/*
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <Python.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include "pst-image.h"

/*** type object *************************************************************/

static PyObject *pst_image_new (PyTypeObject *type,
				PyObject *args,
				PyObject *keywords);
static void pst_image_dealloc (PSTImage *self);

PyTypeObject PSTImage_Type = {
  PyObject_HEAD_INIT(NULL)
  0,				/* ob_size */
  "ST.Image",			/* tp_name */
  sizeof(PSTImage),		/* tp_basicsize */
  0,				/* tp_itemsize */
  (destructor) pst_image_dealloc, /* tp_dealloc */
  NULL,				/* tp_print */
  NULL,				/* tp_getattr */
  NULL,				/* tp_setattr */
  NULL,				/* tp_compare */
  NULL,				/* tp_repr */
  NULL,				/* tp_as_number */
  NULL,				/* tp_as_sequence */
  NULL,				/* tp_as_mapping */
  NULL,				/* tp_hash */
  NULL,				/* tp_call */
  NULL,				/* tp_str */
  NULL,				/* tp_getattro */
  NULL,				/* tp_setattro */
  NULL,				/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
  NULL,				/* tp_doc */
  NULL,				/* tp_traverse */
  NULL,				/* tp_clear */
  NULL,				/* tp_richcompare */
  0,				/* tp_weaklistoffset */
  NULL,				/* tp_iter */
  NULL,				/* tp_iternext */
  NULL,				/* tp_methods */
  NULL,				/* tp_members */
  NULL,				/* tp_getset */
  NULL,				/* tp_base */
  NULL,				/* tp_dict */
  NULL,				/* tp_descr_get */
  NULL,				/* tp_descr_set */
  0,				/* tp_dictoffset */
  NULL,				/* tp_init */
  NULL,				/* tp_alloc */
  pst_image_new			/* tp_new */
};

/*** type methods ************************************************************/

static PyObject *
pst_image_new (PyTypeObject *type, PyObject *args, PyObject *keywords)
{
  PSTImage *self;
  static char *keyword_list[] = { "filename", "buffer", NULL };
  const char *filename = NULL;
  PyObject *buffer = NULL;
  GdkPixbuf *pixbuf;
  GError *err = NULL;
      
  if (! PyArg_ParseTupleAndKeywords(args, keywords, "|sO!", keyword_list,
				    &filename, &PyBuffer_Type, &buffer))
    return NULL;

  if (! filename && ! buffer)
    {
      PyErr_SetString(PyExc_TypeError, _("either filename or buffer must be given"));
      return NULL;
    }
  if (filename && buffer)
    {
      PyErr_SetString(PyExc_TypeError, _("only one of filename or buffer must be given"));
      return NULL;
    }
  
  if (filename)
    {
      pixbuf = gdk_pixbuf_new_from_file(filename, &err);
      if (! pixbuf)
	{
	  PyErr_SetString(PyExc_RuntimeError, err->message);
	  g_error_free(err);

	  return NULL;
	}
    }
  else if (buffer)
    {
      GdkPixbufLoader *loader;
      const char *buf;
      int buf_len;

      if (PyObject_AsReadBuffer(buffer, (const void **) &buf, &buf_len) == -1)
	return NULL;		/* exception is set */

      loader = gdk_pixbuf_loader_new();
      if (! gdk_pixbuf_loader_write(loader, buf, buf_len, &err))
	{
	  PyErr_SetString(PyExc_RuntimeError, err->message);
	  g_error_free(err);

	  return NULL;
	}

      pixbuf = gdk_pixbuf_loader_get_pixbuf(loader);
      g_return_val_if_fail(pixbuf != NULL, NULL);

      g_object_ref(pixbuf);
      gdk_pixbuf_loader_close(loader, NULL);
    }
  else
    g_return_val_if_reached(NULL);
  
  self = (PSTImage *) type->tp_alloc(type, 0);
  self->pixbuf = pixbuf;

  return (PyObject *) self;
}

static void
pst_image_dealloc (PSTImage *self)
{
  g_object_unref(self->pixbuf);

  self->ob_type->tp_free((PyObject *) self);
}

/*** C API *******************************************************************/

gboolean
pst_image_register (PyObject *module)
{
  g_return_val_if_fail(module != NULL, FALSE);

  if (PyType_Ready(&PSTImage_Type) < 0)
    return FALSE;

  Py_INCREF(&PSTImage_Type);
  PyModule_AddObject(module, "Image", (PyObject *) &PSTImage_Type);

  return TRUE;
}

PSTImage *
PSTImage_New (GdkPixbuf *pixbuf)
{
  PSTImage *self;

  g_return_val_if_fail(GDK_IS_PIXBUF(pixbuf), NULL);

  self = PyObject_New(PSTImage, &PSTImage_Type);

  g_object_ref(pixbuf);
  self->pixbuf = pixbuf;

  return self;
}
