# Copyright (c) 2004 Jean-Yves Lefort
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
#    may be used to endorse or promote products derived from this software
#    without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
# CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
# BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
# TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
# ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

import sys, re, ST
from ST import _

### constants #################################################################

class FIELD:
    BANNER, DESCRIPTION, HOMEPAGE = range(3)

PUNKCAST_COM_ROOT	= "http://punkcast.com/"

re_url			= re.compile("href=(http://punkcast.com/)?([0-9]+.*?)>")
re_special_url		= re.compile("href=(http://209.*?)>")
re_img			= re.compile('SRC="(http://punkcast.com/)?(.*?)"')
re_alt			= re.compile('ALT="(.*?)"')

### helpers ###################################################################

class struct:
    def __init__(self, **entries):
        self.__dict__.update(entries)

def streamtunerize (str):
    "Convert a string to UTF-8 and expand its SGML character references."

    # punkcast.com returns no charset information in the Content-Type
    # HTTP header, so we assume that the data is encoded in ISO8859-1
    return ST.sgml_ref_expand(unicode(str, "iso-8859-1").encode("utf8"))

### transfer callbacks ########################################################

def streams_line_cb (line, info):
    if not hasattr(info, "aborted"):
        url = None
        match = re_url.search(line)
        if match is not None:
            url = PUNKCAST_COM_ROOT + match.group(2)
        else:
            match = re_special_url.search(line)
            if match is not None:
                url = match.group(1)

        if url is not None:
            if hasattr(info, "stream"):
                handler.notice(_("found incomplete stream"))

            info.stream = PunkcastComStream()
            info.stream.name = url
            info.stream.set_field(FIELD.HOMEPAGE, url)
            info.streams.append(info.stream)

        match = re_img.search(line)
        if match is not None:
            if not hasattr(info, "stream"):
                handler.notice(_("found misplaced image"))
            else:
                info.stream.image_url = PUNKCAST_COM_ROOT + match.group(2)

        match = re_alt.search(line)
        if match is not None:
            if not hasattr(info, "stream"):
                handler.notice(_("found misplaced alt"))
            else:
                info.stream.set_field(FIELD.DESCRIPTION, streamtunerize(match.group(1)))
                del info.stream
                        
### stream implementation #####################################################

class PunkcastComStream (ST.Stream):
    image_url = None

### handler implementation ####################################################

class PunkcastComHandler (ST.Handler):
    def __new__(cls):
        self = ST.Handler.__new__(cls, "punkcast.com.py")

        self.label = "punkcast.com"
        self.copyright = "Copyright \302\251 2004 Jean-Yves Lefort"
        self.home = PUNKCAST_COM_ROOT
        self.flags = ST.HANDLER_NO_CATEGORIES
        self.icon = ST.find_icon("punkcast.com.png")

        self.add_field(ST.HandlerField(FIELD.BANNER,
                                       _("Show"),
                                       ST.HANDLER_FIELD_TYPE_IMAGE,
                                       ST.HANDLER_FIELD_VISIBLE))
        self.add_field(ST.HandlerField(FIELD.DESCRIPTION,
                                       _("Description"),
                                       ST.HANDLER_FIELD_TYPE_STRING,
                                       ST.HANDLER_FIELD_VISIBLE))
        self.add_field(ST.HandlerField(FIELD.HOMEPAGE,
                                       _("Homepage"),
                                       ST.HANDLER_FIELD_TYPE_STRING,
                                       ST.HANDLER_FIELD_VISIBLE
                                       | ST.HANDLER_FIELD_START_HIDDEN))

        return self

    def reload (self, category):
        info = struct(streams = [])
        session = ST.TransferSession()
        session.get_by_line(PUNKCAST_COM_ROOT,
                            body_cb = streams_line_cb,
                            body_data = info)

        # download the images
        for stream in info.streams:
            if stream.image_url is not None:
                try:
                    buffer = session.get_binary(stream.image_url)
                except ST.AbortError:
                    break
                except:
                    handler.notice(_("unable to download image %s: %s") % (stream.image_url, str(sys.exc_value)))
                else:
                    try:
                        stream.set_field(FIELD.BANNER, ST.Image(buffer = buffer))
                    except:
                        handler.notice(_("unable to load image %s: %s") % (stream.image_url, str(sys.exc_value)))
                
        return (ST.CategoryNode(), info.streams)

    def stream_get_stock_field (self, stream, stock_field):
        if stock_field == ST.HANDLER_STOCK_FIELD_NAME:
            return stream.get_field(FIELD.DESCRIPTION)

    def stream_browse (self, stream):
    	ST.action_run("view-web", stream.get_field(FIELD.HOMEPAGE))

### initialization ############################################################

def init ():
    global handler

    if not ST.check_api_version(1, 1):
        raise RuntimeError, _("API version mismatch")
    
    ST.action_register("view-web", _("Open a web page"), "epiphany %q")

    handler = PunkcastComHandler()
    ST.handlers_add(handler)

init()
