/*
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>
#include <glib-object.h>
#include "st-category-bag.h"
#include "st-category-view.h"
#include "st-handler.h"

/*** type definitions ********************************************************/

struct _STCategoryBagPrivate
{
  GMutex		*mutex;

  unsigned int		flags;
};

/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

/*** function declarations ***************************************************/

static void st_category_bag_class_init (STCategoryBagClass *class);
static void st_category_bag_init (STCategoryBag *bag);
static void st_category_bag_finalize (GObject *object);

static void st_category_bag_set_or_unset_remarkable (STCategoryBag *bag);

/*** implementation **********************************************************/

GType
st_category_bag_get_type (void)
{
  static GType category_bag_type = 0;
  
  if (! category_bag_type)
    {
      static const GTypeInfo category_bag_info = {
	sizeof(STCategoryBagClass),
	NULL,
	NULL,
	(GClassInitFunc) st_category_bag_class_init,
	NULL,
	NULL,
	sizeof(STCategoryBag),
	0,
	(GInstanceInitFunc) st_category_bag_init,
      };
      
      category_bag_type = g_type_register_static(G_TYPE_OBJECT,
						 "STCategoryBag",
						 &category_bag_info,
						 0);
    }

  return category_bag_type;
}

static void
st_category_bag_class_init (STCategoryBagClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);

  parent_class = g_type_class_peek_parent(class);

  object_class->finalize = st_category_bag_finalize;
}

static void
st_category_bag_init (STCategoryBag *bag)
{
  bag->category = NULL;
  bag->handler = NULL;
  memset(&bag->iter, 0, sizeof(bag->iter));
  
  bag->priv = g_new0(STCategoryBagPrivate, 1);
  bag->priv->mutex = g_mutex_new();
}

static void
st_category_bag_finalize (GObject *object)
{
  STCategoryBag *bag = ST_CATEGORY_BAG(object);

  st_handler_event_category_free(bag->handler, bag->category);
  g_mutex_free(bag->priv->mutex);
  g_free(bag->priv);

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

STCategoryBag *
st_category_bag_new (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  return st_category_bag_new_from_category(handler, st_handler_event_category_new(handler));
}

STCategoryBag *
st_category_bag_new_from_category (STHandler *handler, STCategory *category)
{
  STCategoryBag *bag;

  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);
  g_return_val_if_fail(category != NULL, NULL);

  bag = g_object_new(ST_TYPE_CATEGORY_BAG, NULL);

  bag->category = category;
  bag->handler = handler;

  return bag;
}

void
st_category_bag_set_flags (STCategoryBag *bag, unsigned int flags)
{
  g_return_if_fail(ST_IS_CATEGORY_BAG(bag));

  g_mutex_lock(bag->priv->mutex);
  bag->priv->flags |= flags;
  st_category_bag_set_or_unset_remarkable(bag);
  g_mutex_unlock(bag->priv->mutex);
}

void
st_category_bag_unset_flags (STCategoryBag *bag, unsigned int flags)
{
  g_return_if_fail(ST_IS_CATEGORY_BAG(bag));

  g_mutex_lock(bag->priv->mutex);
  bag->priv->flags &= ~flags;
  st_category_bag_set_or_unset_remarkable(bag);
  g_mutex_unlock(bag->priv->mutex);
}

unsigned int
st_category_bag_get_flags (STCategoryBag *bag)
{
  unsigned int flags;

  g_return_val_if_fail(ST_IS_CATEGORY_BAG(bag), 0);

  g_mutex_lock(bag->priv->mutex);
  flags = bag->priv->flags;
  g_mutex_unlock(bag->priv->mutex);

  return flags;
}

gboolean
st_category_bag_apply_url_cb (STCategoryBag *bag)
{
  g_return_val_if_fail(ST_IS_CATEGORY_BAG(bag), FALSE);

  if (ST_CATEGORY(bag)->url_cb)
    {
      g_free(ST_CATEGORY(bag)->url_postfix);
      ST_CATEGORY(bag)->url_postfix = NULL;

      if (! ST_CATEGORY(bag)->url_cb(ST_CATEGORY(bag)))
	return FALSE;
    }

  return TRUE;
}

/*
 * Must be called with mutex held.
 */
static void
st_category_bag_set_or_unset_remarkable (STCategoryBag *bag)
{
  g_return_if_fail(ST_IS_CATEGORY_BAG(bag));

  if (bag->priv->flags != 0)
    st_handler_add_remarkable_category(bag->handler, bag);
  else
    st_handler_remove_remarkable_category(bag->handler, bag);
}

void
st_category_bag_select (STCategoryBag *bag)
{
  STCategoryBag *selected_bag;

  g_return_if_fail(ST_IS_CATEGORY_BAG(bag));

  selected_bag = st_handler_get_selected_category(bag->handler);
  if (selected_bag)
    {
      st_category_bag_unset_flags(selected_bag, ST_CATEGORY_BAG_SELECTED);
      g_object_unref(selected_bag);
    }

  st_category_bag_set_flags(bag, ST_CATEGORY_BAG_SELECTED);
}
