/*
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <string.h>
#include <stdlib.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include "sg-parser.h"
#include "st-handler.h"
#include "st-handler-field.h"
#include "st-handlers.h"
#include "st-parser.h"
#include "st-action.h"
#include "st-settings.h"

/*** type definitions ********************************************************/

enum {
  STATEMENT_OPTIONS = 1,
  STATEMENT_OPTIONS_VIEW_MENUBAR,		/* obsolete */
  STATEMENT_OPTIONS_VIEW_TOOLBAR,
  STATEMENT_OPTIONS_VIEW_TABS,
  STATEMENT_OPTIONS_VIEW_TAB_ICONS,
  STATEMENT_OPTIONS_VIEW_STATUSBAR,
  STATEMENT_OPTIONS_TOOLBAR_THEME,		/* obsolete */
  STATEMENT_OPTIONS_TOOLBAR_STYLE,
  STATEMENT_OPTIONS_TOOLBAR_SIZE,
  STATEMENT_OPTIONS_SAVE_AUTOMATIC,		/* obsolete */
  STATEMENT_OPTIONS_MAIN_WINDOW_WIDTH,
  STATEMENT_OPTIONS_MAIN_WINDOW_HEIGHT,
  STATEMENT_OPTIONS_MAIN_WINDOW_DIVIDER,	/* deprecated */
  STATEMENT_OPTIONS_WINDOW_WIDTH,		/* deprecated */
  STATEMENT_OPTIONS_WINDOW_HEIGHT,		/* deprecated */
  STATEMENT_OPTIONS_WINDOW_DIVIDER,		/* deprecated */
  STATEMENT_OPTIONS_PREFERENCES_WINDOW_WIDTH,
  STATEMENT_OPTIONS_PREFERENCES_WINDOW_HEIGHT,
  STATEMENT_OPTIONS_STREAM_PROPERTIES_WINDOW_WIDTH,
  STATEMENT_OPTIONS_SELECTED_PREFERENCES_PAGE,
  STATEMENT_OPTIONS_PROXY_ENABLED,
  STATEMENT_OPTIONS_PROXY_TYPE,
  STATEMENT_OPTIONS_PROXY_URL,			/* deprecated */
  STATEMENT_OPTIONS_PROXY_SERVER,
  STATEMENT_OPTIONS_PROXY_PORT,
  STATEMENT_OPTIONS_PROXY_AUTH_ENABLED,
  STATEMENT_OPTIONS_PROXY_AUTH_NAME,
  STATEMENT_OPTIONS_PROXY_AUTH_PASSWORD,
  STATEMENT_OPTIONS_GALEON_THEMES_ENABLED,	/* obsolete */
  STATEMENT_OPTIONS_GALEON_THEMES_SYSTEM_DIR,	/* obsolete */
  STATEMENT_OPTIONS_GALEON_THEMES_USER_DIR,	/* obsolete */
  STATEMENT_OPTIONS_SELECTED_HANDLER,
  STATEMENT_OPTIONS_ALWAYS_REFRESH,		/* deprecated */
  STATEMENT_OPTIONS_ALWAYS_RELOAD,
  STATEMENT_OPTIONS_FIND_TOKEN,
  STATEMENT_OPTIONS_FIND_CASE_SENSITIVE,
  STATEMENT_OPTIONS_FIND_WRAP_AROUND,
  STATEMENT_OPTIONS_FIND_HISTORY,
  STATEMENT_OPTIONS_MUSIC_DIR,

  STATEMENT_ACTION,
  STATEMENT_ACTION_COMMAND,
  STATEMENT_ACTION_PROGRAM,			/* deprecated */

  STATEMENT_HANDLER,
  STATEMENT_HANDLER_FIELDS_SORT_INDEX,
  STATEMENT_HANDLER_FIELDS_SORT_ORDER,
  STATEMENT_HANDLER_PANED_POSITION,
  STATEMENT_HANDLER_SELECTED_CATEGORY,		/* obsolete */
  STATEMENT_HANDLER_SELECTED_STREAM,		/* obsolete */
  STATEMENT_HANDLER_FIELDS_WIDTH,		/* deprecated */
  STATEMENT_HANDLER_FIELD,
  STATEMENT_HANDLER_FIELD_VISIBLE,
  STATEMENT_HANDLER_FIELD_WIDTH,
  STATEMENT_HANDLER_EXPANDED_CATEGORIES,	/* obsolete */
  STATEMENT_HANDLER_SELECTED_STREAMS		/* obsolete */
};

typedef struct
{
  SGParser		*parser;
  SGParserStatement	*statement;

  char			*action;
  
  GSList		*handlers;
  STHandler		*handler;
  GSList		*handler_field_iter;
} STConfigLoadInfo;

/*** constant definitions ****************************************************/

static SGParserDefinition config_definitions[] = {
  {
    0,			STATEMENT_OPTIONS,
    "options",		TRUE,		G_TYPE_NONE
  },
  {				/* obsolete */
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_VIEW_MENUBAR,
    "view_menubar",	FALSE,		G_TYPE_INT
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_VIEW_TOOLBAR,
    "view_toolbar",	FALSE,		G_TYPE_BOOLEAN
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_VIEW_TABS,
    "view_tabs",	FALSE,		G_TYPE_BOOLEAN
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_VIEW_TAB_ICONS,
    "view_tab_icons",	FALSE,		G_TYPE_BOOLEAN
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_VIEW_STATUSBAR,
    "view_statusbar",	FALSE,		G_TYPE_BOOLEAN
  },
  {				/* obsolete */
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_TOOLBAR_THEME,
    "toolbar_theme",	FALSE,		G_TYPE_STRING
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_TOOLBAR_STYLE,
    "toolbar_style",	FALSE,		G_TYPE_INT
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_TOOLBAR_SIZE,
    "toolbar_size",	FALSE,		G_TYPE_INT
  },
  {				/* obsolete */
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_SAVE_AUTOMATIC,
    "save_automatic",	FALSE,		G_TYPE_BOOLEAN
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_MAIN_WINDOW_WIDTH,
    "main_window_width",	FALSE,	G_TYPE_INT
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_MAIN_WINDOW_HEIGHT,
    "main_window_height",	FALSE,	G_TYPE_INT
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_MAIN_WINDOW_HEIGHT,
    "main_window_height",	FALSE,	G_TYPE_INT
  },
  {				/* deprecated */
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_MAIN_WINDOW_DIVIDER,
    "main_window_divider",	FALSE,	G_TYPE_INT
  },
  {				/* deprecated */
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_WINDOW_WIDTH,
    "window_width",	FALSE,	G_TYPE_INT
  },
  {				/* deprecated */
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_WINDOW_HEIGHT,
    "window_height",	FALSE,	G_TYPE_INT
  },
  {				/* deprecated */
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_WINDOW_DIVIDER,
    "window_divider",	FALSE,	G_TYPE_INT
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_PREFERENCES_WINDOW_WIDTH,
    "preferences_window_width", FALSE,	G_TYPE_INT
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_PREFERENCES_WINDOW_HEIGHT,
    "preferences_window_height", FALSE,	G_TYPE_INT
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_STREAM_PROPERTIES_WINDOW_WIDTH,
    "stream_properties_window_width", FALSE, G_TYPE_INT
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_SELECTED_PREFERENCES_PAGE,
    "selected_preferences_page", FALSE,	G_TYPE_INT
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_PROXY_ENABLED,
    "proxy_enabled",	FALSE,		G_TYPE_BOOLEAN
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_PROXY_TYPE,
    "proxy_type",	FALSE,		G_TYPE_INT
  },
  {				/* deprecated */
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_PROXY_URL,
    "proxy_url",	FALSE,		G_TYPE_STRING
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_PROXY_SERVER,
    "proxy_server",	FALSE,		G_TYPE_STRING
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_PROXY_PORT,
    "proxy_port",	FALSE,		G_TYPE_INT
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_PROXY_AUTH_ENABLED,
    "proxy_auth_enabled",	FALSE,	G_TYPE_BOOLEAN
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_PROXY_AUTH_NAME,
    "proxy_auth_name",	FALSE,		G_TYPE_STRING
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_PROXY_AUTH_PASSWORD,
    "proxy_auth_password",	FALSE,	G_TYPE_STRING
  },
  {				/* obsolete */
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_GALEON_THEMES_ENABLED,
    "galeon_themes_enabled",	FALSE,	G_TYPE_INT
  },
  {				/* obsolete */
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_GALEON_THEMES_SYSTEM_DIR,
    "galeon_themes_system_dir",	FALSE,	G_TYPE_STRING
  },
  {				/* obsolete */
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_GALEON_THEMES_USER_DIR,
    "galeon_themes_user_dir",	FALSE,	G_TYPE_STRING
  },
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_SELECTED_HANDLER,
    "selected_handler",	FALSE,		G_TYPE_STRING
  }, 
  {				/* deprecated */
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_ALWAYS_REFRESH,
    "always_refresh",	FALSE,		G_TYPE_BOOLEAN
  }, 
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_ALWAYS_RELOAD,
    "always_reload",	FALSE,		G_TYPE_BOOLEAN
  }, 
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_FIND_TOKEN,
    "find_token",	FALSE,		G_TYPE_STRING
  }, 
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_FIND_CASE_SENSITIVE,
    "find_case_sensitive",	FALSE,	G_TYPE_BOOLEAN
  }, 
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_FIND_WRAP_AROUND,
    "find_wrap_around",		FALSE,	G_TYPE_BOOLEAN
  }, 
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_FIND_HISTORY,
    "find_history",	TRUE,		G_TYPE_NONE
  }, 
  {
    STATEMENT_OPTIONS,	STATEMENT_OPTIONS_MUSIC_DIR,
    "music_dir",	FALSE,		G_TYPE_STRING
  }, 
  {
    0,			STATEMENT_ACTION,
    "action",		TRUE,		G_TYPE_STRING
  },
  {
    STATEMENT_ACTION,	STATEMENT_ACTION_COMMAND,
    "command",		FALSE,		G_TYPE_STRING
  },
  {				/* deprecated */
    STATEMENT_ACTION,	STATEMENT_ACTION_PROGRAM,
    "program",		FALSE,		G_TYPE_STRING
  },
  {
    0,			STATEMENT_HANDLER,
    "handler",		TRUE,		G_TYPE_STRING
  },
  {
    STATEMENT_HANDLER,	STATEMENT_HANDLER_FIELDS_SORT_INDEX,
    "fields_sort_index",	FALSE,	G_TYPE_INT
  },
  {
    STATEMENT_HANDLER,	STATEMENT_HANDLER_FIELDS_SORT_ORDER,
    "fields_sort_order",	FALSE,	G_TYPE_INT
  },
  {
    STATEMENT_HANDLER,	STATEMENT_HANDLER_PANED_POSITION,
    "paned_position",		FALSE,	G_TYPE_INT
  },
  {				/* obsolete */
    STATEMENT_HANDLER,	STATEMENT_HANDLER_SELECTED_CATEGORY,
    "selected_category",	FALSE,	G_TYPE_STRING
  },
  {				/* obsolete */
    STATEMENT_HANDLER,	STATEMENT_HANDLER_SELECTED_STREAM,
    "selected_stream",	FALSE,		G_TYPE_STRING
  },
  {				/* deprecated */
    STATEMENT_HANDLER,	STATEMENT_HANDLER_FIELDS_WIDTH,
    "fields_width",	TRUE,		G_TYPE_NONE
  },
  {
    STATEMENT_HANDLER,	STATEMENT_HANDLER_FIELD,
    "field",		TRUE,		G_TYPE_NONE
  },
  {
    STATEMENT_HANDLER_FIELD,	STATEMENT_HANDLER_FIELD_VISIBLE,
    "visible",		FALSE,		G_TYPE_BOOLEAN
  },
  {
    STATEMENT_HANDLER_FIELD,	STATEMENT_HANDLER_FIELD_WIDTH,
    "width",		FALSE,		G_TYPE_INT
  },
  {				/* obsolete */
    STATEMENT_HANDLER,	STATEMENT_HANDLER_EXPANDED_CATEGORIES,
    "expanded_categories",	TRUE,	G_TYPE_NONE
  },
  {				/* obsolete */
    STATEMENT_HANDLER,	STATEMENT_HANDLER_SELECTED_STREAMS,
    "selected_streams",	TRUE,		G_TYPE_NONE
  },
  { 0, 0, NULL, 0, 0 }
};
  
/*** function declarations ***************************************************/

static void st_config_load_find_history              (STConfigLoadInfo *info);

static void st_config_load_handler_begin	     (STConfigLoadInfo *info);
static void st_config_load_handler_end		     (STConfigLoadInfo *info);

static void st_config_load_handler_field_begin	     (STConfigLoadInfo *info);
static void st_config_load_handler_field_end	     (STConfigLoadInfo *info);

static void st_config_load_command		     (STConfigLoadInfo *info);
static void st_config_load_program		     (STConfigLoadInfo *info);
static void st_config_load_handler_field_width	     (STConfigLoadInfo *info);

/*** implementation **********************************************************/

gboolean
st_config_load (const char *filename, GError **err)
{ 
  STConfigLoadInfo info = { NULL, };

  g_return_val_if_fail(filename != NULL, FALSE);

  info.parser = sg_parser_new(filename, err);
  if (! info.parser)
    return FALSE;
  
  sg_parser_set_message_cb(info.parser, st_parser_message_cb);
  sg_parser_definev(info.parser, config_definitions);

  while ((info.statement = sg_parser_get_statement(info.parser)))
    {
      if (SG_PARSER_STATEMENT_IS_END(info.statement))
	{
	  switch (SG_PARSER_SCOPE(info.parser))
	    {
	    case STATEMENT_HANDLER:
	      st_config_load_handler_end(&info);
	      break;

	    case STATEMENT_HANDLER_FIELD:
	      st_config_load_handler_field_end(&info);
	      break;
	    }
	}
    
      if (info.statement->definition)
	{
	  switch (info.statement->definition->id)
	    {
	    case STATEMENT_OPTIONS_VIEW_TOOLBAR:
	      st_settings.view_toolbar = g_value_get_boolean(&info.statement->value);
	      break;
	
	    case STATEMENT_OPTIONS_VIEW_TABS:
	      st_settings.view_tabs = g_value_get_boolean(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_VIEW_TAB_ICONS:
	      st_settings.view_tab_icons = g_value_get_boolean(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_VIEW_STATUSBAR:
	      st_settings.view_statusbar = g_value_get_boolean(&info.statement->value);
	      break;
	      
	    case STATEMENT_OPTIONS_TOOLBAR_STYLE:
	      st_settings.toolbar_style = g_value_get_int(&info.statement->value);
	      break;
	      
	    case STATEMENT_OPTIONS_TOOLBAR_SIZE:
	      st_settings.toolbar_size = g_value_get_int(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_WINDOW_WIDTH:	/* deprecated */
	    case STATEMENT_OPTIONS_MAIN_WINDOW_WIDTH:
	      st_settings.main_window_width = g_value_get_int(&info.statement->value);
	      break;
	      
	    case STATEMENT_OPTIONS_WINDOW_HEIGHT:	/* deprecated */
	    case STATEMENT_OPTIONS_MAIN_WINDOW_HEIGHT:
	      st_settings.main_window_height = g_value_get_int(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_SELECTED_PREFERENCES_PAGE:
	      st_settings.selected_preferences_page = g_value_get_int(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_PREFERENCES_WINDOW_WIDTH:
	      st_settings.preferences_window_width = g_value_get_int(&info.statement->value);
	      break;
	      
	    case STATEMENT_OPTIONS_PREFERENCES_WINDOW_HEIGHT:
	      st_settings.preferences_window_height = g_value_get_int(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_STREAM_PROPERTIES_WINDOW_WIDTH:
	      st_settings.stream_properties_window_width = g_value_get_int(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_PROXY_ENABLED:
	      st_settings.proxy_enabled = g_value_get_boolean(&info.statement->value);
	      break;
	      
	    case STATEMENT_OPTIONS_PROXY_TYPE:
	      st_settings.proxy_type = g_value_get_int(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_PROXY_URL:
	      {
		const char *proxy_url;
		char *server = NULL;
		int port = 0;
		
		proxy_url = g_value_get_string(&info.statement->value);
		if (proxy_url)
		  {
		    const char *protocol;
		    const char *portstr;

		    /* skip protocol */

		    protocol = strstr(proxy_url, "://");
		    if (protocol)
		      proxy_url = protocol + 3;

		    /* extract port */

		    portstr = strrchr(proxy_url, ':');
		    if (portstr)
		      {
			server = g_strndup(proxy_url, portstr - proxy_url);
			port = atoi(portstr + 1);
		      }
		    else
		      server = g_strdup(proxy_url);
		  }

		g_free(st_settings.proxy_server);
		st_settings.proxy_server = server;
		st_settings.proxy_port = port;

		break;
	      }

	    case STATEMENT_OPTIONS_PROXY_SERVER:
	      g_free(st_settings.proxy_server);
	      st_settings.proxy_server = g_value_dup_string(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_PROXY_PORT:
	      st_settings.proxy_port = g_value_get_int(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_PROXY_AUTH_ENABLED:
	      st_settings.proxy_auth_enabled = g_value_get_boolean(&info.statement->value);
	      break;
	      
	    case STATEMENT_OPTIONS_PROXY_AUTH_NAME:
	      g_free(st_settings.proxy_auth_name);
	      st_settings.proxy_auth_name = g_value_dup_string(&info.statement->value);
	      break;
	      
	    case STATEMENT_OPTIONS_PROXY_AUTH_PASSWORD:
	      g_free(st_settings.proxy_auth_password);
	      st_settings.proxy_auth_password = g_value_dup_string(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_VIEW_MENUBAR:
	    case STATEMENT_OPTIONS_TOOLBAR_THEME:
	    case STATEMENT_OPTIONS_GALEON_THEMES_ENABLED:
	    case STATEMENT_OPTIONS_GALEON_THEMES_SYSTEM_DIR:
	    case STATEMENT_OPTIONS_GALEON_THEMES_USER_DIR:
	    case STATEMENT_HANDLER_SELECTED_CATEGORY:
	    case STATEMENT_HANDLER_SELECTED_STREAM:
	    case STATEMENT_HANDLER_EXPANDED_CATEGORIES:
	    case STATEMENT_HANDLER_SELECTED_STREAMS:
	    case STATEMENT_OPTIONS_SAVE_AUTOMATIC:
	      sg_parser_warn(info.parser, _("obsolete statement, ignored"));
	      break;

	    case STATEMENT_OPTIONS_SELECTED_HANDLER:
	      g_free(st_settings.selected_handler_name);
	      st_settings.selected_handler_name = g_value_dup_string(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_ALWAYS_REFRESH: /* deprecated */
	    case STATEMENT_OPTIONS_ALWAYS_RELOAD:
	      st_settings.always_reload = g_value_get_boolean(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_FIND_TOKEN:
	      g_free(st_settings.find_token);
	      st_settings.find_token = g_value_dup_string(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_FIND_CASE_SENSITIVE:
	      st_settings.find_case_sensitive = g_value_get_boolean(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_FIND_WRAP_AROUND:
	      st_settings.find_wrap_around = g_value_get_boolean(&info.statement->value);
	      break;

	    case STATEMENT_OPTIONS_MUSIC_DIR:
	      g_free(st_settings.music_dir);
	      st_settings.music_dir = g_value_dup_string(&info.statement->value);
	      break;

	    case STATEMENT_ACTION:
	      info.action = g_value_dup_string(&info.statement->value);
	      break;

	    case STATEMENT_ACTION_COMMAND:
	      st_config_load_command(&info);
	      break;

	    case STATEMENT_ACTION_PROGRAM:
	      st_config_load_program(&info);
	      break;
	      
	    case STATEMENT_HANDLER:
	      st_config_load_handler_begin(&info);
	      break;
	      
	    case STATEMENT_HANDLER_FIELDS_SORT_INDEX:
	      if (info.handler)
		{
		  int sort_index = g_value_get_int(&info.statement->value);
		  
		  if (sort_index > st_handler_count_fields(info.handler, ST_HANDLER_FIELD_VISIBLE))
		    sg_parser_warn(info.parser, _("index too big, ignored"));
		  else
		    st_handler_set_fields_sort_index(info.handler, sort_index);
		}
	      break;
	      
	    case STATEMENT_HANDLER_FIELDS_SORT_ORDER:
	      if (info.handler)
		st_handler_set_fields_sort_order(info.handler, g_value_get_int(&info.statement->value));
	      break;
	      
	    case STATEMENT_HANDLER_PANED_POSITION:
	      if (info.handler)
		st_handler_set_paned_position(info.handler, g_value_get_int(&info.statement->value));
	      break;

	    case STATEMENT_HANDLER_FIELD:
	      st_config_load_handler_field_begin(&info);
	      break;

	    case STATEMENT_HANDLER_FIELD_VISIBLE:
	      if (info.handler_field_iter)
		{
		  STHandlerField *field = info.handler_field_iter->data;
		  st_handler_field_set_user_visible(field, g_value_get_boolean(&info.statement->value));
		}
	      break;

	    case STATEMENT_HANDLER_FIELD_WIDTH:
	      if (info.handler_field_iter)
		{
		  STHandlerField *field = info.handler_field_iter->data;
		  st_handler_field_set_width(field, g_value_get_int(&info.statement->value));
		}
	      break;
	    }
	}
      else if (G_IS_VALUE(&info.statement->value))
	{
	  switch (SG_PARSER_SCOPE(info.parser))
	    {
	    case STATEMENT_OPTIONS_FIND_HISTORY:
	      st_config_load_find_history(&info);
	      break;

	    case STATEMENT_HANDLER_FIELDS_WIDTH:
	      st_config_load_handler_field_width(&info);
	      break;
	      
	    case STATEMENT_HANDLER_EXPANDED_CATEGORIES:
	    case STATEMENT_HANDLER_SELECTED_STREAMS:
	      /* obsolete */
	      break;

	    default:
	      sg_parser_warn(info.parser, _("unexpected value"));
	    }
	}

      sg_parser_statement_evaluate(info.statement);
      sg_parser_statement_free(info.statement);
    }
  
  sg_parser_free(info.parser);
  st_handlers_reorder(info.handlers);
  
  return TRUE;
}

static void
st_config_load_find_history (STConfigLoadInfo *info)
{
  if (! G_VALUE_HOLDS_STRING(&info->statement->value))
    {
      sg_parser_warn(info->parser, _("expected string value"));
      return;
    }

  if (g_slist_length(st_settings.find_history) < ST_SETTINGS_FIND_HISTORY_MAX_LENGTH)
    st_settings.find_history = g_slist_append(st_settings.find_history, g_value_dup_string(&info->statement->value));
  else
    sg_parser_warn(info->parser, _("too many find history tokens"));
}

static void
st_config_load_handler_begin (STConfigLoadInfo *info)
{
  const char *name;

  name = g_value_get_string(&info->statement->value);
  info->handler = st_handlers_find_by_name(name);
  
  if (info->handler)
    {
      info->handler_field_iter = st_handler_get_fields(info->handler);
      info->handlers = g_slist_append(info->handlers, info->handler);
    }
  else
    sg_parser_warn(info->parser, _("%s: no such handler"), name);
}

static void
st_config_load_handler_end (STConfigLoadInfo *info)
{
  info->handler = NULL;
  info->handler_field_iter = NULL;
}

static void
st_config_load_handler_field_begin (STConfigLoadInfo *info)
{
  if (! info->handler)
    return;
  
  while (info->handler_field_iter)
    {
      STHandlerField *field = info->handler_field_iter->data;
      
      if (ST_HANDLER_FIELD_IS_VISIBLE(field))
	break;
      
      info->handler_field_iter = info->handler_field_iter->next;
    }
		  
  if (! info->handler_field_iter)
    sg_parser_warn(info->parser, _("too many fields"));
}

static void
st_config_load_handler_field_end (STConfigLoadInfo *info)
{
  if (info->handler_field_iter)
    info->handler_field_iter = info->handler_field_iter->next;
  /* otherwise there has been a "too many fields" error -> nop */
}

static void
st_config_load_command (STConfigLoadInfo *info)
{
  if (! info->action)
    {
      sg_parser_warn(info->parser, _("command already specified"));
      return;
    }

  if (st_action_exists(info->action))
    st_action_associate(info->action, g_value_get_string(&info->statement->value));

  g_free(info->action);
  info->action = NULL;
}

static void
st_config_load_program (STConfigLoadInfo *info)
{
  if (! info->action)
    {
      sg_parser_warn(info->parser, _("program already specified"));
      return;
    }

  if (st_action_exists(info->action))
    {
      const char *command;
      char *new_command;

      command = g_value_get_string(&info->statement->value);
      new_command = st_action_convert_old_style_command(command);

      st_action_associate(info->action, new_command);
      g_free(new_command);
    }

  g_free(info->action);
  info->action = NULL;
}

static void
st_config_load_handler_field_width (STConfigLoadInfo *info)
{
  STHandlerField *field = NULL;

  if (! info->handler)
    return;

  if (! G_VALUE_HOLDS_INT(&info->statement->value))
    {
      sg_parser_warn(info->parser, _("expected integer value"));
      return;
    }

  while (info->handler_field_iter)
    {
      field = info->handler_field_iter->data;

      info->handler_field_iter = info->handler_field_iter->next;

      if (ST_HANDLER_FIELD_IS_VISIBLE(field))
	break;
      else
	field = NULL;
    }
  
  if (field)
    st_handler_field_set_width(field, g_value_get_int(&info->statement->value));
  else
    sg_parser_warn(info->parser, _("too many fields"));
}
