/*
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include "art/general_preferences.h"
#include "sgtk-hig.h"
#include "st-general-tab.h"
#include "st-preferences.h"
#include "st-settings.h"
#include "st-preferences-tab.h"
#include "st-dialog-api.h"

/*** type definitions ********************************************************/

struct _STGeneralTabPrivate
{
  GtkTooltips	*tooltips;
  GtkWidget	*folder_entry;
};
  
/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

/*** function declarations ***************************************************/

static void st_general_tab_class_init (STGeneralTabClass *class);
static void st_general_tab_init (STGeneralTab *tab);
static void st_general_tab_finalize (GObject *object);

static void st_general_tab_browse_clicked_h (GtkButton *button,
					     gpointer user_data);

/*** implementation **********************************************************/

GType
st_general_tab_get_type (void)
{
  static GType general_tab_type = 0;
  
  if (! general_tab_type)
    {
      static const GTypeInfo general_tab_info = {
	sizeof(STGeneralTabClass),
	NULL,
	NULL,
	(GClassInitFunc) st_general_tab_class_init,
	NULL,
	NULL,
	sizeof(STGeneralTab),
	0,
	(GInstanceInitFunc) st_general_tab_init,
      };

      general_tab_type = g_type_register_static(ST_TYPE_PREFERENCES_TAB,
						"STGeneralTab",
						&general_tab_info,
						0);
    }

  return general_tab_type;
}

static void
st_general_tab_class_init (STGeneralTabClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);
  STPreferencesTabClass *preferences_tab_class = ST_PREFERENCES_TAB_CLASS(class);

  parent_class = g_type_class_peek_parent(class);
  
  object_class->finalize = st_general_tab_finalize;

  preferences_tab_class->link_id = "preferences-general";
}

static void
st_general_tab_init (STGeneralTab *tab)
{
  GdkPixbuf *pixbuf;
  GtkWidget *vbox;
  GtkWidget *splash_enabled;
  GtkWidget *always_reload;
  GtkWidget *folder_hbox1;
  GtkWidget *folder_label;
  GtkWidget *folder_hbox2;
  GtkWidget *folder_button;

  tab->priv = g_new0(STGeneralTabPrivate, 1);
  tab->priv->tooltips = gtk_tooltips_new();

  pixbuf = gdk_pixbuf_new_from_inline(sizeof(art_general_preferences),
				      art_general_preferences,
				      FALSE,
				      NULL);

  st_preferences_tab_construct_label(ST_PREFERENCES_TAB(tab), pixbuf, _("General"));
  g_object_unref(pixbuf);

  gtk_container_set_border_width(GTK_CONTAINER(tab), SGTK_HIG_CONTROLS_BORDER_WIDTH);

  splash_enabled = gtk_check_button_new_with_mnemonic(_("Sh_ow splash screen at startup"));
  gtk_tooltips_set_tip(tab->priv->tooltips, splash_enabled,
		       _("If this option is enabled, a splash screen will be shown when starting streamtuner."),
		       NULL);

  always_reload = gtk_check_button_new_with_mnemonic(_("_Always reload categories"));
  gtk_tooltips_set_tip(tab->priv->tooltips, always_reload,
		       _("If this option is enabled, categories will always be reloaded when you select them."),
		       NULL);

  folder_hbox1 = gtk_hbox_new(FALSE, SGTK_HIG_CONTROL_LABEL_SPACING);
  folder_label = gtk_label_new_with_mnemonic(_("_Music folder:"));

  folder_hbox2 = gtk_hbox_new(FALSE, SGTK_HIG_CONTROL_SPACING);

  tab->priv->folder_entry = gtk_entry_new();
  gtk_tooltips_set_tip(tab->priv->tooltips, tab->priv->folder_entry,
		       _("The path to your music collection"),
		       NULL);

  gtk_label_set_mnemonic_widget(GTK_LABEL(folder_label), tab->priv->folder_entry);

  folder_button = gtk_button_new_with_mnemonic(_("_Browse..."));
  g_signal_connect(folder_button, "clicked", G_CALLBACK(st_general_tab_browse_clicked_h), tab);

  gtk_box_pack_start(GTK_BOX(folder_hbox2), tab->priv->folder_entry, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(folder_hbox2), folder_button, FALSE, FALSE, 0);
  
  gtk_box_pack_start(GTK_BOX(folder_hbox1), folder_label, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(folder_hbox1), folder_hbox2, TRUE, TRUE, 0);

  st_preferences_bind_boolean(GTK_TOGGLE_BUTTON(splash_enabled), &st_settings.splash_enabled);
  st_preferences_bind_boolean(GTK_TOGGLE_BUTTON(always_reload), &st_settings.always_reload);
  st_preferences_bind_string(GTK_ENTRY(tab->priv->folder_entry), &st_settings.music_dir);
 
  vbox = gtk_vbox_new(FALSE, SGTK_HIG_CONTROL_SPACING);

  gtk_box_pack_start(GTK_BOX(vbox), splash_enabled, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), always_reload, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), folder_hbox1, FALSE, FALSE, 0);

  gtk_widget_show_all(vbox);
  gtk_container_add(GTK_CONTAINER(tab), vbox);
}

static void
st_general_tab_finalize (GObject *object)
{
  STGeneralTab *tab = ST_GENERAL_TAB(object);

  gtk_object_destroy(GTK_OBJECT(tab->priv->tooltips));
  g_free(tab->priv);

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
st_general_tab_browse_clicked_h (GtkButton *button, gpointer user_data)
{
  STGeneralTab *tab = user_data;
  GtkWidget *toplevel;
  GtkWidget *chooser;
  const char *folder;

  toplevel = gtk_widget_get_toplevel(GTK_WIDGET(tab));
  chooser = gtk_file_chooser_dialog_new(_("Select a Music Folder"),
					GTK_WINDOW(toplevel),
					GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_OPEN, 1,
					NULL);

  folder = gtk_entry_get_text(GTK_ENTRY(tab->priv->folder_entry));
  if (*folder)
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(chooser), folder);

  if (gtk_dialog_run(GTK_DIALOG(chooser)) == 1)
    {
      char *filename;
      char *converted = NULL;

      filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(chooser));
      if (filename)
	{
	  GError *err = NULL;

	  converted = g_filename_to_utf8(filename, -1, NULL, NULL, &err);
	  g_free(filename);

	  if (! converted)
	    {
	      st_notice(_("unable to convert filename to UTF-8 encoding: %s"), err->message);
	      g_error_free(err);
	    }
	}
      
      gtk_entry_set_text(GTK_ENTRY(tab->priv->folder_entry), converted ? converted : "");
      g_free(converted);

      /* focus-out-event will not be emitted, so we have to commit manually */
      st_preferences_commit_string(GTK_ENTRY(tab->priv->folder_entry));
    }

  gtk_widget_destroy(chooser);
}

GtkWidget *
st_general_tab_new (void)
{
  return g_object_new(ST_TYPE_GENERAL_TAB, NULL);
}
