/*
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include "art/logo.h"
#include "st-action.h"
#include "st-dialog-api.h"

/*** API implementation ******************************************************/

char *
st_format_bitrate (int bitrate)
{
  g_return_val_if_fail(bitrate > 0, NULL);
  return g_strdup_printf(_("%i kbps"), bitrate);
}

char *
st_format_samplerate (int samplerate)
{
  g_return_val_if_fail(samplerate > 0, NULL);
  return g_strdup_printf(_("%i Hz"), samplerate);
}

char *
st_format_channels (int channels)
{
  g_return_val_if_fail(channels > 0, NULL);
  switch (channels)
    {
    case 1:	return g_strdup(_("Mono"));
    case 2:	return g_strdup(_("Stereo"));
    default:	return g_strdup_printf(ngettext("%i channel", "%i channels", channels), channels);
    }
}

char *
st_format_audio_properties (int bitrate,
			    int samplerate,
			    int channels)
{
  GString *audio;
  char *str;

  audio = g_string_new(NULL);

  if (bitrate > 0)
    {
      str = st_format_bitrate(bitrate);
      g_string_append(audio, str);
      g_free(str);
    }
  
  if (samplerate > 0)
    {
      if (*audio->str)
	g_string_append(audio, ", ");

      str = st_format_samplerate(samplerate);
      g_string_append(audio, str);
      g_free(str);
    }
  
  if (channels > 0)
    {
      if (*audio->str)
	g_string_append(audio, ", ");

      str = st_format_channels(channels);
      g_string_append(audio, str);
      g_free(str);
    }

  if (*audio->str)
    return g_string_free(audio, FALSE);
  else
    {
      g_string_free(audio, TRUE);
      return NULL;
    }
}

/*** private implementation **************************************************/

GtkWidget *
st_logo_new (void)
{
  GdkPixbuf *logo;
  GtkWidget *image;

  logo = gdk_pixbuf_new_from_inline(sizeof(art_logo),
				    art_logo,
				    FALSE,
				    NULL);
  image = gtk_image_new_from_pixbuf(logo);
  g_object_unref(logo);

  return image;
}

void
st_show_help (const char *link_id)
{
  char *filename;

  filename = g_build_filename(BASEDOCDIR, "C", "streamtuner.xml", NULL);

  if (g_file_test(filename, G_FILE_TEST_EXISTS))
    {
      char *uri;
      gboolean status;
      GError *err = NULL;
      
      uri = link_id
	? g_strconcat("ghelp://", filename, "?", link_id, NULL)
	: g_strconcat("ghelp://", filename, NULL);

      status = st_action_run("view-ghelp", uri, &err);
      g_free(uri);

      if (! status)
	{
	  char *normalized;

	  normalized = st_dialog_normalize(err->message);
	  g_error_free(err);

	  st_error_dialog(_("Unable to view the manual"), "%s", normalized);
	  g_free(normalized);
	}
    }
  else
    st_error_dialog(_("Unable to view the manual"),
		    _("The manual file %s does not exist."), filename);
  
  g_free(filename);
}

void
st_visit_homepage (const char *url)
{
  GError *err = NULL;

  g_return_if_fail(url != NULL);

  if (! st_action_run("view-web", url, &err))
    {
      char *normalized;

      normalized = st_dialog_normalize(err->message);

      st_error_dialog(_("Unable to visit the homepage"), "%s", normalized);

      g_free(normalized);
      g_error_free(err);
    }
}
