/*
 * $Id: live365.c,v 1.48 2004/03/28 17:16:04 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <streamtuner/streamtuner.h>
#include <string.h>
#include "art/icon.h"
#include "gettext.h"

/*** cpp *********************************************************************/

#define COPYRIGHT		"Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort"

#define LIVE365_ROOT		"http://www.live365.com/"
#define DIRECTORY_PREFIX	"cgi-bin/directory.cgi?genre="

/*** types *******************************************************************/

typedef struct
{
  STStream	stream;
  
  char		*title;
  char		*genre;
  char		*description;
  char		*broadcaster;
  char		*audio;
  char		*url_postfix;
  char		*homepage;

  char		*url;		/* will be read and written in threads... */
  GMutex	*url_mutex;	/* ...so we protect it */
} Live365Stream;

typedef struct
{
  GSList	*labels;
  GSList	*names;
} Genres;

typedef struct
{
  char		*charset;

  GList		**streams;
  Live365Stream	*stream;
} RefreshStreamsInfo;

enum {
  FIELD_TITLE,
  FIELD_GENRE,
  FIELD_DESCRIPTION,
  FIELD_BROADCASTER,
  FIELD_AUDIO,
  FIELD_URL_POSTFIX,
  FIELD_HOMEPAGE,
  FIELD_URL
};

/*** variables ***************************************************************/

static regex_t re_header_charset;
static regex_t re_body_charset;
static regex_t re_title;
static regex_t re_broadcaster;
static regex_t re_genre;
static regex_t re_audio;
static regex_t re_description;
static regex_t re_stream;
static regex_t re_start_genre_list;
static regex_t re_start_genre_id_list;
static regex_t re_end_list;
static regex_t re_item;
static regex_t re_stationid;

static GNode *session_categories = NULL;
static const char *ident_user;
static const char *ident_session;

/*** functions ***************************************************************/

static Live365Stream	*stream_new_cb		(gpointer	data);
static void		stream_field_get_cb	(Live365Stream	*stream,
						 STHandlerField	*field,
						 GValue		*value,
						 gpointer	data);
static void		stream_field_set_cb	(Live365Stream	*stream,
						 STHandlerField	*field,
						 const GValue	*value,
						 gpointer	data);
static void		stream_stock_field_get_cb (Live365Stream	*stream,
						   STHandlerStockField	stock_field,
						   GValue		*value,
						   gpointer		data);
static void		stream_free_cb		(Live365Stream	*stream,
						 gpointer	data);

static gboolean		stream_resolve		(Live365Stream	*stream,
						 GError		**err);
static void		stream_resolve_line_cb	(const char	*line,
						 gpointer	data);

static gboolean		stream_resolve_cb	(Live365Stream	*stream,
						 gpointer	data,
						 GError		**err);
static gboolean		stream_tune_in_cb	(Live365Stream	*stream,
						 gpointer	data,
						 GError		**err);
static gboolean		stream_record_cb	(Live365Stream	*stream,
						 gpointer	data,
						 GError		**err);
static gboolean		stream_browse_cb	(Live365Stream	*stream,
						 gpointer	data,
						 GError		**err);
static gboolean		refresh_cb		(STCategory	*category,
						 GNode		**categories,
						 GList		**streams,
						 gpointer	data,
						 GError		**err);

static GNode *		categories_copy		(void);
static gboolean		categories_copy_cb	(GNode		*node,
						 gpointer	data);
static gboolean		refresh_categories	(GNode		**categories,
						 GError		**err);
static void		genres_init		(Genres		*genres);
static void		genreslist_get_genres	(const char	*genreslist,
						 Genres		*genres);
static GNode		*genres_get_categories	(const Genres	*genres);
static void		genres_free		(Genres		*genres);

static gboolean		refresh_streams		(STCategory	*category,
						 GList		**streams,
						 GError		**err);
static void		refresh_streams_header_cb (const char	*line,
						   gpointer	data);
static void		refresh_streams_body_cb	(const char	*line,
						 gpointer	data);

static gboolean		search_url_cb		(STCategory	*category);

/*** implementation **********************************************************/

static Live365Stream *
stream_new_cb (gpointer data)
{
  Live365Stream *stream;

  stream = g_new0(Live365Stream, 1);
  stream->url_mutex = g_mutex_new();

  return stream;
}

static void
stream_field_get_cb (Live365Stream *stream,
		     STHandlerField *field,
		     GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case FIELD_TITLE:
      g_value_set_string(value, stream->title);
      break;

    case FIELD_GENRE:
      g_value_set_string(value, stream->genre);
      break;

    case FIELD_DESCRIPTION:
      g_value_set_string(value, stream->description);
      break;

    case FIELD_BROADCASTER:
      g_value_set_string(value, stream->broadcaster);
      break;

    case FIELD_AUDIO: 
      g_value_set_string(value, stream->audio);
      break;

    case FIELD_URL_POSTFIX:
      g_value_set_string(value, stream->url_postfix);
      break;

    case FIELD_HOMEPAGE:
      g_value_set_string(value, stream->homepage);
      break;

    case FIELD_URL:
      g_mutex_lock(stream->url_mutex);
      g_value_set_string(value, stream->url);
      g_mutex_unlock(stream->url_mutex);
      break;

    default:
      g_assert_not_reached();
    }
}

static void
stream_field_set_cb (Live365Stream *stream,
		     STHandlerField *field,
		     const GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case FIELD_TITLE:
      stream->title = g_value_dup_string(value);
      break;

    case FIELD_GENRE:
      stream->genre = g_value_dup_string(value);
      break;

    case FIELD_DESCRIPTION: 
      stream->description = g_value_dup_string(value);
      break;

    case FIELD_BROADCASTER:
      stream->broadcaster = g_value_dup_string(value);
      break;

    case FIELD_AUDIO: 
      stream->audio = g_value_dup_string(value);
      break;

    case FIELD_URL_POSTFIX:
      stream->url_postfix = g_value_dup_string(value);
      break;

    case FIELD_HOMEPAGE:
      stream->homepage = g_value_dup_string(value);
      break;

    case FIELD_URL:
      g_mutex_lock(stream->url_mutex);
      stream->url = g_value_dup_string(value);
      g_mutex_unlock(stream->url_mutex);
      break;

    default:
      g_assert_not_reached();
    }
}

static void
stream_stock_field_get_cb (Live365Stream *stream,
			   STHandlerStockField stock_field,
			   GValue *value,
			   gpointer data)
{
  switch (stock_field)
    {
    case ST_HANDLER_STOCK_FIELD_NAME:
      g_value_set_string(value, stream->title);
      break;

    case ST_HANDLER_STOCK_FIELD_GENRE:
      g_value_set_string(value, stream->genre);
      break;
    }
}

static void
stream_free_cb (Live365Stream *stream, gpointer data)
{
  g_free(stream->title);
  g_free(stream->genre);
  g_free(stream->description);
  g_free(stream->broadcaster);
  g_free(stream->audio);
  g_free(stream->url_postfix);
  g_free(stream->homepage);

  g_free(stream->url);
  g_mutex_free(stream->url_mutex);

  st_stream_free((STStream *) stream);
}

static gboolean
stream_resolve (Live365Stream *stream, GError **err)
{
  gboolean already_resolved;
  STTransferSession *session;
  char *url;
  gboolean status;
  char *station_id = NULL;

  g_return_val_if_fail(stream != NULL, FALSE);

  g_mutex_lock(stream->url_mutex);
  already_resolved = stream->url != NULL;
  g_mutex_unlock(stream->url_mutex);

  if (already_resolved)
    return TRUE;

  session = st_transfer_session_new();
  url = g_strconcat(LIVE365_ROOT, stream->url_postfix, NULL);
  status = st_transfer_session_get_by_line(session,
					   url,
					   0,
					   NULL,
					   NULL,
					   stream_resolve_line_cb,
					   &station_id,
					   err);
  g_free(url);
  st_transfer_session_free(session);

  if (status)
    {
      if (station_id)
	{
	  g_mutex_lock(stream->url_mutex);
	  stream->url = ident_user && ident_session
	    ? g_strdup_printf(LIVE365_ROOT "play/%s&session=%s&membername=%s", station_id, ident_session, ident_user)
	    : g_strconcat(LIVE365_ROOT, "play/", station_id, NULL);
	  g_mutex_unlock(stream->url_mutex);
	}
      else
	{
	  g_set_error(err, 0, 0, _("stream is empty"));
	  status = FALSE;
	}
    }
  g_free(station_id);

  return status;
}

static void
stream_resolve_line_cb (const char *line, gpointer data)
{
  char **station_id = data;

  if (! *station_id)
    st_re_parse(&re_stationid, line, station_id);
}

static gboolean
stream_resolve_cb (Live365Stream *stream, gpointer data, GError **err)
{
  return stream_resolve(stream, err);
}

static gboolean
stream_tune_in_cb (Live365Stream *stream,
		   gpointer data,
		   GError **err)
{
  gboolean status;

  if (! stream_resolve(stream, err))
    return FALSE;

  g_mutex_lock(stream->url_mutex);
  status = st_action_run("play-stream", stream->url, err);
  g_mutex_unlock(stream->url_mutex);

  return status;
}

static gboolean
stream_record_cb (Live365Stream *stream,
		  gpointer data,
		  GError **err)
{
  gboolean status;

  if (! stream_resolve(stream, err))
    return FALSE;

  g_mutex_lock(stream->url_mutex);
  status = st_action_run("record-stream", stream->url, err);
  g_mutex_unlock(stream->url_mutex);

  return status;
}

static gboolean
stream_browse_cb (Live365Stream *stream,
		  gpointer data,
		  GError **err)
{
  return st_action_run("view-web", stream->homepage, err);
}

static gboolean
refresh_cb (STCategory *category,
	    GNode **categories,
	    GList **streams,
	    gpointer data,
	    GError **err)
{
  if (! session_categories)
    {
      if (! refresh_categories(&session_categories, err))
	return FALSE;
    }
  
  *categories = categories_copy();

  if (st_is_aborted())
    return FALSE;

  return refresh_streams(category, streams, err);
}

static GNode *
categories_copy (void)
{
  GNode *node;

  node = g_node_copy(session_categories);
  g_node_traverse(node, G_IN_ORDER, G_TRAVERSE_ALL, -1, categories_copy_cb, NULL);

  return node;
}

static gboolean
categories_copy_cb (GNode *node, gpointer data)
{
  STCategory *category = node->data;

  if (category)
    {
      STCategory *copy;

      copy = st_category_new();
      copy->name = g_strdup(category->name);
      copy->label = g_strdup(category->label);
      copy->url_postfix = g_strdup(category->url_postfix);

      node->data = copy;
    }

  return FALSE;
}

static gboolean
refresh_categories (GNode **categories, GError **err)
{
  STTransferSession *session;
  gboolean status;
  char *url;
  char *genreslist;
  Genres genres;

  session = st_transfer_session_new();
  url = g_strconcat(LIVE365_ROOT, "scripts/genredata.js", NULL);
  status = st_transfer_session_get(session, url, 0, NULL, &genreslist, err);
  g_free(url);
  st_transfer_session_free(session);

  if (! status)
    return FALSE;

  genres_init(&genres);
  genreslist_get_genres(genreslist, &genres);
  g_free(genreslist);

  *categories = genres_get_categories(&genres);
  genres_free(&genres);
  
  return TRUE;
}

static void
genres_init (Genres *genres)
{
  genres->names = NULL;
  genres->labels = NULL;
}

static void
genreslist_get_genres (const char *genreslist, Genres *genres)
{
  char **lines;
  int i;
  
  enum
    {
      SECTION_NULL,
      SECTION_LABELS,
      SECTION_NAMES,
    } section = SECTION_NULL;
      
  lines = g_strsplit(genreslist, "\n", 0);

  for (i = 0; lines[i]; i++)
    {
      char *sub;

      if (st_re_match(&re_start_genre_list, lines[i]))
	section = SECTION_LABELS;
      else if (st_re_match(&re_start_genre_id_list, lines[i]))
	section = SECTION_NAMES;
      else if (st_re_match(&re_end_list, lines[i]))
	section = SECTION_NULL;
      else if (section && st_re_parse(&re_item, lines[i], &sub))
	{
	  if (section == SECTION_LABELS)
	    genres->labels = g_slist_append(genres->labels, sub);
	  else
	    genres->names = g_slist_append(genres->names, sub);
	}
    }
  
  g_strfreev(lines);
}

static GNode *
genres_get_categories (const Genres *genres)
{
  GNode *categories;
  STCategory *category;
  GSList *labels_iter, *names_iter;

  categories = g_node_new(NULL);

  labels_iter = genres->labels;
  names_iter = genres->names;

  while (labels_iter && names_iter)
    {
      category = st_category_new();

      category->name = names_iter->data;
      category->label = labels_iter->data;
      category->url_postfix = g_strconcat(DIRECTORY_PREFIX, category->name,
					  NULL);

      g_node_append_data(categories, category);

      labels_iter = g_slist_next(labels_iter);
      names_iter = g_slist_next(names_iter);
    }
  
  return categories;
}

static void
genres_free (Genres *genres)
{
  g_slist_free(genres->labels);
  g_slist_free(genres->names);
}

static gboolean
refresh_streams (STCategory *category,
		 GList **streams,
		 GError **err)
{
  STTransferSession *session;
  gboolean status;
  char *url;
  RefreshStreamsInfo info;

  *streams = NULL;

  info.charset = NULL;
  info.streams = streams;
  info.stream = NULL;

  session = st_transfer_session_new();
  url = g_strconcat(LIVE365_ROOT, category->url_postfix, NULL);
  status = st_transfer_session_get_by_line(session,
					   url,
					   0,
					   refresh_streams_header_cb,
					   &info,
					   refresh_streams_body_cb,
					   &info,
					   err);
  g_free(url);
  st_transfer_session_free(session);

  g_free(info.charset);

  if (info.stream)
    {
      stream_free_cb(info.stream, NULL);
      if (status) /* only display warning if the transfer was otherwise correct */
	st_notice(_("Live365:EOF: found unterminated stream"));
    }

  return status;
}

static void
refresh_streams_header_cb (const char *line, gpointer data)
{
  RefreshStreamsInfo *info = data;

  if (! info->charset)
    st_re_parse(&re_header_charset, line, &info->charset);
}

static void
refresh_streams_body_cb (const char *line, gpointer data)
{
  RefreshStreamsInfo *info = data;
  char *sub1, *sub2, *sub3;
  char *converted = NULL;
  char *expanded;

  if (! info->charset && ! strncasecmp(line, "</head>", 6))
    info->charset = g_strdup("ISO8859-1"); /* we got no charset info, use a fallback */
  else
    {
      char *charset;

      if (st_re_parse(&re_body_charset, line, &charset))
	{			/* body charset overrides header charset */
	  g_free(info->charset);
	  info->charset = charset;
	}
    }
  
  if (info->charset)
    {
      GError *convert_err = NULL;
      
      if ((converted = g_convert(line, strlen(line), "UTF-8", info->charset,
				 NULL, NULL, &convert_err)))
	line = converted;
      else
	{
	  st_notice(_("Live365: unable to convert line to UTF-8: %s"),
		    convert_err->message);
	  
	  g_error_free(convert_err);
	  return;
	}
    }

  expanded = st_sgml_ref_expand(line);
  line = expanded;

  if (st_re_parse(&re_description, line, &sub1))
    {
      if (info->stream)
	{
	  info->stream->description = sub1;

	  if (info->stream->genre
	      && info->stream->description
	      && info->stream->audio)
	    {
	      ((STStream *) info->stream)->name =
		g_strconcat(info->stream->genre, info->stream->description,
			    info->stream->audio, NULL);
	  
	      *(info->streams) = g_list_append(*(info->streams), info->stream);
	    }
	  else
	    {
	      st_notice(_("Live365: found incomplete stream"));
	      stream_free_cb(info->stream, NULL);
	    }

	  info->stream = NULL;
	}
      else
	{
	  st_notice(_("Live365: found misplaced description"));
	  g_free(sub1);
	}
    }
  else if (st_re_parse(&re_stream, line, &sub1))
    {
      if (info->stream)	/* a malformed stream remains, free it */
	{
	  st_notice(_("Live365: found unterminated stream"));
	  stream_free_cb(info->stream, NULL);
	}

      info->stream = stream_new_cb(NULL);
      info->stream->url_postfix =
	g_strdup_printf("cgi-bin/mini.cgi?stream=%s&bitratebypass=1", sub1);

      g_free(sub1);
    }
  else if (st_re_parse(&re_title, line, &sub1, &sub2))
    {
      if (info->stream)
	{
	  info->stream->homepage = sub1;
	  info->stream->title = sub2;
	}
      else
	{
	  st_notice(_("Live365: found misplaced homepage and title"));
	  g_free(sub1);
	  g_free(sub2);
	}
    }
  else if (st_re_parse(&re_broadcaster, line, &sub1))
    {
      if (info->stream)
	info->stream->broadcaster = sub1;
      else
	{
	  st_notice(_("Live365: found misplaced broadcaster"));
	  g_free(sub1);
	}
    }
  else if (st_re_parse(&re_genre, line, &sub1))
    {
      if (info->stream)
	info->stream->genre = sub1;
      else
	{
	  st_notice(_("Live365: found misplaced genre"));
	  g_free(sub1);
	}
    }
  else if (st_re_parse(&re_audio, line, &sub1, &sub2, &sub3))
    {
      if (info->stream)
	{
	  GString *string;

	  string = g_string_new(NULL);
	  g_string_append_printf(string, "%s %s", sub2, sub1);

	  if (*sub3)
	    g_string_append(string, ", MP3Pro");

	  info->stream->audio = g_string_free(string, FALSE);

	  g_free(sub1);
	  g_free(sub2);
	  g_free(sub3);
	}
      else
	{
	  st_notice(_("Live365: found misplaced audio"));
	  g_free(sub1);
	}
    }

  g_free(converted);
  g_free(expanded);
}

static gboolean
search_url_cb (STCategory *category)
{
  char *str;

  str = st_search_dialog();
  if (str)
    {
      char *escaped;

      g_free(category->label);
      category->label = g_strdup_printf(_("Search results for \"%s\""), str);

      escaped = st_transfer_escape(str);
      g_free(str);

      g_free(category->url_postfix);
      category->url_postfix = g_strconcat(DIRECTORY_PREFIX,
					  "search&searchdesc=",
					  escaped,
					  NULL);
      g_free(escaped);

      return TRUE;
    }
  else
    return FALSE;
}

static gboolean
init_re (void)
{
  int status;

  status = regcomp(&re_header_charset, "^Content-Type: .*charset=(.*)", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_body_charset, "<meta http-equiv=.* content=.*charset=(.*)\"", REG_EXTENDED | REG_ICASE);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_title, "<a class='title-enhanced-link' href='(.*)'>(.*)</a>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_broadcaster, "<a class=\"handle-link\" href=\".*\" alt=\".*\" TARGET=\"?.*\"?>(.*)</a>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_genre, "^<TD  CLASS=\"genre\" >(.*)</TD>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_audio, "^<TD  CLASS=\"connection\" WIDTH=\"[0-9]+\" >(.*)<br>([0-9k]+)(<img src='/images/mp3pro.*>)?</TD>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_description, "<a class='desc-link' href='.*'>([^<]*)", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_stream, "href=\\\\'javascript:Launch\\(([0-9]+),", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_start_genre_list, "^// START GENRE LIST", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_start_genre_id_list, "^// START GENRE ID LIST", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_end_list, "^// END .* LIST", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_item, "\"(.*)\",", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_stationid, "^var stationID.*= \"(.*)\";$", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);

  return TRUE;
}

static void
init_handler (void)
{
  STHandler *handler;

  GNode *stock_categories;
  STCategory *category;

  handler = st_handler_new("live365");

  st_handler_set_label(handler, "Live365");
  st_handler_set_copyright(handler, COPYRIGHT);
  st_handler_set_description(handler, "Live365 Internet Radio");
  st_handler_set_home(handler, LIVE365_ROOT);

  stock_categories = g_node_new(NULL);

  category = st_category_new();
  category->name = "__main";
  category->label = _("Editor's Picks");
  category->url_postfix = DIRECTORY_PREFIX "ESP";
  
  g_node_append_data(stock_categories, category);

  category = st_category_new();
  category->name = "__search";
  category->label = g_strdup(_("Search"));
  category->url_cb = search_url_cb;
  
  g_node_append_data(stock_categories, category);
  
  st_handler_set_icon_from_inline(handler, sizeof(art_icon), art_icon);
  st_handler_set_stock_categories(handler, stock_categories);

  st_handler_bind(handler, ST_HANDLER_EVENT_REFRESH, refresh_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_NEW, stream_new_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_GET, stream_field_get_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_SET, stream_field_set_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_STOCK_FIELD_GET, stream_stock_field_get_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FREE, stream_free_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_RESOLVE, stream_resolve_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_TUNE_IN, stream_tune_in_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_RECORD, stream_record_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_BROWSE, stream_browse_cb, NULL);

  st_handler_add_field(handler, st_handler_field_new(FIELD_TITLE,
						     _("Title"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_GENRE,
						     _("Genre"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_DESCRIPTION,
						     _("Description"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_BROADCASTER,
						     _("Broadcaster"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_AUDIO,
						     _("Audio"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_URL_POSTFIX,
						     _("URL postfix"),
						     G_TYPE_STRING,
						     0));
  st_handler_add_field(handler, st_handler_field_new(FIELD_HOMEPAGE,
						     _("Homepage"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE
						     | ST_HANDLER_FIELD_START_HIDDEN));
  st_handler_add_field(handler, st_handler_field_new(FIELD_URL,
						     _("URL"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE
						     | ST_HANDLER_FIELD_START_HIDDEN));

  st_handlers_add(handler);
}

gboolean
plugin_init (GError **err)
{
  gboolean status;

  bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");

  if (! st_check_api_version(5, 5))
    {
      g_set_error(err, 0, 0, _("API version mismatch"));
      return FALSE;
    }

  ident_user = g_getenv("STREAMTUNER_LIVE365_USER");
  ident_session = g_getenv("STREAMTUNER_LIVE365_SESSION");

  if (ident_user && ! ident_session)
    {
      g_set_error(err, 0, 0, _("STREAMTUNER_LIVE365_USER is set but STREAMTUNER_LIVE365_SESSION isn't"));
      return FALSE;
    }
  else if (ident_session && ! ident_user)
    {
      g_set_error(err, 0, 0, _("STREAMTUNER_LIVE365_SESSION is set but STREAMTUNER_LIVE365_USER isn't"));
      return FALSE;
    }

  status = init_re();
  g_return_val_if_fail(status == TRUE, FALSE);

  init_handler();

  st_action_register("record-stream", _("Record a stream"), "xterm -hold -e streamripper %q");
  st_action_register("view-web", _("Open a web page"), "epiphany %q");
  st_action_register("play-stream", _("Listen to a stream"), "xmms %q");

  return TRUE;
}
