/*
 * $Id: pg.c,v 1.17 2004/01/27 13:52:34 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <Python.h>
#include <glib.h>
#include <glib-object.h>
#include "gettext.h"

static PyObject *pg_node_new (PyObject *dummy, PyObject *args);
static PyObject *pg_node_append (PyObject *dummy, PyObject *args);

static PyMethodDef pg_methods[] = {
  { "g_node_new", pg_node_new, METH_VARARGS, "g_node_new()", },
  { "g_node_append", pg_node_append, METH_VARARGS, "g_node_append()", },

  { NULL, NULL, 0, NULL },
};

void
pg_init (void)
{
  PyObject *module;

  module = Py_InitModule("glib", pg_methods);

  PyModule_AddIntConstant(module, "G_TYPE_BOOLEAN", G_TYPE_BOOLEAN);
  PyModule_AddIntConstant(module, "G_TYPE_INT", G_TYPE_INT);
  PyModule_AddIntConstant(module, "G_TYPE_STRING", G_TYPE_STRING);
}

/*** exported C API **********************************************************/

PyObject *
PGNode_New (PyObject *data)
{
  PyObject *pargs;
  PyObject *pnode;

  pargs = Py_BuildValue("(O)", data);
  pnode = pg_node_new(Py_None, pargs);
  Py_DECREF(pargs);

  return pnode;
}

GList *
PyList_AsGList (PyObject *list)
{
  GList *g_list = NULL;
  int i;
  int len;

  len = PyList_Size(list);

  for (i = 0; i < len; i++)
    {
      PyObject *item;
      gpointer ptr;
      
      item = PyList_GetItem(list, i);
      ptr = PyCObject_AsVoidPtr(item);

      g_list = g_list_append(g_list, ptr);
    }
  
  return g_list;
}

/*
 * Converts a tuple or list of Python strings to a GSList of
 * newly-allocated C strings.
 *
 * Returns true if succeeds, false otherwise (in such case, a Python
 * exception will be raised).
 */
gboolean
PyStrings_AsGSList (PyObject *strings, GSList **g_slist)
{
  int i;
  int len;
  gboolean is_tuple;		/* FALSE = list */

  g_return_val_if_fail(g_slist != NULL, FALSE);

  if (PyTuple_Check(strings))
    is_tuple = TRUE;
  else if (PyList_Check(strings))
    is_tuple = FALSE;
  else
    {
      PyErr_SetString(PyExc_TypeError, _("expected tuple or list object"));
      return FALSE;
    }
  
  *g_slist = NULL;
  len = is_tuple ? PyTuple_Size(strings) : PyList_Size(strings);

  for (i = 0; i < len; i++)
    {
      PyObject *item;
      const char *str;
      
      item = is_tuple ? PyTuple_GetItem(strings, i) : PyList_GetItem(strings, i);
      str = PyString_AsString(item);

      if (! str)
	{			/* exception raised by PyString_AsString */
	  g_slist_free(*g_slist);
	  return FALSE;
	}

      *g_slist = g_slist_append(*g_slist, g_strdup(str));
    }
  
  return TRUE;
}

/*
 * Converts a GSList of C strings to a tuple of Python strings.
 */
PyObject *
PyTuple_FromGSList (GSList *g_slist)
{
  PyObject *tuple;
  GSList *l;
  int i = 0;

  tuple = PyTuple_New(g_slist_length(g_slist));
  for (l = g_slist; l; l = l->next)
    {
      PyObject *str;

      str = PyString_FromString(l->data);
      PyTuple_SetItem(tuple, i++, str);
    }

  return tuple;
}

/*** gnode.h *****************************************************************/

static PyObject *
pg_node_new (PyObject *dummy, PyObject *args)
{
  PyObject *pdata;

  PyObject *pnode;
  gpointer data = NULL;
  GNode *node;

  if (! PyArg_ParseTuple(args, "O", &pdata))
    return NULL;
  
  if (pdata != Py_None)
    data = PyCObject_AsVoidPtr(pdata);
  
  node = g_node_new(data);

  pnode = PyCObject_FromVoidPtr(node, NULL);

  return pnode;
}

static PyObject *
pg_node_append (PyObject *dummy, PyObject *args)
{
  PyObject *pparent;
  PyObject *pnode;

  GNode *parent;
  GNode *node;
  
  if (! PyArg_ParseTuple(args, "OO", &pparent, &pnode))
    return NULL;
  
  parent = PyCObject_AsVoidPtr(pparent);
  node = PyCObject_AsVoidPtr(pnode);

  g_node_append(parent, node);

  Py_INCREF(Py_None);
  return Py_None;
}
