/*
 * $Id: spy.c,v 1.5 2004/01/27 13:52:34 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <errno.h>
#include <Python.h>
#include <glib.h>
#include "gettext.h"
#include "pg.h"
#include "pst.h"

PyThreadState *spy_main_thread_state;
static PyObject *spy_globals;

static void spy_sys_path_append (const char *dir);

void
spy_init (void)
{
  int status;
  PyObject *m;

  Py_Initialize();
  PyEval_InitThreads();

  spy_main_thread_state = PyThreadState_Get();
  PyEval_ReleaseLock();

  spy_sys_path_append(MODULES_DIR);

  m = PyImport_AddModule("__main__");
  g_return_if_fail(m != NULL);

  status = PyModule_AddStringConstant(m, "GETTEXT_PACKAGE", GETTEXT_PACKAGE);
  g_return_if_fail(status == 0);

  status = PyModule_AddStringConstant(m, "LOCALEDIR", LOCALEDIR);
  g_return_if_fail(status == 0);

  spy_globals = PyModule_GetDict(m);

  pg_init();
  pst_init();
}

static void
spy_sys_path_append (const char *dir)
{
  PyObject *module;
  PyObject *dict;
  PyObject *path;
  PyObject *pdir;

  module = PyImport_ImportModule("sys");
  g_return_if_fail(module != NULL);

  dict = PyModule_GetDict(module);
  path = PyDict_GetItemString(dict, "path");
  g_return_if_fail(path != NULL);

  pdir = PyString_FromString(dir);
  PyList_Append(path, pdir);	/* may return -1 */
  Py_DECREF(pdir);

  Py_DECREF(module);
}

gboolean
spy_load_script (const char *filename, GError **err)
{
  FILE *file;
  gboolean status;

  if (! (file = fopen(filename, "r")))
    {
      g_set_error(err, 0, 0, _("unable to open script for reading: %s"),
		  g_strerror(errno));
      return FALSE;
    }
  
  status = PyRun_File(file, g_strdup(filename), Py_file_input,
		      spy_globals, spy_globals) != NULL;

  fclose(file);

  if (! status)
    {
      PyErr_Print();
      g_set_error(err, 0, 0, _("Python exception occurred"));
    }

  return status;
}
