/*
 * $Id: spy-main.c,v 1.8 2004/03/28 22:34:29 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <string.h>
#include <Python.h>
#include <streamtuner.h>
#include "gettext.h"
#include "pst.h"
#include "pst-helpers.h"

/*** variable declarations ***************************************************/

const char *spy_main_current_script;
static GHashTable *main_thread_states;

/*** function declarations ***************************************************/

static void spy_main_init_python (void);

static void spy_main_load_all_scripts (void);
static gboolean spy_main_load_scripts (const char *dirname, GError **err);

static PyObject *spy_main_interpreter_new (const char *script_name);
static gboolean spy_main_load_script (const char *filename, GError **err);

/*** implementation **********************************************************/

gboolean
plugin_init (GError **err)
{
  bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");

  if (! st_check_api_version(5, 5))
    {
      g_set_error(err, 0, 0, _("API version mismatch"));
      return FALSE;
    }
  
  spy_main_init_python();
  spy_main_load_all_scripts();

  return TRUE;
}

static void
spy_main_init_python (void)
{
  Py_Initialize();
  PyEval_InitThreads();
  PyEval_ReleaseLock();

  main_thread_states = g_hash_table_new(g_str_hash, g_str_equal);
}

static void
spy_main_load_all_scripts (void)
{
  GError *tmp_err = NULL;
  char *private_scriptsdir;

  if (g_file_test(SCRIPTS_DIR, G_FILE_TEST_IS_DIR))
    {
      if (! spy_main_load_scripts(SCRIPTS_DIR, &tmp_err))
	{
	  char *secondary;
	  char *normalized;

	  secondary = g_strdup_printf(_("Unable to scan system Python scripts directory: %s"), tmp_err->message);
	  g_clear_error(&tmp_err);

	  normalized = st_dialog_normalize(secondary);
	  g_free(secondary);
	  
	  st_error_dialog(_("A script error has occurred."), "%s", normalized);
	  g_free(normalized);
	}
    }

  private_scriptsdir = g_build_filename(st_settings_get_private_dir(),
					"streamtuner-python", "scripts", NULL);
  if (g_file_test(private_scriptsdir, G_FILE_TEST_IS_DIR))
    {
      if (! spy_main_load_scripts(private_scriptsdir, &tmp_err))
	{
	  char *secondary;
	  char *normalized;

	  secondary = g_strdup_printf(_("Unable to scan user Python scripts directory: %s"), tmp_err->message);
	  g_clear_error(&tmp_err);

	  normalized = st_dialog_normalize(secondary);
	  g_free(secondary);
	  
	  st_error_dialog(_("A script error has occurred."), "%s", normalized);
	  g_free(normalized);
	}
    }
  g_free(private_scriptsdir);
}

static gboolean
spy_main_load_scripts (const char *dirname, GError **err)
{
  GDir *dir;
  const char *filename;

  g_return_val_if_fail(dirname != NULL, FALSE);

  if (! (dir = g_dir_open(dirname, 0, err)))
    return FALSE;

  while ((filename = g_dir_read_name(dir)))
    {
      GError *load_err = NULL;
      char *pathname;
      char *extension;

      pathname = g_build_filename(dirname, filename, NULL);

      if (g_file_test(pathname, G_FILE_TEST_IS_REGULAR)
	  && (extension = strrchr(filename, '.'))
	  && (! strcmp(++extension, "py")))
	{
	  if (! spy_main_load_script(pathname, &load_err))
	    {
	      char *secondary;
	      char *normalized;

	      secondary = g_strdup_printf(_("Script <i>%s</i> couldn't be loaded: %s"), pathname, load_err->message);
	      g_clear_error(&load_err);

	      normalized = st_dialog_normalize(secondary);
	      g_free(secondary);

	      st_error_dialog(_("A script error has occurred."), "%s", normalized);
	      g_free(normalized);
	    }
	}
      
      g_free(pathname);
    }

  g_dir_close(dir);
  
  return TRUE;
}

static PyObject *
spy_main_interpreter_new (const char *script_name)
{
  PyThreadState *state;
  PyObject *module;

  state = Py_NewInterpreter();
  g_return_val_if_fail(state != NULL, NULL);
  g_hash_table_insert(main_thread_states, g_strdup(script_name), state);

  module = PyImport_AddModule("__main__");
  if (! module)
    return FALSE;

  if (PyModule_AddStringConstant(module, "GETTEXT_PACKAGE", GETTEXT_PACKAGE) == -1)
    return FALSE;
  if (PyModule_AddStringConstant(module, "LOCALEDIR", LOCALEDIR) == -1)
    return FALSE;
  if (! pst_init())
    return FALSE;

  return PyModule_GetDict(module);
}

static gboolean
spy_main_load_script (const char *filename, GError **err)
{
  FILE *file;
  PyObject *globals;
  gboolean status = FALSE;

  if (! (file = fopen(filename, "r")))
    {
      g_set_error(err, 0, 0, _("unable to open script for reading: %s"), g_strerror(errno));
      return FALSE;
    }
  
  globals = spy_main_interpreter_new(filename);
  if (globals)
    {
      spy_main_current_script = filename;
      status = PyRun_File(file, filename, Py_file_input, globals, globals) != NULL;
      spy_main_current_script = NULL;
    }

  fclose(file);

  if (! status)
    pst_set_error(err);

  return status;
}

gpointer
spy_main_thread_begin_cb (gpointer data)
{
  const char *script_name = data;
  PyThreadState *main_state;
  PyThreadState *state;

  main_state = g_hash_table_lookup(main_thread_states, script_name);
  g_return_val_if_fail(main_state != NULL, NULL);

  PyEval_AcquireLock();

  state = PyThreadState_New(main_state->interp);
  PyThreadState_Swap(state);

  return state;
}

void
spy_main_thread_end_cb (gpointer thread_data, gpointer data)
{
  const char *script_name = data;
  PyThreadState *main_state;
  PyThreadState *state = thread_data;

  main_state = g_hash_table_lookup(main_thread_states, script_name);
  g_return_if_fail(main_state != NULL);

  PyThreadState_Swap(main_state);

  PyThreadState_Clear(state);
  PyThreadState_Delete(state);

  PyEval_ReleaseLock();
}
