/*
 * $Id: pst-handler.c,v 1.26 2004/03/29 15:19:13 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <Python.h>
#include <streamtuner.h>
#include "gettext.h"
#include "pst-category.h"
#include "pst-category-node.h"
#include "pst-handler.h"
#include "pst-handler-field.h"
#include "pst-stream.h"
#include "pst-helpers.h"
#include "pst-transfer-session.h"
#include "spy-main.h"

/*** attributes **************************************************************/

static int pst_handler_set_label (PSTHandler *self, PyObject *value, void *closure);
static int pst_handler_set_copyright (PSTHandler *self, PyObject *value, void *closure);
static int pst_handler_set_description (PSTHandler *self, PyObject *value, void *closure);
static int pst_handler_set_home (PSTHandler *self, PyObject *value, void *closure);
static int pst_handler_set_icon (PSTHandler *self, PyObject *value, void *closure);
static int pst_handler_set_stock_categories (PSTHandler *self, PyObject *value, void *closure);
static int pst_handler_set_flags (PSTHandler *self, PyObject *value, void *closure);

static PyGetSetDef getsetters[] = {
  { "label", NULL, (setter) pst_handler_set_label },
  { "copyright", NULL, (setter) pst_handler_set_copyright },
  { "description", NULL, (setter) pst_handler_set_description },
  { "home", NULL, (setter) pst_handler_set_home },
  { "icon", NULL, (setter) pst_handler_set_icon },
  { "stock_categories", NULL, (setter) pst_handler_set_stock_categories },
  { "flags", NULL, (setter) pst_handler_set_flags },
  
  { NULL }
};

/*** methods *****************************************************************/

static PyObject *pst_handler_add_field (PSTHandler *self, PyObject *args);

static PyMethodDef methods[] = {
  { "add_field", (PyCFunction) pst_handler_add_field, METH_VARARGS },
  
  { NULL }
};

/*** type object *************************************************************/

static PyObject *pst_handler_new (PyTypeObject *type, PyObject *args, PyObject *keywords);

PyTypeObject PSTHandler_Type = {
  PyObject_HEAD_INIT(NULL)
  0,				/* ob_size */
  "ST.Handler",			/* tp_name */
  sizeof(PSTHandler),		/* tp_basicsize */
  0,				/* tp_itemsize */
  NULL,				/* tp_dealloc */
  NULL,				/* tp_print */
  NULL,				/* tp_getattr */
  NULL,				/* tp_setattr */
  NULL,				/* tp_compare */
  NULL,				/* tp_repr */
  NULL,				/* tp_as_number */
  NULL,				/* tp_as_sequence */
  NULL,				/* tp_as_mapping */
  NULL,				/* tp_hash */
  NULL,				/* tp_call */
  NULL,				/* tp_str */
  NULL,				/* tp_getattro */
  NULL,				/* tp_setattro */
  NULL,				/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
  NULL,				/* tp_doc */
  NULL,				/* tp_traverse */
  NULL,				/* tp_clear */
  NULL,				/* tp_richcompare */
  0,				/* tp_weaklistoffset */
  NULL,				/* tp_iter */
  NULL,				/* tp_iternext */
  methods,			/* tp_methods */
  NULL,				/* tp_members */
  getsetters,			/* tp_getset */
  NULL,				/* tp_base */
  NULL,				/* tp_dict */
  NULL,				/* tp_descr_get */
  NULL,				/* tp_descr_set */
  0,				/* tp_dictoffset */
  NULL,				/* tp_init */
  NULL,				/* tp_alloc */
  pst_handler_new		/* tp_new */
};

/*** function declarations ***************************************************/

static void pst_handler_bind_events (PSTHandler *self);
static int pst_handler_string_setter (void (*func) (STHandler *, const char *),
				      STHandler *handler,
				      PyObject *value);

static gboolean pst_handler_refresh_cb (PythonCategory *category,
					GNode **categories,
					GList **streams,
					PSTCallbackInfo *info,
					GError **err);
static gboolean pst_handler_refresh_multiple_cb (GNode **categories,
						 GHashTable **streams,
						 PSTCallbackInfo *info,
						 GError **err);

static gboolean pst_handler_refresh_split_result (PyObject *result,
						  PyObject **item1,
						  PyObject **item2,
						  GError **err);

/*** type methods ************************************************************/

static PyObject *
pst_handler_new (PyTypeObject *type, PyObject *args, PyObject *keywords)
{
  PSTHandler *self;
  static char *keyword_list[] = { "name", NULL };
  const char *name;
      
  if (! PyArg_ParseTupleAndKeywords(args, keywords, "s", keyword_list,
				    &name))
    return NULL;

  self = (PSTHandler *) type->tp_alloc(type, 0);
  self->handler = st_handler_new(name);

  pst_handler_bind_events(self);
  
  return (PyObject *) self;
}

static void
pst_handler_bind_events (PSTHandler *self)
{
  char *script;
  static const struct
  {
    STHandlerEvent	event;
    gpointer		cb;
  } stock_events[] = {
    { ST_HANDLER_EVENT_CATEGORY_NEW, pst_category_new_cb },
    { ST_HANDLER_EVENT_CATEGORY_FREE, pst_category_free_cb },
    { ST_HANDLER_EVENT_STREAM_NEW, pst_stream_new_cb },
    { ST_HANDLER_EVENT_STREAM_FIELD_SET, pst_stream_field_set_cb },
    { ST_HANDLER_EVENT_STREAM_FIELD_GET, pst_stream_field_get_cb },
    { ST_HANDLER_EVENT_STREAM_FREE, pst_stream_free_cb },
    { ST_HANDLER_EVENT_THREAD_BEGIN, spy_main_thread_begin_cb },
    { ST_HANDLER_EVENT_THREAD_END, spy_main_thread_end_cb }
  };
  static const struct
  {
    STHandlerEvent	event;
    const char		*method_name;
    gpointer		cb;
  } custom_events[] = {
    { ST_HANDLER_EVENT_REFRESH, "refresh", pst_handler_refresh_cb },
    { ST_HANDLER_EVENT_REFRESH_MULTIPLE, "refresh_multiple", pst_handler_refresh_multiple_cb },
    { ST_HANDLER_EVENT_STREAM_RESOLVE, "stream_resolve", pst_stream_cb },
    { ST_HANDLER_EVENT_STREAM_TUNE_IN, "stream_tune_in", pst_stream_cb },
    { ST_HANDLER_EVENT_STREAM_RECORD, "stream_record", pst_stream_cb },
    { ST_HANDLER_EVENT_STREAM_BROWSE, "stream_browse", pst_stream_cb },
    { ST_HANDLER_EVENT_STREAM_DELETE, "stream_delete", pst_stream_cb },
    { ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE, "stream_tune_in_multiple", pst_stream_tune_in_multiple_cb },
    { ST_HANDLER_EVENT_STREAM_STOCK_FIELD_GET, "stream_get_stock_field", pst_stream_stock_field_get_cb },
    { ST_HANDLER_EVENT_STREAM_MODIFY, "stream_modify", pst_stream_modify_cb }
  };
  int i;

  /* stock events */

  script = g_strdup(spy_main_current_script);
  for (i = 0; i < G_N_ELEMENTS(stock_events); i++)
    st_handler_bind(self->handler, stock_events[i].event, stock_events[i].cb, script);

  /* custom events */

  for (i = 0; i < G_N_ELEMENTS(custom_events); i++)
    {
      PyObject *pname;
      PyObject *pattr;

      pname = PyString_FromString(custom_events[i].method_name);
      pattr = PyObject_HasAttr((PyObject *) self, pname)
	? PyObject_GetAttr((PyObject *) self, pname)
	: NULL;
      Py_DECREF(pname);

      if (pattr)
	{
	  if (PyMethod_Check(pattr))
	    {
	      PSTCallbackInfo *info = g_new(PSTCallbackInfo, 1);

	      Py_INCREF(self);
	      info->object = (PyObject *) self;
	      info->method = (char *) custom_events[i].method_name;

	      st_handler_bind(self->handler,
			      custom_events[i].event,
			      custom_events[i].cb,
			      info);
	    }
	  Py_DECREF(pattr);
	}
    }
}

static int
pst_handler_string_setter (void (*func) (STHandler *, const char *),
			   STHandler *handler,
			   PyObject *value)
{
  if (value)
    {
      const char *str;

      str = PyString_AsString(value);
      if (! str)
	return -1;

      func(handler, str);
    }
  else
    func(handler, NULL);

  return 0;
}

static int
pst_handler_set_label (PSTHandler *self, PyObject *value, void *closure)
{
  return pst_handler_string_setter(st_handler_set_label, self->handler, value);
}

static int
pst_handler_set_copyright (PSTHandler *self, PyObject *value, void *closure)
{
  return pst_handler_string_setter(st_handler_set_copyright, self->handler, value);
}

static int
pst_handler_set_description (PSTHandler *self, PyObject *value, void *closure)
{
  return pst_handler_string_setter(st_handler_set_description, self->handler, value);
}

static int
pst_handler_set_home (PSTHandler *self, PyObject *value, void *closure)
{
  return pst_handler_string_setter(st_handler_set_home, self->handler, value);
}

static int
pst_handler_set_icon (PSTHandler *self, PyObject *value, void *closure)
{
  const char *filename;
  GError *err = NULL;

  if (! value)
    {
      PyErr_SetString(PyExc_TypeError, _("cannot unset icon"));
      return -1;
    }
  
  filename = PyString_AsString(value);
  if (! filename)
    return -1;

  if (! st_handler_set_icon_from_file(self->handler, filename, &err))
    {
      PyErr_SetString(PyExc_RuntimeError, err->message);
      g_error_free(err);

      return -1;
    }

  return 0;
}

static int
pst_handler_set_stock_categories (PSTHandler *self, PyObject *value, void *closure)
{
  if (value)
    {
      if (! PyObject_IsInstance(value, (PyObject *) &PSTCategoryNode_Type))
	{
	  PyErr_SetString(PyExc_TypeError, _("stock categories must be a ST.CategoryNode object"));
	  return -1;
	}

      st_handler_set_stock_categories(self->handler, PSTCategoryNode_AsNode((PSTCategoryNode *) value));
    }
  else
    st_handler_set_stock_categories(self->handler, NULL);

  return 0;
}

static int
pst_handler_set_flags (PSTHandler *self, PyObject *value, void *closure)
{
  if (value)
    {
      if (! PyInt_Check(value))
	{
	  PyErr_SetString(PyExc_TypeError, _("flags must be an integer"));
	  return -1;
	}
  
      st_handler_set_flags(self->handler, PyInt_AsLong(value));
    }
  else
    st_handler_set_flags(self->handler, 0);

  return 0;
}

/*** object methods **********************************************************/

static PyObject *
pst_handler_add_field (PSTHandler *self, PyObject *args)
{
  PSTHandlerField *pfield;

  if (! PyArg_ParseTuple(args, "O!", &PSTHandlerField_Type, &pfield))
    return NULL;

  st_handler_add_field(self->handler, pfield->field);

  return pst_none();
}

/*** streamtuner callbacks ***************************************************/

static gboolean
pst_handler_refresh_cb (PythonCategory *category,
			GNode **categories,
			GList **streams,
			PSTCallbackInfo *info,
			GError **err)
{
  PyObject *result;
  PyObject *pcategories;
  PyObject *pstreams;
  GError *tmp_err = NULL;
  gboolean status = FALSE;

  /* might have been aborted while we were waiting for the GIL */
  if (st_is_aborted())
    return FALSE;

  result = PyObject_CallMethod(info->object, info->method, "O", category->p);
  if (pst_handler_refresh_split_result(result, &pcategories, &pstreams, err))
    {
      if (PySequence_Check(pstreams))
	{
	  if (pst_streams_as_glist(pstreams, streams, &tmp_err))
	    {
	      *categories = PSTCategoryNode_AsNode((PSTCategoryNode *) pcategories);
	      status = TRUE;
	    }
	  else
	    {
	      g_set_error(err, 0, 0, _("in second item of returned sequence: %s"), tmp_err->message);
	      g_error_free(tmp_err);
	    }
	}
      else
	g_set_error(err, 0, 0, _("second item of returned sequence is not a sequence"));
    }
  Py_XDECREF(result);		/* may be null */

  return status;
}

static gboolean
pst_handler_refresh_multiple_cb (GNode **categories,
				 GHashTable **streams,
				 PSTCallbackInfo *info,
				 GError **err)
{
  PyObject *result;
  PyObject *pcategories;
  PyObject *pstreams;
  GError *tmp_err = NULL;
  gboolean status = FALSE;

  /* might have been aborted while we were waiting for the GIL */
  if (st_is_aborted())
    return FALSE;
  
  result = PyObject_CallMethod(info->object, info->method, NULL);
  if (pst_handler_refresh_split_result(result, &pcategories, &pstreams, err))
    {
      if (PyMapping_Check(pstreams))
	{
	  *streams = pst_streams_mapping_as_ghashtable(pstreams, &tmp_err);
	  if (*streams)
	    {
	      *categories = PSTCategoryNode_AsNode((PSTCategoryNode *) pcategories);
	      status = TRUE;
	    }
	  else
	    {
	      g_set_error(err, 0, 0, _("in second item of returned sequence: %s"), tmp_err->message);
	      g_error_free(tmp_err);
	    }
	}
      else
	g_set_error(err, 0, 0, _("second item of returned sequence is not a mapping"));
    }
  Py_XDECREF(result);		/* may be null */

  return status;
}

static gboolean
pst_handler_refresh_split_result (PyObject *result,
				  PyObject **item1,
				  PyObject **item2,
				  GError **err)
{
  g_return_val_if_fail(item1 != NULL, FALSE);
  g_return_val_if_fail(item2 != NULL, FALSE);

  if (! result)
    {
      if (! PyErr_ExceptionMatches(PyExc_AbortError))
	pst_set_error(err);
      return FALSE;
    }

  if (! PySequence_Check(result))
    {
      g_set_error(err, 0, 0, _("returned value is not a sequence"));
      return FALSE;
    }
  if (PySequence_Length(result) != 2)
    {
      g_set_error(err, 0, 0, _("returned sequence length is not 2"));
      return FALSE;
    }

  *item1 = PySequence_GetItem(result, 0);
  g_return_val_if_fail(*item1 != NULL, FALSE);
  Py_DECREF(*item1);

  if (! PyObject_IsInstance(*item1, (PyObject *) &PSTCategoryNode_Type))
    {
      g_set_error(err, 0, 0, _("first item of returned sequence is not a ST.CategoryNode object"));
      return FALSE;
    }

  *item2 = PySequence_GetItem(result, 1);
  g_return_val_if_fail(*item2 != NULL, FALSE);
  Py_DECREF(*item2);

  return TRUE;
}

/*** C API *******************************************************************/

gboolean
pst_handler_register (PyObject *module)
{
  g_return_val_if_fail(module != NULL, FALSE);

  if (PyType_Ready(&PSTHandler_Type) < 0)
    return FALSE;

  Py_INCREF(&PSTHandler_Type);
  PyModule_AddObject(module, "Handler", (PyObject *) &PSTHandler_Type);

  PyModule_AddIntConstant(module, "HANDLER_NO_CATEGORIES", ST_HANDLER_NO_CATEGORIES);
  PyModule_AddIntConstant(module, "HANDLER_CONFIRM_DELETION", ST_HANDLER_CONFIRM_DELETION);

  PyModule_AddIntConstant(module, "HANDLER_STOCK_FIELD_NAME", ST_HANDLER_STOCK_FIELD_NAME);
  PyModule_AddIntConstant(module, "HANDLER_STOCK_FIELD_GENRE", ST_HANDLER_STOCK_FIELD_GENRE);

  PyModule_AddIntConstant(module, "HANDLER_EVENT_REFRESH", ST_HANDLER_EVENT_REFRESH);
  PyModule_AddIntConstant(module, "HANDLER_EVENT_REFRESH_MULTIPLE", ST_HANDLER_EVENT_REFRESH_MULTIPLE);
  PyModule_AddIntConstant(module, "HANDLER_EVENT_STREAM_RESOLVE", ST_HANDLER_EVENT_STREAM_RESOLVE);
  PyModule_AddIntConstant(module, "HANDLER_EVENT_STREAM_TUNE_IN", ST_HANDLER_EVENT_STREAM_TUNE_IN);
  PyModule_AddIntConstant(module, "HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE", ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE);
  PyModule_AddIntConstant(module, "HANDLER_EVENT_STREAM_RECORD", ST_HANDLER_EVENT_STREAM_RECORD);
  PyModule_AddIntConstant(module, "HANDLER_EVENT_STREAM_BROWSE", ST_HANDLER_EVENT_STREAM_BROWSE);
  PyModule_AddIntConstant(module, "HANDLER_EVENT_STREAM_DELETE", ST_HANDLER_EVENT_STREAM_DELETE);
  PyModule_AddIntConstant(module, "HANDLER_EVENT_STREAM_STOCK_FIELD_GET", ST_HANDLER_EVENT_STREAM_STOCK_FIELD_GET);
  PyModule_AddIntConstant(module, "HANDLER_EVENT_STREAM_MODIFY", ST_HANDLER_EVENT_STREAM_MODIFY);

  return TRUE;
}
