/*
 * $Id: xiph.c,v 1.17 2004/01/27 11:42:02 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <streamtuner/streamtuner.h>
#include <string.h>
#include "gettext.h"
#include "art/icon.h"

/*** cpp *********************************************************************/

#define COPYRIGHT		"Copyright (c) 2003, 2004 Jean-Yves Lefort"

#define XIPH_ROOT		"http://dir.xiph.org/cgi-bin/dir.py"

/*** types *******************************************************************/

typedef struct
{
  STStream	stream;
  
  char		*visible_name;
  char		*stream_type;
  char		*stream_description;
  char		*current_song;
  char		*genre;
  char		*audio_info;
  
  char		*listen_url;
} XiphStream;

enum {
  FIELD_VISIBLE_NAME,
  FIELD_STREAM_TYPE,
  FIELD_STREAM_DESCRIPTION,
  FIELD_CURRENT_SONG,
  FIELD_GENRE,
  FIELD_AUDIO_INFO,
  FIELD_LISTEN_URL
};

enum {
  TAG_ROOT,
  TAG_ENTRY_LIST,
  TAG_ENTRY,
  TAG_STREAM,
  TAG_LISTEN_URL,
  TAG_STREAM_TYPE,
  TAG_STREAM_DESCRIPTION,
  TAG_CURRENT_SONG,
  TAG_GENRE,
  TAG_AUDIO_INFO
};

typedef struct
{
  GNode			*parent_node;
  char			*charset;
} RefreshGenresInfo;

typedef struct
{
  int			page;
  int			current_tag;

  GMarkupParseContext	*context;
  gboolean		parse_error;

  char			*entry_name;
  XiphStream		*stream;
  GList			**streams;
} RefreshStreamsInfo;
  
/*** variables ***************************************************************/

static regex_t re_header_charset;
static regex_t re_genre;

static GNode *genres = NULL;

/*** functions ***************************************************************/

static XiphStream *stream_new_cb (gpointer data);
static void stream_field_get_cb (XiphStream *stream,
				 STHandlerField *field,
				 GValue *value,
				 gpointer data);
static void stream_field_set_cb (XiphStream *stream,
				 STHandlerField *field,
				 const GValue *value,
				 gpointer data);
static void stream_free_cb (XiphStream *stream, gpointer data);

static gboolean stream_tune_in_cb (XiphStream *stream,
				   gpointer data,
				   GError **err);
static gboolean stream_record_cb (XiphStream *stream,
				  gpointer data,
				  GError **err);

static GNode *genres_copy (void);
static gboolean genres_copy_cb (GNode *node, gpointer data);

static gboolean refresh_cb (STCategory *category,
			    GNode **categories,
			    GList **streams,
			    gpointer data,
			    GError **err);

static gboolean refresh_genres (GError **err);
static void refresh_genres_header_cb (const char *line, gpointer data);
static void refresh_genres_body_cb (const char *line, gpointer data);

static gboolean refresh_streams (STCategory *category,
				 GList **streams,
				 GError **err);
static void refresh_streams_line_cb (const char *line, gpointer data);
static void refresh_streams_start_element_cb (GMarkupParseContext *context,
					      const char *element_name,
					      const char **attribute_names,
					      const char **attribute_values,
					      gpointer user_data,
					      GError **err);
static void refresh_streams_end_element_cb (GMarkupParseContext *context,
					    const char *element_name,
					    gpointer user_data,
					    GError **err);
static void refresh_streams_text_cb (GMarkupParseContext *context,
				     const char *text,
				     gsize text_len,
				     gpointer user_data,
				     GError **err);
static void refresh_streams_error_cb (GMarkupParseContext *context,
				      GError *err,
				      gpointer user_data);

static gboolean search_url_cb (STCategory *category);

static gboolean init_re (void);
static void init_handler (void);

/*** implementation **********************************************************/

static XiphStream *
stream_new_cb (gpointer data)
{
  return g_new0(XiphStream, 1);
}

static void
stream_field_get_cb (XiphStream *stream,
		     STHandlerField *field,
		     GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case FIELD_VISIBLE_NAME:
      g_value_set_string(value, stream->visible_name);
      break;
      
    case FIELD_STREAM_TYPE:
      g_value_set_string(value, stream->stream_type);
      break;

    case FIELD_STREAM_DESCRIPTION:
      g_value_set_string(value, stream->stream_description);
      break;

    case FIELD_CURRENT_SONG:
      g_value_set_string(value, stream->current_song);
      break;

    case FIELD_GENRE:
      g_value_set_string(value, stream->genre);
      break;

    case FIELD_AUDIO_INFO:
      g_value_set_string(value, stream->audio_info);
      break;

    case FIELD_LISTEN_URL:
      g_value_set_string(value, stream->listen_url);
      break;

    default:
      g_assert_not_reached();
    }
}

static void
stream_field_set_cb (XiphStream *stream,
		     STHandlerField *field,
		     const GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case FIELD_VISIBLE_NAME:
      stream->visible_name = g_value_dup_string(value);
      break;

    case FIELD_STREAM_TYPE:
      stream->stream_type = g_value_dup_string(value);
      break;

    case FIELD_STREAM_DESCRIPTION:
      stream->stream_description = g_value_dup_string(value);
      break;

    case FIELD_CURRENT_SONG:
      stream->current_song = g_value_dup_string(value);
      break;

    case FIELD_GENRE:
      stream->genre = g_value_dup_string(value);
      break;

    case FIELD_AUDIO_INFO:
      stream->audio_info = g_value_dup_string(value);
      break;

    case FIELD_LISTEN_URL:
      stream->listen_url = g_value_dup_string(value);
      break;

    default:
      g_assert_not_reached();
    }
}

static void
stream_free_cb (XiphStream *stream, gpointer data)
{
  g_free(stream->visible_name);
  g_free(stream->stream_type);
  g_free(stream->stream_description);
  g_free(stream->current_song);
  g_free(stream->genre);
  g_free(stream->audio_info);
  g_free(stream->listen_url);

  st_stream_free((STStream *) stream);
}

static gboolean
stream_tune_in_cb (XiphStream *stream,
		   gpointer data,
		   GError **err)
{
  return st_action_run("play-stream", stream->listen_url, err);
}

static gboolean
stream_record_cb (XiphStream *stream,
		  gpointer data,
		  GError **err)
{
  return st_action_run("record-stream", stream->listen_url, err);
}

static GNode *
genres_copy (void)
{
  GNode *node;

  node = g_node_copy(genres);
  g_node_traverse(node, G_IN_ORDER, G_TRAVERSE_ALL, -1, genres_copy_cb, NULL);

  return node;
}

static gboolean
genres_copy_cb (GNode *node, gpointer data)
{
  STCategory *category = node->data;

  if (category)
    {
      STCategory *copy;

      copy = st_category_new();
      copy->name = g_strdup(category->name);
      copy->label = g_strdup(category->label);
      copy->url_postfix = g_strdup(category->url_postfix);

      node->data = copy;
    }

  return FALSE;
}

static gboolean
refresh_cb (STCategory *category,
	    GNode **categories,
	    GList **streams,
	    gpointer data,
	    GError **err)
{
  if (! genres)
    {
      if (! refresh_genres(err))
	return FALSE;
    }

  *categories = genres_copy();

  if (st_is_aborted())
    return FALSE;

  return refresh_streams(category, streams, err);
}

static gboolean
refresh_genres (GError **err)
{
  STTransferSession *session;
  gboolean status;
  RefreshGenresInfo info = { NULL, NULL };

  genres = g_node_new(NULL);

  session = st_transfer_session_new();
  status = st_transfer_session_get_by_line(session,
					   XIPH_ROOT,
					   0,
					   refresh_genres_header_cb,
					   &info,
					   refresh_genres_body_cb,
					   &info,
					   err);
  st_transfer_session_free(session);
  g_free(info.charset);

  if (! status)
    {
      g_node_destroy(genres);
      genres = NULL;
      
      return FALSE;
    }

  return TRUE;
}

static void
refresh_genres_header_cb (const char *line, gpointer data)
{
  RefreshGenresInfo *info = data;

  if (! info->charset)
    st_re_parse(&re_header_charset, line, &info->charset);
}

static void
refresh_genres_body_cb (const char *line, gpointer data)
{
  RefreshGenresInfo *info = data;
  char *converted;
  GError *convert_err = NULL;
  char *name;
  char *label;

  if (! info->charset)
    info->charset = g_strdup("ISO8859-1");

  converted = g_convert(line,
			strlen(line),
			"UTF-8",
			info->charset,
			NULL,
			NULL,
			&convert_err);
  if (! converted)
    {
      st_notice(_("Xiph: unable to convert line to UTF-8: %s"), convert_err->message);
      g_error_free(convert_err);
      return;
    }

  if (st_re_parse(&re_genre, converted, &name, &label))
    {
      if (*name)
	{
	  STCategory *category;
	  GNode *node;
      
	  category = st_category_new();
	  category->name = name;
	  category->url_postfix = g_strdup_printf("?xml=1&sgenre=%s", name);
	  
	  node = g_node_new(category);
	  
	  if (label[0] == '-' && label[1] == '-')
	    {
	      if (info->parent_node)
		{
		  char *display_label = g_strdup(label + 2);
		  
		  category->label = display_label;
		  g_free(label);
		  
		  g_node_append(info->parent_node, node);
		}
	      else
		{
		  st_notice(_("Xiph: parent genre not found"));
		  st_category_free(category);
		  g_node_destroy(node);
		}
	    }
	  else
	    {
	      category->label = label;
	      g_node_append(genres, node);
	      info->parent_node = node;
	    }
	}
      else			/* the index */
	{
	  g_free(name);
	  g_free(label);
	}
    }

  g_free(converted);
}

static gboolean
refresh_streams (STCategory *category,
		 GList **streams,
		 GError **err)
{
  static GMarkupParser parser = 
    {
      refresh_streams_start_element_cb,
      refresh_streams_end_element_cb,
      refresh_streams_text_cb,
      NULL,
      refresh_streams_error_cb
    };
  STTransferSession *session;
  RefreshStreamsInfo info;
  GList *page_streams;
  char *url;
  gboolean status = TRUE;

  *streams = NULL;

  info.page = 0;
  info.streams = &page_streams;

  session = st_transfer_session_new();
  do
    {
      info.current_tag = TAG_ROOT;
      info.stream = NULL;
      info.entry_name = NULL;
      *info.streams = NULL;

      info.context = g_markup_parse_context_new(&parser, 0, &info, NULL);
      info.parse_error = FALSE;

      url = g_strdup_printf(XIPH_ROOT "%s&start=%i", category->url_postfix, info.page++ * 10);
      status = st_transfer_session_get_by_line(session,
					       url,
					       ST_TRANSFER_PASS_NEWLINE,
					       NULL,
					       NULL,
					       refresh_streams_line_cb,
					       &info,
					       err);
      g_free(url);

      if (! info.parse_error)
	g_markup_parse_context_end_parse(info.context, NULL);
      g_markup_parse_context_free(info.context);
      
      if (info.stream)
	{
	  stream_free_cb(info.stream, NULL);
	  if (status && ! info.parse_error) /* only display warning if the transfer was otherwise correct */
	    st_notice(_("Xiph:EOF: found unterminated stream"));
	}

      if (! status)
	goto end;

      *streams = g_list_concat(*streams, *info.streams);
      g_free(info.entry_name);
    }
  while (g_list_length(*info.streams) > 0);

 end:
  st_transfer_session_free(session);

  return status;
}

static void
refresh_streams_line_cb (const char *line, gpointer data)
{
  RefreshStreamsInfo *info = data;
  GError *err = NULL;

  if (info->parse_error)
    return;

  if (! g_markup_parse_context_parse(info->context, line, strlen(line), &err))
    {
      info->parse_error = TRUE;
      g_error_free(err);
    }
}

static void
refresh_streams_start_element_cb (GMarkupParseContext *context,
				  const char *element_name,
				  const char **attribute_names,
				  const char **attribute_values,
				  gpointer user_data,
				  GError **err)
{
  RefreshStreamsInfo *info = user_data;

  if (info->parse_error)
    return;

  switch (info->current_tag)
    {
    case TAG_ROOT:
      if (! strcmp(element_name, "entry_list"))
	info->current_tag = TAG_ENTRY_LIST;
      break;

    case TAG_ENTRY_LIST:
      if (! strcmp(element_name, "entry"))
	{
	  int i;

	  for (i = 0; attribute_names[i]; i++)
	    if (! strcmp(attribute_names[i], "name"))
	      {
		g_free(info->entry_name);
		info->entry_name = g_strdup(attribute_values[i]);
		break;
	      }
	  
	  info->current_tag = TAG_ENTRY;
	}
      break;

    case TAG_ENTRY:
      if (! strcmp(element_name, "stream"))
	{
	  if (info->stream)	/* a malformed stream remains, free it */
	    {
	      st_notice(_("Xiph: found unterminated stream"));
	      stream_free_cb(info->stream, NULL);
	    }

	  info->stream = stream_new_cb(NULL);
	  info->stream->visible_name = g_strdup(info->entry_name);
	  
	  info->current_tag = TAG_STREAM;
	}
      break;

    case TAG_STREAM:
      if (! strcmp(element_name, "listen_url"))
	info->current_tag = TAG_LISTEN_URL;
      else if (! strcmp(element_name, "stream_type"))
	info->current_tag = TAG_STREAM_TYPE;
      else if (! strcmp(element_name, "stream_description"))
	info->current_tag = TAG_STREAM_DESCRIPTION;
      else if (! strcmp(element_name, "current_song"))
	info->current_tag = TAG_CURRENT_SONG;
      else if (! strcmp(element_name, "genre"))
	info->current_tag = TAG_GENRE;
      else if (! strcmp(element_name, "audio_info"))
	info->current_tag = TAG_AUDIO_INFO;
      break;
    }
}

static void
refresh_streams_end_element_cb (GMarkupParseContext *context,
				const char *element_name,
				gpointer user_data,
				GError **err)
{
  RefreshStreamsInfo *info = user_data;

  if (info->parse_error)
    return;

  switch (info->current_tag)
    {
    case TAG_LISTEN_URL:
    case TAG_STREAM_TYPE:
    case TAG_STREAM_DESCRIPTION:
    case TAG_CURRENT_SONG:
    case TAG_GENRE:
    case TAG_AUDIO_INFO:
      info->current_tag = TAG_STREAM;
      break;

    case TAG_STREAM:
      *info->streams = g_list_append(*info->streams, info->stream);
      info->stream = NULL;
      info->current_tag = TAG_ENTRY;
      break;

    case TAG_ENTRY:
      g_free(info->entry_name);
      info->entry_name = NULL;
      info->current_tag = TAG_ENTRY_LIST;
      break;

    case TAG_ENTRY_LIST:
      info->current_tag = TAG_ROOT;
      break;
    }
}

static void
refresh_streams_text_cb (GMarkupParseContext *context,
			 const char *text,
			 gsize text_len,
			 gpointer user_data,
			 GError **err)
{
  RefreshStreamsInfo *info = user_data;

  if (info->parse_error)
    return;

  switch (info->current_tag)
    {
    case TAG_LISTEN_URL:
      info->stream->listen_url = g_strndup(text, text_len);
      ((STStream *) info->stream)->name = g_strdup(info->stream->listen_url);
      break;

    case TAG_STREAM_TYPE:
      info->stream->stream_type = g_strndup(text, text_len);
      break;

    case TAG_STREAM_DESCRIPTION:
      info->stream->stream_description = g_strndup(text, text_len);
      break;

    case TAG_CURRENT_SONG:
      info->stream->current_song = g_strndup(text, text_len);
      break;

    case TAG_GENRE:
      info->stream->genre = g_strndup(text, text_len);
      break;

    case TAG_AUDIO_INFO:
      info->stream->audio_info = g_strndup(text, text_len);
      break;
    }
}

static void
refresh_streams_error_cb (GMarkupParseContext *context,
			  GError *err,
			  gpointer user_data)
{
  RefreshStreamsInfo *info = user_data;

  info->parse_error = TRUE;
  st_notice(_("Xiph: XML parse error: %s"), err->message);
}

static gboolean
search_url_cb (STCategory *category)
{
  char *str;

  str = st_search_dialog();
  if (str)
    {
      char *escaped;

      g_free(category->label);
      category->label = g_strdup_printf(_("Search results for \"%s\""), str);

      escaped = st_transfer_escape(str);
      g_free(str);

      g_free(category->url_postfix);
      category->url_postfix = g_strconcat("?xml=1&sgenre=%5BChoose%20Genre%5D&search=", escaped, NULL);
      g_free(escaped);

      return TRUE;
    }
  else
    return FALSE;
}

static gboolean
init_re (void)
{
  int status;

  status = regcomp(&re_header_charset, "^Content-Type: .*charset=(.*)", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_genre, "^<option value=\"(.*)\">(.*)</option>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);

  return TRUE;
}

static void
init_handler (void)
{
  STHandler *handler;
  GNode *stock_categories;
  STCategory *category;

  handler = st_handler_new("xiph");

  st_handler_set_label(handler, "Xiph");
  st_handler_set_copyright(handler, COPYRIGHT);
  st_handler_set_description(handler, _("Xiph.org Streaming Directory"));
  st_handler_set_home(handler, XIPH_ROOT);

  stock_categories = g_node_new(NULL);

  category = st_category_new();
  category->name = "__main";
  category->label = _("All [INDEX]");
  category->url_postfix = "?xml=1";
  
  g_node_append_data(stock_categories, category);

  category = st_category_new();
  category->name = "__search";
  category->label = g_strdup(_("Search"));
  category->url_cb = search_url_cb;

  g_node_append_data(stock_categories, category);

  st_handler_set_icon(handler, sizeof(art_icon), art_icon);
  st_handler_set_stock_categories(handler, stock_categories);

  st_handler_bind(handler, ST_HANDLER_EVENT_REFRESH, refresh_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_NEW, stream_new_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_GET, stream_field_get_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_SET, stream_field_set_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FREE, stream_free_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_TUNE_IN, stream_tune_in_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_RECORD, stream_record_cb, NULL);

  st_handler_add_field(handler, st_handler_field_new(FIELD_VISIBLE_NAME,
						     _("Name"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_STREAM_TYPE,
						     _("Type"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_STREAM_DESCRIPTION,
						     _("Description"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_CURRENT_SONG,
						     _("Current song"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_GENRE,
						     _("Genre"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_AUDIO_INFO,
						     _("Audio"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_LISTEN_URL,
						     _("Listen URL"),
						     G_TYPE_STRING,
						     FALSE));

  st_handlers_add(handler);
}

gboolean
plugin_init (GError **err)
{
  gboolean status;

  bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");

  if (! st_check_api_version(5, 4))
    {
      g_set_error(err, 0, 0, _("API version mismatch"));
      return FALSE;
    }

  status = init_re();
  g_return_val_if_fail(status == TRUE, FALSE);

  init_handler();

  st_action_register("record-stream", _("Record a stream"), "xterm -hold -e streamripper %q");
  st_action_register("play-stream", _("Listen to a stream"), "xmms %q");

  return TRUE;
}
