/*
 * $Id: xiph.c,v 1.33 2004/03/29 14:19:16 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <streamtuner/streamtuner.h>
#include <string.h>
#include <stdlib.h>
#include "gettext.h"
#include "art/icon.h"

/*** cpp *********************************************************************/

#define COPYRIGHT		"Copyright \302\251 2003, 2004 Jean-Yves Lefort"

#define XIPH_HOME		"http://dir.xiph.org/"
#define XIPH_XML		"http://dir.xiph.org/yp.xml"

/*** types *******************************************************************/

typedef struct
{
  STStream	stream;
  
  char		*server_name;
  char		*listen_url;
  char		*server_type;
  char		*bitrate;
  int		channels;
  int		samplerate;
  char		*genre;
  char		*current_song;
  char		*server_desc;
} XiphStream;

enum {
  FIELD_SERVER_NAME,
  FIELD_LISTEN_URL,
  FIELD_SERVER_TYPE,
  FIELD_BITRATE,
  FIELD_CHANNELS,
  FIELD_SAMPLERATE,
  FIELD_GENRE,
  FIELD_CURRENT_SONG,
  FIELD_SERVER_DESC,
  FIELD_AUDIO		/* meta field (bitrate, channels and samplerate) */
};

enum {
  TAG_ROOT,
  TAG_DIRECTORY,
  TAG_ENTRY,
  TAG_SERVER_NAME,
  TAG_LISTEN_URL,
  TAG_SERVER_TYPE,
  TAG_BITRATE,
  TAG_CHANNELS,
  TAG_SAMPLERATE,
  TAG_GENRE,
  TAG_CURRENT_SONG,
  TAG_SERVER_DESC,

  TAG_UNSUPPORTED
};

typedef struct
{
  GMarkupParseContext	*context;
  gboolean		parse_error;
  GError		**err;

  GQueue		*tag_queue;

  XiphStream		*stream;
  GList			**streams;
} RefreshStreamsInfo;
  
/*** variables ***************************************************************/

static struct
{
  char		*name;
  char		*label;
  char		*re;
  regex_t	compiled_re;
} stock_genres[] = {
  { "__alternative",	N_("Alternative"),	"alternative|indie|goth|college|industrial|punk|hardcore|ska" },
  { "__classical",	N_("Classical"),	"classical|opera|symphonic" },
  { "__country",	N_("Country"),		"country|swing" },
  { "__electronic",	N_("Electronic"),	"electronic|ambient|drum.*bass|trance|techno|house|downtempo|breakbeat|jungle|garage" },
  { "__rap",		N_("Hip-Hop/Rap"),	"hip ?hop|rap|turntabl|old school|new school" },
  { "__jazz",		N_("Jazz"),		"jazz|swing|big ?band" },
  { "__oldies",		N_("Oldies"),		"oldies|disco|50s|60s|70s|80s|90s" },
  { "__rock",		N_("Pop/Rock"),		"pop|rock|top ?40|metal" },
  { "__soul",		N_("R&B/Soul"),		"r ?(&|'? ?n ?'?) ?b|funk|soul|urban" },
  { "__spiritual",	N_("Spiritual"),	"spiritual|gospel|christian|muslim|jewish|religio" },
  { "__spoken",		N_("Spoken"),		"spoken|talk|comedy" },
  { "__world",		N_("World"),		"world|reggae|island|african|european|middle ?east|asia" },
  { "__other",		N_("Other"),		"various|mixed|misc|eclectic|film|show|instrumental" },
  { NULL }
};

static char *search_token = NULL;

/*** functions ***************************************************************/

static XiphStream *stream_new_cb (gpointer data);
static void stream_field_get_cb (XiphStream *stream,
				 STHandlerField *field,
				 GValue *value,
				 gpointer data);
static void stream_field_set_cb (XiphStream *stream,
				 STHandlerField *field,
				 const GValue *value,
				 gpointer data);
static void stream_stock_field_get_cb (XiphStream *stream,
				       STHandlerStockField stock_field,
				       GValue *value,
				       gpointer data);
static void stream_free_cb (XiphStream *stream, gpointer data);
static XiphStream *stream_copy (XiphStream *stream);

gboolean str_isnumeric (const char *str);
static char *stream_get_audio (XiphStream *stream);

static gboolean stream_tune_in_cb (XiphStream *stream,
				   gpointer data,
				   GError **err);
static gboolean stream_record_cb (XiphStream *stream,
				  gpointer data,
				  GError **err);

static GList *streams_match_genre (GList *streams, regex_t *regexp);
static GList *streams_match_any (GList *streams, const char *token);

static gboolean utf8_strcasecontains (const char *big, const char *little);

static gboolean refresh_streams (GList **streams, GError **err);
static void refresh_streams_line_cb (const char *line, gpointer data);
static void refresh_streams_start_element_cb (GMarkupParseContext *context,
					      const char *element_name,
					      const char **attribute_names,
					      const char **attribute_values,
					      gpointer user_data,
					      GError **err);
static void refresh_streams_end_element_cb (GMarkupParseContext *context,
					    const char *element_name,
					    gpointer user_data,
					    GError **err);
static void refresh_streams_text_cb (GMarkupParseContext *context,
				     const char *text,
				     gsize text_len,
				     gpointer user_data,
				     GError **err);

static gboolean search_url_cb (STCategory *category);

static void init_handler (void);

/*** implementation **********************************************************/

static XiphStream *
stream_new_cb (gpointer data)
{
  return g_new0(XiphStream, 1);
}

static void
stream_field_get_cb (XiphStream *stream,
		     STHandlerField *field,
		     GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case FIELD_SERVER_NAME:
      g_value_set_string(value, stream->server_name);
      break;
      
    case FIELD_LISTEN_URL:
      g_value_set_string(value, stream->listen_url);
      break;

    case FIELD_SERVER_TYPE:
      g_value_set_string(value, stream->server_type);
      break;

    case FIELD_BITRATE:
      g_value_set_string(value, stream->bitrate);
      break;

    case FIELD_CHANNELS:
      g_value_set_int(value, stream->channels);
      break;

    case FIELD_SAMPLERATE:
      g_value_set_int(value, stream->samplerate);
      break;

    case FIELD_GENRE:
      g_value_set_string(value, stream->genre);
      break;

    case FIELD_CURRENT_SONG:
      g_value_set_string(value, stream->current_song);
      break;

    case FIELD_SERVER_DESC:
      g_value_set_string(value, stream->server_desc);
      break;

    case FIELD_AUDIO:
      g_value_set_string_take_ownership(value, stream_get_audio(stream));
      break;

    default:
      g_assert_not_reached();
    }
}

static void
stream_field_set_cb (XiphStream *stream,
		     STHandlerField *field,
		     const GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case FIELD_SERVER_NAME:
      stream->server_name = g_value_dup_string(value);
      break;

    case FIELD_LISTEN_URL:
      stream->listen_url = g_value_dup_string(value);
      break;

    case FIELD_SERVER_TYPE:
      stream->server_type = g_value_dup_string(value);
      break;

    case FIELD_BITRATE:
      stream->bitrate = g_value_dup_string(value);
      break;

    case FIELD_CHANNELS:
      stream->channels = g_value_get_int(value);
      break;

    case FIELD_SAMPLERATE:
      stream->samplerate = g_value_get_int(value);
      break;

    case FIELD_GENRE:
      stream->genre = g_value_dup_string(value);
      break;

    case FIELD_CURRENT_SONG:
      stream->current_song = g_value_dup_string(value);
      break;

    case FIELD_SERVER_DESC:
      stream->server_desc = g_value_dup_string(value);
      break;

    default:
      g_assert_not_reached();
    }
}

static void
stream_stock_field_get_cb (XiphStream *stream,
			   STHandlerStockField stock_field,
			   GValue *value,
			   gpointer data)
{
  switch (stock_field)
    {
    case ST_HANDLER_STOCK_FIELD_NAME:
      g_value_set_string(value, stream->server_name);
      break;

    case ST_HANDLER_STOCK_FIELD_GENRE:
      g_value_set_string(value, stream->genre);
      break;
    }
}

static void
stream_free_cb (XiphStream *stream, gpointer data)
{
  g_free(stream->server_name);
  g_free(stream->listen_url);
  g_free(stream->server_type);
  g_free(stream->bitrate);
  g_free(stream->genre);
  g_free(stream->current_song);
  g_free(stream->server_desc);

  st_stream_free((STStream *) stream);
}

static XiphStream *
stream_copy (XiphStream *stream)
{
  XiphStream *copy;

  copy = stream_new_cb(NULL);
  ((STStream *) copy)->name = g_strdup(((STStream *) stream)->name);
  copy->server_name = g_strdup(stream->server_name);
  copy->listen_url = g_strdup(stream->listen_url);
  copy->server_type = g_strdup(stream->server_type);
  copy->bitrate = g_strdup(stream->bitrate);
  copy->channels = stream->channels;
  copy->samplerate = stream->samplerate;
  copy->genre = g_strdup(stream->genre);
  copy->current_song = g_strdup(stream->current_song);
  copy->server_desc = g_strdup(stream->server_desc);

  return copy;
}

gboolean
str_isnumeric (const char *str)
{
  int i;

  g_return_val_if_fail(str != NULL, FALSE);

  for (i = 0; str[i]; i++)
    if (! g_ascii_isdigit(str[i]))
      return FALSE;

  return TRUE;
}

static char *
stream_get_audio (XiphStream *stream)
{
  GString *audio = g_string_new(NULL);

  if (stream->bitrate)
    {
      if (! strncmp(stream->bitrate, "Quality", 7))
	g_string_append(audio, stream->bitrate);
      else if (str_isnumeric(stream->bitrate))
	{
	  int bitrate = atoi(stream->bitrate);

	  if (bitrate > 0 && bitrate < 1000000) /* avoid bogus bitrates */
	    {
	      /*
	       * Some bitrates are given in bps. To properly convert
	       * bps to kbps, we consider that if the bitrate is
	       * superior to 1000, the unit is bps.
	       *
	       * Also, bitrates such as "16000" probably mean
	       * "16kbps", so we use a kilo of 1000, not 1024.
	       */
	      if (bitrate > 1000)
		bitrate /= 1000;

	      g_string_append_printf(audio, "%ikbps", bitrate);
	    }
	}
    }

  if (stream->samplerate > 0)
    {
      if (*audio->str)
	g_string_append_c(audio, ' ');
      g_string_append_printf(audio, "%iHz", stream->samplerate);
    }

  if (*audio->str)
    g_string_append_c(audio, ' ');

  g_string_append(audio, stream->channels == 2 ? _("Stereo") : _("Mono"));
  
  return g_string_free(audio, FALSE);
}

static gboolean
stream_tune_in_cb (XiphStream *stream,
		   gpointer data,
		   GError **err)
{
  return st_action_run("play-stream", stream->listen_url, err);
}

static gboolean
stream_record_cb (XiphStream *stream,
		  gpointer data,
		  GError **err)
{
  return st_action_run("record-stream", stream->listen_url, err);
}

static GList *
streams_match_genre (GList *streams, regex_t *regexp)
{
  GList *matching = NULL;
  GList *l;

  for (l = streams; l; l = l->next)
    {
      XiphStream *stream = l->data;

      if (st_re_match(regexp, stream->genre))
	matching = g_list_append(matching, stream_copy(stream));
    }

  return matching;
}

static GList *
streams_match_any (GList *streams, const char *token)
{
  GList *matching = NULL;
  GList *l;

  for (l = streams; l; l = l->next)
    {
      XiphStream *stream = l->data;

      if (utf8_strcasecontains(stream->server_name, token)
	  || utf8_strcasecontains(stream->listen_url, token)
	  || utf8_strcasecontains(stream->server_type, token)
	  || utf8_strcasecontains(stream->genre, token)
	  || utf8_strcasecontains(stream->current_song, token)
	  || utf8_strcasecontains(stream->server_desc, token))
	matching = g_list_append(matching, stream_copy(stream));
    }

  return matching;
}

static gboolean
utf8_strcasecontains (const char *big, const char *little)
{
  gboolean contains;
  char *normalized_big;
  char *normalized_little;
  char *case_normalized_big;
  char *case_normalized_little;

  g_return_val_if_fail(big != NULL, FALSE);
  g_return_val_if_fail(little != NULL, FALSE);

  normalized_big = g_utf8_normalize(big, -1, G_NORMALIZE_ALL);
  normalized_little = g_utf8_normalize(little, -1, G_NORMALIZE_ALL);
  case_normalized_big = g_utf8_casefold(normalized_big, -1);
  case_normalized_little = g_utf8_casefold(normalized_little, -1);

  contains = strstr(case_normalized_big, case_normalized_little) != NULL;

  g_free(normalized_big);
  g_free(normalized_little);
  g_free(case_normalized_big);
  g_free(case_normalized_little);

  return contains;
}

static gboolean
refresh_multiple_cb (GNode **categories,
		     GHashTable **streams,
		     gpointer data,
		     GError **err)
{
  GList *streams_list = NULL;
  int i;

  if (! refresh_streams(&streams_list, err))
    return FALSE;

  *streams = g_hash_table_new(g_str_hash, g_str_equal);

  g_hash_table_insert(*streams, "__main", streams_list);

  if (search_token)
    g_hash_table_insert(*streams,
			"__search",
			streams_match_any(streams_list, search_token));
  
  for (i = 0; stock_genres[i].name; i++)
    g_hash_table_insert(*streams,
			stock_genres[i].name,
			streams_match_genre(streams_list, &stock_genres[i].compiled_re));

  return TRUE;
}

static gboolean
refresh_streams (GList **streams, GError **err)
{
  static GMarkupParser parser = 
    {
      refresh_streams_start_element_cb,
      refresh_streams_end_element_cb,
      refresh_streams_text_cb,
      NULL,
      NULL,
    };
  STTransferSession *session;
  RefreshStreamsInfo info;
  gboolean status = TRUE;

  info.context = g_markup_parse_context_new(&parser, 0, &info, NULL);
  info.parse_error = FALSE;
  info.err = err;
  info.tag_queue = g_queue_new();
  info.stream = NULL;
  info.streams = streams;
  *info.streams = NULL;

  session = st_transfer_session_new();
  status = st_transfer_session_get_by_line(session,
					   XIPH_XML,
					   ST_TRANSFER_PASS_NEWLINE,
					   NULL,
					   NULL,
					   refresh_streams_line_cb,
					   &info,
					   err);
  st_transfer_session_free(session);

  if (status && ! info.parse_error)
    g_markup_parse_context_end_parse(info.context, NULL); /* ignore errors */

  g_markup_parse_context_free(info.context);
  g_queue_free(info.tag_queue);
      
  if (info.stream)
    {
      stream_free_cb(info.stream, NULL);
      if (status && ! info.parse_error) /* only display warning if the transfer was otherwise correct */
	st_notice(_("Xiph:EOF: found unterminated stream"));
    }

  return status;
}

static void
refresh_streams_line_cb (const char *line, gpointer data)
{
  RefreshStreamsInfo *info = data;

  if (! info->parse_error)
    {
      if (! g_markup_parse_context_parse(info->context, line, strlen(line), info->err))
	info->parse_error = TRUE;
    }
}

static void
refresh_streams_start_element_cb (GMarkupParseContext *context,
				  const char *element_name,
				  const char **attribute_names,
				  const char **attribute_values,
				  gpointer user_data,
				  GError **err)
{
  RefreshStreamsInfo *info = user_data;
  int tag = TAG_UNSUPPORTED;

  switch (GPOINTER_TO_INT(g_queue_peek_head(info->tag_queue)))
    {
    case TAG_ROOT:		/* empty queue */
      if (! strcmp(element_name, "directory"))
	tag = TAG_DIRECTORY;
      break;

    case TAG_DIRECTORY:
      if (! strcmp(element_name, "entry"))
	{
	  if (info->stream)	/* a malformed stream remains, free it */
	    {
	      st_notice(_("Xiph: found unterminated stream"));
	      stream_free_cb(info->stream, NULL);
	    }

	  info->stream = stream_new_cb(NULL);
	  tag = TAG_ENTRY;
	}
      break;

    case TAG_ENTRY:
      if (! strcmp(element_name, "server_name"))
	tag = TAG_SERVER_NAME;
      else if (! strcmp(element_name, "listen_url"))
	tag = TAG_LISTEN_URL;
      else if (! strcmp(element_name, "server_type"))
	tag = TAG_SERVER_TYPE;
      else if (! strcmp(element_name, "bitrate"))
	tag = TAG_BITRATE;
      else if (! strcmp(element_name, "channels"))
	tag = TAG_CHANNELS;
      else if (! strcmp(element_name, "samplerate"))
	tag = TAG_SAMPLERATE;
      else if (! strcmp(element_name, "genre"))
	tag = TAG_GENRE;
      else if (! strcmp(element_name, "current_song"))
	tag = TAG_CURRENT_SONG;
      else if (! strcmp(element_name, "server_desc"))
	tag = TAG_SERVER_DESC;
      break;
    }

  g_queue_push_head(info->tag_queue, GINT_TO_POINTER(tag));
}

static void
refresh_streams_end_element_cb (GMarkupParseContext *context,
				const char *element_name,
				gpointer user_data,
				GError **err)
{
  RefreshStreamsInfo *info = user_data;

  switch (GPOINTER_TO_INT(g_queue_pop_head(info->tag_queue)))
    {
    case TAG_ENTRY:
      if (((STStream *) info->stream)->name)
	*info->streams = g_list_append(*info->streams, info->stream);
      else
	{
	  stream_free_cb(info->stream, NULL);
	  st_notice(_("Xiph: found unnamed stream"));
	}
      info->stream = NULL;
      break;
    }
}

static void
refresh_streams_text_cb (GMarkupParseContext *context,
			 const char *text,
			 gsize text_len,
			 gpointer user_data,
			 GError **err)
{
  RefreshStreamsInfo *info = user_data;
  char *tmp;

  switch (GPOINTER_TO_INT(g_queue_peek_head(info->tag_queue)))
    {
    case TAG_SERVER_NAME:
      info->stream->server_name = g_strndup(text, text_len);
      break;

    case TAG_LISTEN_URL:
      info->stream->listen_url = g_strndup(text, text_len);
      ((STStream *) info->stream)->name = g_strdup(info->stream->listen_url);
      break;

    case TAG_SERVER_TYPE:
      info->stream->server_type = g_strndup(text, text_len);
      break;

    case TAG_BITRATE:
      info->stream->bitrate = g_strndup(text, text_len);
      break;

    case TAG_CHANNELS:
      tmp = g_strndup(text, text_len);
      info->stream->channels = atoi(tmp);
      g_free(tmp);
      break;
	
    case TAG_SAMPLERATE:
      tmp = g_strndup(text, text_len);
      info->stream->samplerate = atoi(tmp);
      g_free(tmp);
      break;

    case TAG_GENRE:
      info->stream->genre = g_strndup(text, text_len);
      break;

    case TAG_CURRENT_SONG:
      info->stream->current_song = g_strndup(text, text_len);
      break;

    case TAG_SERVER_DESC:
      info->stream->server_desc = g_strndup(text, text_len);
      break;
    }
}

static gboolean
search_url_cb (STCategory *category)
{
  char *str;

  str = st_search_dialog();
  if (str)
    {
      g_free(category->label);
      category->label = g_strdup_printf(_("Search results for \"%s\""), str);

      g_free(search_token);
      search_token = str;

      return TRUE;
    }
  else
    return FALSE;
}

static void
init_handler (void)
{
  STHandler *handler;
  GNode *stock_categories;
  STCategory *category;
  int i;

  handler = st_handler_new("xiph");

  st_handler_set_label(handler, "Xiph");
  st_handler_set_copyright(handler, COPYRIGHT);
  st_handler_set_description(handler, _("Xiph.org Streaming Directory"));
  st_handler_set_home(handler, XIPH_HOME);

  stock_categories = g_node_new(NULL);

  category = st_category_new();
  category->name = "__main";
  category->label = _("All");
  
  g_node_append_data(stock_categories, category);

  category = st_category_new();
  category->name = "__search";
  category->label = g_strdup(_("Search"));
  category->url_cb = search_url_cb;

  g_node_append_data(stock_categories, category);

  for (i = 0; stock_genres[i].name; i++)
    {
      int status;

      /* compile the regexp */
      status = regcomp(&stock_genres[i].compiled_re, stock_genres[i].re, REG_EXTENDED | REG_ICASE);
      g_return_if_fail(status == 0);

      category = st_category_new();
      category->name = stock_genres[i].name;
      category->label = _(stock_genres[i].label);

      g_node_append_data(stock_categories, category);
    }
  
  st_handler_set_icon(handler, sizeof(art_icon), art_icon);
  st_handler_set_stock_categories(handler, stock_categories);

  st_handler_bind(handler, ST_HANDLER_EVENT_REFRESH_MULTIPLE, refresh_multiple_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_NEW, stream_new_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_GET, stream_field_get_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_SET, stream_field_set_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_STOCK_FIELD_GET, stream_stock_field_get_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FREE, stream_free_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_TUNE_IN, stream_tune_in_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_RECORD, stream_record_cb, NULL);

  /* visible fields */

  st_handler_add_field(handler, st_handler_field_new(FIELD_SERVER_NAME,
						     _("Name"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_SERVER_DESC,
						     _("Description"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_GENRE,
						     _("Genre"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_CURRENT_SONG,
						     _("Current song"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_SERVER_TYPE,
						     _("Type"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_AUDIO,
						     _("Audio"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE
						     | ST_HANDLER_FIELD_VOLATILE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_LISTEN_URL,
						     _("URL"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE
						     | ST_HANDLER_FIELD_START_HIDDEN));
  
  /* invisible fields */

  st_handler_add_field(handler, st_handler_field_new(FIELD_BITRATE,
						     _("Bitrate"),
						     G_TYPE_STRING,
						     0));
  st_handler_add_field(handler, st_handler_field_new(FIELD_CHANNELS,
						     _("Channels"),
						     G_TYPE_INT,
						     0));
  st_handler_add_field(handler, st_handler_field_new(FIELD_SAMPLERATE,
						     _("Sample rate"),
						     G_TYPE_INT,
						     0));

  st_handlers_add(handler);
}

gboolean
plugin_init (GError **err)
{
  bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");

  if (! st_check_api_version(5, 5))
    {
      g_set_error(err, 0, 0, _("API version mismatch"));
      return FALSE;
    }

  init_handler();

  st_action_register("record-stream", _("Record a stream"), "xterm -hold -e streamripper %q");
  st_action_register("play-stream", _("Listen to a stream"), "xmms %q");

  return TRUE;
}
