#########################################################################################
# SVM (Statechart Virtual Machine), an interpreter for an extended statechart formalism #
#########################################################################################

# Copyright (C) 2003 Thomas Huining Feng

# Address:      MSDL, SOCS, McGill Univ., Montreal, Canada
# HomePage:     http://msdl.cs.mcgill.ca/people/tfeng/
# SVM HomePage: http://msdl.cs.mcgill.ca/people/tfeng/?research=svm
# Download:     http://sourceforge.net/projects/pyuml/
# CVS:          :pserver:anonymous@cvs.sourceforge.net:/cvsroot/pyuml
# Email:        hfeng2@cs.mcgill.ca

# This file is part of SVM.

# SVM is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your
# option) any later version.

# SVM is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
# License for more details.

# You should have received a copy of the GNU General Public License
# along with SVM; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA


from java.awt import Frame, Button, Color, Font, Dimension
from java.awt import Panel, BorderLayout, FlowLayout, Event
from javax.swing import JTextPane, JList, BorderFactory, JLabel
from javax.swing import JScrollPane, JSplitPane, JPanel, JButton
from javax.swing import ListCellRenderer, KeyStroke
from javax.swing.text import StyleConstants, TextAction, TabStop, TabSet, SimpleAttributeSet
from java.awt.event import MouseAdapter, KeyAdapter, KeyEvent
from java.lang import System
import javax.swing.SwingConstants as swingconst

from string import *
import Debugger
import sys
import StringUtil
import thread
import os.path
import re
from string import *
import thread
import time

try:
  from AboutBox import AboutBox
  from MyStyle import MyStyle
except:
  1

DEBUG_STRING='debug'

class CommandKeyAction(TextAction):
  def __init__(self, name, action):
    TextAction.__init__(self, name)
    self.action=action

  def actionPerformed(self, e):
    self.action()

class CommandKeyAdapter(KeyAdapter):
  def __init__(self, action):
    self.action=action

  def keyTyped(self, e):
    self.action(e)

class SingleLineText(JTextPane):
  def setSize(self, *args):
    if len(args)==1:
      d=args[0]
      w=d.width
      h=d.height
      if w<self.getParent().getSize().width:
        w=self.getParent().getSize().width
        args=(Dimension(w,h),)
    apply(JTextPane.setSize, (self,)+args)
    
  def getScrollableTracksViewportWidth(self):
    return 0

class StateNode:
  def __init__(self, path):
    paths=split(path, '.')
    self.depth=len(paths)-1
    self.path=path
    self.name=paths[self.depth]
    self.has_substate=0

  def __str__(self):
    if self.has_substate:
      return '  '*self.depth+"+ "+self.name
    else:
      return '  '*self.depth+"- "+self.name

  def getPath(self):
    return self.path

class ListMouseAdapter(MouseAdapter):
  def __init__(self, clickAction=None, dblClickAction=None):
    self.clickAction=clickAction
    self.dblClickAction=dblClickAction
    
  def mouseClicked(self, e):
   if e.getClickCount()==1:
     if self.clickAction:
       self.clickAction(e)
   elif e.getClickCount()==2:
     if self.dblClickAction:
       self.dblClickAction(e)

class ColorCellRenderer(JLabel, ListCellRenderer):
  def __init__(self, eventhandler=None):
    JLabel.__init__(self, preferredSize=(10, 20))
    self.eventhandler=eventhandler
  
  def getListCellRendererComponent(self, list, value, index, isSelected, cellHasFocus):
    s=str(value)
    self.setText(s)
    if self.eventhandler and value.getPath() in self.eventhandler.state:
      self.setBackground(Color.yellow)
    elif index%2==0:
      self.setBackground(Color(240,240,255))
    else:
      self.setBackground(Color.white)
    if isSelected:
      self.setBackground(Color(94,121,173))
      self.setBorder(BorderFactory.createRaisedBevelBorder())
      self.setForeground(Color.white)
    else:
      self.setBorder(BorderFactory.createEmptyBorder(1,1,1,1))
      self.setForeground(Color(1,1,139))
    self.setEnabled(list.isEnabled())
    self.setFont(list.getFont())
    self.setOpaque(1)
    return self

class StateList(Frame):
  def close(self, e):
    self.par.sshbtn.setEnabled(1)
    self.hide()

  def setData(self, data):
    self.data=data
    self.state_list.setListData(data)
    index=self.state_list.getSelectedIndex()
    if index<0:
      index=0
      self.state_list.setSelectedIndex(index)
    dlm=self.state_list.getModel()
    item=dlm.getElementAt(index)
    self.state_text.text=item.getPath()
    self.ensureView()

  def ensureView(self):
    i=0
    while i<len(self.data):
      if self.data[i].getPath() in self.par.eventhandler.state:
        self.state_list.ensureIndexIsVisible(i)
      i+=1

  def change_state(self, e):
    index=self.state_list.locationToIndex(e.getPoint())
    dlm=self.state_list.getModel()
    item=dlm.getElementAt(index)
    self.state_list.ensureIndexIsVisible(index)
    self.state_text.text=item.getPath()
  
  def __init__(self, parent, adapter):
    Frame.__init__(self, "State Hierarchy", windowClosing=self.close)
    self.par=parent
    self.data=[]
    p=Panel()
    tpan=Panel()
    tpan.setLayout(BorderLayout())
    self.mdllabel=JLabel(parent.eventhandler.model_name, swingconst.CENTER,
		    preferredSize=(250, 23), font=MyStyle.vfont1b,
		    foreground=Color.blue, background=MyStyle.lightblue,
		    opaque=1, border=MyStyle.border)
    tpan.add("North", self.mdllabel)
    
    self.state_list=JList()
    self.state_list.setCellRenderer(ColorCellRenderer(parent.eventhandler))
    self.state_list.setFont(MyStyle.ffont1)
    self.state_list.addMouseListener(ListMouseAdapter(clickAction=self.change_state))
    self.state_list.addKeyListener(adapter)
    scrollpane=JScrollPane(self.state_list, border=MyStyle.border)
    self.state_pane=scrollpane
    tpan.add("Center", scrollpane)
    
    p.setLayout(BorderLayout())
    p.add("Center", tpan)
    
    self.state_text=JTextPane(background=MyStyle.lightblue,
			      border=MyStyle.border, preferredSize=(100,50))
    self.state_text.addKeyListener(adapter)
    self.state_text.setEditable(0)
    
    doc=self.state_text.getStyledDocument()
    style=self.state_text.addStyle("normal", None)
    StyleConstants.setFontFamily(style, "DialogInput")
    StyleConstants.setForeground(style, Color.black)
    StyleConstants.setLineSpacing(style, 0)
    StyleConstants.setLeftIndent(style, 5)
    StyleConstants.setRightIndent(style, 5)
    StyleConstants.setSpaceAbove(style, 0)
    doc.setLogicalStyle(0, style)
    
    p.add("South", self.state_text)
    self.add(p)
    self.resize(250, 500)

class DefaultGUI(Frame):
  def window_close(self, e=None):
    self.dispose()
    self.state_window.dispose()
    System.exit(0)

  def setfocus(self, e):
    self.command_text.requestFocus()

  def append_command(self, command):
    l=len(self.command_list)
    if l==0 or command!=self.command_list[l-1]:
      self.command_list.append(command)
      self.command_index+=1

  def command_enter(self):
    command=self.gui.command_text.text
    command=command[0:len(command)-1]
    command=replace(command, '\r', '')   # for windows, line-end is \r\n instead of \n
    self.gui.execute_command(command)

  def up_command(self):
    command=self.command_text.text
    l=len(self.command_list)
    if l>0 and self.command_index>0 and (not command or \
           (self.command_index<l and self.command_list[self.command_index]==command)):
      self.command_index=self.command_index-1
      self.command_text.text=self.command_list[self.command_index]
    self.check_syntax()
 
  def down_command(self):
    command=self.command_text.text
    l=len(self.command_list)
    if l>0 and self.command_index<l-1 and (not command or \
           (self.command_index>=0 and self.command_list[self.command_index]==command)):
      self.command_index=self.command_index+1
      self.command_text.text=self.command_list[self.command_index]
    self.check_syntax()

  def handle_keystroke(self, e):
    self.command_index=len(self.command_list)
    self.check_syntax()

  def __init__(self, eventhandler, debugger):
    Frame.__init__(self, "Statechart Virtual Machine - Thomas (Feb. 2003)",
		   windowClosing=self.window_close, windowOpened=self.setfocus)
    self.eventhandler=eventhandler
    self.debugger=debugger
    self.events=[]
    self.debug=0
    self.command_list=[]
    self.command_index=0
    self.syntax_lock=thread.allocate_lock()
    self.syn=re.compile('([a-zA-Z0-9_]+)|(".*?")|(\'.*?\')|([%s]+)' % StringUtil.OPERATORS)
    self.debugger.CustomizeEvent(self.event_callback, None, 1)
    
    adapter=CommandKeyAdapter(self.handle_hotkeys)
    self.state_window=StateList(self, adapter)
    self.createWidgets(adapter)

    wndSize=self.getToolkit().getScreenSize()
    left=(wndSize.width-self.state_window.size.width-self.size.width)/2
    top=(wndSize.height-self.state_window.size.height)/2
    self.state_window.setBounds(left, top, self.state_window.size.width, self.state_window.size.height)
    left+=self.state_window.size.width
    self.setBounds(left, top, self.size.width, self.size.height)

    self.update_state_list()
    self.show_state_list()
    self.show()
    
  def handle_hotkeys(self, e):
    if e.isControlDown():
      if ord(e.getKeyChar())==4:
	if self.debug:
          self.stop_debug()
        else:
          self.start_debug()
      elif ord(e.getKeyChar())==19:
        self.take_snapshot()

  def is_keyword(self, word, level=None, keywords=StringUtil.KEYWORDS):
    if not level:
      for k in keywords:
        if word in k:
          return 1
      return 0
    else:
      return word in keywords[level-1]

  def check_syntax_thread(self):
    time.sleep(0.1)
    self.syntax_lock.acquire()
    text=self.command_text.text
    text=replace(text, '\r', '')
    cp=self.command_text.getCaretPosition()
    cppos=find(text, '\n', cp)
    if cppos>0:
      left=text[0:cppos]
    else:
      left=text
    startpos=rfind(left, '\n')+1
    text=split(text, '\n')
    maxline=len(split(left, '\n'))-1
    line=text[maxline]
    if self.debug:
      keywords=StringUtil.KEYWORDS+[StringUtil.DEBUG_KEYWORDS]
    else:
      keywords=[StringUtil.NONDEBUG_KEYWORDS+self.events]
    pos=0
    doc=self.command_text.getStyledDocument()
    while pos>=0:
      s=self.syn.search(line[pos:])
      if s:
	start=s.start()+pos
	end=s.end()+pos
	if start>pos:
	  style=self.command_text.getStyle('normal')
	  doc.setCharacterAttributes(pos, start-pos, style, 1)
        pos=end
        word=line[start:end]
	l=len(word)
	colorset=0
        for t in range(len(keywords)):
          if self.is_keyword(word, t+1, keywords):
            if t+1==len(keywords):
	      style=self.command_text.getStyle('ultra')
              doc.setCharacterAttributes(startpos+start, l, style, 1)
	      colorset=1
	      break
            else:
              style=self.command_text.getStyle('level_%d' % (t+1))
              doc.setCharacterAttributes(startpos+start, l, style, 1)
	      colorset=1
	      break
          else:
            if len(word)>=2 and ((word[0]=='"' and word[len(word)-1]=='"') or (word[0]=='\'' and word[len(word)-1]=='\'')):
              style=self.command_text.getStyle('string')
              doc.setCharacterAttributes(startpos+start+1, l-2, style, 1)
	      colorset=1
	      break
	  if not colorset:  
            style=self.command_text.getStyle('normal')
            doc.setCharacterAttributes(startpos+start, l, style, 1)
      else:
	if len(line)>pos:
	  style=self.command_text.getStyle('normal')
	  doc.setCharacterAttributes(pos, len(line)-pos, style, 1)
	pos=-1
    style=self.command_text.getStyle('normal')
    doc.setLogicalStyle(0, style)
    self.syntax_lock.release()

  def check_syntax(self):
    thread.start_new_thread(self.check_syntax_thread, ())

  def show_about_box(self, e=None):
    about=AboutBox(self)
    about.show()

  def createWidgets(self, adapter):
    p=Panel()
    p.setLayout(BorderLayout())
    bp=Panel()
    tp=Panel()
    p.add("Center", tp)
    p.add("South", bp)

    tp.setLayout(BorderLayout())
    ltp=Panel()
    rtp=Panel()

    ltp.setLayout(BorderLayout())
    ltplabel=JLabel("Enabled Events", swingconst.CENTER,
		    preferredSize=(150, 23),
		    foreground=Color.blue, background=MyStyle.lightblue,
		    opaque=1, border=MyStyle.border)
    ltplabel.setFont(MyStyle.vfont1b)
    self.event_list=JList()
    self.event_list.addKeyListener(adapter)
    self.event_list.setFont(MyStyle.vfont1i)
    self.event_list.setCellRenderer(ColorCellRenderer())
    self.event_list.addMouseListener(ListMouseAdapter(dblClickAction=self.trigger_event))
    lscrollpane=JScrollPane(self.event_list, border=MyStyle.border, preferredSize=(150, 10))
    ltp.add("North", ltplabel)
    ltp.add("Center", lscrollpane)
    
    rtp.setLayout(BorderLayout())
    rtplabel=JLabel("Output", swingconst.CENTER,
		    preferredSize=(150, 23),
		    foreground=Color.blue, background=MyStyle.lightblue,
		    opaque=1, border=MyStyle.border)
    rtplabel.setFont(MyStyle.vfont1b)
    
    self.output_text=JTextPane()
    self.output_text.addKeyListener(adapter)

    style=self.output_text.addStyle("normal", None)
    StyleConstants.setFontFamily(style, "DialogInput")
    StyleConstants.setForeground(style, Color.black)
    StyleConstants.setBackground(style, Color.white)
    StyleConstants.setLineSpacing(style, 0)
    StyleConstants.setSpaceAbove(style, 0)

    style=self.output_text.addStyle("event", None)
    StyleConstants.setFontFamily(style, "DialogInput")
    StyleConstants.setForeground(style, Color.blue)
    StyleConstants.setBackground(style, Color.white)
    StyleConstants.setLineSpacing(style, 0)
    StyleConstants.setSpaceAbove(style, 0)
    
    style=self.output_text.addStyle("debug", None)
    StyleConstants.setFontFamily(style, "DialogInput")
    StyleConstants.setForeground(style, Color.magenta)
    StyleConstants.setBackground(style, Color.white)
    StyleConstants.setLineSpacing(style, 0)
    StyleConstants.setSpaceAbove(style, 0)

    style=self.output_text.addStyle("message", None)
    StyleConstants.setFontFamily(style, "DialogInput")
    StyleConstants.setForeground(style, Color.red)
    StyleConstants.setBackground(style, Color.yellow)
    StyleConstants.setLineSpacing(style, 0)
    StyleConstants.setSpaceAbove(style, 0)

    self.output_text.setEditable(0)

    bbbrtp=Panel(background=Color(255,255,224))
    bbbrtp.setLayout(BorderLayout())
    exitbtn=JButton("Exit", border=MyStyle.border, background=MyStyle.lightyellow,
                    foreground=MyStyle.brown, preferredSize=(50,28),
                    actionPerformed=self.window_close, font=MyStyle.vfont1b)
    exitbtn.addKeyListener(adapter)
    aboutbtn=JButton("About", border=MyStyle.border, background=MyStyle.lightyellow,
                    foreground=MyStyle.brown, preferredSize=(60,28),
                    font=MyStyle.vfont1b, actionPerformed=self.show_about_box)
    aboutbtn.addKeyListener(adapter)
    snapbtn=JButton("Snapshot", border=MyStyle.border, background=MyStyle.lightyellow,
                    foreground=MyStyle.brown, preferredSize=(80,28),
                    font=MyStyle.vfont1b, actionPerformed=self.take_snapshot)
    snapbtn.addKeyListener(adapter)
    self.sshbtn=JButton("Show State Hierarchy", border=MyStyle.border, background=MyStyle.lightyellow,
		    foreground=MyStyle.brown, preferredSize=(160,28),
                    actionPerformed=self.show_state_list, font=MyStyle.vfont1b)
    self.sshbtn.addKeyListener(adapter)

    bbbrtp.add("East", exitbtn)
    bbbrtp1=Panel(background=Color(255,255,224))
    bbbrtp.add("Center", bbbrtp1)
    bbbrtp1.setLayout(BorderLayout())
    bbbrtp1.add("East", aboutbtn)
    bbbrtp2=Panel(background=Color(255,255,224))
    bbbrtp1.add("Center", bbbrtp2)
    bbbrtp2.setLayout(BorderLayout())
    bbbrtp2.add("East", snapbtn)
    bbbrtp3=Panel(background=Color(255,255,224))
    bbbrtp2.add("Center", bbbrtp3)
    bbbrtp3.setLayout(FlowLayout(FlowLayout.CENTER,0,0))
    bbbrtp3.add(self.sshbtn)

    rtp2=Panel()
    rtp2.setLayout(BorderLayout())
    rscrollpane=JScrollPane(self.output_text, border=MyStyle.border)
    mtplabel=JLabel("Command", swingconst.CENTER,
       	    preferredSize=(150, 23),
       	    foreground=Color.blue, background=MyStyle.pink,
       	    opaque=1, border=MyStyle.border)
    mtplabel.setFont(MyStyle.vfont1b)
    rtp2.add("South", mtplabel)
    rtp2.add("North", rtplabel)
    rtp2.add("Center", rscrollpane)
    self.command_above=rtp2

    self.create_command_box()
    self.command_text.addKeyListener(adapter)
    
    rtp.add("Center", self.command_pane)
    rtp.add("South", bbbrtp)

    hsplit=JSplitPane(JSplitPane.HORIZONTAL_SPLIT, ltp, rtp)
    hsplit.setOneTouchExpandable(1)
    tp.add("Center", hsplit)

    bp.setLayout(BorderLayout())
    self.status_text=JTextPane(background=MyStyle.lightblue,
			       border=MyStyle.border, preferredSize=(100,23))
    self.status_text.addKeyListener(adapter)
    self.status_text.setEditable(0)
    
    doc=self.status_text.getStyledDocument()
    style=self.status_text.addStyle("normal", None)
    StyleConstants.setFontFamily(style, "DialogInput")
    StyleConstants.setForeground(style, Color.black)
    StyleConstants.setLineSpacing(style, 0)
    StyleConstants.setAlignment(style, StyleConstants.ALIGN_RIGHT)
    StyleConstants.setLeftIndent(style, 5)
    StyleConstants.setRightIndent(style, 5)
    StyleConstants.setSpaceAbove(style, 0)
    doc.setLogicalStyle(0, style)
    
    bp.add("Center", self.status_text)
    
    self.resize(600,400)
    self.add(p)

  def setTabs(self, textPane, charactersPerTab):
    fm=textPane.getFontMetrics(textPane.getFont())
    charWidth=fm.charWidth('w')
    tabWidth=charWidth*charactersPerTab

    tabs=[]
    j=0
    while j<20:
      tab=j+1
      tabs.append(TabStop(tab*tabWidth))
      j+=1

    tabSet=TabSet(tabs);
    attributes=SimpleAttributeSet()
    StyleConstants.setTabSet(attributes, tabSet)
    length=textPane.getDocument().getLength()
    textPane.getStyledDocument().setParagraphAttributes(0, length, attributes, 1)

  def create_command_box(self, height=23):
    self.command_text=SingleLineText()
    km=self.command_text.getKeymap();
    ks = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0, 1)
    km.addActionForKeyStroke(ks, CommandKeyAction("Enter", self.command_enter))
    ks = KeyStroke.getKeyStroke(KeyEvent.VK_UP, 0, 1)
    km.addActionForKeyStroke(ks, CommandKeyAction("Up", self.up_command))
    ks = KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, 0, 1)
    km.addActionForKeyStroke(ks, CommandKeyAction("Down", self.down_command))
    self.command_text.addKeyListener(CommandKeyAdapter(self.handle_keystroke))
    scrollPane=JScrollPane(self.command_text, preferredSize=(10,height), border=MyStyle.border)
    scrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER)
    scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_NEVER)
    self.command_scroll=scrollPane
                                            
    doc=self.command_text.getStyledDocument()
    style=self.command_text.addStyle("normal", None)
    StyleConstants.setFontFamily(style, "DialogInput")
    StyleConstants.setForeground(style, Color.black)
    StyleConstants.setLineSpacing(style, 0)
    StyleConstants.setSpaceAbove(style, 0)
    doc.setLogicalStyle(0, style)

    style=self.command_text.addStyle("ultra", None)
    StyleConstants.setFontFamily(style, "DialogInput")
    StyleConstants.setForeground(style, Color.red)
    StyleConstants.setBackground(style, Color.yellow)
    StyleConstants.setLineSpacing(style, 0)
    StyleConstants.setSpaceAbove(style, 0)

    style=self.command_text.addStyle("level_1", None)
    StyleConstants.setFontFamily(style, "DialogInput")
    StyleConstants.setForeground(style, Color.blue)
    StyleConstants.setLineSpacing(style, 0)
    StyleConstants.setSpaceAbove(style, 0)
    
    style=self.command_text.addStyle("level_2", None)
    StyleConstants.setFontFamily(style, "DialogInput")
    StyleConstants.setForeground(style, Color.red)
    StyleConstants.setLineSpacing(style, 0)
    StyleConstants.setSpaceAbove(style, 0)

    style=self.command_text.addStyle("level_3", None)
    StyleConstants.setFontFamily(style, "DialogInput")
    StyleConstants.setForeground(style, Color.magenta)
    StyleConstants.setLineSpacing(style, 0)
    StyleConstants.setSpaceAbove(style, 0)
    
    style=self.command_text.addStyle("string", None)
    StyleConstants.setFontFamily(style, "DialogInput")
    StyleConstants.setForeground(style, Color(0x008000))
    StyleConstants.setLineSpacing(style, 0)
    StyleConstants.setSpaceAbove(style, 0)    

    self.command_pane=JSplitPane(JSplitPane.VERTICAL_SPLIT, self.command_above, scrollPane)
    self.command_pane.setResizeWeight(1)
    self.command_pane.setDividerSize(0)
    self.command_pane.setEnabled(0)

    self.setTabs(self.command_text, 4)

  def resize_command_box(self, newweight=1):
    self.command_pane.setResizeWeight(newweight)
    self.command_pane.resetToPreferredSizes()

  def trigger_event(self, e):
    if not self.debug:
      index=self.event_list.getSelectedIndex()
      dlm=self.event_list.getModel()
      item=dlm.getElementAt(index)
      self.event_list.ensureIndexIsVisible(index)
      self.execute_command(str(item))
      
  def take_snapshot(self, e=None):
    name=os.path.splitext(self.eventhandler.model_name)[0]+'.snp'
    self.eventhandler.snap_to_file(name)
    self.send_output('*** Execution snapshot saved to file "'+name+'" ***\n', 'message')

  def show_state_list(self, e=None):
    self.state_window.show()
    self.state_window.setState(Frame.NORMAL)
    self.sshbtn.setEnabled(0)

  def send_output(self, data, tag='normal'):
    doc=self.output_text.getStyledDocument()
    style=self.output_text.getStyle(tag)
    doc.insertString(doc.getLength(), data, style);
    self.output_text.setCaretPosition(doc.getLength());

  def update_event_list(self):
    self.events=self.eventhandler.get_enabled_events()
    self.event_list.setListData(self.events)
    
  def event_callback(self, event, before, oldstate, newstate):
    if DefaultGUI.gui:
      self.status_text.text=str(self.eventhandler.state)
      self.update_event_list()
      if event==StringUtil.IMPORT_EVENT:
        self.update_state_list()
      self.state_window.state_pane.revalidate()
      self.state_window.state_pane.repaint()
      self.state_window.ensureView()
    
  def add_state_to_list(self, states, path, stateh):
    k=stateh.keys()
    k.sort()
    added=0
    for s in k:
      if not s in StringUtil.StateProperties:
        newpath=self.eventhandler.append_path(path, s)
	state=StateNode(newpath)
        states.append(state)
	added=1
        state.has_substate=self.add_state_to_list(states, newpath, stateh[s])
    return added

  def update_state_list(self):
    states=[]
    self.add_state_to_list(states, '', self.eventhandler.stateH)
    self.state_window.setData(states)

  def start_debug(self):
    self.eventhandler.lock.acquire()
    Debugger.DefaultInterpreter.output=self.send_output
    self.old_stdout=sys.__stdout__
    sys.stdout=Debugger.DefaultInterpreter
    self.debug=1
    self.lastprompt=''
    self.debugger.OutputInterruptLogo(DEBUG_STRING)
    self.command_text.text=""
    self.command_text.requestFocus()
    self.resize_command_box(0.7)
    self.command_scroll.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED)
    self.command_scroll.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED)

  def stop_debug(self):
    sys.stdout=self.old_stdout
    self.debug=0
    self.debugger.OutputLeaveMessage()
    self.resize_command_box()
    self.update_event_list()
    self.eventhandler.lock.release()
    self.command_text.text=""
    self.command_scroll.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER)
    self.command_scroll.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_NEVER)

  def execute_command(self, command):
    scommand=strip(command)
    self.command_index=len(self.command_list)
    if scommand=='exit' and not self.debug:
      self.window_close()
    elif scommand=='exit' and self.debug:
      self.append_command('exit')
      self.send_output('>>> exit\n', 'debug')
      self.stop_debug()
    elif scommand==DEBUG_STRING and not self.debug:
      self.append_command(DEBUG_STRING)
      self.send_output('[EVENT] > %s\n'%DEBUG_STRING, 'event')
      self.start_debug()
    elif self.debug:
      err=self.debugger.CheckSource(command)
      comp=self.check_completeness(command)
      if err==1 or comp:
        self.send_output('>>> '+replace(command, '\n', '\n... ')+'\n', 'debug')
      if comp:
        self.lastprompt=self.debugger.RunSource(command)
      else:
	self.lastprompt=command
      if not self.lastprompt:
        self.append_command(command)
        self.command_text.text=""
    else:
      self.command_text.text=""
      self.append_command(scommand)
      self.send_output('[EVENT] > '+scommand+'\n', 'event')
      self.eventhandler.event(scommand)

  def check_completeness(self, command):
    cs=split(command, '\n')
    if len(cs)==1:
      s=cs[0]
      if len(s)>0 and s[len(s)-1]==':':
        return 0
      else:
        return 1
    else:
      if strip(cs[len(cs)-1])=='':
        return 1
      else:
        return 0

DefaultGUI.gui=None

def setup_gui_debugger(eventhandler, debugger):
  try:
    DefaultGUI.gui=DefaultGUI(eventhandler, debugger)
    if eventhandler.description:
      DefaultGUI.gui.send_output(eventhandler.description+'\n')
    eventhandler.start()
    DefaultGUI.gui.status_text.text=str(eventhandler.state)
    DefaultGUI.gui.update_event_list()
  except:
    setup_tui_debugger(eventhandler, debugger)

def setup_tui_debugger(eventhandler, debugger):
  DefaultGUI.gui=None
  debugger.InterruptEvent('debug')
  if eventhandler.description:
    print eventhandler.description
  eventhandler.start()
  sys.__stdout__.write(str(eventhandler.state)+' > ')
  prompt=strip(sys.__stdin__.readline())
  while prompt!='exit':
    eventhandler.event(prompt)
    sys.__stdout__.write(str(eventhandler.state)+' > ')
    prompt=strip(sys.__stdin__.readline())

def sortlist(s):
  s.sort()
  return s

def dump_message(s):
  if DefaultGUI.gui:
    DefaultGUI.gui.send_output('%s\n'%s)
  else:
    print s
