/****************************************************************
*
*                  Container World Project
*
*           (c) 2002-2004 Imperial College London
*
* $RCSfile: Date.java,v $
* $Revision: 1.1 $
* $Date: 2003/04/16 13:15:43 $
* $Author: marsha02 $
* Log: See end of file
*
****************************************************************/

package cosmic.util;

/**
 * Class with date methods to convert between Julian day and Gregorian date.
 * Author: James.A.Marshall@imperial.ac.uk
 *
 */

public class Date
{
	/**
	 * Converts Gregorian Date into Julian Day
	 *
	 * @param gregorianDate ("yyyymmdd")
	 *
	 * @return Julian Day
	 *
	 * @throws IllegalArgumentException if gregorianDate is invalid
	 *
	 */
	public static int getJulianDay(int gregorianDate) throws IllegalArgumentException
	{
		if (gregorianDate < 10000101)
		{
			throw new IllegalArgumentException("getJulianDay called with illegal gregorianDate (gregorianDate = " + gregorianDate + ")");
		}
		int jd, y, m, d;
		Integer integer;
		String date, year, month, day;

		date = String.valueOf(gregorianDate);
		year = date.substring(0, 4);
		month = date.substring(4, 6);
		day = date.substring(6, 8);
		integer = new Integer(year);
		y = integer.intValue();
		integer = new Integer(month);
		m = integer.intValue();
		if (m < 1 || m > 12)
		{
			throw new IllegalArgumentException("getJulianDay called with illegal gregorianDate (gregorianDate = " + gregorianDate + ")");
		}
		integer = new Integer(day);
		d = integer.intValue();
		if (m == 1 || m == 3 || m == 5 || m == 7 || m == 8 || m == 10 || m == 12)
		{
			if (d < 1 || d > 31)
			{
				throw new IllegalArgumentException("getJulianDay called with illegal gregorianDate (gregorianDate = " + gregorianDate + ")");
			}
		}
		else if (m == 4 || m == 6 || m == 9 || m == 11)
		{
			if (d < 1 || d > 30)
			{
				throw new IllegalArgumentException("getJulianDay called with illegal gregorianDate (gregorianDate = " + gregorianDate + ")");
			}
		}
		else if (m == 2)
		{
			if (y % 4 == 0 ^ (y % 100 == 0 ^ y % 400 == 0))
			{
				if (d < 1 || d > 29)
				{
					throw new IllegalArgumentException("getJulianDay called with illegal gregorianDate (gregorianDate = " + gregorianDate + ")");
				}
			}
			else
			{
				if (d < 1 || d > 28)
				{
					throw new IllegalArgumentException("getJulianDay called with illegal gregorianDate (gregorianDate = " + gregorianDate + ")");
				}
			}
		}

		jd = ( 1461 * ( y + 4800 + ( m - 14 ) / 12 ) ) / 4 +
			 ( 367 * ( m - 2 - 12 * ( ( m - 14 ) / 12 ) ) ) / 12 -
			 ( 3 * ( ( y + 4900 + ( m - 14 ) / 12 ) / 100 ) ) / 4 +
			 d - 32075;

		return jd;
	}

	/**
	 * Converts Julian Day into Gregorian Date integer in format "yyyymmdd"
	 *
	 * @param julianDay (>= 0)
	 *
	 * @return Gregorian Date (yyyymmdd)
	 *
	 */
	public static int getGregorianDate(int julianDay)
	{
		if (julianDay < 0)
		{
			throw new IllegalArgumentException("getGregorianDate called with julianDay out of range (julianDay = " + julianDay + ")");
		}
		int l, n, i, j, d, m, y, gregorianDate;
		Integer integer;
		String date, year, month, day;

		l = julianDay + 68569;
		n = ( 4 * l ) / 146097;
		l = l - ( 146097 * n + 3 ) / 4;
		i = ( 4000 * ( l + 1 ) ) / 1461001;
		l = l - ( 1461 * i ) / 4 + 31;
		j = ( 80 * l ) / 2447;
		d = l - ( 2447 * j ) / 80;
		l = j / 11;
		m = j + 2 - ( 12 * l );
		y = 100 * ( n - 49 ) + i + l;
		day = new String(Integer.toString(d));
		if (day.length() == 1)
		{
			day = new String("0").concat(day);
		}
		month = new String(Integer.toString(m));
		if (month.length() == 1)
		{
			month = new String("0").concat(month);
		}
		year = new String(Integer.toString(y));
		date = year.concat(month.concat(day));
		integer = new Integer(date);

		return (integer.intValue());
	}

	/**
     * Private constructor for the Date class (objects of class Date should never be constructed)
     *
     */
	private Date()
	{
	}
}

/****************************************************************
*
*                              File log
*
* $Log: Date.java,v $
* Revision 1.1  2003/04/16 13:15:43  marsha02
* James Marshall: moved Julian Day/Gregorian Date methods into separate utility class
*
*
*
****************************************************************/