/*$$
 * packages uchicago.src.*
 * Copyright (c) 1999, Trustees of the University of Chicago
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with 
 * or without modification, are permitted provided that the following 
 * conditions are met:
 *
 *	 Redistributions of source code must retain the above copyright notice,
 *	 this list of conditions and the following disclaimer.
 *
 *	 Redistributions in binary form must reproduce the above copyright notice,
 *	 this list of conditions and the following disclaimer in the documentation
 *	 and/or other materials provided with the distribution.
 *
 *	 Neither the name of the University of Chicago nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE TRUSTEES OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Nick Collier
 * nick@src.uchicago.edu
 *
 * packages cern.jet.random.*
 * Copyright (c) 1999 CERN - European Laboratory for Particle
 * Physics. Permission to use, copy, modify, distribute and sell this
 * software and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice appear in
 * supporting documentation. CERN makes no representations about the 
 * suitability of this software for any purpose. It is provided "as is" 
 * without expressed or implied warranty. 
 *
 * Wolfgang Hoschek
 * wolfgang.hoschek@cern.ch
 *$$*/
import java.awt.Dimension;
import java.awt.Point;

import uchicago.src.sim.space.*;
import cern.jet.random.Uniform;

import gui.Drawable;
import gui.SimGraphics;
/**
 * The agent for the Heat Bugs simulation. This pretty much follows the
 * Swarm code.
 *
 * @author Swarm Project and Nick Collier
 * @version $Revision: 1.1.1.1 $ $Date: 2000/03/10 03:11:12 $
 * @see HeatBugsModel
 */

public class Heatbug implements Drawable {

  private double unhappiness = 0;
  private int x, y;
  private int idealTemp, outputHeat;
  private float randomMoveProb;

  private HeatSpace space;
  private Object2DTorus world;
  private Dimension worldSize;

  private int xSize;
  private int ySize;


  public Heatbug(HeatSpace space, Object2DTorus world, int x,
    int y, int idealTemp, int outputHeat, float randomMoveProb)
  {
    this.x = x;
    this.y = y;
    this.idealTemp = idealTemp;
    this.outputHeat = outputHeat;
    this.randomMoveProb = randomMoveProb;
    this.space = space;
    this.world = world;
    worldSize = world.getSize();
    xSize = worldSize.width;
    ySize = worldSize.height;
  }

  public void setXY(int x, int y) {
    this.x = x;
    this.y = y;
    world.putObjectAt(x, y, this);
  }

  public void heatbugStep() {
    long heatHere = (long)space.getValueAt(x, y);

    if (heatHere < idealTemp) {
      unhappiness = (double) (idealTemp - heatHere) / space.MAX;
    } else {
      unhappiness = (double) (heatHere - idealTemp) / space.MAX;
    }

    int type = (heatHere < idealTemp) ? space.HOT : space.COLD;
    Point p = space.findExtreme(type, x, y);

    if (Uniform.staticNextFloatFromTo(0.0f, 1.0f) < randomMoveProb) {
      p.x = x + Uniform.staticNextIntFromTo(-1, 1);
      p.y = y + Uniform.staticNextIntFromTo(-1, 1);
    }

    if (unhappiness == 0) {
      space.addHeat(x, y, outputHeat);
    } else {
      int tries = 0;

      if (p.x != x || p.y != y) {
        while ((world.getObjectAt(p.x, p.y) != null) && tries < 10) {

          int location = Uniform.staticNextIntFromTo(1, 8);
          // get the neighbors
          int prevX = (x + xSize - 1) % xSize;
          int nextX = (x + 1) % xSize;
          int prevY = (y + ySize - 1) % ySize;
          int nextY = (y + 1) % ySize;

          switch (location) {
            case 1:
              p.x = prevX;
              p.y = prevY;
              break;
            case 2:
              p.x = x;
              p.y = prevY;
              break;
            case 3:
              p.x = nextX;
              p.y = prevY;
              break;
            case 4:
              p.x = nextX;
              p.y = y;
              break;
            case 5:
              p.x = prevX;
              p.y = y;
              break;
            case 6:
              p.x = prevX;
              p.y = nextY;
              break;
            case 7:
              p.x = x;
              p.y = nextY;
              break;
            case 8:
              p.x = nextX;
              p.y = nextY;
            default:
              break;
          }
          tries++;
        }
        if (tries == 10) {
          p.x = x;
          p.y = y;
        }

      }
      space.addHeat(x, y, outputHeat);
      world.putObjectAt(x, y, null);
      x = p.x;
      y = p.y;
      world.putObjectAt(x, y, this);
    }
  }

  public double getUnhappiness() {
    return unhappiness;
  }

  public void setUnhappiness(double value) {
    unhappiness = value;
  }

  public int getX() {
    return x;
  }

  public void setX(int x) {
    this.x = x;
  }

   public int getY() {
    return y;
  }

  public void setY(int y) {
    this.y = y;
  }

  public int getIdealTemp() {
    return idealTemp;
  }

  public void setIdealTemp(int idealTemp) {
    this.idealTemp = idealTemp;
  }

  public int getOutputHeat() {
    return outputHeat;
  }

  public void setOutputHeat(int outputHeat) {
    this.outputHeat = outputHeat;
  }

  public float getRandomMoveProb() {
    return randomMoveProb;
  }

  public void setRandomMoveProb(float f) {
    randomMoveProb = f;
  }

  // drawable interface
  public void draw(SimGraphics g) {
    g.drawFastRoundRect(java.awt.Color.green);
  }
}
