/*$$
 * packages uchicago.src.*
 * Copyright (c) 1999, Trustees of the University of Chicago
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with 
 * or without modification, are permitted provided that the following 
 * conditions are met:
 *
 *	 Redistributions of source code must retain the above copyright notice,
 *	 this list of conditions and the following disclaimer.
 *
 *	 Redistributions in binary form must reproduce the above copyright notice,
 *	 this list of conditions and the following disclaimer in the documentation
 *	 and/or other materials provided with the distribution.
 *
 *	 Neither the name of the University of Chicago nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE TRUSTEES OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Nick Collier
 * nick@src.uchicago.edu
 *
 * packages cern.jet.random.*
 * Copyright (c) 1999 CERN - European Laboratory for Particle
 * Physics. Permission to use, copy, modify, distribute and sell this
 * software and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice appear in
 * supporting documentation. CERN makes no representations about the 
 * suitability of this software for any purpose. It is provided "as is" 
 * without expressed or implied warranty. 
 *
 * Wolfgang Hoschek
 * wolfgang.hoschek@cern.ch
 *$$*/
package gui;
import java.util.ArrayList;

import uchicago.src.sim.space.*;
import uchicago.src.collection.*;

/**
 * Displays 2d arrays of values (Integers or ints). As a Displayable this class
 * iterates through a Discrete2DSpace, turns the integers into
 * colors according to a {@link uchicago.src.sim.gui.ColorMap ColorMap}, and
 * draws these colors in the appropriate cells.
 *
 * @author Nick Collier
 * @version $Revision: 1.1.1.1 $ $Date: 2000/03/10 03:16:29 $
 */


public class Value2DDisplay extends Display2D {

  private static final int TOGGLE_VIEW = 0;
  private ColorMap colorMap;

  private int mapping = 1;
  private int mappingColor = 0;
  private boolean colorMapping = false;

  private boolean view = true;

  /**
   * Creates a Value2DDisplay to display the specified Discrete2DSpace
   * using the specified ColorMap.
   */
  public Value2DDisplay(Discrete2DSpace grid, ColorMap map) { // Integer noColorInt) {
    super(grid);
    colorMap = map;
  }

  /**
   * Linear transform of states (doubles, floats, integers etc.) to
   * colors for drawing. color = state / m + c
   */
  public void setDisplayMapping(int m, int c) {
    mapping = m;
    mappingColor = c;
    colorMapping = true;
  }

  /**
   * Draws the Discrete2DSpace converting the Numbers contained therein to
   * Colors according to the ColorMap specified in the constructor
   */
  public void drawDisplay(SimGraphics g) {
    if (!view)
      return;
      
    BaseMatrix matrix = grid.getMatrix();
    int xTrans = DisplayConstants.CELL_WIDTH;
    int yTrans = DisplayConstants.CELL_HEIGHT;
    if (matrix instanceof DoubleMatrix) {
      DoubleMatrix dm = (DoubleMatrix)matrix;
      double value;
      Integer colorValue;
      for (int i = 0; i < grid.getSizeX(); i++) {
        for (int j = 0; j < grid.getSizeY(); j++) {
          value = dm.getDoubleAt(i, j);
          colorValue = new Integer((int)(value / mapping + mappingColor));
          g.setDrawingCoordinates(i * xTrans, j * yTrans, 0);
          g.drawFastRect(colorMap.getColor(colorValue));
        }
      }
    } else {
      Integer colorValue;
      Number value;
      for (int i = 0; i < grid.getSizeX(); i++) {
        for (int j = 0; j < grid.getSizeY(); j++) {
          value = (Number)matrix.get(i, j);
          colorValue = new Integer((int)(value.doubleValue() / mapping + mappingColor));

          g.setDrawingCoordinates(i * xTrans, j * yTrans, 0);
          g.drawFastRect(colorMap.getColor(colorValue));

        }
      }
    }
  }

  /**
   * Gets a list of the DisplayInfo objects associated with this Object2DDisplay.
   */
  public ArrayList getDisplayableInfo() {
    ArrayList list = new ArrayList();
    list.add(new DisplayInfo("", TOGGLE_VIEW, this));
    return list;
  }

  /**
   * Invoked when a viewEvent for this display is fired by the
   * DisplaySurface.
   */
  public void viewEventPerformed(ViewEvent evt) {
    view = evt.showView();
  }
}




