/*$$
 * packages uchicago.src.*
 * Copyright (c) 1999, Trustees of the University of Chicago
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with 
 * or without modification, are permitted provided that the following 
 * conditions are met:
 *
 *	 Redistributions of source code must retain the above copyright notice,
 *	 this list of conditions and the following disclaimer.
 *
 *	 Redistributions in binary form must reproduce the above copyright notice,
 *	 this list of conditions and the following disclaimer in the documentation
 *	 and/or other materials provided with the distribution.
 *
 *	 Neither the name of the University of Chicago nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE TRUSTEES OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Nick Collier
 * nick@src.uchicago.edu
 *
 * packages cern.jet.random.*
 * Copyright (c) 1999 CERN - European Laboratory for Particle
 * Physics. Permission to use, copy, modify, distribute and sell this
 * software and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice appear in
 * supporting documentation. CERN makes no representations about the 
 * suitability of this software for any purpose. It is provided "as is" 
 * without expressed or implied warranty. 
 *
 * Wolfgang Hoschek
 * wolfgang.hoschek@cern.ch
 *$$*/


package collection;

import java.util.ListIterator;
import java.util.Enumeration;

/**
 * An iterator for iterating through BaseMatrices.
 *
 * @author Nick Collier
 * @version $Revision: 1.1.1.1 $ $Date: 2000/03/10 03:18:29 $
 */

public class MatrixIterator implements ListIterator, Enumeration {

  BaseMatrix myMatrix = null;
  int mpos = 0;
  int num_cols;
  int num_rows;

  MatrixIterator(BaseMatrix matrix, int pos) {
    myMatrix = matrix;
    mpos = pos;
    num_rows = matrix.getNumRows();
    num_cols = matrix.getNumCols();
  }

  /**
   * Returns current position
   */
  public int getIndex() {
    return mpos;
  }

   /**
   * Advance the current position one row forward, remaining in the
   * same column.
   */
  public void advanceRow() {
    mpos += num_cols;
  }

  /**
   * Advance the current position n rows forward, remaining in the
   * same column.
   * @param n The number of rows to advance
   */
  public void advanceRow(int n) {
    mpos += num_cols * n;
  }

  /**
   * Advance the current position to the beginning of the next row.
   */
  public void advanceBegRow() {
    int cur_col = mpos % num_cols;
    mpos += (num_cols - cur_col);
  }

  /**
   * Advance the current position to the beginning of the nth row from
   * the current row.
   * @param n The number of rows to advance
   */
  public void advanceBegRow(int n) {
    int cur_col = mpos % num_cols;
    mpos += (num_cols - cur_col) + num_cols * (n - 1);
  }

   /**
   * Move one row backward, remaining in the
   * same column.
   */
  public void retreatRow() {
    mpos -= num_cols;
  }

  /**
   * Move n rows backward, remaining in the
   * same column.
   * @param n The number of rows to advance
   */
  public void retreatRow(int n) {
    mpos -= num_cols * n;
  }

  /**
   * Move the current position to the beginning of the previous row.
   */
  public void retreatBegRow() {
    if (mpos == myMatrix.size()) {
      mpos -= num_cols;
    } else {
      int cur_col = mpos % num_cols;
      mpos -= (num_cols + cur_col);
    }
  }

  /**
   * Move to the beginning of the row n rows from the current position.
   * @param n The number of rows to retreat
   */
  public void retreatBegRow(int n) {
    if (mpos == myMatrix.size()) {
      mpos -= num_cols;
      mpos -= num_cols * (n - 1);
    } else {
      int cur_col = mpos % num_cols;
      mpos -= (num_cols + cur_col) + num_cols * (n - 1);
    }
  }


  // ListIterator interface

  public void add(Object o) {
    throw new UnsupportedOperationException("Unsuppored Operation");
  }

  public boolean hasNext() {
    return mpos < myMatrix.size();
  }

  public boolean hasPrevious() {
    return mpos > 0;

  }

  /**
   * Returns the next element in the matrix. This method may be called
   * repeatedly to iterate through the matrix, or intermixed with calls
   * to previous to go back and forth. (Note that alternating calls to next and
   * previous will return the same element repeatedly.)
   * @return the next element in the matrix.
   */
  public Object next() {
    Object retval = myMatrix.get(mpos);
    mpos++;
    return retval;
  }

  /**
   * Returns the index of the element that would be returned by a subsequent
   * call to next. (Returns list size if the list iterator is at the end of
   * the matrix.)
   * @return the index of the element that would be returned by a subsequent
   * call to next, or list size if list iterator is at end of matrix.
   */
  public int nextIndex() {
   if (mpos == num_rows * num_cols - 1)
    return num_rows * num_cols;
   else
    return mpos + 1;
  }


  /**
   * Returns the previous element in the matrix. This method may be called
   * repeatedly to iterate through the matrix backwards, or intermixed with
   * calls to next to go back and forth. (Note that alternating calls to next
   * and previous will return the same element repeatedly.)
   * @return the previous elment in the matrix.
   */
  public Object previous() {
    mpos--;
    return myMatrix.get(mpos);
  }

  /**
   * Returns the index of the element that would be returned by a subsequent
   * call to previous. (Returns -1 if the matrix iterator is at the beginning
   * of the list.
   * @return the index of the element that would be returned by a subsequent
   * call to previous, or -1 if iterator is at the beginning the of matrix.
   */
  public int previousIndex() {
   if (mpos == 0)
    return -1;
   else
    return mpos - 1;
  }

  /**
   * Removes from the matrix the last element that was returned by next
   * or previous, setting the value of the matrix at that coordinate to null.
   */
  public void remove() {
    mpos--;
    myMatrix.remove(mpos);
    mpos++;
  }
  /**
   * Replaces the last element returned by next or previous with the specified
   * element.
   * @param o the object to set the previous element to.
   */
  public void set(Object o) {
    mpos--;
    myMatrix.put(mpos, o);
    mpos++;
  }

  // Enumeration interface
  public boolean hasMoreElements() {
    return this.hasNext();
  }

  public Object nextElement() {
    return this.next();
  }



  /**
   * Return true if a specified object is the same kind of iterator as me
   * and is positioned at the same element.
   * @param object Any object.
   */
   public boolean equals(Object obj) {
    if (obj instanceof MatrixIterator) {

      MatrixIterator m = (MatrixIterator)obj;
      return (myMatrix == m.myMatrix && mpos == m.mpos);
    }
    return false;
   }

  /**
   * Return the object at the current index
   * @exception IndexOutOfBoundsException If the current index is invalid.
   */

  public Object get() {
    return myMatrix.get(mpos);
  }

  /**
   * Return the object that is a specified distance from my current position.
   * @param offset The offset from my current position.
   * @exception IndexOutOfBoundsException If the adjusted index is invalid.
   */
  public Object get(int offset) {
    return myMatrix.get(mpos + offset);
  }

  /**
   * Return the Matrix associated with this Iterator
   * @return The associated Matrix
   */
  public BaseMatrix getMatrix() {
    return (BaseMatrix)myMatrix;
  }

  /**
   * Set the object at my current position to objdct.
   * @param object The object to be written at my current position.
   * @exception ArrayIndexOutOfBoundsException If I'm positioned at an invalid index.
   */
  public void put(Object object) {
    myMatrix.put(mpos, object);
  }


  /**
   * Write an object at a specified distance from my current position.
   * @param offset The offset from my current position.
   * @param object The object to write.
   * @exception ArrayIndexOutOfBoundsException If the adjusted index is invalid.
   */
  public void put(int offset, Object object) {
    myMatrix.put(mpos + offset, object);
  }




  public Object clone() {
    try {
      return super.clone();
    } catch (Exception e) {}
    return null;
  }
}
