/*$$
 * packages uchicago.src.*
 * Copyright (c) 1999, Trustees of the University of Chicago
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with 
 * or without modification, are permitted provided that the following 
 * conditions are met:
 *
 *	 Redistributions of source code must retain the above copyright notice,
 *	 this list of conditions and the following disclaimer.
 *
 *	 Redistributions in binary form must reproduce the above copyright notice,
 *	 this list of conditions and the following disclaimer in the documentation
 *	 and/or other materials provided with the distribution.
 *
 *	 Neither the name of the University of Chicago nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE TRUSTEES OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Nick Collier
 * nick@src.uchicago.edu
 *
 * packages cern.jet.random.*
 * Copyright (c) 1999 CERN - European Laboratory for Particle
 * Physics. Permission to use, copy, modify, distribute and sell this
 * software and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice appear in
 * supporting documentation. CERN makes no representations about the 
 * suitability of this software for any purpose. It is provided "as is" 
 * without expressed or implied warranty. 
 *
 * Wolfgang Hoschek
 * wolfgang.hoschek@cern.ch
 *$$*/

/**
 * Handles the drawing of Displayables and the probing of probeables.
 * Displayables are added to a DisplaySurface which is then responsible for
 * drawing them, and handling probing (single left-click) of them. Displayables
 * are drawn in the order they are added to the DisplaySurface. Typically,
 * a DisplaySurface is created by the model and has displayables added to it.
 * DisplaySurface also handles the creation of movies from and snapshots of
 * displays.<p>
 *
 * When drawing discrete (cell-based displays) the actual drawing surface
 * divided up into a number of cells equal to the DisplaySurface's width * the
 * DisplaySuface's height (and in the future, the DisplaySurface's depth).
 * Drawing at x, y coordinates via {@link uchicago.src.sim.gui.SimGraphics
 * SimGraphics} draws in the cell with these coordinates.<p>
 *
 * When drawing non-discrete displays(e.g Network2DDisplay), drawing is
 * done to actual screen coordinates, where x and y refer to a screen
 * coordinate.<p>
 *
 * @author Nick Collier
 * @version $Revision: 1.1.1.1 $ $Date: 2000/03/10 03:18:16 $
 * @see Displayable
 * @see Probeable
 */

package gui;

import javax.swing.*;
import java.awt.image.*;
import java.awt.*;
import java.awt.event.*;
import java.io.FileOutputStream;
import java.io.DataOutputStream;
import java.util.*;
import java.lang.reflect.*;
// import javax.media.protocol.FileTypeDescriptor;

//import uchicago.src.sim.remote.SimRunThread;
//import uchicago.src.sim.engine.SimEventListener;
//import uchicago.src.sim.engine.SimEvent;
//import uchicago.src.sim.engine.BaseController;
// import uchicago.src.sim.util.SimUtilities;
// import uchicago.src.reflector.IntrospectFrame;

public class DisplaySurface extends Container
                                    /* implements SimEventListener */ {
  private Toolkit toolkit;
  private int bufImgHeight, bufImgWidth;

  private Painter painter;
  private Vector probeables = new Vector();
  private JFrame f;
  private boolean needsUpdate = true;
  private String snapshotFile = null;

  private Hashtable displays = new Hashtable();
  private JMenu menu = new JMenu("View");
  private String name = "";

  private WindowAdapter dsWindowAdapter = new WindowAdapter() {
    public void windowIconified(WindowEvent evt) {
      needsUpdate = false;
    }

    public void windowDeiconified(WindowEvent evt) {
      needsUpdate = true;
    }

    public void windowClosing(WindowEvent evt) {
      needsUpdate = false;
      dispose();
    }
  };

  /**
   * Creates a DisplaySurface of the specified size and with the
   * specified model and the specified name. The size of the display surface
   * should be equal to
   * the size of the largest displayable is going to display. For example,
   * <code><br>Object2DDisplay display = new Object2DDisplay(someGrid);<br>
   * DisplaySurface ds = new DisplaySurface(display.getSize(), someModel,
   * "Display");<br></code>. The name appears in the title bar of the
   * actual screen window that contains the display.
   *
   * @param size the size of the DisplaySurface
   * @param model the model associated with this display surface
   * @param name the name that appears in the title bar of the physical
   * display
   */
  public DisplaySurface(Dimension size, String name) {
    this.name = name;
    setSize(size);

    // remote should be a controller scope variable
    // here for testing remote display
    //boolean remote = false;
    // if (remote) {
      //SimRunThread t = (SimRunThread)Thread.currentThread();
      //painter = new RemotePainter(this, size.width, size.height, t.getServer());
    // } else {
    painter = new LocalPainter(this, size.width, size.height);
    //}

    /*
    // ties probing to a left mouseclick on a display.
    addMouseListener(new MouseAdapter() {
      public void mouseClicked(MouseEvent evt) {
        //if (evt.isPopupTrigger()) {
          //System.out.println("Probing...");
          int x = evt.getX();
          int y = evt.getY();

          for (int i = 0; i < probeables.size(); i++) {
            Probeable p = (Probeable)probeables.elementAt(i);
            ArrayList list = p.getObjectsAt(x, y);
            for (int j = 0; j < list.size(); j++) {
              Object o = list.get(j);
              if (o != null) {
                final IntrospectFrame spector;
                if (o instanceof Named) {
                  Named n = (Named)o;
                  spector = new IntrospectFrame(o, n.getName());
                } else {
                  spector = new IntrospectFrame(o);
                }
                Runnable specDisplay = new Runnable() {
                  public void run() {
                    try {
                      spector.display();
                    } catch (InvocationTargetException ex) {
                      SimUtilities.showError("Probing error", ex);
                      ex.printStackTrace();
                      System.exit(0);
                    } catch (IllegalAccessException ex) {
                      SimUtilities.showError("Probing error", ex);
                      ex.printStackTrace();
                      System.exit(0);
                    }
                  }
                };
                SwingUtilities.invokeLater(specDisplay);
              }
            }
          }
        //}
      }
      }); */
  }

  /**
   * Sets the background color for this display
   *
   * @param c the background color
   */
  public void setBackground(Color c) {
    super.setBackground(c);
    painter.setBackgroundColor(c);
  }

  /**
   * Adds a Displayable to the list of displayables that are displayed when
   * {@link #updateDisplay() updateDisplay} is called.
   *
   * @param display the displayable to add
   * @param name the name of the displayable (is shown under the view menu)
   */
  public void addDisplayable(Displayable display, String name) {
    addDisplay(display, name);
    painter.addDisplayable(display);
  }

  /**
   * Removes the specified Displayable from the list of displayables to
   * display
   *
   * @param display the displayable to remove
   */
  public void removeDisplayable(Displayable display) {
    painter.removeDisplayable(display);
  }

  /**
   * Adds the specified Probeable to the list of probeables.
   *
   * @param probeable the probeable to add
   */
  public void addProbeable(Probeable probeable) {
    probeables.addElement(probeable);
  }

  /**
   * Removes the specified Probeable from the list of probeables.
   *
   * @param probeable the probeable to remove
   */
  public void removeProbeable(Probeable probeable) {
    probeables.remove(probeable);
  }

  /**
   * Adds the specified object to the displayables list and the probeables
   * list.
   *
   * @param o the object to add to the lists
   * @param name the name of the displayable to add. Name will be shown
   * on the view menu
   * @exception java.lang.IllegalArgumentException if the specified object
   * is not a probable and a displayable
   */
  public void addDisplayableProbeable(Displayable o, String name) {
    if (o instanceof Probeable) {
      probeables.addElement(o);
      painter.addDisplayable(o);
      addDisplay(o, name);
    } else {
      throw new IllegalArgumentException("Object is not a Probleable and a Displayable");
    }
  }

  /**
   * Updates the display. Painting all the displayables to the screen. This
   * method is typically added to the schedule with an interval of however
   * often the user wants the display to refresh.
   */
 public void updateDisplay() {
    // need some polymorphic solution to this instanceof test
   // if (painter instanceof RemotePainter) {
     // painter.paint(null);
   // } else if (needsUpdate) {
    //repaint();
    //}

    if (needsUpdate) {
      painter.paint(getGraphics());
    }
  }

  /**
   * Paints this surface.
   */
  public void paint(Graphics g) {
    painter.paint(g);
  }

  private void addDisplay(Displayable d, String name) {
    ArrayList infoList = d.getDisplayableInfo();
    for (int i = 0; i < infoList.size(); i++) {
      DisplayInfo info = (DisplayInfo)infoList.get(i);
      String menuText = info.getMenuText();
      if (menuText.length() == 0) {
        menuText = name;
      } else {
        menuText = name + " " + menuText;
      }

      if (displays.containsKey(menuText)) {
        throw new IllegalArgumentException("Display Surface already contains a menu item with this menu text");
        //System.exit(0);
      }

      JCheckBoxMenuItem item = new JCheckBoxMenuItem(menuText, true);
      item.addActionListener(viewAction);
      item.setActionCommand(menuText);
      menu.add(item);
      displays.put(menuText, info);
    }
  }

  private Action viewAction = new AbstractAction("") {
    public void actionPerformed(ActionEvent evt) {
      JCheckBoxMenuItem item = (JCheckBoxMenuItem)evt.getSource();
      DisplayInfo info = (DisplayInfo)displays.get(item.getActionCommand());
      ViewEvent event = new ViewEvent(this, info.getId(), item.isSelected());
      info.getDisplayable().viewEventPerformed(event);
      repaint();
    }
  };

  /**
   * Displays this DisplaySurface, that is, makes it visible on the screen in
   * a JFrame etc.
   */
  public void display() {
    f = new JFrame(name);
    f.getContentPane().setLayout(new BorderLayout());
    f.getContentPane().add(this, BorderLayout.CENTER);
    f.getRootPane().setDoubleBuffered(false);
    f.addWindowListener(dsWindowAdapter);
    JMenuBar bar = new JMenuBar();
    bar.add(menu);
    f.setJMenuBar(bar);
    f.pack();
    f.setVisible(true);
  }

  /**
   * Dispose this DisplaySurface
   */
  public void dispose() {
    painter.dispose();
    if (f != null) {
      f.dispose();
      f = null;
    }
    System.gc();
  }

  /**
   * Sets the snapshot file name.
   *
   * @param fileName then file name to record to image to.
   */
  public void setSnapshotFileName(String fileName) {
    snapshotFile = fileName;
  }

  /**
   * Takes a snapshot of the current displayed image and writes it to
   * the file specified by setSnapshotFileName(String fileName). The name of
   * the file has the tickcount appended to it, as well as an appropriate
   * extension. For example, a fileName of 'SugarScape', and calling
   * takeSnapShot every 25 ticks would generate files like the following:
   * SugarScape25.gif, SugarScape50.gif, and so on.
   *
   */

  /**
   * Gets the preferred size of this DisplaySurface.
   */

  public Dimension getPreferredSize() {
    return this.getSize();
  }

  /**
   * Sets the size of this DisplaySurface.
   *
   * @param width the new width
   * @param height the new height
   */
  public void setSize(int width, int height) {
    super.setSize(width, height);
    if (f != null)
      f.pack();
  }

  /**
   * Repaints the display on a pause or a stop event. Consequently, adding
   * a DisplaySurface as SimEventListener to a SimModel causes the display
   * to update whenever a simulation run is paused or stopped.
   */
    /*
  public void simEventPerformed(SimEvent evt) {
    int id = evt.getId();
    if (id == SimEvent.PAUSE_EVENT || id == SimEvent.STOP_EVENT) {
      //System.out.println("updating display");
      //updateDisplay();
      repaint();
    }
  }
    */
}


