// Swarm library. Copyright (C) 1996 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

// Generic support for 3d discrete lattices.
// Each point in the lattice stores either an id or a integer.
// Ie: 32 bits per point. This is not 100% general, but is the best
// compromise I know of.

// See FUTURE-DESIGN for notes on where this is going.

#import <swarmobject.h>

@interface Discrete3d: SwarmObject {
@public
  int xsize, ysize, zsize;
  id * lattice;
  int * planeOffsets, * columnOffsets;
}

-setSizeX: (int) x Y: (int) y Z: (int) z;
-createEnd;
-makeOffsets;
-(id *)allocLattice;

-(int) getSizeX;
-(int) getSizeY;
-(int) getSizeZ;

-getObjectAtX: (int) x Y: (int) y Z: (int) z;
-(int) getValueAtX: (int) x Y: (int) y Z: (int) z;

-putObject: anObject atX: (int) x Y: (int) y Z: (int) z;
-putValue: (int) v atX: (int) x Y: (int) y Z: (int) z;

-fastFillWithValue: (int) aValue ;
-fastFillWithObject: anObj ;

-fillWithValue: (int) aValue ;
-fillWithObject: anObj ;

-(id *) getLattice;
-(int *) getOffsets;
-(int *) getColumnOffsets;
-(int *) getPlaneOffsets;

@end

/* fast macro to access lattice array. Use this cautiously.  We define
 * this here to allow library authors to get around the getFooAtX:Y:
 * and setFooAtX:Y: methods. This plays havoc with inheritance, of
 * course.
 *
 * The underlying function here is:
 *
 *   *(p + z(nX)(nY) + y(nX) + x)
 *
 *   where
 *          * => pointer dereference
 *          p => pointer to the address where the vector begins
 *      x,y,z => coordinates of the point of interest
 *         nX => number of rows in the vector
 *         nY => number of columns in the vector
 *         nZ => number of planes in the vector
 *
 * So, what one ends up with is an offset that consists of a number of
 * areas (planes) added to a number of distances (columns) added to a
 * number of elements.  To generalize to four dimensions would involve
 * adding an offset consisting of volumes + planes + distances +
 * elements.  And so on.  */
#define discrete3dSiteAt(l, planeOffsets, columnOffsets, x, y, z) ((l) + (planeOffsets)[(z)] + columnOffsets[(y)] + (x))
