// Swarm library. Copyright (C) 1996 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.
#define __USE_FIXED_PROTOTYPES__  // for gcc headers

#import <stdlib.h>
#import <space/Discrete3d.h>
#import <string.h>

// Note - this code assumes that ints can be written in where ids have been
// allocated. It uses casts to do this, and I think is portable to all
// modern architectures.

@implementation Discrete3d

-setSizeX: (int) x Y: (int) y Z: (int) z {
  if (lattice)
    [InvalidArgument raiseEvent: "You cannot reset the grid size after creation.\n"];
  xsize = x;
  ysize = y;
  zsize = z;
  return self;
}

-createEnd {
  if (xsize <= 0 || ysize <= 0 || zsize <= 0)
    [InvalidCombination raiseEvent: "invalid size in creation of Discrete2d\n"];
  lattice = [self allocLattice];
  [self makeOffsets];

  return self;
}

-(id *) allocLattice {
  void * p;
  p = [[self getZone] alloc: xsize * ysize * zsize * sizeof(id)];
  memset(p, 0, xsize * ysize * zsize * sizeof (id));
  return p;
}

-makeOffsets {
  int i;

  // precalculate offsets based on the y and z coordinate. This lets
  // us avoid arbitrary multiplication in array lookup.
  planeOffsets = [[self getZone] alloc: zsize * sizeof(*planeOffsets)];
  columnOffsets = [[self getZone] alloc: ysize * sizeof(*columnOffsets)];

  for (i = 0; i < zsize; i++)
    planeOffsets[i] = i * (xsize*ysize);

  for (i = 0; i < ysize; i++)
    columnOffsets[i] = xsize * i;

  return self;
}

-(int) getSizeX {
  return xsize;
}

-(int) getSizeY {
  return ysize;
}

-(int) getSizeZ {
  return zsize;
}

-getObjectAtX: (int) x Y: (int) y Z: (int) z {
  return *discrete3dSiteAt(lattice, planeOffsets, columnOffsets, x, y, z);
}

-(int) getValueAtX: (int) x Y: (int) y Z: (int) z {
  return (int) *discrete3dSiteAt(lattice, planeOffsets, columnOffsets, x, y, z);
}

-putObject: anObject atX: (int) x Y: (int) y Z: (int) z {
  *discrete3dSiteAt(lattice, planeOffsets, columnOffsets, x, y, z) = anObject;
  return self;
}

-putValue: (int) v atX: (int) x Y: (int) y Z: (int) z {
  *discrete3dSiteAt(lattice, planeOffsets, columnOffsets, x, y, z) = (id) v;
  return self;
}

-fastFillWithValue: (int) aValue {
  int i, lcount ;
  
  lcount = xsize * ysize * zsize ;

  for(i = 0 ; i < lcount ; i++)
    lattice[i] = (id) aValue ;

  return self ;
}

-fastFillWithObject: anObj {
  int i, lcount ;
  
  lcount = xsize * ysize * zsize ;

  for(i = 0 ; i < lcount ; i++)
    lattice[i] = anObj ;

  return self ;
}

-fillWithValue: (int) aValue {
  unsigned x, y, z;

  for (y = 0; y < ysize; y++)
    for (x = 0; x < xsize; x++)
      for (z = 0; z < zsize; z++)
	[self putValue: aValue atX: x Y: y Z: z];

  return self;
}

-fillWithObject: anObj {
  unsigned x, y, z;

  for (y = 0; y < ysize; y++)
    for (x = 0; x < xsize; x++)
      for (z = 0; z < zsize; z++)
	[self putObject: anObj atX: x Y: y Z: z];

  return self;
}

-(id *) getLattice {
  return lattice;
}

-(int *) getOffsets {
  return columnOffsets;
}

-(int *) getColumnOffsets {
  return columnOffsets; 
}

-(int *) getPlaneOffsets {
  return planeOffsets; 
}

@end
