//Copyright Simon Tokumine 2003. Freely distributable under the GNU General Public Licence

/**
 * @author 	S. Tokumine
 * @date	12-July-2003
 * @time	17:04:50
 * @file	ScoreMapFactory.java
 * 
 * @desc 	Generates 2D large Arrays of fitness scores
 * 			Responisble for n-space mapping and drawing initial landscape.
 *
 */
public class ScoreMapFactory {

	private int smallArraySize = 32;
	private int arraySizeX = 160;
	private int arraySizeY = 288;
	private int bufferPixels = 2;
	int moveUp = 0;
	int moveDown = 0;
	int moveLeft = 0;
	int moveRight = 0;
	SlidePanel slidePanel;
	SSMPanel sSMPanel;

	private int arraylengthX = arraySizeX + bufferPixels;
	private int arraylengthY = arraySizeY + bufferPixels;
	private double[][] fitArray = new double[arraylengthY][arraylengthX];
	private int sSMatrix[][] = { // search space matrix
		{ 21, 29, 23, 31 }, {
			5, 13, 7, 15 }, {
			17, 25, 19, 27 }, {
			1, 9, 3, 11, }, {
			20, 28, 22, 30, }, {
			4, 12, 6, 14 }, {
			16, 24, 18, 26 }, {
			0, 8, 2, 10 }
	};
	//matrix used for interpolation is larger by 1 all arounf
	private int sSMatrixI[][] =
		new int[sSMatrix.length
			+ bufferPixels][sSMatrix[1].length
			+ bufferPixels];

	/**
	 * constructor fills in fitlist, and initialises/fills 2d arrays
	 */
	public ScoreMapFactory() {
		super();

		//create the gene matrix with proper sides ready for interpolation.
		createMatrix();

		//initialise to 0 and create large 2d array of heights
		for (int i = 0; i < arraySizeY; i++) {
			for (int j = 0; j < arraySizeX; j++) {
				fitArray[i][j] = 0;
			}
		}
	}

	public void setSSMPanel(SSMPanel sSMPanel_in) {
		sSMPanel = sSMPanel_in;
	}

	//increments the movement counters (to buffer movement
	//and tests to see if should fire 
	//alter the "4" to increase/ddecrease sensitivity
	public void moveUp() {
		moveDown -= 4;
		moveUp += 4;
		fireTest();
	}

	public void moveDown() {
		moveUp -= 4;
		moveDown += 4;
		fireTest();
	}

	public void moveLeft() {
		moveRight -= 4;
		moveLeft += 4;
		fireTest();

	}

	public void moveRight() {
		moveLeft -= 4;
		moveRight += 4;
		fireTest();
	}

	//resets movemenet counters
	public void setZero() {
		moveUp = 0;
		moveDown = 0;
		moveLeft = 0;
		moveRight = 0;

	}

	//firetest tests to see if mouse has moved
	//enough to warrant a SSM shift
	private void fireTest() {
		if (moveUp >= 32) {
			moveUp = 0;
			moveDown = 0;
			shiftUp();
			sSMPanel.createMatrix();
			slidePanel.printOut();
		}

		if (moveDown >= 32) {
			moveDown = 0;
			moveUp = 0;
			shiftDown();
			sSMPanel.createMatrix();
			slidePanel.printOut();
		}

		if (moveLeft >= 32) {
			moveLeft = 0;
			moveRight = 0;
			shiftLeft();
			sSMPanel.createMatrix();
			slidePanel.printOut();
		}

		if (moveRight >= 32) {
			moveRight = 0;
			moveLeft = 0;
			shiftRight();
			sSMPanel.createMatrix();
			slidePanel.printOut();
		}

	}

	//shifts matrix up 1
	public void shiftUp() {

		int temp[] = new int[sSMatrix[0].length];

		//move top line of matrix to temp file
		System.arraycopy(sSMatrix[0], 0, temp, 0, sSMatrix[0].length);

		//shift up rest of search space matrix
		for (int x = 0; x < sSMatrix[0].length; x++) {
			for (int y = 1; y < sSMatrix.length; y++) {
				sSMatrix[y - 1][x] = sSMatrix[y][x];
			}
		}

		//copy temp array to bottom
		System.arraycopy(
			temp,
			0,
			sSMatrix[sSMatrix.length - 1],
			0,
			temp.length);

		//update matrix in use
		createMatrix();
	}

	//	shifts matrix down 1
	public void shiftDown() {
		int temp[] = new int[sSMatrix[0].length];

		//move bottom line of matrix to temp file
		System.arraycopy(
			sSMatrix[sSMatrix.length - 1],
			0,
			temp,
			0,
			sSMatrix[0].length);

		//shift down rest of search space matrix
		for (int x = 0; x < sSMatrix[0].length; x++) {
			for (int y = sSMatrix.length - 2; y >= 0; y--) {
				sSMatrix[y + 1][x] = sSMatrix[y][x];
			}
		}

		//copy temp array to top
		System.arraycopy(temp, 0, sSMatrix[0], 0, temp.length);

		//update matrix in use
		createMatrix();
	}

	//	shifts matrix left 1
	public void shiftLeft() {
		int temp[] = new int[sSMatrix.length];

		//move left line to temp
		for (int i = 0; i < sSMatrix.length; i++) {
			temp[i] = sSMatrix[i][0];
		}

		//Shift rest of ssm left
		for (int x = 1; x < sSMatrix[0].length; x++) {
			for (int y = 0; y < sSMatrix.length; y++) {
				sSMatrix[y][x - 1] = sSMatrix[y][x];
			}
		}

		//copy temp array to right
		for (int i = 0; i < sSMatrix.length; i++) {
			sSMatrix[i][sSMatrix[0].length - 1] = temp[i];
		}

		//update matrix in use
		createMatrix();

	}

	//	shifts matrix right 1
	public void shiftRight() {
		int temp[] = new int[sSMatrix.length];

		//move right line to temp
		for (int i = 0; i < sSMatrix.length; i++) {
			temp[i] = sSMatrix[i][sSMatrix[0].length - 1];
		}

		//Shift rest of ssm right
		for (int x = sSMatrix[0].length - 2; x >= 0; x--) {
			for (int y = 0; y < sSMatrix.length; y++) {
				sSMatrix[y][x + 1] = sSMatrix[y][x];
			}
		}

		//copy temp array to left
		for (int i = 0; i < sSMatrix.length; i++) {
			sSMatrix[i][0] = temp[i];
		}

		//update matrix in use
		createMatrix();

	}

	//link in the slide panel
	public void setSlider(SlidePanel slidePanel_in) {
		slidePanel = slidePanel_in;
	}

	/**
	 * This copies the matrix as presented above, and copies it to another 2d array with
	 * the correct surrounding cell types. Ready for using as a reference for the
	 * interpolation map.
	 */
	private void createMatrix() {
		//copy sSMatrux to sSMatrixI and also fill in edges.

		//left/right/centre
		for (int i = 1; i < sSMatrixI.length - 1; i++) {
			//left
			sSMatrixI[i][0] = sSMatrix[i - 1][sSMatrix[i - 1].length - 1];
			//right
			sSMatrixI[i][sSMatrixI[i].length - 1] = sSMatrix[i - 1][0];
			//centre
			for (int j = 1; j <= sSMatrix[1].length; j++) {
				sSMatrixI[i][j] = sSMatrix[i - 1][j - 1];
			}
		}
		//top/bottom
		for (int i = 0; i < sSMatrixI[i].length; i++) {
			//top
			sSMatrixI[0][i] = sSMatrixI[sSMatrixI.length - 2][i];
			//bottom
			sSMatrixI[sSMatrixI.length - 1][i] = sSMatrixI[1][i];
		}
	}

	/**
	 * This method creates the large 2d array of heights for the image. 
	 * it creates a seperate 2d bell curve for each gene based upon it's 
	 * fitness value, and stitches them all together according to the SSM.
	 */
	private double[][] createLargeArray(double[] fitList) {
		createPoints(fitList);
		interpolateVert();
		interpolateHoriz();
		return toSquare();
	}

	//enalrge rectangualr grid to square - to the same size x as the y
	//also, remove left and top row of 32 pixel cells
	public double[][] toSquare() {
		int tX = arraySizeY + bufferPixels - smallArraySize;
		int tY = arraySizeY + bufferPixels - smallArraySize;
		double[][] temp = new double[tY][tX];

		for (int i = 0; i < tY; i++) {
			for (int j = 0; j < tX; j++) {
				temp[i][j] = 0;
			}
		}
		
		// calc offset	
		int start =
			(((arraySizeY + bufferPixels - smallArraySize)
				- (arraySizeX + bufferPixels - smallArraySize)))
				/ 2;

		//copies array to square, and clips off the top and right hand side duplicates
		for (int x = 0;
			x < arraySizeX + bufferPixels - smallArraySize;
			x++) { //used to be 0
			for (int y = 32; y < arraySizeY + bufferPixels; y++) {
				temp[y - smallArraySize][x + start - 1] = fitArray[y][x];
			} // used to have no -smallArraySize
		}
		return temp;
	}

	//creates shromasome sample points through the 2D array
	private void createPoints(double[] fitList) {
		for (int gene = 0; gene < fitList.length; gene++) {
			for (int x = 0; x < sSMatrixI[1].length; x++) {
				for (int y = 0; y < sSMatrixI.length; y++) {
					if (sSMatrixI[y][x] == gene) {
						//for fitness image
						fitArray[y * smallArraySize][x * smallArraySize] =
							fitList[gene];
						fitArray[y * smallArraySize][x * smallArraySize + 1] =
							fitList[gene];
						fitArray[y * smallArraySize + 1][x * smallArraySize] =
							fitList[gene];
						fitArray[y * smallArraySize
							+ 1][x * smallArraySize
							+ 1] =
							fitList[gene];
					}
				}
			}
		}
	}

	//interpolates down the y axis
	private void interpolateVert() {
		for (int x = 0; x < sSMatrixI[1].length; x++) { 
			for (int y = 0; y < sSMatrixI.length - 1; y++) {
				interpolateLine(y * smallArraySize + 1, x * smallArraySize + 1);
			}
		}
	}

	//interpolate along a line
	private void interpolateLine(int topY, int topX) {
		int botY = topY + smallArraySize; 
		double from = fitArray[topY][topX];
		double to = fitArray[botY][topX];
		double increment = (to - from) / (smallArraySize);

		for (int y = topY + 1; y < botY; y++) {
			fitArray[y][topX] = fitArray[y - 1][topX] + increment;
		}
	}

	//interpolate along the X axis
	private void interpolateHoriz() {
		for (int x = 0; x < sSMatrixI[1].length - 1; x++) {
			for (int y = 0; y < sSMatrixI.length - 1; y++) {
				interpolateCentre(
					y * smallArraySize + 1,
					x * smallArraySize + 1);
			}
		}
	}

	//fill in the centre
	private void interpolateCentre(int topY, int topX) {
		double from;
		double to;
		for (int y = topY; y < topY + smallArraySize; y++) {
			from = fitArray[y][topX];
			to = fitArray[y][topX + smallArraySize];

			double increment = (to - from) / (smallArraySize);

			for (int x = topX + 1;
				x < topX + smallArraySize;
				x++) { 
				fitArray[y][x] = fitArray[y][x - 1] + increment;
			}
		}
	}

	//print out methods
	private void printLarge() {
		for (int i = 0; i < fitArray.length; i++) {
			System.out.print("Row: " + i + ") ");
			for (int j = 0; j < fitArray[i].length; j++) {
				System.out.print(fitArray[i][j] + " ");
			}
			System.out.println("");
		}
		System.out.println("");
		System.out.println("end");
	}

	public void printFirst() {
		for (int i = 0; i < sSMatrix.length; i++) {
			for (int j = 0; j < sSMatrix[i].length; j++) {
				System.out.print(sSMatrix[i][j]);
			}
			System.out.println("");
		}
		System.out.println("");
	}

	public void printSecond() {
		for (int i = 0; i < sSMatrixI.length; i++) {
			for (int j = 0; j < sSMatrixI[i].length; j++) {
				System.out.print(sSMatrixI[i][j]);
			}
			System.out.println("");
		}
	}

	//accessor methods for the scoremap
	public double[][] get2DArray(double[] inputArray) {
		return createLargeArray(inputArray);
	}

	//accessor methods for the x size
	public int getArraySizeX() {
		return arraySizeY - smallArraySize;
		//used to be arraySizeX, but now returns large square
	}

	//accessor methods for the y size
	public int getArraySizeY() {
		return arraySizeY - smallArraySize; //formerlly no -small arraysize
	}

	//accessor methods for the SSM
	public int[][] getSSM() {
		return sSMatrix;
	}

}