//Copyright Simon Tokumine 2003. Freely distributable under the GNU General Public Licence

import java.io.File;
import java.io.IOException;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * @author 	S. Tokumine
 * @date	20-Jun-2003
 * @time	12:48:50
 * @file	XMLReader.java
 * 
 * @desc  SAX XML reader app for the EPD log.xml file
 *
 */
public class XMLReader extends DefaultHandler {

	SliceList sliceList; //list of timeslices to add too

	int no_strategies = 32;
	int timeslice;
	int datatype = 0; // 0 = type, 1 = freq, 2 = interactionlength, 3 = aapayoff, 4 = abpayoff, 5 = bapayoff, 6 = bbpayoff
	int typeNo;

	int[] freqList = new int[no_strategies]; //could be an arraylist

	//CONSTRUCTOR SETS UP PARSER
	public XMLReader(
		File fileIn,
		SliceList sliceList_in,
		StatusBar statusBar_in) {
		super();

		sliceList = sliceList_in;
		StatusBar statusBar = statusBar_in;

		try {
			// use the JAXP API to instantiate validating SAXParser in a vendor neutral way
			SAXParserFactory factory = SAXParserFactory.newInstance();
			factory.setValidating(true);

			SAXParser xParser = factory.newSAXParser();

			// invoke the instantiated parser with the input XML document
			xParser.parse(fileIn, this);
			statusBar.setStatus("Parsed: " + fileIn.getName());

		} catch (IOException xEx) {
			// error reading the file
			statusBar.setStatus(
				"Error reading the file. . Please make sure it is in E3D XML format.");
			//System.exit(1);
		} catch (SAXException xEx) {
			// error parsing the document
			statusBar.setStatus(
				"Error parsing the document. Please make sure it is in E3D XML format.");
			//System.exit(1);
		} catch (ParserConfigurationException xEx) {
			// error parsing the document
			statusBar.setStatus(
				"Error parsing the document. Please make sure it is in E3D XML format.");
			//System.exit(1);
		}
	}

	/**
	 * AT EVERY timeslice element DO THIS - 
	 * populate a vector list of objects, 1 per timeline. :
	 * Each object will contain a timeslice worth of data 
	 * in the form of fitness per genecombination.. 
	 */

	/**
	 * This method looks for the start of the elements
	 */
	public void startElement(
		String namespaceURI,
		String localName,
		String qName,
		Attributes atts)
		throws SAXException {
		super.startElement(namespaceURI, localName, qName, atts);

		String eName = localName;
		if ("".equals(eName))
			eName = qName; //not namespaceaware

		if (eName.equalsIgnoreCase("timeslice")) {
			if (atts != null) {
				// crop off "t" from attributes, and passs timeslice value along.
				String tempString = atts.getValue(0);

				Integer temp =
					new Integer(tempString.substring(1, tempString.length()));

				timeslice = temp.intValue();
			}
		}

		if (eName.equalsIgnoreCase("type")) {
			datatype = 0;
		}

		if (eName.equalsIgnoreCase("sfreq")) {
			datatype = 1;
		}
		if (eName.equalsIgnoreCase("interactionlength"))
		{
			datatype = 2;
		}
		if (eName.equalsIgnoreCase("aapayoff"))
		{
			datatype = 3;
		}
		if (eName.equalsIgnoreCase("abpayoff"))
		{
			datatype = 4;
		}
		if (eName.equalsIgnoreCase("bapayoff"))
		{
			datatype = 5;
		}
		if (eName.equalsIgnoreCase("bbpayoff"))
		{
			datatype = 6;
		}
	}

	/**
	 * When the closing timeslice element is found, add the timeslice
	 * to the slice list by passing in the time (timeslice) and the data (freqList).
	 */
	public void endElement(String namespaceURI, String sName, // simple name
	String qName // qualified name
	) throws SAXException {
		if (qName.equalsIgnoreCase("timeslice")) {
			sliceList.addSlice(freqList, timeslice);
		}
	}

	public void characters(char buf[], int offset, int len)
		throws SAXException {

		String s = new String(buf, offset, len);
		Integer tempint;
		Double tempdouble;

		switch (datatype)
		{
			case 0:
				tempint = new Integer(s);
				typeNo = tempint.intValue();
				break;
			case 1:
				tempint = new Integer(s);
				freqList[typeNo] = tempint.intValue();
				break;
			case 2:
				tempdouble = new Double(s);
				Payoffs.mGameLength = tempdouble.doubleValue();
				break;
			case 3:
				tempint = new Integer(s);
				Payoffs.mReward = tempint.intValue();
				break;
			case 4:
				tempint = new Integer(s);
				Payoffs.mSuckersPayoff = tempint.intValue();
				break;
			case 5:
				tempint = new Integer(s);
				Payoffs.mTemptation = tempint.intValue();
				break;
			case 6:
				tempint = new Integer(s);
				Payoffs.mPunishment = tempint.intValue();
				break;
		}
		if (datatype == 6)
		{
			Timeslice.payOffFact.calculatePayoffs();
			for (int l = 0; l < sliceList.size(); l++)
			{
				Timeslice slice = sliceList.getSlice(l);
				slice.populateList();
			}
		}
	}
}
