/*

FEARLUS Model 0-2 Swarm Implementation

Gary Polhill, Macaulay Land Use Research Institute, Aberdeen, UK

Hack of EZBin to do bar charts.

EZBin is Copyright (C) 1996-1998 Santa Fe Institute.

*/

#import <simtoolsgui.h>
#import <simtools.h> // OutFile

#import <stdlib.h> // free
#import <misc.h> // xmalloc
#import <math.h> // sqrt

#import <objc/objc-api.h>
#import "EZBar.h"
#import <string.h>


@implementation EZBar

PHASE(Creating)

/****
  
  These methods are all pretty much copied straight out of EZ Bin

  ****/

+createBegin: aZone {
  EZBar *anObj;
  
  anObj = [super createBegin: aZone];
  anObj->fileOutput = 0;
  anObj->graphics = 1;
  anObj->showAverage = 1;
  anObj->theTitle = NULL;
  anObj->filename = NULL;
  anObj->xLabel = NULL;
  anObj->yLabel = NULL;
  anObj->barColourmap = NULL;
  anObj->barLabels = 0;
  return anObj;
}

-setGraphics: (BOOL)state {
  graphics = state;
  return self;
}

-setFileOutput: (BOOL)state {
  fileOutput = state;
  return self;
}

-setTitle: (const char *)aTitle {
  theTitle = aTitle;
  if(!filename) {
    filename = aTitle;		// Default filename for file output is the
				// title of the graph
  }
  return self;
}

-setCollection: aCollection {
  collection = aCollection;
  return self;
}

-setProbedSelector: (SEL) aSel {
  probedSelector = aSel;
  return self;
}

/****

  Now we start to get some of the more serious changes

  ****/

/*

setOutputFilename:

Allow the possibility to set a filename for the output of this graph to be
a non-default value

*/

-setOutputFilename: (const char *)aFilename {
  filename = aFilename;
  return self;
}

/*

setColourSelector:andColourmap:

If each bar will have a unique colour, then each object in the collection must
have a method to return that colour. They should return the colour as an index
on the colourmap. An alternative way to do this would be to have an array of
colours, one for each bar, and you could do setColourOfBar: barnumber to:
colournumber. But then there'd be hassle about when to allocate the array, how
to make sure you weren't setting the colour before you allocated the array,
etc. etc. No. A further problem would be having to separate setting the colour
selector from setting the colourmap. And that just wouldn't do.

*/

-setColourSelector: (SEL)aSel andColourmap: (DBColormap *)cmap {
  colourSelector = aSel;
  barColourmap = cmap;
  return self;
}

/*

setLabelSelector:

Provide a method to give a selector for each object to give itself a label on
the bar chart.

*/

-setLabelSelector: (SEL)aSel {
  labelSelector = aSel;
  barLabels = 1;
  return self;
}

/*

createEnd

Check everything has been set up. Set up the histogram. Set up the colours, if
we've been given coloury stuff. Allocate space for the distribution.

*/

-createEnd {
  int i;
  char **colours = NULL;
  char **labels = NULL;
  id iter, obj;
  id <String> colstr = nil;

  barZone = [Zone create: [self getZone]];

  // Check that all is present and correct
  
  if(collection == nil) {
    [InvalidCombination raiseEvent: "EZBin created without a collection\n"];
  }
  if(!theTitle) {
    [InvalidCombination raiseEvent: "EZBin without a title!!\n"]; 
  }

  // Call createEnd of the superclass

  [super createEnd];

  // Set up the histogram

  binNum = [collection getCount];
                                // The number of bars to draw is equal to the
				// number of items in the collection as called
				// here.
  distribution = (double *)xmalloc(binNum * sizeof(double));
				// Allocate memory for the current bar values
  averages = (double *)xmalloc(binNum * sizeof(double));
				// Allocate memory for the average bar values
  originalCollection = [Array create: barZone setCount: binNum];
				// Allocate memory for an array to store the
				// initial members of the collection.
  if(barColourmap != nil) {
    colours = (char **)xmalloc(binNum * sizeof(char *));
				// Allocate space for the array of colours
    colstr = [String create: barZone setC: "#FFFFFF"];
  }
  if(barLabels) {
    labels = (char **)xmalloc(binNum * sizeof(char *));
                                // Allocate space for the array of bar labels
  }
  if(!(distribution && averages &&
       (!barColourmap || colours) && (!barLabels || labels))) {
    [OutOfMemory raiseEvent];
  }

  // Iterate through the collection to (a) keep a backup of all items in the
  // collection, (b) set up the array of colours, if required, and (c) set
  // up the array of labels if required.

  for(i = 0, iter = [collection begin: barZone], [iter next];
      (i < binNum) && ([iter getLoc] == Member);
      [iter next], i++) {
    obj = [iter get];
    [originalCollection atOffset: i put: obj];
				// Make a backup of the original objects in
				// the collection
    
    if(barColourmap != nil) {	// Set up the pointers to colours
      Color c;

      c = (Color)(* ((int (*) (id, SEL, ...))
		     [obj methodFor: colourSelector])) (obj, colourSelector);
				// If this works it'll be a miracle! I just
				// hope this is the standard way you call a
				// method of an object using a selector
      [barColourmap getColor: c inString: colstr];
				// This is why we need a DBColormap instead of
				// an ordinary one.
      colours[i] = (char *)xmalloc(([colstr getCount] + 1) * sizeof(char));
				// Allocate space for the new colour
      if(!colours[i]) {
	[OutOfMemory raiseEvent];
      }
      strcpy(colours[i], [colstr getC]);
    }
    if(barLabels) {		// Set up the pointers to bar labels
      char *c;

      c = (char *)(* ((char * (*) (id, SEL, ...))
		      [obj methodFor: labelSelector])) (obj, labelSelector);
                                // With a bit of luck, this will get us a
				// string from the object.
      labels[i] = (char *)xmalloc((strlen(c) + 1) * sizeof(char));
				// Allocate space for the label
      if(!labels[i]) {
	[OutOfMemory raiseEvent];
      }
      strcpy(labels[i], c);
    }
  }
  if (colstr) [colstr drop];

  [self reset];
  
  if(graphics) {		// Set up the histogram object that will
				// actually display the graph
    aHisto = [Histogram createBegin: barZone];
    SET_COMPONENT_WINDOW_GEOMETRY_RECORD_NAME (aHisto);
       [aHisto setBinCount: binNum];  
    aHisto = [aHisto createEnd];
    [aHisto setTitle: theTitle];
    if(xLabel && yLabel) {
      [aHisto setAxisLabelsX: xLabel Y: yLabel];
    }
    [aHisto setLabels: (const char * const *)labels count: binNum]; 
    [aHisto setColors: (const char * const *)colours count: binNum];
    [aHisto pack];
    if(!barLabels) {
      [aHisto hideLegend];
    }
    [aHisto setupZoomStack];
    [aHisto setupActiveOutlierMarker];
    [aHisto setupActiveItemInfo];
  }
  if(fileOutput) {
    anOutFile = [OutFile create: barZone withName: filename];
    [anOutFile putString: theTitle];
    [anOutFile putNewLine];
				// Write the title to the output file in case
				// a non-default filename has been used.
  }
  if(barColourmap != nil) {	// Free up the space allocated when processing
				// the colourmap
    for(i = 0; i < binNum; i++) {
      free(colours[i]);
    }
    free(colours);
  }
  if(barLabels) {
    for(i = 0; i < binNum; i++) {
      free(labels[i]);
    }
    free(labels);
  }
  return self;
}

PHASE(Using)

/*

showAverage and showCurrent -- If you create probes for these methods, then
clicking on the button should change the kind of graph it draws.

*/

-showAverage {			// Show an average over time
  showAverage = 1;
  return self;
}

-showCurrent {			// Show the currently obtained values
  showAverage = 0;
  return self;
}

/*

setAxisLabelsX:Y:

This is just as for EZBin -- set the labels for each axis

*/

-setAxisLabelsX: (const char *)xl Y: (const char *)yl {
  xLabel = xl;
  yLabel = yl;
  return self;
}

/*

reset

Reset everything -- the averages, the time counter. Modified from EZBin.

*/

-reset {
  int i;
  
  count = 0;			// Reset the time (for averaging purposes) to 0
  outliers = 0;			// Reset the number of outliers.
  clean = 1;			// Set a flag to say that everything is reset

  for (i = 0; i < binNum; i++) {
    averages[i] = 0.0;		// Reset all the averages
  }
  return self;
}

/*

step

Modified update method for bar charts. First, set outliers to be the number of
items in the current collection that weren't in the original collection when
the EZBar was created. (I don't understand how outliers work in EZBin, so you
won't be able to print the outliers anyway.) Then, loop through the items in
the original collection, and if they are in the current collection, get the
value for each to display in its bar.

Combines in the output method. There doesn't seem to be any point in having
separate methods to update and output EZBars, as is the case for EZBins.

*/

-(void) step {
  id iter, obj;
  char type0 = sel_get_type (sel_get_any_typed_uid 
			     (sel_get_name (probedSelector)))[0];
  int i;

  outliers = 0;
  for(iter = [collection begin: barZone], [iter next];
      [iter getLoc] == Member;
      [iter next]) {
    if(![originalCollection contains: [iter get]]) {
      outliers++;
    }
  }
  [iter drop];
  count++;
  // Initialise the distribution (in case the collection doesn't contain some
  // of the items in the original collection)
  for(i = 0; i < binNum; i++) {
    double v;

    distribution[i] = 0.0;

    // Check that the original collection contains the object.

    obj = [originalCollection atOffset: i];
    if(![collection contains: obj]) {
      continue;
    }
    
    // This next bit apparently gets the value to be plotted for the object
    // using the probedSelector.
    
    if (type0 == _C_DBL) {
      v = (* ((double (*) (id, SEL, ...))
	      [obj methodFor: probedSelector])) (obj, probedSelector);
    }
    else if (type0 == _C_FLT) {
      v = (double)(* ((float (*) (id, SEL, ...))
		      [obj methodFor: probedSelector])) (obj, probedSelector);
    }
    else {
      v = (double)(* ((int (*) (id, SEL, ...))
		      [obj methodFor: probedSelector])) (obj, probedSelector);
    }


    // If we've been reset, then reset all the statistics.

    if(clean) {
      maxval = v;
      minval = v;
      average = v;
      average2 = v * v;
      std = 0.0;
      clean = 0;
    }
    else {		// We haven't been reset -- accumulate the stats.
      if(v < minval) {
	minval = v;
      }  
      if(v > maxval) {
	maxval = v;
      }  
      average = ((average * ((double)((count * binNum) + i - 1))) + v)
	/ ((double) (((count * binNum) + i)));
      
      average2 = ((average2 * ((double)((count * binNum) + i - 1))) + v * v)
	/ ((double) (((count * binNum) + i)));
      
      std = sqrt(average2 - average*average);
    }
  
    distribution[i] = v;
    // Keep a running average of each member of the collection.
    averages[i] = (((((double)count - 1.0) * averages[i]) + distribution[i]) 
		   / (double)count);
  }


  // Draw the graph. In EZBin, this method is separated into two -- update and
  // output. Don't see any point in doing that for EZBar.

  if(graphics) {
/*
  If I understood the documentation, I'd know what to send this method, but I
  don't, so we're not going to deal with the outliers properly.
  */
//      [aHisto setActiveOutlierText: outliers count: count];
// Above line of code stays commented out until I know what it does properly.

    // Send the histogram a set of numbers to draw.
    if(showAverage) {
      [aHisto drawHistogramWithDouble: averages];
				// Draw the average over time for each bar
    }
    else {
      [aHisto drawHistogramWithDouble: distribution];
				// Draw the current value of each bar
    }
  }
  
  if(fileOutput) {
    [anOutFile putDouble: distribution[0]];
    for(i = 1; i < binNum; i++){
      [anOutFile putTab];
      [anOutFile putDouble: distribution[i]];
    }
    [anOutFile putNewLine];
  }
}

/*

The following methods are all accessor methods for the stats, and other bits
and bobs. No changes from EZBin.

*/

-(double *)getDistribution {
  return distribution;
}

-(int)getBarCount {
  return count;
}

-(int)getOutliers {
  return outliers;
}

-(int)getBinNum {
  return binNum;
}

-(double)getMin {
  if (clean)
    {
      [InvalidOperation 
        raiseEvent:
          "Attempted to getMin from a reset EZBin (no data available).\n"];
    }
  return minval;
}

-(double)getMax {
  if (clean)
    {
      [InvalidOperation
        raiseEvent:
          "Attempted to getMax from a reset EZBin (no data available).\n"];
    }
  
  return maxval;
}

-(double)getAverage {
  if (clean)
    {
      [InvalidOperation
        raiseEvent:
          "Attempted to getAverage from a reset EZBin (no data available).\n"];
    }
  return average;
}

-(double)getStd {
  if (clean)
    {
      [InvalidOperation
        raiseEvent:
          "Attempted to getStd from a reset EZBin (no data available).\n"];
    }
  
  return std;
}

/*

drop

Over-ride the drop method to free up the objects created for this graph

*/

-(void)drop {
  free(distribution);
  free(averages);
  [barZone drop];
  if (graphics)
    [aHisto drop];
  if (fileOutput)
    [anOutFile drop];
  
  [super drop];
}

@end
