// RuleMaker.m
//--------------------------------------------------------------------

#import "RuleMaker.h"

@implementation RuleMaker

//====================================================================
//  Part 1 - Methods to receive parameters and data, usually performed
//           during creations of instances before createEnd
//====================================================================

//--------------------------------------------------------------------

- setLengthOfGenoma: (int) gL
{

TRACE4(printf("\n%8X RuleMaker - entry setLengthOfGenoma: %d\n",
             (int) self,gL);)

   lengthOfGenoma = gL;

TRACE4(printf("%8X RuleMaker - exit setLengthOfGenoma %8X\n\n",
             (int) self,(int) self);)

   return self;

}

//--------------------------------------------------------------------

- setTurnoverRate: (float) tR
{

TRACE4(printf("\n%8X RuleMaker - entry setTurnoverRate: %f\n",
             (int) self,tR);)

   turnoverRate = tR;

TRACE4(printf("%8X RuleMaker - exit setTurnoverRate %8X\n\n",
             (int) self,(int) self);)

   return self;

}

//--------------------------------------------------------------------

- setCrossoverRate: (float) cR
{

TRACE4(printf("\n%8X RuleMaker - entry setCrossoverRate: %f\n",
             (int) self,cR);)

   crossoverRate = cR;

TRACE4(printf("%8X RuleMaker - exit setCrossoverRate %8X\n\n",
             (int) self,(int) self);)

   return self;

}

//--------------------------------------------------------------------

- setMutationRate: (float) mR
{

TRACE4(printf("\n%8X RuleMaker - entry setMutationRate: %f\n",
             (int) self,mR);)

   mutationRate = mR;

TRACE4(printf("%8X RuleMaker - exit setMutationRate %8X\n\n",
             (int) self,(int) self);)

   return self;

}

//--------------------------------------------------------------------

- setEvolutionFrequency: (float) eF
{

TRACE4(printf("\n%8X RuleMaker - entry setEvolutionFrequency: %f\n",
             (int) self,eF);)

   evolutionFrequency = eF;

TRACE4(printf("%8X RuleMaker - exit setEvolutionFrequency %8X\n\n",
             (int) self,(int) self);)

   return self;

}

//--------------------------------------------------------------------

- setChildrenFitness: (float) cF
{

TRACE4(printf("\n%8X RuleMaker - entry setChildrenFitness: %f\n",
             (int) self,cF);)

   childrenFitness = cF;

TRACE4(printf("%8X RuleMaker - exit setChildrenFitness %8X\n\n",
             (int) self,(int) self);)

   return self;

}

//--------------------------------------------------------------------

- setUseDeltaFitness: (float) dF
{

TRACE4(printf("\n%8X RuleMaker - entry setUseDeltaFitness: %f\n",
             (int) self,dF);)

   useDeltaFitness = dF;

TRACE4(printf("%8X RuleMaker - exit setUseDeltaFitness %8X\n\n",
             (int) self,(int) self);)

   return self;

}

//--------------------------------------------------------------------

- setPopulationList: pL
{

TRACE4(printf("\n%8X RuleMaker - entry setPopulationList: %8X\n",
             (int) self,(int) pL);)

   populationList = pL;

TRACE4(printf("%8X RuleMaker - exit setPopulationList %8X\n\n",
             (int) self,(int) self);)

   return self;

}

//====================================================================
//  Part 2 - createEnd
//====================================================================

//--------------------------------------------------------------------
//Here we build lists to handle evolution and set to zero statistics
//counters

- createEnd
{

   id obj;

TRACE4(printf("\n%8X RuleMaker - entry createEnd\n",(int) self);)

   workList   = [List create: [self getZone]];
   parentList = [List create: [self getZone]];
   dyingList  = [List create: [self getZone]];
   embryoList = [List create: [self getZone]];
   evolutions = reproductions = crossovers = mutations = nullSteps = 0;

   obj = [super createEnd];

TRACE4(printf("%8X RuleMaker - exit createEnd %8X\n\n",(int) self,
                (int) obj);)

   return obj;

}

//====================================================================
//  Part 3 - Methods to give or print statistics data
//====================================================================

//--------------------------------------------------------------------

- (int) getEvolutions
{

TRACE4(printf("\n%8X RuleMaker - entry getEvolutions\n",
             (int) self);)
TRACE4(printf("%8X RuleMaker - exit getEvolutions %d\n\n",
             (int) self,evolutions);)

   return evolutions;

}

//--------------------------------------------------------------------

- (int) getReproductions
{

TRACE4(printf("\n%8X RuleMaker - entry getReproductions\n",
             (int) self);)
TRACE4(printf("%8X RuleMaker - exit getReproductions %d\n\n",
             (int) self,reproductions);)

   return reproductions;

}

//--------------------------------------------------------------------

- (int) getCrossovers
{

TRACE4(printf("\n%8X RuleMaker - entry getCrossovers\n",
             (int) self);)
TRACE4(printf("%8X RuleMaker - exit getCrossovers %d\n\n",
             (int) self,crossovers);)

   return crossovers;

}

//--------------------------------------------------------------------

- (int) getMutations
{

TRACE4(printf("\n%8X RuleMaker - entry getMutations\n",
             (int) self);)
TRACE4(printf("%8X RuleMaker - exit getMutations %d\n\n",
             (int) self,mutations);)

   return mutations;

}

//--------------------------------------------------------------------

- (int) getNullSteps
{

TRACE4(printf("\n%8X RuleMaker - entry getNullSteps\n",
             (int) self);)
TRACE4(printf("%8X RuleMaker - exit getNullSteps %d\n\n",
             (int) self,nullSteps);)

   return nullSteps;

}

//--------------------------------------------------------------------

- printStatistics
{

TRACE4(printf("\n%8X RuleMaker - entry printStatistics\n",(int) self);)

   printf("\n------------- RuleMaker's statistics ------------\n");
   printf("%d evolution's steps have been performed\n",evolutions);
   printf("with %d reproductions %d crossovers %d mutations\n",
           reproductions,crossovers,mutations);
   printf("-----------------------------------------------------\n\n");

TRACE4(printf("%8X RuleMaker - exit printStatistics %8X\n\n",
              (int) self,(int) self);)

   return self;

}

//====================================================================
//  Part 4 - core: main-line and service-methods
//====================================================================

//--------------------------------------------------------------------
//This part of code contains the main-line of the process, single
//function will be performed by specific methods

- step
{

   int i;

TRACE4(printf("\n%8X RuleMaker - entry step\n",(int) self);)

   if ([self verify] == 1)
   {
      evolutions++;

TRACE6(printf("Evolution n. %d\n",evolutions);)

      [self keepData];

TRACE6(printf("workList contains %d elements\n",
              [workList getCount]);)

TRACE6([workList forEach: M(print)];)

      [self selectBestN: numberOfParents fromList: workList
                 toList: parentList      withTot: totalFitness];

      [self selectBestN: numberOfIndividuals-numberOfParents
            fromList: dyingList toList: workList  withTot: totalFitness];

TRACE6(printf("parentList contains %d elements\n",
              [parentList getCount]);)

TRACE6([parentList forEach: M(print)];)

TRACE6(printf("dyingList contains %d elements\n",
              [dyingList getCount]);)
TRACE6([dyingList forEach: M(print)];)

   //Performing reproduction without shaffling parentList means
   //that better parents (i. e. high fitted parents) are, probably,
   //crossed between theyselves. If you prefere cross parents
   //at random you must enable SHUFFLE macro modifying file Macro.h

SHUFFLE([self shuffle: parentList];)

      [self reproduce];

      for(i=0;i<numberOfParents;i++)
      {

         anInterface = [dyingList atOffset: i];

TRACE6(printf("replacing %8X\n",(int) anInterface);)

        [anInterface setGenoma:  [[embryoList atOffset: i] getGenoma]];
        [anInterface setFitness: [[embryoList atOffset: i] getFitness]];

      }

   }

TRACE4(printf("%8X RuleMaker - exit step %8X\n\n",(int) self,(int) self);)

   return self;

}

//--------------------------------------------------------------------
//Here we verify operational condition about evolution frequency
//if evolutionFrequency == 1 evolution is done everytime
//
//if evolutionFrequency >  1 evolution is done every evolutionFrequency
//                                             times
//if evolutionFrequency is between 0 and 1 evolution is done if a
//   random real number between 0 and 1 is less than evolutionFrequency
//
//if evolutionFrequency == 0 no evolutions are done at all
//
- (int) verify
{

   int i=0;

TRACE4(printf("\n%8X RuleMaker - entry verify\n",(int) self);)


   if (evolutionFrequency > 1)
   {

      nullSteps ++;
      if (nullSteps == evolutionFrequency)
      {

         i = 1;
         nullSteps = 0;

      }

   }

   if ((evolutionFrequency > 0) && (evolutionFrequency < 1))
   {

      if ([uniformDblRand getDoubleWithMin: 0 withMax: 1] <
          evolutionFrequency) i = 1;

   }

   if (evolutionFrequency == 1) i = 1;

TRACE4(printf("%8X RuleMaker - exit verify: %d\n\n",(int) self,i);)

   return i;

}

//--------------------------------------------------------------------
//Here we map data received from the caller into work-variables to
//allow caller doing anything on these data, particularry shuffling
//lists, removing or adding objects and so on. The only action that is
//not allowed consist in dropping objects.

- keepData
{

   float s;
   int i;

TRACE4(printf("\n%8X RuleMaker - entry keepData\n",(int) self);)

   //The number of individuals in the list is computed all times,
   //this way, users can remove or add several individuals between
   //two evolutions.

   [workList removeAll];
   [dyingList removeAll];
   [parentList removeAll];
   freeEmbryo          = [embryoList getCount];
   nextEmbryo          = 0;
   numberOfIndividuals = [populationList getCount];
   totalFitness        = 0;

   //Then let's copy all individuals-Id contained in populationList in
   //the workList and compute the total fitness of the population and
   //the number of parents we need, based upon the turnoverRate value;
   //because we intend handle with couples of parent the number
   //will be reduced to less multiple of 2.

   for(i=0;i<numberOfIndividuals;i++)
   {

      //Usually ruleMaker talk with Interface, if you prefere you can
      //allow talking with Agents enabling AGENT macro in file Macro.h

      anInterface   = (Interface *) [populationList atOffset: i];
AGENT(anAgent       = [populationList atOffset: i];)
AGENT(anInterface   = [anAgent getInterface];)
      s             = [anInterface getFitness];
      totalFitness  = totalFitness + s;
      if ((minFitness > s) || (i == 0)) minFitness = s;
      [workList  addLast: anInterface];
      [dyingList addLast: anInterface];

   }

   if (useDeltaFitness != 1) minFitness = 0;
   totalFitness    = totalFitness - minFitness * numberOfIndividuals;
   numberOfParents = (numberOfIndividuals * turnoverRate);
   numberOfParents = numberOfParents - (numberOfParents % 2);
   if (numberOfParents < 2) numberOfParents = 2;
   if (numberOfIndividuals < 2) numberOfParents = 0;

TRACE4(printf("%8X RuleMaker - exit keepData %8X\n\n",
              (int) self,(int) self);)

   return self;

}

//--------------------------------------------------------------------
//Here we select parents using a roulette-weel algorithm based upon
//fitness value

- selectBestN: (int) n fromList: (id) l0 toList: (id) l1
                        withTot: (float) tF;
{

   int i,j;
   double s,totFitness,randomFitness,sumFitness;

TRACE4(printf("\n%8X RuleMaker - entry selectBestN: %d\n",(int) self,n);)
TRACE4(printf("                           fromList: %8X\n",(int) l0);)
TRACE4(printf("                             toList: %8X\n",(int) l1);)
TRACE4(printf("                            withTot: %f\n",tF);)

   totFitness = tF;
   [l1 removeAll];

   for(i=0;i<n;i++)
   {

      j = 0;
      sumFitness = 0;
      randomFitness = [uniformDblRand getDoubleWithMin: 0
                                               withMax: totFitness];

      do
      {

         anInterface = [l0 atOffset: j];
         s           = [anInterface getFitness] - minFitness;
         sumFitness  = sumFitness + s;
         j++;

      } while (sumFitness < randomFitness);

      [l1 addLast: anInterface];
      [l0  remove: anInterface];
      totFitness = totFitness - s;

   }

TRACE4(printf("%8X RuleMaker - exit selectBestN %8X\n",(int) self,
              (int) self);)
   return self;

}

//--------------------------------------------------------------------
//Here we perform reproduction applying crossover and mutation
//accordingly with parameters received by "set" methods.

- reproduce
{

   Interface * interface0;
   Interface * interface1;
   char      * genoma0;
   char      * genoma1;
   char        newGenoma0[MAXGL];
   char        newGenoma1[MAXGL];
   char        allele[2];
   char        mutantAllele[2];
   float       fitness0,fitness1,work0;
   int         i,j,k,xP;

TRACE4(printf("\n%8X RuleMaker - entry reproduce\n",(int) self);)

   for (i=0;i<numberOfParents/2;i++)
   {

      reproductions++;

      //If you prefere following 4 lines can be reduced to 2 modifying
      //instruction to:
      //
      // genoma0 = [[parentList atOffset: i*2]   getGenoma];
      // genoma1 = [[parentList atOffset: i*2+1] getGenoma];
      //
      //This way you can also remove initial declaration of interface0
      //and interface1, but you must modify an istruction under TRACE6
      //(see the note just before the line) and the compiler will not
      //be able to verifiy if objects in parentList are understanding
      //methods getGenoma

      interface0      = [parentList atOffset: i*2];
      genoma0         = [interface0 getGenoma];
      interface1      = [parentList atOffset: i*2+1];
      genoma1         = [interface1 getGenoma];

      //let's perform crossover accordingly with percentage
      //of crossover contained in crossoverRate parameter.
      //xP is used to store crossover point; if xP = 0 we'll
      //have no crossover, the same if xP = lengthOfGenoma;
      //to avoid this accident let's randomize
      //crossover-point between 1 and lengthOfGenoma-1

      xP = 0;
      if ([uniformDblRand getDoubleWithMin: 0 withMax: 1]
          < crossoverRate)
      {

         crossovers++;
         xP = [uniformIntRand getIntegerWithMin: 1
                                        withMax: lengthOfGenoma-1];

      }

      //If you have removed declarations of interface0 and interface1
      //you can put in theyre place, in the following macro's parameter,
      //genoma0 and genoma1 and remember that addresses aren't the same.

TRACE6(printf("crossing %8X - %8X at %d\n",
               (int) interface0,(int) interface1,xP);)

      for(j=0;j<lengthOfGenoma;j++)
      {

         allele[0] = mutantAllele[0] = genoma0[j];
         allele[1] = mutantAllele[1] = genoma1[j];

         for(k=0;k<2;k++)
         {

            //We've also to perform some mutation

            if ([uniformDblRand getDoubleWithMin: 0 withMax: 1]
                < mutationRate)
            {

TRACE6(printf("allele at offset %d of genoma%d is mutating\n",j,k);)

               mutations++;
               mutantAllele[k] = '0';
               if (allele[k] == '0') mutantAllele[k] = '1';

            }

         }

         if (j<xP)
         {

            newGenoma0[j] = mutantAllele[1];
            newGenoma1[j] = mutantAllele[0];

         }
         else
         {

            newGenoma0[j] = mutantAllele[0];
            newGenoma1[j] = mutantAllele[1];

         }

      }

      if (childrenFitness < 0)
      {

         fitness0 = work0 = [interface0 getFitness];
         fitness1 = [interface1 getFitness];

         fitness0 = (1.0 + childrenFitness) * fitness1 -
                    childrenFitness * fitness0;
         fitness1 = (1.0 + childrenFitness) * work0 -
                    childrenFitness * fitness1;

      }

      else fitness0 = fitness1 = childrenFitness;

      [[self getEmbryo] setGenoma: &newGenoma0[0]
                      withFitness: fitness0];
      [[self getEmbryo] setGenoma: &newGenoma1[0]
                      withFitness: fitness1];

   }

TRACE4(printf("%8X RuleMaker - exit reproduce %8X\n\n",
              (int) self,(int) self);)

   return self;

}

//--------------------------------------------------------------------

- (Embryo *) getEmbryo
{

   Embryo * anEmbryo;

TRACE4(printf("\n%8X RuleMaker - entry getEmbryo\n",(int) self);)

   if (freeEmbryo == nextEmbryo)
   {

      anEmbryo = [Embryo createBegin: [self getZone]];
      anEmbryo = [anEmbryo createEnd];
      [embryoList addLast: anEmbryo];

   }

   else
   {

      anEmbryo = [embryoList atOffset: nextEmbryo];
      nextEmbryo++;

   }

   [anEmbryo setLengthOfGenoma: lengthOfGenoma];

TRACE4(printf("%8X RuleMaker - exit getEmbryo %8X\n\n",
              (int) self,(int) anEmbryo);)

   return anEmbryo;

}

//--------------------------------------------------------------------
//Here we shuffle a list

- shuffle: (id) list
{

   int i,num;

TRACE4(printf("\n%8X RuleMaker - entry shuffle: %8X\n",
               (int) self,(int) list);)

   num = [list getCount];

   for(i=0;i<num;i++)
   [list addLast: [list remove : [list atOffset:
   [uniformIntRand getIntegerWithMin: 0 withMax: num-1]]]];

TRACE4(printf("%8X RuleMaker - exit shuffle %8X\n\n",(int) self,(int) self);)

   return self;

}

//--------------------------------------------------------------------
@end
