#import "Boid.h"
#import <math.h>

#define GRAVITY 9.806650
#define desiredCruisingSpeed 0.2*maxVelocity

@implementation Boid

+ (Boid *) createBegin: aZone {
  int i;
  Boid *obj = [super createBegin: aZone];

  obj->acceleration = [Vector create: aZone]; // initially {0,0}
  obj->flockCenter  = [Vector create: aZone];
  obj->calc = [Vector create: aZone]; 



  for (i=0; i<NUMOBJECTTYPES; i++)
  {

     obj->closestBoid[i] = nil;

  }


  return obj;
}


///////////////////////////////////////////////////
//
// drop
//
////////////////////////////////////////////////////
- (void) drop {

  int i;

  for (i=0; i<NUMOBJECTTYPES; i++) {

      closestBoid[i] = nil;

  }
  
  [acceleration drop];
  [flockCenter  drop];
  [super drop];

}

- createEnd {

  return [super createEnd];

}



//////////////////////////////////////////////////////////
//
// init
//
////////////////////////////////////////////////////////
- init: (int) id : (int) type
      : (id <Vector>) p
      : (id <Vector>) v
      : (double  ) r {

    [super init: id : type : p : v : r];

    maxVelocity = 10.0;
    maxAcceleration = 0.5;
    cruiseDistance = 2.0*radius; // Try to stay at least one bodywidth apart

    return self;

}

///////////////////////////////////////////////////////////////
//
// visible
//
//////////////////////////////////////////////////////////////
- (double) visible: (SimObject *)b {

  double res;
  int dx, dy;

  // find out if the boid b is within our field of view
  [[calculation init: b->position] sub: position];
  res = [calculation getLength];

  if (res > worldX/2) {    // correct for torus world

     [calc init: b->position];

     dx = [calculation getX]; if (dx<0) dx = -dx;
     dy = [calculation getY]; if (dy<0) dy = -dy;
    
     if (dx>worldX/2) {

        if ([calc getX] < [position getX])
            [calc setX: [calc getX] + worldX];
        else
            [calc setX: [calc getX] - worldX];

     } //if dx


    if (dy>worldY/2) {

        if ([calc getY] < [position getY])
	    [calc setY: [calc getY] + worldY];
        else
	    [calc setY: [calc getY] - worldY];

     } //if dy


      [[calculation init: calc] sub: position];
      res = [calculation getLength];
 
  }  //if res

   if (res > cruiseDistance && [calculation angle: velocity] > 1.0471967) { // pi/3 radians is our FOV
    
      res = -1.0; // ignore object

   }

  return res;
}



///////////////////////////////////////////////////
//
// getProbeLength
//
///////////////////////////////////////////////////
- (double) getProbeLength
{
  double maxScale = 5.0;

  // When we're at maxVelocity, scalefactor = maxScale.
  // When our velocity is 0, scalefactor = 1.
  // Linearly scale in between.
  double scaleFactor = ((maxScale-1.0)/maxVelocity)*[velocity getLength]+1.0;

  return 10 * radius * scaleFactor;

}




/////////////////////////////////////////////////
//
// updateVisibilityList
//
/////////////////////////////////////////////////
- updateVisibilityList {

  id <Index> index;
  SimObject *member;
  int i, bType;
  double dist;

  unsigned int flockCount=0;
  double closestDist[NUMOBJECTTYPES];


  //  printf("Boid\tupdateVisibilityList\n");

  index  = [[model getObjectList] listBegin: scratchZone];

  for (i=0; i<NUMOBJECTTYPES; i++)
  {

      closestBoid[i] = nil;
      closestDist[i] = LARGEINT;

  }

  [flockCenter init];

  flockCount = 0;

  for (member = [index next]; [index getLoc] == Member; member = [index next])
  {

      if (member != self && (dist = [self visible: member])>0.0)
      {

          bType = [member getObjectType];


          if(dist < closestDist[bType]) 
          {
               closestDist[bType] = dist;
               closestBoid[bType] = member;
          } 

          if(bType == objectType)
          {
 
	      [flockCenter add: [member getPosition]];
              flockCount++;

          }

      } //if member

  } //for member

  [index drop];


  if(flockCount) 
  {

      [flockCenter div: flockCount];
    

  }
  else 
  {

      [flockCenter init: -1.0: -1.0];

  }

  //  printf("Boid\tupdateVisibilityList done.\n");
  return self;
}


////////////////////////////////////////////////////////
//
// accumulate
//
///////////////////////////////////////////////////////
- (double) accumulate: (id <Vector>)valueToAdd
{
  double newLength;
  if ([valueToAdd getLength] > 1.0) {

      [valueToAdd normalize];

  }

  [acceleration add: valueToAdd];

  newLength = [acceleration getLength];

  if (newLength > 1.0) {

       [acceleration normalize];

  }

  return newLength;
}



///////////////////////////////////////////////////
//
// navigator
//
//////////////////////////////////////////////////
- navigator
{
  double length = 0.0;

  // Calculate the visibility matrix so that visibility computations are
  // much more efficient.
  //  printf("Boid\tnavigator\n");
  [self updateVisibilityList];

  [acceleration init];

  length = [self accumulate: [self collisionAvoidance]];
  if (length < 1.0)
    length = [self accumulate: [self predatorAvoidance]];
  if (length < 1.0)
    length = [self accumulate: [self maintainingCruisingDistance]];
  if (length < 1.0)
    length = [self accumulate: [self wander]];
  if (length < 1.0)
    length = [self accumulate: [self flockCentering]];
  if (length < 1.0)
    length = [self accumulate: [self velocityMatching]];

  // should be done with a swarm and a scedule, no?

  // printf("acceleration:\t"); [acceleration print];
  
  // IMPORTANT:
  // Since the FlockCentering, CollisionAvoidance, and VelocityMatching modules
  // return a vector whose magnitude is a percentage of the maximum acceleration,
  // we need to convert this to a real magnitude before we hand it off to the Flight
  // module. 
  // Remember, maxAcceleration is in terms of a fraction (0.0 to 1.0) of maxVelocity

  [acceleration mult: maxAcceleration * maxVelocity];
  //  printf("Boid\tnavigator\n");
  return self;
}



/////////////////////////////////////////////////
//
// collisionAvoidance
//
/////////////////////////////////////////////////
- (id <Vector>)collisionAvoidance {
  SimObject *obs;
  double mag;

  [calculation init];

  


  if ((obs = closestBoid[OBSTACLE]) != nil) {
    // Obstacle with lowest priority == closest seen
  
    mag = 1.0 - [obs avoid: position with: calculation]/[self getProbeLength];

     //printf("Avoid:\t"); [calculation print];

    if (mag < 0.001) {

         [calculation init]; // ignore objects far away

    }
    else {

        [calculation setLength: mag];
      
        if(0) {
 
           fprintf(stderr, "Avoid:\t"); 
           [calculation print];

       }

    }
  }

  return calculation; 
}


/////////////////////////////////////////////////
//
// maintainingCruisingDistance
//
/////////////////////////////////////////////////
- (id <Vector>)maintainingCruisingDistance
{
  Boid *boid;
  double dist;

  [calculation init];

  if ((boid = closestBoid[objectType]) != nil) {

     // Boid with lowest priority == closest seen
     [[calculation init: [boid getPosition]] sub: position];
     dist = [calculation getLength];

     if (dist > cruiseDistance) {

        [calculation setLength: 0.1];
    }
    else {
     
        [calculation setLength: -0.1];

    }

  } //if boid

  return calculation;
}



/////////////////////////////////////////////////////
//
// velocityMatching
//
////////////////////////////////////////////////////
- (id <Vector>)velocityMatching
{
  Boid *boid;   

  [calculation init];

  if ((boid = closestBoid[objectType]) != nil) {

      // Boid with lowest priority == closest seen
      [calculation init: [boid getVelocity]];
 
      [calculation mult: 0.05]; // importance factor of velocityMatching

  }

  return calculation;
}



/////////////////////////////////////////////////////
//
// flockCentering
//
/////////////////////////////////////////////////////
- (id <Vector>)flockCentering
{
  [calculation init: -1.0: -1.0];

  if ([calculation eq: flockCenter]) { // sybol: no others

      [calculation init];

  }
  else {

      [[[calculation init: flockCenter] sub: position] mult: 0.1];

  }

  return calculation;
}





/////////////////////////////////////////////////////////
//
// predatorAvoidance
//
////////////////////////////////////////////////////////
- (id <Vector>)predatorAvoidance
{
  Boid *obj;
  double mag, dist;
  [calculation init];

  // avoid agents with objectType = [self objectType]+1

  switch (objectType) {
  default: 
           //[InternalError raiseEvent: "ERROR: BOID has no type\n"];
           //break;

  case PREY:
       if ((obj = closestBoid[PREDATOR]) != nil) { // lowest priority == closest seen
           dist = [self visible: obj];
           [[calculation init: position] sub: [obj getPosition]];
           if (dist < 0) { // pretty far away
	       mag = 1.0 - [calculation getLength]/[self getProbeLength]; }
           else {
	       mag = 1.2 * (1.0 - dist/[self getProbeLength]);
           }

           [[calculation sub:[obj getVelocity]] setLength: mag]; // go the other way
       }

    break;


  case PREDATOR:
      if ((obj = closestBoid[PREY]) != nil) { // lowest priority == closest seen
          [[calculation init: [obj getPosition]] sub: position]; // vector towards PREY
          mag = 1.2 * (1.0 - [calculation getLength]/[self getProbeLength]);
          [calculation setLength: mag];
      }
  
      break;
     

  }  //switch

  return calculation;
}


/////////////////////////////////////////////////////////////
//
// step
//
/////////////////////////////////////////////////////////////
- step {

  double x, y;

  //  double dt = 1.0;
  
  //  [position print];

  if(objectType == OBSTACLE) 
  {

       fprintf(stderr, "BOID >>>> step objectType = %d OBSTACLE = %d\n", objectType, OBSTACLE);
       fflush(0);

  }



  [super step];     // p' = p0 + v0 * dt

  [self navigator]; // calculate new acceleration vector

  // new position p = p' + 1/2 * dt^2 * a
  [position add: [[calculation init: acceleration] mult: 0.5]];

  // new velocity = v0 + a * dt
  [velocity add: acceleration];

  if ([velocity getLength] > maxVelocity) // cap off at top speed
    [velocity setLength: maxVelocity];

  x = [position getX];
  y = [position getY];

  if (x<0.0) { while (x<0.0) x += worldX; [position setX: x]; }
  if (x>worldX)  { while (x>worldX)  x -= worldX; [position setX: x]; }
  if (y<0.0) { while (y<0.0) y += worldY; [position setY: y]; }
  if (y>worldY)  { while (y>worldY)  y -= worldY; [position setY: y]; }


   

  return self;
}



/////////////////////////////////////////////////////////
//
// wander
//
/////////////////////////////////////////////////////////
- (id <Vector>)wander {
  [[calculation init: velocity]
    mult: ([velocity getLength] - desiredCruisingSpeed) / maxVelocity];

  return calculation;
}



- printSelf {

    fprintf(stderr, "BOID >>>> printSelf self = %p \n", self);
    fflush(0);

    return self;

}

@end
