/*
   Simulation Player (Player.c)
   by Patrick McNeill
   Revision 1.6 (April 30, 1997)

   This is an OpenGL application (also works in Mesa) that processes and replays
   the contents of the log files generated by the UAV3d series of simulations.
   The replay is in full 3d, with the user in control of all three axises of
   rotation (via the z,a,x,s,c,d, and q keys).  The output is basically a 
   3d map of the terrain with the UAVs, Tanks, and Buildings (if any) on
   top of it.

   If you have any questions or comments about the Player, feel free to email
   me.  My address is pmcneill@ida.org until mid-August, then I go to college.
   Most likely, I'll be going to Georgia Tech so you can look me up there 
   (though I'll probably still be on the Swarm lists).

   New in Revision 1.6:
	Added in OptimizeTerrain and moved the help to PrintHelp.  Help must
		specifically be asked for now, and the Player doesn't run when
		help is displayed.  A command line interpretter has been added
		for future options (OptimizeTerrain is an option).
	Also added -wf for wireframe drawing rather than solid fills.  Speeds
		up some things.

   New in Revision 1.5:
	Created the DrawObj function to allow easier visual differentiation
		of objects (ie sphere vs cube).  This is currently determined
		by the ClassID.

   New in Revision 1.4.1:
	Fixed bug in sight lines that caused misalignment.  Also involved 
		reworking part of the log formats.

   New in Revision 1.4:
	"Sight Lines" added to allow for visual display of both object
		sightings and communications.

   New in Revision 1.3:
	Added jumps w/ numeric input.  Valid modes are currently turn and
		probed object.

   New in Revision 1.2:
	Supports the new file format that includes class information about
		each object.
	"UAV View" slightly more functional...
		I need to figure out the xyz offsets to make UAV View work...

   New in Revision 1.1:
	Camera movement along any axis
	Probing of individual objects
	Maximum array dimensions definable at compile time
	Experimental implementation of "UAV View" - see the sim through the
		eyes of the probed object.  Still needs a lot of work.
*/

#ifndef XSIZE
#define XSIZE 100
#endif

#ifndef YSIZE
#define YSIZE 100
#endif

#ifndef MAXUAV
#define MAXUAV 501
#endif

#ifndef TURNS
#define TURNS 1000
#endif

#ifndef MAXLINES
#define MAXLINES 300000
#endif

#ifndef OPTSIZE
#define OPTSIZE 10000
#endif

#define JUMP_TURN 0
#define JUMP_PROBE 1

#include <GL/gl.h>
#include <GL/glu.h>
#include <stdlib.h>
#include <stdio.h>
#include <aux.h>

// Global variables used to save time and resources.
// xyz spin variables for the rotation
static GLfloat xspin=0.0, yspin=0.0, zspin=0.0;
// matrix containing elevation coordinates
static float ys[XSIZE][YSIZE];
// rgb values to coorespond to the ys matrix
static float clrs[XSIZE][YSIZE][3];
// matrix containing turn data
static float upos[TURNS][MAXUAV][6];
// ClassIds cooresponding to each object
static int ucls[MAXUAV];
// Matrix to contain the lists of lines...
static float ulines[MAXLINES][6];
// various other variables
static int row, col, uavs, turns, lines, curTurn=0;
static float boxsize=0.75, agentsize=0.75;
// camera offsets
static float xos=0.0,yos=-5.0,zos=-45.0;
// selected object, probing boolean, UAV View boolean
static int selobj=0,probing=1,uavview=0;
// number for numeric input and type
static int jumpNum, jumpMode;
// variables for optimized terrain
static int optimized=0, optMax;
static float optCoords[OPTSIZE][4];
// other stuff
static int wireframe=0,noterrain=0,nolines=0,flatshade=0;

void DrawObj (int clsid)
/*
   Modify this function to control how you want to draw the objects.  If you 
   translate the cursor, make sure you translate it back to where it started.
   If you don't, weird things may happen.
*/
{
	switch (clsid)
	{
		case 0: case 1:
			if (wireframe)
				auxWireSphere(agentsize/8);
			else
				auxSolidSphere(agentsize/8);
			break;
		case 2:
			if (wireframe)
				auxWireBox(agentsize/4,agentsize/4,agentsize/4);
			else
				auxSolidBox(agentsize/4,agentsize/4,agentsize/4);
			break;
		default:
			if (wireframe)
				auxWireBox(agentsize/4,agentsize/4,agentsize/4);
			else
				auxSolidBox(agentsize/4,agentsize/4,agentsize/4);
	}
}

void display (void)
{
	int i,j,k,l,m,id1,id2;
	float dx,dy,dz,oldx,oldy,oldz;
	float camx=0,camy=0,camz=0;

	glClearColor(0.0,0.0,0.0,0.0);
	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

	glPushMatrix();

	glLoadIdentity();

	// translate the camera
	if (uavview)
	{
	        glTranslatef(-(boxsize*(row-1.5)),0.0,boxsize*1.5);
		glTranslatef(upos[curTurn][selobj][0],upos[curTurn][selobj][1],upos[curTurn][selobj][2]);
	}
	else
		glTranslatef(xos,yos,zos);
	camx+=xos; camy+=yos; camz+=zos;

	// rotate the view
	glRotatef(xspin,1.0,0.0,0.0);
	glRotatef(yspin,0.0,1.0,0.0);
	glRotatef(zspin,0.0,0.0,1.0);

	if (!noterrain)
	{
		// set up for the drawing
		glTranslatef(-((boxsize/2.0)+(row*boxsize)/2.0),0.0,-((boxsize/2.0)+(col*boxsize)/2.0));
		camx+=-((boxsize/2.0)+(row*boxsize)/2.0);
		camz+=-((boxsize/2.0)+(col*boxsize)/2.0);

		// draw the terrain...
		if (!optimized)
			for (i=0;i<(row-1);i++)
			{
				glTranslatef(boxsize,0.0,0.0);
				camx+=boxsize;
				for (j=0;j<(col-1);j++)
				{
					glTranslatef(0.0,0.0,boxsize);
					camz+=boxsize;
					if (wireframe)
						glBegin(GL_LINE_LOOP);
					else
						glBegin(GL_QUADS);
					glColor3f(clrs[i][j][0],clrs[i][j][1],clrs[i][j][2]);
					glVertex3f(0.0,ys[i][j],0.0);
					glColor3f(clrs[i+1][j][0],clrs[i+1][j][1],clrs[i+1][j][2]);
					glVertex3f(boxsize,ys[i+1][j],0.0);
					glColor3f(clrs[i+1][j+1][0],clrs[i+1][j+1][1],clrs[i+1][j+1][2]);
					glVertex3f(boxsize,ys[i+1][j+1],boxsize);
					glColor3f(clrs[i][j+1][0],clrs[i][j+1][1],clrs[i][j+1][2]);
					glVertex3f(0.0,ys[i][j+1],boxsize);
					glEnd();
				}
				glTranslatef(0.0,0.0,-(col*boxsize)+boxsize);
				camz+=-(col*boxsize)+boxsize;
			}
		else
		{
			for (i=0;i<optMax;i++);
			{
				j=optCoords[i][0];
				k=optCoords[i][1];
				l=optCoords[i][2];
				m=optCoords[i][3];
				glTranslatef((l-j),0.0,(m-k));
				camx+=(l-j); camz+=(m-k);
				if (wireframe)
					glBegin(GL_LINE_LOOP);
				else
					glBegin(GL_QUADS);
				glColor3f(clrs[j][k][0],clrs[j][k][1],clrs[j][k][2]);
				glVertex3f(j,0.0,k);
				glColor3f(clrs[l][k][0],clrs[l][k][1],clrs[l][k][2]);
				glVertex3f(l,0.0,k);
				glColor3f(clrs[l][m][0],clrs[l][m][1],clrs[l][m][2]);
				glVertex3f(l,0.0,m);
				glColor3f(clrs[j][m][0],clrs[j][m][1],clrs[j][m][2]);
				glVertex3f(j,0.0,m);
				glEnd();
			}
			glTranslatef((boxsize*(row-1.0)),0.0,0.0);
			camx+=(boxsize*(row-1.0));
		}
		// Reposition to the origin
		glTranslatef(-(boxsize*(row-1.5)),0.0,boxsize*1.5);
		camx+=-(boxsize*(row-1.5));
		camz+=boxsize*1.5;
	}

	// ...and now draw the objects for the current turn
	oldx=0;oldy=-agentsize/8.0;oldz=0;
	for (i=0;i<=uavs;i++)
	{
		dx=upos[curTurn][i][0]-oldx;
		dy=upos[curTurn][i][1]-oldy;
		dz=upos[curTurn][i][2]-oldz;
		oldx=upos[curTurn][i][0];
		oldy=upos[curTurn][i][1];
		oldz=upos[curTurn][i][2];
		if ((i==selobj)&&(probing))
			glColor3f(1.0-upos[curTurn][i][3],1.0-upos[curTurn][i][4],1.0-upos[curTurn][i][5]);
		else
			glColor3f(upos[curTurn][i][3],upos[curTurn][i][4],upos[curTurn][i][5]);
		glTranslatef(dx,dy,dz);
		camx+=dx; camy+=dy; camz+=dz;
		DrawObj(ucls[i]);
	}

	if (!nolines)
	{
		glBegin(GL_LINES);
		for (i=0;i<lines;i++)
			if (curTurn==ulines[i][0])
			{
				id1 = ulines[i][1];
				id2 = ulines[i][2];
				glColor4f(ulines[i][3],ulines[i][4],ulines[i][5],0.5);
				glVertex3f(upos[curTurn][id1][0]-oldx,upos[curTurn][id1][1]-oldy,upos[curTurn][id1][2]-oldz);
				glVertex3f(upos[curTurn][id2][0]-oldx,upos[curTurn][id2][1]-oldy,upos[curTurn][id2][2]-oldz);
			}
		glEnd();
	}

	glPopMatrix();
	glFlush();
	glXSwapBuffers(auxXDisplay(),auxXWindow());
//	fprintf(stderr,"Camera location: %f %f %f\n",camx,camy,camz);
}

// These are functions that are called in response to mouse and keyboard events
// that rotate the terrain.  Just for [xyz] forward and reverse spins.

void xfwdspin (void)
{
        xspin+=1.5;
        if (xspin>360.0)
                xspin-=360.0;

        display();
}

void xrevspin (void)
{
        xspin-=1.5;
        if (xspin<0.0)
                xspin+=360.0;

        display();
}

void yfwdspin (void)
{
        yspin+=1.5;
        if (yspin>360.0)
                yspin-=360.0;

        display();
}

void yrevspin (void)
{
        yspin-=1.5;
        if (yspin<0.0)
                yspin+=360.0;

        display();
}

void zfwdspin (void)
{
	zspin+=1.5;
	if (zspin>360.0)
		zspin-=360.0;

	display();
}

void zrevspin (void)
{
	zspin-=1.5;
	if (zspin<0.0)
		zspin+=360.0;

	display();
}

// And now the functions that set the idle functions.
// These tell it what to do before redrawing the scene.

void stopIdleFunc (AUX_EVENTREC *event)
{
	auxIdleFunc(0);
}

void startXFunc (AUX_EVENTREC *event)
{
	auxIdleFunc(xfwdspin);
}

void startXRevFunc (AUX_EVENTREC *event)
{
	auxIdleFunc(xrevspin);
}

void startYFunc (AUX_EVENTREC *event)
{
        auxIdleFunc(yfwdspin);
}

void startYRevFunc (AUX_EVENTREC *event)
{
        auxIdleFunc(yrevspin);
}

void startZFunc (AUX_EVENTREC *event)
{
        auxIdleFunc(zfwdspin);
}

void startZRevFunc (AUX_EVENTREC *event)
{
        auxIdleFunc(zrevspin);
}

void myinit (void)
{
	// Set up smooth shading (flat is ugly) and depth buffering.
	if (flatshade||wireframe)
		glShadeModel(GL_FLAT);
	else
		glShadeModel(GL_SMOOTH);
	glEnable(GL_DEPTH_TEST);
}

void myReshape (int w, int h)
{
	// resize function...  I don't even know what some of these things do:)
	glViewport(0,0,w,h);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	// If the terrain is disappearing at the back, make the last number in
	// this next function call larger.
	gluPerspective(60.0,(GLfloat)w/(GLfloat)h,1.0,100.0);
	glMatrixMode(GL_MODELVIEW);
}

// Reset the rotation
void resetFlippers (void)
{
	xspin=yspin=zspin=0.0;
}

// Reset the camera
void resetView (void)
{
	xos=0.0;zos=-45.0; yos=-5.0;
}

// This could be incorporated into the individual functions, but this way makes
// the code easier to maintain (if you want to add a lot of display information)
// and a lot easier to read (I think).
void dispObjInfo (void)
{
	if (probing)
		fprintf(stderr,"Object %d (Class %d) is currently at position %f, %f, %f.\n",selobj,ucls[selobj],upos[curTurn][selobj][0],upos[curTurn][selobj][1],upos[curTurn][selobj][2]);
}

// Increment the turn counter
void nextTurn (void)
{
	if (curTurn < turns)
		curTurn++;
	fprintf(stderr,"Currently displaying turn %d.\n",curTurn);
	dispObjInfo();
}

// Decrement the turn counter
void prevTurn (void)
{
	if (curTurn > 0)
		curTurn--;
	fprintf(stderr,"Currently displaying turn %d.\n",curTurn);
	dispObjInfo();
}

// Increment the selobj counter
void nextObj (void)
{
	if (selobj < uavs)
		selobj++;
	fprintf(stderr,"Currently probing object %d (Class %d).\n",selobj,ucls[selobj]);
	dispObjInfo();
}

// Decrement the selobj counter
void prevObj (void)
{
	if (selobj > 0)
		selobj--;
	fprintf(stderr,"Currently probing object %d (Class %d).\n",selobj,ucls[selobj]);
	dispObjInfo();
}

// Toggle probing
void tglProbe (void)
{
	if (probing)
	{
		probing=0;
		fprintf(stderr,"Probing is now off.\n");
	}
	else
	{
		probing=1;
		fprintf(stderr,"Probing is now on.\n");
	}
}

// Toggle UAV View
void tglView (void)
{
	if (uavview)
	{
		uavview=0;
		fprintf(stderr,"Viewing from probed object is now off.\n");
	}
	else
	{
		uavview=1;
		fprintf(stderr,"Viewing from probed object is now on.\n");
	}
}

// Camera Offset functions

// These two seem backwards, but the orientation is weird so it's necessary.
void incYOS (void)
{
	yos++;
}

void decYOS (void)
{
	yos--;
}

void incZOS (void)
{
	zos--;
}

void decZOS (void)
{
	zos++;
}

void incXOS (void)
{
	xos++;
}

void decXOS (void)
{
	xos--;
}

void spin180 (void)
{
	yspin=(int)(yspin+540)%(int)360;
}

// The Jump Functions -- j0-9 for input, jReturn for enter, jTurn for
// setting JUMP_TURN, jProbe for JUMP_PROBE

void j0 (void)
{
	jumpNum *= 10; 
}

void j1 (void)
{
	jumpNum = jumpNum * 10 + 1;
}

void j2 (void)
{
        jumpNum = jumpNum * 10 + 2;
}

void j3 (void)
{
        jumpNum = jumpNum * 10 + 3;
}

void j4 (void)
{
        jumpNum = jumpNum * 10 + 4;
}

void j5 (void)
{
        jumpNum = jumpNum * 10 + 5;
}

void j6 (void)
{
        jumpNum = jumpNum * 10 + 6;
}

void j7 (void)
{
        jumpNum = jumpNum * 10 + 7;
}

void j8 (void)
{
        jumpNum = jumpNum * 10 + 8;
}

void j9 (void)
{
        jumpNum = jumpNum * 10 + 9;
}

void jReturn (void)
{
	switch (jumpMode)
	{
		case JUMP_TURN:
			if (jumpNum > turns)
				fprintf(stderr,"Jump unsuccessful -- number out of range.\n");
			else
			{
				fprintf(stderr,"Jump successful -- current turn is now %d.\n",jumpNum);
				curTurn = jumpNum;
			}
			break;
		case JUMP_PROBE:
			if (jumpNum > uavs)
				fprintf(stderr,"Object does not exist.\n");
			else
			{
				fprintf(stderr,"Now probing object %d.\n",jumpNum);
				selobj = jumpNum;
			}
			break;
	}

	jumpNum = 0;
}

void jTurn (void)
{
	jumpMode=JUMP_TURN;
}

void jProbe (void)
{
	jumpMode=JUMP_PROBE;
}

int full (int curX[YSIZE-1])
{
	int i, ret=1;

	for (i=0;i<(YSIZE),ret==1;i++)
		if (curX[i]>0) ret=0;

	return ret;
}

void OptimizeTerrain (void)
/*
   This probably isn't the best way to do this, but here's how it works:
   It calculates the slope of each of the boundary lines of each square
   and stores them in an matrix.  It then begins a search across the matrix
   comparing the squares' slopes, and stopping when it finds one that doesn't
   match.  At the same time, it is counting how far down the continuity lasts.
   Finally, it finds the minimum value for the downward search.  This, and the
   length, provides a rectangle that fits the necessary area.  It repeats this
   for the rest of the matrix, until the whole terrain has been covered.
*/
{
	float slopes[XSIZE-1][YSIZE-1][4];
	float temp1, temp2;
	int curX[YSIZE-1];
	int i,j,k,l,m,n,mind;

//	fprintf(stderr,"Optimizing terrain display...\n");

	for (i=0;i<(YSIZE-1);i++)
		curX[i]=0;

	for (i=0;i<(XSIZE-1);i++)
		for (j=0;j<(YSIZE-1);j++)
		{
			// slope is dy/dx, but we can pretend that the difference
			// between two x coords 1 so we don't have to divide.
			slopes[i][j][0]=(ys[i][j]-ys[i+1][j]);
			slopes[i][j][1]=(ys[i+1][j]-ys[i+1][j+1]);
			slopes[i][j][2]=(ys[i+1][j+1]-ys[i][j+1]);
			slopes[i][j][3]=(ys[i][j+1]-ys[i][j]);
		}

	i=1;
	j=k=l=n=mind=0;

	while ((!full(curX))&&(n<YSIZE)&&(l<(YSIZE-1)))
	{
		while ((slopes[k][l][0]==slopes[i][l+mind][0])
		     &&(slopes[k][l][1]==slopes[i][l+mind][1])
		     &&(slopes[k][l][2]==slopes[i][l+mind][2])
		     &&(slopes[k][l][3]==slopes[i][l+mind][3])
		     &&(mind<(YSIZE-1)))
			mind++;
//		fprintf(stderr,"Mind, check 1=%d\n",mind);

		while ((slopes[k][l][0]==slopes[i][j][0])
		     &&(slopes[k][l][1]==slopes[i][j][1])
		     &&(slopes[k][l][2]==slopes[i][j][2])
		     &&(slopes[k][l][3]==slopes[i][j][3])
		     &&(i<(XSIZE-1)))
		{
			m=j;
			while ((slopes[k][l][0]==slopes[i][m][0])
			     &&(slopes[k][l][1]==slopes[i][m][1])
			     &&(slopes[k][l][2]==slopes[i][m][2])
			     &&(slopes[k][l][3]==slopes[i][m][3])
			     &&(m<(YSIZE-1)))
				m++;
			if (m<mind) mind=m;
//			fprintf(stderr,"Mind, turn %d = %d\n",i,mind);
			i++;
			if (i==YSIZE)
			{
				i--;
				break;
			}
		}
//		fprintf(stderr,"rect (%d,%d)-(%d,%d).\n",k,l,i,mind);
//		fprintf(stderr,"n=%d\n",n);
		optCoords[n][0]=k;
		optCoords[n][1]=l;
		optCoords[n][2]=i;
		optCoords[n][3]=mind;
//		fprintf(stderr,"Entering assignment loops (l=%d,mind=%d)\n",l,mind);
		for (m=l;m<=mind;m++)
		{
			curX[m]=i+1;
//			fprintf(stderr,"assigned curX[%d] to %d.\n",m,i+1);
		}
//		fprintf(stderr,"Part 1 done\n");
		while (curX[l]>=100)
			l++;
//		fprintf(stderr,"l is now %d\n",l);
		k=i=curX[l];
		mind=j=l;
		n++;
	}

	optMax = n;
}

void PrintHelp (void)
{
	fprintf(stderr,"Instructions:\n");
	fprintf(stderr,"Keyboard Commands --\n");
	fprintf(stderr,"m,n           - Increase/Decrease turn counter\n");
	fprintf(stderr,"b,v           - Increase/Decrease probed object counter\n");
	fprintf(stderr,"z,a           - Rotate around y axis\n");
	fprintf(stderr,"x,s           - Rotate around x axis\n");
	fprintf(stderr,"c,d           - Rotate around z axis\n");
	fprintf(stderr,"f,r           - Increase/Decrease camera X position\n");
	fprintf(stderr,"g,t           - Increase/Decrease camera Y position\n");
	fprintf(stderr,"h,y           - Increase/Decreate camera Z position\n");
	fprintf(stderr,"q             - Stop rotation\n");
	fprintf(stderr,"p             - Toggle probing\n");
	fprintf(stderr,"o             - Restore orientation\n");
	fprintf(stderr,"i             - Restore view\n");
	fprintf(stderr,"l             - Rotate the display 180 degrees\n");
	fprintf(stderr,"0-9           - Enter number for Jump command\n");
	fprintf(stderr,"j,k           - Set Jump mode (TURN or PROBE)\n");
	fprintf(stderr,"Enter         - Execute Jump command.\n");
	fprintf(stderr,"\nMouse Commands --\n");
	fprintf(stderr,"Left Button   - Rotate around y axis\n");
	fprintf(stderr,"Right Button  - Rotate around y axis\n");
	fprintf(stderr,"Middle Button - Rotate around x axis\n");
	fprintf(stderr,"\nCommand Line Parameters --\n");
	fprintf(stderr,"-wf           - Render in wire-frame mode\n");
	fprintf(stderr,"-nl           - Do not draw lines (from sightlines file)\n");
	fprintf(stderr,"-fs           - Use flat shading\n");
	fprintf(stderr,"-help         - Show this message\n");
	fprintf(stderr,"\nAlso - Make sure CAPS LOCK is OFF!  If it is on, nothing will work.\n");
}

void main (int argc, char **argv)
{
	FILE *infile, *fp;
	int i, j, k, l, redobox=1;
	float temp,xx,yy,zz,r,g,b;

	fprintf(stderr,"Simulation Player version 1.6\n");
	fprintf(stderr,"by Patrick McNeill\n");

	// Now go through any command line arguments...

	if (argc > 1)
	{
		for (i=1;i<argc;i++)
		{
			if (strcmp("-o",argv[i])==0)
				optimized=1;
			else if ((strcmp("-h",argv[i])==0)||(strcmp("-help",argv[i])==0))
			{
				PrintHelp();
				return;
			}
			else if (strcmp("-wf",argv[i])==0)
				wireframe=1;
			else if (strcmp("-nt",argv[i])==0)
				noterrain=1;
			else if (strcmp("-nl",argv[i])==0)
				nolines=1;
			else if (strcmp("-nr",argv[i])==0)
				redobox=0;
			else if (strcmp("-fs",argv[i])==0)
				flatshade=1;
			else
			{
				fprintf(stderr,"Unknown option %s.\n",argv[i]);
				fprintf(stderr,"Correct usage: Player [-o] [-wf]  [-nl] [-nr] [-fs] [-h|-help]\n");	
				fprintf(stderr,"Use -help for help.\n");
				return;
			}
		}
	}

	// Process the terrain data files
	if (!noterrain)
	{
		fprintf(stderr,"\nProcessing Terrain File...\n");
		infile=fopen("t","r");
		fp=fopen("tclrs","r");
		fscanf(infile,"%d",&row);
		fscanf(infile,"%d",&col);
		for (i=0;i<row;i++)
		{
			for (j=0;j<col;j++)
			{
				fscanf(infile,"%f",&temp);
				ys[j][i]=temp;
				fscanf(fp,"%f %f %f",&r,&g,&b);
				clrs[j][i][0]=r;
				clrs[j][i][1]=g;
				clrs[j][i][2]=b;
			}
			fscanf(infile,"\n");
			fscanf(fp,"\n");
		}
		fclose(fp);
		fclose(infile);
		if (redobox)
			boxsize*=(100/row);  // make it fit in the 100x100 grid
	}

	// now process the log files
	fprintf(stderr,"Processing Position File...\n");
	fp=fopen("uavpos","r");
	fscanf(fp,"%d",&uavs);
	fscanf(fp,"%d",&turns);
	for (i=0;i<turns;i++)
		for (j=0;j<uavs;j++)
		{
			fscanf(fp,"%d %f %f %f %f %f %f cls%d",&l,&xx,&yy,&zz,&r,&g,&b,&k);
			// X and Z must be multiplied by boxsize to allign
			// the terrain and the objects.
			upos[i][l][0]=xx*agentsize;
			// Player uses a height model 1/2 of UAV3d
			// You may want to use a direct scale, but my personal
			// feeling that a dy of 1.0 is too much of a slope.
			upos[i][l][1]=yy/2.0;
			// So it isn't black when in UAV view if it's on the 
			// ground.  If the camera is at the same level as the
			// ground, it doesn't see it.
			if (upos[i][l][1]==0.0)
				upos[i][l][1]=0.01;
			upos[i][l][2]=zz*agentsize;
			upos[i][l][3]=r;
			upos[i][l][4]=g;
			upos[i][l][5]=b;
			ucls[l]=k;
		}
	fclose(fp);
	turns--; // just to make the log files logical - humans start at 1,
	uavs--;  // arrays start at 0.

	if (!nolines)
	{
		fprintf(stderr,"Processing Lines File...\n");
		fp = fopen("sightlines","r");
		fscanf(fp,"%d",&lines);
		for (i=0;i<lines;i++)
		{
			fscanf(fp,"%f %d %d %f %f %f",&temp,&k,&l,&r,&g,&b);
			ulines[i][0]=temp;
			ulines[i][1]=k;
			ulines[i][2]=l;
			ulines[i][3]=r;
			ulines[i][4]=g;
			ulines[i][5]=b;
		}
		fclose(fp);
	}

	if ((!noterrain)&&(optimized)) OptimizeTerrain();

	// set up the OpenGL stuff - AUX_DEPTH is needed for Mesa, but OpenGL
	// seems to work without it...
	// you shouldn't need to do anything with this...
	auxInitDisplayMode(AUX_DOUBLE|AUX_RGBA|AUX_DEPTH);
	auxInitPosition(0,0,400,300);
	auxInitWindow("Simulation Player");
	myinit();
	auxReshapeFunc(myReshape);

	// binds keys and mouse buttons to functions...
	// too add more, just make the function and add in a auxKeyFunc
	// line with the AUX_x const.  You can look in aux.h (glaux.h in Mesa)
	// to get a list of all of the valid constants.
	auxMouseFunc(AUX_LEFTBUTTON,AUX_MOUSEUP,stopIdleFunc);
	auxMouseFunc(AUX_RIGHTBUTTON,AUX_MOUSEUP,stopIdleFunc);
	auxMouseFunc(AUX_MIDDLEBUTTON,AUX_MOUSEUP,stopIdleFunc);
	auxKeyFunc(AUX_q,stopIdleFunc);

	auxMouseFunc(AUX_LEFTBUTTON,AUX_MOUSEDOWN,startYFunc);
	auxMouseFunc(AUX_RIGHTBUTTON,AUX_MOUSEDOWN,startYRevFunc);
	auxMouseFunc(AUX_MIDDLEBUTTON,AUX_MOUSEDOWN,startXFunc);
	auxKeyFunc(AUX_z,startXFunc);
	auxKeyFunc(AUX_a,startXRevFunc);
	auxKeyFunc(AUX_x,startYFunc);
	auxKeyFunc(AUX_s,startYRevFunc);
	auxKeyFunc(AUX_c,startZFunc);
	auxKeyFunc(AUX_d,startZRevFunc);

	auxKeyFunc(AUX_o,resetFlippers);
	auxKeyFunc(AUX_i,resetView);

	auxKeyFunc(AUX_n,prevTurn);
	auxKeyFunc(AUX_m,nextTurn);
	auxKeyFunc(AUX_v,prevObj);
	auxKeyFunc(AUX_b,nextObj);
	auxKeyFunc(AUX_p,tglProbe);
	auxKeyFunc(AUX_u,tglView);

	auxKeyFunc(AUX_f,incXOS);
	auxKeyFunc(AUX_r,decXOS);
	auxKeyFunc(AUX_g,incYOS);
	auxKeyFunc(AUX_t,decYOS);
	auxKeyFunc(AUX_h,incZOS);
	auxKeyFunc(AUX_y,decZOS);

	auxKeyFunc(AUX_l,spin180);

	auxKeyFunc(AUX_0,j0);
	auxKeyFunc(AUX_1,j1);
	auxKeyFunc(AUX_2,j2);
	auxKeyFunc(AUX_3,j3);
	auxKeyFunc(AUX_4,j4);
	auxKeyFunc(AUX_5,j5);
	auxKeyFunc(AUX_6,j6);
	auxKeyFunc(AUX_7,j7);
	auxKeyFunc(AUX_8,j8);
	auxKeyFunc(AUX_9,j9);
	auxKeyFunc(AUX_RETURN,jReturn);
	auxKeyFunc(AUX_j,jTurn);
	auxKeyFunc(AUX_k,jProbe);

	auxMainLoop(display);
}
