// Copyright (C) 1999 University of Washington, Columbia Basin Research.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#import <math.h>
#import "SearchableGridSpace.h"
 
@implementation SearchableGridSpace
+ createBegin: aZone {
  return [super createBegin: aZone];
}
 
- createEnd {
  [super createEnd];
  return self;
}

- setAllowDups: (BOOL) bAllow {
  allowDups = bAllow;
  return self;
}

- setIntCoords: (BOOL) bInt {
  intCoords = bInt;
  return self;
}

// Set the limits of the underlying space and the size of a grid
//  cell. This also determines the number of grid cells, and lets the
//  underlying Discrete2D code create the storage lattice.
// NOTE: this message should be sent once, during initialization.
- setMinX: (double) minx maxX: (double) maxx minY: (double) miny
     maxY: (double) maxy gridSpacing: (double) spacing {
  minX = minx;
  minY = miny;
  maxX = maxx;
  maxY = maxy;
  gridSpacing = spacing;
  sizeX = 1 + (int)((maxX - minX) / gridSpacing);
  sizeY = 1 + (int)((maxY - minY) / gridSpacing);
  [self setSizeX: (unsigned) sizeX Y: (unsigned) sizeY];
  return self;
}

- (double) getSpacing {
  return gridSpacing;
}

- (double) getMinX {
  return minX;
}

- (double) getMinY {
  return minY;
}

- clear {
  [self fastFillWithObject: nil];
  return self;
}

- clearAndPutObjects: (id <List>) aList {
  [self clear];
  return [self putObjects: aList];
}

// This routine puts a list of objects into the underlying lattice. More than
//  one object can exist in a given grid cell, living in a simple list
//  supported by the objects themselves. This is done for searching efficiency
//  (allocating an iterator for each cell for each search would be slow,
//  allocating a list for each non-empty cell would consume much memory).
- putObjects: (id <List>) aList {
  id <Index> indx;
  id object, oldObj;
  double x, y, oldX, oldY;
  int xg, yg;
  BOOL skip;

  indx = [aList begin: [self getZone]];
  while ((object = [indx next]) != nil) {
    // Decide which grid cell to put object in.
    if (intCoords) {
      x = [object getIntX];
      y = [object getIntY];
    }
    else {
      x = [object getDoubleX];
      y = [object getDoubleY];
    }
    xg = (int)((x - minX) / gridSpacing);
    yg = (int)((y - minY) / gridSpacing);
    skip = FALSE;

    // Complain if it doesn't fit.
    if (x < minX || x > maxX || y < minY || y > maxY) {
      printf( "Error, object at %f %f is outside grid, discarding\n", x, y );
      skip = TRUE;
    }

    // Complain if it is at the same place as another object.
    if (!skip && !allowDups) {
      oldObj = [self getObjectAtX: (unsigned) xg Y: (unsigned) yg];
      while (oldObj != nil) {
        if (intCoords) {
          oldX = [oldObj getIntX];
          oldY = [oldObj getIntY];
        }
        else {
          oldX = [oldObj getDoubleX];
          oldY = [oldObj getDoubleY];
        }
        if (x == oldX && y == oldY) {
          printf( "Error, two objects at same location: %f %f. Discarding\n",
                    x, y );
          skip = TRUE;
          break;
        }
        oldObj = [oldObj getNext];
      }
    }

    // Add object to head of list of objects in given grid cell.
    if (!skip) {
      [object setNext: [self getObjectAtX: (unsigned) xg Y: (unsigned) yg]];
      [self putObject: object atX: (unsigned) xg Y: (unsigned) yg];
    }
  }
  [indx drop];
  return self;
}

@end

