//  2d Feeding Optimization
//  Copyright 8/18/00 Andrew Lovett, under the supervision of Louis Gross
//  Check the file Info/CopyrightInfo.txt for copyright information

//Sets up and schedules the modelswarm for each simulation run when we're in //optimization mode

import swarm.Globals;
import swarm.objectbase.Swarm;
import swarm.objectbase.SwarmImpl;
import swarm.activity.Activity;
import swarm.activity.Schedule;
import swarm.activity.ScheduleImpl;
import swarm.defobj.Zone;
import swarm.Selector;

public class Middleman extends SwarmImpl {
	public Landscape modelswarm; //the modelswarm, in which agents are created
	public static SimulationState state; //info on the state of the simulation is 									//viewed and edited here
	public Schedule updateSchedule; //schedules any visual updates
	public Schedule killSchedule; //schedules the simulation's end

	//I have left the following three variables in all-caps despite the fact that 	//they are no longer constants, technically speaking; the fact is, they are 	//given valuables by the Optimizer class (which takes the data from the file 	//SumulationData.txt) before an instance of Middleman is created, and these 	//values do not chance throughout the program, so it is easier to think of them 	//as constants
	public static int SIMU_SIZE; //total number of time steps per simulation

	public static int START_TIME; //the time at which the program 
									//begins taking measurements
	public static int VIABILITY_LEVEL; //the minumum population required 									//for a species to be considered viable

	public int dumpType; //refers to the method which will be used to dump food in 				//the Foodspaces

	//Middleman variables used to measure the success of each dumpType
	public int animalYears = 0; //the sume of each animal * the number of years 							//that animal was alive
	//each Animal type's success in staying above a given population 
	public double cowViability;
	public double horseViability;
	public double monkeyViability;
	public double humanViability;

	//and the ints used to determine viability by giving the number of times an 	//animal successfully succeeded the viability level
	public int cowSuccesses = 0;
	public int horseSuccesses = 0;
	public int monkeySuccesses = 0;
	public int humanSuccesses = 0;

	//the deaths and births per time step
	public double deathRate;
	public double birthRate;

	//the total births and deaths, used to determine birth and death rates
	//these are, of course, only the births and deaths that occur after we've 	//started counting
	public int birthCount;
	public int deathCount;

	//the death rate for each region and the death count for each region
	public int[] deathCountForRegion;
	public double[] deathRateForRegion;

	Info simuInfo; //holds all info about a run of the simulation so that the 					//Optimizer class can retreive it
	
	//constructor
	public Middleman (Zone thZone, int i, int dumpType, Info simuInfo) {
		super (thZone);  //thZone is the memory zone in which this class is found
				//any swarms beneath it will be found in its memory zone
		this.simuInfo = simuInfo;
		this.dumpType = dumpType;

		modelswarm = new Landscape (getZone ());
		modelswarm.makeWorld (20, 20, Master.xRegions, Master.yRegions);
	}

	//build all the graphical and interface objects, and tell modelswarm to build 	//its objects
	public Object buildObjects () {
		super.buildObjects ();
		modelswarm.buildObjects ();

		//now we'll tell the Space2d how to dump food
		modelswarm.world.setFoodDumpType (dumpType);
		if (dumpType == 1) //for now, let's just have it dump 
					//even amounts in each region
			modelswarm.world.evenRegions ();

		return this;
	}

	//sets up the our display schedule; modelswarm doesn't really use buildActions
	public Object buildActions () {
		super.buildActions ();
		modelswarm.buildActions ();

		updateSchedule = new ScheduleImpl (getZone(), 1);
		killSchedule = new ScheduleImpl (getZone(), true);
		try {
			updateSchedule.at$createActionTo$message (0, this,
				new Selector (getClass (), "updateDisplay", false));

			killSchedule.at$createActionTo$message (SIMU_SIZE - 1, this,
				new Selector (getClass (), "endSimulation", false));
		}
		catch (Exception e) {
			e.printStackTrace (System.err);
			System.exit (1);
		}
		
		return this;
	}

	//activate our class and our schedule, and tell modelswarm to activate all the 	//agents
	public Activity activateIn (Swarm papaswarm) {
		super.activateIn (papaswarm);
		modelswarm.activateIn (this);
		updateSchedule.activateIn (this);
		killSchedule.activateIn (this);
	
		return getActivity ();
	}

	//ends the current simulation run, dropping this instance of Middleman
	public void endSimulation () {
		double minDeathCount = 9999, maxDeathCount = 0; //the greatest and 								//smallest deathCounts among the regions


		if (state.Time >= START_TIME) {
			double measureTime = state.Time - START_TIME + 1;
			cowViability = cowSuccesses / measureTime;
			horseViability = horseSuccesses / measureTime;
			monkeyViability = monkeySuccesses / measureTime;
			humanViability = humanSuccesses / measureTime;
			
			deathCount = state.Death_Count - deathCount;
			birthCount = state.Birth_Count - birthCount;

			deathRate = deathCount / measureTime;
			birthRate = birthCount / measureTime;

			for (int i = 0; i <= Space2d.numRegions - 1; i++) {
				deathCountForRegion[i] = Space2d.deathsInRegion[i] - 											deathCountForRegion[i];
				deathRateForRegion[i] = deathCountForRegion[i] / measureTime;
		
				double holder = deathCountForRegion[i];
				if (holder < minDeathCount)
					minDeathCount = holder;
				if (holder > maxDeathCount)
					maxDeathCount = holder;
			}
		
			//use simuInfo to report on simulation
			simuInfo.totalTime = state.Time;
			simuInfo.startTime = START_TIME;
			simuInfo.animalYears = this.animalYears;
			simuInfo.viabilityLevel = VIABILITY_LEVEL;
			simuInfo.cowViability = this.cowViability;
			simuInfo.horseViability = this.horseViability;
			simuInfo.monkeyViability = this.monkeyViability;
			simuInfo.humanViability = this.humanViability;
			simuInfo.deathRate = this.deathRate;
			simuInfo.birthRate = this.birthRate;
			simuInfo.deathDist = maxDeathCount - minDeathCount;
		}
		else
			System.out.println ("oops");
		
		//out.close ();
		Globals.env.getCurrentSwarmActivity ().terminate ();
	}

	//the function the display schedule calls, it updates the display, tells the 	//modelswarm to draw food the first time this timestep it is asked to do so, 	//updates Landscape's Time and Total_Animal fields, etc
	public void updateDisplay () {
		state.Time++;
		state.Total_Animals = modelswarm.allAnimals.size ();

		//initialize our death and birth counts when we reach START_TIME
		if (state.Time == START_TIME) {
			//we'll set all these equal to the current amounts; then, when 				//we're done, we'll subtract them from the final amounts to get the 			//change over the measurement period
			birthCount = state.Birth_Count;
			deathCount = state.Death_Count;
			deathCountForRegion = new int[Space2d.numRegions];
			deathRateForRegion = new double[Space2d.numRegions];
			for (int i = 0; i <= Space2d.numRegions - 1; i++)
				deathCountForRegion[i] = Space2d.deathsInRegion[i];
		}
		
		if (state.Time >= START_TIME) {
			animalYears += state.Total_Animals;
		
			if (state.Total_Cows >= VIABILITY_LEVEL)
				cowSuccesses++;
			if (state.Total_Horses >= VIABILITY_LEVEL)
				horseSuccesses++;
			if (state.Total_Monkeys >= VIABILITY_LEVEL)
				monkeySuccesses++;
			if (state.Total_Humans >= VIABILITY_LEVEL)
				humanSuccesses++;
		}

		if ((modelswarm.allAnimals.size ()) == 0)
			endSimulation ();

	}
	
	//used to begin a simulation 
	public Object go () {
		System.out.println ("This use of dump method " + dumpType + 
				" will take " + SIMU_SIZE + " time steps.");
		(getActivity ().getSwarmActivity ()).run ();

		return getActivity ().getStatus ();
	}
}

