// AgentCamera.m

// Copyright (C) 1996 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

#import "AgentCamera.h"
#import <simtools.h>
#import <math.h>
#import <vision/TwoD.h>
#import <vision/ThreeD.h>

@implementation AgentCamera

// Initialize crucial state for the agent.

-setWorlds: (Grid2d *) wAgt: (WorldObject *) wObj {
  if (worldAgent != nil || worldObject != nil) {
    [InvalidArgument raiseEvent: "You can only set the worldAgent/worldObjects at once at creation time\n"];
  }
  worldAgent = wAgt;
  worldObject = wObj;

  return self;
}

// createEnd. This is a good place to put code that does various sorts
// of initialization that can only be done after some parameters of the
// object are set. It's also a good time to check for errors in creation.
-createEnd {
  // make sure the user set up worldAgent and worldObject.
  if (worldAgent == nil || worldObject == nil) {
    [InvalidCombination raiseEvent: "Agent was created without a worldAgent or worldObject.\n"];
  }

  worldXSize = [worldAgent getSizeX];
  worldYSize = [worldAgent getSizeY];

  return self;
}

-setX: (int) inX Y: (int) inY {
  x = inX;
  y = inY;
  [worldAgent putObject: self atX: x Y: y];
  //randGAgt = [[PMMLCG1 alloc] initSeed: seedValueAgt];	  // seeds from clock
  //uRandAgt = [[[Uniform alloc] init] setGenerator: randGAgt];
  //for the generator 
  randGAgt = [PMMLCG1 createBegin: [self getZone]];
  [randGAgt setStateFromSeed: seedValueAgt];
  randGAgt = [randGAgt createEnd];
  // for the distribution
  uRandAgt = [UniformInteger createBegin: [self getZone]];
  [uRandAgt setGenerator: randGAgt];
  uRandAgt = [uRandAgt createEnd];

  return self;
}

-setCameraInit
{
  //int i;

  //builds camera and gives a couple of parameters

  camera = [Camera createBegin: [self getZone]]; // camera allocation
  camera = [camera createEnd];
  [camera initCamera: "EYE"]; // init parameters, with a name for instance
  //[[camera sendIntr] print]; // print these parameters
  //[[camera sendIntr] ViewAngle]; // give additional information such as view angle

  [[camera sendExtr] initialize: 0: 0: 0: 0];

  pixellist = [List create: [self getZone]];
  [[camera sendExtr] print: [camera sendName]]; // print Extr parameters

  return self;
}


-setRotAngle: (double)val
{
  rotAngle=val;
  return self;
}

-(double)getRotAngle
{
  return rotAngle;
}

-setNbPointSeen: (int)val
{
  nbPointSeen = val;
  return self;
}

-(int)getNbPointSeen
{
  return nbPointSeen;
}


-setSeed: (int)value
{
  seedValueAgt=value;
  return self;
}


// Agent behaviour is actually implemented here. The notion of a "step"
// method is a nice simplification for basic simulations.
-step {
  int newX, newY;
  int i;
  int pixelseen=0;
  int val, tries;
  double rotation_angle;


  // X world is Y of swarm world, Y world is virtual X of swarm (virtual height), Z world is Y swarm size - Y swarm
  // do not change the height x

  printf("Camera is in : x = %d, y = %d, orientation = %g\n", x, y, [self getRotAngle]);

  // update the camera coordinates with the position of the agent
  [[camera sendExtr] modifytheta: rotAngle]; // rotate clockwise
  [[camera sendExtr] modifyy: x];
  [[camera sendExtr] modifyz: y]; 
  //[[camera sendExtr] print: [camera sendName]]; // print Extr parameters

  // computes the images of the objects and returns the number of objects seen and the list of images
  pixelseen = [camera ComputeImage: [worldObject getThreeDList]: pixellist: [worldObject getInitObj]: [worldAgent getSizeY]];
  [self setNbPointSeen: pixelseen];

  printf("nber of objects seen : %d\n", pixelseen);
  if (pixelseen != 0) {
    for (i=0; i< pixelseen; i++) {
      printf("pixel %d: u=%d, v=%d, size=%d\n", [[pixellist atOffset: i] sendlab], [[pixellist atOffset: i] sendu], 					[[pixellist atOffset: i] sendv], [[pixellist atOffset: i] sendHeapHeight]);
    }
  }
  printf("\n\n");

  // we choose randomly the new position and new orientation.
  // orientation is as follows:
  //
  //            7    0    1
  //             \   |   /
  //              \  |  /
  //               \ | /
  //        6  ---      --- 2
  //               / | \
  //              /  |  \
  //             /   |   \
  //            5    4    3
  //
  // 0 ----> 0 rad
  // 1 ----> PI/4
  // 2 ----> PI/2
  // 3 ----> 3PI/4 etc...

  val = [uRandAgt getIntegerWithMin: 0L withMax: 2];
 // val = [uRandAgt rMax: 3]; // we choose the cell randomly: we intentionnally
			    // restricted for this demo the number of directions (0, 1, 2)
  switch(val) {
    case 0:
      newX = x;
      newY = y-1;
      rotation_angle = 0;
      break;
    case 1:
      newX = x+1;
      newY = y-1;
      rotation_angle = PI_4;
      break;
    case 2:
      newX = x+1;
      newY = y;
      rotation_angle = PI_2;
      break;
    default:
      break; 
  }
  newX = (newX+ worldXSize) % worldXSize;
  newY = (newY+ worldYSize) % worldYSize;

  tries = 0;
  // we are checking if there is already another agent present in the location we plan to move to
  while ([worldAgent getObjectAtX: newX Y: newY] != nil && tries < 10) {

    val = [uRandAgt getIntegerWithMin: 0L withMax: 2];
    //val = [uRandAgt rMax: 3]; // we choose the cell randomly
    switch(val) {
      case 0:
        newX = x;
        newY = y-1;
        rotation_angle = 0;
        break;
      case 1:
        newX = x+1;
        newY = y-1;
        rotation_angle = PI_4;
        break;
      case 2:
        newX = x+1;
        newY = y;
        rotation_angle = PI_2;
        break;
      default:
        break; 
    }
    newX = (newX+ worldXSize) % worldXSize;
    newY = (newY+ worldYSize) % worldYSize;
    tries++;					  // don't try too hard.
  }
  if (tries == 10) {				 
    newX = x;		// so just don't move and keep the orientation
    newY = y;
    rotation_angle = rotAngle;
  }

  // Now move ourselves in the grid and update our coordinates.
  [worldAgent putObject: nil atX: x Y: y];
  x = newX;
  y = newY;
  [self setRotAngle: rotation_angle];
  // printf("move to x=%d, y=%d, rotation is %f\n", newX, newY, rotAngle);
  [worldAgent putObject: self atX: newX Y: newY];

  i=0;
  while (i++<pixelseen){
    [pixellist remove: [pixellist atOffset: 0]];
    //printf("removed %d \n", i);
  };
  
  // all done moving! Return self.
  return self;
}

// Extra bits of display code: setting our colour, drawing on a window.
// This code works, but it'd be better if there were a generic object
// that knew how to draw agents on grids.
-setAgtColor: (Color) c {
  agtColor = c;
  return self;
}

-drawSelfOn: (Raster *) r {
  [r drawPointX: x Y: y Color: agtColor];
  return self;
}

@end
