// AgentModelSwarm.m

// Copyright (C) 1996 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

#import "AgentModelSwarm.h"
#import <simtools.h>

@implementation AgentModelSwarm

// These methods provide access to the objects inside the ModelSwarm.
// These objects are the ones visible to other classes via message call.
// In theory we could just let other objects use Probes to read our state,
// but message access is frequently more convenient.

-getAgentList {
  return agentList;
}

-getWorldAgent {
  return worldAgent;
}

-getWorldObject {
  return worldObject;
}


-IncrementNumIterations
{
  numIterations +=1;
  return self;
}

-(int)getNumIterations
{
  return numIterations;
}

// This method isn't normally used, but is convenient when running probes:
// it lets you easily clone a agent and drag it into the model.
-addAgent: (id) agt {
  [agentList addLast: agt];
  return self;
}


// createBegin: here we set up the default simulation parameters.
+createBegin: (id) aZone {
  AgentModelSwarm * obj;
  ProbeMap * probeMap;

  // First, call our superclass createBegin - the return value is the
  // allocated AgentModelSwarm object.
  obj = [super createBegin: aZone];

  // Now fill in various simulation parameters with default values.
  obj->numAgents = 1;
  obj->numAgentCamera = 1;
  if (obj->numAgents != obj->numAgentCamera) printf ("Bad selection of parameters\n");
  obj->worldXSize = 50;
  obj->worldYSize = 50;
  obj->InitNumObjects = 10;
  obj->seedValueObjects = 100;
  obj->seedValueLocAgt = 50;
  obj->numIterations = 0;

  // And build a customized probe map. Without a probe map, the default
  // is to show all variables and messages. Here we choose to
  // customize the appearance of the probe, give a nicer interface.
  probeMap = [EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [self class]];
  probeMap = [probeMap createEnd];

  // Add in a bunch of variables, one per simulation parameter
  [probeMap addProbe: [probeLibrary getProbeForVariable: "numAgents"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "numAgentCamera"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "worldXSize"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "worldYSize"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "InitNumObjects"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "seedValueObjects"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "seedValueLocAgt"
				    inClass: [self class]]];
  // And one method, the "addagent" method for a probe demo.
  // we also hide the return value for this message probe, just for nicety.
  //[probeMap addProbe: [[probeLibrary getProbeForMessage: "addAgent:"
  //			     inClass: [self class]]
  //			setHideResult: 1]];

  // Now install our custom probeMap into the probeLibrary.
  [probeLibrary setProbeMap: probeMap For: [self class]];
  
  return obj;
}

// createEnd: we could create some objects here if we knew we needed
// them. But this method is called before the user is allowed to fill
// in any customization of the model, so we defer most object creation
// to later. (In this example, this method does nothing at all and could
// just be inherited. But it's here to show you a place to customize.)
-createEnd {
  return [super createEnd];
}

// Now it's time to build the model objects. We use various parameters
// inside ourselves to choose how to create things.
-buildObjects {
  int i;
  id randG;
  id uRand;

  //randG = [[PMMLCG1 alloc] initSeed: seedValueLocAgt];	  // seeds from clock
  //uRand = [[[Uniform alloc] init] setGenerator: randG];

  // for the generator
  randG = [PMMLCG1 createBegin: [self getZone]];
  [randG setStateFromSeed: seedValueLocAgt];
  randG = [randG createEnd]; 
  // for the distribution
  uRand = [UniformInteger createBegin: [self getZone]];
  [uRand setGenerator: randG];
  uRand = [uRand createEnd];
  
  // allow our parent class to build anything.
  [super buildObjects];
  
  // First, set up objects used to represent the worlds.
  worldObject = [WorldObject createBegin: [self getZone]];
  [worldObject setSizeX: worldXSize Y: worldYSize];
  worldObject = [worldObject createEnd];
  [worldObject InitNumberOfObj: InitNumObjects];
  [worldObject InitSetObjects: seedValueObjects];

  // Now set up the grid used to represent agent position
  worldAgent = [Grid2d createBegin: [self getZone]];
  [worldAgent setSizeX: worldXSize Y: worldYSize];
  worldAgent = [worldAgent createEnd];

  // Create a list to keep track of the agents in the model.
  agentList = [List create: [self getZone]];
  
  // Create agents themselves. This is a fairly complex step, as is
  // appropriate: the agents are essential aspects of the simulation.
  // First, a quick hack. During creation we might put several agents
  // in the same square. This is a design flaw, but it's one that's not
  // fatal, so we ask the world object not to warn us about it. This is
  // not an example to be emulated :-)
  [worldAgent setOverwriteWarnings: 0];

  // Now a loop to create a bunch of agents.
  // for AgentProbaCamera
  for (i = 0; i < numAgentCamera; i++) {
    AgentCamera *aagt;
    int coordX, coordY;

    // Create the agent, set the creation time variables
    aagt = [AgentCamera createBegin: [self getZone]];
    [aagt setWorlds: worldAgent: worldObject];
    aagt = [aagt createEnd];

    // Add the agt to the end of the list.
    [agentList addLast: aagt];

    //coordX = [uRand rMax: worldXSize];
    //coordY = [uRand rMax: worldYSize];
    coordX = [uRand getIntegerWithMin: 0L withMax: worldXSize-1];
    coordY = [uRand getIntegerWithMin: 0L withMax: worldYSize-1];

    // to avoid to place an agent on a cell containing an object
    while (([worldObject getValueAtX: coordX Y: coordY] != 0) || ([worldAgent getValueAtX: coordX Y: coordY] != 0)){
      //printf("occupied: x= %d, y= %d\n", coordX, coordY);
      coordX = [uRand getIntegerWithMin: 0L withMax: worldXSize-1];
      coordY = [uRand getIntegerWithMin: 0L withMax: worldYSize-1];
      //coordX = [uRand rMax: worldXSize];
      //coordY = [uRand rMax: worldYSize];
    }
    //printf("agt: x= %d, y= %d\n", coordX, coordY);
    [aagt setSeed: 200+i]; // seed for the random generator of the agent
    [aagt setX: coordX Y: coordY];
  
    [aagt setRotAngle: 0];
    [aagt setNbPointSeen: 0];

    [aagt setCameraInit];
    
  }

  [worldAgent setOverwriteWarnings: 1];		  // ok, done cheating.

  return self;
}

// Here is where the model schedule is built, the data structures
// that define the simulation of time in the mode. The core is an
// actionGroup that has a list of actions. That's then put in a Schedule.
-buildActions {
  [super buildActions];

  modelActions = [ActionGroup create: [self getZone]];
  [modelActions createActionForEach: agentList message: M(step)];

  // Then we create a schedule that executes the modelActions. modelActions
  // is an ActionGroup, by itself it has no notion of time. In order to
  // have it executed in time, we create a Schedule that says to use
  // the modelActions ActionGroup at particular times.
  // This schedule has a repeat interval of 1, it will loop every time step.
  // The action is executed at time 0 relative to the beginning of the loop.
  
  modelSchedule = [Schedule createBegin: [self getZone]];
  [modelSchedule setRepeatInterval: 1];
  modelSchedule = [modelSchedule createEnd];
  [modelSchedule at: 0 createAction: modelActions];

  return self;
}

// Now set up the model's activation. swarmContext indicates where
// we're being started in - typically, this model is run as a subswarm
// of an observer swarm.
-activateIn: (id) swarmContext {
  // First, activate ourselves via the superclass activateIn: method.
  // Just pass along the context: the activity library does the right thing.
  [super activateIn: swarmContext];

  // Now activate our own schedule.
  [modelSchedule activateIn: self];

  // Finally, return our activity.
  return [self getSwarmActivity];
}

@end

