// Copyright  1995-2000 Swarm Development Group.
// No warranty implied, see LICENSE for terms.

// SortingAnts are agents in a 2d world with simple behaviour:
//   wander around randomly
//   if currently carrying an object
//     if near object, try to drop with prob pDropWhenNear
//     else try to drop with prob pDropWhenAlone
//   else (not carrying) if standing on object
//     if object is alone, pick up with prob pPickupWhenAlone
//     else pick up with prob pPickupWhenNear
// SortingAnts can occupy the same space at the same time.
// SortSpace enforces details about blocks on each other.

#import "SortingAnt.h"
#import <random.h>

// class variables: probability of pickups.
double pPickupOnNbs[5] = {0, 0, 0, 0, 0}, pDropOnNbs[5];
extern double pickupExponent;

@implementation SortingAnt

// In addition to setting the world, we also cache the worldSize for speed.
- setWorld: (id <Grid2d>)w Space: (SortSpace *)s
{
  world = w;
  sortSpace = s;
  worldSize = [world getSizeX];
  if (worldSize != [world getSizeY])
    {
      fprintf(stderr, "Sorting ants currently only work in square worlds.\n");
      return nil;
    }
  return self;
}

// after we set our own internal coordinates, we also tell the 2d grid
// where we are sitting.
- setX: (int)inX Y: (int)inY
{
  x = inX;
  y = inY;
  [world putObject: self atX: x Y: y];
  return self;
}

// Extra bits of display code: setting our colour, drawing on a window.
- setColors: (Color)c : (Color)c2
{
  carryColor = c;
  emptyColor = c2;
  return self;
}

- setPixmaps: ep : cp
{
  emptyPixmap = ep;
  carryPixmap = cp;
  return self;
}

- createEnd
{
  if (sortSpace == nil || world == nil)
    {
      [InvalidCombination raiseEvent: "Ant not set up correctly.\n"];
      return nil;
    }

  // do we need to initialize?  
  if (pPickupOnNbs[0] == 0)
    {
      int i;
      
      pPickupOnNbs[0] = 1.0;			  // always.
      pDropOnNbs[4] = pPickupOnNbs[0];		  // symmetry
      for (i = 1; i < 4; i++)
        {
          pPickupOnNbs[i] = pPickupOnNbs[i-1]*pickupExponent;
          pDropOnNbs[4-i] = pPickupOnNbs[i];		  // symmetry
        }
      pPickupOnNbs[4] = 0.0;			  // always.
      pDropOnNbs[0] = pPickupOnNbs[4];		  // symmetry
    }
  
  return self;
}

- step
{
  int newX, newY;
  SortParcel objectHere;
  int numNeighboursHere;

  objectHere = [sortSpace getValueAtX: x Y: y];

  if (carriedObject && objectHere == 0)
    {
      // if carrying and spot empty
      float pDrop;
      numNeighboursHere = [sortSpace num4NeighboursAtX: x Y: y Type: carriedObject];
      pDrop = pDropOnNbs[numNeighboursHere];
      if ([uniformDblRand getDoubleWithMin: 0.0 withMax: 1.0] < pDrop)
        {
          [sortSpace drop: carriedObject AtX: x Y: y];
          carriedObject = 0; // forget we're carrying
        }
    }
  else if (carriedObject == 0 && objectHere)
    { 
      // spot not empty, no carry.
      float pPickup;
      
      numNeighboursHere = [sortSpace num4NeighboursAtX: x Y: y Type: objectHere];
      pPickup = pPickupOnNbs[numNeighboursHere];
      if ([uniformDblRand getDoubleWithMin: 0.0 withMax: 1.0] < pPickup)
        carriedObject = [sortSpace pickupAtX: x Y: y];
    }
  
  // now move randomly. Note: we can walk on other ants, which screws up
  // the Grid2d code because it puts two objects in one place. Bad!
  newX = ((int)x + [uniformIntRand getIntegerWithMin: -1 withMax: 1] + worldSize) % worldSize;
  newY = ((int)y + [uniformIntRand getIntegerWithMin: -1 withMax: 1] + worldSize) % worldSize;
  
  [world putObject: nil atX: x Y: y];
  x = newX;
  y = newY;
  [world putObject: self atX: newX Y: newY];
  
  return self;
}

- drawSelfOn: (id <Raster>)r
{
#ifdef COLORS
  if (carriedObject)
    [r drawPointX: x Y: y Color: carryColor];
  else
    [r drawPointX: x Y: y Color: emptyColor];
#else
  if (carriedObject)
    [r draw: carryPixmap X: x Y: y];
  else
    [r draw: emptyPixmap X: x Y: y];
#endif
  return self;
}

@end
