// Swarm library. Copyright (C) 1996 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.
#define __USE_FIXED_PROTOTYPES__  // for gcc headers

// with Pietro Terna's changements, se //pt


#import <stdlib.h>
#import <string.h>

#import <collections.h>

//pt #import "ListShuffler.h"
//pt with 1.3 use the following (ListShuffler is included in simtools)
#import <simtools.h>

#import "MoGrid2d.h"		// Multiple Occupancy Discrete2d grid

// When an object is added to an empty cell on the grid, a List is created
// at that cell to hold the object. Further additions of objects to the
// cell are just added to the list.

// Whenever objects are removed from the list, if the list becomes empty
// it is dropped (removed).

// The code for Lists allow for adding arbitrary data values rather than
// object id's, and the code for this grid allows for that.

@implementation MoGrid2d

// Private methods:

-makeOffsets {
 unsigned i;

  // precalculate offsets based on the y coordinate. This lets
  // us avoid arbitrary multiplication in array lookup.
  offsets = [[self getZone] alloc: ysize * sizeof(*offsets)];

  for (i = 0; i < ysize; i++)
    offsets[i] = xsize * i;		// cache this multiplication
  return self;
}

-(id *) allocLattice {
  void * p;
  p = [[self getZone] alloc: xsize * ysize * sizeof(id)];
  memset(p, 0, xsize * ysize * sizeof (id));
  return p;
}

-setSizeX: (int) x Y: (int) y {
  if (lattice)
    [InvalidArgument raiseEvent: 
    "MoGrid2d: You cannot reset the grid size after creation.\n"];
  xsize = x;
  ysize = y;
  return self;
}

-createEnd {
  if (xsize <= 0 || ysize <= 0)
    [InvalidCombination raiseEvent: 
    "MoGrid2d: Invalid size in creation: %d %d\n", xsize, ysize];
  lattice = [self allocLattice];
  [self makeOffsets];
  shuffler = [ListShuffler create: [self getZone]];
  return self;
}

// Creation and parameter setting:

+create: (id) aZone setSizeX: (unsigned) x Y: (unsigned) y {
   MoGrid2d * grid;

   grid = [MoGrid2d createBegin: aZone];
   [grid setSizeX: (int) x Y: (int) y];
   grid = [grid createEnd];

   return grid;
}

// Accessing data values:

-(int) getSizeX {
  return xsize;
}

-(int) getSizeY {
  return ysize;
}

-(id *) getLattice {
  return lattice;
}

//pt -(int *) getOffsets {
-(long *) getOffsets {  //pt long with Grid2d inheritance
  return offsets;
 }

// Add objects to lists at cells:

-addFirst: (id) anObject atX: (int) x Y: (int) y {
  id list;

  list = *mo2dSiteAt(lattice, offsets, x, y);

  if (!list) {						// site is empty now
     list = [List create: [self getZone]];		// create empty list
     *mo2dSiteAt(lattice, offsets, x, y) = list;	// put list at cell
  }

  [list addFirst: anObject];				// add object to list

   return self;
}

-pushObject: (id) anObject atX: (int) x Y: (int) y {
  id list;

  list = *mo2dSiteAt(lattice, offsets, x, y);
  if (!list) {						// site is empty now
     list = [List create: [self getZone]];		// create empty list
     *mo2dSiteAt(lattice, offsets, x, y) = list;	// put list at cell
  }
  [list addFirst: anObject];				// add object to list
  return self;
}

-addLast: (id) anObject atX: (int) x Y: (int) y {
  id list;

  list = *mo2dSiteAt(lattice, offsets, x, y);

  if (!list) {						// site is empty now
     list = [List create: [self getZone]];		// create empty list
     *mo2dSiteAt(lattice, offsets, x, y) = list;	// put list at cell
  }

  [list addLast: anObject];				// add object to list

  return self;
}

-enQueObject: (id) anObject atX: (int) x Y: (int) y {
  id list;

  list = *mo2dSiteAt(lattice, offsets, x, y);
  if (!list) {						// site is empty now
     list = [List create: [self getZone]];		// create empty list
     *mo2dSiteAt(lattice, offsets, x, y) = list;	// put list at cell
  }
  [list addLast: anObject];				// add object to list
  return self;
}

// Remove objects from lists at cells:

-(id) getFirstAtX: (int) x Y: (int) y {
   id list,item;
   int num;

   list = *mo2dSiteAt(lattice, offsets, x, y);
   if (!list) return list;	// i.e. nil; cell has no list

   num = [list getCount];
   
   if (num > 1)
     return [list removeFirst];

   if (num == 1) {
     item = [list removeFirst];
     // Now list should be empty; remove list
     [list drop];
      *mo2dSiteAt(lattice, offsets, x, y) = nil;	// zero the cell value
     return item;  
   }

   // If we're here, num = 0 which should not happen!

     [list drop];
      *mo2dSiteAt(lattice, offsets, x, y) = nil;	// zero the cell value
     printf("MoGrid2d: deleteFirst: found empty list!\n");
     return nil;

}

-(id) popObjectAtX: (int) x Y: (int) y {
   id list,item;
   int num;

   list = *mo2dSiteAt(lattice, offsets, x, y);
   if (!list) return list;	// i.e. nil; cell has no list
   num = [list getCount];
   if (num > 1)
     return [list removeFirst];
   if (num == 1) {
     item = [list removeFirst];
     [list drop];
      *mo2dSiteAt(lattice, offsets, x, y) = nil;	// zero the cell value
     return item;  
   }
// If we're here, num = 0 which should not happen!
     [list drop];
      *mo2dSiteAt(lattice, offsets, x, y) = nil;	// zero the cell value
     printf("MoGrid2d: deleteFirst: found empty list!\n");
     return nil;
}

-(id) deQueObjectAtX: (int) x Y: (int) y {
   id list,item;
   int num;

   list = *mo2dSiteAt(lattice, offsets, x, y);
   if (!list) return list;	// i.e. nil; cell has no list

   num = [list getCount];
   if (num > 1)
     return [list removeFirst];
   if (num == 1) {
     item = [list removeFirst];
     [list drop];
      *mo2dSiteAt(lattice, offsets, x, y) = nil;	// zero the cell value
     return item;  
   }
// If we're here, num = 0 which should not happen!
     [list drop];
      *mo2dSiteAt(lattice, offsets, x, y) = nil;	// zero the cell value
     printf("MoGrid2d: deleteFirst: found empty list!\n");
     return nil;
}

-(id) getLastAtX: (int) x Y: (int) y {
   id list,item;
   int num;

   list = *mo2dSiteAt(lattice, offsets, x, y);
   if (!list) return list;	// i.e. nil; cell has no list

   num = [list getCount];
   
   if (num > 1)
     return [list removeLast];

   if (num == 1) {
     item = [list removeLast];
     [list drop];
      *mo2dSiteAt(lattice, offsets, x, y) = nil;	// zero the cell value
     return item;  
   }

   // If we're here, num = 0 which should not happen!

     [list drop];
      *mo2dSiteAt(lattice, offsets, x, y) = nil;	// zero the cell value
     printf("MoGrid2d: deleteFirst: found empty list!\n");
     return nil;

}

-(id) deleteObject: (id) anObject atX: (int) x Y: (int) y {
   id list;

   list = *mo2dSiteAt(lattice, offsets, x, y);
   if (!list) return list;	// no list there, so return nil

   return [list remove: anObject];
}


-(void) dropListAtX: (int) x Y: (int) y {
   id list;

   list = *mo2dSiteAt(lattice, offsets, x, y);
   if (list) {
     // [list forEach: M(drop)]; // do *not* drop the objects
     [list drop];
     *mo2dSiteAt(lattice, offsets, x, y) = nil;	// zero the cell value
   }
   // no return
}

-(void) dropAllAtX: (int) x Y: (int) y {
   id list;

   list = *mo2dSiteAt(lattice, offsets, x, y);
   if (list) {
     [list forEach: M(drop)];
     [list drop];
     *mo2dSiteAt(lattice, offsets, x, y) = nil;	// zero the cell value
   }
   // no return
}

// Retrieve object data from cells:

-(int) numberOfObjectsAtX: (int) x Y: (int) y {
   id list;

   list = *mo2dSiteAt(lattice, offsets, x, y);

   if (!list) return 0;
   else return [list getCount];
}


-getListAtX: (int) x Y: (int) y {
  return *mo2dSiteAt(lattice, offsets, x, y);
}

-getCopyAtX: (int) x Y: (int) y {
   id list, newlist;
   int i, num;

   list = *mo2dSiteAt(lattice, offsets, x, y);
   if (!list) return list;

   num = [list getCount];
   if (num == 0) return nil;

   // copy the list:
   newlist = [List create: [self getZone]];
   for (i=0; i<num; i++)
     [newlist addLast: [list atOffset: i]];

   return newlist;
}


// Retrieve randomized lists from cells:

-(id) getShuffledListAtX: (int) x Y: (int) y {
   id list;

   list = *mo2dSiteAt(lattice, offsets, x, y);
   if (!list) return list;	// nil, i.e. no list

   if ([list getCount] == 0)	// Should Not Happen
     return nil;
   else {
     [shuffler shuffleWholeList: list];
     return list;
   }
}


-getShuffledCopyAtX: (int) x Y: (int) y {
   id list,newlist;
   int num,i;

   list = *mo2dSiteAt(lattice, offsets, x, y);
   if (!list) return list;	// nil, i.e. no list

   if ([list getCount] == 0)	// Should Not Happen
     return nil;
   else {
     newlist = [List create: [self getZone]];
     num = [list getCount];
     for (i=0; i<num; i++)
       [newlist addLast: [list atOffset: i]];
     return [shuffler shuffleWholeList: newlist];
   }
}


-(void) shuffleListAtX: (int) x Y: (int) y {
   id list;

   list = *mo2dSiteAt(lattice, offsets, x, y);
   if (!list) return;

   [shuffler shuffleWholeList: list];
}

-(void) shuffleAll {
   id list;
   unsigned ii,jj;

   for  (ii=0; ii<xsize; ii++) 
     {
       for  (jj=0; jj<ysize; jj++) 
	 {
	   list = *mo2dSiteAt(lattice, offsets, ii, jj);
	   if (list) [shuffler shuffleWholeList: list];
	 } 
     }
}

-setList: (id) newlist atX: (int) x Y: (int) y {
   id list;

   list = *mo2dSiteAt(lattice, offsets, x, y);

   if (list) 
   [InvalidCombination raiseEvent:
   "MoGrid2d: setList:atX:Y:  (x,y) already occupied!\n"];

   *mo2dSiteAt(lattice, offsets, x, y) = newlist;

   return self;
}

@end
