// Metabolizing Agents. Copyright (C) 1998-1999 Peter Zvirinsky
// This program is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

// geneticsInt.m                                    Peter Zvirinsky oct,1998
// Library for implementing genetics in SWARM agents with Integer values
//    genome:  binary string
//    gene:    set of bits, type INT,BITS
//    classes: GeneMap, GeneType				      

#import "GeneticsInt.h"

//====================================================================
// class GeneType, structure for describing type and parameters
// of particular gene
//====================================================================

@implementation Gene

- setGName:   (char *)nameEntry
{ if( strlen(nameEntry) > 20 ) 
    [InvalidCombination raiseEvent: "GeneticsInt: GeneName exceeding 20 chars\n"];
  else
    strcpy(name,nameEntry);  
  return self; 
}

- setGType:   (char *)ty   
{ if( strlen(ty) > 4 ) 
    [InvalidCombination raiseEvent: "GeneticsInt: GeneType exceeding 4 chars\n"];
  else
    strcpy(type,ty);  
  return self; 
}

- setGStart:  (unsigned)st      {   start = st;  return self; }
- setGLength: (unsigned)le      {   len = le;  return self; }
- setGMin:    (int) mi          {   min = mi;  return self; }
- setGMax:    (int) ma          {   max=ma;  return self;}
- (char *)   getGName    { return name; }
- (char *)   getGType    { return type; }
- (unsigned) getGStart   { return start; }
- (unsigned) getGLength  { return len; } 
- (int)      getGMin     { return min; }
- (int)      getGMax     { return max; }

@end

//====================================================================
// class GeneMap, List of pointers to GeneTypes
//====================================================================


@implementation GeneMap

- createEnd
{
  id tempObj;
  
  tempObj = [super createEnd];
  geneMap = [List create: [self getZone]];
  gMapLength = 0;
  return tempObj;
}

- createGeneType: (char *)   ty
            Name: (char *)   na
           Start: (unsigned) st
          Length: (unsigned) len
             Min: (int)      mi
             Max: (int)      ma
{
  Gene *gene;

  gene = [Gene create: [self getZone]];
  [gene setGName:   na];
  [gene setGType:   ty];
  [gene setGStart:  st];
  [gene setGLength: len];
  [gene setGMin:    mi];
  [gene setGMax:    ma];
  [geneMap addLast: gene];
  gMapLength += len; 
//  printf("\nCreating Gene:%s Type:%s St:%d Len:%d <%d %d>",na,ty,st,len,mi,ma);
  return self;
}

- getGeneWithName: (char *) nam
{
  Gene *gene;
  unsigned i;
  char *txt;
  
  for(i=0; i< [geneMap getCount]; i++)
  {  
    gene = [geneMap atOffset: i];
    txt = [gene getGName];
    if( strcmp(nam,txt)==0 )
      return gene;
  }    
  [InvalidArgument raiseEvent: "GeneticsInt: Gene Name not found\n"];
}

- (char *)   getNameOfGene: (unsigned) index
{
  return [[geneMap atOffset: index] getGName];
}

- (char *)   getTypeOfGene: (char *)nam
{  return [[self getGeneWithName: nam] getGType]; }

- (unsigned) getStartOfGene: (char *)nam
{  return [[self getGeneWithName: nam] getGStart]; }

- (unsigned) getLengthOfGene: (char *)nam
{  return [[self getGeneWithName: nam] getGLength]; }

- (int)      getMinOfGene: (char *)nam
{  return [[self getGeneWithName: nam] getGMin]; }

- (int)      getMaxOfGene: (char *)nam
{  return [[self getGeneWithName: nam] getGMax]; }

- (unsigned) getGMapLength { return gMapLength; }


// genome is array of chars , here with values 0 or 1
- (char *) createGenome
{
  char *p;

  if( (p= (char *) calloc( gMapLength, sizeof(char)))==NULL)
    [InvalidCombination raiseEvent: "Genome: Cannot allocate Genome\n"];
  return p;
}

- randInitGenome: (char *) p
{
  int i;

  for( i=0; i< gMapLength; i++)
    p[i]= [uniformIntRand getIntegerWithMin: 0 withMax: 1];
  return self;
}

- (int) getValueOfGene: (char *)nam ofGenome: (char *)p
{
  int value,start,len, i,mi,ma;
  
  value=0;
  start = [self getStartOfGene: nam];
  len =[self getLengthOfGene: nam];
  for(i=0; i< len; i++)
  {  
    value+= (1<<i) * p[start+len-i-1];
  }
    //scaling if float else bits type i.e. no scaling
  if( ! strcmp("int",[self getTypeOfGene: nam]) )
  {
    mi = [self getMinOfGene:nam];
    ma = [self getMaxOfGene:nam];
    value=(int)( mi+(ma-mi)*((float)value/((1<<len)-1)) + 0.5);    // scaling
  }  
  return value;
}

-printGenome: (char *)p
{
  int i;

  printf("\n");
  for( i=0; i< gMapLength; i++)
    printf("%d",p[i]);
  return self;
}

- (char *) cloneGenome: (char *)p1
{
  char *p2;
  int i;
  
  p2=[self createGenome];
  for( i=0; i< gMapLength; i++)
    p2[i]=p1[i];
  return p2;
}

- (int) mutateGenome: (char *) p withProb: (float) pmut
{
  unsigned pos;

  if ([uniformDblRand getDoubleWithMin: 0.0 withMax: 1.0] <= pmut)
  {
    pos=[uniformIntRand getIntegerWithMin: 0 withMax: gMapLength-1];
    p[pos]= (p[pos]+1)%2;           // flip bin0 <-> bin1
    return 1;
  }
  return 0;
}

@end

        

