// Copyright (C) 1995 The Santa Fe Institute.
// No warranty implied, see LICENSE for terms.

#include <math.h>

#include "Hebb2Layer.h"

@implementation Hebb2Layer
- setTrainingConstant: (float) _alpha : (float) _beta {
    alpha =_alpha;
    beta = _beta;
    return self;
}

- setLayerSize: (unsigned) inpLyUnits: (unsigned) opLyUnits
	      : (unsigned) hidLyUnits; {

    inpSize = inpLyUnits;
    opSize = opLyUnits;
    hidSize = hidLyUnits;
  
    inpLy = (float * ) calloc( inpSize, sizeof( float ) ); 
    Wgts1 = (float * ) calloc( inpSize*hidSize, sizeof( float ) );
    hidLy = (float * ) calloc( hidSize, sizeof( float ) ); 
    Wgts2 = (float * ) calloc( opSize*hidSize, sizeof( float ) );
    opLy = (float * ) calloc( opSize, sizeof( float ) ); 

    if ( !Wgts1 || !Wgts2 || !opLy || !hidLy 
	 ||!inpLy || !opSize || !hidSize) {
	[OutOfMemory raiseEvent: "Problems with  malloc in Hebb2Layer" ];
	return nil;
    }
    return self;
}

//--------------------------------------------
-free {
    free(inpLy );
    free( Wgts1 );
    free( hidLy );
    free( Wgts2 );
    free( opLy );
    [self drop];
    return nil;
}

//-----------------------------------------------------
- createEnd {
    if ( !Wgts1 || !Wgts2 || !opLy || !hidLy 
	 || !inpLy || !opSize || !hidSize ) {
	[InvalidCombination raiseEvent:
				"Problems with  malloc or network size\n" ];
	return nil;
    }
    return self;
}


//------------------------------------------------------
// Dummy to avoid warnings
- setWeightsTo : (float * ) wgtsInputLayer: (float* )wgtsOutputLayer {

    int i, j, k;

    for ( i = 0; i < hidSize; i++ ) {
	for ( j = 0; j < inpSize; j++ ) {
	     Wgts1[ i*inpSize + j ] =  wgtsInputLayer[ i*inpSize + j ];
	}
    }

    for ( k = 0; i < opSize; i++ ) {
	for ( j = 0; j < hidSize; j++ ) {
	    Wgts2[ i*hidSize + j ] = wgtsOutputLayer[ i*hidSize + j ];
	}
    }
    return self;
}

//------------------------------------------------------
-setFromString: (unsigned char *) _str {
// 2 bytes by weight inits weights with  floats between -1 and 1
//------------------------------------------------------

    unsigned i, j, foo;
    unsigned powers[BYTESWGT] = {1,256};

    if ( !Wgts1 || !Wgts2 ) {
	[WarningMessage raiseEvent: "Warning: Weights not set..."];
	return nil;
    }

    for ( i = 0; i < inpSize*hidSize; i++ ) {
	foo = 0;
	for ( j = 0; j < BYTESWGT; j ++ ) { // Little-endian order + sign
	    foo += powers[j]*_str[i*BYTESWGT+j];
	}
	Wgts1[i] = (foo*2)/(powers[BYTESWGT-1]*256.0) -1;
//	printf( "Weight %d %f\n", i, Wgts1[i] );
    } 

    for ( i = 0; i < opSize*hidSize; i++ ) {
	foo = 0;
	for ( j = 0; j < BYTESWGT; j ++ ) { // Little-endian order + sign
	    foo += powers[j]*_str[i*BYTESWGT+j];
	}
	Wgts2[i] = (foo*2)/(powers[BYTESWGT-1]*256.0) -1;
//	printf( "Weight %d %f\n", i, Wgts2[i] );
    }
    
    return self;
}

//------------------------------------------------------
- setRandomWeights {
// Inits weights with random floats between -1 and 1
//------------------------------------------------------

    unsigned i;
    for ( i = 0; i < inpSize*hidSize; i++ ) {
	Wgts1[i] = (float)[uniformDblRand getDoubleWithMin:-1.0L withMax: 1.0L];
//	printf( "Weight %d %f\n", i, Wgts[i] );
    } 

    for ( i = 0; i < opSize*hidSize; i++ ) {
	Wgts2[i]=(float)[uniformDblRand getDoubleWithMin:-1.0L withMax: 1.0L];
//	printf( "Weight %d %f\n", i, Wgts[i] );
    }
    
    return self;
}


//------------------------------------------------------------
- (float)   getWeights: (unsigned) Layer: (unsigned) i: (unsigned) j {
// Returns a weight in a layer, implements a protocol function
// Only one layer this time, so Layer acts as dummy
// First index is always the lowest layer
// -----------------------------------------------------------

   if ( Layer > 1 )
       return 0;		// 0 is first layer, 1 second

   switch( Layer ) {
       case 0:
	   if ( ( i < hidSize ) && ( j < inpSize) )
	       return Wgts1[ i*inpSize + j];
	   else
	       return 0; 
	   break;
       case 1:
	   if ( ( i < opSize ) && ( j < hidSize) )
	       return Wgts2[ i*hidSize + j];
	   else
	       return 0;
   }

   return 0;			// To make the compiler happy
	   
}


//-----------------------------------------------------
//            Learning and recalling functions
//-----------------------------------------------------

//-----------------------------------------------------
-(unsigned ) feedForward: (float *) inpVec {
// dimension should be the same than the input layer

    unsigned i, j, maxOp;
    float maxVal = -1e22;	// Smallest number around
    memcpy( inpLy, inpVec, inpSize*sizeof(float) );

//    printf ( "\nHidlayer : " );
    for ( i = 0; i < hidSize; i++ ) {
	hidLy[i] = 0;		// Reset output layer
	for ( j = 0; j < inpSize; j++ ) {
	    hidLy[i] += Wgts1[ i*inpSize + j ]*inpVec[j];
	}
	hidLy[i] = ( hidLy[i] > 0 )?1:-1;
//	printf ( "%.1f ", hidLy[i] );
    }

    for ( i = 0; i < opSize; i++ ) {
	opLy[i] = 0;		// Reset output layer
	for ( j = 0; j < hidSize; j++ ) {
	    opLy[i] += Wgts2[ i*hidSize + j ]*hidLy[j];
	}
//	printf( "Output Layer %f\n", opLy[i] );
	if ( opLy[i] > maxVal ) {
//	    printf( "Changing to %d %f\n", i, opLy[i]  );
	    maxOp = i;
	    maxVal = opLy[i];
	}
    }
    // Change activations of output layer
    for ( i = 0; i < opSize; i++ ) {
	opLy[i] = ( i == maxOp )?1:-1;
    }

    return maxOp;
}

-normalize {
// It normalizes the weights that flow out of a input or 
// hidden neuron. Their squares should be one.
    float sum = 0;
    int i,j;
    for ( j = 0; j < inpSize; j++ ) {
	sum = 0;
	for ( i = 0; i < hidSize; i++ ) {
	    float foo = Wgts1[ i*inpSize + j ];
	    sum += foo * foo;
	}
	sum = sqrt( sum );	// square root of squared terms
	// Now divide everything
	for ( i = 0; i < hidSize; i++ ) {
	    Wgts1[ i*inpSize + j ] /= sum;
	}
    } 
    // Ditto for second layer
    for ( j = 0; j < hidSize; j++ ) {
	sum = 0;
	for ( i = 0; i < opSize; i++ ) {
	    float foo = Wgts2[ i*hidSize + j ];
	    sum += foo * foo;
	}
	sum = sqrt( sum );	// square root of squared terms
	// Now divide everything
	for ( i = 0; i < opSize; i++ ) {
	    Wgts1[ i*hidSize + j ] /= sum;
	}
    } 
    return self;
}


//------------------------------------------------------
- train: (float) delta {
//  With renormalization
//------------------------------------------------------

    int sgn;
    unsigned i, j;
    for ( i = 0; i < hidSize; i++ ) {
	for ( j = 0; j < inpSize; j++ ) {
	    sgn = (inpLy[j]*hidLy[i] > 0 )?1:-1; 
//	    printf( "Sg*delta %f\n", sgn*fabs(delta) );
	    Wgts1[ i*inpSize + j ] += sgn*alpha;
	}
    }    
    
    // reinforcement is applied to the second layer
    for ( i = 0; i < opSize; i++ ) {
	for ( j = 0; j < hidSize; j++ ) {
	    sgn = (hidLy[j]*opLy[i] > 0 )?1:-1;
//	    printf( "Sg*delta %f\n", sgn*delta);
	    Wgts2[ i*hidSize + j ] += delta*beta*sgn;
	}
    }   
    [self normalize];
    return self;
}


// ------------------------- Info functions ------------------

-(unsigned) getInpSize {
    return inpSize;
}

-(BOOL) getLayers {
    return 1;
}

-(unsigned) getHiddenSize {
    return hidSize;
}

@end
