// Copyright (C) 1995-1998 The Santa Fe Institute.
// No warranty implied, see LICENSE for terms.

#import "BaseSingleLayer.h"
#import <random.h>

#include <math.h>
#include <misc.h>

@implementation BaseSingleLayer

//--------------------------------------------
// Instance objects, taken from the protocol
- setLayerSize: (unsigned)inpLyUnits: (unsigned)opLyUnits
{
    inpSize = inpLyUnits;
    numNeurons = opLyUnits;
  
    inpLy = (float *)xcalloc (inpSize, sizeof (float)); 
    Wgts = (float *)xcalloc (inpSize * numNeurons, sizeof (float));
    opLy = (float *)xcalloc (numNeurons, sizeof (float)); 

    if (!Wgts || !opLy || !inpLy || !numNeurons)
      {
	[InvalidCombination raiseEvent:  
                              "Problems with  malloc or network size\n" ];
	return nil;
      }
    return self;
}

- setTrainingConstant: (float)_alpha
{
    alpha = _alpha;
    return self;
}

//--------------------------------------------
- free
{
  xfree (inpLy);
  xfree (Wgts);
  xfree (opLy);
  [self drop];
  return nil;
}

//-----------------------------------------------------
- createEnd
{
  if ( !Wgts || !opLy || !inpLy || !numNeurons ) {
    [InvalidCombination
      raiseEvent: "Problems with  malloc or network size\n" ];
  }
  return self;
}

//------------------------------------------------------
// wgts should have the same dimensions than Wgts
- setWeightsTo: (float *)wgts
{
  unsigned i, j;

  for (i = 0; i < numNeurons; i++)
    for (j = 0; j < inpSize; j++)
      Wgts[ i*inpSize + j ] = wgts[ i*inpSize + j];
  return self;
}


//------------------------------------------------------
// Inits weights with random floats between -1 and 1
//------------------------------------------------------
- setRandomWeights
{
    unsigned i;

    for (i = 0; i < inpSize * numNeurons; i++)
      Wgts[i]=(float)[uniformDblRand getDoubleWithMin:-1.0L withMax: 1.0L];

    return self;
}

//------------------------------------------------------
// 2 bytes by weight inits weights with  floats between -1 and 1
//------------------------------------------------------
- setFromString: (unsigned char *)_str
{
    unsigned i, j, foo;
    unsigned powers[BYTESWGT] = {1,256};

    for (i = 0; i < inpSize * numNeurons; i++)
      {
	foo = 0;
	for ( j = 0; j < BYTESWGT; j ++ ) 
          // Little-endian order + sign
          foo += powers[j]*_str[i*BYTESWGT+j];
	Wgts[i] =  (1 - foo / powers[BYTESWGT] * 256.0) * 2 - 1;
//	printf( "Weight %d %f\n", i, Wgts[i] );
      } 
    
    return self;
}

//------------------------------------------------------------
// Returns a weight in a layer, implements a protocol function
// Only one layer this time, so Layer acts as dummy
// -----------------------------------------------------------
- (float)getWeights: (unsigned)i: (unsigned)j
{
  if ((i < numNeurons) && (j < inpSize))
    return Wgts[i * inpSize + j];
  else
    return 0.0;
}

//-----------------------------------------------------
//            Learning and recalling functions
//-----------------------------------------------------

//-----------------------------------------------------
- (unsigned)feedForward: (float *)inpVec
{
// To implement in subClasses
  return numNeurons + 1;
  //    [SubclassMustImplement raiseEvent: "Error in BaseSingleLayer feedforward method"];
}


//------------------------------------------------------
- train: (float)delta
{
  // To implement in subclasses
  return nil;
  //    [SubclassMustImplement raiseEvent: "Error in BaseSingleLayer train method"];
}


//----------- Info functions --------------------------

- (unsigned)getInpSize
{
    return inpSize;
}

- (BOOL)getLayers
{
    return 0;
}

@end
