/* 
  Coord.h

  This defines the <Coord> protocol, which is an
  attempt at a generalised protocol suitable for
  "coordinate" objects on "any" kind of "regular",
  discrete lattice.

  It is unlikely to be appropriate for use in
  *continuous* spaces.

  A Coord object is always specific to some particular
  Lattice object. This must be specified at create
  time and cannot be changed thereafer.

  Thus, the direct +create: or +createBegin: methods
  are *not* supported (and should cause a fatal
  exception if invoked).

  Instead, Coord objects should typically be created
  either by invoking a -createOriginCoord: method on
  some Lattice object (see the <Lattice> protocol in
  Lattice.h) or by using the +create:withLattice:
  method, specifying the lattice, by the or -copy:
  method on some already existing Coord object.

  The primary functionality of a Coord object is to
  give access to the object (if any) at the
  corresponding position in the Lattice.  Coords can
  also be operated upon to change their position in the
  space, both by translating relative to another Coord
  (vector addition, in effect), or translating in a
  specified "neighbor direction" supported by the
  particular Lattice.  Possible neighbor directions are
  given symbolic names in the NeighborCode enumerated
  type (see NeighborCode.h).  A Coord can also be used
  to systematically traverse the entire Lattice.


  Barry McMullin <mcmullin@eeng.dcu.ie>
  FEB-1997

*/

#import <swarmobject.h>

#import "NeighborCode.h"

@protocol Coord <Copy>

+create: (id) aZone withLattice: (id) inLattice;

-(id)   getLatticeObject;
// Get (a pointer to) the corresponding Lattice Object.
// This might be better called simply -getLattice, but
// that method name was already in use in a core Swarm
// Lib (Space), with a different interface
// specification. Darn.


-(void) traverseLattice;
// Repeated invocations of traverseLattice will (sooner or
// later!) cause the coordinate to traverse the entire
// lattice in some closed cycle where every site is
// visited exactly once per cycle.


-(void) translateByCoord: (id) coord;
-(void) translateByNeighborCode: (NeighborCode) neighborCode;
-(BOOL) isEqualToCoord:   (id) coord;


-(id)   getObject;
-(void) putObject: (id) anObject;

@end
