/* 
  Coord.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  FEB-1997

*/

#import "Lattice2d.h"
#import "Coord2d.h"

@implementation Coord2d

+(id) createBegin: (id) aZone {
  [InternalError raiseEvent:
    "createBegin: and create: not supported on Coord2d class.\n"
    "Use create:withLattice: instead.\n"];

  return (nil);
}

+(id) create: (id) aZone withLattice: (id) inLattice {
  Coord2d *finalSelf;

  finalSelf = [super createBegin: aZone];
  finalSelf->lattice = inLattice;
  finalSelf = [finalSelf createEnd];

  return (finalSelf);
}

-(id) getLatticeObject {
  return (lattice);
}

-(void) setX: (int) inX {
  x = inX;
}

-(int) getX {
  return x;
}

-(void) setY: (int) inY {
  y = inY;
}

-(int) getY {
  return y;
}


-(void) setRasterX: (int) inRasterX {
  x = (inRasterX / [lattice getRasterScaleFactor]);
}

-(int) getRasterX {
  return ([lattice getRasterScaleFactor] * x);
}

-(void) setRasterY: (int) inRasterY {
  unsigned scaleFactor;

  scaleFactor = [lattice getRasterScaleFactor];
  if ([lattice getTilingCode] == TriTiling)
    if ((x % 2) == 1)
      inRasterY -= (scaleFactor / 2);
  y = (inRasterY / scaleFactor);
}

-(int) getRasterY {
  unsigned scaleFactor;
  int rasterY;

  scaleFactor = [lattice getRasterScaleFactor];
  rasterY = scaleFactor * y;
  if ([lattice getTilingCode] == TriTiling)
    if ((x % 2) == 1)
      rasterY += (scaleFactor / 2);

  // Note that we do *not* get the aspect ratio right
  // on the raster co-ordinates of a Triangular tiling;
  // this is, in part at least, a limitation of wanting
  // to be able to use the same size raster to display
  // both square and triangular tilings...

  return rasterY;
}


-(id) copy: (id) aZone {
  return ([aZone copyIVars: self]);
}

-(int) normalizeScalar: (int) inScalar atModulus: (int) inModulus
{
  /* I have just tried and failed to find out
     definitively whether the behaviour of the %
     operator is well defined if either operand is
     negative.  I don't have a copy of the ISO C
     standard to hand.  My code here will at least work
     robustly (if inefficiently) *even* in the
     pathological case of large negative inScalar.  In the
     "typical" case of small negative inScalar, this code
     is fine. */
  while ((inScalar) < 0) inScalar += inModulus;
  return (inScalar % inModulus);
}

-(void) normalizeX {
  x = [self normalizeScalar: x atModulus: [lattice getSizeX]];
}

-(void) normalizeY {
  y = [self normalizeScalar: y atModulus: [lattice getSizeY]];
}

-(void) normalize {
  [self normalizeX];
  [self normalizeY];
}

-(void) traverseLattice {
  x++;
  [self normalizeX];
  if (x == 0) {
    y++;
    [self normalizeY];
  }
}

-(void) translateByCoord: (id) coord {
  x = x + [coord getX];
  y = y + [coord getY];
  [self normalize];
}

-(void) translateXByNeighborCode: (NeighborCode) neighborCode {
  switch (neighborCode) {
    case SqrEast:     
    case SqrNorthEast:
    case SqrSouthEast:     x++;      break;

    case SqrWest:
    case SqrSouthWest:
    case SqrNorthWest:     x--;      break;

    case SqrFarEast:       x += 2;   break;

    case SqrFarWest:       x -= 2;   break;

    case Center:
    case SqrNorth:
    case SqrSouth:
    case SqrFarNorth:
    case SqrFarSouth:                          break;

  default:
    [InternalError raiseEvent:
    "Invalid neighbor code (%d) encountered.", neighborCode];
  }

  [self normalizeX];
}

-(void) translateYByNeighborCode: (NeighborCode) neighborCode {
  /* Increasing y is taken as SqrSouth-ward; this is consistent
     with the swarm *raster* coordinate convention, where 
     increasing y is downward... */

  switch (neighborCode) {
    case SqrNorth:     
    case SqrNorthEast:
    case SqrNorthWest:     y--;      break;

    case SqrSouth:
    case SqrSouthEast:
    case SqrSouthWest:     y++;      break;

    case SqrFarNorth:      y -= 2;   break;

    case SqrFarSouth:      y += 2;   break;

    case Center:
    case SqrEast:
    case SqrWest:
    case SqrFarEast:
    case SqrFarWest:                           break;

  default:
    [InternalError raiseEvent:
    "Invalid neighbor code (%d) encountered.", neighborCode];
  }

  [self normalizeY];
}

-(BOOL) isTriNeighbor: (NeighborCode) neighborCode {
  /* Private Method.
     Classify neighborhood as Tri (or not). */

  BOOL isTri;

  switch (neighborCode) {
    case TriNorth:      
    case TriNorthEast:  
    case TriSouthEast:  
    case TriSouth:      
    case TriSouthWest:  
    case TriNorthWest:  
      isTri = YES;
      break;
    default:
      isTri = NO;
  }

  return (isTri);
}


-(unsigned) deformTriNeighbor: (unsigned) neighborCode {
  /* Private method.
     Deform a Tri neighbor into a "corresponding" Sqr neighbor on the 
     "underlying" lattice... */

  if ((x % 2) == 0)
    switch (neighborCode) {
      case TriNorth:      neighborCode = SqrNorth;     break;
      case TriNorthEast:  neighborCode = SqrNorthEast; break;
      case TriSouthEast:  neighborCode = SqrEast;      break;
      case TriSouth:      neighborCode = SqrSouth;     break;
      case TriSouthWest:  neighborCode = SqrWest;      break;
      case TriNorthWest:  neighborCode = SqrNorthWest; break;
      default:
	[InternalError raiseEvent:
	 "Invalid Tri neighbor code (%d) encountered.\n", 
	  neighborCode];
	neighborCode = SqrNorth; // Defensive...
      }
  else
    switch (neighborCode) {
      case TriNorth:      neighborCode = SqrNorth;     break;
      case TriNorthEast:  neighborCode = SqrEast;      break;
      case TriSouthEast:  neighborCode = SqrSouthEast; break;
      case TriSouth:      neighborCode = SqrSouth;     break;
      case TriSouthWest:  neighborCode = SqrSouthWest; break;
      case TriNorthWest:  neighborCode = SqrWest;      break;
      default:
	[InternalError raiseEvent:
	 "Invalid Tri neighbor code (%d) encountered.\n", 
	  neighborCode];
	neighborCode = SqrNorth; // Defensive...
      }
  
  return neighborCode;
}

-(void) translateByNeighborCode: (NeighborCode) neighborCode {

  if ([self isTriNeighbor: neighborCode])
    neighborCode = [self deformTriNeighbor:neighborCode];

  [self translateXByNeighborCode: neighborCode];
  [self translateYByNeighborCode: neighborCode];
}

-(BOOL) isEqualToCoord:   (id) coord {
  return (
    (lattice == [coord getLatticeObject]) &&
    (x == [coord getX]) && 
    (y == [coord getY]));
}

-(id)   getObject {
  [self normalize]; // Defensive...
  return [lattice getObjectAtX: x Y: y];
}

-(void) putObject: (id) anObject {
  [self normalize]; // Defensive...
  [lattice putObject: anObject atX: x Y: y];
}

