/* 
  LatticeAgent.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  FEB-1997

*/

#import "Lattice.h"
#import "LatticeAgent.h"


@implementation LatticeAgent



-(id) copyCoord: (id) aZone {
  id newCoord;

  newCoord = nil;
  if (coord != nil)
    newCoord = [coord copy: aZone];

  return (newCoord);
}

-(BOOL) attemptUnwarpToCoord: (id) newCoord {
  id currentOccupant;
  BOOL ok;

  ok = NO;

  if (coord != nil)
    [InternalError raiseEvent: 
      "Attempt to reembed latticeAgent %d;\n"
      "but already embedded.\n",
      self];
  else {
    currentOccupant = [newCoord getObject];
    if (currentOccupant != nil)
      [InternalError raiseEvent: 
	"Attempt to embed latticeAgent %d;\n"
	"but already occupied by agent %d.\n",
	self, currentOccupant];
    else {
      coord = [newCoord copy: [self getZone]];
      [coord putObject: self];
      ok = YES;
    }
  }

  return ok;
}

-warp 
{
  if (coord != nil) {
    [coord putObject: nil];
    [coord drop];
    coord = nil;
  }

  return self;
}


-moveToCoord: (id) newCoord {
  [self warp];
  if (![self attemptUnwarpToCoord: newCoord])
    [InternalError raiseEvent:
      "unwarp failed on attempted move of latticeAgent %d.\n",
      self];

  return self;
}

-moveToNeighbor: (NeighborCode) neighborCode {
  id newCoord;

  newCoord = [self copyCoord: [self getZone]];
  if (newCoord == nil)
    [InternalError raiseEvent: 
      "Attempt to moveToNeighbor: on latticeAgent %d,\n"
      "while not embedded.\n",
      self];
  else {
    [newCoord translateByNeighborCode: neighborCode];
    [self moveToCoord: newCoord];
    [newCoord drop];;
  }

  return self;
}


-(BOOL) attemptSwapSelfFromOldCoord: (id) oldCoord 
  withAgent: (id) agent
  atNewCoord: (id) newCoord {
  /* Private method; no sanity checks. */
  BOOL primaryOK, secondaryOK, ok;

  [self warp];
  [agent warp]; 

  primaryOK = [self attemptUnwarpToCoord: newCoord];
  secondaryOK = [agent attemptUnwarpToCoord: oldCoord];

  ok = (primaryOK && secondaryOK);
  if (!ok) {
    // Backout!
    [self warp];
    [agent warp];     
    if (![self attemptUnwarpToCoord: oldCoord]) 
      [InternalError raiseEvent:
        "Can't unwarp primary latticeAgent %d back to original coordinate.\n",
        self];
    if (![agent attemptUnwarpToCoord: newCoord]) 
      [InternalError raiseEvent:
        "Can't unwarp secondary latticeAgent %d back "
		     "to original coordinate).\n",
        agent];
  }

  return ok;
}


-(BOOL) attemptSwapToCoord: (id) newCoord {
  id agent;
  id oldCoord;
  BOOL ok;

  ok = NO;

  if (coord == nil)
    [InternalError raiseEvent: 
      "attemptSwapToCoord: on latticeAgent %d,\n"
      "while not embedded.\n",
      self];
  else {
    agent = [newCoord getObject];
    if (agent == nil) {
      [self moveToCoord: newCoord];
      ok = YES;
    }
    else {
      oldCoord = [self copyCoord: [self getZone]];
      ok = ([self attemptSwapSelfFromOldCoord: oldCoord
	withAgent: agent
	atNewCoord: newCoord]);
      [oldCoord drop];
    }
  }

  return ok;
}

-(BOOL) attemptSwapToNeighbor: (NeighborCode) neighborCode {
  id newCoord;
  BOOL ok;

  ok = NO;

  if (coord == nil)
    [InternalError raiseEvent: 
      "attemptSwapToNeighbor: on latticeAgent %d,\n"
      "while not embedded.\n",
      self];
  else {
    newCoord = [coord copy: [self getZone]];
    [newCoord translateByNeighborCode: neighborCode];

    ok = ([self attemptSwapToCoord: newCoord]);

    [newCoord drop];
  }

  return (ok);
}

-(BOOL) attemptSwapWithAgent: (LatticeAgent *) agent {
  id newCoord;
  BOOL ok;

  ok = NO;

  if (agent == nil)
    [InternalError raiseEvent:
     "attemptSwapWithAgent on latticeAgent %d with nil agent.\n", self];
  else {
    newCoord = [agent copyCoord: [self getZone]];
    if (newCoord == nil)
      [InternalError raiseEvent:
	"attemptSwapWithAgent on latticeAgent %d with latticeAgent%d;\n"
	"but latter not embedded.\n",
	self, agent];
    else {
      ok = [self attemptSwapToCoord: newCoord];
      [newCoord drop];
    }
  }

  return ok;
}

-(void) drop {
  if (coord != nil) [self warp];
  [super drop];
}

-saveTo: (OutFile *) file {
  [super saveTo: file];

  if (coord == nil)
    [file putInt: 0];
  else {
    [file putInt: 1];
    [coord saveTo: file];
  }

  return self;
}


-loadFrom: (InFile *) file toLattice: (id) inLattice {
  int embeddedFlag;
  id newCoord;

  [super loadFrom: file];

  [file getInt: &embeddedFlag];
  if (embeddedFlag) {
    newCoord = [inLattice createOriginCoord: [self getZone]];
    [newCoord loadFrom: file];
    
    if (![self attemptUnwarpToCoord: newCoord])
    [InternalError raiseEvent:
      "unwarp failed on attempted load of latticeAgent %d.\n",
      self];
  }

  return self;
}




@end
