/* 
  Lattice2d.h

  This provides the Lattice2d class.  This is
  subclassed from the standard Swarm Discrete2d.  It
  extends that class by providing support for the
  "Coord" mechanism of interaction (see Coord.h). 

  A Lattice2d can be (dynamically) configured with
  either a Square or (Equilateral) Triangular
  tiling. It can also be dynamically configured with a
  "default" Neighborhood object, being either Von
  Neumann or Moore (on a Square tiling) or
  Frisch-Hasslacher-Pomeau (on a triangular tiling).
  The latter is the "nearest neighbor" Neighborhood on
  a Triangular tiling, having 6 members, and is also
  referred to as the "hexagonal" neighborhood.

  Note carefully that the tiling and the neighborhood
  are *not* independent: so changing either one
  potentially forces a change in the other also.

  Lattice2d adopts the generic <Lattice> protocol (see
  Lattice.h), and also the <Neighborhood> protocol (see
  Neighborhood.h).  The latter is "sort of" an attempt
  at faking multiple inheritance in ObjC: the actual
  implementation of the <Neighborhood> methods is based
  simply on forwarding to the currently specified
  default Neighborhood object (see VNneighborhood.h,
  MooreNeighborhood.h, and FFHPNeighborhood.h).

  Note that, in general, an application *might* want to
  use different differhoods on the same Lattice2d on an
  ongoing basis (specifically, both VN and Moore on a
  Square tiling).  In this case, the app may prefer to
  maintain it's own, multiple, Neighborhood objects,
  rather than relying on the default object accessed
  via the Lattice2d (and incurring the overhead of
  frequently switching this).

  Barry McMullin <mcmullin@eeng.dcu.ie>
  FEB-1997

*/

#import <space/Discrete2d.h>
#import "Lattice.h"

typedef enum {
  SqrTiling, TriTiling
} TilingCode;

typedef enum {
  VN, Moore, FHP
} NeighborhoodCode;

@interface Lattice2d : Discrete2d <Lattice> 
{
  unsigned rasterScaleFactor;
  TilingCode tilingCode;
  NeighborhoodCode neighborhoodCode;
  id <Neighborhood, Drop> neighborhood;
  id prng;
}

+ create: aZone withSizeX: (unsigned)sizeX Y: (unsigned)sizeY;

- (TilingCode)getTilingCode;
- (void)setTilingCode: (TilingCode)newTilingCode;

/*
  Setting SqrTiling forces the VN neighborhood;
  setting TriTiling forces the FHP neighborhood.
*/


- (NeighborhoodCode)getNeighborhoodCode;
- (void)setNeighborhoodCode: (NeighborhoodCode) newNeighborhoodCode;

/*
  Setting VN or Moore Neighborhood forces SqrTiling;
  setting FHP Neighborhood forces TriTiling.
*/


- (void)setRasterScaleFactor: (unsigned)newScaleFactor;;
- (unsigned)getRasterScaleFactor;

@end

