/* 
  AgentManager.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  07-OCT-1996

*/

#import <collections.h>

#import "SCLGlobals.h"

#import "PRNG.h"

#import "Agent.h"
#import "AgentManager.h"

@implementation AgentManager

+createBegin: (id) aZone {
  AgentManager *obj;

  obj = [super createBegin: aZone];
  obj->agentList = [List create: aZone];
  obj->addedAgentQueue = [List create: aZone];
  obj->droppedAgentQueue = [List create: aZone];

  return obj;
}


-setAgentClass: (id) classId {
  if (agentClass != nil)
    [InternalError raiseEvent:
      "AgentManager %d: attempt to setAgentClass when already set to %d.\n",
      self, agentClass];
  agentClass = classId;

  return self;
}

-createEnd {
  if (agentClass == nil)
    [InternalError raiseEvent:
      "AgentManager object %d -createEnd while agentClass == nil.\n",
      self];

  return [super createEnd];
}


-(id) getAgentList {
  return agentList;
}

-(id) createAgent {
  id agent;

  agent = [agentClass create: [self getZone]];
  [addedAgentQueue addLast: agent];

  return agent;
}


-dropAgent: (id) agent {
  [droppedAgentQueue addLast: agent];

  return self;
}

-(unsigned) getCount {
  return [agentList getCount];
}


-(id) randomAgent {
  int offset;
  int count;
  id agent;

  agent = nil;
  count = [agentList getCount];
  if (count > 0) {
    offset = [prng getUniformWithRange: count];
    agent = [agentList atOffset: offset];
  }

  return agent;
}


-processQueues {
  id agent;

  while ([addedAgentQueue getCount] > 0) {
    agent = [addedAgentQueue getFirst];
    [agentList addLast: agent];
    [addedAgentQueue removeFirst];
  }

  while ([droppedAgentQueue getCount] > 0) {
    agent = [droppedAgentQueue getFirst];
    [agentList remove: agent];
    [droppedAgentQueue removeFirst];
    [agent drop];
  }

  return self;
}

-saveTo: (id <OutFile>) file {
  int i, count;

  [self processQueues];  // Defensive...

  count = [agentList getCount];
  [file putString: "# Start agentList (Class: "];
  [file putString: [agentClass name]];
  [file putString: ")\n"];
  [file putString: "# Agent count:\n"];
  [file putInt: count];
  [file putString: "\n# Agents:\n"];

  for (i = 0; i < count; i++) {
    [[agentList atOffset: i] saveTo: file];
    [file putNewLine];
  }

  return self;
}

-loadFrom: (id <InFile>) file {
  int i, count;
  id agent;

  [self processQueues];  // Defensive...

  [file skipLine]; // "# Start agentList (Class: ... )\n"
  [file skipLine]; // "# Agent count:\n"
  [file getInt: &count];
  [file skipLine]; 
  [file skipLine]; // "# Agents:\n"];

  for (i = 0; i < count; i++) {
    agent = [self createAgent];
    [agent loadFrom: file];
    [file skipLine];
  }

  [self processQueues];

  return self;
}



-(void) drop {
  id agent;

  [self processQueues]; // Defensive...

  while ([agentList getCount] > 0) {
    agent = [agentList getFirst];
    [agentList remove: agent];
    [agent drop];
  }

  [super drop];
}

@end
