// Neural Network Library. Copyright (C) 1996 Juan J. Merelo.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.


#include "BaseSingleLayer.h"

#include <math.h>

@implementation BaseSingleLayer

//--------------------------------------------
// Instance objects, taken from the protocol
- setLayerSize: (unsigned) inpLyUnits: (unsigned) opLyUnits {

    inpSize = inpLyUnits;
    numNeurons = opLyUnits;
  
    inpLy = (float*) calloc( inpSize,sizeof( float ) ); 
    Wgts = (float*) calloc( inpSize*numNeurons, sizeof( float ) );
    opLy = (float*) calloc( numNeurons, sizeof( float ) ); 

    if ( !Wgts || !opLy || !inpLy || !numNeurons ) {
	[InvalidCombination raiseEvent:  
				"Problems with  malloc or network size\n" ];
	return nil;
    }
    return self;
}

- setTrainingConstant: (float) _alpha {
    alpha = _alpha;
    return self;
}

//--------------------------------------------
-free {
    free( inpLy );
    free( Wgts );
    free( opLy );
    [self drop];
    return nil;
}

//-----------------------------------------------------
- createEnd {
    if ( !Wgts || !opLy || !inpLy || !numNeurons ) {
	[InvalidCombination raiseEvent: "Problems with  malloc or network size\n" ];
    }
    return self;
}

//------------------------------------------------------
// wgts should have the same dimensions than Wgts
- setWeightsTo : (float * ) wgts {
    unsigned i, j;
    for ( i = 0; i < numNeurons; i++ ) {
	for ( j = 0; j < inpSize; j++ ) {
	    Wgts[ i*inpSize + j ] = wgts[ i*inpSize + j ];
	}
    }
    return self;

}


//------------------------------------------------------
- setRandomWeights {
// Inits weights with random floats between -1 and 1
//------------------------------------------------------

    unsigned i;
    for ( i = 0; i < inpSize*numNeurons; i++ ) {
	Wgts[i] = [uniformRandom rFloat] * 2 -1;
    }
    return self;
}

//------------------------------------------------------
-setFromString: (unsigned char *) _str {
// 2 bytes by weight inits weights with  floats between -1 and 1
//------------------------------------------------------

    unsigned i, j, foo;
    unsigned powers[BYTESWGT] = {1,256};

    for ( i = 0; i < inpSize*numNeurons; i++ ) {
	foo = 0;
	for ( j = 0; j < BYTESWGT; j ++ ) { // Little-endian order + sign
	    foo += powers[j]*_str[i*BYTESWGT+j];
	}
	Wgts[i] =  (1 - foo/powers[BYTESWGT]*256.0)*2 -1;
//	printf( "Weight %d %f\n", i, Wgts[i] );
    } 
    
    return self;
}

//------------------------------------------------------------
- (float)   getWeights: (unsigned) i: (unsigned) j {
// Returns a weight in a layer, implements a protocol function
// Only one layer this time, so Layer acts as dummy
// -----------------------------------------------------------

    if ( ( i < numNeurons ) && ( j < inpSize) )
	return Wgts[ i*inpSize + j];
    else
	return 0;
}


//-----------------------------------------------------
//            Learning and recalling functions
//-----------------------------------------------------

//-----------------------------------------------------
-(unsigned ) feedForward: (float *) inpVec {
// To implement in subClasses
    return numNeurons + 1;
//    [SubclassMustImplement raiseEvent: "Error in BaseSingleLayer feedforward method"];
}


//------------------------------------------------------
- train : (float) delta {
// To implement in subclasses
    return nil;
//    [SubclassMustImplement raiseEvent: "Error in BaseSingleLayer train method"];
}


//----------- Info functions --------------------------

-(unsigned) getInpSize {
    return inpSize;
}

-(BOOL) getLayers {
    return 0;
}

@end
