/* 
  MooreNeighbor.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  JAN-1997

*/

#import "MooreNeighborhood.h"
#import <random.h>


@implementation MooreNeighborhood

+ create: aZone setGenerator: generator
{
  MooreNeighborhood *neighborhood;
  
  neighborhood = [super create: aZone];
  neighborhood->uniformUnsigned =
    [UniformUnsignedDist create: aZone setGenerator: generator];
  
  return neighborhood;
}

- (void)drop
{
  if (uniformUnsigned != nil)
    [uniformUnsigned drop];
  [super drop];
}

- (NeighborCode)getInitialNeighborCode
{
  return SqrNorth;
}

- (BOOL)isValidNeighborCode: (NeighborCode)inNeighborCode
{
  BOOL isValid;
  
  switch (inNeighborCode)
    {
    case SqrNorth: 
    case SqrNorthEast:
    case SqrEast:  
    case SqrSouthEast:
    case SqrSouth: 
    case SqrSouthWest:
    case SqrWest:  
    case SqrNorthWest:
      isValid = YES;
      break;
    default:
      isValid = NO;
    }
  
  return isValid;
}


- (NeighborCode)getNextNeighborCode: (NeighborCode)inNeighborCode
{
  NeighborCode nextNeighborCode;
  
  switch (inNeighborCode)
    {
    case SqrNorth:     nextNeighborCode = SqrNorthEast;    break;
    case SqrNorthEast: nextNeighborCode = SqrEast;         break;
    case SqrEast:      nextNeighborCode = SqrSouthEast;    break;
    case SqrSouthEast: nextNeighborCode = SqrSouth;        break;
    case SqrSouth:     nextNeighborCode = SqrSouthWest;    break;
    case SqrSouthWest: nextNeighborCode = SqrWest;            break;
    case SqrWest:      nextNeighborCode = SqrNorthWest;    break;
    case SqrNorthWest: nextNeighborCode = SqrNorth;        break;
    default:
      [InternalError raiseEvent:
                       "Invalid Moore neighbor code (%d) encountered.\n", 
                     inNeighborCode];
      nextNeighborCode = SqrNorth; // Defensive...
    }
  
  return nextNeighborCode;
}

- (NeighborCode)getPreviousNeighborCode: (NeighborCode)inNeighborCode
{
  NeighborCode previousNeighborCode;
  
  switch (inNeighborCode)
    {
    case SqrNorth:     previousNeighborCode = SqrNorthWest;    break;
    case SqrNorthWest: previousNeighborCode = SqrWest;         break;
    case SqrWest:      previousNeighborCode = SqrSouthWest;    break;
    case SqrSouthWest: previousNeighborCode = SqrSouth;        break;
    case SqrSouth:     previousNeighborCode = SqrSouthEast;    break;
    case SqrSouthEast: previousNeighborCode = SqrEast;         break;
    case SqrEast:      previousNeighborCode = SqrNorthEast;    break;
    case SqrNorthEast: previousNeighborCode = SqrNorth;        break;
    default:
      [InternalError raiseEvent:
                       "Invalid Moore neighbor code (%d) encountered.\n", 
                     inNeighborCode];
      previousNeighborCode = SqrNorth; // Defensive...
    }
  
  return previousNeighborCode;
}

- (NeighborCode)getOppositeNeighborCode: (NeighborCode)inNeighborCode
{
  NeighborCode oppositeNeighborCode;
  
  switch (inNeighborCode)
    {
    case SqrNorth:     oppositeNeighborCode = SqrSouth;        break;
    case SqrNorthEast: oppositeNeighborCode = SqrSouthWest;    break;
    case SqrWest:      oppositeNeighborCode = SqrEast;         break;
    case SqrNorthWest: oppositeNeighborCode = SqrSouthEast;    break;
    case SqrSouth:     oppositeNeighborCode = SqrNorth;        break;
    case SqrSouthEast: oppositeNeighborCode = SqrNorthWest;    break;
    case SqrEast:      oppositeNeighborCode = SqrWest;         break;
    case SqrSouthWest: oppositeNeighborCode = SqrNorthEast;    break;
    default:
      [InternalError raiseEvent:
                       "Invalid Moore neighbor code (%d) encountered.\n", 
                     inNeighborCode];
      oppositeNeighborCode = SqrNorth; // Defensive...
    }
  
  return oppositeNeighborCode;
  
}

- (NeighborCode)getRandomNeighborCode
{
  unsigned randomNumber;
  NeighborCode randomNeighborCode;
  
  randomNeighborCode = SqrNorth; // Defensive...
  
  if (uniformUnsigned == nil)
    [InternalError 
      raiseEvent:
        "Attempt to -getRandomNeighborCode, but uniformUnsigned is nil.\n"];
  else
    {
      randomNumber = [uniformUnsigned getUnsignedWithMin: 0 withMax: 7];
      switch (randomNumber)
        {
        case 0: randomNeighborCode = SqrNorth;        break;
        case 1: randomNeighborCode = SqrNorthEast;    break;
        case 2: randomNeighborCode = SqrEast;         break;
        case 3: randomNeighborCode = SqrSouthEast;    break;
        case 4: randomNeighborCode = SqrSouth;        break;
        case 5: randomNeighborCode = SqrSouthWest;    break;
        case 6: randomNeighborCode = SqrWest;         break;
        case 7: randomNeighborCode = SqrNorthWest;    break;
        default:
          [InternalError
            raiseEvent:
              "Out-of-range random number (%d) encountered.\n", randomNumber];
        }
    }
  
  return randomNeighborCode;
}

- (NeighborCode)getRandomNeighborCodeExcluding: (NeighborCode)inNeighborCode
{
  // This is a cheap `n cheerful, totally unoptimised version...
  unsigned randomNumber;
  NeighborCode randomNeighborCode;
  
  randomNeighborCode = [self getNextNeighborCode: inNeighborCode];
  
  if (uniformUnsigned == nil)
    [InternalError raiseEvent:
                     "Attempt to -getRandomNeighborCodeExcluding,\n"
                   "but uniformUnsigned is nil.\n"];
  else
    {
      randomNumber = [uniformUnsigned getUnsignedWithMin: 0 withMax: 6];
      while (randomNumber > 0) {
        randomNeighborCode = [self getNextNeighborCode: inNeighborCode];
        randomNumber--;
      }
    }
  
  return randomNeighborCode;
}


@end
