/* 
  VNneighborhood.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  JAN-1997

*/

#import "VNneighborhood.h"
#import <random.h>

@implementation VNneighborhood

+ create: aZone setGenerator: generator
{
  VNneighborhood *neighborhood;
  
  neighborhood = [super create: aZone];
  neighborhood->uniformUnsigned =
    [UniformUnsignedDist create: aZone setGenerator: generator];

  return neighborhood;
}

- (void) drop
{
  if (uniformUnsigned != nil)
    [uniformUnsigned drop];
  [super drop];
} 

- (NeighborCode)getInitialNeighborCode
{
  return SqrNorth;
}

- (BOOL)isValidNeighborCode: (NeighborCode)inNeighborCode
{
  BOOL isValid;
  
  switch (inNeighborCode) 
    {
    case SqrNorth: 
    case SqrEast:  
    case SqrSouth: 
    case SqrWest:  
      isValid = YES;
      break;
    default:
      isValid = NO;
    }
  
  return(isValid);
}

/* 
   The -next, -previous and -randomise methods *could* be made
   marginally more efficient if they took advantage of the
   underlying ordering of the SqrNeighborCode enumerated type;
   but that would then mean that any change in that underlying
   ordering would (silently) break the VNneighborhood class.
   I prefer to accept the small performace penalty in making
   this class robust against changes in the underlying ordering
   of SqrNeighborCode.
*/


- (NeighborCode)getNextNeighborCode: (NeighborCode)inNeighborCode
{
  NeighborCode nextNeighborCode;
  
  switch (inNeighborCode)
    {
    case SqrNorth: nextNeighborCode = SqrEast;    break;
    case SqrEast:  nextNeighborCode = SqrSouth;   break;
    case SqrSouth: nextNeighborCode = SqrWest;    break;
    case SqrWest:  nextNeighborCode = SqrNorth;   break;
    default:
      [InternalError
        raiseEvent:
          "Invalid von Neumann neighbor code (%d) encountered.\n", 
        inNeighborCode];
      nextNeighborCode = SqrNorth; // Defensive...
    }
  
  return (nextNeighborCode);
}

- (NeighborCode)getPreviousNeighborCode: (NeighborCode)inNeighborCode
{
  NeighborCode previousNeighborCode;
  
  switch (inNeighborCode)
    {
    case SqrNorth: previousNeighborCode = SqrWest;    break;
    case SqrWest:  previousNeighborCode = SqrSouth;   break;
    case SqrSouth: previousNeighborCode = SqrEast;    break;
    case SqrEast:  previousNeighborCode = SqrNorth;   break;
    default:
      [InternalError
        raiseEvent:
          "Invalid von Neumann neighbor code (%d) encountered.\n", 
        inNeighborCode];
      previousNeighborCode = SqrNorth; // Defensive...
    }
  
  return previousNeighborCode;
}

- (NeighborCode)getOppositeNeighborCode: (NeighborCode)inNeighborCode
{
  NeighborCode oppositeNeighborCode;
  
  switch (inNeighborCode)
    {
    case SqrNorth: oppositeNeighborCode = SqrSouth;    break;
    case SqrWest:  oppositeNeighborCode = SqrEast;     break;
    case SqrSouth: oppositeNeighborCode = SqrNorth;    break;
    case SqrEast:  oppositeNeighborCode = SqrWest;   break;
    default:
      [InternalError
        raiseEvent:
          "Invalid von Neumann neighbor code (%d) encountered.\n", 
        inNeighborCode];
      oppositeNeighborCode = SqrNorth; // Defensive...
    }
  
  return oppositeNeighborCode;
  
}

- (NeighborCode)getRandomNeighborCode
{
  unsigned randomNumber;
  NeighborCode randomNeighborCode;
  
  randomNeighborCode = SqrNorth; // Defensive...
  
  if (uniformUnsigned == nil)
    [InternalError
      raiseEvent:
        "Attempt to -getRandomNeighborCode, but uniformUnsigned is nil.\n"];
  else
    {
      randomNumber = [uniformUnsigned getUnsignedWithMin: 0 withMax: 3];
      switch (randomNumber)
        {
        case 0: randomNeighborCode = SqrNorth;    break;
        case 1: randomNeighborCode = SqrEast;     break;
        case 2: randomNeighborCode = SqrSouth;    break;
        case 3: randomNeighborCode = SqrWest;     break;
        default:
          [InternalError
            raiseEvent:
              "Out-of-range random number (%d) encountered.\n", randomNumber];
        }
    }
  
  return randomNeighborCode;
}

- (NeighborCode)getRandomNeighborCodeExcluding: (NeighborCode)inNeighborCode
{
  // This is a cheap `n cheerful, totally unoptimised version...
  
  unsigned randomNumber;
  NeighborCode randomNeighborCode;
  
  randomNeighborCode = [self getNextNeighborCode: inNeighborCode];
  
  if (uniformUnsigned == nil)
    [InternalError raiseEvent:
                     "Attempt to -getRandomNeighborCodeExcluding,\n"
                   "but uniformUnsigned is nil.\n"];
  else
    {
      randomNumber = [uniformUnsigned getUnsignedWithMin: 0 withMax: 2];
      while (randomNumber > 0)
        {
          randomNeighborCode = [self getNextNeighborCode: inNeighborCode];
          randomNumber--;
        }
    }
  
  return randomNeighborCode;
}


@end
