// Swarm library. Copyright (C) 1996-1998 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

/* 
  Agent2d.h

  Barry McMullin <mcmullin@eeng.dcu.ie>
  19-SEP-1996

*/

#import "Agent2d.h"


@implementation Agent2d

+createBegin: (id) aZone {
  Agent2d *agent;

  agent = [super createBegin: aZone];

  agent->world = nil;  // Defensive...
  agent->embedded = NO; // ditto
  agent->x = -1;       // ditto
  agent->y = -1;       // ditto

  return (agent);
}

-setWorld: (DiscreteToroid *) inWorld {
  if (world != nil) {
    [InternalError raiseEvent: 
    "You can only set the world of an agent2d once -\n"
    "at creation time\n"];
  }
  world = inWorld;

  return self;
}

-createEnd{
  if (world == nil) {
    [InternalError raiseEvent: 
    "agent2d created without a world.\n"];
  }

  return self;
}


-(Discrete2d *) getWorld {
  return world;
}

-(BOOL) getEmbedded {
  return embedded;
}

-unwarpToX: (int) newX Y: (int) newY {
  id currentOccupant;

  newX = [world wrapXCoord: newX];
  newY = [world wrapYCoord: newY];

  if (embedded)
    [InternalError raiseEvent: 
      "Attempt to reembed agent2d %d at (%d, %d);\n"
      "but already embedded at (%d, %d).\n",
      self, newX, newY, x, y];
  else {
    currentOccupant = [world getObjectAtX: newX Y: newY];
    if (currentOccupant != nil)
      [InternalError raiseEvent: 
        "Attempt to embed agent2d %d at (%d, %d);\n"
        "but already occupied by agent %d.\n",
        self, newX, newY, currentOccupant];
    else {
      x = newX;
      y = newY;
      [world putObject: self atX: x Y: y];
      embedded = YES;
    }
  }

  return self;
}

-warp 
{
  if (!embedded) 
    [InternalError raiseEvent: 
      "Attempt to warp already unembedded agent2d %d;\n",
      self];
  else {
    [world putObject: nil atX: x Y: y];
    x = y = -1; // Defensive...
    embedded = NO;
  }

  return self;
}


-(int) getX {
  if (!embedded)
    [InternalError raiseEvent: 
      "Attempt to -getX on non-embedded agent2d %d", self];

  return x;
}

-(int) getY {
  if (!embedded)
    [InternalError raiseEvent: 
      "Attempt to -getY on non-embedded agent2d %d", self];

  return y;
}


-(int) getNeighborX: (neighbor_t) neighbor {
  int neighborX;

  neighborX = x;

  if (!embedded)
    [InternalError raiseEvent: 
      "Attempt to -getNeighborX on non-embedded agent2d %d", self];
  else {
    switch (neighbor) {
      case East:     
      case NorthEast:
      case SouthEast:     (neighborX)++;      break;

      case West:
      case SouthWest:
      case NorthWest:     (neighborX)--;      break;

      case FarEast:       (neighborX) += 2;   break;

      case FarWest:       (neighborX) -= 2;   break;

      case Center:
      case North:
      case South:
      case FarNorth:
      case FarSouth:                          break;

    default:
      [InternalError raiseEvent:
      "Invalid neighbor code (%d) encountered.", neighbor];
    }
  }  
  return ([world wrapXCoord: neighborX]);
}


-(int) getNeighborY: (neighbor_t) neighbor {
  int neighborY;

  neighborY = y;

  if (!embedded)
    [InternalError raiseEvent: 
      "Attempt to -getNeighborY on non-embedded agent2d %d", self];
  else {
    switch (neighbor) {
      case North:     
      case NorthEast:
      case NorthWest:     (neighborY)--;      break;

      case South:
      case SouthEast:
      case SouthWest:     (neighborY)++;      break;

      case FarNorth:      (neighborY) -= 2;   break;

      case FarSouth:      (neighborY) += 2;   break;

      case Center:
      case East:
      case West:
      case FarEast:
      case FarWest:                           break;

    default:
      [InternalError raiseEvent:
      "Invalid neighbor code (%d) encountered.", neighbor];
    }
  }

  return ([world wrapYCoord: neighborY]);
}


-moveToX: (int) inX Y: (int) inY {
  [self warp];
  [self unwarpToX: inX Y: inY];

  return self;
}


-moveToNeighbor: (neighbor_t) neighbor {
  int neighborX, neighborY;

  neighborX = [self getNeighborX: neighbor];
  neighborY = [self getNeighborY: neighbor];

  [self moveToX: neighborX Y: neighborY];

  return self;
}


-swapWithAgentAtX: (int) inX Y: (int) inY {
  int oldX, oldY;
  id currentOccupant;

  oldX = x;
  oldY = y;
  [self warp];

  currentOccupant = [world getObjectAtX: inX Y: inY];
  if (currentOccupant != nil) {
    [currentOccupant warp]; 
    [currentOccupant unwarpToX: oldX Y: oldY];
  }

  [self unwarpToX: inX Y: inY];

  return self;
}

-swapWithAgentAtNeighbor: (neighbor_t) neighbor {
  int neighborX, neighborY;

  neighborX = [self getNeighborX: neighbor];
  neighborY = [self getNeighborY: neighbor];
  [self swapWithAgentAtX: neighborX Y: neighborY];

  return self;
}

-swapWithAgent: (Agent2d *) agent {
  Discrete2d *newWorld;
  int newX, newY;
  int oldX, oldY;

  if (agent == nil)
    [InternalError raiseEvent:
     "Attempt to swap agent2d %d with nil agent.\n", self];
  else {
    newWorld = [agent getWorld];
    if (newWorld != world)
      [InternalError raiseEvent:
        "Attempt to swap agent2d %d in world %d\n"
        "with agent2d %d in world %d.\n",
        self, world, agent, newWorld];
    else {
      oldX = x;
      oldY = y;
      [self warp];

      newX = [agent getX];
      newY = [agent getY];
      [agent warp];
      [agent unwarpToX: oldX Y: oldY];

      [self unwarpToX: newX Y: newY];
    }
  }

  return self;
}

-(Agent2d *) getAgentAtNeighbor: (neighbor_t) neighbor {
  int neighborX, neighborY;

  neighborX = [self getNeighborX: neighbor];
  neighborY = [self getNeighborY: neighbor];

  return ([world getObjectAtX: neighborX Y: neighborY]);
}

-(void) drop {
  if (embedded) [self warp];
  [super drop];
}

@end
