// Swarm library. Copyright (C) 1996 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

/*
Name:		 random.m
Description:	 startup functions for the random library
Library:	 random
Original Author: Glen E. Ropella
Date:		 1997-09-01 (v. 0.7)
*/

#import <sys/time.h>

#import <defobj.h>
#import <random.h>

id <MT19937gen>          randomGenerator;
id <UniformIntegerDist>   uniformIntRand;
id <UniformUnsignedDist>  uniformUnsRand;
id <UniformDoubleDist>    uniformDblRand;

unsigned int              randomSeed;
BOOL                      useFixedSeed;

unsigned int              _timeThen, _timeNow;

void
initRandom(int argc, char ** argv) {
  int i;
  struct timeval then;
 
// Default mode is that starting seeds are fixed (as in 0.6):

  useFixedSeed = YES;
  randomSeed = DEFAULTSEED;

// But if the user says '-varySeed', starting seeds are randomized:
  for(i = 1 ; i < argc ; i++) {
    if ( !strcmp(argv[i],"-varySeed") ) {
      useFixedSeed = NO;
      // use FIRSTSEED because "clock()" is cpu time, which could be zero
      randomSeed = FIRSTSEED;
    }
  }

// Save the time-of-day in microseconds for use later in computing RANDOMSEED:

   gettimeofday(&then, NULL);
   _timeThen = then.tv_usec + 1000000 * ( then.tv_sec % 2048 );


// Create the utility objects here:

   randomGenerator = [ MT19937gen create: globalZone 
			setStateFromSeed: randomSeed ];

   uniformIntRand  = [ UniformIntegerDist create: globalZone
			setGenerator: randomGenerator ];
   uniformUnsRand  = [ UniformUnsignedDist create: globalZone
			setGenerator: randomGenerator ];
   uniformDblRand  = [ UniformDoubleDist create: globalZone
			setGenerator: randomGenerator ];
// Comments: 
//  1. the generator is fast and has a *long* period. It is initialized
//     with either a fixed or a randomized seed, depending on the use
//     of the 'varySeed' command line flag.
//  2. the distribution objects are all connected to this generator,
//     getting their random numbers from it in an interleaved fashion.

}

unsigned int 
tempusFugit(void) {
  struct timeval now;

  clock();  // initialize clock()

  gettimeofday(&now, NULL);
  _timeNow = now.tv_usec + 1000000 * ( now.tv_sec % 2048 );

  if ( _timeNow > _timeThen ) 
     return ( _timeNow - _timeThen );
  else if ( _timeNow < _timeThen )
     return ( _timeThen - _timeNow );
  else // do *not* return 0:
     return DEFAULTSEED;
}

