#!/bin/bash
#
#   cygbuild.sh --- A generic Cygwin Net release package builder script
#
#       Time-stamp: <2004-01-27 16:33:12>
#       Author: <jari.aalto@poboxes.com>
#
#   Copyright
#
#       This program is free software; you can redistribute it and/or
#       modify it under the terms of the GNU General Public License as
#       published by the Free Software Foundation; either version 2 of
#       the License, or (at your option) any later version.
#
#       This program is distributed in the hope that it will be useful, but
#       WITHOUT ANY WARRANTY; without even the implied warranty of
#       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#       General Public License for more details.
#
#       You should have received a copy of the GNU General Public License along
#       with this program; if not, write to the Free Software Foundation,
#       Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#   Code notes
#
#       Underscore_variables are used for globals (exported). Local variables
#       used in functions areLikeThis.
#
#       See option --help and read the instructions
#
#       This script's startup time is slow due to multiple calls to
#       variable settings. This isssue will hopefully addressed in later
#       releases, so don't be suprised to see multiple calls to function
#       CygbuildVersionInfo() with call to "bash -x"
#
# TODO: Daniel Reed <n@ml.org> writes 2003-10-14 in cygwin-apps-L
# http://sources.redhat.com/ml/cygwin-apps/2002-07/msg00144.html
#
#       prefix=/usr/X11R6
#       includedir=${prefix}/include    # /usr/X11R6/include
#       sysconfdir=/etc
#       localstatedir=/var
#
#       exec_prefix=${prefix}
#       bindir=${exec_prefix}/bin       # /usr/X11R6/bin
#       libdir=${exec_prefix}/lib       # /usr/X11R6/lib
#
#       datadir=/usr/share
#       pkgdatadir=${datadir}/${PACKAGE_TARNAME}-${PACKAGE_VERSION}
#                                       # /usr/share/tcm-2.20
#       docdir=${datadir}/doc           # /usr/share/doc
#       pkgdocdir=${docdir}/${PACKAGE_TARNAME}-${PACKAGE_VERSION}
#                                       # /usr/share/doc/tcm-2.20
#       cygdocdir=${docdir}/Cygwin      # /usr/share/doc/Cygwin
#       mandir=${datadir}/man           # /usr/share/man
#       man1dir=${mandir}/man1          # /usr/share/man/man1
#
#
# If you use:
#       --prefix=/usr/X11R6 \
#       --sysconfdir=/etc \
#       --libexecdir=/usr/sbin \
#       --localstatedir=/var \
#       --datadir=/usr/share \
#       --mandir='${datadir}/man' \
#       --infodir='${datadir}/info'
# or:
#       --prefix=/usr \
#       --exec-prefix=/usr/X11R6 \
#       --includedir=/usr/X11R6/include \
#       --sysconfdir=/etc \
#       --libexecdir=/usr/sbin \
#       --localstatedir=/var \
#       --mandir='${datadir}/man' \
#       --infodir='${datadir}/info'
#
# everything should end up in the proper place. The last few lines in both
# are because mandir and infodir default to ${prefix}/man and
# ${prefix}/info, libexec defaults to ${exec_prefix}/libexec, sysconfdir
# defaults to ${prefix}/etc, and localstatedir defaults to ${prefix}/var.
#
# not sure if this is what existing X packages use, but this seems to be
# the latest reference to paths from the archive.



CYGBUILD_HOMEPAGE_URL="http://cygbuild.sourceforge.net/"
CYGBUILD_ID="$Id: cygbuild.sh,v 1.234 2004/01/27 08:17:54 jaalto Exp $"

#   Extract 'cygbuild.sh' from version string

CYGBUILD_NAME=${CYGBUILD_ID#*:[ ]}
CYGBUILD_NAME=${CYGBUILD_NAME%%,v*}

CYGBUILD_VERSION=${CYGBUILD_ID##*,v[ ]}
CYGBUILD_VERSION=${CYGBUILD_VERSION%%[ ]*}


#   This could be more easily done in sed or awk, but Let's
#   make bash to convert YYYY/MM/DD into ISO8601 format YYYY-MM-DD

CYGBUILD_DATE=${CYGBUILD_ID#*.[0-9][0-9][0-9][ ]}   # Delete leading
CYGBUILD_DATE=${CYGBUILD_DATE%%[ ]*}                # And trailing
CYGBUILD_DATE=${CYGBUILD_DATE//\//-}       # And substitute / with -



CYGBUILD_PROGRAM="$CYGBUILD_NAME $CYGBUILD_DATE $CYGBUILD_VERSION"


#   Do not add leading slash. We export, because variables
#   are used in subshells

export CYGBUILD_DOCDIR_PREFIX_RELATIVE=usr/share

export CYGBUILD_DOCDIR_RELATIVE=share/doc
export CYGBUILD_DOCDIR_FULL=usr/share/doc

export CYGBUILD_MANDIR_RELATIVE=share/man
export CYGBUILD_MANDIR_FULL=usr/share/man



#   Global options included whan making packages (source, binary, devel))
#   SVN = See http://subversion.tigris.org/ (a CVS replacement)

# --exclude='Makefile*' \
# --exclude='MAKEFILE*' \

CYGBUILD_TAR_OPT="\
 --exclude=CVS \
 --exclude=RCS \
 --exclude=SVN \
 --exclude=SCCS \
"


#   What files to ignore while running CygbuildInstallPackageDocs
#   Manual files are already andled by "make install". If not,
#   then you better write custom install script of hack the original
#   Makefile

CYGBUILD_TAR_INSTALL_EXCLUDE="\
 --exclude=CVS \
 --exclude=RCS \
 --exclude=SVN \
 --exclude=SCCS \
 --exclude=*.xml \
 --exclude=*.xsl \
 --exclude=*.sgml \
 --exclude=*.man \
 --exclude=*.[0-9] \
 --exclude=man \
"


CYGBUILD_TEMPLATE_DIR="/etc/cygbuild/template"

CYGBUILD_CVSDIFF_OPTIONS="\
 -u \
 --recursive \
 --new-file \
 --ignore-all-space \
 --ignore-blank-lines \
 --exclude=.build \
 --exclude=.inst \
 --exclude=.sinst \
 --exclude=tmp \
 --exclude=*.lo \
 --exclude=*.o \
 --exclude=*.a \
 --exclude=*.sa \
 --exclude=*.la \
 --exclude=*.dll \
 --exclude=*.dll.a \
 --exclude=*.tmp \
 --exclude=*.bak \
 --exclude=*.orig \
 --exclude=*.rej \
 --exclude=config.guess \
 --exclude=config.cache \
 --exclude=config.log \
 --exclude=config.status \
 --exclude=config.sub \
 --exclude=config.h \
 --exclude=.cvsignore \
 --exclude=*[~#] \
 --exclude=CVS \
 --exclude=RCS \
 --exclude=SCCS \
 --exclude=SVN \
"

CYGBUILD_DIFF_OPTIONS="\
 -u \
 --recursive \
 --new-file \
 --exclude=.build \
 --exclude=.inst \
 --exclude=.sinst \
 --exclude=tmp \
 --exclude=*[~#] \
 --exclude=*.o \
 --exclude=*.lo \
 --exclude=*.a \
 --exclude=*.la \
 --exclude=*.dll \
 --exclude=*.exe \
 --exclude=*.bak \
 --exclude=*.tmp \
 --exclude=*.log \
 --exclude=.deps \
 --exclude=*.Plo \
 --exclude=*.Tpo \
 --exclude=*.swp \
 --exclude=config.* \
 --exclude=.#* \
 --exclude=*.tar.gz \
 --exclude=*.tar.bz2 \
 --exclude=*.tgz \
 --exclude=*.zip \
 --exclude=*.rar \
 --exclude=*.[zZ] \
 --exclude=CVS \
 --exclude=RCS \
 --exclude=SCCS \
 --exclude=SVN \
"

CYGBUILD_PATCH_OPT="\
 --forward \
 --strip=0 \
 --fuzz=3 \
"



#   Not included:
#    --host $host
#    --target $target
#    --srcdir $srcdir
#    --includedir $prefix/include

prefix=/usr

CYGBUILD_CONFIGURE_OPTIONS="\
 --prefix=/usr \
 --sysconfdir=/etc \
 --libexecdir=/usr/sbin \
 --exec-prefix=/usr \
 --localstatedir=/var \
 --datadir=$prefix/share \
 --mandir=$prefix/share/man \
 --infodir=$prefix/share/info \
 --libdir=$prefix/lib \
 --includedir=$prefix/include \
"



#######################################################################
#
#       RUN TIME STATIC GLOBALS: Variables whose values are "cached"
#
#######################################################################

#  Function return values are stored to file, because bash cannot be
#  called in the current shell environment. The only way to
#  call bash function would be:
#
#       local val=$(func "param")
#
#  But as since that is subshell call, any variables defined globally
#  in "func" would vanish after funrtion returns.
#
#  To call function, AND define global variables it must be done like
#  this. The return value is stored to file and the results are then read.
#  the return value file must be made unique to each function.
#
#       local retval=$RETVAL.$FUNCNAME
#       func "param" > $retval
#       local val=$retval

declare RETVAL=${TEMP:-$TMP:-/tmp}/CYGBUILD_NAME.func.$$

declare -a STATIC_ABSOLUTE_SCRIPT_PATH      # (bin path)

declare -a STATIC_VER_ARRAY                 # (pkg ver release)
declare STATIC_VER_PACKAGE=""
declare STATIC_VER_VERSION=""
declare STATIC_VER_RELEASE=""
declare STATIC_VER_STRING=""


#######################################################################
#
#       EXPORTS: Functions available for external scripts
#
#######################################################################

function CygbuildLibInstallEnvironment()
{
    #   This function can be used in install.sh, so that it can set up
    #   all the environment variables at startup
    #
    #   Include it as a first call:
    #
    #       CygbuildLibEnvironmentInstall $* && InstallUsingMake && ..

    instdir=${1:-""}     # ROOT DIR passed to script
    instdir=${instdir%/}  # Delete trailing slash

    export instdir
    export exec_instdir=$instdir

    export bindir=$instdir/usr/bin
    export includedir=$instdir/usr/include
    export libdir=$instdir/usr/lib
    export infodir=$instdir/usr/info
    export mandir=$instdir/usr/man
    export docdir=$instdir/$CYGBUILD_DOCDIR_FULL

    export infobin="/usr/bin/install-info"

    export INSTALL=${CYGWIN_BUILD_INSTALL:-"/usr/bin/install"}
    export INSTALL_DATA=${CYGWIN_BUILD_F_MODES:-"-m 644"}
    export INSTALL_BIN=${CYGWIN_BUILD_X_MODES:-"-m 755"}
}




#######################################################################
#
#       Utility functions
#
#######################################################################


function CygbuildVersionInfo()
{

    local str="$1"

    #   Debian uses scheme: package_VERSION-REL.orig.tar.gz
    #
    #   1. Delete path and see if the lats name matches:
    #
    #       /this//dir/package-VER-REL
    #
    #   2. If that does not work, then perhaps the package is being
    #   verified:
    #
    #       /usr/src/build/neon/foo-NN.NN/.build/tmp/verify
    #

    echo "$str" | perl -e '
$_ = <>;
$temp = $_;
$temp =~ s,.+/,,;

if ( @a = ($temp =~ /^(.*?)[-_]([\d.]+\d)-?(\d+)*/) )
{
    print qq(@a);
}
elsif ( @a = ( m,/([^/]+)-([\d.]+\d)-?(\d+)*,) )
{
    print qq(@a);
}

'

}

function CygbuildDefineVersionVariables()
{
    local str="$1"
    local -a arr

    if [ "$STATIC_VER_STRING" = "$str" ]; then
        arr=( ${STATIC_VER_ARRAY[*]} )
    else
        arr=( $(CygbuildVersionInfo "$str") )
        STATIC_VER_ARRAY=( ${arr[*]} )
    fi

    local count=${#arr[*]}

    if [ $count -gt 1 ]; then
        STATIC_VER_PACKAGE=${arr[0]}
        STATIC_VER_VERSION=${arr[1]}
        STATIC_VER_RELEASE=${arr[2]}
        STATIC_VER_STRING="$str"
        xxx=$str
    fi

    #  Return status: Do we have the VERSION?

    [[ $STATIC_VER_VERSION == [0-9]*.[0-9]* ]]

}


function CygbuildStrRemoveExt()
{
    # Remove compression extensions
    # foo-1.13-src.tar.gz => foo-1.13

    local str="$1"

    str=${str##*/}          # Remove path
    str=${str%.tar.bz2}
    str=${str%.tar.gz}
    str=${str%.tgz}
    str=${str%-src}
    str=${str%.orig}

    echo $str
}




function CygbuildStrPackage()
{
    # Like reading PACKAGE-1.13-1-src.tar.gz
    # foo-1.13-1-src.tar.gz => foo

    local str="$1"

    if CygbuildDefineVersionVariables $str ; then
        echo $STATIC_VER_PACKAGE
    fi
}

function CygbuildStrVersionRelease()
{
    # Like reading foo-VERSION-RELEASE-src.tar.gz
    # foo-1.13-1-src.tar.gz => 1.13-1

    local str="$1"

    if CygbuildDefineVersionVariables $str ; then
        if [ ! -z "$STATIC_VER_RELEASE" ]; then
            echo $STATIC_VER_VERSION-$STATIC_VER_RELEASE
        fi
    fi

}

function CygbuildStrRelease()
{
    # Like reading foo-1.13-RELEASE-src.tar.gz
    # foo-1.13-1-src.tar.gz => 1

    local str="$1"

    if CygbuildDefineVersionVariables $str ; then
        echo $STATIC_VER_RELEASE
    fi

}


function CygbuildStrVersion()
{
    # Like reading foo-VERSION-1-src.tar.gz
    # foo-1.13-1-src.tar.gz => 1.13

    local str="$1"

    if CygbuildDefineVersionVariables $str ; then
        echo $STATIC_VER_VERSION
    fi

}

function CygbuildCygbuildIsVersionMatch()
{
    # If it looks anything like a version number
    [[ $1 == [0-9][0-9.]* ]]
}


function CygbuildIsVersion()
{
    local str="$1"
    local ver=$(CygbuildStrVersion $str)

    [ ! -z "$ver" ]
}


function CygbuildFileExists()
{
    local file="$1"

    [ ! -z "$file" ] && [ -f $file ]
}


function CygbuildDirExists()
{
    local file="$1"

    [ ! -z "$file" ] && [ -d $file ]
}



function CygbuildDate()
{
    date "+%Y%m%d%H%M"
}


function CygbuildAbsoluteScriptPathSearch()
{
    local bin="$1"

    if [ -z "$bin" ]; then
        return 1
    fi

    if [[ $bin != */* ]]; then
        bin=$(/bin/which $bin)
    fi


    #   Try to resolve all symbolic links

    local try

    if [ -x /bin/chase ]; then

        try=$(/bin/chase $bin)

    elif [ -x /bin/namei ]; then

        #   Output:
        #   l file -> /path/file

        local -a arr=( $(/bin/namei $bin | grep -i ' l .* -> ') )
        local count=${arr[*]}


        if [ $count -eq 3 ]; then
            try=${arr[2]}
        fi

    elif [ -x /bin/readlink ]; then

        #  readlink is unreliable, because it doesn't return path, if the
        #  path is not a symlink. It returns empty.

        try=$(/bin/readlink $bin)

    fi


    if [ ! -z "$try" ]; then
        bin="$try"
    fi

    echo $bin
}


function CygbuildAbsoluteScriptPath()
{
    local bin="$1"
    local -a cache=( ${STATIC_ABSOLUTE_SCRIPT_PATH[*]} )
    local ret

    if [ -z "$bin" ]; then
        return 1
    fi

    if [[ "${cache[0]}" == $bin* ]]; then
        ret=${cache[1]}
    else
        ret=$(CygbuildAbsoluteScriptPathSearch $bin)

        if [ ! -z "$ret" ]; then
            STATIC_ABSOLUTE_SCRIPT_PATH=($bin $ret)
        fi
    fi

    echo $ret
}

function CygbuildAbsolutePath()
{
    local p="$1"

    if [ -d $p ]; then
        p=$(cd $p; pwd)
    elif [[ $p == /*  && -f $p ]]; then
        # Nothing to do, it's absolute already
        :
    elif [[ $p == */* ]]; then

        #   Perhaps there is filename too? dir/dir/file.txt
        #   Remove last portion

        local file=${p##*/}
        local dir=${p%/*}

        if [ -d $dir ]; then
            dir=$(cd $dir; pwd)
            p=$dir/$file
        fi
    else
        if [ -f $p ]; then
            p=$(cd . ; pwd)/$p
        fi
    fi

    echo $p
}


function CygbuildCommandPath()
{
    local cmd="$1"

    #  Find out where is the absolute location of CMD
    #  type: cygbuild.sh is /cygdrive/...
    #        ls is hashed (/bin/ls)

    if [[ $cmd == */* ]]; then
        cmd=$(CygbuildAbsolutePath $cmd)
    else
        local saved="$IFS"
        local IFS=" "

        set -- $(type $cmd)
        IFS="$saved"

        local path=$3

        if [ "$path" = "hashed" ]; then
            path=$4
        fi

        path=${path%)}      # Remove possible parentheses
        path=${path#(}
        cmd=$path
    fi

    echo $cmd
}


function CygbuildBuildScriptPath()
{
    local id="$0.$FUNCNAME"

    #   Note, that source packages include script-VERSION-RELEASE.sh
    #   not just script.sh, so $0 cannot always be used directly.

    local name="$0"

    #   If there is path component, then perhaps "verify" is calling the script
    #   like ./script-NN.NN-1.sh, skip that case

    if [[ $name != */* ]]; then
        name=${name%%-*}        # Delete up til "-VER-REL"
        name=${name%.sh}        # Delete .sh, because it is attached below
        name="$name.sh"
    fi


    path=$(CygbuildAbsoluteScriptPath $name)

    echo $path
}


function CygbuildTemplatePath()
{
    #   Find out where template files are located

    local dir=$CYGBUILD_TEMPLATE_DIR

    if [ -d $dir ]; then
        echo $dir
    else
        dir=$(CygbuildBuildScriptPath)
        #   It's relative to the installation of the *.sh file, this may be an
        #   CVS checkout directory
        #
        #   dir/bin/cygbuild.sh
        #   dir/etc/template        << Here

        dir=${dir%/*}       # Remove filename: /path/to/FILE.SH => /path/to

        if [ -d $dir/../etc/template ]; then
            echo $dir
        elif [ -d $dir/etc/template ]; then
            echo $dir/etc/template
        fi
    fi

}

function CygbuildTarOptionCompress()
{
    #   Return correct packaging command based on the filename
    #   .tar.gz or .tgz     => "z" option
    #   .bz2                => "j" option"

    case "$1" in
        *tar.gz|tgz)    echo "z" ;;
        *bz2)           echo "j" ;;
        *)              echo ""  ;;
    esac
}


function CygbuildMakefileName ()
{
    local dir=${1:-"."}
    local file path

    for file in GNUMakefile Makefile makefile
    do
        path=$dir/$file

        if [ -f $path ]; then
            echo $path
            break
        fi
    done
}

function CygbuildIsMakefileTarget()
{
    local id="$0.$FUNCNAME"
    local target="$1"
    local file=$(CygbuildMakefileName)

    if [ -z "$file" ]; then
        echo "$id: [WARN] No Makefile found" 1>&2
        return 1
    fi

    grep "^$target:" $file > /dev/null
}

function CygbuildIsPerlPackage()
{
    [ -f $srcdir/Makefile.PL ]
}


function CygbuildIsPythonPackage()
{
    [ -f $srcdir/setup.py ]
}


function CygbuildSourceDownloadScript ()
{
    local -a arr=( $(ls *source-install* 2> /dev/null) )
    local len=${#arr[*]}
    local file

    if [ $len -eq 1 ]; then
        file=${arr[0]}
    fi

    echo $file
}

function CygbuildGetOneDir()
{
    #   Return one Directory, if there is only one.

    local from=${1:-"."}

    #   AWK get all entries that include "/" and then deleted trailing "/"

    local -a arr=( $(ls -F $from | awk  '/\/$/ && ! /tmp/ {   \
        sub("/$", "");                                        \
        print;                                                \
        exit;                                                 \
        }') )

    local count=${#arr[*]}
    local dir

    if [ $count -eq 1 ]; then
        dir=${arr[0]}
    fi

    echo $dir
}

function CygbuildMoveToTempDir ()
{


    local id="$0.$FUNCNAME"

    #   Move all files execpt cygbuild*.sh from DIR to temporary directory,
    #   which is deleted beforehand.
    #
    #   Return TEMP DIR
    #
    #   This function is menat for packages that do not contain directory
    #   structure at all, but unpack in place. The idea is to move files
    #   to separate directory to get clean unpack.

    local dir="$1"
    local dest=${2:-"tempdir"}     # optional parameter, if no name given

    dir=$(cd $dir && pwd)

    if [ -z "$dir" ]; then
        echo "$id: [ERROR] DIR input parameter is empty"  1>&2
        return 1
    fi

    local temp=$dir/$dest

    if [ -d $temp ]; then
        rm -rf $temp
    fi


    mkdir $temp || return 1

    #   Move everything else, but the directory itself and
    #   the build script, that is not belong to the original
    #   package

    (
        cd $dir
        mv $(ls | egrep -v "$dest|cygbuild*.sh") $dest
    )


    echo $temp
}

#######################################################################
#
#       Core functions: Define globals and do checks
#
#######################################################################


function CygbuildDefineGlobalsRoot()
{
    #   Define generic globals. However this hass been split to two
    #   functions which define complete set of globas:
    #
    #   CygbuildDefineGlobalsRoot         This
    #   CygbuildDefineGlobalsSrcOrig      And the sister function


    local id="$0.$FUNCNAME"
    local retval=$RETVAL.$FUNCNAME

    local top="$1"
    local src="$2"
    local release="$3"
    local pkg="$4"
    local directive="$5"


    #   - If filename or explicit release was given
    #   - or examine source directory name package-NN.NN/ name

    local templatepkg=${pkg:-$src}

    if  [[ $templatepkg != *[0-9]*.[0-9.]* ]] &&
        [[ $templaterel == *[0-9]*.[0-9.]* ]]; then

        #   Fix it. This is better
        templatepkg=$templaterel
    fi

    if  [[ $templatepkg != *[0-9]*.[0-9.]* ]] &&
        [[ $release     == *[0-9]*.[0-9.]* ]]; then

        #   Fix it. This is better
        templatepkg=$release
    fi




    if [[ $templatepkg != *[0-9]*.[0-9.]* ]]; then

        # Does not look like a correct version, complain
        echo "$id: [WARN] Can't derive VERSION from variable [$templatepkg]." \
             "Perhaps you need option -f" 1>&2
    fi

    #   Pick any of these, in this order as  avariable where
    #   we dig out the version information.

    local templaterel=${release:-${package:-$src}}

    CygbuildStrPackage $templatepkg > $retval
    local pkgname=$(< $retval)

    CygbuildStrVersion  $templatepkg > $retval
    local pkgver=$(< $retval)

    if [[ $pkgver != [0-9]*.[0-9]* ]]; then
        echo "$id: [ERROR] Cannot determine VERSION from $templatepkg" 1>&2
        echo "$id: [ERROR] Are you inside directory package-NN.NN/ ?" 1>&2
        return 1
    fi


    local relver=$templaterel

    if [[ $relver == *[!0-9]* ]]; then
        CygbuildStrRelease $relver > $retval
        relver=$(< $retval)
    fi


    if [[ $directive != *noCheckRelease ]]; then

        if [[ ! $relver > 0  ]]; then
            # Does not look like a correct version, complain
            echo "$id: [ERROR] Can't derive RELEASE from $src. " \
                 " See option -r" 1>&2
            exit 1
        fi
    elif [[ $relver != [0-9]* ]]; then
        relver=
    fi

    if [[ $relver != [0-9]* ]]; then
        echo "$id: [WARN] RELEASE '$relver' is not a number" 1>&2
    fi


    export buildscript=$(CygbuildBuildScriptPath)

    export PKG=$pkgname
    export VER=$pkgver
    export REL=$relver
    export FULLPKG=${PKG}-${VER}-${REL}


    # run on
    export host=i686-pc-cygwin

    # if this package creates binaries, they run on
    export target=i686-pc-cygwin
    export prefix=/usr
    export sysconfdir=/etc
    export CYGBUILD_CFLAGS="-O2 -g"
    export CYGBUILD_LDFLAGS=
    export CYGBUILD_MAKEFLAGS="CC=gcc CXX=g++"

    # top=${top%/*}

    # if the orig src package is bzip2'ed, remember to
    # change 'z' to 'j' in the 'tar xzvf' commands in the
    # prep) and mkpatch) sections

    export src_pkg_build_script=$FULLPKG.sh
    export src_pkg_name=$FULLPKG-src.tar.bz2
    export src_patch_name=$FULLPKG.patch
    export bin_pkg_name=$FULLPKG.tar.bz2
    export dev_pkg_name=$PKG-devel-$VER-$REL.tar.bz2
    export doc_pkg_name=$PKG-doc-$VER-$REL.tar.bz2
    export lib_pkg_name=$LIBPKG.tar.bz2

    export LIBPKG=$FULLPKG

    if [[ $PKG != lib* ]]; then
        export LIBPKG=lib$FULLPKG
    fi

    # TOPDIR files


    export src_pkg=$top/$src_pkg_name
    export src_patch=$top/$src_patch_name
    export bin_pkg=$top/$bin_pkg_name


    export dev_pkg=$top/$dev_pkg_name
    export doc_pkg=$top/$doc_pkg_name
    export lib_pkg=$top/$lib_pkg_name


    export TOPDIR=$top
    export srcdir=$top/$PKG-$VER


    # export objdir=${srcdir}/.build
    export objdir=$srcdir

    #   DO NOT CHANGE, .sinst and .inst and other are in fact hard coded
    #   elsewhere too to prevent accidental rm -rf

    export instdir_relative=.inst
    export instdir=$srcdir/$instdir_relative


    export srcinstdir_relative=.sinst
    export srcinstdir=$srcdir/$srcinstdir_relative

    export builddir_relative=.build
    export builddir=$srcdir/$builddir_relative

    export checkfile=$top/${FULLPKG}.check

    #  Documentation and setup directories

    local _docdir=$CYGBUILD_DOCDIR_RELATIVE   # temp cariable

    export cygpatch=${srcdir}/CYGWIN-PATCHES

    export cygdoc=${instdir}${prefix}/$_docdir/Cygwin
    export cygpostinstall=${instdir}$sysconfdir/postinstall
    export docdir=${instdir}${prefix}/$_docdir/$PKG-$VER
    export infodir=${instdir}${prefix}/usr/info


    export scriptDiff=$cygpatch/diff.sh
    export scriptConfigure=$cygpatch/configure.sh
    export scriptBuild=$cygpatch/build.sh
    export scriptInstall=$cygpatch/install.sh
    export scriptInstallAfter=$cygpatch/install-after.sh
    export cygpostinstall=$cygpatch/postinstall.sh
    export scriptPackageBin=$cygpatch/package-bin.sh
    export scriptPackageSource=$cygpatch/package-source.sh

    export scriptSourceGet=$srcinstdir/source-install.sh

    export installScript=${CYGBUILD_INSTALL:-"/usr/bin/install"}
    export installFmodes=${INSTALL_DATA:-"-m 644"}
    export installXmodes=${NSTALL_BIN:-"-m 755"}

}


function CygbuildCygbuildDefineGlobalsSrcOrigGuess()
{
    #   Define source package related globals. CygbuildDefineGlobalsRoot must
    #   have been called prior this function.

    local id="$0.$FUNCNAME"
    local name
    local pkg

    local dummy=$(pwd)    # for debugging


    if [[ $packageGuess == *tar.* ]]; then
        #  The Main function set this variable and set the source package
        pkg=$packageGuess
        name=${pkg##*/}     # Delete path
    else
        local ext

        for ext in .tar.gz .tar.tgz .tar.bz2
        do

            #  Standard version uses hyphen  : package-NN.NN.tar.gz
            #  Debian version uses underscore: package_NN.NN.tar.gz

            local file try

            for file in $PKG-$VER$ext       \
                        $PKG-$VER-src$ext   \
                        ${PKG}_$VER$ext     \
                        ${PKG}_$VER.orig$ext

            do

                try=$TOPDIR/$file

                if [ -f $try ]; then
                    name=$file
                    pkg=$try
                    break 2
                elif [ -h $try ]; then
                    echo "--    [WARN] Dangling symlink found: $TOPDIR" 1>&2
                    ls -l $try
                fi

            done

        done
    fi

    export src_orig_pkg_name=$name
    export src_orig_pkg=$pkg


}

function CygbuildDefineGlobalsSrcOrig()
{

    #   Define Source package related globals. CygbuildDefineGlobalsRoot
    #   must have been called prior this function.

    local id="$0.$FUNCNAME"

    local dummy=$(pwd)    # for debugging

    if [ -z "$PKG" ] || [ -z "$VER" ]; then
        echo "$id: [FATAL] variables PKG and VER not known." \
             "Are you inside package-NN.NN/ ?" 1>&2
        return 1
    fi


    if CygbuildFileExists $OPTION_FILE ; then
        #  If user told where the source files was, then examine that
        local name=${OPTION_FILE##*/}     # Remove path
        export src_orig_pkg_name
        export src_orig_pkg=$OPTION_FILE
    else
        #  No chance. Try guessing where that source file is
        CygbuildCygbuildDefineGlobalsSrcOrigGuess
    fi


    if [ -z "$src_orig_pkg" ]; then
        echo -e "$id: [FATAL] src_orig_pkg $PKG-$VER.tar.gz not found." \
                "Perhaps you have to make a symbolic link from original" \
                "to that file? See manual for details." 1>&2
        return 1
    else
        export src_orig_pkg=$(CygbuildAbsolutePath $src_orig_pkg)
    fi


}



function CygbuildSrcDirCheck()
{
    #   We must know where the sources are, in orger to run conf, make or
    #   mkpatch etc.

    local id="$0.$FUNCNAME"
    local dir="$1"

    if [ -z "$dir" ]; then
        echo "$id: [FATAL] dir is empty" 1>&2
        exit 1
    fi

    dir=${dir##*/}
    local pkg=${dir%%-*}
    local ver=${dir##*-}


    if  ! echo "$ver" | egrep -e '^[0-9]+\.[0-9][0-9.]*$' > /dev/null; then
        cat <<EOF 1>&2
$id: [ERROR] Cannot determine plain numeric VERSION in format NN.NN

The directory $dir
does not look like package-VERSION. Variables cannot be contructed.
You have options:

- chdir to package-NN.NN/ directory and use -f package-NN.NN.tar.gz.
  If package name does not have VERSION, and is something like
  foo-latest.tar.gz, make a symbolic link to foo-1.3.tar.gz and try -f again.
- Extract that package, and chdir to package-NN.NN/ and try separate
  options: 'mkdirs' 'files' 'conf' 'make'
- If the package does not extract to package-NN.NN/ make a symbolic link
  and chdir into it. ln -s foo3.3alpha3 foo-3.3.1.3; cd  foo-3.3.1.3/

A VERSION must be present either in package name or in directory name

EOF
        exit 1
    fi
}

function CygbuildSrcDirLocation()
{
    local dir="$1"

    #   find src directory.
    #   If scriptdir ends in SPECS, then top is $scriptdir/..
    #   If scriptdir ends in CYGWIN-PATCHES, then top is $scriptdir/../..
    #   Otherwise, we assume that top = scriptdir

    local top=$dir
    local src=$dir

    if [ -f $dir/configure  -o -f $dir/buildconf ]; then
        top=$(cd $dir/..; pwd)
    elif echo $top | egrep -e '-[0-9]+\.[0-9][0-9.]*$' > /dev/null; then
        # Looks like we're inside package-NN.NN/
        top=$(cd $dir/..; pwd)
    elif [[     $dir == */CYGWIN-PATCHES
             || $dir == */SPECS
             || $dir == */debian
         ]]
    then
        src=$(cd $dir/..; pwd)
        top=$(cd $src/..; pwd)
    else
        top=$(cd $dir; pwd)
        src=$top
    fi

    echo $top $src


}



#######################################################################
#
#       Documentation functions
#
#######################################################################


function CygbuildHelpShort()
{
    local bin=$(CygbuildBuildScriptPath)

    cat <<EOF

Call syntax: $bin [--help -v -x] [-f ORIG-SRC.tar.gz] -r RELEASE-NBR CMD ...

Tool for making, bulding and maintaining Cygwin source and binary
packages. Like Debian dh_make or other similar build tools.

INSTALLING CYGBUILD CONTROLLED SOURCE PACKAGE

    If you downloaded a Cygwin source package, like
    package-NN.NN-RELEASE-src.tar.gz, it should unpacked to these three
    files:

        foo-NN.NN-RELEASE-src.tar.bz2
        foo-NN.NN-RELEASE.patch
        foo-NN.NN-RELEASE.sh

    Run included shell script:

        ./foo-NN.NN-RELEASE.sh all

    The command builds a binary package and it also tries to build a source
    package - which - will fail if you do not have cygbuild suite
    installed.

TO USE FULL CYGBUILD SUITE TO PORT PACKAGES

    In cygbuild, the CMD can be one of the following:

        To prepare port: mkdirs files conf
        To port        : build strip install
        To package     : package source-package
        To publish     : publish (copy files to publish area)

NOTES

    There is also long --help option, but it would consult a separate
    manual. To read it, you need to install cygbuild suite from
    $CYGBUILD_HOMEPAGE_URL

    To build source package, you need full cygbuild suite.

    For more information about porting packages to cygwin, refer to
    <http://cygwin.com/setup.html>.

EOF

}


function CygbuildHelpLong()
{
    local bin=$(CygbuildCommandPath cygbuild.pl)

    if [ ! -z "$bin" ]; then
        $bin help
    else
        echo "[ERROR] Cannot find Long help page." \
             "You need to install 'cygbuild' suite." 1>&2
        CygbuildHelpShort
    fi
}


function CygbuildHelpSourcePackage()
{
    local bin=$(CygbuildCommandPath cygbuild.pl)

    if [ -z "$bin" ]; then

        echo "NOTE: To build a source package, install Cygbuild from" \
             "$CYGBUILD_HOMEPAGE_URL"
    fi
}

#######################################################################
#
#       Misc functions
#
#######################################################################


function CygbuildNoticeCygwinPatches ()
{
    cat<<EOF
It appears that there is no directory
$cygpatch

It should at minimum include 'package.README' and 'setup.hint'.
The files must be in place before making a binary or source package.

You can generate template files with command [files].
EOF

}


function CygbuildNoticeMaybe()
{
    if [ ! -d $cygpatch ]; then
        CygbuildNoticeCygwinPatches
    fi
}



function CygbuildCygDirCheck()
{
    local id="$0.$FUNCNAME"

    # Make sure there is a README at /usr/share/doc/Cygwin/

    if [ ! -d $cygdoc ]; then
        echo "$id: [ERROR] no $cygdoc. " \
             "Did forget to run [files] before [install]?" 1>&2
        exit 1
    fi

    local readme=$(ls $cygdoc/*README*)

    if [ -z "$readme" ]; then
        echo "$id: [ERROR] no $cygdoc/package.README; " \
             "Did forget to run [files]'?" 1>&2
        exit 1
    fi

    egrep -n -e '\<(PKG|VER|REL)\>' $readme /dev/null

    if [ $? == 0 ]; then
        echo "$id: [WARN] $cygdoc/package.README contains tags. " \
             "Edit or see [readmefix]; and run [install] " 1>&2

    fi

}

function CygbuildDetermineReadmeFile()
{
    local id="$0.$FUNCNAME"
    local ret
    local file

    for file in  $cygpatch/$PKG.README $cygpatch/README $cygpatch/*.README
    do
        #   install first found file
        if [ -f $file ]; then
            ret=$file
            break
        fi
    done

    echo $ret
}


function CygbuildDetermineDocDir()
{
    local id="$0.$FUNCNAME"
    local dir=$1

    if [ -z "$dir" ]; then
        echo "$id: [ERROR]: call parameter DIR is missing."
        exit 1
    fi

    if [ ! -d  $dir ]; then
        return 1
    fi


    local ret
    local try

    #   http:://wwww.fprot.org/ uses doc_ws

    for try in $dir/doc $dir/docs dir/doc_ws
    do
        if [ -d $try ]; then
            ret=$try
            break
        fi
    done


    echo $ret
}


#######################################################################
#
#       Publish functions
#
#######################################################################

function CygbuildPerlModuleLocation()
{
    #   Find out if we can use cygbuild.pl module
    #   Return 1) Perl interpreter and 2) module path

    local id="$0.$FUNCNAME"

    local perl=cygbuild.pl
    local module=$(CygbuildCommandPath $perl)
    local PERL=$(CygbuildCommandPath perl)

    if [ -z "$PERL" ]; then
        echo "$id: [ERROR] perl is not in PATH. Cannot run 'readmefix'." \
            1>&2
        return 1
    fi


    if [ -z "$module" ]; then
        echo "$id: [ERROR] file not found: $perl $module" \
             "Have you $CYGBUILD_HOMEPAGE_URL installed?" \
             1>&2
        return 1
    fi

    echo $PERL $module

}


function CygbuildCmdReadmeFix()
{
    local id="$0.$FUNCNAME"

    local readme=$(CygbuildDetermineReadmeFile)

    echo "-- Fixing $readme"

    if [ -z "$readme" ]; then
        echo "$id: [ERROR] Not found $cygpatch/package.README"
        return 1
    fi

    if [ ! -r $readme ]; then
        echo "$id: [ERROR] not readable $readme"
        return 1
    fi





    if ! CygbuildFileExists $bin_pkg ; then
        echo "$id: [ERROR] Cannot read files from '$bin_pkg' " \
             " Run 'package' first." 1>&2
        return 1
    fi


    if [ -z "$srcdir"  -o ! -d "$srcdir" ]; then
        return 1
    fi

    local -a arr=( $(CygbuildPerlModuleLocation) )
    local PERL=${arr[0]}
    local module=${arr[1]}

    if [ -z "$module" ]; then
        return 1
    fi


    # CygbuildDefineGlobalsSrcOrig || return 1

    #   1. Load library MODULE
    #   2. Call function Readmefix() with parameters. It will handle the
    #      text manipulation details.

    cp $readme $readme.bak                                      && \
    $PERL -e "require qq($module);                                 \
      ReadmeFix(qq($readme), qq($FULLPKG), qq($bin_pkg));"         \
      > $readme.tmp                                             && \
    mv $readme.tmp $readme

}

function CygbuildCmdPublishSetupFix()
{
    local id="$0.$FUNCNAME"
    local dest=${CYGWIN_BUILD_PUBLISH_DIR:-"/usr/src/cygwin-packages"}
    dest=$dest/$PKG

    #  Rename setup files

    local to dir file suffix base

    for dir in $dest/devel $dest/doc
    do
        if [ -d $dir ]; then
            suffix=${dir##*/}
            base=setup-$suffix.hint
            file=$dir/$base
            to=$dir/setup.hint

            if [ -f $file ]; then
                mv $file  $to
            fi

            if [ ! -f $to ]; then
                echo "--    [WARN] Cannot rename $file => $to" 1>&2
                echo "--    [WARN] Did you write CYGWIN-PATCHES/$base ?" 1>&2
            fi
        fi

    done

}

function CygbuildCmdPublishNormal()
{

    local id="$0.$FUNCNAME"
    local dest=${CYGWIN_BUILD_PUBLISH_DIR:-"/usr/src/cygwin-packages"}

    dest=${dest%/}  # Delete possible trailing slash

    if [ ! -d $dest ]; then

        echo "$id: [ERROR] No dir $dest, set CYGWIN_BUILD_PUBLISH_DIR"
        return 1

    else

        dest=$dest/$PKG

        echo "-- Publishing to $dest"

        if [ ! -d $dest ]; then
            mkdir $verbose  -p $dest || return 1
        fi

        #  For library packages, the hierarchy is
        #  base/
        #  base/devel/
        #  base/doc/

        local dirDevel=$dest/devel
        local dirDoc=$dest/doc

        local file

        for file in $(ls $TOPDIR/$PKG-${VER}-*tar.bz2 )         \
                    $(ls $TOPDIR/$PKG-devel-${VER}-*tar.bz2 2> /dev/null )   \
                    $(ls $TOPDIR/$PKG-doc-${VER}-*tar.bz2 2> /dev/null )     \
                    $cygpatch/setup.hint                        \
                    $cygpatch/setup-devel.hint                  \
                    $cygpatch/setup-doc.hint
        do

            [ ! -f $file ] && continue

            if [[ $file == *-devel* ]]; then
                mkdir -p $dirDevel || return 1
                cp $file $dirDevel || return 1
            elif [[ $file == *-doc* ]]; then
                mkdir -p $dirDoc || return 1
                cp $file $dirDoc || return 1
            else
                cp $file $dest || return 1
            fi

            file=${file##*/}
            echo "--    $file"
        done

        CygbuildCmdPublishSetupFix

    fi
}

function CygbuildCmdPublishExternal()
{
    local id="$0.$FUNCNAME"
    local prg="$CYGBUILD_PUBLISH_BIN"

    echo "-- [Publish] with external call: $prg $TOPDIR"

    $prg $TOPDIR
}

function CygbuildCmdPublishMain()
{
    local id="$0.$FUNCNAME"

    if [ ! -z "$CYGBUILD_PUBLISH_BIN" ]; then
        CygbuildCmdPublishExternal
    else
        CygbuildCmdPublishNormal
    fi
}


function CygbuildCmdPkgExternal()
{
    local id="$0.$FUNCNAME"
    local prg=$scriptPackageBin

    (
        cd $instdir &&
        echo "-- Making package [binary] with external " \
             " $prg $PKG $VER $REL" && \
        $rg $PKG $VER $REL $TOPDIR

        if [ $? -ne 0  ]; then
            local stat=$?
            echo "$id: [ERROR] Failed create binary package."
            return $stat
        fi
    )
}

function CygbuildCmdPkgDevelStandard()
{
    (

        docdir=$CYGBUILD_DOCDIR_FULL

        echo "-- Making packages [devel] from $instdir"

        cd $instdir || return 1

        files="$(find usr -name '*.dll')"

        if [ -z "$files" ]; then
            echo "--    [devel-lib] WARNING; No *.dll files found. " \
                 "Skipping $lib_pkg" 1>&2
        else
            echo "--    [devel-lib] $lib_pkg"  &&
            tar $verbose $CYGBUILD_TAR_OPT -jcf $lib_pkg $files
        fi


        echo "--    [devel-dev] $dev_pkg"
        tar $CYGBUILD_TAR_OPT $verbose -jcf $dev_pkg usr/include usr/lib &&

        dirlist=""
        docdir=$CYGBUILD_MANDIR_FULL

        echo "--    [devel-doc] $doc_pkg"

        if [ -d $docdir ]; then
            dirlist="$docdir"
        else
            echo "--    [WARN] $docdir not found. Nothing to do." 1>&2
        fi

        docdir=$CYGBUILD_DOCDIR_FULL

        if [ -d $docdir ]; then
            dirlist="$dirlist $docdir"
        else
            echo "--    [devel-doc] [WARN] $docdir not found. Nothing to do." \
                1>&2
        fi


        if [ -z "$docdir" ]; then
            echo "--    [devel-doc] [WARN] No doc files to package. " \
                 "Skipping $doc_pkg" 1>&2
        else
            tar $CYGBUILD_TAR_OPT $verbose -jcf $doc_pkg $dirlist
        fi

    )

}


function CygbuildCmdPkgDevelMain()
{
    local id="$0.$FUNCNAME"

    CygbuildCygDirCheck  || return $?

    if CygbuildFileExists $scriptPackageBin ; then
        CygbuildCmdPkgExternal
    else
        CygbuildCmdPkgDevelStandard
    fi
}


function CygbuildCmdPkgBinaryStandard()
{
    (
        echo "-- Making package [binary] $bin_pkg"
        cd $instdir &&
        tar $CYGBUILD_TAR_OPT $verbose -jcf $bin_pkg *
    )
}

function CygbuildCmdPkgBinaryMain()
{
    local id="$0.$FUNCNAME"

    CygbuildCygDirCheck  || return $?

    if CygbuildFileExists $scriptPackageBin ; then
        CygbuildCmdPkgExternal
    else
        CygbuildCmdPkgBinaryStandard
    fi
}


CygbuildPackageSourceDirClean()
{
    # Clean previous sourcepacakge install and start from fresh.
    # Make sure it looks like .sinst or rm -rf will do havoc

    if [[  $srcinstdir == *.sinst* ]]; then
    (
        cd $srcinstdir && rm -rf *
    )
    fi

}

function CygbuildCmdPkgSourceStandard()
{
    local id="$0.$FUNCNAME"


    if [ ! -d $srcinstdir ]; then
        echo "$id: [FATAL] No directory $srcinstdir. Try running [mkdirs]"
        exit 1
    fi


    if ! CygbuildDefineGlobalsSrcOrig ; then
        echo "$id: [ERROR] Original source kit must be available."
        return 1
    fi


    if ! CygbuildFileExists $buildscript ; then
        #  Still not found, complain to user
        echo "$id: [ERROR] Can't locate build script [$buildscript]" 1>&2
        exit 1
    fi


    local orig=$src_orig_pkg

    if ! CygbuildFileExists $orig ; then
        echo "$id: [FATAL] No src_orig_pkg defined."
        exit 1
    fi

    local makepatch="yes"

    if ! CygbuildFileExists $orig ; then
        echo "$id: [WARN] Cannot diff. Don't know where original " \
             "source package is. Do you need -f or make a symbolic " \
             "link to PKG-VER.tar.gz?" 1>&2
        makepatch=
    fi



    CygbuildPackageSourceDirClean

    if [ ! -z "$makepatch" ]; then
        CygbuildCmdMkpatch || return $?
    fi


    if ! CygbuildFileExists $buildscript ; then
        echo "$id: [FATAL] Cannot find buildscript, please debug." 1>&2
        return 1
    fi

    # .......................................... make source package ...

    local name=$src_pkg_build_script    # script-VERSION-RELEASE.sh
    local tarOpt="$verbose -jcf"

    echo "-- Making package [source] $src_pkg"

    cp $orig $srcinstdir/$src_orig_pkg_name                     &&
    cp $buildscript $srcinstdir/$name                           &&
    (
        cd $srcinstdir                                          &&
        tar $CYGBUILD_TAR_OPT $tarOpt $src_pkg *
    )

    return $?

}

function CygbuildCmdPkgSourceExternal ()
{
    local id="$0.$FUNCNAME"
    local prg=$scriptPackagesSource

    (
        cd $instdir &&
        echo "-- Making package [source] with external " \
             " $prg $PKG $VER $REL" && \
        $prg $PKG $VER $REL $TOPDIR

        if [ $? -ne 0  ]; then
            local stat=$?
            echo "$id: [ERROR] Failed create source package."
            return $stat
        fi
    )
}

function CygbuildCmdPkgSourceCVSdiff ()
{
    local id="$0.$FUNCNAME"
    local out="$srcinstdir/$src_patch_name"
    local orig=$builddir/tmp/cvsout
    local prg=$scriptSourceGet

    if ! CygbuildFileExists $prg ; then
        echo "$id: [ERROR] Can't find $prg" 1>&2
        return 1
    fi

    echo "-- Making [cvs patch] getting sources"

    #   If already there is a checkout, use it
    #   Otherwise download a fresh copy from pserver where we
    #   compare

    local try=$(cd $orig/*/ && pwd)

    if [ ! -z "$try" ] && [ -d $try ]; then
    (
        #   There is only one directory, so go there
        cd $try && cvs update
    )
    else
    (
        mkdir -p $orig          &&
        cd $orig                &&
        $prg --quiet            &&
        cd ./*/                 &&
        try=$(pwd)
    )
    fi

    if [ -z "$try" ] || [ ! -d $try ]; then
        echo "$id: [FATAL] Something went wrong with cvs up. Run debug." \
            1>&2
        return 1
    fi


    #   Files that were generated after checkout must be filtered out
    #   Examine how the current directory is different from the
    #   original chekout (Perl function does this)

    echo "-- Making [cvs patch] Examining files to exclude, wait."

    local exclude=$( \
        $PERL -e "require qq($module); DiffToExclude(qq($try));" \
    )

    local status=$?

    if [ $status -ne 0 ]; then
        echo "$id: [ERROR] Hm, either full 'cygbuild' suite is not" \
             "installed or cygbuild.pl perl module is not in PATH." \
             "See $CYGBUILD_HOMEPAGE_URL" 1>&2
        return $status
    fi

    #   Good. Now we have all the details to make the patch
    #   It is not guaranteed that this is perfectly okay.
    #   => Every user added directory except CYGWIN-PATHCES are
    #   => filtered out.

    echo "-- Making [cvs patch] $out"

    #   We must make the patch relative to the current directory.
    #
    #       diff -r there/cvs-dir/ package-NN.NN/

    local compare=$(basename $(pwd))

    (
        cd ..
        LC_ALL=C TZ=UTC0 diff $CYGBUILD_CVSDIFF_OPTIONS \
            $exclude $try $compare \
            > $out

        #   GNU diff(1) changed return codes.
        #   Return code 1 is OK and value > 1 is an error
        #
        #   0 = no differences
        #   1 = differences
        #   2 = some sort of recoverable error occurred
        #       => that means that there were biary files. Look for
        #       message "Files X and Y differ"


        local stat=$?

        if [ "$stat" = "1" ]; then
            true                    #  That's OK!
        else
            echo "$id: [ERROR] Making a patch failed, check $out"
            egrep -n -i 'files.*differ' $out
            return $stat
        fi

    )

}

function CygbuildCmdPkgSourceCVS ()
{
    local id="$0.$FUNCNAME"
    local dummy

    local -a arr=( $(CygbuildPerlModuleLocation) )
    local PERL=${arr[0]}
    local module=${arr[1]}

    if [ -z "$module" ]; then
        return 1
    fi

    if ! CygbuildFileExists $buildscript ; then
        echo "$id: [FATAL] Cannot find buildscript, please debug." 1>&2
        return 1
    fi



    CygbuildPackageSourceDirClean

    if CygbuildFileExists $scriptSourceGet ; then
        rm $scriptSourceGet
    fi

    local dir=dummy=$(pwd)                # For debugger

    $PERL -e "require qq($module);                 \
      PackageCVSmain(qq($srcdir), qq($FULLPKG));" \
    > $scriptSourceGet

    local stat=$?

    if [ $stat -ne 0 ]; then
        return $stat
    fi

    if  [ ! -f $scriptSourceGet ] || [ ! -s $scriptSourceGet ]; then
        echo "$id: [ERROR] Dind't see $scriptSourceGet"
        return 1
    fi

    echo "--    Wrote $scriptSourceGet"

    CygbuildCmdPkgSourceCVSdiff  || return $?

    # .......................................... make source package ...

    local name=${buildscript##*/}        # Remove path
    name=${name%.sh}                     # Remove extension
    name=$name-$VER-$REL.sh              # script-VERSION-RELEASE.sh

    local tarOpt="$verbose -jcf"

    #   There is no source.tar.gz to go with this, but we include
    #   a shell script that gets the source from CVS. Rename the
    #   Checkout script with 'mv'.
    #
    #       source-install.sh => <PKG>-source-install-NN.NN-REL.sh

    local script=${scriptSourceGet##*/}
    script=${script%.sh}
    script=$PKG-$script-$VER-$REL.sh


    mv $scriptSourceGet $srcinstdir/$script                     &&
    cp $buildscript $srcinstdir/$name                           &&
    echo "-- Making package [source] $src_pkg"                  &&
    (
        cd $srcinstdir                                          &&
        tar $CYGBUILD_TAR_OPT $tarOpt $src_pkg *
    )


}

function CygbuildCmdPkgSourceMain()
{
    local id="$0.$FUNCNAME"

    dummy=$(pwd)                # For debugger

    if CygbuildFileExists $scriptPackageSource ; then
        CygbuildCmdPkgSourceExternal
    elif [ -d $srcdir/CVS ]; then
        echo "-- Erm, packaging sources from CVS checkout dir is" \
             "still highly experimental."
        CygbuildCmdPkgSourceCVS
    else
        CygbuildCmdPkgSourceStandard
    fi
}



#######################################################################
#
#       Makefile functions
#
#######################################################################


function CygbuildPostinstallWrite()
{
    local id="$0.$FUNCNAME"
    local str="$1"

    if [ -z "$str" ]; then
        echo "$id: [FATAL] command string is empty"
        return 1
    elif CygbuildFileExists $cygpostinstall ; then
        if [ ! -z "$verbose" ]; then
            echo "$id: [WARN] cannot write to $cygpostinstall => $str" 1>&2
        fi
    else
        echo "$str" > $cygpostinstall
    fi
}


function CygbuildDebianRules2Makefile()
{

    local id="$0.$FUNCNAME"
    local file="$1"

    if [ -z "$file" ]; then
        echo "$id: [ERROR] argument FILE is missing"
        return 1
    fi


    local -a arr=( $(CygbuildPerlModuleLocation) )
    local PERL=${arr[0]}
    local module=${arr[1]}

    if [ -z "$module" ]; then
        return 1
    elif [ -z "$srcdir"  -o ! -d "$srcdir" ]; then
        return 1
    else

        local out=$srcdir/Makefile

        echo "--    Debian: writing $out" \
             "--    Debian: Makefile written was based on debian/rules. Please check!"

        $PERL -e "require qq($module);              \
          DebianRulesMain(qq($file), -makefile);"   \
        > $out

        if [ ! -s $out ]; then
            #  Some error happened if there was no output from perl
            echo "$id: [ERROR] failed to write $out" 1>&2
            rm -f $out
            return 1
        fi
    fi
}

function CygbuildDebianRules2MakefileMaybe()
{
    local id="$0.$FUNCNAME"
    local file="$srcdir/debian/rules"

    if [ -f $file ]; then
        echo "--    Debian: examining 'debian/rules'"
        CygbuildDebianRules2Makefile $file || return $?

        ( cd $srcdir ; make prefix=$instdir install )

    else
        return 1
    fi

}


function CygbuildMakeRunInstallFixPerl()
{
    local id="$0.$FUNCNAME"

    local -a arr=( $(CygbuildPerlModuleLocation) )
    local PERL=${arr[0]}
    local module=${arr[1]}

    if [ -z "$module" ]; then
        return 1
    fi


    #  There is one problem. make install want's to append to file:
    #  /usr/lib/perl5/5.8.0/cygwin-multi-64int/perllocal.pod
    #
    #  which get's installed as
    #  .inst/usr/lib/perl5/5.8.0/cygwin-multi-64int/perllocal.pod
    #
    #  and upon unpack it would replace the existing file. Tackle that.

    local file
    local ext=".postinstall_append"
    local commands
    local found


    for file in $(find $instdir -name perllocal.pod)
    do
        found="yes"

        mv $file $file$ext || return $?

        local dir=${file%/*}
        local name=${file##*/}
        local realdir=${dir#*.inst}    # relative .inst/usr => absolute /usr

        local from=$realdir/$name$ext
        local to=$realdir/$name

        echo "--    Perl install fix: $realdir/$name$ext"

        commands="
$commands
#  Append new utility to Perl installation
from=$from
to=$to
cat \$from >> \$to && rm \$from"

    done

    if [ $found ]; then
        CygbuildPostinstallWrite "$commands" || return $?
    fi

}



function CygbuildMakefilePrefixCheck()
{
    local id="$0.$FUNCNAME"
    local file=$(CygbuildMakefileName $objdir)

    if ! CygbuildFileExists $file ; then
        return 0
    fi

    grep 'DESTDIR' $file  >  /dev/null  2>&1

    if [ $? != 0 ]; then

        egrep '^[ \t]*(prefix|PREFIX)[ \t]*=' $file # >  /dev/null  2>&1

        local stat=$?

        if [ -z "$OPTION_PREFIX"  ] && [ $stat = 0 ]; then
            #  There was statement "prefix="
            echo "--    Hm, Makefile does not use DESTDIR but prefix." \
                 " Turning on option -P"
            export OPTION_PREFIX="automatic"
        else
            echo "--    [WARN] Makefile does not use variables 'DESTDIR' " \
                 " or 'prefix'. You may need to write custom install.sh" 1>&2
        fi
    fi

}


function CygbuildMakefileRunClean()
{
    #   Before making a patch, a "make distclean" should be run

    local id="$0.$FUNCNAME"

    echo "-- Running make(1)) clean distclean"

    (
        cd $objdir &&
        make clean distclean
    )

}

function CygbuildMakefileRunInstall()
{
    local id="$0.$FUNCNAME"
    local makefile=$(CygbuildMakefileName $objdir)

    # install under .inst/

    echo "--    Running 'make install'"
    echo "--    See also: find .inst/ -print"

    if CygbuildIsPerlPackage ; then

        #  Perl makefles use DESTDIR, but the configure phase already
        #  set the PREFIX, so DESTDIR would cause bad karma at this point

        Echo "--    ... Looks like Perl package"

        ( cd  $objdir && make install && CygbuildMakeRunInstallFixPerl )

    elif CygbuildFileExists $makefile ; then

        #   DESTDIR is standard GNU ./configure macro,
        #   which points to root of install.
        #   prefix and exec_prefix are relative to it.
        #
        #   Debian package uses @bin_prefi@PROGRAm to install
        #   programs under another name. Do not set it

        local _prefix=$prefix

        CygbuildMakefilePrefixCheck

        if [ ! -z "$OPTION_PREFIX" ]; then

            #  Debian packages don't use DESTDIR, so the only possibility to
            #  guide the installation process is to set PREFIX.

            _prefix=$instdir/usr
        fi

        local _docprefix=/$CYGBUILD_DOCDIR_PREFIX_RELATIVE

        (
            #  GNU autoconf uses 'prefix'


            cd $objdir                     &&
            make                            \
                 DESTDIR=$instdir           \
                 prefix=$_prefix            \
                 exec_prefix=$_prefix       \
                 man_prefix=$_docprefix     \
                 info_prefix=$_docprefix    \
                 bin_prefix=                \
                 $CYGBUILD_MAKEFLAGS        \
                 install

        )

    else

        echo "--    [WARN] There is no Makefile." 1>&2
        CygbuildDebianRules2MakefileMaybe


    fi

}



#######################################################################
#
#       Build functions
#
#######################################################################


function CygbuildCmdMkdirs()
{
    echo "-- Making Cygwin directories under $srcdir"

    (
        cd $srcdir                     &&
        mkdir $verbose -p $objdir      &&
        mkdir $verbose -p $builddir    &&
        mkdir $verbose -p $instdir     &&
        mkdir $verbose -p $srcinstdir  &&
        mkdir $verbose -p $cygpatch
    )

}


function CygbuildExtractTar()
{
    local id="$0.$FUNCNAME"
    local file=$src_orig_pkg

    #   Check that CygbuildDefineGlobalsSrcOrig
    #   defined variables correctly


    if ! CygbuildDefineGlobalsSrcOrig ; then
        echo "$id: [ERROR] Original source kit must be available."
        return 1
    fi


    local file=$src_orig_pkg

    if [ ! -f $file ]; then
        echo "$id: [FATAL] $file not found. Check" \
             "function CygbuildDefineGlobalsRoot()" 1>&2
        return 1
    fi


    local package=$(CygbuildStrPackage $file)
    local ver=$(CygbuildStrVersion $file)

    if [ -z "$package" ] || [ -z "$ver" ]; then
        echo "$id: [FATAL] $file does not look like package-NN.NN.tar.* " 1>&2
        return 1
    fi


    local expectDir=$package-$ver

    local z=$(CygbuildTarOptionCompress  $file)
    local opt="$verbose -${z}xf"

    echo "-- Extracting $file"
    tar $opt $file ||  return $?

    #  This archive did not upack to a directory package-NN.NN/

    if [ ! -d $expectDir ]; then

        echo "--    Hm, archive did not extract into directory $expectDir ..."

        local -a arr=( $(ls -F | grep '/$') )
        local count=${#arr[*]}


        if [ $count -eq 1 ]; then
            local dir=${arr[0]}
            echo "--    Renaming unpack dir $dir to $expectDir"
            mv $dir $expectDir || return $?

        elif [ $count -eq 0 ]; then

            local temp=$(CygbuildMoveToTempDir . $expectDir)
            local stat=$?

            echo "--    Hm, unpacking did not create any directories." \
            echo " Moving manually to $temp"

            return $stat

        else
            echo "--    Too many directories here, quitting. ${arr[*]}"
            return 1
        fi

    fi

}

function CygbuildExtractWithScript()
{
    local id="$0.$FUNCNAME"
    local prg="$1"

    if [ -z "$srcdir" ]; then
        echo "$id: [FATAL] 'srcdir' not defined"
        return 1
    fi

    echo "--    [!!] Getting external sources with $file"

    #   Now run the script and if it succeeds, we're ready to proceed to
    #   patching

    ./$prg  && \
    {
        if [ ! -d "$srcdir" ]; then
            #  The sript did not uplack to package-NN.NN, fix it

            dir=$(CygbuildGetOneDir)

            #   Good, there no more than ONE directory, which
            #   was just made by that script.

            to=$(basename $srcdir)

            if [ ! -z "$dir" ]; then
                echo "--    [!!] Download done. Symlinking $dir => $to" \
                     "in $(pwd)"
                ln -s $dir $to
                mkdir -p $srcdir
            fi
        fi
    }



}

function CygbuildExtractMain()
{
    local id="$0.$FUNCNAME"

    local file=$(CygbuildSourceDownloadScript)

    if [ ! -z "$file" ]; then
        CygbuildExtractWithScript $file
    else
        CygbuildExtractTar
    fi

}


function CygbuildApplyPatch()
{
    local id="$0.$FUNCNAME"

    #  --forward  Ignore patches that seem to be reversed
    #  --strip=N  Strip the smallest prefix containing num leading slashes
    #             setting 0 gives the entire file name unmodified
    #  --fuzz=N   Set the maximum fuzz factor.(default is 2)

    local patch_opt=$CYGBUILD_PATCH_OPT

    if [ -z "$verbose" ]; then
        patch_opt="$patch_opt --quiet"
    fi


   if CygbuildFileExists $src_patch ; then
        echo "-- Patching $patch_opt < $src_patch"
        echo "--    If patch fails, you may need rm -rf $srcdir"
        patch $patch_opt < $src_patch
    else
        echo "$id: [WARN] No cygwin specific path src_patch [$src_patch]" \
            1>&2
    fi


}

function CygbuildCmdPrepMain()
{
    local id="$0.$FUNCNAME"

    if [ -z "$src_pkg" ]; then
        if ! CygbuildDefineGlobalsSrcOrig ; then
            return 1
        fi
    fi

    if CygbuildDirExists $instdir ; then

        echo "--    Skipping 'prep' (and patching), source already unpacked"
    else
    (
        cd $TOPDIR          &&
        CygbuildExtractMain &&
        CygbuildCmdMkdirs   &&
        CygbuildApplyPatch
    )
    fi

}


function CygbuildCmdMkpatch()
{

    if ! CygbuildDefineGlobalsSrcOrig; then
        return 1
    fi

    local file=$src_orig_pkg
    local z=$(CygbuildTarOptionCompress  $file)
    local opt="$verbose -${z}xf"

    local orig=$TOPDIR/$PKG-$VER-orig
    local out="$srcinstdir/$src_patch_name"



    if [ -z "$orig" ]; then

        #  This may be impossible error message, but there is that rm -rf
        #  below, so no measure is enough to guard against errors.

        echo "$id: [ERROR] variable 'orig' is empty."
        exit 1
    fi

    if [ -d $orig ]; then
        rm -rf $orig
    fi

    CygbuildCmdCleanMain
    # CygbuildMakefileRunClean


    echo "-- Making [patch] $out"


    if [[ $builddir == *.build* ]]; then
        # Clear everything first.
        # ( cd $builddir ; rm -rf * )
        :
    fi

    #  It's impossible to know to what directory source was unpacked.
    #  Find it out with 'ls'.

    (

        cd $builddir  || return $?

        tar $opt $file || \
        {
            echo "$id: [ERROR] tar $opt $file"
            return 1
        }

        #   After unpack there should be only one directory
        #   However, e.g. [verify] creates a temporary directory
        #   here too, so filter out anything that read "tmp"

        dir=$(CygbuildGetOneDir)

        if [ -z "$dir" ]; then

            #  Hm, sometimes packages do not create subdirectories
            #  at all.

            if  ls -F | grep "/"  | grep -v tmp ; then
                echo "--    Found more than one dir at $builddir. " \
                     "Please clean directory"
                return 1
            else
                echo "--    [WARN] Original package did not unpack to a" \
                     "separate directory package-NN.NN. In fact there is" \
                     "no directories at all, so moving all files manually." \
                     1>&2

                dir=$(CygbuildMoveToTempDir .)
            fi

        fi

        mv $dir $orig                 &&
        cd $TOPDIR                    &&
        if CygbuildFileExists $scriptDiff ; then
            #   If there is custom script, run it.
            echo "--   Diff external $scriptDiff"
            $scriptDiff "${PKG}-${VER}-orig"  "${PKG}-${VER}" "$out"
        else

            local dummy=$(pwd)    # For debugging.

            LC_ALL=C TZ=UTC0                    \
                diff $CYGBUILD_DIFF_OPTIONS     \
                $PKG-$VER-orig   $PKG-$VER      \
                > $out

            #   GNU diff(1) changed return codes.
            #   Return code 1 is OK and value > 1 is an error

            stat=$?

            if [ "$stat" = "0" ]; then
                echo "$id: [ERROR] Making a patch failed, check $out"
                egrep -i -n 'files.*differ' $out
                return $stat
            else
                rm -rf $orig
            fi

        fi
    )

}


function CygbuildConfOld()
{
    local conf="$1"

    while [ ! -z "$opt" ]
    do
        $conf $CYGBUILD_CONFIGURE_OPTIONS

        if [ $? -eq 0 ]; then
            break
        else
            #  Remove element by element until success
            set -- ${list[@]}
            shift
            list=($@)
        fi
    done

}

function CygbuildConfDepend
{
    local id="$0.$FUNCNAME"

    #  if there is target 'depend' in Makefile, run it.


    echo "--   Running 'make depend'. Ignore possible error message."

    make depend

    return  # return ok status
}


function CygbuildConfCC()
{
    local id="$0.$FUNCNAME"
    local conf=$srcdir/configure
    local make=$(CygbuildMakefileName $srcdir)
    local opt="$CYGBUILD_CONFIGURE_OPTIONS"

    if ! CygbuildFileExists $conf ; then

        echo "--    [WARN] Hm, there is no $conf" 1>&2

        if CygbuildFileExists $make ; then
            echo "--    [WARN] Found only a $make. " \
                 " No configuring to do." 1>&2
        fi

        return 0  # OK status
    else
    (

        echo "--   Running ./configure with Cygwin specific options"

        if [ ! -z "$verbose" ]; then

            #   print the listing more nicely. Get a hand from perl here
            #   to format the option listing

            echo "$opt" | perl -ane "
            s/\s+/,/g; print '  ', join( qq(\n  ), split ',', $_), qq(\n)"
        fi

        cd $objdir                   &&
        CFLAGS= LDFLAGS= $conf $opt
#        || CygbuildConfOld $conf

    )
    fi
}


function CygbuildConfPerlCheck()
{
    local id="$0.$FUNCNAME"

    perl -e "use ExtUtils::MakeMaker 6.10"  # at least 6.10 works ok

    if [ $? != 0 ]; then

        cat<<EOF
$id [ERROR] It is not possible to make Perl source package.

Standard Perl (5.8.0) MM:MakeMaker 6.05 does not handle PREFIX variable
correctly to install files into separate directory. YInstall latest
MakeMaker from <http://search.cpan.org/author/MSCHWERN/ =>
ExtUtils-MakeMaker

  1. Download tar.gz and unpack, chdir to unpack directory
  2. Run: perl Makefile.PL
  3. Run: make install

EOF

        return 1
    fi


}

function CygbuildConfPerlMain()
{
    local id="$0.$FUNCNAME"
    local conf="$srcdir/Makefile.PL"

    #   Not good, Perl make Maker is broken. User cannot control PREFIX.
    #   See message http://www.makemaker.org/drafts/prefixification.txt


    if [ -f $conf ]; then
    (
        echo "--  Running: perl Makefile.PL PREFIX=$instdir/usr"

        local _prefix=$instdir/usr

        cd $objdir
        perl Makefile.PL # PREFIX=$_prefix SITEPREFIX=$_prefix

    )
    fi
}


function CygbuildCmdConfMain()
{
    local id="$0.$FUNCNAME"
    local perlconf="$srcdir/Makefile.PL"

    local dummy=$(pwd)      # For debugger

    if CygbuildFileExists $scriptConfigure ; then
    (
        echo "--   External $scriptConfigure"
        cd $objdir && $scriptConfigure $instdir
    )
    elif [ -f $perlconf ]; then
        CygbuildConfPerlCheck || return $?
        CygbuildConfPerlMain
    elif CygbuildIsMakefileTarget configure ; then
        echo "--   running: make configure (auto detected; no ./configure)"
        make configure
    else
        CygbuildConfCC
    fi
}


function CygbuildSetLDPATHpython ()
{
    local id="$0.$FUNCNAME"

    #  Make sure all paths are there

    local try=$(cd /usr/lib/python2.*/config; pwd)

    if  [ ! -z "$try" ] && [ -d $try ]; then
        [ ! -z "$verbose" ] &&  echo "--    Added Python to LD_LIBRARY_PATH"
        export LD_LIBRARY_PATH=$LD_LIBRARY_PATH:$try
        export LD_RUN_PATH=$LD_RUN_PATH:$try
    fi

}

function CygbuildCmdBuildPython()
{
    local id="$0.$FUNCNAME"

    (
        CygbuildSetLDPATHpython

        cd $objdir                                  &&
        echo "-- Building: python setup.py build"   &&
        python setup.py build
    )
}

function CygbuildCmdBuildMainMakefile()
{
    local id="$0.$FUNCNAME"

    (
        cd $objdir                                  &&
        echo "-- Building with standard make(1)"    &&
        CFLAGS="$CYGBUILD_CFLAGS" LDFLAGS="$CYGBUILD_LDFLAGS" \
            make $CYGBUILD_MAKEFLAGS
#                 $DESTDIR=$instdir
#                 $CYGBUILD_CONFIGURE_OPTIONS \


    )
}


function CygbuildCmdBuildMain()
{
    local id="$0.$FUNCNAME"

    if CygbuildFileExists $scriptBuild ; then
    (
        echo "-- Building with external $scriptBuild"
        cd $objdir && $scriptBuild
    )
    elif CygbuildIsPythonPackage ; then
        CygbuildCmdBuildPython
    else
        CygbuildCmdBuildMainMakefile
    fi
}



function CygbuildCmdCheckMain()
{
    local id="$0.$FUNCNAME"

    if [ -z "$objdir" ]; then
        echo "$id: [ERROR] variable 'objdir' is empty"
        exit 1
    fi


    (
        cd $objdir &&
        make test | tee ${checkfile} 2>&1
    )
}

function CygbuildCmdCleanMain()
{
    local id="$0.$FUNCNAME"

    echo "-- Running make(1) clean"

    if [ -z "$objdir" ]; then
        echo "$id: [ERROR] variable 'objdir' is empty"
        exit 1
    fi


    local missing

    if  CygbuildIsMakefileTarget "clean" ; then
        echo "--   [WARN] Makefile may not have target 'clean:'" 1>&2
        missing="yes"
    fi


    (
        cd $objdir                         &&
        make distclean clean || \
        {
            echo "--   [FIX] Hm, running rm *.o instead"
            rm -f *.o
        }

    )
}



function CygbuildInstallPackageInfo()
{
    local id="$0.$FUNCNAME"

    local dest=$infodir
    local file
    local done

    for file in $(\
        find $srcdir                              \
        -type d \( -name ".inst" -name ".build" \) -prune   \
        -o -name "*.info"                                   \
        -o -name "*.info-*"                                 \
        | sort                                              \
        )

    do

        if [ -z "$done" ]; then
            mkdir -p $infodir || return 1
            done=1
            echo "--   Installing info to $dest"
        fi

        if [ -f $file ]; then
            $installScript $installFmodes $file $dest
        fi
    done


    # Package must not supply central 'dir' file

    file=$infodir/dir

    if [ -f $file ] ; then
        rm $file
    fi


    return 0
}

function CygbuildInstallPackageDocs()
{
    local id="$0.$FUNCNAME"


    local dest=$docdir
    local done
    local file

    for file in $srcdir/[A-Z][A-Z][A-Z]* \
                $srcdir/changelog        \
                $srcdir/ChangeLog        \
                $srcdir/*html            \
                $srcdir/*txt

    do

        #   BCC_MAKEFILE, WCC_MAKEFILE => drop these

        [[ $file == *MAKEFILE* ]]   && continue
        [[ $file == *Makefile* ]]   && continue
        [[ $file == *CVS* ]]        && continue
        [[ $file == *RCS* ]]        && continue
        [[ $file == *SVN* ]]        && continue

        if [ -f $file ] ; then         # Not interested in directories

            if [ -z "$done" ]; then
                $installScript -d $dest || return $?
                done=1
                echo "--   Installing docs to $dest"
            fi

            $installScript $installFmodes $file $dest
        fi
    done


    # Install whole doc/ directory

    local dir=$(CygbuildDetermineDocDir $srcdir)

    if [ ! -z "$dir" ]; then

        #   Are there any files in it?

        ls $dir/* > /dev/null 2>&1


        if [ $? = 0 ]; then

            $installScript -d $dest || return $?

            (
                cd $dir

                #   Remove manual pages, there are already installed in
                #   man/manN/


                tar $CYGBUILD_TAR_INSTALL_EXCLUDE \
                    $verbose --create --file=- *    \
                | ( cd $dest  ; tar --extract --file=- )


            )

            local modes=${installFmodes##*[!0-9]}   # remove "-m" from "-m 644"

            chmod -R $modes $dest  || return $?
        fi
    fi



}


function CygbuildInstallCygwinPart()
{
    local id="$0.$FUNCNAME"

    dest=$cygdoc
    local file=$(CygbuildDetermineReadmeFile)

    if [ -z "$file" ]; then
        echo "$id: [ERROR] $cygpatch/README file is missing"
        return 1
    else
        mkdir -p $dest || return 1
        echo "--   Installing [Cygwin] docs $cygdoc"
        $installScript $installFmodes $file $dest/$PKG-$VER.README
    fi


    file=$cygpostinstall
    dest=${instdir}$sysconfdir/postinstall

    if CygbuildFileExists $file ; then
        echo "--   Installing [Cygwin] postinstall $dest"
        $installScript -D $installXmodes $file $dest/$PKG.sh
    fi
}

function CygbuildCmdInstallMain()
{
    local id="$0.$FUNCNAME"


    if [ ! -d $objdir ]; then
        echo "$id: objdir $objdir does not exist"
        return 1;
    fi

    local saved=$(pwd)
    cd $objdir || return 1


    if [ -z "$instdir_relative" ]; then
        echo "$id: [ERROR] instdir_relative is empty";
        exit 1
    fi

    if [ -d $instdir_relative ]; then

        #  -rf is too dangerous to run without check

        if [[ $instdir_relative == *.inst* ]]; then
            rm -rf $instdir_relative/*
        else
            echo "$id: [ERROR] Suspicious instdir_relative [$instdir_relative]"
            exit 1
        fi

    fi


    CygbuildInstallPackageDocs   &&
    CygbuildInstallPackageInfo   &&
    CygbuildInstallCygwinPart



    if CygbuildFileExists $scriptInstall ; then
        mkdir -p $instdir
        echo "--   Installing with external $scriptInstall $instdir_relative"
        echo "--    See also: find .inst/ -print"

        $scriptInstall "$instdir_relative"

        if [ $? -ne 0  ]; then
            local stat=$?
            echo "$id: [ERROR] Failed to run $scriptInstall $instdir_relative"
            return $stat
        fi

    else
        echo "-- Running install to $instdir"
        CygbuildMakefileRunInstall || return $?
    fi

    if CygbuildFileExists $scriptInstallAfter ; then
        echo "--   Running post-install " \
             "$scriptInstallAfter $instdir_relative"
        $scriptInstallAfter "$instdir_relative"
    fi

    if  [ -z "$srcdir" ] || [ ! -d $srcdir ]; then
        echo "$id: [FATAL] srcdir not found" 1>&2
        return 1
    fi

    cd $saved

}


function CygbuildCmdPostInstall()
{
    if CygbuildFileExists $cygpostinstall ; then
        $cygpostinstall $instdir
    fi
}



function CygbuildCmdStripMain()
{
    local id="$0.$FUNCNAME"


    if  [ -z "$instdir" ] || [ ! -d $instdir ]; then
        echo "$id: instdir [$instdir] not found"
        return 1
    fi

    local done
    local file

    for file in $(find $instdir \
                  -name "*.dll" \
                  -o -name "*.exe" \
                  -print )
    do
        if [ -z "$done" ]; then
            echo "-- Stripping *.exe and *.dll"
            done=1
        fi

        if [ ! -z "$verbose" ]; then
            echo "   Stripping $file"
        fi

        strip $file # > /dev/null 2>&1
    done



}



function CygbuildStripCheck()
{
    local id="$0.$FUNCNAME"

    local file=$(find $instdir -type f -name "*.exe" -o -name "*dll" | head -1)

    if [ -z "$file" ]; then

        file=$(find $instdir -type f -name "*.a" -o -name "*.la" )

        if [ -z "$file" ]; then
            file=
            echo "-- Hm, no *.exe or *.dll files, skipping strip."
            return 0
        fi


        file=$(find $instdir -type f \
            -name "*.py"             \
            -o -name "*.pl"          \
            -o -name "*.sh"          \
            )

        if [ -z "$file" ]; then
            file=
            echo "-- Hm, looks like a library .a or .la package," \
                 " skipping strip."
            return 0
        else
            echo "$id: [ERROR] No *.exe, *.dll files found at $instdir."
            echo "$id: [ERROR] Is this a script  See -x" 1>&2
            return 1
        fi


    fi

    #   If strip has been run, then
    #   nm fil.exe =>  nm: file.exe: no symbols

    local saved="$IFS"
    local IFS=" "

    set -- $(nm $file 2>&1 | head -1)
    IFS="$saved"

    if [ "$3 $4" = "no symbols" ]; then
        return 0
    else
        echo "$id: [WARN] Symbols found. I'm going to run [strip] first" 1>&2
        CygbuildCmdStripMain
    fi

}

function CygbuildCmdFilesWrite()
{

    local id="$0.$FUNCNAME"
    local from="$1"
    local to="$2"

    if [ ! -d $from ]; then
        echo "$id: [ERROR] Template directory does not exist: [$from]" 1>&2
        echo "$id: it should be at /etc/cygbuild/template if you installed cygbuild package. See Web page for download." 1>&2
        return 1
    fi

    if [ ! -d $to ]; then
        echo "$id: [ERROR] Write directory does not exist: $to" 1>&2
        return 1
    fi


    echo "-- Writing default files under $to"


    local file=$PKG.README

    if [ ! -f $to/$file ]; then
        cp $verbose $from/*README $to/$file || return $?
    fi

    file=setup.hint

    if [ ! -f $to/$file ]; then
        cp $verbose $from/$file $to || return $?
    fi


    for file in $from/*.tmp
    do
        if [ ! -f $file ]; then
            echo "$id: [ERROR] No such file $file"  1>&2
            return 1
        fi

        local sh=${file##*/}      # /path/to/file.sh.tmp => file.sh.tmp
        script=${sh%.tmp}           # file.sh.tmp => file.sh

        local dest=$to/$script

        if [ -f $dest ]; then
            [ ! -z "$verbose" ] && echo "--    Skip, already exists $dest"
        else
            cp $verbose $file $to || return $?
        fi
    done
}


function CygbuildCmdFilesMain()
{

    local id="$0.$FUNCNAME"
    local file

    local templatedir=$(CygbuildTemplatePath)
    local destdir=$cygpatch

    if [ -z "$templatedir" ]; then
        echo "$id [ERROR] variable 'templatedir' is empty" 1>&2
        return 1
    fi


    if [ -z "$destdir" ]; then
        echo "$id [ERROR] variable 'destdir' is empty" 1>&2
        return 1
    fi


    if [ ! -d $destdir ]; then
        echo "$id: [WARN] Hm, forgot to run 'mkdirs'? " \
             "Running it for you now." 1>&2
        CygbuildCmdMkdirs  || return 1
    fi

    CygbuildCmdFilesWrite $templatedir $destdir


}

function CygbuildCmdFinishMain()
{
    # Too dangerous, commented out
    # rm -rf $objdir
    :
}

function CygbuildCmdVerifyMain()
{
    local id="$0.$FUNCNAME"
    local script=$src_pkg_build_script
    local file

    if [ -f $TOPDIR/$src_pkg_name ]; then
        file=$TOPDIR/$src_pkg_name
    elif [ ! -z "$REL" ]; then
        local file=$(ls $TOPDIR/$PKG*$VER-$REL-src.tar.bz2)
    else
        local -a arr=$(ls $TOPDIR/$PKG*$REL-src.tar.bz2)
        local count=${#arr[*]}

        if [ $count -eq 1 ]; then
            file=${arr[0]}
        fi
    fi


    if [ ! -f $file ]; then
        echo "$id: [WARN] Original source package '$file' missing " \
             "or RELEASE unknown." 1>&2
        echo "$id: [WARN] See manual and options -f or -r" 1>&2
        return 1
    fi

    if [ -z "$builddir" ]; then
        echo "$id: [ERROR] variable 'builddir' is empty" 1>&2
        exit 1
    fi

    local name=${file%%*/}
    local dir=$builddir/tmp/verify
    local z=$(CygbuildTarOptionCompress  $file)
    local opt="$verbose -${z}xf"

    (
       set -- ''

        mkdir -p $dir   &&
        cd $dir         &&
        echo "--    cd $dir"
        rm -rf *        &&
        cp $file .      &&
        echo "--    Unpacking $name ..."
        tar $opt $name  &&
        echo "--    Building ..."

        cat<<EOF

***********************************************************************
VERIFYING source-package
$dir
***********************************************************************

EOF
        echo "--    Running ./$script with option 'all'."

        if [ -z "$OPTION_DEBUG_VERIFY" ]; then
            ./$script all
        else
            bash -x ./$script all
        fi
    )


}


#######################################################################
#
#       Guess functions: so that -f or -r need not be supplied
#
#######################################################################

function CygbuildFilePackageGuessFromDirectory()
{
    local id="$0.$FUNCNAME"
    local dir=$(pwd)
    local ret

    if CygbuildDefineVersionVariables $dir ; then

        ret=$STATIC_VER_PACKAGE-$STATIC_VER_VERSION

        #  Directory looks like package-NN.NN/ add RELEASE

        if [ ! -z $OPTION_RELEASE ]; then
            ret=$ret-$OPTION_RELEASE
        fi
    fi

    dummy="$id: RETURN"         # Will show in debugger

    echo $ret
}


function CygbuildFilePackageGuessMain()
{
    #  DEVEL: This function must not 'echo' a thing. The values
    #  are stored to valiables from this function.

    local id="$0.$FUNCNAME"
    local script=$(CygbuildBuildScriptPath)

    #   If there is only one PACKAGE in current directory, make an educated
    #   guess and use that in case user did not supply -f
    #
    #   We expect to see these files:
    #
    #       package-NN.NN-1.patch
    #       package-NN-NN.tar.gz
    #
    #   If they are not there and there is only this file.
    #   then user forgot to unpack the file.
    #
    #       package-NN.NN-RELEASE-src.tar.gz
    #
    #   Finally, if there is a script that gets sources from external
    #   source, run it.

    local ret
    local -a arr

    #  The SRC package is not counted in, because that would
    #  be the Cygwin Net release source package. We need to find out
    #  the original developer package.

    arr=( $(ls | \
            egrep -ve '-src.tar.bz2' | \
            egrep '[0-9]\.[0-9.].(tar.gz|tgz)' \
            2> /dev/null ))


    local dummy="${arr[*]}"          # For bash debug only, what we got?
    local len=${#arr[*]}

    if [ $len -eq 0 ]; then
        #   If no files were found, it means, that we cannot find
        #   the developer's original package.

        arr=( $(ls | \
                egrep '[0-9]\.[0-9.]+-[0-9].*(tar.bz2)' \
                2> /dev/null ))

    fi


    #   Check if all files have same first work: PACKAGENAME, if not
    #   we don't know what package use wants to use

    if [ $len -gt 1 ]; then

        local tmp=${arr[0]}     # package-nn.nn.tar.gz
        local word=${tmp%%-*}   # package
        local element fail

        for element in ${arr[*]}
        do
            if [[ $element != $word* ]]; then
                echo "--    [WARN] Different tar files. $word <> $element" \
                     "      [WARN] consider using -f FILE option." 2>&1
                fail=1
                break
            fi
        done

        if [ -z $fail ]; then
            ret=${arr[0]}
        fi

    elif [ $len -eq 1 ]; then     # Fresh and empty dir. Good. One tar file.

        ret=${arr[0]}

    else

        #  No tar files around to guess, try if this directory holds
        #  package name and user is currently porting a package

        ret=$(CygbuildFilePackageGuessFromDirectory)
    fi

    echo $ret
}

function CygbuildFileReleaseGuess()
{
    local id="$0.$FUNCNAME"
    local ret

    #   Debian source packages are in format
    #   package_NN.NN.orig.tar.gz

    let -a arr

    arr=( $(ls  2> /dev/null | \
            egrep -e '[-_][0-9]+(-src\.tar|\.orig\.tar|\.patch)' ) )

    local count=${#arr[*]}

    if [ $count -eq 1 ]; then
        ret=${arr[0]}
    elif [ $count -eq 2 ]; then

        #  Found exactly two, source and binary package. Pick source
        #  package-NN.NN-RELEASE-src.tar.bz2
        #  package-NN.NN-RELEASE.tar.bz2

        ret=$(echo "${arr[*]}" | tr ' ' '\n' | egrep -e '-src|\.orig.' )
    fi

    echo $ret

}



#######################################################################
#
#       Main
#
#######################################################################


function CygbuildCommandMain()
{
    local id="$0.$FUNCNAME"

    if [ "$1" = "-h" ]; then
        CygbuildHelpShort
        exit 1
    elif [ "$1" = "--help" ]; then
        CygbuildHelpLong
        exit 1
    fi

    # ................................................. read options ...

    local arg
    local args
    local dir
    local quiet

    local release
    local package

    OPTIND=1

    unset verbose
    local stripFlag="yes"
    export OPTION_FILE=
    export OPTION_PREFIX=
    export OPTION_RELEASE=


    # #todo: check if this can take long options names
    # -l file:,Prefix,release:,Version,verbose  \

    while getopts \
          "Dd:r:f:PvVx" arg $*
    do
      case $arg in

            f|file)

                export OPTION_FILE="$OPTARG"
                package="$OPTARG"
                package=$(CygbuildStrRemoveExt "$package")
                ;;

            D|DEBUG)

                export OPTION_DEBUG_VERIFY="yes"
                ;;

            r|release)

                if [ "$OPTARG" = "date" ]; then
                    release=$(CygbuildDate)
                else
                    release="$OPTARG"
                fi

                if ! echo $release | grep -e '^[0-9]*$' ; then
                    echo "$id: [ERROR] release must all be numeric. " \
                         "Got [$release]"  1>&2
                    exit
                fi


                export OPTION_RELEASE=$release
                ;;

            P|Prefix)

                export OPTION_PREFIX="yes";
                ;;

            v|verbose)

                export verbose="--verbose"
                ;;

            V|Version)

                echo $CYGBUILD_VERSION
                return
                ;;

            x)  stripFlag=
                ;;

      esac
    done

    #  Remove options

    shift $((OPTIND - 1))


    echo "-- $CYGBUILD_PROGRAM $CYGBUILD_HOMEPAGE_URL"
    echo "-- Wait, examining environment and preparing variables"

    #  See if user supplied the RELEASE. This can be implicit in the
    #  package name, in which case it's ok. Otherwise user have to
    #  explicitly give it. Either way, it is important, otherwise the build
    #  directories cannot be determined correctly

    export packageGuess=

    if [ -z "$package" ]; then
        export packageGuess=$(CygbuildFilePackageGuessMain)
        package=$packageGuess
    fi



    local releaseGuess

    if [ -z "$release" ]; then
        releaseGuess=$(CygbuildFileReleaseGuess)
        release=$releaseGuess
    fi


    if [ $# -lt 1 ]; then
        CygbuildHelpShort
        echo "$id: [ERROR] COMMAND is missing." 1>&2
        exit 1
    fi

    # ................................................ set variables ...


    tdir=$(pwd)

    local -a array=( $(CygbuildSrcDirLocation $tdir) )

    local top=${array[0]}
    local src=${array[1]}


    CygbuildDefineGlobalsRoot           \
        "$top" "$src" "$release"        \
        "$package" "noCheckRelease"

    #  If we don't know where source dir is, then we can do nothing.
    #  This is our basic compass bearing

    CygbuildSrcDirCheck "$srcdir"

    local arg_count=$#
    local STATUS=0

    # ............................................... other commands ...

    if [ "$1" != 'all'  ]; then

        #  option 'all' run all steps, so ignore check

        if [ ! -d $srcdir ]; then
            echo "$id: [ERROR] No directory $srcdir"                      \
                 "$id: [ERROR] Are you inside directory package-NN.NN/ ?" \
                 "$id: [ERROR] Perhasps source kit is not unpacked."
            exit 1
        fi
    fi

    #   RELEASE version must be known because som commands below expect it
    #   That's why two calls to CygbuildDefineGlobalsRoot


    if [[ "$*"  == *package*  || "$*"  == *pkg* || "$*"  == *spkg*  ]]; then

        local tmprel

        if [ ! -z "$package" ]; then

            #  This var is not used anywhere, it's a check if we can parse the
            #  release from package name NN.NN-RELEASE

            tmprel=$(CygbuildStrRelease $package)
        fi

        if [ -z "$release" ] && [ -z "$tmprel" ]; then
            #   User didn't supply -r, and we cannot parse release from -f NAME

            echo "$id: [ERROR] -r RELEASE option must be given." \
                  "$id: [ERROR] See also -f SRC-FILE.tar.gz" 1>&2
            exit 1
        fi

        CygbuildDefineGlobalsRoot "$top" "$src" "$release" "$package"
    fi



    for opt in $*
    do
        case $opt in

          check)                CygbuildCmdCheckMain
                                STATUS=$?
                                ;;

          configure|conf)       CygbuildCmdConfMain
                                STATUS=$?
                                ;;

          clean)                CygbuildCmdCleanMain
                                STATUS=$?
                                ;;


          make)                 CygbuildCmdBuildMain
                                STATUS=$?
                                CygbuildNoticeMaybe
                                ;;
          finish)               CygbuildCmdFinishMain
                                STATUS=$?
                                ;;

          files)                CygbuildCmdFilesMain
                                STATUS=$?
                                ;;

          install)              CygbuildCmdInstallMain
                                STATUS=$?
                                ;;

          iprep)                InstallPrepare
                                STATUS=$?
                                ;;


          makedirs|mkdirs)      CygbuildCmdMkdirs
                                STATUS=$?
                                ;;

          makepatch|mkpatch)    CygbuildCmdMkpatch
                                STATUS=$?
                                ;;

          package|pkg)          if [ $stripFlag ]; then
                                    CygbuildStripCheck &&
                                    CygbuildCmdPkgBinaryMain
                                    STATUS=$?
                                else
                                    CygbuildCmdPkgBinaryMain
                                    STATUS=$?
                                fi
                                ;;

          package-devel|pkg-dev*)
                                if [ $stripFlag ]; then
                                    CygbuildStripCheck      &&
                                    CygbuildCmdPkgDevelMain
                                    STATUS=$?
                                else
                                    CygbuildCmdPkgDevelMain
                                    STATUS=$?
                                fi
                                ;;

          prep|prepare)         CygbuildCmdPrepMain
                                STATUS=$?
                                ;;

          publish|pub)          CygbuildCmdPublishMain
                                STATUS=$?
                                ;;

          source-package|package-source|spkg)

                                CygbuildCmdPkgSourceMain
                                STATUS=$?
                                ;;

          source-package-verify|spkgv)
                                CygbuildCmdVerifyMain
                                STATUS=$?
                                ;;

          package-all)
                                CygbuildCmdInstallMain     &&
                                CygbuildCmdPkgBinaryMain   &&
                                CygbuildCmdReadmeFix       &&
                                CygbuildCmdPkgBinaryMain   &&
                                CygbuildCmdPkgSourceMain   &&
                                CygbuildCmdPublishMain
                                STATUS=$?
                                echo "-- See also command  [readmefix]"
                                ;;

          postinstall)          CygbuildCmdPostInstall
                                STATUS=$?
                                ;;

          readme)               CygbuildDocFileReadme
                                STATUS=$?
                                ;;

          readmefix)            CygbuildCmdReadmeFix
                                STATUS=$?
                                ;;


          strip)                if [ $stripFlag ]; then
                                    CygbuildCmdStripMain
                                    STATUS=$?
                                else
                                    STATUS=0
                                fi
                                ;;


          all)  CygbuildCmdPrepMain         &&
                CygbuildCmdConfMain         &&
                CygbuildCmdBuildMain        &&
                CygbuildCmdInstallMain      &&
                CygbuildCmdStripMain        &&
                CygbuildCmdPkgBinaryMain    &&
                CygbuildCmdPkgSourceMain    &&
                CygbuildCmdFinishMain

                STATUS=$?

                [ $STATUS -eq 0 ] && CygbuildHelpSourcePackage
                ;;

          vars) set -x
                CygbuildDefineGlobalsRoot "$TOPDIR" "$srcdir" \
                    "$release" "$package"
                return
                ;;


          *)    echo "$id: [ERROR] bad argument [$opt]. See --help" 1>&2
                exit 1
                ;;

        esac


        if [ $STATUS -ne 0 ]; then
            echo "$id: [FATAL] status is $STATUS." 1>&2
            exit $STATUS
        fi

    done

}

function CygbuildMain ()
{

    #   This file can be included as a bash library, so don't run Main
    #   unless we're called with an argument.
    #
    #       #!/bin/bash
    #       ... load library
    #       export CYGBUILD_LIB=1
    #       source $(/bin/which cygbuild.sh)
    #       ... call functions

    if [ $# -gt 0 ]; then

        CygbuildCommandMain $*

    elif [ ! -z "$CYGBUILD_LIB" ]; then

        #   Externally called custom scripts may want to call back to us
        #   referring to these functions

        export -f CygbuildMakeRunInstallFixPerl
        export -f CygbuildPostinstallWrite
        export -f CygbuildVersionInfo
        export -f CygbuildDetermineReadmeFile
        export -f CygbuildLibInstallEnvironment

    else
        CygbuildHelpShort
    fi
}

CygbuildMain $*

# End of file