#!/bin/sh
#
# Copyright (C) 1997-2002 Vladimir Makarov.
# 
# Written by Vladimir Makarov <vmakarov@users.sourceforge.net>
# 
# This file is part of interpreter of DINO.
# 
# This is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# This software is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU CC; see the file COPYING.  If not, write to the Free
# Software Foundation, 59 Temple Place - Suite 330, Boston, MA
# 02111-1307, USA.
#
# Script for compare Dino and perl, tclsh, python, awk
# The most scripts are taken from The Great Computer Language Shootout
# (http://www.bagley.org/~doug/shootout)
#

DINO=./dino
srcdir=`dirname $0`
ftest=__ftest.d
input=__input
temp=__temp
temp2=__temp2

if test x`echo -n` != "x-n";then
  NECHO="echo -n"
else
  NECHO=
fi

cat <<EOF >$ftest
EOF

if perl $ftest 2>/dev/null;then
  PERL=perl
  echo '>>>> ' `perl -v 2>&1|fgrep 'This is'`
else
  echo We have no perl
  PERL=
fi

if python $ftest 2>/dev/null;then
  PYTHON=python
  echo '>>>> ' `echo|python -v 2>&1|fgrep Python`
else
  echo We have no python
  PYTHON=
fi

if tclsh $ftest 2>/dev/null;then
  TCLSH=tclsh
  echo '>>>> ' TCL version is `echo 'puts $tcl_version'|tclsh`
else
  echo We have no tclsh
  TCLSH=
fi

if awk -f $ftest 2>/dev/null </dev/null;then
  AWK=awk
  echo '>>>> ' `echo|awk --version 2>&1|fgrep wk`
else
  echo We have no awk
  AWK=
fi

echo '>>>> ' dino: `$DINO 2>&1|fgrep Version`

if test x$1 = x
then
    start_test_number=1
else
    if expr $1 + 0 >/dev/null
    then
        start_test_number=`expr $1 + 0`
    else
        echo $script_file:invalid argument $1 2>&1
        exit 1
    fi
fi

if test x$2 != x; then
  PERL=
  PYTHON=
  TCLSH=
  AWK=
fi

# Test 1.
if test $start_test_number -le 1; then

######################################################
echo 
echo '+++++ Test #1 ackermann (good test for recursive functions N=7):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

use integer;

# It's prettier but slower to do this
#sub Ack {
#    my($M, $N) = @_;
#    return( $N + 1 )         if ($M == 0);
#    return( Ack($M - 1, 1) ) if ($N == 0);
#    Ack($M - 1, Ack($M, $N - 1));
#}

# in our quest for speed, we must get ugly:
# it helps reduce stack frame size a little bit
# from Leif Stensson
sub Ack {
    return $_[0] ? ($_[1] ? Ack($_[0]-1, Ack($_[0], $_[1]-1))
		    : Ack($_[0]-1, 1))
	: $_[1]+1;
}

my $NUM = $ARGV[0];
$NUM = 1 if ($NUM < 1);
my $ack = Ack(3, $NUM);
print "Ack(3,$NUM): $ack\n";
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 7 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# from Brad Knotwell
import sys

def Ack(M, N):
    if (not M):
        return( N + 1 )
    if (not N):
        return( Ack(M-1, 1) )
    return( Ack(M-1, Ack(M, N-1)) )

def main():
    NUM = int(sys.argv[1])
    print "Ack(3,%d): %d" % (NUM, Ack(3, NUM))

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 7 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

echo
echo  Tcl has stack overflow on the test
echo

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

function ack(m, n) {
    if (m == 0) return( n + 1 );
    if (n == 0) return( ack(m - 1, 1) );
    return( ack(m - 1, ack(m, (n - 1))) );
}

BEGIN {
    n = (ARGV[1] < 1) ? 1 : ARGV[1];
    printf("Ack(3,%d): %d\n", n, ack(3, n));
    exit;
}
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest 7 </dev/null >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
func ack (m, n) {
    if (m == 0) return n + 1;
    if (n == 0) return ack (m - 1, 1);
    return ack (m - 1, ack (m, (n - 1)));
}

var n = int (argv [0] < 1 ? 1 : argv [0]);
putln ("Ack(3,", n, "): ", ack (3, n));
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 7 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 2.
if test $start_test_number -le 2; then

######################################################
echo 
echo '+++++ Test #2: Array access (N=700):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

# this program is modified from:
#   http://cm.bell-labs.com/cm/cs/who/bwk/interps/pap.html
# Timing Trials, or, the Trials of Timing: Experiments with Scripting
# and User-Interface Languages</a> by Brian W. Kernighan and
# Christopher J. Van Wyk.

my $n = @ARGV[0] || 1;
my @X;
my @Y;

my $last = $n - 1;
for my $i (0..$last) {
    $X[$i] = $i + 1;
}
for my $k (0..999) {
    for my $i (reverse 0..$last) {
	$Y[$i] += $X[$i];
    }
}

print "$Y[0] $Y[$last]\n";
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 700 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# with help from Brad Knotwell

import sys

def main():
    n = int(sys.argv[1])
    x = n * [0]       
    y = n * [0]
    for i in xrange(0,n):
        x[i] = i + 1
    for k in xrange(0,1000):
        for i in xrange(n-1,-1,-1):
            y[i] = y[i] + x[i]
    print y[0], y[-1]

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 700 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

# this program is modified from:
#   http://cm.bell-labs.com/cm/cs/who/bwk/interps/pap.html
# Timing Trials, or, the Trials of Timing: Experiments with Scripting
# and User-Interface Languages</a> by Brian W. Kernighan and
# Christopher J. Van Wyk.

proc main {} {
    global argv
    set n [lindex $argv 0]
    set last [expr {$n - 1}]
    for {set i 0} {$i < $n} {incr i} {
	set x($i) [expr {$i + 1}]
	set y($i) 0
    }
    for {set k 0} {$k < 1000} {incr k} {
	for {set j $last} {$j >= 0} {incr j -1} {
	    set y($j) [expr {$x($j) + $y($j)}]
	}
    }
    puts "$y(0) $y($last)"
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 700 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

# this program modified from:
#   http://cm.bell-labs.com/cm/cs/who/bwk/interps/pap.html
# Timing Trials, or, the Trials of Timing: Experiments with Scripting
# and User-Interface Languages</a> by Brian W. Kernighan and
# Christopher J. Van Wyk.

BEGIN {
    n = (ARGV[1] < 1) ? 1 : ARGV[1];

    for (i = 0; i < n; i++)
	x[i] = i + 1
    for (k = 0; k < 1000; k++) {
	for (j = n-1; j >= 0; j--)
	    y[j] += x[j]
    }

    print y[0], y[n-1]
}
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest 700 </dev/null >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
var i, j, k, n = int (argv [0] < 1 ? 1 : argv [0]);
var x = [n:0], y = [n:0];

for (i = 0; i < n; i++)
  x [i] = i + 1;
for (k = 0; k < 1000; k++) {
  for (j = n - 1; j >= 0; j--)
    y [j] += x [j];
}

putln (y [0], " ", y [n - 1]);
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 700 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 3.
if test $start_test_number -le 3; then

######################################################
echo 
echo '+++++ Test #3:  Count lines/words/chars (N=2000):  +++++'

cat <<'EOF' >$temp
Subject:      Re: Who was Izchak Miller?
From:         "Jane D. Anonymous" <nobody@yale.edu>
Date:         1996/04/28
Message-Id:   <4lv7bc$oh@news.ycc.yale.edu>
References:   <317C405E.5DFA@panix.com> <4lk6vl$gde@ns.oar.net>
To:           75176.2330@compuserve.com
Content-Type: text/plain; charset=us-ascii
Organization: Yale University
X-Url:        news:4lk6vl$gde@ns.oar.net
Mime-Version: 1.0
Newsgroups:   rec.games.roguelike.nethack
X-Mailer:     Mozilla 1.1N (Macintosh; I; 68K)

Hello there, Izchak Miller was my father.  When I was younger I spent 
many a night, hunched over the keyboard with a cup of tea, playing 
nethack with him and my brother.  my dad was a philosopher with a strong 
weakness for fantasy/sci fi.  I remember when he started to get involved 
with the Nethack team- my brother's Dungeons and Dragons monster book 
found a regular place beside my dad's desk. it's nice to see him living 
on in the game he loved so much :-).  
                                                                  Tamar Miller

The following is a really long word of 5000 characters:

wwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwwww
EOF

rm -f $input; touch $input

i=0
while test $i -lt 2000;do cat $temp >>$input; i=`expr $i + 1`;done

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

# this program is modified from:
#   http://cm.bell-labs.com/cm/cs/who/bwk/interps/pap.html
# Timing Trials, or, the Trials of Timing: Experiments with Scripting
# and User-Interface Languages</a> by Brian W. Kernighan and
# Christopher J. Van Wyk.

use strict;

my($nl, $nw, $nc);
while (read(STDIN, $_, 4095)) {
    $_ .= <STDIN>;
    $nl += scalar(split(/\n/));
    $nc += length;
    $nw += scalar(split);
}
print "$nl $nw $nc\n";
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# with help from Brad Knotwell

import sys

def main():
    nl = nw = nc = 0
    rl = sys.stdin.readlines
    lines = rl(4096)

    while lines:
        nl += len(lines)
        for line in lines:
            nc += len(line)
            nw += len(line.split())
        lines = rl(4096)

    print "%d %d %d" % (nl, nw, nc)

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

# this program is modified from:
#   http://cm.bell-labs.com/cm/cs/who/bwk/interps/pap.html
# Timing Trials, or, the Trials of Timing: Experiments with Scripting
# and User-Interface Languages</a> by Brian W. Kernighan and
# Christopher J. Van Wyk.

# Modified by Miguel Sofer

proc main {} {
    set nl 0
    set nc 0
    set nw 0

    while {1} {
	set data [read stdin 4096]
	if {![string length $data]} {break}
	if {[gets stdin extra] >= 0} {
	    append data $extra
	    incr nc
	}
	incr nc [string length $data]
	incr nw [regexp -all {(?:^|\s)\S} $data]
	incr nl [regexp -all -line {^} $data]
    }
    puts "$nl $nw $nc"
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

# this program modified from:
#   http://cm.bell-labs.com/cm/cs/who/bwk/interps/pap.html
# Timing Trials, or, the Trials of Timing: Experiments with Scripting
# and User-Interface Languages</a> by Brian W. Kernighan and
# Christopher J. Van Wyk.

# this version is a little more efficient than the original via
# use of NR

BEGIN { delete ARGV }
{
    nc += length($0) + 1
    nw += NF
}
END { print NR, nw, nc }
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
var ln, v, nc = 0, nw = 0, nl = 0, l;

try {
  for (;;) {
    ln = getln ();
    nc += #ln + 1;
    v = split (ln);
    l = #v;
    nw += (#v [l - 1] == 0 ? l - 1 : l);
    nl++;
  }
} catch (invcalls.eof) {
  putln (nl, " ", nw, " ", nc);
  exit (0);
}
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 4.
if test $start_test_number -le 4; then

######################################################
echo 
echo '+++++ Test #4: Echo client/server (N=50000):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

use Socket;

my $DATA = "Hello there sailor\n";

sub server_sock {
    local *SS;
    socket(SS, PF_INET, SOCK_STREAM, 0) or
	die "server/socket ($!)";
    setsockopt(SS, SOL_SOCKET, SO_REUSEADDR, pack("l", 1)) or
	die "server/setsockopt ($!)";
    bind(SS, sockaddr_in(0, INADDR_LOOPBACK)) or
	die "server/bind ($!)";
    listen(SS, 2);
    return(*SS);
}

sub get_port {
    local *SK = shift;
    (sockaddr_in(getsockname(SK)))[0];
}

sub client_sock {
    my $port = shift;
    local *CS;
    socket(CS, PF_INET, SOCK_STREAM, getprotobyname('tcp')) or
	die "client/socket ($!)";
    connect(CS, sockaddr_in($port, INADDR_LOOPBACK)) or
	die "client/connect ($!)";
    return(*CS);
}

sub echo_client {
    my($N, $port) = @_;
    local *SOCK = client_sock($port);
    select(SOCK);
    $| = 1;
    for my $i (0..($N-1)) {
	print $DATA;
	my $ans = <SOCK>;
	($ans eq $DATA) or die qq{client: "$DATA" ne "$ans"};
    }
    close SOCK;
}

sub echo_server {
    my($N) = @_;
    local *SSOCK = server_sock();
    my $port = get_port(*SSOCK);
    my $pid = fork;
    defined $pid or die "server/fork ($!)";
    if ($pid) {
	# parent is server
	local *CSOCK;
	accept(CSOCK, SSOCK) or die "server/accept ($!)";
	select(CSOCK);
	$| = 1;
	my $n = 0;
	while (<CSOCK>) {
	    print $_;
	    $n += length($_);
	}
	select(STDOUT);
	print "server processed $n bytes\n";
    } else {
	# child is client
	echo_client($N, $port);
    }
    wait();
}

sub main {
    my $N = $ARGV[0] || 1;
    echo_server($N);
    exit(0);
}

main();
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 50000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# with help from Brad Knotwell

import sys, os
from socket import *

DATA = "Hello there sailor\n"
bufferSize = len(DATA)

def server_sock():
    sock = socket(AF_INET, SOCK_STREAM)
    sock.setsockopt(SOL_SOCKET, SO_REUSEADDR, 1)
    sock.bind(('127.0.0.1', 0));
    sock.listen(2)
    return(sock)

def get_port(sock):
    host, port = sock.getsockname()
    return(port)

def client_sock(port):
    sock = socket(AF_INET, SOCK_STREAM)
    sock.connect(('127.0.0.1', port))
    return(sock)

def echo_client(n, port):
    sock = client_sock(port)
    sender,receiver = sock.send,sock.recv
    for i in range(0,n):
        sender(DATA)
        ans = receiver(bufferSize)
        while ans[-1] != "\n":
            ans = ans + receiver(bufferSize - len(ans))
        if ans <> DATA:
            raise("client: \"%s\" ne \"%s\"" % (DATA, ans))
    sock.close()

def echo_server(n):
    ssock = server_sock()
    if os.fork() > 0:
        # parent is server
        csock, addr = ssock.accept()
        n = 0
        sender,receiver = csock.send,csock.recv
        while 1:
            dat = receiver(bufferSize)
            if not dat: break
            sender(dat)
            n = n + len(dat)
        print "server processed %d bytes" % n
        os.wait()
    else:
        # child is client
        echo_client(n, get_port(ssock))

def main():
    n = int(sys.argv[1])
    if n < 1:
        n = 1
    echo_server(n)

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 50000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

# from: Kristoffer Lawson
# with help from Miguel Sofer

proc newClient {sock addr port} {
    fconfigure $sock -buffering line
    set r [gets $sock]
    set rLength 0
    while {![eof $sock]} {
	# Extra increase because [gets] doesn't return \n
	incr rLength [expr {[string length $r] + 1}]
	puts $sock $r
	set r [gets $sock]
    }
    puts "server processed $rLength bytes"
    exit
}


proc runClient {n addr port} {
    set sock [socket $addr $port]
    fconfigure $sock -buffering line
    set msg "Hello there sailor"

    incr n
    while {[incr n -1]} {
	puts $sock $msg
	if {[string compare [gets $sock] $msg] == 0} continue
	error "Received different message: $r."
    }
}


set n [lindex $argv 0]

if {[llength $argv] < 2} {
    socket -server newClient -myaddr localhost 10004
    # exec tclsh [info script] $n client &
    vwait forever
} else {
    runClient $n localhost 10004
}
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  $TCLSH $ftest 50000&
  time $TCLSH $ftest 50000 client 2>&1|fgrep user
fi

#
cat <<'EOF' >$ftest
include "socket";
if (#argv < 2) {
  var s, cl, str, l = 0;
  s = sockets.stream_server (10003, 4);
  cl = s.accept ();
  try {
    for (;;) {
      str = cl.read (64); l+= #str; cl.write (str);
    }
  } catch (socket_excepts.eof) {
    putln ("i got ", l, " bytes");
  }
} else {
  var cl, send, rec, i, n = int (argv [0]);
  cl = sockets.stream_client ("localhost", 10003);
  send = "Hello there sailor\n";
  for (i = 0; i < n; i++) {
    cl.write (send);
    rec = cl.read (19);
    if (send != rec) {
      put ("different strings"); println (send, rec);
    }
  }
}
EOF

echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi

  $DINO -I$srcdir -L./ipcerr.so -L./socket.so $ftest 50000&
  time $DINO -I$srcdir -L./ipcerr.so -L./socket.so $ftest 50000 client 2>&1|fgrep user

fi

# Test 5.
if test $start_test_number -le 5; then

######################################################
echo 
echo '+++++ Test #5: Exceptions mechanism (N=100000):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

use integer;

my $HI = 0;
my $LO = 0;
my $NUM = $ARGV[0];
$NUM = 1 if ($NUM < 1);

package Lo_Exception;

sub new {
    bless({Val => shift}, __PACKAGE__);
}

package Hi_Exception;

sub new {
    bless({Val => shift}, __PACKAGE__);
}

package main;

sub some_function {
    my $num = shift;
    eval {
	&hi_function($num);
    };
    if ($@) {
	die "We shouldn't get here ($@)";
    }
}

sub hi_function {
    my $num = shift;
    eval {
	&lo_function($num);
    };
    if (ref($@) eq "Hi_Exception") {
	$HI++;		# handle
    } elsif ($@) {
	die $@;		# rethrow
    }
}

sub lo_function {
    my $num = shift;
    eval {
	&blowup($num);
    };
    if (ref($@) eq "Lo_Exception") {
	$LO++;		# handle
    } elsif ($@) {
	die $@;		# rethrow
    }
}

sub blowup {
    my $num = shift;
    if ($num % 2) {
	die Lo_Exception->new(Num => $num);
    } else {
	die Hi_Exception->new(Num => $num);
    }
}

$NUM = $ARGV[0];
while ($NUM--) {
    &some_function($NUM);
}
print "Exceptions: HI=$HI / LO=$LO\n";
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 100000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

import sys

HI = 0
LO = 0


class Hi_exception:
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return `self.value`


class Lo_exception:
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return `self.value`


def some_function(num):
    try:
	hi_function(num)
    except:
	raise "We shouldn't get here (%s)" % sys.exc_info()[0]


def hi_function(num):
    global HI
    try:
	lo_function(num)
    except Hi_exception, ex:
	HI = HI + 1
	#print 'Hi_exception occurred, value:', ex.value


def lo_function(num):
    global LO
    try:
        blowup(num)
    except Lo_exception, ex:
	LO = LO + 1
        #print 'Lo_exception occurred, value:', ex.value


def blowup(num): 
    raise (((num & 1) and Lo_exception) or Hi_exception)(num)

def main():
    global LO, HI
    NUM = int(sys.argv[1])
    if NUM < 1:
        NUM = 1
    for i in xrange(NUM-1,-1,-1):
	some_function(i)
    print "Exceptions: HI=%d / LO=%d" % (HI, LO)


main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 100000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# with help from Kristoffer Lawson
# modified by Miguel Sofer

set HI 0
set LO 0

proc some_function {num} {
    if {[catch {hi_function $num} result]} {
        puts stderr "We shouldn't get here ($result)"
    }
}

proc hi_function {num} {
    if {[set exc [catch {lo_function $num}]] == 11} {
        # handle
	incr ::HI
    } else {
        # rethrow
	return -code $exc
    }
}

proc lo_function {num} {
    if {[set exc [catch {blowup $num}]] == 10} {
        # handle
	incr ::LO
    } else {
        # rethrow
	return -code $exc
    }
}

proc blowup {num} {
    if {$num % 2} {
        #error "Lo_exception"
	return -code 10
    } else {
        #error "Hi_exception"
	return -code 11
    }
}

proc main {} {
    global argv HI LO
    set NUM [lindex $argv 0]
    if {$NUM < 1} {
        set NUM 1
    }
    incr NUM
    while {[incr NUM -1]} {
        some_function $NUM
    }
    puts "Exceptions: HI=$HI / LO=$LO"
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 100000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
var HI = 0;
var LO = 0;

ext except {
  class Hi_exception (value) {}
  class Lo_exception (value) {}
}

func some_function (num) {
  try { 
    hi_function (num);
  } catch (except) {
    fputln (stderr, "We shouldn't get here (", e.value , ")");
  }
}

func hi_function (num) {
  try {
    lo_function (num);
  } catch (excepts.Hi_exception) {
    HI++;
    //    putln ("Hi_exception occurred, value:", e.value);
  }
}

func lo_function (num) {
  try {
    blowup (num);
  } catch (excepts.Lo_exception) {
    LO++;
    //    putln ("Lo_exception occurred, value:", e.value);
  }
}


func blowup (num) { 
  throw ((num & 1) ? excepts.Lo_exception (num) : excepts.Hi_exception (num));
}

func main () {
  var i, num = int (argv [0]);

  if (num < 1) num = 1;
  for (i = num - 1; i >= 0; i--)
    some_function (i);
  putln ("Exceptions: HI=", HI, " / LO=", LO);
}

main ();
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 100000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 6.
if test $start_test_number -le 6; then

######################################################
echo 
echo '+++++ Test 6: Fibonacci Numbers (N=28):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/perl
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

use strict;
use integer;

# from Leif Stensson
sub fib {
    return $_[0] < 2 ? 1 : fib($_[0]-2) + fib($_[0]-1);
}

my $N = ($ARGV[0] < 1) ? 1 : $ARGV[0];
my $fib = fib($N);
print "$fib\n";
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 28 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/python
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

import sys

def fib(n):
    if (n < 2):
        return(1)
    return( fib(n-2) + fib(n-1) )

def main():
    N = int(sys.argv[1])
    #sys.setrecursionlimit(3000)
    print fib(N)

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 28 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/tclsh
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

# with help from: Kristoffer Lawson

proc fib {n} {
    if {$n < 2} {
	return 1
    } else {
	return [expr {[fib [expr {$n-2}]] + [fib [expr {$n-1}]]}]
    }
}

set N [lindex $argv 0]
if {$N < 1} { set N 1 }
puts [fib $N]
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 28 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

function fib(n) {
    if (n < 2) return(1);
    return(fib(n-2) + fib(n-1));
}

BEGIN {
    n = (ARGV[1] < 1) ? 1 : ARGV[1];
    printf("%d\n", fib(n));
    exit;
}
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest 28 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
func fibonacci (n)
  {
     if (n < 2) return 1;
     return (fibonacci (n-2) + fibonacci (n-1));
  }

var n = argv [0] < 1 ? 1 : int (argv [0]);
putln (fibonacci (n)); 
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 28 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 7.
if test $start_test_number -le 7; then

######################################################
echo 
echo '+++++ Test #7: Hash (Associative Array) Access (N=80000):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

# this program is modified from:
#   http:#cm.bell-labs.com/cm/cs/who/bwk/interps/pap.html
# Timing Trials, or, the Trials of Timing: Experiments with Scripting
# and User-Interface Languages</a> by Brian W. Kernighan and
# Christopher J. Van Wyk.

use strict;

my $n = $ARGV[0] || 1;
my %X = ();
my $c = 0;

for my $i (1..$n) {
    $X{sprintf('%x', $i)} = $i;
}
for my $i (reverse 1..$n) {
    ++$c if exists $X{$i};
}
print "$c\n";
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 80000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# with help from from Gustavo Niemeyer

import sys
#sys.setcheckinterval(10000)

def main():
    n = int(sys.argv[1])
    X = {}
    myhex = hex
    for i in xrange(1,n+1):
        X[myhex(i)[2:]] = i
    c = 0
    has_key = X.has_key
    for i in xrange(n, 0, -1):
        c = c + has_key(`i`)
    print c

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 80000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

# this program is modified from:
#   http://cm.bell-labs.com/cm/cs/who/bwk/interps/pap.html
# Timing Trials, or, the Trials of Timing: Experiments with Scripting
# and User-Interface Languages</a> by Brian W. Kernighan and
# Christopher J. Van Wyk.

proc main {} {
    global argv
    set n [lindex $argv 0]
    for {set i 1} {$i <= $n} {incr i} {
        set x([format {%x} $i]) $i
    }
    set c 0
    for {set i $n} {$i > 0} {incr i -1} {
	if {[info exists x($i)]} {
	    incr c
	}
    }
    puts $c
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 80000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

# this program modified from:
#   http://cm.bell-labs.com/cm/cs/who/bwk/interps/pap.html
# Timing Trials, or, the Trials of Timing: Experiments with Scripting
# and User-Interface Languages</a> by Brian W. Kernighan and
# Christopher J. Van Wyk.

BEGIN {
    n = (ARGV[1] < 1) ? 1 : ARGV[1];

    for (i = 1; i <= n; i++)
	x[sprintf("%x", i)] = i
    for (i = n; i > 0; i--)
	if (i in x)
	    c++
    print c
}
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest 80000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
var i, c = 0, n = argv [0] < 1 ? 1 : int (argv [0]);
var f = "%x", x = {};

for (i = 1; i <= n; i++)
 x {vector (i, f)} = i;
for (i = n; i > 0; i--)
 if (vector (i) in x)
   c++;
putln (c);
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 80000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 8.
if test $start_test_number -le 8; then

######################################################
echo 
echo '+++++ Test #8: Hashes, Part II (N=50):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# with help from Steve Fink

use strict;

my $n = ($ARGV[0] > 0) ? $ARGV[0] : 1;
my %hash1 = ();
$hash1{"foo_$_"} = $_ for 0..9999;
my %hash2 = ();
my($k, $v);
for (1..$n) {
    $hash2{$_} += $hash1{$_} while (defined ($_ = each %hash1));
}
print "$hash1{foo_1} $hash1{foo_9999} $hash2{foo_1} $hash2{foo_9999}\n";
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 50 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# from Mark Baker

import sys

n = int(sys.argv[1])
hash1 = {}
for i in xrange(10000):
    hash1['foo_' + `i`] = i

hash2 = {}
for i in xrange(n):
    for k in hash1.keys():
        try:
            hash2[k] = hash2[k] + hash1[k]
        except KeyError:
            hash2[k] = hash1[k]

print hash1['foo_1'], hash1['foo_9999'], hash2['foo_1'], hash2['foo_9999']
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 50 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# with help from Branko Vesligaj

proc main {} {
    global argv
    set n [lindex $argv 0]
    for {set i 0} {$i < 10000} {incr i} {
	set hash1(foo_$i) $i
    }
    for {set i $n} {$i > 0} {incr i -1} {
	foreach k [array names hash1] {
	    if {[catch {set hash2($k) [expr {$hash1($k) + $hash2($k)}]}]} {
		set hash2($k) $hash1($k)
	    }
	}
    }
    puts [join [list $hash1(foo_1) $hash1(foo_9999) $hash2(foo_1) $hash2(foo_9999) ] " "]
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 50 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

BEGIN {
    n = (ARGV[1] < 1) ? 1 : ARGV[1];

    for (i=0; i<10000; i++)
	hash1[sprintf("foo_%d", i)] = i
    for (i=0; i<n; i++)
	for (k in hash1)
	    hash2[k] += hash1[k]
    print hash1["foo_1"], hash1["foo_9999"], hash2["foo_1"], hash2["foo_9999"]
}
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest 50 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
var i, n = int (argv[0]);
var hash1 = {};

for (i = 0; i < 10000; i++)
  hash1 {vector (i, "foo_%d")} = i;

var k, hash2 = {};
for (i = 0 ; i < n; i++)
  for (k in hash1)
    try {
      hash2 {k} += hash1 {k};
    } catch (except) {
      hash2 {k} = hash1 {k};
    }

putln (hash1 {"foo_1"}, " ", hash1 {"foo_9999"}, " ",
       hash2 {"foo_1"}, " ", hash2 {"foo_9999"});
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 50 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 9.
if test $start_test_number -le 9; then

######################################################
echo 
echo '+++++ Test #9: Heapsort (N=20000):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# Matt Harris suggested passing the array via typeglob

use strict;

use constant IM => 139968;
use constant IA =>   3877;
use constant IC =>  29573;

use vars qw(@ra);

my $LAST = 42;
sub gen_random { ($_[0] * ($LAST = ($LAST * IA + IC) % IM)) / IM }

sub heapsort ($\@) {
    my $n = shift;
    # use typeglob ra to refer to array.
    local *ra = shift;

    my($rra, $i, $j);

    my $l = ($n >> 1) + 1;
    my $ir = $n;
    while (1) {
	if ($l > 1) {
	    $rra = $ra[--$l];
	} else {
	    $rra = $ra[$ir];
	    $ra[$ir] = $ra[1];
	    if (--$ir == 1) {
		$ra[1] = $rra;
		return;
	    }
	}
	$i = $l;
	$j = $l << 1;
	while ($j <= $ir) {
	    $j++ if (($j < $ir) && ($ra[$j] < $ra[$j+1]));
	    if ($rra < $ra[$j]) {
		$ra[$i] = $ra[$j];
		$j += ($i = $j);
	    } else {
		$j = $ir + 1;
	    }
	}
	$ra[$i] = $rra;
    }
}


my $N = $ARGV[0];
$N = 1 if ($N < 1);

# create an array of N random doubles
my @ary = ();
for (my $i=1; $i<=$N; $i++) {
    $ary[$i] = gen_random(1.0);
}

heapsort($N, @ary);

printf("%.10f\n", $ary[-1]);
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 20000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

import sys

IM = 139968
IA =   3877
IC =  29573

LAST = 42
def gen_random(max):
    global LAST
    LAST = (LAST * IA + IC) % IM
    return( (max * LAST) / IM )

def heapsort(n, ra):
    rra = i = j = 0
    l = (n >> 1) + 1
    ir = n

    while (1):
        if (l > 1):
            l = l - 1
            rra = ra[l]
        else:
            rra = ra[ir]
            ra[ir] = ra[1]
	    ir = ir - 1
            if (ir == 1):
                ra[1] = rra
                return
        i = l
        j = l << 1
        while (j <= ir):
            if ((j < ir) and (ra[j] < ra[j+1])):
                j = j + 1
            if (rra < ra[j]):
                ra[i] = ra[j]
                i = j
                j = j + i
            else:
                j = ir + 1
        ra[i] = rra

def main():
    N = int(sys.argv[1])
    if N < 1:
        N = 1

    ary = range(N+1)
    for i in xrange(1,N+1):
        ary[i] = gen_random(1.0)

    heapsort(N, ary)

    print "%.10f" % ary[N]

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 20000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# sped up by Miguel Sofer's function generator

set IM 139968
set IA   3877
set IC  29573

set last 42

proc make_gen_random {} {
    global IM IA IC
    set params [list IM $IM IA $IA IC $IC]
    set body [string map $params {
        global last
        expr {($max * [set last [expr {($last * IA + IC) % IM}]]) / IM}
    }]
    proc gen_random {max} $body
}

proc heapsort {n ra_name} {
    upvar $ra_name ra

    set j 0
    set i 0
    set rra 0.0
    set l [expr {($n >> 1) + 1}]
    set ir $n
    while 1 {
        if {$l > 1} {
	    incr l -1
            set rra $ra($l)
        } else {
	    set rra $ra($ir)
	    set ra($ir) $ra(1)
	    incr ir -1
	    if {$ir == 1} {
                set ra(1) $rra
                return
            }
        }
	set i $l
	set j [expr {$l << 1}]
        while {$j <= $ir} {
	    if {($j < $ir) && ($ra($j) < $ra([expr {$j + 1}]))} {
		incr j
	    }
            if {$rra < $ra($j)} {
		set ra($i) $ra($j)
		set i $j
                set j [expr {$j + $i}]
            } else {
		set j [expr {$ir + 1}]
            }
        }
        set ra($i) $rra
    }
}

proc main {} {
    global argv
    set n [lindex $argv 0]
    make_gen_random

    for {set i 1} {$i <= $n} {incr i} {
	set ary($i) [gen_random 1.0]
    }
    heapsort $n ary
    puts [format "%.10f" $ary($n)]
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 20000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

function gen_random(n) { return( (n * (LAST = (LAST * IA + IC) % IM)) / IM ); }

function heapsort (n, ra) {
    l = int(0.5+n/2) + 1
    ir = n;
    for (;;) {
        if (l > 1) {
            rra = ra[--l];
        } else {
            rra = ra[ir];
            ra[ir] = ra[1];
            if (--ir == 1) {
                ra[1] = rra;
                return;
            }
        }
        i = l;
        j = l * 2;
        while (j <= ir) {
            if (j < ir && ra[j] < ra[j+1]) { ++j; }
            if (rra < ra[j]) {
                ra[i] = ra[j];
                j = j + (i = j);
            } else {
                j = ir + 1;
            }
        }
        ra[i] = rra;
    }
}

BEGIN {
    IM = 139968;
    IA = 3877;
    IC = 29573;
    LAST = 42;

    n = (ARGV[1] < 1) ? 1 : ARGV[1];
    ary[0] = 0;
    for (i=1; i<=n; i++) {
	ary[i] = gen_random(1.0);
    }

    heapsort(n, ary);

    printf("%.10f\n", ary[n]);

    exit;
}
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest 20000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
var IM = 139968;
var IA = 3877;
var IC = 29573;
var LAST = 42;

func gen_random (max) {
  LAST = (LAST * IA + IC) % IM;
  return (max * LAST) / IM;
}

func heapsort (n, ra) {
  var rra = 0, i = 0, j = 0;
  var l = (n >> 1) + 1;
  var ir = n;

  for (;;) {
    if (l > 1) {
      l--;
      rra = ra [l];
    } else {
      rra = ra [ir];
      ra [ir] = ra [1];
      ir--;
      if (ir == 1) {
	ra [1] = rra;
	return;
      }
    }
    i = l;
    j = l << 1;
    for (; j <= ir; ) {
      if (j < ir && ra [j] < ra [j + 1]) j++;
      if (rra < ra [j]) {
	ra [i] = ra [j];
	i = j;
	j = j + i;
      } else
	j = ir + 1;
    }
    ra [i] = rra;
  }
}

func main () {
  var i, n = (argv [0] < 1) ? 1 : int (argv [0]);
  var ary = [n + 1 : 0];

  for (i = 1; i <= n; i++)
    ary [i] = gen_random (1.0);
  
  heapsort (n, ary);
  
  putln (vector (ary[n], "%.10f"));
}

main ();
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 20000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 10.
if test $start_test_number -le 10; then

######################################################
echo 
echo '+++++ Test #10: Hello: Start up time (N=200)+++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

print "hello world\n";
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  time sh -c 'i=0; while test $i -lt 200;do $0 $*; i=`expr $i + 1`;done' $PERL $ftest|fgrep user
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

print "hello world"
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  time sh -c 'i=0; while test $i -lt 200;do $0 $*; i=`expr $i + 1`;done' $PYTHON $ftest|fgrep user
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

puts "hello world"
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  time sh -c 'i=0; while test $i -lt 200;do $0 $*; i=`expr $i + 1`;done' $TCLSH $ftest|fgrep user
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

BEGIN { print "hello world" }
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  time sh -c 'i=0; while test $i -lt 200;do $0 $*; i=`expr $i + 1`;done' $AWK -f $ftest </dev/null |fgrep user
fi

cat <<'EOF' >$ftest
putln ("hello world");
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
time sh -c 'i=0; while test $i -lt 200;do $0 $*; i=`expr $i + 1`;done' $DINO $ftest|fgrep user
fi

# Test 11.
if test $start_test_number -le 11; then

######################################################
echo 
echo '+++++ Test #11: List Processing (N=16):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
use strict;

my $SIZE = 10000;

my $ITER = $ARGV[0];
$ITER = 1 if ($ITER < 1);

my $result = 0;
while ($ITER--) {
    $result = &test_lists();
}
print "$result\n";

sub test_lists {
    # create a list of integers (Li1) from 1 to SIZE
    my @Li1 = (1..$SIZE);
    # copy the list to Li2 (not by individual items)
    my @Li2 = @Li1;
    my @Li3 = ();
    # remove each individual item from left side of Li2 and
    # append to right side of Li3 (preserving order)
    push(@Li3, shift @Li2) while (@Li2);
    # Li2 must now be empty
    # remove each individual item from right side of Li3 and
    # append to right side of Li2 (reversing list)
    push(@Li2, pop @Li3) while (@Li3);
    # Li3 must now be empty
    # reverse Li1 in place
    @Li1 = reverse @Li1;
    # check that first item is now SIZE
    return(0) if $Li1[0] != $SIZE;
    # compare Li1 and Li2 for equality
    my $len1 = scalar(@Li1);
    my $len2 = scalar(@Li2);
    my $lists_equal = ($len1 == $len2);
    return(0) if not $lists_equal;
    for my $i (0..($len1-1)) {
	if ($Li1[$i] != $Li2[$i]) {
	    $lists_equal = 0;
	    last;
	}
    }
    return(0) if not $lists_equal;
    # return the length of the list
    return($len1);
}
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 16 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# with improvements from Mark Baker

import sys

SIZE = 10000

def test_lists():
    Li1 = range(1, SIZE + 1)
    Li2 = Li1[:]
    Li3 = []

    # remove each individual item from left side of Li2 and
    # append to right side of Li3 (preserving order)
    # 
    # popping the first element is *expensive*
    #
    while Li2:
        Li3.append(Li2.pop(0))   
    #Li2.reverse()
    #while Li2:
    #    Li3.append(Li2.pop())
    while Li3:
        Li2.append(Li3.pop())
    Li1.reverse()
    if Li1[0] != SIZE:
        return 0
    if Li1 == Li2:
        return len(Li1)
    else:
        return 0

def main():
    NUM = int(sys.argv[1])
    if NUM < 1:
        NUM = 1
    while NUM > 0:
        result = test_lists()
        NUM = NUM - 1
    print result

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 16 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# from Kristoffer Lawson
# Modified by Tom Wilkason

set SIZE 10000

proc K {a b} {set a}

proc ldelete {listName index} {
    upvar $listName list
    ;# Replace a deletion with null, much faster
    set list [lreplace [K $list [set list {}]] $index $index]
}

proc lreverse {_list} {
    upvar $_list List
    for {set i [expr {[llength $List] - 1}]} {$i >= 0} {incr i -1} {
	lappend Li1r [lindex $List $i]
    }
    set List $Li1r
    unset Li1r
}

proc test_lists {args} {
    # create a list of integers (Li1) from 1 to SIZE
    for {set i 1} {$i <= $::SIZE} {incr i} {lappend Li1 $i}
    # copy the list to Li2 (not by individual items)
    set Li2 $Li1
    # remove each individual item from left side of Li2 and
    # append to right side of Li3 (preserving order)
    lreverse Li2
    foreach {item} $Li2 {
	lappend Li3 [lindex $Li2 end]
	ldelete Li2 end
    }
    # Li2 must now be empty
    # remove each individual item from right side of Li3 and
    # append to right side of Li2 (reversing list)
    foreach {item} $Li3 {
	lappend Li2 [lindex $Li3 end]
	ldelete Li3 end
    }
    # Li3 must now be empty
    # reverse Li1 in place
    lreverse Li1
    # check that first item is now SIZE
    if {[lindex $Li1 0] != $::SIZE} {
	return "fail size [lindex $Li1 0]"
    }
    # compare Li1 and Li2 for equality
    # and return length of the list
    if {$Li1 == $Li2} {
	return [llength $Li1]
    } else {
	return "fail compare"
    }
}

proc main {args} {
    global argv
    set NUM [lindex $argv 0]
    if {$NUM < 1} {
	set NUM 1
    }
    while {$NUM > 0} {
	set result [test_lists]
	incr NUM -1
    }
    puts $result
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 16 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
var SIZE = 10000;

func test_lists () {
  var i, Li1 = [SIZE : 0];

  // create a list of integers (Li1) from 1 to SIZE
  for (i = 0; i < SIZE; i++) Li1 [i] = i + 1;

  // copy the list to Li2 (not by individual items)
  var Li2 = new Li1;
  var Li3 = [];

  // remove each individual item from left side of Li2 and
  // append to right side of Li3 (preserving order)

  for (;#Li2;) {
    ins (Li3, Li2 [0], -1);
    del (Li2, 0);
  }

  // Li2 must now be empty
  // remove each individual item from right side of Li3 and
  // append to right side of Li2 (reversing list)
  for (;#Li3;) {
    ins (Li2, Li3 [#Li3 - 1], -1);
    del (Li3, #Li3 - 1);
  }
  rev (Li1);
  if (Li1 [0] != SIZE)
    return 0;
  return (Li1 == Li2 ? #Li1 : 0);
}

func main() {
  var result, NUM;
  for (NUM = argv [0] < 1 ? 1 : int (argv [0]); NUM > 0; NUM--)
    result = test_lists ();
  putln (result);
}

main ();
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 16 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 12.
if test $start_test_number -le 12; then

######################################################
echo 
echo '+++++ Test #12: Matrix Multiplication (N=30):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/perl 
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

# This program based on the original from:
# "The What, Why, Who, and Where of Python" By Aaron R. Watters
# http://www.networkcomputing.com/unixworld/tutorial/005/005.html

# modified to pass rows and cols, and avoid matrix size checks
# I've sped up the original quite a bit by removing some loop
# invariants and declaring "use integer"

use strict;
use integer;

my $size = 30;

sub mkmatrix {
    my($rows, $cols) = @_;
    --$rows; --$cols;
    my $count = 1;
    my @mx = ();
    foreach (0 .. $rows) {
	my @row = ();
	$row[$_] = $count++ foreach (0 .. $cols);
	push(@mx, \@row);
    }
    return(\@mx);
}

sub mmult {
    my ($rows, $cols, $m1, $m2) = @_;
    my @m3 = ();
    --$rows; --$cols;
    for my $i (0 .. $rows) {
	my @row = ();
	my $m1i = $m1->[$i];
	for my $j (0 .. $cols) {
	    my $val = 0;
	    for my $k (0 .. $cols) {
		$val += $m1i->[$k] * $m2->[$k]->[$j];
	    }
	    push(@row, $val);
	}
	push(@m3, \@row);
    }
    return(\@m3);
}

my $N = $ARGV[0] || 1;

my $m1 = mkmatrix($size, $size);
my $m2 = mkmatrix($size, $size);
my $mm;
while ($N--) {
    $mm = mmult($size, $size, $m1, $m2);
}
print "$mm->[0]->[0] $mm->[2]->[3] $mm->[3]->[2] $mm->[4]->[4]\n";

EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 30 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/python
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

# This program based on the original from:
# "The What, Why, Who, and Where of Python" By Aaron R. Watters
# http://www.networkcomputing.com/unixworld/tutorial/005/005.html

# modified to pass rows and cols, and avoid matrix size checks
# and added one optimization to reduce subscripted references in
# inner loop.

import sys

size = 30

def mkmatrix(rows, cols):
    count = 1
    mx = [ None ] * rows
    for i in range(rows):
        mx[i] = [0] * cols
        for j in range(cols):
            mx[i][j] = count
            count += 1
    return mx

def mmult(rows, cols, m1, m2):
    m3 = [ None ] * rows
    for i in range( rows ):
        m3[i] = [0] * cols
        for j in range( cols ):
            val = 0
            for k in range( cols ):
                val += m1[i][k] * m2[k][j]
            m3[i][j] = val
    return m3

def mxprint(m):
    for i in range(size):
        for j in range(size):
            print m[i][j],
        print ""

def main():
    iter = int(sys.argv[1])
    m1 = mkmatrix(size, size)
    m2 = mkmatrix(size, size)
    for i in xrange(iter):
        mm = mmult(size, size, m1, m2)
    print mm[0][0], mm[2][3], mm[3][2], mm[4][4]

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 30 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/tclsh
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

# This program based on the original from:
# "The What, Why, Who, and Where of Python" By Aaron R. Watters
# http://www.networkcomputing.com/unixworld/tutorial/005/005.html

# modified to avoid matrix size checks
# --Doug

# additional speedups by Kristoffer Lawson and Miguel Sofer

set size 30;

proc mkmatrix {rows cols} {
    set count 1;
    set mx [list]
    for { set i 0 } { $i < $rows } { incr i } {
	set row [list]
	for { set j 0 } { $j < $cols } { incr j } {
	    lappend row $count;
	    incr count;
	}
	lappend mx $row;
    }
    return $mx;
}

proc mmult {m1 m2} {
    set cols [lindex $m2 0]
    foreach row1 $m1 {
        set row [list]
        set i 0
        foreach - $cols {
            set elem 0
            foreach elem1 $row1 row2 $m2 {
                set elem [expr {$elem + $elem1 * [lindex $row2 $i]}]
            }
            lappend row $elem
            incr i
        }
        lappend result $row
    }
    return $result
}

proc main {} {
    global argv size
    set num [lindex $argv 0]
    if {$num < 1} {
	set num 1
    }

    set m1 [mkmatrix $size $size]
    set m2 [mkmatrix $size $size]
    while {$num > 0} {
        incr num -1
        set m [mmult $m1 $m2]
    }

    puts "[lindex [lindex $m 0] 0] [lindex [lindex $m 2] 3] [lindex [lindex $m 3] 2] [lindex [lindex $m 4] 4]"
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 30 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

echo
echo AWK is too slow for this test.
echo

cat <<'EOF' >$ftest
var size = 30;

func mkmatrix (rows, cols) {
  var i, j, count = 1;
  var tm, mx = [rows:1];

  for (i = 0; i < rows; i++) {
    mx [i] = [cols:1];
    tm = mx [i];
    for (j = 0; j < cols; j++) {
      tm [j] = count;
      count++;
    }
  }
  return mx;
}

func mmult (rows, cols, m1, m2) {
  var i, j, k, val, tm, m3 = [rows:1];

  for (i = 0; i < rows; i++) {
    m3 [i] = [cols:1];
    for (j = 0; j < cols; j++) {
       val = 0;
       tm = m1 [i];
       for (k = 0; k < cols; k++)
         val += tm [k] * m2 [k][j];
       m3 [i][j] = val;
    }
  }
  return m3;
}

func main () {
  var m1, m2, mm;
  var i, iter = argv [0] < 1 ? 1 : int (argv [0]);

  m1 = mkmatrix (size, size);
  m2 = mkmatrix (size, size);
  for (i = 0; i < iter; i++)
     mm = mmult (size, size, m1, m2);
  putln (mm [0][0], ' ', mm [2][3], ' ', mm [3][2], ' ', mm [4][4]);
}
main ();
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 30 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 13.
if test $start_test_number -le 13; then

######################################################
echo 
echo '+++++ Test #13: Method Calls (N=100000):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# with help from Ben Tilly

package Toggle;

sub new {
    my($class, $start_state) = @_;
    bless( { Bool => $start_state }, $class );
}

sub value {
    (shift)->{Bool};
}

sub activate {
    my $self = shift;
    $self->{Bool} ^= 1;
    return($self);
}


package NthToggle;
our @ISA = qw(Toggle);

sub new {
    my($class, $start_state, $max_counter) = @_;
    my $self = $class->SUPER::new($start_state);
    $self->{CountMax} = $max_counter;
    $self->{Counter} = 0;
    return($self);
}

sub activate {
    my $self = shift;
    if (++$self->{Counter} >= $self->{CountMax}) {
	$self->{Bool} ^= 1;
	$self->{Counter} = 0;
    }
    return($self);
}


package main;

sub main {
    my $NUM = $ARGV[0];
    $NUM = 1 if ($NUM < 1);

    my $val = 1;
    my $toggle = Toggle->new($val);
    for (1..$NUM) {
	$val = $toggle->activate->value;
    }
    print (($val) ? "true\n" : "false\n");

    $val = 1;
    my $ntoggle = NthToggle->new($val, 3);
    for (1..$NUM) {
	$val = $ntoggle->activate->value;
    }
    print (($val) ? "true\n" : "false\n");
}

main();
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 100000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

import sys

class Toggle:
    def __init__(self, start_state):
        self.bool = start_state
    def value(self):
        return(self.bool)
    def activate(self):
        self.bool = not self.bool
        return(self)

class NthToggle(Toggle):
    def __init__(self, start_state, max_counter):
        Toggle.__init__(self, start_state)
        self.count_max = max_counter
        self.counter = 0
    def activate(self):
        self.counter = self.counter + 1
        if (self.counter >= self.count_max):
            self.bool = not self.bool
            self.counter = 0
        return(self)


def main():
    NUM = int(sys.argv[1])
    if NUM < 1:
        NUM = 1

    val = 1
    toggle = Toggle(val)
    for i in xrange(0,NUM):
	val = toggle.activate().value()
    if val:
        print "true"
    else:
        print "false"

    val = 1
    ntoggle = NthToggle(val, 3)
    for i in xrange(0,NUM):
        val = ntoggle.activate().value()
    if val:
        print "true"
    else:
        print "false"

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 100000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi


cat <<'EOF' >$ftest
class Toggle (start_state) {
  var bool = start_state;
  func value () { return bool; }
  func activate () { bool = !bool; }
}

ext Toggle {
  class NthToggle (max_counter) {
    var count_max = max_counter;
    var counter = 0;
    func activate () {
      counter++;
      if (counter >= count_max) {
	bool = !bool;
	counter = 0;
      }
    }
  }
}

func main () {
  var NUM = argv [0] < 1 ? 1 : int (argv [0]);
  var i, val = 1;

  var toggle = Toggle (val);
  for (i = 1; i <= NUM; i++) {
    toggle.activate ();
    val = toggle.value ();
  }
  putln (val ? "true" : "false");

  val = 1;
  var ntoggle = Toggle (val).NthToggle (3);
  for (i = 1; i <= NUM; i++) {
    ntoggle.activate ();
    val = ntoggle.value ();
  }
  putln (val ? "true" : "false");
}

main();
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 100000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 14.
if test $start_test_number -le 14; then

######################################################
echo 
echo '+++++ Test #14: Nested Loops (N=11):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

use strict;

my $n = ($ARGV[0] > 0) ? $ARGV[0] : 1;
my $x = 0;
my $a = $n;
while ($a--) {
    my $b = $n;
    while ($b--) {
	my $c = $n;
	while ($c--) {
	    my $d = $n;
	    while ($d--) {
		my $e = $n;
		while ($e--) {
		    my $f = $n;
		    while ($f--) {
			$x++;
		    }
		}
	    }
	}
    }
}
print "$x\n";
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 11 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# with help from Mark Baker

import sys

def main():
    x = 0
    iter = int(sys.argv[1])
    if iter < 1:
        iter = 1
    
    i_r = range(iter)
    for a in i_r:
        for b in i_r:
            for c in i_r:
                for d in i_r:
                    for e in i_r:
                        for f in i_r:
                            x = x + 1
    print x

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 11 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# from Tom Wilkason

proc main {} {
    global argv
    set n [lindex $argv 0]
    set x 0
    incr n 1
    set a $n
    while {[incr a -1]} {
	set b $n
	while {[incr b -1]} {
	    set c $n
	    while {[incr c -1]} {
		set d $n
		while {[incr d -1]} {
		    set e $n
		    while {[incr e -1]} {
			set f $n
			while {[incr f -1]} {
			    incr x
			}
		    }
		}
	    }
	}
    }
    puts $x
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 11 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

BEGIN {
    n = (ARGV[1] < 1) ? 1 : ARGV[1];

    for (a=0; a<n; a++)
	for (b=0; b<n; b++)
	    for (c=0; c<n; c++)
		for (d=0; d<n; d++)
		    for (e=0; e<n; e++)
			for (f=0; f<n; f++)
			    x += 1
    print x
}
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest 11 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
var n = argv [0] < 1 ? 1 : int (argv[0]);
var a, b, c, d, e, f, x = 0;

for (a = n; a > 0; a--)
  for (b = n; b > 0; b--)
    for (c = n; c > 0; c--)
      for (d = n; d > 0; d--)
	for (e = n; e > 0; e--)
	  for (f = n; f > 0; f--)
	    x++;
putln (x);
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 11 </dev/null >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 15.
if test $start_test_number -le 15; then

######################################################
echo 
echo '+++++ Test #15: Object Instantiation (N=100000):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

use strict;


package Toggle;

sub new {
    my($class, $start_state) = @_;
    bless( { Bool => $start_state }, $class );
}

sub value {
    my $self = shift;
    return($self->{Bool});
}

sub activate {
    my $self = shift;
    $self->{Bool} ^= 1;
    return($self);
}


package NthToggle;
@NthToggle::ISA = qw(Toggle);

sub new {
    my($class, $start_state, $max_counter) = @_;
    my $self = $class->SUPER::new($start_state);
    $self->{CountMax} = $max_counter;
    $self->{Counter} = 0;
    return($self);
}

sub activate {
    my $self = shift;
    if (++$self->{Counter} >= $self->{CountMax}) {
	$self->{Bool} ^= 1;
	$self->{Counter} = 0;
    }
    return($self);
}


package main;

sub main {
    my $NUM = ($ARGV[0] > 0) ? $ARGV[0] : 1;

    my $toggle = Toggle->new(1);
    for (1..5) {
	print (($toggle->activate->value) ? "true\n" : "false\n");
    }
    for (1..$NUM) {
	$toggle = Toggle->new(1);
    }

    print "\n";

    my $ntoggle = NthToggle->new(1, 3);
    for (1..8) {
	print (($ntoggle->activate->value) ? "true\n" : "false\n");
    }
    for (1..$NUM) {
	$ntoggle = NthToggle->new(1, 3);
    }
}

main();
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 100000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

import sys

class Toggle:
    def __init__(self, start_state):
        self.bool = start_state
    def value(self):
        return(self.bool)
    def activate(self):
        self.bool = not self.bool
        return(self)

class NthToggle(Toggle):
    def __init__(self, start_state, max_counter):
        Toggle.__init__(self, start_state)
        self.count_max = max_counter
        self.counter = 0
    def activate(self):
        self.counter = self.counter + 1
        if (self.counter >= self.count_max):
            self.bool = not self.bool
            self.counter = 0
        return(self)


def main():
    NUM = int(sys.argv[1])
    if NUM < 1:
        NUM = 1

    toggle = Toggle(1)
    for i in xrange(0,5):
        if toggle.activate().value():
            print "true"
        else:
            print "false"
    for i in xrange(0,NUM):
	toggle = Toggle(1)

    print ""

    ntoggle = NthToggle(1, 3)
    for i in xrange(0,8):
        if ntoggle.activate().value():
            print "true"
        else:
            print "false"
    for i in xrange(0,NUM):
	ntoggle = NthToggle(1, 3)

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 100000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi


cat <<'EOF' >$ftest
class Toggle (start_state) {
  var bool = start_state;
  func value () { return bool; }
  func activate () { bool = !bool; }
}

ext Toggle {
  class NthToggle (max_counter) {
    var count_max = max_counter;
    var counter = 0;
    func activate () {
      counter++;
      if (counter >= count_max) {
	bool = !bool;
	counter = 0;
      }
    }
  }
}

func main () {
  var NUM = argv [0] < 1 ? 1 : int (argv [0]);
  var i;

  var toggle = Toggle (1);
  for (i = 1; i <= 5; i++) {
    toggle.activate ();
    putln (toggle.value () ? "true" : "false");
  }
  for (i = 1; i <= NUM; i++)
   toggle = Toggle (1);

  putln ();

  var ntoggle = Toggle (1).NthToggle (3);
  for (i = 1; i <= 8; i++) {
    ntoggle.activate ();
    putln (ntoggle.value () ? "true" : "false");
  }
  for (i = 1; i <= NUM; i++)
    ntoggle = Toggle (1).NthToggle (3);
}

main();
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 100000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 16.
if test $start_test_number -le 16; then

######################################################
echo 
echo '+++++ Test #16: Producer/Consumer Threads (N=20000):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
#!/usr/local/test/bin/perl
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

use strict;
use Thread qw(cond_wait cond_signal);

my $count = 0;
my $data = 0;
my $produced = 0;
my $consumed = 0;

sub consumer {
    my $n = shift;
    while (1) {
	lock($count);
	cond_wait($count) while ($count == 0);
	my $i = $data;
	$count = 0;
	$consumed++;
	last if ($i == $n);
	cond_signal($count);
    }
}

sub producer {
    my $n = shift;
    for (my $i=1; $i<=$n; $i++) {
	lock($count);
	cond_wait($count) while ($count == 1);
	$data = $i;
	$count = 1;
	$produced++;
	cond_signal($count);
    }
}

sub main {
    my $n = ($ARGV[0] < 1) ? 1 : $ARGV[0];
    my $p = Thread->new(\&producer, $n);
    my $c = Thread->new(\&consumer, $n);
    $p->join;
    $c->join;
    print "$produced $consumed\n";
}

&main();
EOF

echo
echo Perl threads are not implemented on all systems.
echo

#  echo PERL:
#  if test "x$NECHO" != x;then $NECHO "   ";fi
#  if time $PERL $ftest 20000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/python
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

import sys
from threading import * 

access = Condition()
count = 0
consumed = 0
produced = 0
data = 0

def consumer(n):
    global count, data, consumed
    while 1:
        access.acquire()
        while count == 0:
            access.wait()
        i = data
        count = 0
        access.notify()
        access.release()
        consumed += 1
        if i == n:
            break

def producer(n):
    global count, data, produced
    for i in xrange(1,n+1):
        access.acquire()
        while count == 1:
            access.wait()
        data = i
        count = 1
        access.notify()
        access.release()
        produced += 1

def main(n):
    t1 = Thread(target=producer, args=(n,))
    t2 = Thread(target=consumer, args=(n,))
    t1.start()
    t2.start()
    t1.join()
    t2.join()
    print produced, consumed
    
main(int(sys.argv[1]))
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 20000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
var count = 0;
var consumed = 0;
var produced = 0;
var data = 0;

thread consumer (n) {
  var i;

  for (;;) {
    wait count != 0;
    i = data;
    count = 0;
    consumed++;
    if (i == n - 1)
      break;
  }
}

thread producer (n) {
  var i;

  for (i = 0; i < n; i++) {
    wait count == 0;
    data = i;
    count = 1;
    produced++;
  }
}

func main (n) {
    producer (n);
    consumer (n);
    putln (produced, ' ', consumed);
}
    
main (argv [0] < 1 ? 1 : int (argv [0]));
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 20000 </dev/null >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 17.
if test $start_test_number -le 17; then

######################################################
echo 
echo '+++++ Test #17: Random Number Generator (N=500000):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

use strict;

use constant IM => 139968;
use constant IA => 3877;
use constant IC => 29573;

my $LAST = 42;
sub gen_random { ($_[0] * ($LAST = ($LAST * IA + IC) % IM)) / IM }

my $N = ($ARGV[0] || 1) - 1;
gen_random(100.0) while ($N--);
printf "%.9f\n", gen_random(100.0);
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 500000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# with help from Brent Burley

import sys

IM = 139968
IA = 3877
IC = 29573

LAST = 42
def gen_random(max):
    global LAST
    LAST = (LAST * IA + IC) % IM
    return( (max * LAST) / IM )

def main():
    N = int(sys.argv[1])
    if N < 1:
        N = 1
    gr = gen_random
    for i in xrange(1,N):
        gr(100.0)
    print "%.9f" % gr(100.0)

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 500000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# from Miguel Sofer

set IM 139968
set IA   3877
set IC  29573

set last 42

proc make_gen_random {} {
    global IM IA IC
    set params [list IM $IM IA $IA IC $IC]
    set body [string map $params {
        global last
        expr {($max * [set last [expr {($last * IA + IC) % IM}]]) / IM}
    }]
    proc gen_random {max} $body
}

proc main {} {
    global argv

    set N [expr {[lindex $argv 0] - 1}]
    make_gen_random

    while {$N} {
        gen_random 100.0
        incr N -1
    }

    puts [format "%.9f" [gen_random 100.0]]
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 500000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

function gen_random(max) { return( (max * (LAST = (LAST * IA + IC) % IM)) / IM ); }

BEGIN {
    IM = 139968;
    IA = 3877;
    IC = 29573;
    LAST = 42;

    n = ((ARGV[1] < 1) ? 1 : ARGV[1]) - 1;
    while (n--) {
	gen_random(100);
    }
    printf("%.9f\n", gen_random(100));
    exit;
}
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest 500000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
var IM = 139968;
var IA = 3877;
var IC = 29573;
var LAST = 42;

func gen_random (max) {
  LAST = (LAST * IA + IC) % IM;
  return max * LAST / IM;
}

var n = (argv [0] < 1 ? 1 : argv [0]) - 1;
for (; n; n--)
  gen_random (100);
putln (vector (gen_random (100.0), "%.9f"));
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 500000 </dev/null >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 18.
if test $start_test_number -le 18; then

######################################################
echo 
echo '+++++ Test #18: Regular Expression Matching (N=5000):  +++++'

cat <<'EOF' >$input
datafile for regex matching of phone numbers test

the following numbers should match:

(111) 111 1111
111 222-2222
(111) 333-3333
111 444 4444

foo (111) 555 5555
foo 111 666-6666
foo (111) 777-7777
foo 111 888 8888

foo (111) 999 9999 bar
foo 111 000-0000 bar
foo (111) 232-1111 bar
foo 111 242 1111 bar

The following should fail to match:

2(212) 222 2222
2232 222-2222
(242) 222-22222
2252 222 2222

foo (262) 2222222
foo 272 2222222
foo (282) 222-22227
foo 292 222 22222

foo (202).222 2222 bar
foo 211@22222222 bar
foo (213 222-2222 bar
foo 214) 222 2222 bar

EOF

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

use strict;

my $re = qr{
    (?: ^ | [^\d\(])		# must be preceeded by non-digit
    ( \( )?			# match 1: possible initial left paren
    (\d\d\d)			# match 2: area code is 3 digits
    (?(1) \) )			# if match1 then match right paren
    [ ]				# area code followed by one space
    (\d\d\d)			# match 3: prefix of 3 digits
    [ -]			# separator is either space or dash
    (\d\d\d\d)			# match 4: last 4 digits
    \D				# must be followed by a non-digit
}x;

my $NUM = $ARGV[0];
$NUM = 1 if ($NUM < 1);

my @phones = <STDIN>;
my $count = 0;
my $num;
while ($NUM--) {
    foreach (@phones) {
	if (/$re/o) {
	    $num = "($2) $3-$4";
	    if (0 == $NUM) {
		$count++;
		print "$count: $num\n";
	    }
	}
    }
}
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 5000 <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

import sys, re

def main():
    NUM = int(sys.argv[1])
    if NUM < 1:
        NUM = 1

    phones = sys.stdin.readlines()

    rx = re.compile(
        r'(?:^|[^\d\(])'
        r'(?:\((\d\d\d)\)|(\d\d\d))'
        r'[ ]'
        r'(\d\d\d)'
        r'[ -]'
        r'(\d\d\d\d)'
        r'\D'
        )

    findIt = rx.search
    count = 0
    for i in xrange(0,NUM):
        for line in phones:
            m = findIt(line)
            if m:
                g = m.group
                num = "(" + (g(1) or g(2)) + ") " + g(3) + "-" + g(4)
                if 0 == i:
                    count = count + 1
                    print "%d: %s" % (count, num)

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 5000 <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# from: Miguel Sofer, with modifications by Kristoffer Lawson

proc main {} {
    global argv
    set NUM [lindex $argv 0]
    if {$NUM < 1} {
        set NUM 1
    }

    set data [read stdin]
    set count 0    
    set rExp {(?:^|[^\d(])(\(\d{3}\)|\d{3}) (\d{3}[ -]\d{4})(?:$|[^\d])}

    while {$NUM} {
	incr NUM -1
	foreach {-- area num} [regexp -all -line -inline $rExp $data] {
	    set pnum "([string trim $area () ]) [string map {" " -} $num]"
	    if {!$NUM} {
		puts "[incr count]: $pnum"
	    }
	}
    }
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 5000 <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

BEGIN {
    n = (ARGV[1] < 1) ? 1 : ARGV[1];
    delete ARGV;
}

{ phones[p++] = $0; }

END {
    for (i=0; i<n; i++) {
	for (j=0; j<p; j++) {
	    line = phones[j];
	    if (match(line, /(^|[^0-9])(\([0-9][0-9][0-9]\)|[0-9][0-9][0-9]) [0-9][0-9][0-9][ -][0-9][0-9][0-9][0-9]($|[^0-9])/)) {
		m1 = substr(line, RSTART, RLENGTH);
		num = ""
		if (match(m1, /[0-9][0-9][0-9] [0-9][0-9][0-9][ -][0-9][0-9][0-9][0-9]/)) {
		    if (substr(m1, RSTART-1, 1) == "(") {
			break;
		    }
		    if (x = split(substr(m1, RSTART, RLENGTH), parts, /[ -]/)) {
			num = "(" parts[1] ") " parts[2] "-" parts[3];
		    }
		} else if (match(m1, /\([0-9][0-9][0-9]\) [0-9][0-9][0-9][ -][0-9][0-9][0-9][0-9]/)) {
		    if (x = split(substr(m1, RSTART, RLENGTH), parts, /[ -]/)) {
			num = parts[1] " " parts[2] "-" parts[3];
		    }
		}
		if (i == (n-1)) {
		    count++;
		    printf("%d: %s\n", count, num);
		}
	    }
	}
    }
}
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest 5000 <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
func main () {
  var n = (argv [0] < 1) ? 1 : int (argv [0]);
  var ln, lno = 0, phones = [];
  
  phones = getf (1); lno = #phones;

  var i, j, v, count = 0;
  var area = "([0-9][0-9][0-9]|\\([0-9][0-9][0-9]\\))";
  var exch = "([0-9][0-9][0-9])";
  var last = "([0-9][0-9][0-9][0-9])";
  var re = "^[^0-9\\(]*" @ area @ " " @ exch @ "[ -]" @ last @ "[^0-9]*$";

  for (i = 0; i < n; i++) {
    for (j = 0; j < lno; j++) {
      ln = phones [j];
      v = match (re, ln);
      if (v != nil && i == n - 1) {
	count++;
	if (ln [v [2]] == '(')
	  putln (count, ": ", subv (ln, v[2], v [3] - v[2]), ' ', 
	         subv (ln, v[4], v [5] - v[4]), '-',
	         subv (ln, v[6], v [7] - v[6]));
	else
	  putln (count, ": (", subv (ln, v[2], v [3] - v[2]), ") ", 
	         subv (ln, v[4], v [5] - v[4]), '-',
	         subv (ln, v[6], v [7] - v[6]));
      }
    }
  }
}

main ();
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 5000 <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 19.
if test $start_test_number -le 19; then

######################################################
echo 
echo '+++++ Test #19: Reverse a File (N=20):  +++++'

cat <<'EOF' >$temp
aback
abaft
abandon
abandoned
abandoning
abandonment
abandons
abase
abased
abasement
abasements
abases
abash
abashed
abashes
abashing
abasing
abate
abated
abatement
abatements
abater
abates
abating
abbe
abbey
abbeys
abbot
abbots
abbreviate
abbreviated
abbreviates
abbreviating
abbreviation
abbreviations
abdomen
abdomens
abdominal
abduct
abducted
abduction
abductions
abductor
abductors
abducts
abed
aberrant
aberration
aberrations
abet
abets
abetted
abetter
abetting
abeyance
abhor
abhorred
abhorrent
abhorrer
abhorring
abhors
abide
abided
abides
abiding
abilities
ability
abject
abjection
abjections
abjectly
abjectness
abjure
abjured
abjures
abjuring
ablate
ablated
ablates
ablating
ablation
ablative
ablaze
able
abler
ablest
ably
abnormal
abnormalities
abnormality
abnormally
aboard
abode
abodes
abolish
abolished
abolisher
abolishers
abolishes
abolishing
abolishment
abolishments
abolition
abolitionist
abolitionists
abominable
abominate
aboriginal
aborigine
aborigines
abort
aborted
aborting
abortion
abortions
abortive
abortively
aborts
abound
abounded
abounding
abounds
about
above
aboveboard
aboveground
abovementioned
abrade
abraded
abrades
abrading
abrasion
abrasions
abrasive
abreaction
abreactions
abreast
abridge
abridged
abridges
abridging
abridgment
abroad
abrogate
abrogated
abrogates
abrogating
abrupt
abruptly
abruptness
abscess
abscessed
abscesses
abscissa
abscissas
abscond
absconded
absconding
absconds
absence
absences
absent
absented
absentee
absenteeism
absentees
absentia
absenting
absently
absentminded
absents
absinthe
absolute
absolutely
absoluteness
absolutes
absolution
absolve
absolved
absolves
absolving
absorb
absorbed
absorbency
absorbent
absorber
absorbing
absorbs
absorption
absorptions
absorptive
abstain
abstained
abstainer
abstaining
abstains
abstention
abstentions
abstinence
abstract
abstracted
abstracting
abstraction
abstractionism
abstractionist
abstractions
abstractly
abstractness
abstractor
abstractors
abstracts
abstruse
abstruseness
absurd
absurdities
absurdity
absurdly
abundance
abundant
abundantly
abuse
abused
abuses
abusing
abusive
abut
abutment
abuts
abutted
abutter
abutters
abutting
abysmal
abysmally
abyss
abysses
acacia
academia
academic
academically
academics
academies
academy
accede
acceded
accedes
accelerate
accelerated
accelerates
accelerating
acceleration
accelerations
accelerator
accelerators
accelerometer
accelerometers
accent
accented
accenting
accents
accentual
accentuate
accentuated
accentuates
accentuating
accentuation
accept
acceptability
acceptable
acceptably
acceptance
acceptances
accepted
accepter
accepters
accepting
acceptor
acceptors
accepts
access
accessed
accesses
accessibility
accessible
accessibly
accessing
accession
accessions
accessories
accessors
accessory
accident
accidental
accidentally
accidently
accidents
acclaim
acclaimed
acclaiming
acclaims
acclamation
acclimate
acclimated
acclimates
acclimating
acclimatization
acclimatized
accolade
accolades
accommodate
accommodated
accommodates
accommodating
accommodation
accommodations
accompanied
accompanies
accompaniment
accompaniments
accompanist
accompanists
accompany
accompanying
accomplice
accomplices
accomplish
accomplished
accomplisher
accomplishers
accomplishes
accomplishing
accomplishment
accomplishments
accord
accordance
accorded
accorder
accorders
according
accordingly
accordion
accordions
accords
accost
accosted
accosting
accosts
account
accountability
accountable
accountably
accountancy
accountant
accountants
accounted
accounting
accounts
accredit
accreditation
accreditations
accredited
accretion
accretions
accrue
accrued
accrues
accruing
acculturate
acculturated
acculturates
acculturating
acculturation
accumulate
accumulated
accumulates
accumulating
accumulation
accumulations
accumulator
accumulators
accuracies
accuracy
accurate
accurately
accurateness
accursed
accusal
accusation
accusations
accusative
accuse
accused
accuser
accuses
accusing
accusingly
accustom
accustomed
accustoming
accustoms
ace
aces
acetate
acetone
acetylene
ache
ached
aches
achievable
achieve
achieved
achievement
achievements
achiever
achievers
achieves
achieving
aching
acid
acidic
acidities
acidity
acidly
acids
acidulous
acknowledge
acknowledgeable
acknowledged
acknowledgement
acknowledgements
acknowledger
acknowledgers
acknowledges
acknowledging
acknowledgment
acknowledgments
acme
acne
acolyte
acolytes
acorn
acorns
acoustic
acoustical
acoustically
acoustician
acoustics
acquaint
acquaintance
acquaintances
acquainted
acquainting
acquaints
acquiesce
acquiesced
acquiescence
acquiescent
acquiesces
acquiescing
acquirable
acquire
acquired
acquires
acquiring
acquisition
acquisitions
acquisitive
acquisitiveness
acquit
acquits
acquittal
acquitted
acquitter
acquitting
acre
acreage
acres
acrid
acrimonious
acrimony
acrobat
acrobatic
acrobatics
acrobats
acronym
acronyms
acropolis
across
acrylic
act
acted
acting
actinium
actinometer
actinometers
action
actions
activate
activated
activates
activating
activation
activations
activator
activators
active
actively
activism
activist
activists
activities
activity
actor
actors
actress
actresses
actual
actualities
actuality
actualization
actually
actuals
actuarial
actuarially
actuate
actuated
actuates
actuating
actuator
actuators
acuity
acumen
acute
acutely
acuteness
acyclic
acyclically
ad
adage
adages
adagio
adagios
adamant
adamantly
adapt
adaptability
adaptable
adaptation
adaptations
adapted
adapter
adapters
adapting
adaptive
adaptively
adaptor
adaptors
adapts
add
added
addend
addenda
addendum
adder
adders
addict
addicted
addicting
addiction
addictions
addicts
adding
addition
additional
additionally
additions
additive
additives
additivity
address
addressability
addressable
addressed
addressee
addressees
addresser
addressers
addresses
addressing
adds
adduce
adduced
adduces
adducible
adducing
adduct
adducted
adducting
adduction
adductor
adducts
adept
adequacies
adequacy
adequate
adequately
adhere
adhered
adherence
adherent
adherents
adherer
adherers
adheres
adhering
adhesion
adhesions
adhesive
adhesives
adiabatic
adiabatically
adieu
adjacency
adjacent
adjective
adjectives
adjoin
adjoined
adjoining
adjoins
adjourn
adjourned
adjourning
adjournment
adjourns
adjudge
adjudged
adjudges
adjudging
adjudicate
adjudicated
adjudicates
adjudicating
adjudication
adjudications
adjunct
adjuncts
adjure
adjured
adjures
adjuring
adjust
adjustable
adjustably
adjusted
adjuster
adjusters
adjusting
adjustment
adjustments
adjustor
adjustors
adjusts
adjutant
adjutants
administer
administered
administering
administerings
administers
administrable
administrate
administration
administrations
administrative
administratively
administrator
administrators
admirable
admirably
admiral
admirals
admiralty
admiration
admirations
admire
admired
admirer
admirers
admires
admiring
admiringly
admissibility
admissible
admission
admissions
admit
admits
admittance
admitted
admittedly
admitter
admitters
admitting
admix
admixed
admixes
admixture
admonish
admonished
admonishes
admonishing
admonishment
admonishments
admonition
admonitions
ado
adobe
adolescence
adolescent
adolescents
adopt
adopted
adopter
adopters
adopting
adoption
adoptions
adoptive
adopts
adorable
adoration
adore
adored
adores
adorn
adorned
adornment
adornments
adorns
adrenal
adrenaline
adrift
adroit
adroitness
ads
adsorb
adsorbed
adsorbing
adsorbs
adsorption
adulate
adulating
adulation
adult
adulterate
adulterated
adulterates
adulterating
adulterer
adulterers
adulterous
adulterously
adultery
adulthood
adults
adumbrate
adumbrated
adumbrates
adumbrating
adumbration
advance
advanced
advancement
advancements
advances
advancing
advantage
advantaged
advantageous
advantageously
advantages
advent
adventist
adventists
adventitious
adventure
adventured
adventurer
adventurers
adventures
adventuring
adventurous
adverb
adverbial
adverbs
adversaries
adversary
adverse
adversely
adversities
adversity
advert
advertise
advertised
advertisement
advertisements
advertiser
advertisers
advertises
advertising
advice
advisability
advisable
advisably
advise
advised
advisedly
advisee
advisees
advisement
advisements
adviser
advisers
advises
advising
advisor
advisors
advisory
advocacy
advocate
advocated
advocates
advocating
aegis
aerate
aerated
aerates
aerating
aeration
aerator
aerators
aerial
aerials
aeroacoustic
aerobic
aerobics
aerodynamic
aerodynamics
aeronautic
aeronautical
aeronautics
aerosol
aerosolize
aerosols
aerospace
aesthetic
aesthetically
aesthetics
afar
affable
affair
affairs
affect
affectation
affectations
affected
affecting
affectingly
affection
affectionate
affectionately
affections
affective
affects
afferent
affianced
affidavit
affidavits
affiliate
affiliated
affiliates
affiliating
affiliation
affiliations
affinities
affinity
affirm
affirmation
affirmations
affirmative
affirmatively
affirmed
affirming
affirms
affix
affixed
affixes
affixing
afflict
afflicted
afflicting
affliction
afflictions
afflictive
afflicts
affluence
affluent
afford
affordable
afforded
affording
affords
affricate
affricates
affright
affront
affronted
affronting
affronts
aficionado
afield
afire
aflame
afloat
afoot
afore
aforementioned
aforesaid
aforethought
afoul
afraid
afresh
aft
after
aftereffect
afterglow
afterimage
afterlife
aftermath
aftermost
afternoon
afternoons
aftershock
aftershocks
afterthought
afterthoughts
afterward
afterwards
again
against
agape
agar
agate
agates
age
aged
ageless
agencies
agency
agenda
agendas
agent
agents
ager
agers
ages
agglomerate
agglomerated
agglomerates
agglomeration
agglutinate
agglutinated
agglutinates
agglutinating
agglutination
agglutinin
agglutinins
aggrandize
aggravate
aggravated
aggravates
aggravation
aggregate
aggregated
aggregately
aggregates
aggregating
aggregation
aggregations
aggression
aggressions
aggressive
aggressively
aggressiveness
aggressor
aggressors
aggrieve
aggrieved
aggrieves
aggrieving
aghast
agile
agilely
agility
aging
agitate
agitated
agitates
agitating
agitation
agitations
agitator
agitators
agleam
aglow
agnostic
agnostics
ago
agog
agonies
agonize
agonized
agonizes
agonizing
agonizingly
agony
agrarian
agree
agreeable
agreeably
agreed
agreeing
agreement
agreements
agreer
agreers
agrees
agricultural
agriculturally
agriculture
ague
ah
ahead
aid
aide
aided
aiding
aids
ail
aileron
ailerons
ailing
ailment
ailments
aim
aimed
aimer
aimers
aiming
aimless
aimlessly
aims
air
airbag
airbags
airborne
aircraft
airdrop
airdrops
aired
airer
airers
airfare
airfield
airfields
airflow
airfoil
airfoils
airframe
airframes
airily
airing
airings
airless
airlift
airlifts
airline
airliner
airlines
airlock
airlocks
airmail
airmails
airman
airmen
airplane
airplanes
airport
airports
airs
airship
airships
airspace
airspeed
airstrip
airstrips
airtight
airway
airways
airy
aisle
ajar
akimbo
akin
alabaster
alacrity
alarm
alarmed
alarming
alarmingly
alarmist
alarms
alas
alba
albacore
albatross
albeit
album
albumin
albums
alchemy
alcohol
alcoholic
alcoholics
alcoholism
alcohols
alcove
alcoves
alder
alderman
aldermen
ale
alee
alert
alerted
alertedly
alerter
alerters
alerting
alertly
alertness
alerts
alfalfa
alfresco
alga
algae
algaecide
algebra
algebraic
algebraically
algebras
alginate
algorithm
algorithmic
algorithmically
algorithms
alias
aliased
aliases
aliasing
alibi
alibis
alien
alienate
alienated
alienates
alienating
alienation
aliens
alight
align
aligned
aligning
alignment
alignments
aligns
alike
aliment
aliments
alimony
alive
alkali
alkaline
alkalis
alkaloid
alkaloids
alkyl
all
allay
allayed
allaying
allays
allegation
allegations
allege
alleged
allegedly
alleges
allegiance
allegiances
alleging
allegoric
allegorical
allegorically
allegories
allegory
allegretto
allegrettos
allele
alleles
allemande
allergic
allergies
allergy
alleviate
alleviated
alleviates
alleviating
alleviation
alley
alleys
alleyway
alleyways
alliance
alliances
allied
allies
alligator
alligators
alliteration
alliterations
alliterative
allocatable
allocate
allocated
allocates
allocating
allocation
allocations
allocator
allocators
allophone
allophones
allophonic
allot
allotment
allotments
allots
allotted
allotter
allotting
allow
allowable
allowably
allowance
allowances
allowed
allowing
allows
alloy
alloys
allude
alluded
alludes
alluding
allure
allurement
alluring
allusion
allusions
allusive
allusiveness
ally
allying
alma
almanac
almanacs
almighty
almond
almonds
almoner
almost
alms
almsman
alnico
aloe
aloes
aloft
aloha
alone
aloneness
along
alongside
aloof
aloofness
aloud
alpha
alphabet
alphabetic
alphabetical
alphabetically
alphabetics
alphabetize
alphabetized
alphabetizes
alphabetizing
alphabets
alphanumeric
alpine
already
also
altar
altars
alter
alterable
alteration
alterations
altercation
altercations
altered
alterer
alterers
altering
alternate
alternated
alternately
alternates
alternating
alternation
alternations
alternative
alternatively
alternatives
alternator
alternators
alters
although
altitude
altitudes
altogether
altruism
altruist
altruistic
altruistically
alum
aluminum
alumna
alumnae
alumni
alumnus
alundum
alveolar
alveoli
alveolus
always
am
amain
amalgam
amalgamate
amalgamated
amalgamates
amalgamating
amalgamation
amalgams
amanuensis
amaretto
amass
amassed
amasses
amassing
amateur
amateurish
amateurishness
amateurism
amateurs
amatory
amaze
amazed
amazedly
amazement
amazer
amazers
amazes
amazing
amazingly
ambassador
ambassadors
amber
ambiance
ambidextrous
ambidextrously
ambient
ambiguities
ambiguity
ambiguous
ambiguously
ambition
ambitions
ambitious
ambitiously
ambivalence
ambivalent
ambivalently
amble
ambled
ambler
ambles
ambling
ambrosial
ambulance
ambulances
ambulatory
ambuscade
ambush
ambushed
ambushes
ameliorate
ameliorated
ameliorating
amelioration
amen
amenable
amend
amended
amending
amendment
amendments
amends
amenities
amenity
amenorrhea
americium
amiable
amicable
amicably
amid
amide
amidst
amigo
amino
amiss
amity
ammo
ammonia
ammoniac
ammonium
ammunition
amnesty
amoeba
amoebae
amoebas
amok
among
amongst
amoral
amorality
amorist
amorous
amorphous
amorphously
amortize
amortized
amortizes
amortizing
amount
amounted
amounter
amounters
amounting
amounts
amour
amperage
ampere
amperes
ampersand
ampersands
amphetamine
amphetamines
amphibian
amphibians
amphibious
amphibiously
amphibology
amphitheater
amphitheaters
ample
amplification
amplified
amplifier
amplifiers
amplifies
amplify
amplifying
amplitude
amplitudes
amply
ampoule
ampoules
amputate
amputated
amputates
amputating
amulet
amulets
amuse
amused
amusedly
amusement
amusements
amuser
amusers
amuses
amusing
amusingly
amyl
an
anachronism
anachronisms
anachronistically
anaconda
anacondas
anaerobic
anagram
anagrams
anal
analog
analogical
analogies
analogous
analogously
analogue
analogues
analogy
analyses
analysis
analyst
analysts
analytic
analytical
analytically
analyticities
analyticity
analyzable
analyze
analyzed
analyzer
analyzers
analyzes
analyzing
anaphora
anaphoric
anaphorically
anaplasmosis
anarchic
anarchical
anarchism
anarchist
anarchists
anarchy
anastomoses
anastomosis
anastomotic
anathema
anatomic
anatomical
anatomically
anatomy
ancestor
ancestors
ancestral
ancestry
anchor
anchorage
anchorages
anchored
anchoring
anchorite
anchoritism
anchors
anchovies
anchovy
ancient
anciently
ancients
ancillary
and
anders
anding
anecdotal
anecdote
anecdotes
anechoic
anemia
anemic
anemometer
anemometers
anemometry
anemone
anesthesia
anesthetic
anesthetically
anesthetics
anesthetize
anesthetized
anesthetizes
anesthetizing
anew
angel
angelic
angels
anger
angered
angering
angers
angiography
angle
angled
angler
anglers
angling
angrier
angriest
angrily
angry
angst
angstrom
anguish
anguished
angular
angularly
anhydrous
anhydrously
aniline
animal
animals
animate
animated
animatedly
animately
animateness
animates
animating
animation
animations
animator
animators
animism
animized
animosity
anion
anionic
anions
anise
aniseikonic
anisotropic
anisotropy
ankle
ankles
annal
annals
annex
annexation
annexed
annexes
annexing
annihilate
annihilated
annihilates
annihilating
annihilation
anniversaries
anniversary
annotate
annotated
annotates
annotating
annotation
annotations
announce
announced
announcement
announcements
announcer
announcers
announces
announcing
annoy
annoyance
annoyances
annoyed
annoyer
annoyers
annoying
annoyingly
annoys
annual
annually
annuals
annuity
annul
annular
annuli
annulled
annulling
annulment
annulments
annuls
annulus
annum
annunciate
annunciated
annunciates
annunciating
annunciator
annunciators
anode
anodes
anodize
anodized
anodizes
anoint
anointed
anointing
anoints
anomalies
anomalous
anomalously
anomaly
anomic
anomie
anon
anonymity
anonymous
anonymously
anorexia
another
answer
answerable
answered
answerer
answerers
answering
answers
ant
antagonism
antagonisms
antagonist
antagonistic
antagonistically
antagonists
antagonize
antagonized
antagonizes
antagonizing
antarctic
ante
anteater
anteaters
antecedent
antecedents
antedate
antelope
antelopes
antenna
antennae
antennas
anterior
anthem
anthems
anther
anthologies
anthology
anthracite
anthropological
anthropologically
anthropologist
anthropologists
anthropology
anthropomorphic
anthropomorphically
anti
antibacterial
antibiotic
antibiotics
antibodies
antibody
antic
anticipate
anticipated
anticipates
anticipating
anticipation
anticipations
anticipatory
anticoagulation
anticompetitive
antics
antidisestablishmentarianism
antidote
antidotes
antiformant
antifundamentalist
antigen
antigens
antihistorical
antimicrobial
antimony
antinomian
antinomy
antipathy
antiphonal
antipode
antipodes
antiquarian
antiquarians
antiquate
antiquated
antique
antiques
antiquities
antiquity
antiredeposition
antiresonance
antiresonator
antisemitic
antisemitism
antiseptic
antisera
antiserum
antislavery
antisocial
antisubmarine
antisymmetric
antisymmetry
antithesis
antithetical
antithyroid
antitoxin
antitoxins
antitrust
antler
antlered
ants
anus
anvil
anvils
anxieties
anxiety
anxious
anxiously
any
anybody
anyhow
anymore
anyone
anyplace
anything
anytime
anyway
anywhere
aorta
apace
apart
apartment
apartments
apathetic
apathy
ape
aped
aperiodic
aperiodicity
aperture
apes
apex
aphasia
aphasic
aphelion
aphid
aphids
aphonic
aphorism
aphorisms
apiaries
apiary
apical
apiece
aping
apish
aplenty
aplomb
apocalypse
apocalyptic
apocryphal
apogee
apogees
apologetic
apologetically
apologia
apologies
apologist
apologists
apologize
apologized
apologizes
apologizing
apology
apostate
apostle
apostles
apostolic
apostrophe
apostrophes
apothecary
apothegm
apotheoses
apotheosis
appall
appalled
appalling
appallingly
appanage
apparatus
apparel
appareled
apparent
apparently
apparition
apparitions
appeal
appealed
appealer
appealers
appealing
appealingly
appeals
appear
appearance
appearances
appeared
appearer
appearers
appearing
appears
appease
appeased
appeasement
appeases
appeasing
appellant
appellants
appellate
appellation
append
appendage
appendages
appended
appender
appenders
appendices
appendicitis
appending
appendix
appendixes
appends
appertain
appertains
appetite
appetites
appetizer
appetizing
applaud
applauded
applauding
applauds
applause
apple
applejack
apples
appliance
appliances
applicability
applicable
applicant
applicants
application
applications
applicative
applicatively
applicator
applicators
applied
applier
appliers
applies
applique
apply
applying
appoint
appointed
appointee
appointees
appointer
appointers
appointing
appointive
appointment
appointments
appoints
apportion
apportioned
apportioning
apportionment
apportionments
apportions
apposite
appraisal
appraisals
appraise
appraised
appraiser
appraisers
appraises
appraising
appraisingly
appreciable
appreciably
appreciate
appreciated
appreciates
appreciating
appreciation
appreciations
appreciative
appreciatively
apprehend
apprehended
apprehensible
apprehension
apprehensions
apprehensive
apprehensively
apprehensiveness
apprentice
apprenticed
apprentices
apprenticeship
apprise
apprised
apprises
apprising
approach
approachability
approachable
approached
approacher
approachers
approaches
approaching
approbate
approbation
appropriate
appropriated
appropriately
appropriateness
appropriates
appropriating
appropriation
appropriations
appropriator
appropriators
approval
approvals
approve
approved
approver
approvers
approves
approving
approvingly
approximate
approximated
approximately
approximates
approximating
approximation
approximations
appurtenance
appurtenances
apricot
apricots
apron
aprons
apropos
apse
apsis
apt
aptitude
aptitudes
aptly
aptness
aqua
aquaria
aquarium
aquatic
aqueduct
aqueducts
aqueous
aquifer
aquifers
arabesque
arable
arachnid
arachnids
arbiter
arbiters
arbitrarily
arbitrariness
arbitrary
arbitrate
arbitrated
arbitrates
arbitrating
arbitration
arbitrator
arbitrators
arbor
arboreal
arbors
arc
arcade
arcaded
arcades
arcane
arced
arch
archaic
archaically
archaicness
archaism
archaize
archangel
archangels
archbishop
archdiocese
archdioceses
arched
archenemy
archeological
archeologist
archeology
archers
archery
arches
archetype
archfool
arching
archipelago
archipelagoes
architect
architectonic
architects
architectural
architecturally
architecture
architectures
archival
archive
archived
archiver
archivers
archives
archiving
archivist
archly
arcing
arclike
arcs
arcsine
arctangent
arctic
ardent
ardently
ardor
arduous
arduously
arduousness
are
area
areas
arena
arenas
argon
argonauts
argot
arguable
arguably
argue
argued
arguer
arguers
argues
arguing
argument
argumentation
argumentative
arguments
arid
aridity
aright
arise
arisen
ariser
arises
arising
arisings
aristocracy
aristocrat
aristocratic
aristocratically
aristocrats
arithmetic
arithmetical
arithmetically
arithmetics
arithmetize
arithmetized
arithmetizes
ark
arm
armadillo
armadillos
armament
armaments
armchair
armchairs
armed
armer
armers
armful
armhole
armies
arming
armistice
armload
armor
armored
armorer
armory
armpit
armpits
arms
army
aroma
aromas
aromatic
arose
around
arousal
arouse
aroused
arouses
arousing
arpeggio
arpeggios
arrack
arraign
arraigned
arraigning
arraignment
arraignments
arraigns
arrange
arranged
arrangement
arrangements
arranger
arrangers
arranges
arranging
arrant
array
arrayed
arrays
arrears
arrest
arrested
arrester
arresters
arresting
arrestingly
arrestor
arrestors
arrests
arrival
arrivals
arrive
arrived
arrives
arriving
arrogance
arrogant
arrogantly
arrogate
arrogated
arrogates
arrogating
arrogation
arrow
arrowed
arrowhead
arrowheads
arrows
arroyo
arroyos
arsenal
arsenals
arsenic
arsine
arson
art
arterial
arteries
arteriolar
arteriole
arterioles
arteriosclerosis
artery
artful
artfully
artfulness
arthritis
arthropod
arthropods
artichoke
artichokes
article
articles
articulate
articulated
articulately
articulateness
articulates
articulating
articulation
articulations
articulator
articulators
articulatory
artifact
artifacts
artifice
artificer
artifices
artificial
artificialities
artificiality
artificially
artificialness
artillerist
artillery
artisan
artisans
artist
artistic
artistically
artistry
artists
artless
arts
artwork
as
asbestos
ascend
ascendancy
ascendant
ascended
ascendency
ascendent
ascender
ascenders
ascending
ascends
ascension
ascensions
ascent
ascertain
ascertainable
ascertained
ascertaining
ascertains
ascetic
asceticism
ascetics
ascot
ascribable
ascribe
ascribed
ascribes
ascribing
ascription
aseptic
ash
ashamed
ashamedly
ashen
ashes
ashman
ashore
ashtray
ashtrays
aside
asinine
ask
askance
asked
asker
askers
askew
asking
asks
asleep
asocial
asp
asparagus
aspect
aspects
aspen
aspersion
aspersions
asphalt
asphyxia
aspic
aspirant
aspirants
aspirate
aspirated
aspirates
aspirating
aspiration
aspirations
aspirator
aspirators
aspire
aspired
aspires
aspirin
aspiring
aspirins
ass
assail
assailant
assailants
assailed
assailing
assails
assassin
assassinate
assassinated
assassinates
assassinating
assassination
assassinations
assassins
assault
assaulted
assaulting
assaults
assay
assayed
assaying
assemblage
assemblages
assemble
assembled
assembler
assemblers
assembles
assemblies
assembling
assembly
assent
assented
assenter
assenting
assents
assert
asserted
asserter
asserters
asserting
assertion
assertions
assertive
assertively
assertiveness
asserts
asses
assess
assessed
assesses
assessing
assessment
assessments
assessor
assessors
asset
assets
assiduity
assiduous
assiduously
assign
assignable
assigned
assignee
assignees
assigner
assigners
assigning
assignment
assignments
assigns
assimilate
assimilated
assimilates
assimilating
assimilation
assimilations
assist
assistance
assistances
assistant
assistants
assistantship
assistantships
assisted
assisting
assists
associate
associated
associates
associating
association
associational
associations
associative
associatively
associativity
associator
associators
assonance
assonant
assort
assorted
assortment
assortments
assorts
assuage
assuaged
assuages
assume
assumed
assumes
assuming
assumption
assumptions
assurance
assurances
assure
assured
assuredly
assurer
assurers
assures
assuring
assuringly
astatine
aster
asterisk
asterisks
asteroid
asteroidal
asteroids
asters
asthma
astonish
astonished
astonishes
astonishing
astonishingly
astonishment
astound
astounded
astounding
astounds
astral
astray
astride
astringency
astringent
astrology
astronaut
astronautics
astronauts
astronomer
astronomers
astronomical
astronomically
astronomy
astrophysical
astrophysics
astute
astutely
astuteness
asunder
asylum
asymmetric
asymmetrically
asymmetry
asymptomatically
asymptote
asymptotes
asymptotic
asymptotically
asynchronism
asynchronous
asynchronously
asynchrony
at
atavistic
ate
atemporal
atheism
atheist
atheistic
atheists
atherosclerosis
athlete
athletes
athletic
athleticism
athletics
atlas
atmosphere
atmospheres
atmospheric
atoll
atolls
atom
atomic
atomically
atomics
atomization
atomize
atomized
atomizes
atomizing
atoms
atonal
atonally
atone
atoned
atonement
atones
atop
atrocious
atrociously
atrocities
atrocity
atrophic
atrophied
atrophies
atrophy
atrophying
attach
attache
attached
attacher
attachers
attaches
attaching
attachment
attachments
attack
attackable
attacked
attacker
attackers
attacking
attacks
attain
attainable
attainably
attained
attainer
attainers
attaining
attainment
attainments
attains
attempt
attempted
attempter
attempters
attempting
attempts
attend
attendance
attendances
attendant
attendants
attended
attendee
attendees
attender
attenders
attending
attends
attention
attentional
attentionality
attentions
attentive
attentively
attentiveness
attenuate
attenuated
attenuates
attenuating
attenuation
attenuator
attenuators
attest
attested
attesting
attests
attic
attics
attire
attired
attires
attiring
attitude
attitudes
attitudinal
attorney
attorneys
attract
attracted
attracting
attraction
attractions
attractive
attractively
attractiveness
attractor
attractors
attracts
attributable
attribute
attributed
attributes
attributing
attribution
attributions
attributive
attributively
attrition
attune
attuned
attunes
attuning
atypical
atypically
auburn
auction
auctioneer
auctioneers
audacious
audaciously
audaciousness
audacity
audible
audibly
audience
audiences
audio
audiogram
audiograms
audiological
audiologist
audiologists
audiology
audiometer
audiometers
audiometric
audiometry
audit
audited
auditing
audition
auditioned
auditioning
auditions
auditor
auditorium
auditors
auditory
audits
auger
augers
aught
augment
augmentation
augmented
augmenting
augments
augur
augurs
august
augustly
augustness
aunt
aunts
aura
aural
aurally
auras
aureole
aureomycin
aurora
auscultate
auscultated
auscultates
auscultating
auscultation
auscultations
auspice
auspices
auspicious
auspiciously
austere
austerely
austerity
authentic
authentically
authenticate
authenticated
authenticates
authenticating
authentication
authentications
authenticator
authenticators
authenticity
author
authored
authoring
authoritarian
authoritarianism
authoritative
authoritatively
authorities
authority
authorization
authorizations
authorize
authorized
authorizer
authorizers
authorizes
authorizing
authors
authorship
autism
autistic
auto
autobiographic
autobiographical
autobiographies
autobiography
autocollimator
autocorrelate
autocorrelation
autocracies
autocracy
autocrat
autocratic
autocratically
autocrats
autodecrement
autodecremented
autodecrements
autodialer
autofluorescence
autograph
autographed
autographing
autographs
autoincrement
autoincremented
autoincrements
autoindex
autoindexing
automata
automate
automated
automates
automatic
automatically
automating
automation
automaton
automobile
automobiles
automotive
autonavigator
autonavigators
autonomic
autonomous
autonomously
autonomy
autopilot
autopilots
autopsied
autopsies
autopsy
autoregressive
autos
autosuggestibility
autotransformer
autumn
autumnal
autumns
auxiliaries
auxiliary
avail
availabilities
availability
available
availably
availed
availer
availers
availing
avails
avalanche
avalanched
avalanches
avalanching
avant
avarice
avaricious
avariciously
avenge
avenged
avenger
avenges
avenging
avenue
avenues
aver
average
averaged
averages
averaging
averred
averrer
averring
avers
averse
aversion
aversions
avert
averted
averting
averts
avian
aviaries
aviary
aviation
aviator
aviators
avid
avidity
avidly
avionic
avionics
avocado
avocados
avocation
avocations
avoid
avoidable
avoidably
avoidance
avoided
avoider
avoiders
avoiding
avoids
avouch
avow
avowal
avowed
avows
await
awaited
awaiting
awaits
awake
awaken
awakened
awakening
awakens
awakes
awaking
award
awarded
awarder
awarders
awarding
awards
aware
awareness
awash
away
awe
awed
awesome
awful
awfully
awfulness
awhile
awkward
awkwardly
awkwardness
awl
awls
awning
awnings
awoke
awry
ax
axed
axer
axers
axes
axial
axially
axing
axiological
axiom
axiomatic
axiomatically
axiomatization
axiomatizations
axiomatize
axiomatized
axiomatizes
axiomatizing
axioms
axis
axle
axles
axolotl
axolotls
axon
axons
aye
ayes
azalea
azaleas
azimuth
azimuths
azure
babble
babbled
babbles
babbling
babe
babes
babied
babies
baboon
baboons
baby
babyhood
babying
babyish
babysit
babysitting
baccalaureate
bachelor
bachelors
bacilli
bacillus
back
backache
backaches
backarrow
backbend
backbends
backboard
backbone
backbones
backdrop
backdrops
backed
backer
backers
backfill
backfiring
background
backgrounds
backhand
backing
backlash
backlog
backlogged
backlogs
backorder
backpack
backpacks
backplane
backplanes
backplate
backs
backscatter
backscattered
backscattering
backscatters
backside
backslash
backslashes
backspace
backspaced
backspaces
backspacing
backstage
backstairs
backstitch
backstitched
backstitches
backstitching
backstop
backtrack
backtracked
backtracker
backtrackers
backtracking
backtracks
backup
backups
backward
backwardness
backwards
backwater
backwaters
backwoods
backyard
backyards
bacon
bacteria
bacterial
bacterium
bad
bade
badge
badger
badgered
badgering
badgers
badges
badlands
badly
badminton
badness
baffle
baffled
baffler
bafflers
baffling
bag
bagatelle
bagatelles
bagel
bagels
baggage
bagged
bagger
baggers
bagging
baggy
bagpipe
bagpipes
bags
bah
bail
bailiff
bailiffs
bailing
bait
baited
baiter
baiting
baits
bake
baked
baker
bakeries
bakers
bakery
bakes
baking
baklava
balalaika
balalaikas
balance
balanced
balancer
balancers
balances
balancing
balconies
balcony
bald
balding
baldly
baldness
bale
baleful
baler
bales
balk
balkanized
balkanizing
balked
balkiness
balking
balks
balky
ball
ballad
ballads
ballast
ballasts
balled
baller
ballerina
ballerinas
ballers
ballet
ballets
ballgown
balling
ballistic
ballistics
balloon
ballooned
ballooner
ballooners
ballooning
balloons
ballot
ballots
ballpark
ballparks
ballplayer
ballplayers
ballroom
ballrooms
balls
ballyhoo
balm
balms
balmy
balsa
balsam
balustrade
balustrades
bamboo
ban
banal
banally
banana
bananas
band
bandage
bandaged
bandages
bandaging
banded
bandied
bandies
banding
bandit
bandits
bandpass
bands
bandstand
bandstands
bandwagon
bandwagons
bandwidth
bandwidths
bandy
bandying
bane
baneful
bang
banged
banging
bangle
bangles
bangs
banish
banished
banishes
banishing
banishment
banister
banisters
banjo
banjos
bank
banked
banker
bankers
banking
bankrupt
bankruptcies
bankruptcy
bankrupted
bankrupting
bankrupts
banned
banner
banners
banning
banquet
banqueting
banquetings
banquets
bans
banshee
banshees
bantam
banter
bantered
bantering
banters
baptism
baptismal
baptisms
baptistery
baptistries
baptistry
baptize
baptized
baptizes
baptizing
bar
barb
barbarian
barbarians
barbaric
barbarism
barbarities
barbarity
barbarous
barbarously
barbecue
barbecued
barbecues
barbed
barbell
barbells
barber
barbital
barbiturate
barbiturates
barbs
bard
bards
bare
bared
barefaced
barefoot
barefooted
barely
bareness
barer
bares
barest
barflies
barfly
bargain
bargained
bargaining
bargains
barge
barges
barging
baring
baritone
baritones
barium
bark
barked
barker
barkers
barking
barks
barley
barn
barns
barnstorm
barnstormed
barnstorming
barnstorms
barnyard
barnyards
barometer
barometers
barometric
baron
baroness
baronial
baronies
barons
barony
baroque
baroqueness
barrack
barracks
barrage
barrages
barred
barrel
barrelled
barrelling
barrels
barren
barrenness
barricade
barricades
barrier
barriers
barring
barringer
barrow
bars
bartender
bartenders
barter
bartered
bartering
barters
basal
basalt
base
baseball
baseballs
baseband
baseboard
baseboards
based
baseless
baseline
baselines
basely
baseman
basement
basements
baseness
baser
bases
bash
bashed
bashes
bashful
bashfulness
bashing
basic
basically
basics
basil
basin
basing
basins
basis
bask
basked
basket
basketball
basketballs
baskets
basking
bass
basses
basset
bassinet
bassinets
bastard
bastards
baste
basted
bastes
basting
bastion
bastions
bat
batch
batched
batches
bath
bathe
bathed
bather
bathers
bathes
bathing
bathos
bathrobe
bathrobes
bathroom
bathrooms
baths
bathtub
bathtubs
baton
batons
bats
battalion
battalions
batted
batten
battens
batter
battered
batteries
battering
batters
battery
batting
battle
battled
battlefield
battlefields
battlefront
battlefronts
battleground
battlegrounds
battlement
battlements
battler
battlers
battles
battleship
battleships
battling
bauble
baubles
baud
bauxite
bawdy
bawl
bawled
bawling
bawls
bay
bayed
baying
bayonet
bayonets
bayou
bayous
bays
bazaar
bazaars
be
beach
beached
beaches
beachhead
beachheads
beaching
beacon
beacons
bead
beaded
beading
beadle
beadles
beads
beady
beagle
beagles
beak
beaked
beaker
beakers
beaks
beam
beamed
beamer
beamers
beaming
beams
bean
beanbag
beaned
beaner
beaners
beaning
beans
bear
bearable
bearably
beard
bearded
beardless
beards
bearer
bearers
bearing
bearings
bearish
bears
beast
beastly
beasts
beat
beatable
beatably
beaten
beater
beaters
beatific
beatification
beatify
beating
beatings
beatitude
beatitudes
beatnik
beatniks
beats
beau
beaus
beauteous
beauteously
beauties
beautifications
beautified
beautifier
beautifiers
beautifies
beautiful
beautifully
beautify
beautifying
beauty
beaver
beavers
becalm
becalmed
becalming
becalms
became
because
beck
beckon
beckoned
beckoning
beckons
become
becomes
becoming
becomingly
bed
bedazzle
bedazzled
bedazzlement
bedazzles
bedazzling
bedbug
bedbugs
bedded
bedder
bedders
bedding
bedevil
bedeviled
bedeviling
bedevils
bedfast
bedlam
bedpost
bedposts
bedraggle
bedraggled
bedridden
bedrock
bedroom
bedrooms
beds
bedside
bedspread
bedspreads
bedspring
bedsprings
bedstead
bedsteads
bedtime
bee
beech
beechen
beecher
beef
beefed
beefer
beefers
beefing
beefs
beefsteak
beefy
beehive
beehives
been
beep
beeps
beer
beers
bees
beet
beetle
beetled
beetles
beetling
beets
befall
befallen
befalling
befalls
befell
befit
befits
befitted
befitting
befog
befogged
befogging
before
beforehand
befoul
befouled
befouling
befouls
befriend
befriended
befriending
befriends
befuddle
befuddled
befuddles
befuddling
beg
began
beget
begets
begetting
beggar
beggarly
beggars
beggary
begged
begging
begin
beginner
beginners
beginning
beginnings
begins
begot
begotten
begrudge
begrudged
begrudges
begrudging
begrudgingly
begs
beguile
beguiled
beguiles
beguiling
begun
behalf
behave
behaved
behaves
behaving
behavior
behavioral
behaviorally
behaviorism
behavioristic
behaviors
behead
beheading
beheld
behemoth
behemoths
behest
behind
behold
beholden
beholder
beholders
beholding
beholds
behoove
behooves
beige
being
beings
belabor
belabored
belaboring
belabors
belated
belatedly
belay
belayed
belaying
belays
belch
belched
belches
belching
belfries
belfry
belie
belied
belief
beliefs
belies
believable
believably
believe
believed
believer
believers
believes
believing
belittle
belittled
belittles
belittling
bell
bellboy
bellboys
belle
belles
bellhop
bellhops
bellicose
bellicosity
bellies
belligerence
belligerent
belligerently
belligerents
bellman
bellmen
bellow
bellowed
bellowing
bellows
bells
bellum
bellwether
bellwethers
belly
bellyache
bellyfull
belong
belonged
belonging
belongings
belongs
beloved
below
belt
belted
belting
belts
bely
belying
bemoan
bemoaned
bemoaning
bemoans
bench
benched
benches
benchmark
benchmarking
benchmarks
bend
bendable
benders
bending
bends
beneath
benediction
benedictions
benefactor
benefactors
beneficence
beneficences
beneficent
beneficial
beneficially
beneficiaries
beneficiary
benefit
benefited
benefiting
benefits
benefitted
benefitting
benevolence
benevolent
benighted
benign
benignly
bent
benzene
bequeath
bequeathal
bequeathed
bequeathing
bequeaths
bequest
bequests
berate
berated
berates
berating
bereave
bereaved
bereavement
bereavements
bereaves
bereaving
bereft
beret
berets
beribboned
beriberi
berkelium
berne
berries
berry
berserk
berth
berths
beryl
beryllium
beseech
beseeches
beseeching
beset
besets
besetting
beside
besides
besiege
besieged
besieger
besiegers
besieging
besmirch
besmirched
besmirches
besmirching
besotted
besotter
besotting
besought
bespeak
bespeaks
bespectacled
bespoke
best
bested
bestial
besting
bestir
bestirring
bestow
bestowal
bestowed
bests
bestseller
bestsellers
bestselling
bet
beta
betatron
betel
betide
betray
betrayal
betrayed
betrayer
betraying
betrays
betroth
betrothal
betrothed
bets
better
bettered
bettering
betterment
betterments
betters
betting
between
betwixt
bevel
beveled
beveling
bevels
beverage
beverages
bevy
bewail
bewailed
bewailing
bewails
beware
bewhiskered
bewilder
bewildered
bewildering
bewilderingly
bewilderment
bewilders
bewitch
bewitched
bewitches
bewitching
beyond
biannual
bias
biased
biases
biasing
bib
bibbed
bibbing
bibles
biblical
biblically
bibliographic
bibliographical
bibliographies
bibliography
bibliophile
bibs
bicameral
bicarbonate
bicentennial
bicep
biceps
bicker
bickered
bickering
bickers
biconcave
biconnected
biconvex
bicycle
bicycled
bicycler
bicyclers
bicycles
bicycling
bid
biddable
bidden
bidder
bidders
biddies
bidding
biddy
bide
bidirectional
bids
biennial
biennium
bier
bifocal
bifocals
bifurcate
big
bigger
biggest
bight
bights
bigness
bigot
bigoted
bigotry
bigots
biharmonic
bijection
bijections
bijective
bijectively
bike
bikes
biking
bikini
bikinis
bilabial
bilateral
bilaterally
bile
bilge
bilges
bilinear
bilingual
bilk
bilked
bilking
bilks
bill
billboard
billboards
billed
biller
billers
billet
billeted
billeting
billets
billiard
billiards
billing
billion
billions
billionth
billow
billowed
billows
bills
bimetallic
bimetallism
bimodal
bimolecular
bimonthlies
bimonthly
bin
binaries
binary
binaural
bind
binder
binders
binding
bindings
binds
bing
binge
binges
bingo
binocular
binoculars
binomial
bins
binuclear
biochemical
biochemist
biochemistry
biofeedback
biographer
biographers
biographic
biographical
biographically
biographies
biography
biological
biologically
biologist
biologists
biology
biomedical
biomedicine
biophysical
biophysicist
biophysics
biopsies
biopsy
bioscience
biosphere
biostatistic
biosynthesize
biota
biotic
bipartisan
bipartite
biped
bipeds
biplane
biplanes
bipolar
biracial
birch
birchen
birches
bird
birdbath
birdbaths
birdie
birdied
birdies
birdlike
birds
birefringence
birefringent
birth
birthday
birthdays
birthed
birthplace
birthplaces
birthright
birthrights
births
biscuit
biscuits
bisect
bisected
bisecting
bisection
bisections
bisector
bisectors
bisects
bishop
bishops
bismuth
bison
bisons
bisque
bisques
bistable
bistate
bit
bitch
bitches
bite
biter
biters
bites
biting
bitingly
bitmap
bits
bitten
bitter
bitterer
bitterest
bitterly
bitterness
bitternut
bitterroot
bitters
bittersweet
bitumen
bituminous
bitwise
bivalve
bivalves
bivariate
bivouac
bivouacs
biweekly
bizarre
blab
blabbed
blabbermouth
blabbermouths
blabbing
blabs
black
blackberries
blackberry
blackbird
blackbirds
blackboard
blackboards
blacked
blacken
blackened
blackening
blackens
blacker
blackest
blacking
blackjack
blackjacks
blacklist
blacklisted
blacklisting
blacklists
blackly
blackmail
blackmailed
blackmailer
blackmailers
blackmailing
blackmails
blackness
blackout
blackouts
blacks
blacksmith
blacksmiths
bladder
bladders
blade
blades
blamable
blame
blamed
blameless
blamelessness
blamer
blamers
blames
blameworthy
blaming
blanch
blanched
blanches
blanching
bland
blandly
blandness
blank
blanked
blanker
blankest
blanket
blanketed
blanketer
blanketers
blanketing
blankets
blanking
blankly
blankness
blanks
blare
blared
blares
blaring
blase
blaspheme
blasphemed
blasphemes
blasphemies
blaspheming
blasphemous
blasphemously
blasphemousness
blasphemy
blast
blasted
blaster
blasters
blasting
blasts
blatant
blatantly
blaze
blazed
blazer
blazers
blazes
blazing
bleach
bleached
bleacher
bleachers
bleaches
bleaching
bleak
bleaker
bleakly
bleakness
blear
bleary
bleat
bleating
bleats
bled
bleed
bleeder
bleeding
bleedings
bleeds
blemish
blemishes
blend
blended
blender
blending
blends
bless
blessed
blessing
blessings
blew
blight
blighted
blimp
blimps
blind
blinded
blinder
blinders
blindfold
blindfolded
blindfolding
blindfolds
blinding
blindingly
blindly
blindness
blinds
blink
blinked
blinker
blinkers
blinking
blinks
blip
blips
bliss
blissful
blissfully
blister
blistered
blistering
blisters
blithe
blithely
blitz
blitzes
blitzkrieg
blizzard
blizzards
bloat
bloated
bloater
bloating
bloats
blob
blobs
bloc
block
blockade
blockaded
blockades
blockading
blockage
blockages
blocked
blocker
blockers
blockhouse
blockhouses
blocking
blocks
blocs
bloke
blokes
blond
blonde
blondes
blonds
blood
bloodbath
blooded
bloodhound
bloodhounds
bloodied
bloodiest
bloodless
bloods
bloodshed
bloodshot
bloodstain
bloodstained
bloodstains
bloodstream
bloody
bloom
bloomed
bloomers
blooming
blooms
blooper
blossom
blossomed
blossoms
blot
blots
blotted
blotting
blouse
blouses
blow
blower
blowers
blowfish
blowing
blown
blowout
blows
blowup
blubber
bludgeon
bludgeoned
bludgeoning
bludgeons
blue
blueberries
blueberry
bluebird
bluebirds
bluebonnet
bluebonnets
bluefish
blueness
blueprint
blueprints
bluer
blues
bluest
bluestocking
bluff
bluffing
bluffs
bluing
bluish
blunder
blunderbuss
blundered
blundering
blunderings
blunders
blunt
blunted
blunter
bluntest
blunting
bluntly
bluntness
blunts
blur
blurb
blurred
blurring
blurry
blurs
blurt
blurted
blurting
blurts
blush
blushed
blushes
blushing
bluster
blustered
blustering
blusters
blustery
boa
boar
board
boarded
boarder
boarders
boarding
boardinghouse
boardinghouses
boards
boast
boasted
boaster
boasters
boastful
boastfully
boasting
boastings
boasts
boat
boater
boaters
boathouse
boathouses
boating
boatload
boatloads
boatman
boatmen
boats
boatsman
boatsmen
boatswain
boatswains
boatyard
boatyards
bob
bobbed
bobbin
bobbing
bobbins
bobby
bobolink
bobolinks
bobs
bobwhite
bobwhites
bode
bodes
bodice
bodied
bodies
bodily
body
bodybuilder
bodybuilders
bodybuilding
bodyguard
bodyguards
bodyweight
bog
bogeymen
bogged
boggle
boggled
boggles
boggling
bogs
bogus
boil
boiled
boiler
boilerplate
boilers
boiling
boils
boisterous
boisterously
bold
bolder
boldest
boldface
boldly
boldness
boll
bolster
bolstered
bolstering
bolsters
bolt
bolted
bolting
bolts
bomb
bombard
bombarded
bombarding
bombardment
bombards
bombast
bombastic
bombed
bomber
bombers
bombing
bombings
bombproof
bombs
bonanza
bonanzas
bond
bondage
bonded
bonder
bonders
bonding
bonds
bondsman
bondsmen
bone
boned
boner
boners
bones
bonfire
bonfires
bong
boning
bonnet
bonneted
bonnets
bonny
bonus
bonuses
bony
boo
boob
booboo
booby
book
bookcase
bookcases
booked
booker
bookers
bookie
bookies
booking
bookings
bookish
bookkeeper
bookkeepers
bookkeeping
booklet
booklets
bookmark
books
bookseller
booksellers
bookshelf
bookshelves
bookstore
bookstores
bookworm
boolean
boom
boomed
boomerang
boomerangs
booming
booms
boon
boor
boorish
boors
boos
boost
boosted
booster
boosting
boosts
boot
bootable
booted
booth
booths
booting
bootleg
bootlegged
bootlegger
bootleggers
bootlegging
bootlegs
boots
bootstrap
bootstrapped
bootstrapping
bootstraps
booty
booze
borate
borates
borax
bordello
bordellos
border
bordered
bordering
borderings
borderland
borderlands
borderline
borders
bore
bored
boredom
borer
bores
boric
boring
born
borne
boron
borough
boroughs
borrow
borrowed
borrower
borrowers
borrowing
borrows
bosom
bosoms
boss
bossed
bosses
bosun
botanical
botanist
botanists
botany
botch
botched
botcher
botchers
botches
botching
both
bother
bothered
bothering
bothers
bothersome
bottle
bottled
bottleneck
bottlenecks
bottler
bottlers
bottles
bottling
bottom
bottomed
bottoming
bottomless
bottoms
botulinus
botulism
bouffant
bough
boughs
bought
boulder
boulders
boulevard
boulevards
bounce
bounced
bouncer
bounces
bouncing
bouncy
bound
boundaries
boundary
bounded
bounden
bounding
boundless
boundlessness
bounds
bounteous
bounteously
bounties
bountiful
bounty
bouquet
bouquets
bourbon
bourgeois
bourgeoisie
boustrophedon
boustrophedonic
bout
boutique
bouts
bovine
bovines
bow
bowdlerize
bowdlerized
bowdlerizes
bowdlerizing
bowed
bowel
bowels
bower
bowers
bowing
bowl
bowled
bowler
bowlers
bowline
bowlines
bowling
bowls
bowman
bows
bowstring
bowstrings
box
boxcar
boxcars
boxed
boxer
boxers
boxes
boxing
boxtop
boxtops
boxwood
boy
boycott
boycotted
boycotts
boyfriend
boyfriends
boyhood
boyish
boyishness
boys
bra
brace
braced
bracelet
bracelets
braces
bracing
bracket
bracketed
bracketing
brackets
brackish
brae
braes
brag
bragged
bragger
bragging
brags
braid
braided
braiding
braids
brain
brainchild
brained
braining
brains
brainstem
brainstems
brainstorm
brainstorms
brainwash
brainwashed
brainwashes
brainwashing
brainy
brake
braked
brakeman
brakes
braking
bramble
brambles
brambly
bran
branch
branched
branches
branching
branchings
brand
branded
branding
brandish
brandishes
brandishing
brands
brandy
brandywine
bras
brash
brashly
brashness
brass
brasses
brassiere
brassy
brat
brats
bravado
brave
braved
bravely
braveness
braver
bravery
braves
bravest
braving
bravo
bravos
brawl
brawler
brawling
brawn
bray
brayed
brayer
braying
brays
braze
brazed
brazen
brazenly
brazenness
brazes
brazier
braziers
brazing
breach
breached
breacher
breachers
breaches
breaching
bread
breadboard
breadboards
breadbox
breadboxes
breaded
breading
breads
breadth
breadwinner
breadwinners
break
breakable
breakables
breakage
breakaway
breakdown
breakdowns
breaker
breakers
breakfast
breakfasted
breakfaster
breakfasters
breakfasting
breakfasts
breaking
breakpoint
breakpoints
breaks
breakthrough
breakthroughes
breakthroughs
breakup
breakwater
breakwaters
breast
breasted
breasts
breastwork
breastworks
breath
breathable
breathe
breathed
breather
breathers
breathes
breathing
breathless
breathlessly
breaths
breathtaking
breathtakingly
breathy
bred
breech
breeches
breed
breeder
breeding
breeds
breeze
breezes
breezily
breezy
bremsstrahlung
brethren
breve
brevet
breveted
breveting
brevets
brevity
brew
brewed
brewer
breweries
brewers
brewery
brewing
brews
briar
briars
bribe
bribed
briber
bribers
bribery
bribes
bribing
brick
brickbat
bricked
bricker
bricklayer
bricklayers
bricklaying
bricks
bridal
bride
bridegroom
brides
bridesmaid
bridesmaids
bridge
bridgeable
bridged
bridgehead
bridgeheads
bridges
bridgework
bridging
bridle
bridled
bridles
bridling
brief
briefcase
briefcases
briefed
briefer
briefest
briefing
briefings
briefly
briefness
briefs
brier
brig
brigade
brigades
brigadier
brigadiers
brigantine
bright
brighten
brightened
brightener
brighteners
brightening
brightens
brighter
brightest
brightly
brightness
brigs
brilliance
brilliancy
brilliant
brilliantly
brim
brimful
brimmed
brimming
brimstone
brindle
brindled
brine
bring
bringer
bringers
bringing
brings
brink
brinkmanship
briny
brisk
brisker
briskly
briskness
bristle
bristled
bristles
bristling
britches
brittle
brittleness
broach
broached
broaches
broaching
broad
broadband
broadcast
broadcasted
broadcaster
broadcasters
broadcasting
broadcastings
broadcasts
broaden
broadened
broadener
broadeners
broadening
broadenings
broadens
broader
broadest
broadly
broadness
broadside
brocade
brocaded
broccoli
brochure
brochures
broil
broiled
broiler
broilers
broiling
broils
broke
broken
brokenly
brokenness
broker
brokerage
brokers
bromide
bromides
bromine
bronchi
bronchial
bronchiole
bronchioles
bronchitis
bronchus
bronze
bronzed
bronzes
brooch
brooches
brood
brooder
brooding
broods
brook
brooked
brooks
broom
brooms
broomstick
broomsticks
broth
brothel
brothels
brother
brotherhood
brotherliness
brotherly
brothers
brought
brow
browbeat
browbeaten
browbeating
browbeats
brown
browned
browner
brownest
brownie
brownies
browning
brownish
brownness
browns
brows
browse
browsing
bruise
bruised
bruises
bruising
brunch
brunches
brunette
brunt
brush
brushed
brushes
brushfire
brushfires
brushing
brushlike
brushy
brusque
brusquely
brutal
brutalities
brutality
brutalize
brutalized
brutalizes
brutalizing
brutally
brute
brutes
brutish
bubble
bubbled
bubbles
bubbling
bubbly
buck
buckboard
buckboards
bucked
bucket
buckets
bucking
buckle
buckled
buckler
buckles
buckling
bucks
buckshot
buckskin
buckskins
buckwheat
bucolic
bud
budded
buddies
budding
buddy
budge
budged
budges
budget
budgetary
budgeted
budgeter
budgeters
budgeting
budgets
budging
buds
buff
buffalo
buffaloes
buffer
buffered
buffering
buffers
buffet
buffeted
buffeting
buffetings
buffets
buffoon
buffoons
buffs
bug
bugaboo
bugeyed
bugged
bugger
buggers
buggies
bugging
buggy
bugle
bugled
bugler
bugles
bugling
bugs
build
builder
builders
building
buildings
builds
buildup
buildups
built
builtin
bulb
bulbs
bulge
bulged
bulging
bulk
bulked
bulkhead
bulkheads
bulks
bulky
bull
bulldog
bulldogs
bulldoze
bulldozed
bulldozer
bulldozes
bulldozing
bulled
bullet
bulletin
bulletins
bullets
bullfrog
bullied
bullies
bulling
bullion
bullish
bullock
bulls
bullseye
bully
bullying
bulwark
bum
bumble
bumblebee
bumblebees
bumbled
bumbler
bumblers
bumbles
bumbling
bummed
bumming
bump
bumped
bumper
bumpers
bumping
bumps
bumptious
bumptiously
bumptiousness
bums
bun
bunch
bunched
bunches
bunching
bundle
bundled
bundles
bundling
bungalow
bungalows
bungle
bungled
bungler
bunglers
bungles
bungling
bunion
bunions
bunk
bunker
bunkered
bunkers
bunkhouse
bunkhouses
bunkmate
bunkmates
bunks
bunnies
bunny
buns
bunt
bunted
bunter
bunters
bunting
bunts
buoy
buoyancy
buoyant
buoyed
buoys
burden
burdened
burdening
burdens
burdensome
bureau
bureaucracies
bureaucracy
bureaucrat
bureaucratic
bureaucrats
bureaus
burgeon
burgeoned
burgeoning
burgess
burgesses
burgher
burghers
burglar
burglaries
burglarize
burglarized
burglarizes
burglarizing
burglarproof
burglarproofed
burglarproofing
burglarproofs
burglars
burglary
burial
buried
buries
burl
burlesque
burlesques
burly
burn
burned
burner
burners
burning
burningly
burnings
burnish
burnished
burnishes
burnishing
burns
burnt
burntly
burntness
burp
burped
burping
burps
burrow
burrowed
burrower
burrowing
burrows
burrs
bursa
bursitis
burst
burstiness
bursting
bursts
bursty
bury
burying
bus
busboy
busboys
bused
buses
bush
bushel
bushels
bushes
bushing
bushwhack
bushwhacked
bushwhacking
bushwhacks
bushy
busied
busier
busiest
busily
business
businesses
businesslike
businessman
businessmen
busing
buss
bussed
busses
bussing
bust
bustard
bustards
busted
buster
bustle
bustling
busts
busy
but
butane
butcher
butchered
butchers
butchery
butler
butlers
butt
butte
butted
butter
butterball
buttercup
buttered
butterer
butterers
butterfat
butterflies
butterfly
buttering
buttermilk
butternut
butters
buttery
buttes
butting
buttock
buttocks
button
buttoned
buttonhole
buttonholes
buttoning
buttons
buttress
buttressed
buttresses
buttressing
butts
butyl
butyrate
buxom
buy
buyer
buyers
buying
buys
buzz
buzzards
buzzed
buzzer
buzzes
buzzing
buzzword
buzzwords
buzzy
by
bye
bygone
bylaw
bylaws
byline
bylines
bypass
bypassed
bypasses
bypassing
byproduct
byproducts
bystander
bystanders
byte
bytes
byway
byways
byword
bywords
cab
cabal
cabana
cabaret
cabbage
cabbages
cabdriver
cabin
cabinet
cabinets
cabins
cable
cabled
cables
cabling
caboose
cabs
cache
cached
caches
caching
cackle
cackled
cackler
cackles
cackling
cacti
cactus
cadaver
cadence
cadenced
cadres
cafe
cafes
cafeteria
cage
caged
cager
cagers
cages
caging
caiman
cairn
cajole
cajoled
cajoles
cajoling
cake
caked
cakes
caking
calamities
calamitous
calamity
calcify
calcium
calculate
calculated
calculates
calculating
calculation
calculations
calculative
calculator
calculators
calculi
calculus
caldera
calendar
calendars
calf
calfskin
caliber
calibers
calibrate
calibrated
calibrates
calibrating
calibration
calibrations
calico
caliph
caliphs
call
callable
called
caller
callers
calling
calliope
callous
calloused
callously
callousness
calls
callus
calm
calmed
calmer
calmest
calming
calmingly
calmly
calmness
calms
caloric
calorie
calories
calorimeter
calorimetric
calorimetry
calumny
calve
calves
calypso
cam
came
camel
camels
camera
cameraman
cameramen
cameras
camouflage
camouflaged
camouflages
camouflaging
camp
campaign
campaigned
campaigner
campaigners
campaigning
campaigns
camped
camper
campers
campfire
campground
camping
camps
campsite
campus
campuses
can
canal
canals
canaries
canary
cancel
canceled
canceling
cancellation
cancellations
cancels
cancer
cancerous
cancers
candid
candidacy
candidate
candidates
candidly
candidness
candied
candies
candle
candlelight
candler
candles
candlestick
candlesticks
candor
candy
cane
caner
canine
canister
canker
cankerworm
cannabis
canned
cannel
canner
canners
cannery
cannibal
cannibalize
cannibalized
cannibalizes
cannibalizing
cannibals
canning
cannister
cannisters
cannon
cannonball
cannons
cannot
canny
canoe
canoes
canon
canonic
canonical
canonicalization
canonicalize
canonicalized
canonicalizes
canonicalizing
canonically
canonicals
canons
canopy
cans
cant
cantaloupe
cantankerous
cantankerously
canteen
cantilever
canto
canton
cantons
cantor
cantors
canvas
canvases
canvass
canvassed
canvasser
canvassers
canvasses
canvassing
canyon
canyons
cap
capabilities
capability
capable
capably
capacious
capaciously
capaciousness
capacitance
capacitances
capacities
capacitive
capacitor
capacitors
capacity
cape
caper
capers
capes
capillary
capita
capital
capitalism
capitalist
capitalists
capitalization
capitalizations
capitalize
capitalized
capitalizer
capitalizers
capitalizes
capitalizing
capitally
capitals
capitol
capitols
capped
capping
caprice
capricious
capriciously
capriciousness
caps
capstan
capstone
capsule
captain
captained
captaining
captains
caption
captions
captivate
captivated
captivates
captivating
captivation
captive
captives
captivity
captor
captors
capture
captured
capturer
capturers
captures
capturing
capybara
car
caramel
caravan
caravans
caraway
carbohydrate
carbolic
carbon
carbonate
carbonates
carbonation
carbonic
carbonization
carbonize
carbonized
carbonizer
carbonizers
carbonizes
carbonizing
carbons
carborundum
carbuncle
carcass
carcasses
carcinogen
carcinogenic
carcinoma
card
cardboard
carder
cardiac
cardinal
cardinalities
cardinality
cardinally
cardinals
cardiology
cardiovascular
cards
care
cared
careen
career
careers
carefree
careful
carefully
carefulness
careless
carelessly
carelessness
cares
caress
caressed
caresser
caresses
caressing
caret
caretaker
cargo
cargoes
caribou
caricature
caring
carload
carnage
carnal
carnation
carnival
carnivals
carnivorous
carnivorously
carol
carols
carp
carpenter
carpenters
carpentry
carpet
carpeted
carpeting
carpets
carport
carriage
carriages
carried
carrier
carriers
carries
carrion
carrot
carrots
carry
carrying
carryover
carryovers
cars
cart
carted
cartel
carter
carters
cartilage
carting
cartographer
cartographic
cartography
carton
cartons
cartoon
cartoons
cartridge
cartridges
carts
cartwheel
carve
carved
carver
carves
carving
carvings
cascadable
cascade
cascaded
cascades
cascading
case
cased
casement
casements
cases
casework
cash
cashed
casher
cashers
cashes
cashew
cashier
cashiers
cashing
cashmere
casing
casings
casino
cask
casket
caskets
casks
casserole
casseroles
cassette
cassock
cast
caste
caster
casters
castes
castigate
casting
castle
castled
castles
castor
casts
casual
casually
casualness
casuals
casualties
casualty
cat
cataclysmic
catalog
cataloged
cataloger
cataloging
catalogs
catalyst
catalysts
catalytic
catapult
cataract
catastrophe
catastrophes
catastrophic
catch
catchable
catcher
catchers
catches
catching
categorical
categorically
categories
categorization
categorize
categorized
categorizer
categorizers
categorizes
categorizing
category
cater
catered
caterer
catering
caterpillar
caterpillars
caters
cathedral
cathedrals
catheter
catheters
cathode
cathodes
catlike
catnip
cats
catsup
cattail
cattle
cattleman
cattlemen
caucus
caught
cauldron
cauldrons
cauliflower
caulk
causal
causality
causally
causation
causations
cause
caused
causer
causes
causeway
causeways
causing
caustic
causticly
caustics
caution
cautioned
cautioner
cautioners
cautioning
cautionings
cautions
cautious
cautiously
cautiousness
cavalier
cavalierly
cavalierness
cavalry
cave
caveat
caveats
caved
caveman
cavemen
cavern
cavernous
caverns
caves
caviar
cavil
caving
cavities
cavity
caw
cawing
cease
ceased
ceaseless
ceaselessly
ceaselessness
ceases
ceasing
cedar
cede
ceded
ceding
ceiling
ceilings
celebrate
celebrated
celebrates
celebrating
celebration
celebrations
celebrities
celebrity
celerity
celery
celestial
celestially
cell
cellar
cellars
celled
cellist
cellists
cellophane
cells
cellular
cellulose
cement
cemented
cementing
cements
cemeteries
cemetery
censor
censored
censoring
censors
censorship
censure
censured
censurer
censures
census
censuses
cent
centaur
centenary
centennial
center
centered
centering
centerpiece
centerpieces
centers
centigrade
centimeter
centimeters
centipede
centipedes
central
centralism
centralist
centralization
centralize
centralized
centralizes
centralizing
centrally
centrifugal
centrifuge
centripetal
centrist
centroid
cents
centuries
century
ceramic
cereal
cereals
cerebellum
cerebral
ceremonial
ceremonially
ceremonialness
ceremonies
ceremony
certain
certainly
certainties
certainty
certifiable
certificate
certificates
certification
certifications
certified
certifier
certifiers
certifies
certify
certifying
cessation
cessations
chafe
chafer
chaff
chaffer
chaffing
chafing
chagrin
chain
chained
chaining
chains
chair
chaired
chairing
chairlady
chairman
chairmen
chairperson
chairpersons
chairs
chairwoman
chairwomen
chalice
chalices
chalk
chalked
chalking
chalks
challenge
challenged
challenger
challengers
challenges
challenging
chamber
chambered
chamberlain
chamberlains
chambermaid
chameleon
champagne
champion
championed
championing
champions
championship
championships
chance
chanced
chancellor
chancery
chances
chancing
chandelier
chandeliers
change
changeability
changeable
changeably
changed
changeover
changer
changers
changes
changing
channel
channeled
channeling
channelled
channeller
channellers
channelling
channels
chant
chanted
chanter
chanticleer
chanticleers
chanting
chants
chaos
chaotic
chap
chapel
chapels
chaperon
chaperone
chaperoned
chaplain
chaplains
chaps
chapter
chapters
char
character
characteristic
characteristically
characteristics
characterizable
characterization
characterizations
characterize
characterized
characterizer
characterizers
characterizes
characterizing
characters
charcoal
charcoaled
charge
chargeable
charged
charger
chargers
charges
charging
chariot
chariots
charisma
charismatic
charitable
charitableness
charities
charity
charm
charmed
charmer
charmers
charming
charmingly
charms
chars
chart
chartable
charted
charter
chartered
chartering
charters
charting
chartings
chartreuse
charts
chase
chased
chaser
chasers
chases
chasing
chasm
chasms
chassis
chaste
chastely
chasteness
chastise
chastised
chastiser
chastisers
chastises
chastising
chastity
chat
chateau
chateaus
chattel
chatter
chattered
chatterer
chattering
chatters
chatting
chatty
chauffeur
chauffeured
cheap
cheapen
cheapened
cheapening
cheapens
cheaper
cheapest
cheaply
cheapness
cheat
cheated
cheater
cheaters
cheating
cheats
check
checkable
checkbook
checkbooks
checked
checker
checkerboard
checkerboarded
checkerboarding
checkers
checking
checklist
checkout
checkpoint
checkpoints
checks
checksum
checksummed
checksumming
checksums
checkup
cheek
cheekbone
cheeks
cheeky
cheer
cheered
cheerer
cheerful
cheerfully
cheerfulness
cheerily
cheeriness
cheering
cheerleader
cheerless
cheerlessly
cheerlessness
cheers
cheery
cheese
cheesecloth
cheeses
cheesy
cheetah
chef
chefs
chemical
chemically
chemicals
chemise
chemist
chemistries
chemistry
chemists
cherish
cherished
cherishes
cherishing
cherries
cherry
cherub
cherubim
cherubs
chess
chest
chestnut
chestnuts
chests
chew
chewed
chewer
chewers
chewing
chews
chic
chicanery
chick
chickadee
chickadees
chicken
chickens
chicks
chide
chided
chides
chiding
chief
chiefly
chiefs
chieftain
chieftains
chiffon
child
childbirth
childhood
childish
childishly
childishness
childlike
children
chili
chill
chilled
chiller
chillers
chillier
chilliness
chilling
chillingly
chills
chilly
chime
chimera
chimes
chimney
chimneys
chimpanzee
chin
chink
chinked
chinks
chinned
chinner
chinners
chinning
chins
chintz
chip
chipmunk
chipmunks
chips
chiropractor
chirp
chirped
chirping
chirps
chisel
chiseled
chiseler
chisels
chit
chivalrous
chivalrously
chivalrousness
chivalry
chlorine
chloroform
chlorophyll
chloroplast
chloroplasts
chock
chocks
chocolate
chocolates
choice
choices
choicest
choir
choirs
choke
choked
choker
chokers
chokes
choking
cholera
choose
chooser
choosers
chooses
choosing
chop
chopped
chopper
choppers
chopping
choppy
chops
choral
chord
chordate
chorded
chording
chords
chore
choreograph
choreography
chores
choring
chortle
chorus
chorused
choruses
chose
chosen
chowder
christen
christened
christening
christens
chromatogram
chromatograph
chromatography
chrome
chromium
chromosphere
chronic
chronicle
chronicled
chronicler
chroniclers
chronicles
chronograph
chronography
chronological
chronologically
chronologies
chronology
chrysanthemum
chubbier
chubbiest
chubbiness
chubby
chuck
chuckle
chuckled
chuckles
chucks
chum
chunk
chunks
chunky
church
churches
churchgoer
churchgoing
churchly
churchman
churchmen
churchwoman
churchwomen
churchyard
churchyards
churn
churned
churning
churns
chute
chutes
chutzpah
cicada
cider
cigar
cigarette
cigarettes
cigars
cilia
cinder
cinders
cinema
cinematic
cinnamon
cipher
ciphers
ciphertext
ciphertexts
circa
circle
circled
circles
circlet
circling
circuit
circuitous
circuitously
circuitry
circuits
circulant
circular
circularity
circularly
circulate
circulated
circulates
circulating
circulation
circumcise
circumcision
circumference
circumflex
circumlocution
circumlocutions
circumnavigate
circumnavigated
circumnavigates
circumpolar
circumscribe
circumscribed
circumscribing
circumscription
circumspect
circumspection
circumspectly
circumstance
circumstanced
circumstances
circumstantial
circumstantially
circumvent
circumventable
circumvented
circumventing
circumvents
circus
circuses
cistern
cisterns
citadel
citadels
citation
citations
cite
cited
cites
cities
citing
citizen
citizens
citizenship
citrus
city
cityscape
citywide
civet
civic
civics
civil
civilian
civilians
civility
civilization
civilizations
civilize
civilized
civilizes
civilizing
civilly
clad
cladding
claim
claimable
claimant
claimants
claimed
claiming
claims
clairvoyant
clairvoyantly
clam
clamber
clambered
clambering
clambers
clamor
clamored
clamoring
clamorous
clamors
clamp
clamped
clamping
clamps
clams
clan
clandestine
clang
clanged
clanging
clangs
clank
clannish
clap
clapboard
clapping
claps
clarification
clarifications
clarified
clarifies
clarify
clarifying
clarinet
clarity
clash
clashed
clashes
clashing
clasp
clasped
clasping
clasps
class
classed
classes
classic
classical
classically
classics
classifiable
classification
classifications
classified
classifier
classifiers
classifies
classify
classifying
classmate
classmates
classroom
classrooms
classy
clatter
clattered
clattering
clause
clauses
claustrophobia
claustrophobic
claw
clawed
clawing
claws
clay
clays
clean
cleaned
cleaner
cleaners
cleanest
cleaning
cleanliness
cleanly
cleanness
cleans
cleanse
cleansed
cleanser
cleansers
cleanses
cleansing
cleanup
clear
clearance
clearances
cleared
clearer
clearest
clearing
clearings
clearly
clearness
clears
cleavage
cleave
cleaved
cleaver
cleavers
cleaves
cleaving
cleft
clefts
clemency
clement
clench
clenched
clenches
clergy
clergyman
clergymen
clerical
clerk
clerked
clerking
clerks
clever
cleverer
cleverest
cleverly
cleverness
cliche
cliches
click
clicked
clicking
clicks
client
clientele
clients
cliff
cliffs
climate
climates
climatic
climatically
climatology
climax
climaxed
climaxes
climb
climbed
climber
climbers
climbing
climbs
clime
climes
clinch
clinched
clincher
clinches
cling
clinging
clings
clinic
clinical
clinically
clinician
clinics
clink
clinked
clinker
clip
clipboard
clipped
clipper
clippers
clipping
clippings
clips
clique
cliques
clitoris
cloak
cloakroom
cloaks
clobber
clobbered
clobbering
clobbers
clock
clocked
clocker
clockers
clocking
clockings
clocks
clockwatcher
clockwise
clockwork
clod
clods
clog
clogged
clogging
clogs
cloister
cloisters
clone
cloned
clones
cloning
close
closed
closely
closeness
closenesses
closer
closers
closes
closest
closet
closeted
closets
closeup
closing
closure
closures
clot
cloth
clothe
clothed
clothes
clotheshorse
clothesline
clothing
clotting
cloture
cloud
cloudburst
clouded
cloudier
cloudiest
cloudiness
clouding
cloudless
clouds
cloudy
clout
clove
clover
cloves
clown
clowning
clowns
club
clubbed
clubbing
clubhouse
clubroom
clubs
cluck
clucked
clucking
clucks
clue
clues
clump
clumped
clumping
clumps
clumsily
clumsiness
clumsy
clung
cluster
clustered
clustering
clusterings
clusters
clutch
clutched
clutches
clutching
clutter
cluttered
cluttering
clutters
coach
coached
coacher
coaches
coaching
coachman
coachmen
coagulate
coal
coalesce
coalesced
coalesces
coalescing
coalition
coals
coarse
coarsely
coarsen
coarsened
coarseness
coarser
coarsest
coast
coastal
coasted
coaster
coasters
coasting
coastline
coasts
coat
coated
coating
coatings
coats
coattail
coauthor
coax
coaxed
coaxer
coaxes
coaxial
coaxing
cobalt
cobble
cobbler
cobblers
cobblestone
cobra
cobweb
cobwebs
coca
cocaine
cock
cocked
cocking
cockpit
cockroach
cocks
cocktail
cocktails
cocky
coco
cocoa
coconut
coconuts
cocoon
cocoons
cod
coddle
code
coded
codeine
coder
coders
codes
codeword
codewords
codfish
codicil
codification
codifications
codified
codifier
codifiers
codifies
codify
codifying
coding
codings
codpiece
coed
coeditor
coeducation
coefficient
coefficients
coequal
coerce
coerced
coerces
coercible
coercing
coercion
coercive
coexist
coexisted
coexistence
coexisting
coexists
cofactor
coffee
coffeecup
coffeepot
coffees
coffer
coffers
coffin
coffins
cog
cogent
cogently
cogitate
cogitated
cogitates
cogitating
cogitation
cognac
cognition
cognitive
cognitively
cognizance
cognizant
cogs
cohabitation
cohabitations
cohere
cohered
coherence
coherent
coherently
coheres
cohering
cohesion
cohesive
cohesively
cohesiveness
cohort
coil
coiled
coiling
coils
coin
coinage
coincide
coincided
coincidence
coincidences
coincident
coincidental
coincides
coinciding
coined
coiner
coining
coins
coke
cokes
colander
cold
colder
coldest
coldly
coldness
colds
colicky
coliform
coliseum
collaborate
collaborated
collaborates
collaborating
collaboration
collaborations
collaborative
collaborator
collaborators
collagen
collapse
collapsed
collapses
collapsible
collapsing
collar
collarbone
collared
collaring
collars
collate
collateral
colleague
colleagues
collect
collected
collectible
collecting
collection
collections
collective
collectively
collectives
collector
collectors
collects
college
colleges
collegian
collegiate
collide
collided
collides
colliding
collie
collies
collision
collisions
colloidal
colloquia
colloquial
colloquium
colloquy
collusion
colon
colonel
colonels
colonial
colonially
colonials
colonies
colonist
colonists
colonization
colonize
colonized
colonizer
colonizers
colonizes
colonizing
colons
colony
color
colored
colorer
colorers
colorful
coloring
colorings
colorless
colors
colossal
colt
colts
column
columnize
columnized
columnizes
columnizing
columns
comb
combat
combatant
combatants
combated
combating
combative
combats
combed
comber
combers
combination
combinational
combinations
combinator
combinatorial
combinatorially
combinatoric
combinatorics
combinators
combine
combined
combines
combing
combings
combining
combs
combustible
combustion
come
comeback
comedian
comedians
comedic
comedies
comedy
comeliness
comely
comer
comers
comes
comestible
comet
cometary
comets
comfort
comfortabilities
comfortability
comfortable
comfortably
comforted
comforter
comforters
comforting
comfortingly
comforts
comic
comical
comically
comics
coming
comings
comma
command
commandant
commandants
commanded
commandeer
commander
commanders
commanding
commandingly
commandment
commandments
commando
commands
commas
commemorate
commemorated
commemorates
commemorating
commemoration
commemorative
commence
commenced
commencement
commencements
commences
commencing
commend
commendation
commendations
commended
commending
commends
commensurate
comment
commentaries
commentary
commentator
commentators
commented
commenting
comments
commerce
commercial
commercially
commercialness
commercials
commission
commissioned
commissioner
commissioners
commissioning
commissions
commit
commitment
commitments
commits
committed
committee
committeeman
committeemen
committees
committeewoman
committeewomen
committing
commodities
commodity
commodore
commodores
common
commonalities
commonality
commoner
commoners
commonest
commonly
commonness
commonplace
commonplaces
commons
commonwealth
commonwealths
commotion
communal
communally
commune
communes
communicant
communicants
communicate
communicated
communicates
communicating
communication
communications
communicative
communicator
communicators
communion
communist
communists
communities
community
commutative
commutativity
commute
commuted
commuter
commuters
commutes
commuting
compact
compacted
compacter
compactest
compacting
compaction
compactly
compactness
compactor
compactors
compacts
companies
companion
companionable
companions
companionship
company
comparability
comparable
comparably
comparative
comparatively
comparatives
comparator
comparators
compare
compared
compares
comparing
comparison
comparisons
compartment
compartmentalize
compartmentalized
compartmentalizes
compartmentalizing
compartmented
compartments
compass
compassion
compassionate
compassionately
compatibilities
compatibility
compatible
compatibles
compatibly
compel
compelled
compelling
compellingly
compels
compendium
compensate
compensated
compensates
compensating
compensation
compensations
compensatory
compete
competed
competence
competency
competent
competently
competes
competing
competition
competitions
competitive
competitively
competitor
competitors
compilation
compilations
compile
compiled
compiler
compilers
compiles
compiling
complacency
complain
complained
complainer
complainers
complaining
complains
complaint
complaints
complement
complementary
complemented
complementer
complementers
complementing
complements
complete
completed
completely
completeness
completes
completing
completion
completions
complex
complexes
complexion
complexities
complexity
complexly
compliance
compliant
complicate
complicated
complicates
complicating
complication
complications
complicator
complicators
complicity
complied
compliment
complimentary
complimented
complimenter
complimenters
complimenting
compliments
comply
complying
component
componentry
components
componentwise
compose
composed
composedly
composer
composers
composes
composing
composite
composites
composition
compositional
compositions
compost
composure
compound
compounded
compounding
compounds
comprehend
comprehended
comprehending
comprehends
comprehensibility
comprehensible
comprehension
comprehensive
comprehensively
compress
compressed
compresses
compressible
compressing
compression
compressive
compressor
comprise
comprised
comprises
comprising
compromise
compromised
compromiser
compromisers
compromises
compromising
compromisingly
comptroller
comptrollers
compulsion
compulsions
compulsive
compulsory
compunction
computability
computable
computation
computational
computationally
computations
compute
computed
computer
computerize
computerized
computerizes
computerizing
computers
computes
computing
comrade
comradely
comrades
comradeship
con
concatenate
concatenated
concatenates
concatenating
concatenation
concatenations
concave
conceal
concealed
concealer
concealers
concealing
concealment
conceals
concede
conceded
concedes
conceding
conceit
conceited
conceits
conceivable
conceivably
conceive
conceived
conceives
conceiving
concentrate
concentrated
concentrates
concentrating
concentration
concentrations
concentrator
concentrators
concentric
concept
conception
conceptions
concepts
conceptual
conceptualization
conceptualizations
conceptualize
conceptualized
conceptualizes
conceptualizing
conceptually
concern
concerned
concernedly
concerning
concerns
concert
concerted
concertmaster
concerto
concerts
concession
concessions
conciliate
conciliatory
concise
concisely
conciseness
conclave
conclude
concluded
concludes
concluding
conclusion
conclusions
conclusive
conclusively
concoct
concomitant
concord
concordant
concourse
concrete
concretely
concreteness
concretes
concretion
concubine
concur
concurred
concurrence
concurrencies
concurrency
concurrent
concurrently
concurring
concurs
concussion
condemn
condemnation
condemnations
condemned
condemner
condemners
condemning
condemns
condensation
condense
condensed
condenser
condenses
condensing
condescend
condescending
condition
conditional
conditionally
conditionals
conditioned
conditioner
conditioners
conditioning
conditions
condom
condone
condoned
condones
condoning
conduce
conducive
conduciveness
conduct
conductance
conducted
conducting
conduction
conductive
conductivity
conductor
conductors
conducts
conduit
cone
cones
confectionery
confederacy
confederate
confederates
confederation
confederations
confer
conferee
conference
conferences
conferred
conferrer
conferrers
conferring
confers
confess
confessed
confesses
confessing
confession
confessions
confessor
confessors
confidant
confidants
confide
confided
confidence
confidences
confident
confidential
confidentiality
confidentially
confidently
confides
confiding
confidingly
configurable
configuration
configurations
configure
configured
configures
configuring
confine
confined
confinement
confinements
confiner
confines
confining
confirm
confirmation
confirmations
confirmatory
confirmed
confirming
confirms
confiscate
confiscated
confiscates
confiscating
confiscation
confiscations
conflagration
conflict
conflicted
conflicting
conflicts
confluent
confocal
conform
conformal
conformance
conformed
conforming
conformity
conforms
confound
confounded
confounding
confounds
confront
confrontation
confrontations
confronted
confronter
confronters
confronting
confronts
confuse
confused
confuser
confusers
confuses
confusing
confusingly
confusion
confusions
congenial
congenially
congenital
congest
congested
congestion
congestive
conglomerate
congratulate
congratulated
congratulation
congratulations
congratulatory
congregate
congregated
congregates
congregating
congregation
congregations
congress
congresses
congressional
congressionally
congressman
congressmen
congresswoman
congresswomen
congruence
congruent
conic
conifer
coniferous
conjecture
conjectured
conjectures
conjecturing
conjoined
conjugal
conjugate
conjunct
conjuncted
conjunction
conjunctions
conjunctive
conjunctively
conjuncts
conjuncture
conjure
conjured
conjurer
conjures
conjuring
connect
connected
connectedness
connecting
connection
connectionless
connections
connective
connectives
connectivity
connector
connectors
connects
connivance
connive
connoisseur
connoisseurs
connotation
connotative
connote
connoted
connotes
connoting
connubial
conquer
conquerable
conquered
conquerer
conquerers
conquering
conqueror
conquerors
conquers
conquest
conquests
conscience
consciences
conscientious
conscientiously
conscious
consciously
consciousness
conscript
conscription
consecrate
consecration
consecutive
consecutively
consensual
consensus
consent
consented
consenter
consenters
consenting
consents
consequence
consequences
consequent
consequential
consequentialities
consequentiality
consequently
consequents
conservation
conservationist
conservationists
conservations
conservatism
conservative
conservatively
conservatives
conservator
conserve
conserved
conserves
conserving
consider
considerable
considerably
considerate
considerately
consideration
considerations
considered
considering
considers
consign
consigned
consigning
consigns
consist
consisted
consistency
consistent
consistently
consisting
consists
consolable
consolation
consolations
console
consoled
consoler
consolers
consoles
consolidate
consolidated
consolidates
consolidating
consolidation
consoling
consolingly
consonant
consonants
consort
consorted
consorting
consortium
consorts
conspicuous
conspicuously
conspiracies
conspiracy
conspirator
conspirators
conspire
conspired
conspires
conspiring
constable
constables
constancy
constant
constantly
constants
constellation
constellations
consternation
constituencies
constituency
constituent
constituents
constitute
constituted
constitutes
constituting
constitution
constitutional
constitutionality
constitutionally
constitutions
constitutive
constrain
constrained
constraining
constrains
constraint
constraints
constrict
construct
constructed
constructibility
constructible
constructing
construction
constructions
constructive
constructively
constructor
constructors
constructs
construe
construed
construing
consul
consular
consulate
consulates
consuls
consult
consultant
consultants
consultation
consultations
consultative
consulted
consulting
consults
consumable
consume
consumed
consumer
consumers
consumes
consuming
consummate
consummated
consummately
consummation
consumption
consumptions
consumptive
consumptively
contact
contacted
contacting
contacts
contagion
contagious
contagiously
contain
containable
contained
container
containers
containing
containment
containments
contains
contaminate
contaminated
contaminates
contaminating
contamination
contemplate
contemplated
contemplates
contemplating
contemplation
contemplations
contemplative
contemporaries
contemporariness
contemporary
contempt
contemptible
contemptuous
contemptuously
contend
contended
contender
contenders
contending
contends
content
contented
contenting
contention
contentions
contently
contentment
contents
contest
contestable
contestant
contested
contester
contesters
contesting
contests
context
contexts
contextual
contextually
contiguity
contiguous
contiguously
continent
continental
continentally
continents
contingencies
contingency
contingent
contingents
continual
continually
continuance
continuances
continuation
continuations
continue
continued
continues
continuing
continuities
continuity
continuous
continuously
continuum
contortions
contour
contoured
contouring
contours
contraband
contraception
contraceptive
contract
contracted
contracting
contraction
contractions
contractor
contractors
contracts
contractual
contractually
contradict
contradicted
contradicting
contradiction
contradictions
contradictory
contradicts
contradistinction
contradistinctions
contrapositive
contrapositives
contraption
contraptions
contrariness
contrary
contrast
contrasted
contraster
contrasters
contrasting
contrastingly
contrasts
contribute
contributed
contributes
contributing
contribution
contributions
contributor
contributorily
contributors
contributory
contrite
contrition
contrivance
contrivances
contrive
contrived
contriver
contrives
contriving
control
controllability
controllable
controllably
controlled
controller
controllers
controlling
controls
controversial
controversies
controversy
controvertible
contumacious
contumacy
conundrum
conundrums
convalescent
convect
convene
convened
convenes
convenience
conveniences
convenient
conveniently
convening
convent
convention
conventional
conventionally
conventions
convents
converge
converged
convergence
convergent
converges
converging
conversant
conversantly
conversation
conversational
conversationally
conversations
converse
conversed
conversely
converses
conversing
conversion
conversions
convert
converted
converter
converters
convertibility
convertible
converting
converts
convex
convey
conveyance
conveyances
conveyed
conveyer
conveyers
conveying
conveyor
conveys
convict
convicted
convicting
conviction
convictions
convicts
convince
convinced
convincer
convincers
convinces
convincing
convincingly
convivial
convoke
convoluted
convolution
convoy
convoyed
convoying
convoys
convulse
convulsion
convulsions
coo
cooing
cook
cookbook
cooked
cookery
cookie
cookies
cooking
cooks
cooky
cool
cooled
cooler
coolers
coolest
coolie
coolies
cooling
coolly
coolness
cools
coon
coons
coop
cooped
cooper
cooperate
cooperated
cooperates
cooperating
cooperation
cooperations
cooperative
cooperatively
cooperatives
cooperator
cooperators
coopers
coops
coordinate
coordinated
coordinates
coordinating
coordination
coordinations
coordinator
coordinators
cop
cope
coped
copes
copied
copier
copiers
copies
coping
copings
copious
copiously
copiousness
coplanar
copper
copperhead
coppers
copra
coprocessor
cops
copse
copy
copying
copyright
copyrightable
copyrighted
copyrights
copywriter
coquette
coral
cord
corded
corder
cordial
cordiality
cordially
cords
core
cored
corer
corers
cores
coriander
coring
cork
corked
corker
corkers
corking
corks
corkscrew
cormorant
corn
cornea
corner
cornered
corners
cornerstone
cornerstones
cornet
cornfield
cornfields
corning
cornmeal
corns
cornstarch
cornucopia
corny
corollaries
corollary
coronaries
coronary
coronation
coroner
coronet
coronets
coroutine
coroutines
corporal
corporals
corporate
corporately
corporation
corporations
corps
corpse
corpses
corpulent
corpus
corpuscular
corral
correct
correctable
corrected
correcting
correction
corrections
corrective
correctively
correctives
correctly
correctness
corrector
corrects
correlate
correlated
correlates
correlating
correlation
correlations
correlative
correspond
corresponded
correspondence
correspondences
correspondent
correspondents
corresponding
correspondingly
corresponds
corridor
corridors
corrigenda
corrigendum
corrigible
corroborate
corroborated
corroborates
corroborating
corroboration
corroborations
corroborative
corrode
corrosion
corrosive
corrugate
corrupt
corrupted
corrupter
corruptible
corrupting
corruption
corruptions
corrupts
corset
cortex
cortical
cosine
cosines
cosmetic
cosmetics
cosmic
cosmology
cosmopolitan
cosmos
cosponsor
cost
costed
costing
costly
costs
costume
costumed
costumer
costumes
costuming
cosy
cot
cotangent
cotillion
cots
cottage
cottager
cottages
cotton
cottonmouth
cottons
cottonseed
cottonwood
cotyledon
cotyledons
couch
couched
couches
couching
cougar
cough
coughed
coughing
coughs
could
coulomb
council
councillor
councillors
councilman
councilmen
councils
councilwoman
councilwomen
counsel
counseled
counseling
counselled
counselling
counsellor
counsellors
counselor
counselors
counsels
count
countable
countably
counted
countenance
counter
counteract
counteracted
counteracting
counteractive
counterargument
counterattack
counterbalance
counterclockwise
countered
counterexample
counterexamples
counterfeit
counterfeited
counterfeiter
counterfeiting
counterflow
countering
counterintuitive
counterman
countermeasure
countermeasures
countermen
counterpart
counterparts
counterpoint
counterpointing
counterpoise
counterproductive
counterproposal
counterrevolution
counters
countersink
countersunk
countess
counties
counting
countless
countries
country
countryman
countrymen
countryside
countrywide
counts
county
countywide
couple
coupled
coupler
couplers
couples
coupling
couplings
coupon
coupons
courage
courageous
courageously
courier
couriers
course
coursed
courser
courses
coursing
court
courted
courteous
courteously
courter
courters
courtesan
courtesies
courtesy
courthouse
courthouses
courtier
courtiers
courting
courtly
courtroom
courtrooms
courts
courtship
courtyard
courtyards
cousin
cousins
covalent
covariant
cove
covenant
covenants
cover
coverable
coverage
covered
covering
coverings
coverlet
coverlets
covers
covert
covertly
coves
covet
coveted
coveting
covetous
covetousness
covets
cow
coward
cowardice
cowardly
cowboy
cowboys
cowed
cower
cowered
cowerer
cowerers
cowering
coweringly
cowers
cowherd
cowhide
cowing
cowl
cowlick
cowling
cowls
coworker
cows
cowslip
cowslips
coyote
coyotes
coypu
cozier
coziness
cozy
crab
crabapple
crabs
crack
cracked
cracker
crackers
cracking
crackle
crackled
crackles
crackling
crackpot
cracks
cradle
cradled
cradles
craft
crafted
crafter
craftiness
crafting
crafts
craftsman
craftsmen
craftspeople
craftsperson
crafty
crag
craggy
crags
cram
cramming
cramp
cramps
crams
cranberries
cranberry
crane
cranes
crania
cranium
crank
crankcase
cranked
crankier
crankiest
crankily
cranking
cranks
crankshaft
cranky
cranny
crash
crashed
crasher
crashers
crashes
crashing
crass
crate
crater
craters
crates
cravat
cravats
crave
craved
craven
craves
craving
crawl
crawled
crawler
crawlers
crawling
crawls
crayon
craze
crazed
crazes
crazier
craziest
crazily
craziness
crazing
crazy
creak
creaked
creaking
creaks
creaky
cream
creamed
creamer
creamers
creamery
creaming
creams
creamy
crease
creased
creases
creasing
create
created
creates
creating
creation
creations
creative
creatively
creativeness
creativity
creator
creators
creature
creatures
credence
credential
credibility
credible
credibly
credit
creditable
creditably
credited
crediting
creditor
creditors
credits
credulity
credulous
credulousness
creed
creeds
creek
creeks
creep
creeper
creepers
creeping
creeps
creepy
cremate
cremated
cremates
cremating
cremation
cremations
crematory
crepe
crept
crescent
crescents
crest
crested
crestfallen
crests
cretin
crevice
crevices
crew
crewcut
crewed
crewing
crews
crib
cribs
cricket
crickets
cried
crier
criers
cries
crime
crimes
criminal
criminally
criminals
criminate
crimson
crimsoning
cringe
cringed
cringes
cringing
cripple
crippled
cripples
crippling
crises
crisis
crisp
crisply
crispness
crisscross
criteria
criterion
critic
critical
critically
criticism
criticisms
criticize
criticized
criticizes
criticizing
critics
critique
critiques
critiquing
critter
croak
croaked
croaking
croaks
crochet
crochets
crock
crockery
crocks
crocodile
crocus
croft
crook
crooked
crooks
crop
cropped
cropper
croppers
cropping
crops
cross
crossable
crossbar
crossbars
crossed
crosser
crossers
crosses
crossing
crossings
crossly
crossover
crossovers
crosspoint
crossroad
crosstalk
crosswalk
crossword
crosswords
crotch
crotchety
crouch
crouched
crouching
crow
crowd
crowded
crowder
crowding
crowds
crowed
crowing
crown
crowned
crowning
crowns
crows
crucial
crucially
crucible
crucified
crucifies
crucifix
crucifixion
crucify
crucifying
crud
cruddy
crude
crudely
crudeness
cruder
crudest
cruel
crueler
cruelest
cruelly
cruelty
cruise
cruiser
cruisers
cruises
cruising
crumb
crumble
crumbled
crumbles
crumbling
crumbly
crumbs
crummy
crumple
crumpled
crumples
crumpling
crunch
crunched
crunches
crunchier
crunchiest
crunching
crunchy
crusade
crusader
crusaders
crusades
crusading
crush
crushable
crushed
crusher
crushers
crushes
crushing
crushingly
crust
crustacean
crustaceans
crusts
crutch
crutches
crux
cruxes
cry
crying
cryogenic
crypt
cryptanalysis
cryptanalyst
cryptanalytic
cryptic
cryptogram
cryptographer
cryptographic
cryptographically
cryptography
cryptologist
cryptology
crystal
crystalline
crystallize
crystallized
crystallizes
crystallizing
crystals
cub
cubbyhole
cube
cubed
cubes
cubic
cubs
cuckoo
cuckoos
cucumber
cucumbers
cuddle
cuddled
cuddly
cudgel
cudgels
cue
cued
cues
cuff
cufflink
cuffs
cuisine
culinary
cull
culled
culler
culling
culls
culminate
culminated
culminates
culminating
culmination
culpa
culpable
culprit
culprits
cult
cultivable
cultivate
cultivated
cultivates
cultivating
cultivation
cultivations
cultivator
cultivators
cults
cultural
culturally
culture
cultured
cultures
culturing
cumbersome
cumulative
cumulatively
cunnilingus
cunning
cunningly
cup
cupboard
cupboards
cupful
cupped
cupping
cups
curable
curably
curb
curbing
curbs
curd
curdle
cure
cured
cures
curfew
curfews
curing
curiosities
curiosity
curious
curiouser
curiousest
curiously
curl
curled
curler
curlers
curlicue
curling
curls
curly
currant
currants
currencies
currency
current
currently
currentness
currents
curricular
curriculum
curriculums
curried
curries
curry
currying
curs
curse
cursed
curses
cursing
cursive
cursor
cursorily
cursors
cursory
curt
curtail
curtailed
curtails
curtain
curtained
curtains
curtate
curtly
curtness
curtsies
curtsy
curvaceous
curvature
curve
curved
curves
curvilinear
curving
cushion
cushioned
cushioning
cushions
cusp
cusps
custard
custodial
custodian
custodians
custody
custom
customarily
customary
customer
customers
customizable
customization
customizations
customize
customized
customizer
customizers
customizes
customizing
customs
cut
cutaneous
cutback
cute
cutest
cutlass
cutlet
cutoff
cutout
cutover
cuts
cutter
cutters
cutthroat
cutting
cuttingly
cuttings
cuttlefish
cyanide
cybernetic
cybernetics
cyberspace
cycle
cycled
cycles
cyclic
cyclically
cycling
cycloid
cycloidal
cycloids
cyclone
cyclones
cyclotron
cyclotrons
cylinder
cylinders
cylindrical
cymbal
cymbals
cynic
cynical
cynically
cypress
cyst
cysts
cytology
cytoplasm
czar
dabble
dabbled
dabbler
dabbles
dabbling
dactyl
dactylic
dad
daddy
dads
daemon
daemons
daffodil
daffodils
dagger
dahlia
dailies
daily
daintily
daintiness
dainty
dairy
daisies
daisy
dale
dales
dam
damage
damaged
damager
damagers
damages
damaging
damask
dame
damming
damn
damnation
damned
damning
damns
damp
dampen
dampens
damper
damping
dampness
dams
damsel
damsels
dance
danced
dancer
dancers
dances
dancing
dandelion
dandelions
dandy
danger
dangerous
dangerously
dangers
dangle
dangled
dangles
dangling
dare
dared
darer
darers
dares
daresay
daring
daringly
dark
darken
darker
darkest
darkly
darkness
darkroom
darling
darlings
darn
darned
darner
darning
darns
dart
darted
darter
darting
darts
dash
dashboard
dashed
dasher
dashers
dashes
dashing
dashingly
data
database
databases
datagram
datagrams
date
dated
dateline
dater
dates
dating
dative
datum
daughter
daughterly
daughters
daunt
daunted
dauntless
dawn
dawned
dawning
dawns
day
daybreak
daydream
daydreaming
daydreams
daylight
daylights
days
daytime
daze
dazed
dazzle
dazzled
dazzler
dazzles
dazzling
dazzlingly
deacon
deacons
deactivate
dead
deaden
deadline
deadlines
deadlock
deadlocked
deadlocking
deadlocks
deadly
deadness
deadwood
deaf
deafen
deafer
deafest
deafness
deal
dealer
dealers
dealership
dealing
dealings
deallocate
deallocated
deallocating
deallocation
deallocations
deals
dealt
dean
deans
dear
dearer
dearest
dearly
dearness
dearth
dearths
death
deathbed
deathly
deaths
debacle
debar
debase
debatable
debate
debated
debater
debaters
debates
debating
debauch
debauchery
debilitate
debilitated
debilitates
debilitating
debility
debit
debited
debrief
debris
debt
debtor
debts
debug
debugged
debugger
debuggers
debugging
debugs
debunk
debutante
decade
decadence
decadent
decadently
decades
decal
decathlon
decay
decayed
decaying
decays
decease
deceased
deceases
deceasing
decedent
deceit
deceitful
deceitfully
deceitfulness
deceive
deceived
deceiver
deceivers
deceives
deceiving
decelerate
decelerated
decelerates
decelerating
deceleration
decencies
decency
decennial
decent
decently
decentralization
decentralized
deception
deceptions
deceptive
deceptively
decertify
decibel
decidability
decidable
decide
decided
decidedly
decides
deciding
deciduous
decimal
decimals
decimate
decimated
decimates
decimating
decimation
decipher
deciphered
decipherer
deciphering
deciphers
decision
decisions
decisive
decisively
decisiveness
deck
decked
decking
deckings
decks
declaration
declarations
declarative
declaratively
declaratives
declarator
declaratory
declare
declared
declarer
declarers
declares
declaring
declassify
declination
declinations
decline
declined
decliner
decliners
declines
declining
decode
decoded
decoder
decoders
decodes
decoding
decodings
decolletage
decollimate
decompile
decomposability
decomposable
decompose
decomposed
decomposes
decomposing
decomposition
decompositions
decompress
decompression
decorate
decorated
decorates
decorating
decoration
decorations
decorative
decorum
decouple
decoupled
decouples
decoupling
decoy
decoys
decrease
decreased
decreases
decreasing
decreasingly
decree
decreed
decreeing
decrees
decrement
decremented
decrementing
decrements
decrypt
decrypted
decrypting
decryption
decrypts
dedicate
dedicated
dedicates
dedicating
dedication
deduce
deduced
deducer
deduces
deducible
deducing
deduct
deducted
deductible
deducting
deduction
deductions
deductive
deed
deeded
deeding
deeds
deem
deemed
deeming
deemphasize
deemphasized
deemphasizes
deemphasizing
deems
deep
deepen
deepened
deepening
deepens
deeper
deepest
deeply
deeps
deer
deface
default
defaulted
defaulter
defaulting
defaults
defeat
defeated
defeating
defeats
defecate
defect
defected
defecting
defection
defections
defective
defects
defend
defendant
defendants
defended
defender
defenders
defending
defends
defenestrate
defenestrated
defenestrates
defenestrating
defenestration
defense
defenseless
defenses
defensible
defensive
defer
deference
deferment
deferments
deferrable
deferred
deferrer
deferrers
deferring
defers
defiance
defiant
defiantly
deficiencies
deficiency
deficient
deficit
deficits
defied
defies
defile
defiling
definable
define
defined
definer
defines
defining
definite
definitely
definiteness
definition
definitional
definitions
definitive
deflate
deflater
deflect
defocus
deforest
deforestation
deform
deformation
deformations
deformed
deformities
deformity
defraud
defray
defrost
deftly
defunct
defy
defying
degeneracy
degenerate
degenerated
degenerates
degenerating
degeneration
degenerative
degradable
degradation
degradations
degrade
degraded
degrades
degrading
degree
degrees
dehumidify
dehydrate
deify
deign
deigned
deigning
deigns
deities
deity
dejected
dejectedly
delay
delayed
delaying
delays
delegate
delegated
delegates
delegating
delegation
delegations
delete
deleted
deleter
deleterious
deletes
deleting
deletion
deletions
deliberate
deliberated
deliberately
deliberateness
deliberates
deliberating
deliberation
deliberations
deliberative
deliberator
deliberators
delicacies
delicacy
delicate
delicately
delicatessen
delicious
deliciously
delight
delighted
delightedly
delightful
delightfully
delighting
delights
delimit
delimitation
delimited
delimiter
delimiters
delimiting
delimits
delineament
delineate
delineated
delineates
delineating
delineation
delinquency
delinquent
delirious
deliriously
delirium
deliver
deliverable
deliverables
deliverance
delivered
deliverer
deliverers
deliveries
delivering
delivers
delivery
dell
dells
delta
deltas
delude
deluded
deludes
deluding
deluge
deluged
deluges
delusion
delusions
deluxe
delve
delves
delving
demagnify
demagogue
demand
demanded
demander
demanding
demandingly
demands
demarcate
demeanor
demented
demerit
demigod
demise
demo
democracies
democracy
democrat
democratic
democratically
democrats
demodulate
demodulator
demographic
demolish
demolished
demolishes
demolition
demon
demoniac
demonic
demons
demonstrable
demonstrate
demonstrated
demonstrates
demonstrating
demonstration
demonstrations
demonstrative
demonstratively
demonstrator
demonstrators
demoralize
demoralized
demoralizes
demoralizing
demote
demountable
demultiplex
demultiplexed
demultiplexer
demultiplexers
demultiplexing
demur
demythologize
den
denature
deniable
denial
denials
denied
denier
denies
denigrate
denigrated
denigrates
denigrating
denizen
denominate
denomination
denominations
denominator
denominators
denotable
denotation
denotational
denotationally
denotations
denotative
denote
denoted
denotes
denoting
denounce
denounced
denounces
denouncing
dens
dense
densely
denseness
denser
densest
densities
density
dent
dental
dentally
dented
denting
dentist
dentistry
dentists
dents
denture
denude
denumerable
denunciate
denunciation
deny
denying
deodorant
deoxyribonucleic
depart
departed
departing
department
departmental
departments
departs
departure
departures
depend
dependability
dependable
dependably
depended
dependence
dependencies
dependency
dependent
dependently
dependents
depending
depends
depict
depicted
depicting
depicts
deplete
depleted
depletes
depleting
depletion
depletions
deplorable
deplore
deplored
deplores
deploring
deploy
deployed
deploying
deployment
deployments
deploys
deport
deportation
deportee
deportment
depose
deposed
deposes
deposit
depositary
deposited
depositing
deposition
depositions
depositor
depositors
depository
deposits
depot
depots
deprave
depraved
depravity
deprecate
depreciate
depreciated
depreciates
depreciation
depress
depressed
depresses
depressing
depression
depressions
deprivation
deprivations
deprive
deprived
deprives
depriving
depth
depths
deputies
deputy
dequeue
dequeued
dequeues
dequeuing
derail
derailed
derailing
derails
derby
dereference
deregulate
deregulated
deride
derision
derivable
derivation
derivations
derivative
derivatives
derive
derived
derives
deriving
derogatory
derrick
derriere
dervish
descend
descendant
descendants
descended
descendent
descender
descenders
descending
descends
descent
descents
describable
describe
described
describer
describes
describing
description
descriptions
descriptive
descriptively
descriptives
descriptor
descriptors
descry
desecrate
desegregate
desert
deserted
deserter
deserters
deserting
desertion
desertions
deserts
deserve
deserved
deserves
deserving
deservingly
deservings
desiderata
desideratum
design
designate
designated
designates
designating
designation
designations
designator
designators
designed
designer
designers
designing
designs
desirability
desirable
desirably
desire
desired
desires
desiring
desirous
desist
desk
desks
desktop
desolate
desolately
desolation
desolations
despair
despaired
despairing
despairingly
despairs
despatch
despatched
desperado
desperate
desperately
desperation
despicable
despise
despised
despises
despising
despite
despoil
despondent
despot
despotic
despotism
despots
dessert
desserts
desiccate
destabilize
destination
destinations
destine
destined
destinies
destiny
destitute
destitution
destroy
destroyed
destroyer
destroyers
destroying
destroys
destruct
destruction
destructions
destructive
destructively
destructiveness
destructor
destuff
destuffing
destuffs
desuetude
desultory
desynchronize
detach
detached
detacher
detaches
detaching
detachment
detachments
detail
detailed
detailing
details
detain
detained
detaining
detains
detect
detectable
detectably
detected
detecting
detection
detections
detective
detectives
detector
detectors
detects
detente
detention
deter
detergent
deteriorate
deteriorated
deteriorates
deteriorating
deterioration
determinable
determinacy
determinant
determinants
determinate
determinately
determination
determinations
determinative
determine
determined
determiner
determiners
determines
determining
determinism
deterministic
deterministically
deterred
deterrent
deterring
detest
detestable
detested
detour
detract
detractor
detractors
detracts
detriment
detrimental
deuce
deus
deuterium
devastate
devastated
devastates
devastating
devastation
develop
developed
developer
developers
developing
development
developmental
developments
develops
deviant
deviants
deviate
deviated
deviates
deviating
deviation
deviations
device
devices
devil
devilish
devilishly
devils
devious
devise
devised
devises
devising
devisings
devoid
devolve
devote
devoted
devotedly
devotee
devotees
devotes
devoting
devotion
devotions
devour
devoured
devourer
devours
devout
devoutly
devoutness
dew
dewdrop
dewdrops
dewy
dexterity
diabetes
diabetic
diabolical
diachronic
diacritical
EOF

rm -f $input; touch $input

i=0
while test $i -lt 20;do cat $temp >>$input; i=`expr $i + 1`;done

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

undef($/);
print join("\n", reverse split(/\n/, <STDIN>)),"\n";
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest <$input >$temp2 2>&1;then fgrep 'user ' $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# from Brad Knotwell
# with help from Mark Baker

def main():
    from sys import stdin, stdout
    w = stdin.readlines()
    w.reverse()
    stdout.writelines(w)

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest <$input >$temp2 2>&1;then fgrep 'user ' $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# from: Miguel Sofer

proc main {} {
    set lines [split [read stdin] "\n"]
    
    fconfigure stdout -buffering full

    for {set i [expr {[llength $lines]-2}]} {$i >= 0} {incr i -1} {
        puts [lindex $lines $i]
    }
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest <$input >$temp2 2>&1;then fgrep 'user ' $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

BEGIN { delete ARGV }
{ x[NR] = $0 }
END { for (i = NR; i >= 1; i--)
    print x[i]
}
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest <$input >$temp2 2>&1;then fgrep 'user ' $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
func main () {
  var i, lns;
  
  lns = getf (1);
  for (i = #lns - 1; i >= 0; i--)
    putln (lns [i]);
}

main ();
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest <$input >$temp2 2>&1;then fgrep 'user ' $temp2;else echo FAILED;fi

fi

# Test 20.
if test $start_test_number -le 20; then

######################################################
echo 
echo '+++++ Test #20: Sieve of Eratosthenes (N=50):  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/perl 
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/
# with help from Tony Bowden

use strict;
use integer;

my $NUM = ($ARGV[0] < 1) ? 1 : $ARGV[0];
my $count;
my @flags = ();
while ($NUM--) {
    $count = 0; 
    my @flags = (0 .. 8192);
    for my $i (2 .. 8192 ) {
	next unless defined $flags[$i];
	# remove all multiples of prime: i
	my $k = $i;
	undef $flags[$k] while (($k+=$i) < 8193);
	$count++;
    }
}
print "Count: $count\n";
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 50 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/python 
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/
# with help from Brad Knotwell

import sys

def main():
    NUM = int(sys.argv[1])
    for foo in xrange(0,NUM):
        flags = (8192+1) * [1]
        count = 0
        for i in xrange(2,8192+1):
            if flags[i]:
                # remove all multiples of prime: i
                k = i + i
                while k <= 8192:
                    flags[k] = 0
                    k = k + i
                count = count + 1
    print "Count:", count

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 50 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/tclsh
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/
# with help from: Kristoffer Lawson

proc sieve {num} {
    while {$num > 0} {
	incr num -1
	set count 0
	for {set i 2} {$i <= 8192} {incr i 1} {
	    set flags($i) 1
	}
	for {set i 2} {$i <= 8192} {incr i 1} {
	    if {$flags($i) == 1} {
		# remove all multiples of prime: i
		for {set k [expr {$i+$i}]} {$k <= 8192} {incr k $i} {
		    set flags($k) 0
		}
		incr count 1
	    }
	}
    }
    return $count
}

set NUM [lindex $argv 0]
if {$NUM < 1} {
    set NUM 1
}

set count [sieve $NUM]
puts "Count: $count"
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 50 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

BEGIN {
    n = (ARGV[1] < 1) ? 1 : ARGV[1];
    while (n--) {
        count=0;
        for(i=2; i <= 8192; flags[i++]=1);
        for (i=2; i <= 8192; i++) {
	    if (flags[i]) {
		# remove all multiples of prime: i
		for (k=i+i; k <= 8192; k+=i) {
                    flags[k] = 0;
		}
		count++;
	    }
        }
    }
    printf("Count: %d\n", count);
    exit;
}
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest 50 >$temp2 2>&1;then fgrep 'user ' $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
func main () {
  var i, j, k, count, n = argv [0] < 1 ? 1 : int (argv [0]);
  var flags;
  for (; n >= 0; n--) {
      count = 0;
      flags = [8193:1];
      for (i = 2; i <= 8192; i++) {
  	if (flags [i]) {
          for (k = i + i; k <= 8192; k += i)
            flags [k] = 0;
  	  count++;
  	}
      }
  }
  putln ("Count: ", count);
}
main ();
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 50 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 21.
if test $start_test_number -le 21; then

######################################################
echo 
echo '+++++ Test #21: Spell Checker (N=10):  +++++'

cat <<'EOF' >$temp
aback
abaft
abandon
abandoned
abandoning
abandonment
abandons
abase
abased
abasement
abasements
abases
abash
abashed
abashes
abashing
abasing
abate
abated
abatement
abatements
abater
abates
abating
abbe
abbey
abbeys
abbot
abbots
abbreviate
abbreviated
abbreviates
abbreviating
abbreviation
abbreviations
abdomen
abdomens
abdominal
abduct
abducted
abduction
abductions
abductor
abductors
abducts
abed
aberrant
aberration
aberrations
abet
abets
abetted
abetter
abetting
abeyance
abhor
abhorred
abhorrent
abhorrer
abhorring
abhors
abide
abided
abides
abiding
abilities
ability
abject
abjection
abjections
abjectly
abjectness
abjure
abjured
abjures
abjuring
ablate
ablated
ablates
ablating
ablation
ablative
ablaze
able
abler
ablest
ably
abnormal
abnormalities
abnormality
abnormally
aboard
abode
abodes
abolish
abolished
abolisher
abolishers
abolishes
abolishing
abolishment
abolishments
abolition
abolitionist
abolitionists
abominable
abominate
aboriginal
aborigine
aborigines
abort
aborted
aborting
abortion
abortions
abortive
abortively
aborts
abound
abounded
abounding
abounds
about
above
aboveboard
aboveground
abovementioned
abrade
abraded
abrades
abrading
abrasion
abrasions
abrasive
abreaction
abreactions
abreast
abridge
abridged
abridges
abridging
abridgment
abroad
abrogate
abrogated
abrogates
abrogating
abrupt
abruptly
abruptness
abscess
abscessed
abscesses
abscissa
abscissas
abscond
absconded
absconding
absconds
absence
absences
absent
absented
absentee
absenteeism
absentees
absentia
absenting
absently
absentminded
absents
absinthe
absolute
absolutely
absoluteness
absolutes
absolution
absolve
absolved
absolves
absolving
absorb
absorbed
absorbency
absorbent
absorber
absorbing
absorbs
absorption
absorptions
absorptive
abstain
abstained
abstainer
abstaining
abstains
abstention
abstentions
abstinence
abstract
abstracted
abstracting
abstraction
abstractionism
abstractionist
abstractions
abstractly
abstractness
abstractor
abstractors
abstracts
abstruse
abstruseness
absurd
absurdities
absurdity
absurdly
abundance
abundant
abundantly
abuse
abused
abuses
abusing
abusive
abut
abutment
abuts
abutted
abutter
abutters
abutting
abysmal
abysmally
abyss
abysses
acacia
academia
academic
academically
academics
academies
academy
accede
acceded
accedes
accelerate
accelerated
accelerates
accelerating
acceleration
accelerations
accelerator
accelerators
accelerometer
accelerometers
accent
accented
accenting
accents
accentual
accentuate
accentuated
accentuates
accentuating
accentuation
accept
acceptability
acceptable
acceptably
acceptance
acceptances
accepted
accepter
accepters
accepting
acceptor
acceptors
accepts
access
accessed
accesses
accessibility
accessible
accessibly
accessing
accession
accessions
accessories
accessors
accessory
accident
accidental
accidentally
accidently
accidents
acclaim
acclaimed
acclaiming
acclaims
acclamation
acclimate
acclimated
acclimates
acclimating
acclimatization
acclimatized
accolade
accolades
accommodate
accommodated
accommodates
accommodating
accommodation
accommodations
accompanied
accompanies
accompaniment
accompaniments
accompanist
accompanists
accompany
accompanying
accomplice
accomplices
accomplish
accomplished
accomplisher
accomplishers
accomplishes
accomplishing
accomplishment
accomplishments
accord
accordance
accorded
accorder
accorders
according
accordingly
accordion
accordions
accords
accost
accosted
accosting
accosts
account
accountability
accountable
accountably
accountancy
accountant
accountants
accounted
accounting
accounts
accredit
accreditation
accreditations
accredited
accretion
accretions
accrue
accrued
accrues
accruing
acculturate
acculturated
acculturates
acculturating
acculturation
accumulate
accumulated
accumulates
accumulating
accumulation
accumulations
accumulator
accumulators
accuracies
accuracy
accurate
accurately
accurateness
accursed
accusal
accusation
accusations
accusative
accuse
accused
accuser
accuses
accusing
accusingly
accustom
accustomed
accustoming
accustoms
ace
aces
acetate
acetone
acetylene
ache
ached
aches
achievable
achieve
achieved
achievement
achievements
achiever
achievers
achieves
achieving
aching
acid
acidic
acidities
acidity
acidly
acids
acidulous
acknowledge
acknowledgeable
acknowledged
acknowledgement
acknowledgements
acknowledger
acknowledgers
acknowledges
acknowledging
acknowledgment
acknowledgments
acme
acne
acolyte
acolytes
acorn
acorns
acoustic
acoustical
acoustically
acoustician
acoustics
acquaint
acquaintance
acquaintances
acquainted
acquainting
acquaints
acquiesce
acquiesced
acquiescence
acquiescent
acquiesces
acquiescing
acquirable
acquire
acquired
acquires
acquiring
acquisition
acquisitions
acquisitive
acquisitiveness
acquit
acquits
acquittal
acquitted
acquitter
acquitting
acre
acreage
acres
acrid
acrimonious
acrimony
acrobat
acrobatic
acrobatics
acrobats
acronym
acronyms
acropolis
across
acrylic
act
acted
acting
actinium
actinometer
actinometers
action
actions
activate
activated
activates
activating
activation
activations
activator
activators
active
actively
activism
activist
activists
activities
activity
actor
actors
actress
actresses
actual
actualities
actuality
actualization
actually
actuals
actuarial
actuarially
actuate
actuated
actuates
actuating
actuator
actuators
acuity
acumen
acute
acutely
acuteness
acyclic
acyclically
ad
adage
adages
adagio
adagios
adamant
adamantly
adapt
adaptability
adaptable
adaptation
adaptations
adapted
adapter
adapters
adapting
adaptive
adaptively
adaptor
adaptors
adapts
add
added
addend
addenda
addendum
adder
adders
addict
addicted
addicting
addiction
addictions
addicts
adding
addition
additional
additionally
additions
additive
additives
additivity
address
addressability
addressable
addressed
addressee
addressees
addresser
addressers
addresses
addressing
adds
adduce
adduced
adduces
adducible
adducing
adduct
adducted
adducting
adduction
adductor
adducts
adept
adequacies
adequacy
adequate
adequately
adhere
adhered
adherence
adherent
adherents
adherer
adherers
adheres
adhering
adhesion
adhesions
adhesive
adhesives
adiabatic
adiabatically
adieu
adjacency
adjacent
adjective
adjectives
adjoin
adjoined
adjoining
adjoins
adjourn
adjourned
adjourning
adjournment
adjourns
adjudge
adjudged
adjudges
adjudging
adjudicate
adjudicated
adjudicates
adjudicating
adjudication
adjudications
adjunct
adjuncts
adjure
adjured
adjures
adjuring
adjust
adjustable
adjustably
adjusted
adjuster
adjusters
adjusting
adjustment
adjustments
adjustor
adjustors
adjusts
adjutant
adjutants
administer
administered
administering
administerings
administers
administrable
administrate
administration
administrations
administrative
administratively
administrator
administrators
admirable
admirably
admiral
admirals
admiralty
admiration
admirations
admire
admired
admirer
admirers
admires
admiring
admiringly
admissibility
admissible
admission
admissions
admit
admits
admittance
admitted
admittedly
admitter
admitters
admitting
admix
admixed
admixes
admixture
admonish
admonished
admonishes
admonishing
admonishment
admonishments
admonition
admonitions
ado
adobe
adolescence
adolescent
adolescents
adopt
adopted
adopter
adopters
adopting
adoption
adoptions
adoptive
adopts
adorable
adoration
adore
adored
adores
adorn
adorned
adornment
adornments
adorns
adrenal
adrenaline
adrift
adroit
adroitness
ads
adsorb
adsorbed
adsorbing
adsorbs
adsorption
adulate
adulating
adulation
adult
adulterate
adulterated
adulterates
adulterating
adulterer
adulterers
adulterous
adulterously
adultery
adulthood
adults
adumbrate
adumbrated
adumbrates
adumbrating
adumbration
advance
advanced
advancement
advancements
advances
advancing
advantage
advantaged
advantageous
advantageously
advantages
advent
adventist
adventists
adventitious
adventure
adventured
adventurer
adventurers
adventures
adventuring
adventurous
adverb
adverbial
adverbs
adversaries
adversary
adverse
adversely
adversities
adversity
advert
advertise
advertised
advertisement
advertisements
advertiser
advertisers
advertises
advertising
advice
advisability
advisable
advisably
advise
advised
advisedly
advisee
advisees
advisement
advisements
adviser
advisers
advises
advising
advisor
advisors
advisory
advocacy
advocate
advocated
advocates
advocating
aegis
aerate
aerated
aerates
aerating
aeration
aerator
aerators
aerial
aerials
aeroacoustic
aerobic
aerobics
aerodynamic
aerodynamics
aeronautic
aeronautical
aeronautics
aerosol
aerosolize
aerosols
aerospace
aesthetic
aesthetically
aesthetics
afar
affable
affair
affairs
affect
affectation
affectations
affected
affecting
affectingly
affection
affectionate
affectionately
affections
affective
affects
afferent
affianced
affidavit
affidavits
affiliate
affiliated
affiliates
affiliating
affiliation
affiliations
affinities
affinity
affirm
affirmation
affirmations
affirmative
affirmatively
affirmed
affirming
affirms
affix
affixed
affixes
affixing
afflict
afflicted
afflicting
affliction
afflictions
afflictive
afflicts
affluence
affluent
afford
affordable
afforded
affording
affords
affricate
affricates
affright
affront
affronted
affronting
affronts
aficionado
afield
afire
aflame
afloat
afoot
afore
aforementioned
aforesaid
aforethought
afoul
afraid
afresh
aft
after
aftereffect
afterglow
afterimage
afterlife
aftermath
aftermost
afternoon
afternoons
aftershock
aftershocks
afterthought
afterthoughts
afterward
afterwards
again
against
agape
agar
agate
agates
age
aged
ageless
agencies
agency
agenda
agendas
agent
agents
ager
agers
ages
agglomerate
agglomerated
agglomerates
agglomeration
agglutinate
agglutinated
agglutinates
agglutinating
agglutination
agglutinin
agglutinins
aggrandize
aggravate
aggravated
aggravates
aggravation
aggregate
aggregated
aggregately
aggregates
aggregating
aggregation
aggregations
aggression
aggressions
aggressive
aggressively
aggressiveness
aggressor
aggressors
aggrieve
aggrieved
aggrieves
aggrieving
aghast
agile
agilely
agility
aging
agitate
agitated
agitates
agitating
agitation
agitations
agitator
agitators
agleam
aglow
agnostic
agnostics
ago
agog
agonies
agonize
agonized
agonizes
agonizing
agonizingly
agony
agrarian
agree
agreeable
agreeably
agreed
agreeing
agreement
agreements
agreer
agreers
agrees
agricultural
agriculturally
agriculture
ague
ah
ahead
aid
aide
aided
aiding
aids
ail
aileron
ailerons
ailing
ailment
ailments
aim
aimed
aimer
aimers
aiming
aimless
aimlessly
aims
air
airbag
airbags
airborne
aircraft
airdrop
airdrops
aired
airer
airers
airfare
airfield
airfields
airflow
airfoil
airfoils
airframe
airframes
airily
airing
airings
airless
airlift
airlifts
airline
airliner
airlines
airlock
airlocks
airmail
airmails
airman
airmen
airplane
airplanes
airport
airports
airs
airship
airships
airspace
airspeed
airstrip
airstrips
airtight
airway
airways
airy
aisle
ajar
akimbo
akin
alabaster
alacrity
alarm
alarmed
alarming
alarmingly
alarmist
alarms
alas
alba
albacore
albatross
albeit
album
albumin
albums
alchemy
alcohol
alcoholic
alcoholics
alcoholism
alcohols
alcove
alcoves
alder
alderman
aldermen
ale
alee
alert
alerted
alertedly
alerter
alerters
alerting
alertly
alertness
alerts
alfalfa
alfresco
alga
algae
algaecide
algebra
algebraic
algebraically
algebras
alginate
algorithm
algorithmic
algorithmically
algorithms
alias
aliased
aliases
aliasing
alibi
alibis
alien
alienate
alienated
alienates
alienating
alienation
aliens
alight
align
aligned
aligning
alignment
alignments
aligns
alike
aliment
aliments
alimony
alive
alkali
alkaline
alkalis
alkaloid
alkaloids
alkyl
all
allay
allayed
allaying
allays
allegation
allegations
allege
alleged
allegedly
alleges
allegiance
allegiances
alleging
allegoric
allegorical
allegorically
allegories
allegory
allegretto
allegrettos
allele
alleles
allemande
allergic
allergies
allergy
alleviate
alleviated
alleviates
alleviating
alleviation
alley
alleys
alleyway
alleyways
alliance
alliances
allied
allies
alligator
alligators
alliteration
alliterations
alliterative
allocatable
allocate
allocated
allocates
allocating
allocation
allocations
allocator
allocators
allophone
allophones
allophonic
allot
allotment
allotments
allots
allotted
allotter
allotting
allow
allowable
allowably
allowance
allowances
allowed
allowing
allows
alloy
alloys
allude
alluded
alludes
alluding
allure
allurement
alluring
allusion
allusions
allusive
allusiveness
ally
allying
alma
almanac
almanacs
almighty
almond
almonds
almoner
almost
alms
almsman
alnico
aloe
aloes
aloft
aloha
alone
aloneness
along
alongside
aloof
aloofness
aloud
alpha
alphabet
alphabetic
alphabetical
alphabetically
alphabetics
alphabetize
alphabetized
alphabetizes
alphabetizing
alphabets
alphanumeric
alpine
already
also
altar
altars
alter
alterable
alteration
alterations
altercation
altercations
altered
alterer
alterers
altering
alternate
alternated
alternately
alternates
alternating
alternation
alternations
alternative
alternatively
alternatives
alternator
alternators
alters
although
altitude
altitudes
altogether
altruism
altruist
altruistic
altruistically
alum
aluminum
alumna
alumnae
alumni
alumnus
alundum
alveolar
alveoli
alveolus
always
am
amain
amalgam
amalgamate
amalgamated
amalgamates
amalgamating
amalgamation
amalgams
amanuensis
amaretto
amass
amassed
amasses
amassing
amateur
amateurish
amateurishness
amateurism
amateurs
amatory
amaze
amazed
amazedly
amazement
amazer
amazers
amazes
amazing
amazingly
ambassador
ambassadors
amber
ambiance
ambidextrous
ambidextrously
ambient
ambiguities
ambiguity
ambiguous
ambiguously
ambition
ambitions
ambitious
ambitiously
ambivalence
ambivalent
ambivalently
amble
ambled
ambler
ambles
ambling
ambrosial
ambulance
ambulances
ambulatory
ambuscade
ambush
ambushed
ambushes
ameliorate
ameliorated
ameliorating
amelioration
amen
amenable
amend
amended
amending
amendment
amendments
amends
amenities
amenity
amenorrhea
americium
amiable
amicable
amicably
amid
amide
amidst
amigo
amino
amiss
amity
ammo
ammonia
ammoniac
ammonium
ammunition
amnesty
amoeba
amoebae
amoebas
amok
among
amongst
amoral
amorality
amorist
amorous
amorphous
amorphously
amortize
amortized
amortizes
amortizing
amount
amounted
amounter
amounters
amounting
amounts
amour
amperage
ampere
amperes
ampersand
ampersands
amphetamine
amphetamines
amphibian
amphibians
amphibious
amphibiously
amphibology
amphitheater
amphitheaters
ample
amplification
amplified
amplifier
amplifiers
amplifies
amplify
amplifying
amplitude
amplitudes
amply
ampoule
ampoules
amputate
amputated
amputates
amputating
amulet
amulets
amuse
amused
amusedly
amusement
amusements
amuser
amusers
amuses
amusing
amusingly
amyl
an
anachronism
anachronisms
anachronistically
anaconda
anacondas
anaerobic
anagram
anagrams
anal
analog
analogical
analogies
analogous
analogously
analogue
analogues
analogy
analyses
analysis
analyst
analysts
analytic
analytical
analytically
analyticities
analyticity
analyzable
analyze
analyzed
analyzer
analyzers
analyzes
analyzing
anaphora
anaphoric
anaphorically
anaplasmosis
anarchic
anarchical
anarchism
anarchist
anarchists
anarchy
anastomoses
anastomosis
anastomotic
anathema
anatomic
anatomical
anatomically
anatomy
ancestor
ancestors
ancestral
ancestry
anchor
anchorage
anchorages
anchored
anchoring
anchorite
anchoritism
anchors
anchovies
anchovy
ancient
anciently
ancients
ancillary
and
anders
anding
anecdotal
anecdote
anecdotes
anechoic
anemia
anemic
anemometer
anemometers
anemometry
anemone
anesthesia
anesthetic
anesthetically
anesthetics
anesthetize
anesthetized
anesthetizes
anesthetizing
anew
angel
angelic
angels
anger
angered
angering
angers
angiography
angle
angled
angler
anglers
angling
angrier
angriest
angrily
angry
angst
angstrom
anguish
anguished
angular
angularly
anhydrous
anhydrously
aniline
animal
animals
animate
animated
animatedly
animately
animateness
animates
animating
animation
animations
animator
animators
animism
animized
animosity
anion
anionic
anions
anise
aniseikonic
anisotropic
anisotropy
ankle
ankles
annal
annals
annex
annexation
annexed
annexes
annexing
annihilate
annihilated
annihilates
annihilating
annihilation
anniversaries
anniversary
annotate
annotated
annotates
annotating
annotation
annotations
announce
announced
announcement
announcements
announcer
announcers
announces
announcing
annoy
annoyance
annoyances
annoyed
annoyer
annoyers
annoying
annoyingly
annoys
annual
annually
annuals
annuity
annul
annular
annuli
annulled
annulling
annulment
annulments
annuls
annulus
annum
annunciate
annunciated
annunciates
annunciating
annunciator
annunciators
anode
anodes
anodize
anodized
anodizes
anoint
anointed
anointing
anoints
anomalies
anomalous
anomalously
anomaly
anomic
anomie
anon
anonymity
anonymous
anonymously
anorexia
another
answer
answerable
answered
answerer
answerers
answering
answers
ant
antagonism
antagonisms
antagonist
antagonistic
antagonistically
antagonists
antagonize
antagonized
antagonizes
antagonizing
antarctic
ante
anteater
anteaters
antecedent
antecedents
antedate
antelope
antelopes
antenna
antennae
antennas
anterior
anthem
anthems
anther
anthologies
anthology
anthracite
anthropological
anthropologically
anthropologist
anthropologists
anthropology
anthropomorphic
anthropomorphically
anti
antibacterial
antibiotic
antibiotics
antibodies
antibody
antic
anticipate
anticipated
anticipates
anticipating
anticipation
anticipations
anticipatory
anticoagulation
anticompetitive
antics
antidisestablishmentarianism
antidote
antidotes
antiformant
antifundamentalist
antigen
antigens
antihistorical
antimicrobial
antimony
antinomian
antinomy
antipathy
antiphonal
antipode
antipodes
antiquarian
antiquarians
antiquate
antiquated
antique
antiques
antiquities
antiquity
antiredeposition
antiresonance
antiresonator
antisemitic
antisemitism
antiseptic
antisera
antiserum
antislavery
antisocial
antisubmarine
antisymmetric
antisymmetry
antithesis
antithetical
antithyroid
antitoxin
antitoxins
antitrust
antler
antlered
ants
anus
anvil
anvils
anxieties
anxiety
anxious
anxiously
any
anybody
anyhow
anymore
anyone
anyplace
anything
anytime
anyway
anywhere
aorta
apace
apart
apartment
apartments
apathetic
apathy
ape
aped
aperiodic
aperiodicity
aperture
apes
apex
aphasia
aphasic
aphelion
aphid
aphids
aphonic
aphorism
aphorisms
apiaries
apiary
apical
apiece
aping
apish
aplenty
aplomb
apocalypse
apocalyptic
apocryphal
apogee
apogees
apologetic
apologetically
apologia
apologies
apologist
apologists
apologize
apologized
apologizes
apologizing
apology
apostate
apostle
apostles
apostolic
apostrophe
apostrophes
apothecary
apothegm
apotheoses
apotheosis
appall
appalled
appalling
appallingly
appanage
apparatus
apparel
appareled
apparent
apparently
apparition
apparitions
appeal
appealed
appealer
appealers
appealing
appealingly
appeals
appear
appearance
appearances
appeared
appearer
appearers
appearing
appears
appease
appeased
appeasement
appeases
appeasing
appellant
appellants
appellate
appellation
append
appendage
appendages
appended
appender
appenders
appendices
appendicitis
appending
appendix
appendixes
appends
appertain
appertains
appetite
appetites
appetizer
appetizing
applaud
applauded
applauding
applauds
applause
apple
applejack
apples
appliance
appliances
applicability
applicable
applicant
applicants
application
applications
applicative
applicatively
applicator
applicators
applied
applier
appliers
applies
applique
apply
applying
appoint
appointed
appointee
appointees
appointer
appointers
appointing
appointive
appointment
appointments
appoints
apportion
apportioned
apportioning
apportionment
apportionments
apportions
apposite
appraisal
appraisals
appraise
appraised
appraiser
appraisers
appraises
appraising
appraisingly
appreciable
appreciably
appreciate
appreciated
appreciates
appreciating
appreciation
appreciations
appreciative
appreciatively
apprehend
apprehended
apprehensible
apprehension
apprehensions
apprehensive
apprehensively
apprehensiveness
apprentice
apprenticed
apprentices
apprenticeship
apprise
apprised
apprises
apprising
approach
approachability
approachable
approached
approacher
approachers
approaches
approaching
approbate
approbation
appropriate
appropriated
appropriately
appropriateness
appropriates
appropriating
appropriation
appropriations
appropriator
appropriators
approval
approvals
approve
approved
approver
approvers
approves
approving
approvingly
approximate
approximated
approximately
approximates
approximating
approximation
approximations
appurtenance
appurtenances
apricot
apricots
apron
aprons
apropos
apse
apsis
apt
aptitude
aptitudes
aptly
aptness
aqua
aquaria
aquarium
aquatic
aqueduct
aqueducts
aqueous
aquifer
aquifers
arabesque
arable
arachnid
arachnids
arbiter
arbiters
arbitrarily
arbitrariness
arbitrary
arbitrate
arbitrated
arbitrates
arbitrating
arbitration
arbitrator
arbitrators
arbor
arboreal
arbors
arc
arcade
arcaded
arcades
arcane
arced
arch
archaic
archaically
archaicness
archaism
archaize
archangel
archangels
archbishop
archdiocese
archdioceses
arched
archenemy
archeological
archeologist
archeology
archers
archery
arches
archetype
archfool
arching
archipelago
archipelagoes
architect
architectonic
architects
architectural
architecturally
architecture
architectures
archival
archive
archived
archiver
archivers
archives
archiving
archivist
archly
arcing
arclike
arcs
arcsine
arctangent
arctic
ardent
ardently
ardor
arduous
arduously
arduousness
are
area
areas
arena
arenas
argon
argonauts
argot
arguable
arguably
argue
argued
arguer
arguers
argues
arguing
argument
argumentation
argumentative
arguments
arid
aridity
aright
arise
arisen
ariser
arises
arising
arisings
aristocracy
aristocrat
aristocratic
aristocratically
aristocrats
arithmetic
arithmetical
arithmetically
arithmetics
arithmetize
arithmetized
arithmetizes
ark
arm
armadillo
armadillos
armament
armaments
armchair
armchairs
armed
armer
armers
armful
armhole
armies
arming
armistice
armload
armor
armored
armorer
armory
armpit
armpits
arms
army
aroma
aromas
aromatic
arose
around
arousal
arouse
aroused
arouses
arousing
arpeggio
arpeggios
arrack
arraign
arraigned
arraigning
arraignment
arraignments
arraigns
arrange
arranged
arrangement
arrangements
arranger
arrangers
arranges
arranging
arrant
array
arrayed
arrays
arrears
arrest
arrested
arrester
arresters
arresting
arrestingly
arrestor
arrestors
arrests
arrival
arrivals
arrive
arrived
arrives
arriving
arrogance
arrogant
arrogantly
arrogate
arrogated
arrogates
arrogating
arrogation
arrow
arrowed
arrowhead
arrowheads
arrows
arroyo
arroyos
arsenal
arsenals
arsenic
arsine
arson
art
arterial
arteries
arteriolar
arteriole
arterioles
arteriosclerosis
artery
artful
artfully
artfulness
arthritis
arthropod
arthropods
artichoke
artichokes
article
articles
articulate
articulated
articulately
articulateness
articulates
articulating
articulation
articulations
articulator
articulators
articulatory
artifact
artifacts
artifice
artificer
artifices
artificial
artificialities
artificiality
artificially
artificialness
artillerist
artillery
artisan
artisans
artist
artistic
artistically
artistry
artists
artless
arts
artwork
as
asbestos
ascend
ascendancy
ascendant
ascended
ascendency
ascendent
ascender
ascenders
ascending
ascends
ascension
ascensions
ascent
ascertain
ascertainable
ascertained
ascertaining
ascertains
ascetic
asceticism
ascetics
ascot
ascribable
ascribe
ascribed
ascribes
ascribing
ascription
aseptic
ash
ashamed
ashamedly
ashen
ashes
ashman
ashore
ashtray
ashtrays
aside
asinine
ask
askance
asked
asker
askers
askew
asking
asks
asleep
asocial
asp
asparagus
aspect
aspects
aspen
aspersion
aspersions
asphalt
asphyxia
aspic
aspirant
aspirants
aspirate
aspirated
aspirates
aspirating
aspiration
aspirations
aspirator
aspirators
aspire
aspired
aspires
aspirin
aspiring
aspirins
ass
assail
assailant
assailants
assailed
assailing
assails
assassin
assassinate
assassinated
assassinates
assassinating
assassination
assassinations
assassins
assault
assaulted
assaulting
assaults
assay
assayed
assaying
assemblage
assemblages
assemble
assembled
assembler
assemblers
assembles
assemblies
assembling
assembly
assent
assented
assenter
assenting
assents
assert
asserted
asserter
asserters
asserting
assertion
assertions
assertive
assertively
assertiveness
asserts
asses
assess
assessed
assesses
assessing
assessment
assessments
assessor
assessors
asset
assets
assiduity
assiduous
assiduously
assign
assignable
assigned
assignee
assignees
assigner
assigners
assigning
assignment
assignments
assigns
assimilate
assimilated
assimilates
assimilating
assimilation
assimilations
assist
assistance
assistances
assistant
assistants
assistantship
assistantships
assisted
assisting
assists
associate
associated
associates
associating
association
associational
associations
associative
associatively
associativity
associator
associators
assonance
assonant
assort
assorted
assortment
assortments
assorts
assuage
assuaged
assuages
assume
assumed
assumes
assuming
assumption
assumptions
assurance
assurances
assure
assured
assuredly
assurer
assurers
assures
assuring
assuringly
astatine
aster
asterisk
asterisks
asteroid
asteroidal
asteroids
asters
asthma
astonish
astonished
astonishes
astonishing
astonishingly
astonishment
astound
astounded
astounding
astounds
astral
astray
astride
astringency
astringent
astrology
astronaut
astronautics
astronauts
astronomer
astronomers
astronomical
astronomically
astronomy
astrophysical
astrophysics
astute
astutely
astuteness
asunder
asylum
asymmetric
asymmetrically
asymmetry
asymptomatically
asymptote
asymptotes
asymptotic
asymptotically
asynchronism
asynchronous
asynchronously
asynchrony
at
atavistic
ate
atemporal
atheism
atheist
atheistic
atheists
atherosclerosis
athlete
athletes
athletic
athleticism
athletics
atlas
atmosphere
atmospheres
atmospheric
atoll
atolls
atom
atomic
atomically
atomics
atomization
atomize
atomized
atomizes
atomizing
atoms
atonal
atonally
atone
atoned
atonement
atones
atop
atrocious
atrociously
atrocities
atrocity
atrophic
atrophied
atrophies
atrophy
atrophying
attach
attache
attached
attacher
attachers
attaches
attaching
attachment
attachments
attack
attackable
attacked
attacker
attackers
attacking
attacks
attain
attainable
attainably
attained
attainer
attainers
attaining
attainment
attainments
attains
attempt
attempted
attempter
attempters
attempting
attempts
attend
attendance
attendances
attendant
attendants
attended
attendee
attendees
attender
attenders
attending
attends
attention
attentional
attentionality
attentions
attentive
attentively
attentiveness
attenuate
attenuated
attenuates
attenuating
attenuation
attenuator
attenuators
attest
attested
attesting
attests
attic
attics
attire
attired
attires
attiring
attitude
attitudes
attitudinal
attorney
attorneys
attract
attracted
attracting
attraction
attractions
attractive
attractively
attractiveness
attractor
attractors
attracts
attributable
attribute
attributed
attributes
attributing
attribution
attributions
attributive
attributively
attrition
attune
attuned
attunes
attuning
atypical
atypically
auburn
auction
auctioneer
auctioneers
audacious
audaciously
audaciousness
audacity
audible
audibly
audience
audiences
audio
audiogram
audiograms
audiological
audiologist
audiologists
audiology
audiometer
audiometers
audiometric
audiometry
audit
audited
auditing
audition
auditioned
auditioning
auditions
auditor
auditorium
auditors
auditory
audits
auger
augers
aught
augment
augmentation
augmented
augmenting
augments
augur
augurs
august
augustly
augustness
aunt
aunts
aura
aural
aurally
auras
aureole
aureomycin
aurora
auscultate
auscultated
auscultates
auscultating
auscultation
auscultations
auspice
auspices
auspicious
auspiciously
austere
austerely
austerity
authentic
authentically
authenticate
authenticated
authenticates
authenticating
authentication
authentications
authenticator
authenticators
authenticity
author
authored
authoring
authoritarian
authoritarianism
authoritative
authoritatively
authorities
authority
authorization
authorizations
authorize
authorized
authorizer
authorizers
authorizes
authorizing
authors
authorship
autism
autistic
auto
autobiographic
autobiographical
autobiographies
autobiography
autocollimator
autocorrelate
autocorrelation
autocracies
autocracy
autocrat
autocratic
autocratically
autocrats
autodecrement
autodecremented
autodecrements
autodialer
autofluorescence
autograph
autographed
autographing
autographs
autoincrement
autoincremented
autoincrements
autoindex
autoindexing
automata
automate
automated
automates
automatic
automatically
automating
automation
automaton
automobile
automobiles
automotive
autonavigator
autonavigators
autonomic
autonomous
autonomously
autonomy
autopilot
autopilots
autopsied
autopsies
autopsy
autoregressive
autos
autosuggestibility
autotransformer
autumn
autumnal
autumns
auxiliaries
auxiliary
avail
availabilities
availability
available
availably
availed
availer
availers
availing
avails
avalanche
avalanched
avalanches
avalanching
avant
avarice
avaricious
avariciously
avenge
avenged
avenger
avenges
avenging
avenue
avenues
aver
average
averaged
averages
averaging
averred
averrer
averring
avers
averse
aversion
aversions
avert
averted
averting
averts
avian
aviaries
aviary
aviation
aviator
aviators
avid
avidity
avidly
avionic
avionics
avocado
avocados
avocation
avocations
avoid
avoidable
avoidably
avoidance
avoided
avoider
avoiders
avoiding
avoids
avouch
avow
avowal
avowed
avows
await
awaited
awaiting
awaits
awake
awaken
awakened
awakening
awakens
awakes
awaking
award
awarded
awarder
awarders
awarding
awards
aware
awareness
awash
away
awe
awed
awesome
awful
awfully
awfulness
awhile
awkward
awkwardly
awkwardness
awl
awls
awning
awnings
awoke
awry
ax
axed
axer
axers
axes
axial
axially
axing
axiological
axiom
axiomatic
axiomatically
axiomatization
axiomatizations
axiomatize
axiomatized
axiomatizes
axiomatizing
axioms
axis
axle
axles
axolotl
axolotls
axon
axons
aye
ayes
azalea
azaleas
azimuth
azimuths
azure
babble
babbled
babbles
babbling
babe
babes
babied
babies
baboon
baboons
baby
babyhood
babying
babyish
babysit
babysitting
baccalaureate
bachelor
bachelors
bacilli
bacillus
back
backache
backaches
backarrow
backbend
backbends
backboard
backbone
backbones
backdrop
backdrops
backed
backer
backers
backfill
backfiring
background
backgrounds
backhand
backing
backlash
backlog
backlogged
backlogs
backorder
backpack
backpacks
backplane
backplanes
backplate
backs
backscatter
backscattered
backscattering
backscatters
backside
backslash
backslashes
backspace
backspaced
backspaces
backspacing
backstage
backstairs
backstitch
backstitched
backstitches
backstitching
backstop
backtrack
backtracked
backtracker
backtrackers
backtracking
backtracks
backup
backups
backward
backwardness
backwards
backwater
backwaters
backwoods
backyard
backyards
bacon
bacteria
bacterial
bacterium
bad
bade
badge
badger
badgered
badgering
badgers
badges
badlands
badly
badminton
badness
baffle
baffled
baffler
bafflers
baffling
bag
bagatelle
bagatelles
bagel
bagels
baggage
bagged
bagger
baggers
bagging
baggy
bagpipe
bagpipes
bags
bah
bail
bailiff
bailiffs
bailing
bait
baited
baiter
baiting
baits
bake
baked
baker
bakeries
bakers
bakery
bakes
baking
baklava
balalaika
balalaikas
balance
balanced
balancer
balancers
balances
balancing
balconies
balcony
bald
balding
baldly
baldness
bale
baleful
baler
bales
balk
balkanized
balkanizing
balked
balkiness
balking
balks
balky
ball
ballad
ballads
ballast
ballasts
balled
baller
ballerina
ballerinas
ballers
ballet
ballets
ballgown
balling
ballistic
ballistics
balloon
ballooned
ballooner
ballooners
ballooning
balloons
ballot
ballots
ballpark
ballparks
ballplayer
ballplayers
ballroom
ballrooms
balls
ballyhoo
balm
balms
balmy
balsa
balsam
balustrade
balustrades
bamboo
ban
banal
banally
banana
bananas
band
bandage
bandaged
bandages
bandaging
banded
bandied
bandies
banding
bandit
bandits
bandpass
bands
bandstand
bandstands
bandwagon
bandwagons
bandwidth
bandwidths
bandy
bandying
bane
baneful
bang
banged
banging
bangle
bangles
bangs
banish
banished
banishes
banishing
banishment
banister
banisters
banjo
banjos
bank
banked
banker
bankers
banking
bankrupt
bankruptcies
bankruptcy
bankrupted
bankrupting
bankrupts
banned
banner
banners
banning
banquet
banqueting
banquetings
banquets
bans
banshee
banshees
bantam
banter
bantered
bantering
banters
baptism
baptismal
baptisms
baptistery
baptistries
baptistry
baptize
baptized
baptizes
baptizing
bar
barb
barbarian
barbarians
barbaric
barbarism
barbarities
barbarity
barbarous
barbarously
barbecue
barbecued
barbecues
barbed
barbell
barbells
barber
barbital
barbiturate
barbiturates
barbs
bard
bards
bare
bared
barefaced
barefoot
barefooted
barely
bareness
barer
bares
barest
barflies
barfly
bargain
bargained
bargaining
bargains
barge
barges
barging
baring
baritone
baritones
barium
bark
barked
barker
barkers
barking
barks
barley
barn
barns
barnstorm
barnstormed
barnstorming
barnstorms
barnyard
barnyards
barometer
barometers
barometric
baron
baroness
baronial
baronies
barons
barony
baroque
baroqueness
barrack
barracks
barrage
barrages
barred
barrel
barrelled
barrelling
barrels
barren
barrenness
barricade
barricades
barrier
barriers
barring
barringer
barrow
bars
bartender
bartenders
barter
bartered
bartering
barters
basal
basalt
base
baseball
baseballs
baseband
baseboard
baseboards
based
baseless
baseline
baselines
basely
baseman
basement
basements
baseness
baser
bases
bash
bashed
bashes
bashful
bashfulness
bashing
basic
basically
basics
basil
basin
basing
basins
basis
bask
basked
basket
basketball
basketballs
baskets
basking
bass
basses
basset
bassinet
bassinets
bastard
bastards
baste
basted
bastes
basting
bastion
bastions
bat
batch
batched
batches
bath
bathe
bathed
bather
bathers
bathes
bathing
bathos
bathrobe
bathrobes
bathroom
bathrooms
baths
bathtub
bathtubs
baton
batons
bats
battalion
battalions
batted
batten
battens
batter
battered
batteries
battering
batters
battery
batting
battle
battled
battlefield
battlefields
battlefront
battlefronts
battleground
battlegrounds
battlement
battlements
battler
battlers
battles
battleship
battleships
battling
bauble
baubles
baud
bauxite
bawdy
bawl
bawled
bawling
bawls
bay
bayed
baying
bayonet
bayonets
bayou
bayous
bays
bazaar
bazaars
be
beach
beached
beaches
beachhead
beachheads
beaching
beacon
beacons
bead
beaded
beading
beadle
beadles
beads
beady
beagle
beagles
beak
beaked
beaker
beakers
beaks
beam
beamed
beamer
beamers
beaming
beams
bean
beanbag
beaned
beaner
beaners
beaning
beans
bear
bearable
bearably
beard
bearded
beardless
beards
bearer
bearers
bearing
bearings
bearish
bears
beast
beastly
beasts
beat
beatable
beatably
beaten
beater
beaters
beatific
beatification
beatify
beating
beatings
beatitude
beatitudes
beatnik
beatniks
beats
beau
beaus
beauteous
beauteously
beauties
beautifications
beautified
beautifier
beautifiers
beautifies
beautiful
beautifully
beautify
beautifying
beauty
beaver
beavers
becalm
becalmed
becalming
becalms
became
because
beck
beckon
beckoned
beckoning
beckons
become
becomes
becoming
becomingly
bed
bedazzle
bedazzled
bedazzlement
bedazzles
bedazzling
bedbug
bedbugs
bedded
bedder
bedders
bedding
bedevil
bedeviled
bedeviling
bedevils
bedfast
bedlam
bedpost
bedposts
bedraggle
bedraggled
bedridden
bedrock
bedroom
bedrooms
beds
bedside
bedspread
bedspreads
bedspring
bedsprings
bedstead
bedsteads
bedtime
bee
beech
beechen
beecher
beef
beefed
beefer
beefers
beefing
beefs
beefsteak
beefy
beehive
beehives
been
beep
beeps
beer
beers
bees
beet
beetle
beetled
beetles
beetling
beets
befall
befallen
befalling
befalls
befell
befit
befits
befitted
befitting
befog
befogged
befogging
before
beforehand
befoul
befouled
befouling
befouls
befriend
befriended
befriending
befriends
befuddle
befuddled
befuddles
befuddling
beg
began
beget
begets
begetting
beggar
beggarly
beggars
beggary
begged
begging
begin
beginner
beginners
beginning
beginnings
begins
begot
begotten
begrudge
begrudged
begrudges
begrudging
begrudgingly
begs
beguile
beguiled
beguiles
beguiling
begun
behalf
behave
behaved
behaves
behaving
behavior
behavioral
behaviorally
behaviorism
behavioristic
behaviors
behead
beheading
beheld
behemoth
behemoths
behest
behind
behold
beholden
beholder
beholders
beholding
beholds
behoove
behooves
beige
being
beings
belabor
belabored
belaboring
belabors
belated
belatedly
belay
belayed
belaying
belays
belch
belched
belches
belching
belfries
belfry
belie
belied
belief
beliefs
belies
believable
believably
believe
believed
believer
believers
believes
believing
belittle
belittled
belittles
belittling
bell
bellboy
bellboys
belle
belles
bellhop
bellhops
bellicose
bellicosity
bellies
belligerence
belligerent
belligerently
belligerents
bellman
bellmen
bellow
bellowed
bellowing
bellows
bells
bellum
bellwether
bellwethers
belly
bellyache
bellyfull
belong
belonged
belonging
belongings
belongs
beloved
below
belt
belted
belting
belts
bely
belying
bemoan
bemoaned
bemoaning
bemoans
bench
benched
benches
benchmark
benchmarking
benchmarks
bend
bendable
benders
bending
bends
beneath
benediction
benedictions
benefactor
benefactors
beneficence
beneficences
beneficent
beneficial
beneficially
beneficiaries
beneficiary
benefit
benefited
benefiting
benefits
benefitted
benefitting
benevolence
benevolent
benighted
benign
benignly
bent
benzene
bequeath
bequeathal
bequeathed
bequeathing
bequeaths
bequest
bequests
berate
berated
berates
berating
bereave
bereaved
bereavement
bereavements
bereaves
bereaving
bereft
beret
berets
beribboned
beriberi
berkelium
berne
berries
berry
berserk
berth
berths
beryl
beryllium
beseech
beseeches
beseeching
beset
besets
besetting
beside
besides
besiege
besieged
besieger
besiegers
besieging
besmirch
besmirched
besmirches
besmirching
besotted
besotter
besotting
besought
bespeak
bespeaks
bespectacled
bespoke
best
bested
bestial
besting
bestir
bestirring
bestow
bestowal
bestowed
bests
bestseller
bestsellers
bestselling
bet
beta
betatron
betel
betide
betray
betrayal
betrayed
betrayer
betraying
betrays
betroth
betrothal
betrothed
bets
better
bettered
bettering
betterment
betterments
betters
betting
between
betwixt
bevel
beveled
beveling
bevels
beverage
beverages
bevy
bewail
bewailed
bewailing
bewails
beware
bewhiskered
bewilder
bewildered
bewildering
bewilderingly
bewilderment
bewilders
bewitch
bewitched
bewitches
bewitching
beyond
biannual
bias
biased
biases
biasing
bib
bibbed
bibbing
bibles
biblical
biblically
bibliographic
bibliographical
bibliographies
bibliography
bibliophile
bibs
bicameral
bicarbonate
bicentennial
bicep
biceps
bicker
bickered
bickering
bickers
biconcave
biconnected
biconvex
bicycle
bicycled
bicycler
bicyclers
bicycles
bicycling
bid
biddable
bidden
bidder
bidders
biddies
bidding
biddy
bide
bidirectional
bids
biennial
biennium
bier
bifocal
bifocals
bifurcate
big
bigger
biggest
bight
bights
bigness
bigot
bigoted
bigotry
bigots
biharmonic
bijection
bijections
bijective
bijectively
bike
bikes
biking
bikini
bikinis
bilabial
bilateral
bilaterally
bile
bilge
bilges
bilinear
bilingual
bilk
bilked
bilking
bilks
bill
billboard
billboards
billed
biller
billers
billet
billeted
billeting
billets
billiard
billiards
billing
billion
billions
billionth
billow
billowed
billows
bills
bimetallic
bimetallism
bimodal
bimolecular
bimonthlies
bimonthly
bin
binaries
binary
binaural
bind
binder
binders
binding
bindings
binds
bing
binge
binges
bingo
binocular
binoculars
binomial
bins
binuclear
biochemical
biochemist
biochemistry
biofeedback
biographer
biographers
biographic
biographical
biographically
biographies
biography
biological
biologically
biologist
biologists
biology
biomedical
biomedicine
biophysical
biophysicist
biophysics
biopsies
biopsy
bioscience
biosphere
biostatistic
biosynthesize
biota
biotic
bipartisan
bipartite
biped
bipeds
biplane
biplanes
bipolar
biracial
birch
birchen
birches
bird
birdbath
birdbaths
birdie
birdied
birdies
birdlike
birds
birefringence
birefringent
birth
birthday
birthdays
birthed
birthplace
birthplaces
birthright
birthrights
births
biscuit
biscuits
bisect
bisected
bisecting
bisection
bisections
bisector
bisectors
bisects
bishop
bishops
bismuth
bison
bisons
bisque
bisques
bistable
bistate
bit
bitch
bitches
bite
biter
biters
bites
biting
bitingly
bitmap
bits
bitten
bitter
bitterer
bitterest
bitterly
bitterness
bitternut
bitterroot
bitters
bittersweet
bitumen
bituminous
bitwise
bivalve
bivalves
bivariate
bivouac
bivouacs
biweekly
bizarre
blab
blabbed
blabbermouth
blabbermouths
blabbing
blabs
black
blackberries
blackberry
blackbird
blackbirds
blackboard
blackboards
blacked
blacken
blackened
blackening
blackens
blacker
blackest
blacking
blackjack
blackjacks
blacklist
blacklisted
blacklisting
blacklists
blackly
blackmail
blackmailed
blackmailer
blackmailers
blackmailing
blackmails
blackness
blackout
blackouts
blacks
blacksmith
blacksmiths
bladder
bladders
blade
blades
blamable
blame
blamed
blameless
blamelessness
blamer
blamers
blames
blameworthy
blaming
blanch
blanched
blanches
blanching
bland
blandly
blandness
blank
blanked
blanker
blankest
blanket
blanketed
blanketer
blanketers
blanketing
blankets
blanking
blankly
blankness
blanks
blare
blared
blares
blaring
blase
blaspheme
blasphemed
blasphemes
blasphemies
blaspheming
blasphemous
blasphemously
blasphemousness
blasphemy
blast
blasted
blaster
blasters
blasting
blasts
blatant
blatantly
blaze
blazed
blazer
blazers
blazes
blazing
bleach
bleached
bleacher
bleachers
bleaches
bleaching
bleak
bleaker
bleakly
bleakness
blear
bleary
bleat
bleating
bleats
bled
bleed
bleeder
bleeding
bleedings
bleeds
blemish
blemishes
blend
blended
blender
blending
blends
bless
blessed
blessing
blessings
blew
blight
blighted
blimp
blimps
blind
blinded
blinder
blinders
blindfold
blindfolded
blindfolding
blindfolds
blinding
blindingly
blindly
blindness
blinds
blink
blinked
blinker
blinkers
blinking
blinks
blip
blips
bliss
blissful
blissfully
blister
blistered
blistering
blisters
blithe
blithely
blitz
blitzes
blitzkrieg
blizzard
blizzards
bloat
bloated
bloater
bloating
bloats
blob
blobs
bloc
block
blockade
blockaded
blockades
blockading
blockage
blockages
blocked
blocker
blockers
blockhouse
blockhouses
blocking
blocks
blocs
bloke
blokes
blond
blonde
blondes
blonds
blood
bloodbath
blooded
bloodhound
bloodhounds
bloodied
bloodiest
bloodless
bloods
bloodshed
bloodshot
bloodstain
bloodstained
bloodstains
bloodstream
bloody
bloom
bloomed
bloomers
blooming
blooms
blooper
blossom
blossomed
blossoms
blot
blots
blotted
blotting
blouse
blouses
blow
blower
blowers
blowfish
blowing
blown
blowout
blows
blowup
blubber
bludgeon
bludgeoned
bludgeoning
bludgeons
blue
blueberries
blueberry
bluebird
bluebirds
bluebonnet
bluebonnets
bluefish
blueness
blueprint
blueprints
bluer
blues
bluest
bluestocking
bluff
bluffing
bluffs
bluing
bluish
blunder
blunderbuss
blundered
blundering
blunderings
blunders
blunt
blunted
blunter
bluntest
blunting
bluntly
bluntness
blunts
blur
blurb
blurred
blurring
blurry
blurs
blurt
blurted
blurting
blurts
blush
blushed
blushes
blushing
bluster
blustered
blustering
blusters
blustery
boa
boar
board
boarded
boarder
boarders
boarding
boardinghouse
boardinghouses
boards
boast
boasted
boaster
boasters
boastful
boastfully
boasting
boastings
boasts
boat
boater
boaters
boathouse
boathouses
boating
boatload
boatloads
boatman
boatmen
boats
boatsman
boatsmen
boatswain
boatswains
boatyard
boatyards
bob
bobbed
bobbin
bobbing
bobbins
bobby
bobolink
bobolinks
bobs
bobwhite
bobwhites
bode
bodes
bodice
bodied
bodies
bodily
body
bodybuilder
bodybuilders
bodybuilding
bodyguard
bodyguards
bodyweight
bog
bogeymen
bogged
boggle
boggled
boggles
boggling
bogs
bogus
boil
boiled
boiler
boilerplate
boilers
boiling
boils
boisterous
boisterously
bold
bolder
boldest
boldface
boldly
boldness
boll
bolster
bolstered
bolstering
bolsters
bolt
bolted
bolting
bolts
bomb
bombard
bombarded
bombarding
bombardment
bombards
bombast
bombastic
bombed
bomber
bombers
bombing
bombings
bombproof
bombs
bonanza
bonanzas
bond
bondage
bonded
bonder
bonders
bonding
bonds
bondsman
bondsmen
bone
boned
boner
boners
bones
bonfire
bonfires
bong
boning
bonnet
bonneted
bonnets
bonny
bonus
bonuses
bony
boo
boob
booboo
booby
book
bookcase
bookcases
booked
booker
bookers
bookie
bookies
booking
bookings
bookish
bookkeeper
bookkeepers
bookkeeping
booklet
booklets
bookmark
books
bookseller
booksellers
bookshelf
bookshelves
bookstore
bookstores
bookworm
boolean
boom
boomed
boomerang
boomerangs
booming
booms
boon
boor
boorish
boors
boos
boost
boosted
booster
boosting
boosts
boot
bootable
booted
booth
booths
booting
bootleg
bootlegged
bootlegger
bootleggers
bootlegging
bootlegs
boots
bootstrap
bootstrapped
bootstrapping
bootstraps
booty
booze
borate
borates
borax
bordello
bordellos
border
bordered
bordering
borderings
borderland
borderlands
borderline
borders
bore
bored
boredom
borer
bores
boric
boring
born
borne
boron
borough
boroughs
borrow
borrowed
borrower
borrowers
borrowing
borrows
bosom
bosoms
boss
bossed
bosses
bosun
botanical
botanist
botanists
botany
botch
botched
botcher
botchers
botches
botching
both
bother
bothered
bothering
bothers
bothersome
bottle
bottled
bottleneck
bottlenecks
bottler
bottlers
bottles
bottling
bottom
bottomed
bottoming
bottomless
bottoms
botulinus
botulism
bouffant
bough
boughs
bought
boulder
boulders
boulevard
boulevards
bounce
bounced
bouncer
bounces
bouncing
bouncy
bound
boundaries
boundary
bounded
bounden
bounding
boundless
boundlessness
bounds
bounteous
bounteously
bounties
bountiful
bounty
bouquet
bouquets
bourbon
bourgeois
bourgeoisie
boustrophedon
boustrophedonic
bout
boutique
bouts
bovine
bovines
bow
bowdlerize
bowdlerized
bowdlerizes
bowdlerizing
bowed
bowel
bowels
bower
bowers
bowing
bowl
bowled
bowler
bowlers
bowline
bowlines
bowling
bowls
bowman
bows
bowstring
bowstrings
box
boxcar
boxcars
boxed
boxer
boxers
boxes
boxing
boxtop
boxtops
boxwood
boy
boycott
boycotted
boycotts
boyfriend
boyfriends
boyhood
boyish
boyishness
boys
bra
brace
braced
bracelet
bracelets
braces
bracing
bracket
bracketed
bracketing
brackets
brackish
brae
braes
brag
bragged
bragger
bragging
brags
braid
braided
braiding
braids
brain
brainchild
brained
braining
brains
brainstem
brainstems
brainstorm
brainstorms
brainwash
brainwashed
brainwashes
brainwashing
brainy
brake
braked
brakeman
brakes
braking
bramble
brambles
brambly
bran
branch
branched
branches
branching
branchings
brand
branded
branding
brandish
brandishes
brandishing
brands
brandy
brandywine
bras
brash
brashly
brashness
brass
brasses
brassiere
brassy
brat
brats
bravado
brave
braved
bravely
braveness
braver
bravery
braves
bravest
braving
bravo
bravos
brawl
brawler
brawling
brawn
bray
brayed
brayer
braying
brays
braze
brazed
brazen
brazenly
brazenness
brazes
brazier
braziers
brazing
breach
breached
breacher
breachers
breaches
breaching
bread
breadboard
breadboards
breadbox
breadboxes
breaded
breading
breads
breadth
breadwinner
breadwinners
break
breakable
breakables
breakage
breakaway
breakdown
breakdowns
breaker
breakers
breakfast
breakfasted
breakfaster
breakfasters
breakfasting
breakfasts
breaking
breakpoint
breakpoints
breaks
breakthrough
breakthroughes
breakthroughs
breakup
breakwater
breakwaters
breast
breasted
breasts
breastwork
breastworks
breath
breathable
breathe
breathed
breather
breathers
breathes
breathing
breathless
breathlessly
breaths
breathtaking
breathtakingly
breathy
bred
breech
breeches
breed
breeder
breeding
breeds
breeze
breezes
breezily
breezy
bremsstrahlung
brethren
breve
brevet
breveted
breveting
brevets
brevity
brew
brewed
brewer
breweries
brewers
brewery
brewing
brews
briar
briars
bribe
bribed
briber
bribers
bribery
bribes
bribing
brick
brickbat
bricked
bricker
bricklayer
bricklayers
bricklaying
bricks
bridal
bride
bridegroom
brides
bridesmaid
bridesmaids
bridge
bridgeable
bridged
bridgehead
bridgeheads
bridges
bridgework
bridging
bridle
bridled
bridles
bridling
brief
briefcase
briefcases
briefed
briefer
briefest
briefing
briefings
briefly
briefness
briefs
brier
brig
brigade
brigades
brigadier
brigadiers
brigantine
bright
brighten
brightened
brightener
brighteners
brightening
brightens
brighter
brightest
brightly
brightness
brigs
brilliance
brilliancy
brilliant
brilliantly
brim
brimful
brimmed
brimming
brimstone
brindle
brindled
brine
bring
bringer
bringers
bringing
brings
brink
brinkmanship
briny
brisk
brisker
briskly
briskness
bristle
bristled
bristles
bristling
britches
brittle
brittleness
broach
broached
broaches
broaching
broad
broadband
broadcast
broadcasted
broadcaster
broadcasters
broadcasting
broadcastings
broadcasts
broaden
broadened
broadener
broadeners
broadening
broadenings
broadens
broader
broadest
broadly
broadness
broadside
brocade
brocaded
broccoli
brochure
brochures
broil
broiled
broiler
broilers
broiling
broils
broke
broken
brokenly
brokenness
broker
brokerage
brokers
bromide
bromides
bromine
bronchi
bronchial
bronchiole
bronchioles
bronchitis
bronchus
bronze
bronzed
bronzes
brooch
brooches
brood
brooder
brooding
broods
brook
brooked
brooks
broom
brooms
broomstick
broomsticks
broth
brothel
brothels
brother
brotherhood
brotherliness
brotherly
brothers
brought
brow
browbeat
browbeaten
browbeating
browbeats
brown
browned
browner
brownest
brownie
brownies
browning
brownish
brownness
browns
brows
browse
browsing
bruise
bruised
bruises
bruising
brunch
brunches
brunette
brunt
brush
brushed
brushes
brushfire
brushfires
brushing
brushlike
brushy
brusque
brusquely
brutal
brutalities
brutality
brutalize
brutalized
brutalizes
brutalizing
brutally
brute
brutes
brutish
bubble
bubbled
bubbles
bubbling
bubbly
buck
buckboard
buckboards
bucked
bucket
buckets
bucking
buckle
buckled
buckler
buckles
buckling
bucks
buckshot
buckskin
buckskins
buckwheat
bucolic
bud
budded
buddies
budding
buddy
budge
budged
budges
budget
budgetary
budgeted
budgeter
budgeters
budgeting
budgets
budging
buds
buff
buffalo
buffaloes
buffer
buffered
buffering
buffers
buffet
buffeted
buffeting
buffetings
buffets
buffoon
buffoons
buffs
bug
bugaboo
bugeyed
bugged
bugger
buggers
buggies
bugging
buggy
bugle
bugled
bugler
bugles
bugling
bugs
build
builder
builders
building
buildings
builds
buildup
buildups
built
builtin
bulb
bulbs
bulge
bulged
bulging
bulk
bulked
bulkhead
bulkheads
bulks
bulky
bull
bulldog
bulldogs
bulldoze
bulldozed
bulldozer
bulldozes
bulldozing
bulled
bullet
bulletin
bulletins
bullets
bullfrog
bullied
bullies
bulling
bullion
bullish
bullock
bulls
bullseye
bully
bullying
bulwark
bum
bumble
bumblebee
bumblebees
bumbled
bumbler
bumblers
bumbles
bumbling
bummed
bumming
bump
bumped
bumper
bumpers
bumping
bumps
bumptious
bumptiously
bumptiousness
bums
bun
bunch
bunched
bunches
bunching
bundle
bundled
bundles
bundling
bungalow
bungalows
bungle
bungled
bungler
bunglers
bungles
bungling
bunion
bunions
bunk
bunker
bunkered
bunkers
bunkhouse
bunkhouses
bunkmate
bunkmates
bunks
bunnies
bunny
buns
bunt
bunted
bunter
bunters
bunting
bunts
buoy
buoyancy
buoyant
buoyed
buoys
burden
burdened
burdening
burdens
burdensome
bureau
bureaucracies
bureaucracy
bureaucrat
bureaucratic
bureaucrats
bureaus
burgeon
burgeoned
burgeoning
burgess
burgesses
burgher
burghers
burglar
burglaries
burglarize
burglarized
burglarizes
burglarizing
burglarproof
burglarproofed
burglarproofing
burglarproofs
burglars
burglary
burial
buried
buries
burl
burlesque
burlesques
burly
burn
burned
burner
burners
burning
burningly
burnings
burnish
burnished
burnishes
burnishing
burns
burnt
burntly
burntness
burp
burped
burping
burps
burrow
burrowed
burrower
burrowing
burrows
burrs
bursa
bursitis
burst
burstiness
bursting
bursts
bursty
bury
burying
bus
busboy
busboys
bused
buses
bush
bushel
bushels
bushes
bushing
bushwhack
bushwhacked
bushwhacking
bushwhacks
bushy
busied
busier
busiest
busily
business
businesses
businesslike
businessman
businessmen
busing
buss
bussed
busses
bussing
bust
bustard
bustards
busted
buster
bustle
bustling
busts
busy
but
butane
butcher
butchered
butchers
butchery
butler
butlers
butt
butte
butted
butter
butterball
buttercup
buttered
butterer
butterers
butterfat
butterflies
butterfly
buttering
buttermilk
butternut
butters
buttery
buttes
butting
buttock
buttocks
button
buttoned
buttonhole
buttonholes
buttoning
buttons
buttress
buttressed
buttresses
buttressing
butts
butyl
butyrate
buxom
buy
buyer
buyers
buying
buys
buzz
buzzards
buzzed
buzzer
buzzes
buzzing
buzzword
buzzwords
buzzy
by
bye
bygone
bylaw
bylaws
byline
bylines
bypass
bypassed
bypasses
bypassing
byproduct
byproducts
bystander
bystanders
byte
bytes
byway
byways
byword
bywords
cab
cabal
cabana
cabaret
cabbage
cabbages
cabdriver
cabin
cabinet
cabinets
cabins
cable
cabled
cables
cabling
caboose
cabs
cache
cached
caches
caching
cackle
cackled
cackler
cackles
cackling
cacti
cactus
cadaver
cadence
cadenced
cadres
cafe
cafes
cafeteria
cage
caged
cager
cagers
cages
caging
caiman
cairn
cajole
cajoled
cajoles
cajoling
cake
caked
cakes
caking
calamities
calamitous
calamity
calcify
calcium
calculate
calculated
calculates
calculating
calculation
calculations
calculative
calculator
calculators
calculi
calculus
caldera
calendar
calendars
calf
calfskin
caliber
calibers
calibrate
calibrated
calibrates
calibrating
calibration
calibrations
calico
caliph
caliphs
call
callable
called
caller
callers
calling
calliope
callous
calloused
callously
callousness
calls
callus
calm
calmed
calmer
calmest
calming
calmingly
calmly
calmness
calms
caloric
calorie
calories
calorimeter
calorimetric
calorimetry
calumny
calve
calves
calypso
cam
came
camel
camels
camera
cameraman
cameramen
cameras
camouflage
camouflaged
camouflages
camouflaging
camp
campaign
campaigned
campaigner
campaigners
campaigning
campaigns
camped
camper
campers
campfire
campground
camping
camps
campsite
campus
campuses
can
canal
canals
canaries
canary
cancel
canceled
canceling
cancellation
cancellations
cancels
cancer
cancerous
cancers
candid
candidacy
candidate
candidates
candidly
candidness
candied
candies
candle
candlelight
candler
candles
candlestick
candlesticks
candor
candy
cane
caner
canine
canister
canker
cankerworm
cannabis
canned
cannel
canner
canners
cannery
cannibal
cannibalize
cannibalized
cannibalizes
cannibalizing
cannibals
canning
cannister
cannisters
cannon
cannonball
cannons
cannot
canny
canoe
canoes
canon
canonic
canonical
canonicalization
canonicalize
canonicalized
canonicalizes
canonicalizing
canonically
canonicals
canons
canopy
cans
cant
cantaloupe
cantankerous
cantankerously
canteen
cantilever
canto
canton
cantons
cantor
cantors
canvas
canvases
canvass
canvassed
canvasser
canvassers
canvasses
canvassing
canyon
canyons
cap
capabilities
capability
capable
capably
capacious
capaciously
capaciousness
capacitance
capacitances
capacities
capacitive
capacitor
capacitors
capacity
cape
caper
capers
capes
capillary
capita
capital
capitalism
capitalist
capitalists
capitalization
capitalizations
capitalize
capitalized
capitalizer
capitalizers
capitalizes
capitalizing
capitally
capitals
capitol
capitols
capped
capping
caprice
capricious
capriciously
capriciousness
caps
capstan
capstone
capsule
captain
captained
captaining
captains
caption
captions
captivate
captivated
captivates
captivating
captivation
captive
captives
captivity
captor
captors
capture
captured
capturer
capturers
captures
capturing
capybara
car
caramel
caravan
caravans
caraway
carbohydrate
carbolic
carbon
carbonate
carbonates
carbonation
carbonic
carbonization
carbonize
carbonized
carbonizer
carbonizers
carbonizes
carbonizing
carbons
carborundum
carbuncle
carcass
carcasses
carcinogen
carcinogenic
carcinoma
card
cardboard
carder
cardiac
cardinal
cardinalities
cardinality
cardinally
cardinals
cardiology
cardiovascular
cards
care
cared
careen
career
careers
carefree
careful
carefully
carefulness
careless
carelessly
carelessness
cares
caress
caressed
caresser
caresses
caressing
caret
caretaker
cargo
cargoes
caribou
caricature
caring
carload
carnage
carnal
carnation
carnival
carnivals
carnivorous
carnivorously
carol
carols
carp
carpenter
carpenters
carpentry
carpet
carpeted
carpeting
carpets
carport
carriage
carriages
carried
carrier
carriers
carries
carrion
carrot
carrots
carry
carrying
carryover
carryovers
cars
cart
carted
cartel
carter
carters
cartilage
carting
cartographer
cartographic
cartography
carton
cartons
cartoon
cartoons
cartridge
cartridges
carts
cartwheel
carve
carved
carver
carves
carving
carvings
cascadable
cascade
cascaded
cascades
cascading
case
cased
casement
casements
cases
casework
cash
cashed
casher
cashers
cashes
cashew
cashier
cashiers
cashing
cashmere
casing
casings
casino
cask
casket
caskets
casks
casserole
casseroles
cassette
cassock
cast
caste
caster
casters
castes
castigate
casting
castle
castled
castles
castor
casts
casual
casually
casualness
casuals
casualties
casualty
cat
cataclysmic
catalog
cataloged
cataloger
cataloging
catalogs
catalyst
catalysts
catalytic
catapult
cataract
catastrophe
catastrophes
catastrophic
catch
catchable
catcher
catchers
catches
catching
categorical
categorically
categories
categorization
categorize
categorized
categorizer
categorizers
categorizes
categorizing
category
cater
catered
caterer
catering
caterpillar
caterpillars
caters
cathedral
cathedrals
catheter
catheters
cathode
cathodes
catlike
catnip
cats
catsup
cattail
cattle
cattleman
cattlemen
caucus
caught
cauldron
cauldrons
cauliflower
caulk
causal
causality
causally
causation
causations
cause
caused
causer
causes
causeway
causeways
causing
caustic
causticly
caustics
caution
cautioned
cautioner
cautioners
cautioning
cautionings
cautions
cautious
cautiously
cautiousness
cavalier
cavalierly
cavalierness
cavalry
cave
caveat
caveats
caved
caveman
cavemen
cavern
cavernous
caverns
caves
caviar
cavil
caving
cavities
cavity
caw
cawing
cease
ceased
ceaseless
ceaselessly
ceaselessness
ceases
ceasing
cedar
cede
ceded
ceding
ceiling
ceilings
celebrate
celebrated
celebrates
celebrating
celebration
celebrations
celebrities
celebrity
celerity
celery
celestial
celestially
cell
cellar
cellars
celled
cellist
cellists
cellophane
cells
cellular
cellulose
cement
cemented
cementing
cements
cemeteries
cemetery
censor
censored
censoring
censors
censorship
censure
censured
censurer
censures
census
censuses
cent
centaur
centenary
centennial
center
centered
centering
centerpiece
centerpieces
centers
centigrade
centimeter
centimeters
centipede
centipedes
central
centralism
centralist
centralization
centralize
centralized
centralizes
centralizing
centrally
centrifugal
centrifuge
centripetal
centrist
centroid
cents
centuries
century
ceramic
cereal
cereals
cerebellum
cerebral
ceremonial
ceremonially
ceremonialness
ceremonies
ceremony
certain
certainly
certainties
certainty
certifiable
certificate
certificates
certification
certifications
certified
certifier
certifiers
certifies
certify
certifying
cessation
cessations
chafe
chafer
chaff
chaffer
chaffing
chafing
chagrin
chain
chained
chaining
chains
chair
chaired
chairing
chairlady
chairman
chairmen
chairperson
chairpersons
chairs
chairwoman
chairwomen
chalice
chalices
chalk
chalked
chalking
chalks
challenge
challenged
challenger
challengers
challenges
challenging
chamber
chambered
chamberlain
chamberlains
chambermaid
chameleon
champagne
champion
championed
championing
champions
championship
championships
chance
chanced
chancellor
chancery
chances
chancing
chandelier
chandeliers
change
changeability
changeable
changeably
changed
changeover
changer
changers
changes
changing
channel
channeled
channeling
channelled
channeller
channellers
channelling
channels
chant
chanted
chanter
chanticleer
chanticleers
chanting
chants
chaos
chaotic
chap
chapel
chapels
chaperon
chaperone
chaperoned
chaplain
chaplains
chaps
chapter
chapters
char
character
characteristic
characteristically
characteristics
characterizable
characterization
characterizations
characterize
characterized
characterizer
characterizers
characterizes
characterizing
characters
charcoal
charcoaled
charge
chargeable
charged
charger
chargers
charges
charging
chariot
chariots
charisma
charismatic
charitable
charitableness
charities
charity
charm
charmed
charmer
charmers
charming
charmingly
charms
chars
chart
chartable
charted
charter
chartered
chartering
charters
charting
chartings
chartreuse
charts
chase
chased
chaser
chasers
chases
chasing
chasm
chasms
chassis
chaste
chastely
chasteness
chastise
chastised
chastiser
chastisers
chastises
chastising
chastity
chat
chateau
chateaus
chattel
chatter
chattered
chatterer
chattering
chatters
chatting
chatty
chauffeur
chauffeured
cheap
cheapen
cheapened
cheapening
cheapens
cheaper
cheapest
cheaply
cheapness
cheat
cheated
cheater
cheaters
cheating
cheats
check
checkable
checkbook
checkbooks
checked
checker
checkerboard
checkerboarded
checkerboarding
checkers
checking
checklist
checkout
checkpoint
checkpoints
checks
checksum
checksummed
checksumming
checksums
checkup
cheek
cheekbone
cheeks
cheeky
cheer
cheered
cheerer
cheerful
cheerfully
cheerfulness
cheerily
cheeriness
cheering
cheerleader
cheerless
cheerlessly
cheerlessness
cheers
cheery
cheese
cheesecloth
cheeses
cheesy
cheetah
chef
chefs
chemical
chemically
chemicals
chemise
chemist
chemistries
chemistry
chemists
cherish
cherished
cherishes
cherishing
cherries
cherry
cherub
cherubim
cherubs
chess
chest
chestnut
chestnuts
chests
chew
chewed
chewer
chewers
chewing
chews
chic
chicanery
chick
chickadee
chickadees
chicken
chickens
chicks
chide
chided
chides
chiding
chief
chiefly
chiefs
chieftain
chieftains
chiffon
child
childbirth
childhood
childish
childishly
childishness
childlike
children
chili
chill
chilled
chiller
chillers
chillier
chilliness
chilling
chillingly
chills
chilly
chime
chimera
chimes
chimney
chimneys
chimpanzee
chin
chink
chinked
chinks
chinned
chinner
chinners
chinning
chins
chintz
chip
chipmunk
chipmunks
chips
chiropractor
chirp
chirped
chirping
chirps
chisel
chiseled
chiseler
chisels
chit
chivalrous
chivalrously
chivalrousness
chivalry
chlorine
chloroform
chlorophyll
chloroplast
chloroplasts
chock
chocks
chocolate
chocolates
choice
choices
choicest
choir
choirs
choke
choked
choker
chokers
chokes
choking
cholera
choose
chooser
choosers
chooses
choosing
chop
chopped
chopper
choppers
chopping
choppy
chops
choral
chord
chordate
chorded
chording
chords
chore
choreograph
choreography
chores
choring
chortle
chorus
chorused
choruses
chose
chosen
chowder
christen
christened
christening
christens
chromatogram
chromatograph
chromatography
chrome
chromium
chromosphere
chronic
chronicle
chronicled
chronicler
chroniclers
chronicles
chronograph
chronography
chronological
chronologically
chronologies
chronology
chrysanthemum
chubbier
chubbiest
chubbiness
chubby
chuck
chuckle
chuckled
chuckles
chucks
chum
chunk
chunks
chunky
church
churches
churchgoer
churchgoing
churchly
churchman
churchmen
churchwoman
churchwomen
churchyard
churchyards
churn
churned
churning
churns
chute
chutes
chutzpah
cicada
cider
cigar
cigarette
cigarettes
cigars
cilia
cinder
cinders
cinema
cinematic
cinnamon
cipher
ciphers
ciphertext
ciphertexts
circa
circle
circled
circles
circlet
circling
circuit
circuitous
circuitously
circuitry
circuits
circulant
circular
circularity
circularly
circulate
circulated
circulates
circulating
circulation
circumcise
circumcision
circumference
circumflex
circumlocution
circumlocutions
circumnavigate
circumnavigated
circumnavigates
circumpolar
circumscribe
circumscribed
circumscribing
circumscription
circumspect
circumspection
circumspectly
circumstance
circumstanced
circumstances
circumstantial
circumstantially
circumvent
circumventable
circumvented
circumventing
circumvents
circus
circuses
cistern
cisterns
citadel
citadels
citation
citations
cite
cited
cites
cities
citing
citizen
citizens
citizenship
citrus
city
cityscape
citywide
civet
civic
civics
civil
civilian
civilians
civility
civilization
civilizations
civilize
civilized
civilizes
civilizing
civilly
clad
cladding
claim
claimable
claimant
claimants
claimed
claiming
claims
clairvoyant
clairvoyantly
clam
clamber
clambered
clambering
clambers
clamor
clamored
clamoring
clamorous
clamors
clamp
clamped
clamping
clamps
clams
clan
clandestine
clang
clanged
clanging
clangs
clank
clannish
clap
clapboard
clapping
claps
clarification
clarifications
clarified
clarifies
clarify
clarifying
clarinet
clarity
clash
clashed
clashes
clashing
clasp
clasped
clasping
clasps
class
classed
classes
classic
classical
classically
classics
classifiable
classification
classifications
classified
classifier
classifiers
classifies
classify
classifying
classmate
classmates
classroom
classrooms
classy
clatter
clattered
clattering
clause
clauses
claustrophobia
claustrophobic
claw
clawed
clawing
claws
clay
clays
clean
cleaned
cleaner
cleaners
cleanest
cleaning
cleanliness
cleanly
cleanness
cleans
cleanse
cleansed
cleanser
cleansers
cleanses
cleansing
cleanup
clear
clearance
clearances
cleared
clearer
clearest
clearing
clearings
clearly
clearness
clears
cleavage
cleave
cleaved
cleaver
cleavers
cleaves
cleaving
cleft
clefts
clemency
clement
clench
clenched
clenches
clergy
clergyman
clergymen
clerical
clerk
clerked
clerking
clerks
clever
cleverer
cleverest
cleverly
cleverness
cliche
cliches
click
clicked
clicking
clicks
client
clientele
clients
cliff
cliffs
climate
climates
climatic
climatically
climatology
climax
climaxed
climaxes
climb
climbed
climber
climbers
climbing
climbs
clime
climes
clinch
clinched
clincher
clinches
cling
clinging
clings
clinic
clinical
clinically
clinician
clinics
clink
clinked
clinker
clip
clipboard
clipped
clipper
clippers
clipping
clippings
clips
clique
cliques
clitoris
cloak
cloakroom
cloaks
clobber
clobbered
clobbering
clobbers
clock
clocked
clocker
clockers
clocking
clockings
clocks
clockwatcher
clockwise
clockwork
clod
clods
clog
clogged
clogging
clogs
cloister
cloisters
clone
cloned
clones
cloning
close
closed
closely
closeness
closenesses
closer
closers
closes
closest
closet
closeted
closets
closeup
closing
closure
closures
clot
cloth
clothe
clothed
clothes
clotheshorse
clothesline
clothing
clotting
cloture
cloud
cloudburst
clouded
cloudier
cloudiest
cloudiness
clouding
cloudless
clouds
cloudy
clout
clove
clover
cloves
clown
clowning
clowns
club
clubbed
clubbing
clubhouse
clubroom
clubs
cluck
clucked
clucking
clucks
clue
clues
clump
clumped
clumping
clumps
clumsily
clumsiness
clumsy
clung
cluster
clustered
clustering
clusterings
clusters
clutch
clutched
clutches
clutching
clutter
cluttered
cluttering
clutters
coach
coached
coacher
coaches
coaching
coachman
coachmen
coagulate
coal
coalesce
coalesced
coalesces
coalescing
coalition
coals
coarse
coarsely
coarsen
coarsened
coarseness
coarser
coarsest
coast
coastal
coasted
coaster
coasters
coasting
coastline
coasts
coat
coated
coating
coatings
coats
coattail
coauthor
coax
coaxed
coaxer
coaxes
coaxial
coaxing
cobalt
cobble
cobbler
cobblers
cobblestone
cobra
cobweb
cobwebs
coca
cocaine
cock
cocked
cocking
cockpit
cockroach
cocks
cocktail
cocktails
cocky
coco
cocoa
coconut
coconuts
cocoon
cocoons
cod
coddle
code
coded
codeine
coder
coders
codes
codeword
codewords
codfish
codicil
codification
codifications
codified
codifier
codifiers
codifies
codify
codifying
coding
codings
codpiece
coed
coeditor
coeducation
coefficient
coefficients
coequal
coerce
coerced
coerces
coercible
coercing
coercion
coercive
coexist
coexisted
coexistence
coexisting
coexists
cofactor
coffee
coffeecup
coffeepot
coffees
coffer
coffers
coffin
coffins
cog
cogent
cogently
cogitate
cogitated
cogitates
cogitating
cogitation
cognac
cognition
cognitive
cognitively
cognizance
cognizant
cogs
cohabitation
cohabitations
cohere
cohered
coherence
coherent
coherently
coheres
cohering
cohesion
cohesive
cohesively
cohesiveness
cohort
coil
coiled
coiling
coils
coin
coinage
coincide
coincided
coincidence
coincidences
coincident
coincidental
coincides
coinciding
coined
coiner
coining
coins
coke
cokes
colander
cold
colder
coldest
coldly
coldness
colds
colicky
coliform
coliseum
collaborate
collaborated
collaborates
collaborating
collaboration
collaborations
collaborative
collaborator
collaborators
collagen
collapse
collapsed
collapses
collapsible
collapsing
collar
collarbone
collared
collaring
collars
collate
collateral
colleague
colleagues
collect
collected
collectible
collecting
collection
collections
collective
collectively
collectives
collector
collectors
collects
college
colleges
collegian
collegiate
collide
collided
collides
colliding
collie
collies
collision
collisions
colloidal
colloquia
colloquial
colloquium
colloquy
collusion
colon
colonel
colonels
colonial
colonially
colonials
colonies
colonist
colonists
colonization
colonize
colonized
colonizer
colonizers
colonizes
colonizing
colons
colony
color
colored
colorer
colorers
colorful
coloring
colorings
colorless
colors
colossal
colt
colts
column
columnize
columnized
columnizes
columnizing
columns
comb
combat
combatant
combatants
combated
combating
combative
combats
combed
comber
combers
combination
combinational
combinations
combinator
combinatorial
combinatorially
combinatoric
combinatorics
combinators
combine
combined
combines
combing
combings
combining
combs
combustible
combustion
come
comeback
comedian
comedians
comedic
comedies
comedy
comeliness
comely
comer
comers
comes
comestible
comet
cometary
comets
comfort
comfortabilities
comfortability
comfortable
comfortably
comforted
comforter
comforters
comforting
comfortingly
comforts
comic
comical
comically
comics
coming
comings
comma
command
commandant
commandants
commanded
commandeer
commander
commanders
commanding
commandingly
commandment
commandments
commando
commands
commas
commemorate
commemorated
commemorates
commemorating
commemoration
commemorative
commence
commenced
commencement
commencements
commences
commencing
commend
commendation
commendations
commended
commending
commends
commensurate
comment
commentaries
commentary
commentator
commentators
commented
commenting
comments
commerce
commercial
commercially
commercialness
commercials
commission
commissioned
commissioner
commissioners
commissioning
commissions
commit
commitment
commitments
commits
committed
committee
committeeman
committeemen
committees
committeewoman
committeewomen
committing
commodities
commodity
commodore
commodores
common
commonalities
commonality
commoner
commoners
commonest
commonly
commonness
commonplace
commonplaces
commons
commonwealth
commonwealths
commotion
communal
communally
commune
communes
communicant
communicants
communicate
communicated
communicates
communicating
communication
communications
communicative
communicator
communicators
communion
communist
communists
communities
community
commutative
commutativity
commute
commuted
commuter
commuters
commutes
commuting
compact
compacted
compacter
compactest
compacting
compaction
compactly
compactness
compactor
compactors
compacts
companies
companion
companionable
companions
companionship
company
comparability
comparable
comparably
comparative
comparatively
comparatives
comparator
comparators
compare
compared
compares
comparing
comparison
comparisons
compartment
compartmentalize
compartmentalized
compartmentalizes
compartmentalizing
compartmented
compartments
compass
compassion
compassionate
compassionately
compatibilities
compatibility
compatible
compatibles
compatibly
compel
compelled
compelling
compellingly
compels
compendium
compensate
compensated
compensates
compensating
compensation
compensations
compensatory
compete
competed
competence
competency
competent
competently
competes
competing
competition
competitions
competitive
competitively
competitor
competitors
compilation
compilations
compile
compiled
compiler
compilers
compiles
compiling
complacency
complain
complained
complainer
complainers
complaining
complains
complaint
complaints
complement
complementary
complemented
complementer
complementers
complementing
complements
complete
completed
completely
completeness
completes
completing
completion
completions
complex
complexes
complexion
complexities
complexity
complexly
compliance
compliant
complicate
complicated
complicates
complicating
complication
complications
complicator
complicators
complicity
complied
compliment
complimentary
complimented
complimenter
complimenters
complimenting
compliments
comply
complying
component
componentry
components
componentwise
compose
composed
composedly
composer
composers
composes
composing
composite
composites
composition
compositional
compositions
compost
composure
compound
compounded
compounding
compounds
comprehend
comprehended
comprehending
comprehends
comprehensibility
comprehensible
comprehension
comprehensive
comprehensively
compress
compressed
compresses
compressible
compressing
compression
compressive
compressor
comprise
comprised
comprises
comprising
compromise
compromised
compromiser
compromisers
compromises
compromising
compromisingly
comptroller
comptrollers
compulsion
compulsions
compulsive
compulsory
compunction
computability
computable
computation
computational
computationally
computations
compute
computed
computer
computerize
computerized
computerizes
computerizing
computers
computes
computing
comrade
comradely
comrades
comradeship
con
concatenate
concatenated
concatenates
concatenating
concatenation
concatenations
concave
conceal
concealed
concealer
concealers
concealing
concealment
conceals
concede
conceded
concedes
conceding
conceit
conceited
conceits
conceivable
conceivably
conceive
conceived
conceives
conceiving
concentrate
concentrated
concentrates
concentrating
concentration
concentrations
concentrator
concentrators
concentric
concept
conception
conceptions
concepts
conceptual
conceptualization
conceptualizations
conceptualize
conceptualized
conceptualizes
conceptualizing
conceptually
concern
concerned
concernedly
concerning
concerns
concert
concerted
concertmaster
concerto
concerts
concession
concessions
conciliate
conciliatory
concise
concisely
conciseness
conclave
conclude
concluded
concludes
concluding
conclusion
conclusions
conclusive
conclusively
concoct
concomitant
concord
concordant
concourse
concrete
concretely
concreteness
concretes
concretion
concubine
concur
concurred
concurrence
concurrencies
concurrency
concurrent
concurrently
concurring
concurs
concussion
condemn
condemnation
condemnations
condemned
condemner
condemners
condemning
condemns
condensation
condense
condensed
condenser
condenses
condensing
condescend
condescending
condition
conditional
conditionally
conditionals
conditioned
conditioner
conditioners
conditioning
conditions
condom
condone
condoned
condones
condoning
conduce
conducive
conduciveness
conduct
conductance
conducted
conducting
conduction
conductive
conductivity
conductor
conductors
conducts
conduit
cone
cones
confectionery
confederacy
confederate
confederates
confederation
confederations
confer
conferee
conference
conferences
conferred
conferrer
conferrers
conferring
confers
confess
confessed
confesses
confessing
confession
confessions
confessor
confessors
confidant
confidants
confide
confided
confidence
confidences
confident
confidential
confidentiality
confidentially
confidently
confides
confiding
confidingly
configurable
configuration
configurations
configure
configured
configures
configuring
confine
confined
confinement
confinements
confiner
confines
confining
confirm
confirmation
confirmations
confirmatory
confirmed
confirming
confirms
confiscate
confiscated
confiscates
confiscating
confiscation
confiscations
conflagration
conflict
conflicted
conflicting
conflicts
confluent
confocal
conform
conformal
conformance
conformed
conforming
conformity
conforms
confound
confounded
confounding
confounds
confront
confrontation
confrontations
confronted
confronter
confronters
confronting
confronts
confuse
confused
confuser
confusers
confuses
confusing
confusingly
confusion
confusions
congenial
congenially
congenital
congest
congested
congestion
congestive
conglomerate
congratulate
congratulated
congratulation
congratulations
congratulatory
congregate
congregated
congregates
congregating
congregation
congregations
congress
congresses
congressional
congressionally
congressman
congressmen
congresswoman
congresswomen
congruence
congruent
conic
conifer
coniferous
conjecture
conjectured
conjectures
conjecturing
conjoined
conjugal
conjugate
conjunct
conjuncted
conjunction
conjunctions
conjunctive
conjunctively
conjuncts
conjuncture
conjure
conjured
conjurer
conjures
conjuring
connect
connected
connectedness
connecting
connection
connectionless
connections
connective
connectives
connectivity
connector
connectors
connects
connivance
connive
connoisseur
connoisseurs
connotation
connotative
connote
connoted
connotes
connoting
connubial
conquer
conquerable
conquered
conquerer
conquerers
conquering
conqueror
conquerors
conquers
conquest
conquests
conscience
consciences
conscientious
conscientiously
conscious
consciously
consciousness
conscript
conscription
consecrate
consecration
consecutive
consecutively
consensual
consensus
consent
consented
consenter
consenters
consenting
consents
consequence
consequences
consequent
consequential
consequentialities
consequentiality
consequently
consequents
conservation
conservationist
conservationists
conservations
conservatism
conservative
conservatively
conservatives
conservator
conserve
conserved
conserves
conserving
consider
considerable
considerably
considerate
considerately
consideration
considerations
considered
considering
considers
consign
consigned
consigning
consigns
consist
consisted
consistency
consistent
consistently
consisting
consists
consolable
consolation
consolations
console
consoled
consoler
consolers
consoles
consolidate
consolidated
consolidates
consolidating
consolidation
consoling
consolingly
consonant
consonants
consort
consorted
consorting
consortium
consorts
conspicuous
conspicuously
conspiracies
conspiracy
conspirator
conspirators
conspire
conspired
conspires
conspiring
constable
constables
constancy
constant
constantly
constants
constellation
constellations
consternation
constituencies
constituency
constituent
constituents
constitute
constituted
constitutes
constituting
constitution
constitutional
constitutionality
constitutionally
constitutions
constitutive
constrain
constrained
constraining
constrains
constraint
constraints
constrict
construct
constructed
constructibility
constructible
constructing
construction
constructions
constructive
constructively
constructor
constructors
constructs
construe
construed
construing
consul
consular
consulate
consulates
consuls
consult
consultant
consultants
consultation
consultations
consultative
consulted
consulting
consults
consumable
consume
consumed
consumer
consumers
consumes
consuming
consummate
consummated
consummately
consummation
consumption
consumptions
consumptive
consumptively
contact
contacted
contacting
contacts
contagion
contagious
contagiously
contain
containable
contained
container
containers
containing
containment
containments
contains
contaminate
contaminated
contaminates
contaminating
contamination
contemplate
contemplated
contemplates
contemplating
contemplation
contemplations
contemplative
contemporaries
contemporariness
contemporary
contempt
contemptible
contemptuous
contemptuously
contend
contended
contender
contenders
contending
contends
content
contented
contenting
contention
contentions
contently
contentment
contents
contest
contestable
contestant
contested
contester
contesters
contesting
contests
context
contexts
contextual
contextually
contiguity
contiguous
contiguously
continent
continental
continentally
continents
contingencies
contingency
contingent
contingents
continual
continually
continuance
continuances
continuation
continuations
continue
continued
continues
continuing
continuities
continuity
continuous
continuously
continuum
contortions
contour
contoured
contouring
contours
contraband
contraception
contraceptive
contract
contracted
contracting
contraction
contractions
contractor
contractors
contracts
contractual
contractually
contradict
contradicted
contradicting
contradiction
contradictions
contradictory
contradicts
contradistinction
contradistinctions
contrapositive
contrapositives
contraption
contraptions
contrariness
contrary
contrast
contrasted
contraster
contrasters
contrasting
contrastingly
contrasts
contribute
contributed
contributes
contributing
contribution
contributions
contributor
contributorily
contributors
contributory
contrite
contrition
contrivance
contrivances
contrive
contrived
contriver
contrives
contriving
control
controllability
controllable
controllably
controlled
controller
controllers
controlling
controls
controversial
controversies
controversy
controvertible
contumacious
contumacy
conundrum
conundrums
convalescent
convect
convene
convened
convenes
convenience
conveniences
convenient
conveniently
convening
convent
convention
conventional
conventionally
conventions
convents
converge
converged
convergence
convergent
converges
converging
conversant
conversantly
conversation
conversational
conversationally
conversations
converse
conversed
conversely
converses
conversing
conversion
conversions
convert
converted
converter
converters
convertibility
convertible
converting
converts
convex
convey
conveyance
conveyances
conveyed
conveyer
conveyers
conveying
conveyor
conveys
convict
convicted
convicting
conviction
convictions
convicts
convince
convinced
convincer
convincers
convinces
convincing
convincingly
convivial
convoke
convoluted
convolution
convoy
convoyed
convoying
convoys
convulse
convulsion
convulsions
coo
cooing
cook
cookbook
cooked
cookery
cookie
cookies
cooking
cooks
cooky
cool
cooled
cooler
coolers
coolest
coolie
coolies
cooling
coolly
coolness
cools
coon
coons
coop
cooped
cooper
cooperate
cooperated
cooperates
cooperating
cooperation
cooperations
cooperative
cooperatively
cooperatives
cooperator
cooperators
coopers
coops
coordinate
coordinated
coordinates
coordinating
coordination
coordinations
coordinator
coordinators
cop
cope
coped
copes
copied
copier
copiers
copies
coping
copings
copious
copiously
copiousness
coplanar
copper
copperhead
coppers
copra
coprocessor
cops
copse
copy
copying
copyright
copyrightable
copyrighted
copyrights
copywriter
coquette
coral
cord
corded
corder
cordial
cordiality
cordially
cords
core
cored
corer
corers
cores
coriander
coring
cork
corked
corker
corkers
corking
corks
corkscrew
cormorant
corn
cornea
corner
cornered
corners
cornerstone
cornerstones
cornet
cornfield
cornfields
corning
cornmeal
corns
cornstarch
cornucopia
corny
corollaries
corollary
coronaries
coronary
coronation
coroner
coronet
coronets
coroutine
coroutines
corporal
corporals
corporate
corporately
corporation
corporations
corps
corpse
corpses
corpulent
corpus
corpuscular
corral
correct
correctable
corrected
correcting
correction
corrections
corrective
correctively
correctives
correctly
correctness
corrector
corrects
correlate
correlated
correlates
correlating
correlation
correlations
correlative
correspond
corresponded
correspondence
correspondences
correspondent
correspondents
corresponding
correspondingly
corresponds
corridor
corridors
corrigenda
corrigendum
corrigible
corroborate
corroborated
corroborates
corroborating
corroboration
corroborations
corroborative
corrode
corrosion
corrosive
corrugate
corrupt
corrupted
corrupter
corruptible
corrupting
corruption
corruptions
corrupts
corset
cortex
cortical
cosine
cosines
cosmetic
cosmetics
cosmic
cosmology
cosmopolitan
cosmos
cosponsor
cost
costed
costing
costly
costs
costume
costumed
costumer
costumes
costuming
cosy
cot
cotangent
cotillion
cots
cottage
cottager
cottages
cotton
cottonmouth
cottons
cottonseed
cottonwood
cotyledon
cotyledons
couch
couched
couches
couching
cougar
cough
coughed
coughing
coughs
could
coulomb
council
councillor
councillors
councilman
councilmen
councils
councilwoman
councilwomen
counsel
counseled
counseling
counselled
counselling
counsellor
counsellors
counselor
counselors
counsels
count
countable
countably
counted
countenance
counter
counteract
counteracted
counteracting
counteractive
counterargument
counterattack
counterbalance
counterclockwise
countered
counterexample
counterexamples
counterfeit
counterfeited
counterfeiter
counterfeiting
counterflow
countering
counterintuitive
counterman
countermeasure
countermeasures
countermen
counterpart
counterparts
counterpoint
counterpointing
counterpoise
counterproductive
counterproposal
counterrevolution
counters
countersink
countersunk
countess
counties
counting
countless
countries
country
countryman
countrymen
countryside
countrywide
counts
county
countywide
couple
coupled
coupler
couplers
couples
coupling
couplings
coupon
coupons
courage
courageous
courageously
courier
couriers
course
coursed
courser
courses
coursing
court
courted
courteous
courteously
courter
courters
courtesan
courtesies
courtesy
courthouse
courthouses
courtier
courtiers
courting
courtly
courtroom
courtrooms
courts
courtship
courtyard
courtyards
cousin
cousins
covalent
covariant
cove
covenant
covenants
cover
coverable
coverage
covered
covering
coverings
coverlet
coverlets
covers
covert
covertly
coves
covet
coveted
coveting
covetous
covetousness
covets
cow
coward
cowardice
cowardly
cowboy
cowboys
cowed
cower
cowered
cowerer
cowerers
cowering
coweringly
cowers
cowherd
cowhide
cowing
cowl
cowlick
cowling
cowls
coworker
cows
cowslip
cowslips
coyote
coyotes
coypu
cozier
coziness
cozy
crab
crabapple
crabs
crack
cracked
cracker
crackers
cracking
crackle
crackled
crackles
crackling
crackpot
cracks
cradle
cradled
cradles
craft
crafted
crafter
craftiness
crafting
crafts
craftsman
craftsmen
craftspeople
craftsperson
crafty
crag
craggy
crags
cram
cramming
cramp
cramps
crams
cranberries
cranberry
crane
cranes
crania
cranium
crank
crankcase
cranked
crankier
crankiest
crankily
cranking
cranks
crankshaft
cranky
cranny
crash
crashed
crasher
crashers
crashes
crashing
crass
crate
crater
craters
crates
cravat
cravats
crave
craved
craven
craves
craving
crawl
crawled
crawler
crawlers
crawling
crawls
crayon
craze
crazed
crazes
crazier
craziest
crazily
craziness
crazing
crazy
creak
creaked
creaking
creaks
creaky
cream
creamed
creamer
creamers
creamery
creaming
creams
creamy
crease
creased
creases
creasing
create
created
creates
creating
creation
creations
creative
creatively
creativeness
creativity
creator
creators
creature
creatures
credence
credential
credibility
credible
credibly
credit
creditable
creditably
credited
crediting
creditor
creditors
credits
credulity
credulous
credulousness
creed
creeds
creek
creeks
creep
creeper
creepers
creeping
creeps
creepy
cremate
cremated
cremates
cremating
cremation
cremations
crematory
crepe
crept
crescent
crescents
crest
crested
crestfallen
crests
cretin
crevice
crevices
crew
crewcut
crewed
crewing
crews
crib
cribs
cricket
crickets
cried
crier
criers
cries
crime
crimes
criminal
criminally
criminals
criminate
crimson
crimsoning
cringe
cringed
cringes
cringing
cripple
crippled
cripples
crippling
crises
crisis
crisp
crisply
crispness
crisscross
criteria
criterion
critic
critical
critically
criticism
criticisms
criticize
criticized
criticizes
criticizing
critics
critique
critiques
critiquing
critter
croak
croaked
croaking
croaks
crochet
crochets
crock
crockery
crocks
crocodile
crocus
croft
crook
crooked
crooks
crop
cropped
cropper
croppers
cropping
crops
cross
crossable
crossbar
crossbars
crossed
crosser
crossers
crosses
crossing
crossings
crossly
crossover
crossovers
crosspoint
crossroad
crosstalk
crosswalk
crossword
crosswords
crotch
crotchety
crouch
crouched
crouching
crow
crowd
crowded
crowder
crowding
crowds
crowed
crowing
crown
crowned
crowning
crowns
crows
crucial
crucially
crucible
crucified
crucifies
crucifix
crucifixion
crucify
crucifying
crud
cruddy
crude
crudely
crudeness
cruder
crudest
cruel
crueler
cruelest
cruelly
cruelty
cruise
cruiser
cruisers
cruises
cruising
crumb
crumble
crumbled
crumbles
crumbling
crumbly
crumbs
crummy
crumple
crumpled
crumples
crumpling
crunch
crunched
crunches
crunchier
crunchiest
crunching
crunchy
crusade
crusader
crusaders
crusades
crusading
crush
crushable
crushed
crusher
crushers
crushes
crushing
crushingly
crust
crustacean
crustaceans
crusts
crutch
crutches
crux
cruxes
cry
crying
cryogenic
crypt
cryptanalysis
cryptanalyst
cryptanalytic
cryptic
cryptogram
cryptographer
cryptographic
cryptographically
cryptography
cryptologist
cryptology
crystal
crystalline
crystallize
crystallized
crystallizes
crystallizing
crystals
cub
cubbyhole
cube
cubed
cubes
cubic
cubs
cuckoo
cuckoos
cucumber
cucumbers
cuddle
cuddled
cuddly
cudgel
cudgels
cue
cued
cues
cuff
cufflink
cuffs
cuisine
culinary
cull
culled
culler
culling
culls
culminate
culminated
culminates
culminating
culmination
culpa
culpable
culprit
culprits
cult
cultivable
cultivate
cultivated
cultivates
cultivating
cultivation
cultivations
cultivator
cultivators
cults
cultural
culturally
culture
cultured
cultures
culturing
cumbersome
cumulative
cumulatively
cunnilingus
cunning
cunningly
cup
cupboard
cupboards
cupful
cupped
cupping
cups
curable
curably
curb
curbing
curbs
curd
curdle
cure
cured
cures
curfew
curfews
curing
curiosities
curiosity
curious
curiouser
curiousest
curiously
curl
curled
curler
curlers
curlicue
curling
curls
curly
currant
currants
currencies
currency
current
currently
currentness
currents
curricular
curriculum
curriculums
curried
curries
curry
currying
curs
curse
cursed
curses
cursing
cursive
cursor
cursorily
cursors
cursory
curt
curtail
curtailed
curtails
curtain
curtained
curtains
curtate
curtly
curtness
curtsies
curtsy
curvaceous
curvature
curve
curved
curves
curvilinear
curving
cushion
cushioned
cushioning
cushions
cusp
cusps
custard
custodial
custodian
custodians
custody
custom
customarily
customary
customer
customers
customizable
customization
customizations
customize
customized
customizer
customizers
customizes
customizing
customs
cut
cutaneous
cutback
cute
cutest
cutlass
cutlet
cutoff
cutout
cutover
cuts
cutter
cutters
cutthroat
cutting
cuttingly
cuttings
cuttlefish
cyanide
cybernetic
cybernetics
cyberspace
cycle
cycled
cycles
cyclic
cyclically
cycling
cycloid
cycloidal
cycloids
cyclone
cyclones
cyclotron
cyclotrons
cylinder
cylinders
cylindrical
cymbal
cymbals
cynic
cynical
cynically
cypress
cyst
cysts
cytology
cytoplasm
czar
dabble
dabbled
dabbler
dabbles
dabbling
dactyl
dactylic
dad
daddy
dads
daemon
daemons
daffodil
daffodils
dagger
dahlia
dailies
daily
daintily
daintiness
dainty
dairy
daisies
daisy
dale
dales
dam
damage
damaged
damager
damagers
damages
damaging
damask
dame
damming
damn
damnation
damned
damning
damns
damp
dampen
dampens
damper
damping
dampness
dams
damsel
damsels
dance
danced
dancer
dancers
dances
dancing
dandelion
dandelions
dandy
danger
dangerous
dangerously
dangers
dangle
dangled
dangles
dangling
dare
dared
darer
darers
dares
daresay
daring
daringly
dark
darken
darker
darkest
darkly
darkness
darkroom
darling
darlings
darn
darned
darner
darning
darns
dart
darted
darter
darting
darts
dash
dashboard
dashed
dasher
dashers
dashes
dashing
dashingly
data
database
databases
datagram
datagrams
date
dated
dateline
dater
dates
dating
dative
datum
daughter
daughterly
daughters
daunt
daunted
dauntless
dawn
dawned
dawning
dawns
day
daybreak
daydream
daydreaming
daydreams
daylight
daylights
days
daytime
daze
dazed
dazzle
dazzled
dazzler
dazzles
dazzling
dazzlingly
deacon
deacons
deactivate
dead
deaden
deadline
deadlines
deadlock
deadlocked
deadlocking
deadlocks
deadly
deadness
deadwood
deaf
deafen
deafer
deafest
deafness
deal
dealer
dealers
dealership
dealing
dealings
deallocate
deallocated
deallocating
deallocation
deallocations
deals
dealt
dean
deans
dear
dearer
dearest
dearly
dearness
dearth
dearths
death
deathbed
deathly
deaths
debacle
debar
debase
debatable
debate
debated
debater
debaters
debates
debating
debauch
debauchery
debilitate
debilitated
debilitates
debilitating
debility
debit
debited
debrief
debris
debt
debtor
debts
debug
debugged
debugger
debuggers
debugging
debugs
debunk
debutante
decade
decadence
decadent
decadently
decades
decal
decathlon
decay
decayed
decaying
decays
decease
deceased
deceases
deceasing
decedent
deceit
deceitful
deceitfully
deceitfulness
deceive
deceived
deceiver
deceivers
deceives
deceiving
decelerate
decelerated
decelerates
decelerating
deceleration
decencies
decency
decennial
decent
decently
decentralization
decentralized
deception
deceptions
deceptive
deceptively
decertify
decibel
decidability
decidable
decide
decided
decidedly
decides
deciding
deciduous
decimal
decimals
decimate
decimated
decimates
decimating
decimation
decipher
deciphered
decipherer
deciphering
deciphers
decision
decisions
decisive
decisively
decisiveness
deck
decked
decking
deckings
decks
declaration
declarations
declarative
declaratively
declaratives
declarator
declaratory
declare
declared
declarer
declarers
declares
declaring
declassify
declination
declinations
decline
declined
decliner
decliners
declines
declining
decode
decoded
decoder
decoders
decodes
decoding
decodings
decolletage
decollimate
decompile
decomposability
decomposable
decompose
decomposed
decomposes
decomposing
decomposition
decompositions
decompress
decompression
decorate
decorated
decorates
decorating
decoration
decorations
decorative
decorum
decouple
decoupled
decouples
decoupling
decoy
decoys
decrease
decreased
decreases
decreasing
decreasingly
decree
decreed
decreeing
decrees
decrement
decremented
decrementing
decrements
decrypt
decrypted
decrypting
decryption
decrypts
dedicate
dedicated
dedicates
dedicating
dedication
deduce
deduced
deducer
deduces
deducible
deducing
deduct
deducted
deductible
deducting
deduction
deductions
deductive
deed
deeded
deeding
deeds
deem
deemed
deeming
deemphasize
deemphasized
deemphasizes
deemphasizing
deems
deep
deepen
deepened
deepening
deepens
deeper
deepest
deeply
deeps
deer
deface
default
defaulted
defaulter
defaulting
defaults
defeat
defeated
defeating
defeats
defecate
defect
defected
defecting
defection
defections
defective
defects
defend
defendant
defendants
defended
defender
defenders
defending
defends
defenestrate
defenestrated
defenestrates
defenestrating
defenestration
defense
defenseless
defenses
defensible
defensive
defer
deference
deferment
deferments
deferrable
deferred
deferrer
deferrers
deferring
defers
defiance
defiant
defiantly
deficiencies
deficiency
deficient
deficit
deficits
defied
defies
defile
defiling
definable
define
defined
definer
defines
defining
definite
definitely
definiteness
definition
definitional
definitions
definitive
deflate
deflater
deflect
defocus
deforest
deforestation
deform
deformation
deformations
deformed
deformities
deformity
defraud
defray
defrost
deftly
defunct
defy
defying
degeneracy
degenerate
degenerated
degenerates
degenerating
degeneration
degenerative
degradable
degradation
degradations
degrade
degraded
degrades
degrading
degree
degrees
dehumidify
dehydrate
deify
deign
deigned
deigning
deigns
deities
deity
dejected
dejectedly
delay
delayed
delaying
delays
delegate
delegated
delegates
delegating
delegation
delegations
delete
deleted
deleter
deleterious
deletes
deleting
deletion
deletions
deliberate
deliberated
deliberately
deliberateness
deliberates
deliberating
deliberation
deliberations
deliberative
deliberator
deliberators
delicacies
delicacy
delicate
delicately
delicatessen
delicious
deliciously
delight
delighted
delightedly
delightful
delightfully
delighting
delights
delimit
delimitation
delimited
delimiter
delimiters
delimiting
delimits
delineament
delineate
delineated
delineates
delineating
delineation
delinquency
delinquent
delirious
deliriously
delirium
deliver
deliverable
deliverables
deliverance
delivered
deliverer
deliverers
deliveries
delivering
delivers
delivery
dell
dells
delta
deltas
delude
deluded
deludes
deluding
deluge
deluged
deluges
delusion
delusions
deluxe
delve
delves
delving
demagnify
demagogue
demand
demanded
demander
demanding
demandingly
demands
demarcate
demeanor
demented
demerit
demigod
demise
demo
democracies
democracy
democrat
democratic
democratically
democrats
demodulate
demodulator
demographic
demolish
demolished
demolishes
demolition
demon
demoniac
demonic
demons
demonstrable
demonstrate
demonstrated
demonstrates
demonstrating
demonstration
demonstrations
demonstrative
demonstratively
demonstrator
demonstrators
demoralize
demoralized
demoralizes
demoralizing
demote
demountable
demultiplex
demultiplexed
demultiplexer
demultiplexers
demultiplexing
demur
demythologize
den
denature
deniable
denial
denials
denied
denier
denies
denigrate
denigrated
denigrates
denigrating
denizen
denominate
denomination
denominations
denominator
denominators
denotable
denotation
denotational
denotationally
denotations
denotative
denote
denoted
denotes
denoting
denounce
denounced
denounces
denouncing
dens
dense
densely
denseness
denser
densest
densities
density
dent
dental
dentally
dented
denting
dentist
dentistry
dentists
dents
denture
denude
denumerable
denunciate
denunciation
deny
denying
deodorant
deoxyribonucleic
depart
departed
departing
department
departmental
departments
departs
departure
departures
depend
dependability
dependable
dependably
depended
dependence
dependencies
dependency
dependent
dependently
dependents
depending
depends
depict
depicted
depicting
depicts
deplete
depleted
depletes
depleting
depletion
depletions
deplorable
deplore
deplored
deplores
deploring
deploy
deployed
deploying
deployment
deployments
deploys
deport
deportation
deportee
deportment
depose
deposed
deposes
deposit
depositary
deposited
depositing
deposition
depositions
depositor
depositors
depository
deposits
depot
depots
deprave
depraved
depravity
deprecate
depreciate
depreciated
depreciates
depreciation
depress
depressed
depresses
depressing
depression
depressions
deprivation
deprivations
deprive
deprived
deprives
depriving
depth
depths
deputies
deputy
dequeue
dequeued
dequeues
dequeuing
derail
derailed
derailing
derails
derby
dereference
deregulate
deregulated
deride
derision
derivable
derivation
derivations
derivative
derivatives
derive
derived
derives
deriving
derogatory
derrick
derriere
dervish
descend
descendant
descendants
descended
descendent
descender
descenders
descending
descends
descent
descents
describable
describe
described
describer
describes
describing
description
descriptions
descriptive
descriptively
descriptives
descriptor
descriptors
descry
desecrate
desegregate
desert
deserted
deserter
deserters
deserting
desertion
desertions
deserts
deserve
deserved
deserves
deserving
deservingly
deservings
desiderata
desideratum
design
designate
designated
designates
designating
designation
designations
designator
designators
designed
designer
designers
designing
designs
desirability
desirable
desirably
desire
desired
desires
desiring
desirous
desist
desk
desks
desktop
desolate
desolately
desolation
desolations
despair
despaired
despairing
despairingly
despairs
despatch
despatched
desperado
desperate
desperately
desperation
despicable
despise
despised
despises
despising
despite
despoil
despondent
despot
despotic
despotism
despots
dessert
desserts
desiccate
destabilize
destination
destinations
destine
destined
destinies
destiny
destitute
destitution
destroy
destroyed
destroyer
destroyers
destroying
destroys
destruct
destruction
destructions
destructive
destructively
destructiveness
destructor
destuff
destuffing
destuffs
desuetude
desultory
desynchronize
detach
detached
detacher
detaches
detaching
detachment
detachments
detail
detailed
detailing
details
detain
detained
detaining
detains
detect
detectable
detectably
detected
detecting
detection
detections
detective
detectives
detector
detectors
detects
detente
detention
deter
detergent
deteriorate
deteriorated
deteriorates
deteriorating
deterioration
determinable
determinacy
determinant
determinants
determinate
determinately
determination
determinations
determinative
determine
determined
determiner
determiners
determines
determining
determinism
deterministic
deterministically
deterred
deterrent
deterring
detest
detestable
detested
detour
detract
detractor
detractors
detracts
detriment
detrimental
deuce
deus
deuterium
devastate
devastated
devastates
devastating
devastation
develop
developed
developer
developers
developing
development
developmental
developments
develops
deviant
deviants
deviate
deviated
deviates
deviating
deviation
deviations
device
devices
devil
devilish
devilishly
devils
devious
devise
devised
devises
devising
devisings
devoid
devolve
devote
devoted
devotedly
devotee
devotees
devotes
devoting
devotion
devotions
devour
devoured
devourer
devours
devout
devoutly
devoutness
dew
dewdrop
dewdrops
dewy
dexterity
diabetes
diabetic
diabolic
diachronic
diacritical
diadem
diagnosable
diagnose
diagnosed
diagnoses
diagnosing
diagnosis
diagnostic
diagnostician
diagnostics
diagonal
diagonally
diagonals
diagram
diagrammable
diagrammatic
diagrammatically
diagrammed
diagrammer
diagrammers
diagramming
diagrams
dial
dialect
dialectic
dialects
dialed
dialer
dialers
dialing
dialog
dialogs
dialogue
dialogues
dials
dialup
dialysis
diamagnetic
diameter
diameters
diametric
diametrically
diamond
diamonds
diaper
diapers
diaphragm
diaphragms
diaries
diarrhea
diary
diatribe
diatribes
dibble
dice
dichotomize
dichotomy
dickens
dicky
dictate
dictated
dictates
dictating
dictation
dictations
dictator
dictatorial
dictators
dictatorship
diction
dictionaries
dictionary
dictum
dictums
did
didactic
diddle
die
died
diehard
dielectric
dielectrics
diem
dies
diesel
diet
dietary
dieter
dieters
dietetic
dietician
dietitian
dietitians
diets
differ
differed
difference
differences
different
differentiable
differential
differentials
differentiate
differentiated
differentiates
differentiating
differentiation
differentiations
differentiators
differently
differer
differers
differing
differs
difficult
difficulties
difficultly
difficulty
diffract
diffuse
diffused
diffusely
diffuser
diffusers
diffuses
diffusible
diffusing
diffusion
diffusions
diffusive
dig
digest
digested
digestible
digesting
digestion
digestive
digests
digger
diggers
digging
diggings
digit
digital
digitalis
digitally
digitization
digitize
digitized
digitizes
digitizing
digits
dignified
dignify
dignitary
dignities
dignity
digram
digress
digressed
digresses
digressing
digression
digressions
digressive
digs
dihedral
dike
dikes
dilapidate
dilatation
dilate
dilated
dilates
dilating
dilation
dildo
dilemma
dilemmas
diligence
diligent
diligently
dill
dilogarithm
dilute
diluted
dilutes
diluting
dilution
dim
dime
dimension
dimensional
dimensionality
dimensionally
dimensioned
dimensioning
dimensions
dimes
diminish
diminished
diminishes
diminishing
diminution
diminutive
dimly
dimmed
dimmer
dimmers
dimmest
dimming
dimness
dimple
dims
din
dine
dined
diner
diners
dines
ding
dinghy
dinginess
dingo
dingy
dining
dinner
dinners
dinnertime
dinnerware
dinosaur
dint
diode
diodes
diopter
diorama
dioxide
dip
diphtheria
diphthong
diploma
diplomacy
diplomas
diplomat
diplomatic
diplomats
dipole
dipped
dipper
dippers
dipping
dippings
dips
dire
direct
directed
directing
direction
directional
directionality
directionally
directions
directive
directives
directly
directness
director
directorate
directories
directors
directory
directrices
directrix
directs
dirge
dirges
dirt
dirtier
dirtiest
dirtily
dirtiness
dirts
dirty
disabilities
disability
disable
disabled
disabler
disablers
disables
disabling
disadvantage
disadvantageous
disadvantages
disaffected
disaffection
disagree
disagreeable
disagreed
disagreeing
disagreement
disagreements
disagrees
disallow
disallowed
disallowing
disallows
disambiguate
disambiguated
disambiguates
disambiguating
disambiguation
disambiguations
disappear
disappearance
disappearances
disappeared
disappearing
disappears
disappoint
disappointed
disappointing
disappointment
disappointments
disapproval
disapprove
disapproved
disapproves
disarm
disarmament
disarmed
disarming
disarms
disassemble
disassembled
disassembles
disassembling
disassembly
disaster
disasters
disastrous
disastrously
disband
disbanded
disbanding
disbands
disburse
disbursed
disbursement
disbursements
disburses
disbursing
disc
discard
discarded
discarding
discards
discern
discerned
discernibility
discernible
discernibly
discerning
discerningly
discernment
discerns
discharge
discharged
discharges
discharging
disciple
disciples
disciplinary
discipline
disciplined
disciplines
disciplining
disclaim
disclaimed
disclaimer
disclaims
disclose
disclosed
discloses
disclosing
disclosure
disclosures
discomfort
disconcert
disconcerting
disconcertingly
disconnect
disconnected
disconnecting
disconnection
disconnects
discontent
discontented
discontinuance
discontinue
discontinued
discontinues
discontinuities
discontinuity
discontinuous
discord
discordant
discount
discounted
discounting
discounts
discourage
discouraged
discouragement
discourages
discouraging
discourse
discourses
discover
discovered
discoverer
discoverers
discoveries
discovering
discovers
discovery
discredit
discredited
discreet
discreetly
discrepancies
discrepancy
discrete
discretely
discreteness
discretion
discretionary
discriminant
discriminate
discriminated
discriminates
discriminating
discrimination
discriminatory
discs
discuss
discussant
discussed
discusses
discussing
discussion
discussions
disdain
disdaining
disdains
disease
diseased
diseases
disembowel
disengage
disengaged
disengages
disengaging
disentangle
disentangling
disfigure
disfigured
disfigures
disfiguring
disgorge
disgrace
disgraced
disgraceful
disgracefully
disgraces
disgruntle
disgruntled
disguise
disguised
disguises
disgust
disgusted
disgustedly
disgustful
disgusting
disgustingly
disgusts
dish
dishearten
disheartening
dished
dishes
dishevel
dishing
dishonest
dishonestly
dishonesty
dishonor
dishonorable
dishonored
dishonoring
dishonors
dishwasher
dishwashers
dishwashing
dishwater
disillusion
disillusioned
disillusioning
disillusionment
disillusionments
disinclined
disingenuous
disinterested
disinterestedness
disjoint
disjointed
disjointly
disjointness
disjunct
disjunction
disjunctions
disjunctive
disjunctively
disjuncts
disk
diskette
diskettes
disks
dislike
disliked
dislikes
disliking
dislocate
dislocated
dislocates
dislocating
dislocation
dislocations
dislodge
dislodged
dismal
dismally
dismay
dismayed
dismaying
dismember
dismembered
dismemberment
dismembers
dismiss
dismissal
dismissals
dismissed
dismisser
dismissers
dismisses
dismissing
dismount
dismounted
dismounting
dismounts
disobedience
disobedient
disobey
disobeyed
disobeying
disobeys
disorder
disordered
disorderly
disorders
disorganized
disown
disowned
disowning
disowns
disparage
disparate
disparities
disparity
dispassionate
dispatch
dispatched
dispatcher
dispatchers
dispatches
dispatching
dispel
dispell
dispelled
dispelling
dispels
dispensary
dispensation
dispense
dispensed
dispenser
dispensers
dispenses
dispensing
dispersal
disperse
dispersed
disperses
dispersing
dispersion
dispersions
displace
displaced
displacement
displacements
displaces
displacing
display
displayable
displayed
displayer
displaying
displays
displease
displeased
displeases
displeasing
displeasure
disposable
disposal
disposals
dispose
disposed
disposer
disposes
disposing
disposition
dispositions
dispossessed
disproportionate
disprove
disproved
disproves
disproving
dispute
disputed
disputer
disputers
disputes
disputing
disqualification
disqualified
disqualifies
disqualify
disqualifying
disquiet
disquieting
disregard
disregarded
disregarding
disregards
disrespectful
disrupt
disrupted
disrupting
disruption
disruptions
disruptive
disrupts
dissatisfaction
dissatisfactions
dissatisfactory
dissatisfied
dissect
dissects
dissemble
disseminate
disseminated
disseminates
disseminating
dissemination
dissension
dissensions
dissent
dissented
dissenter
dissenters
dissenting
dissents
dissertation
dissertations
disservice
dissident
dissidents
dissimilar
dissimilarities
dissimilarity
dissipate
dissipated
dissipates
dissipating
dissipation
dissociate
dissociated
dissociates
dissociating
dissociation
dissolution
dissolutions
dissolve
dissolved
dissolves
dissolving
dissonant
dissuade
distaff
distal
distally
distance
distances
distant
distantly
distaste
distasteful
distastefully
distastes
distemper
distempered
distempers
distill
distillation
distilled
distiller
distillers
distillery
distilling
distills
distinct
distinction
distinctions
distinctive
distinctively
distinctiveness
distinctly
distinctness
distinguish
distinguishable
distinguished
distinguishes
distinguishing
distort
distorted
distorting
distortion
distortions
distorts
distract
distracted
distracting
distraction
distractions
distracts
distraught
distress
distressed
distresses
distressing
distribute
distributed
distributes
distributing
distribution
distributional
distributions
distributive
distributivity
distributor
distributors
district
districts
distrust
distrusted
disturb
disturbance
disturbances
disturbed
disturber
disturbing
disturbingly
disturbs
disuse
ditch
ditches
dither
ditto
ditty
diurnal
divan
divans
dive
dived
diver
diverge
diverged
divergence
divergences
divergent
diverges
diverging
divers
diverse
diversely
diversification
diversified
diversifies
diversify
diversifying
diversion
diversionary
diversions
diversities
diversity
divert
diverted
diverting
diverts
dives
divest
divested
divesting
divestiture
divests
divide
divided
dividend
dividends
divider
dividers
divides
dividing
divine
divinely
diviner
diving
divining
divinities
divinity
divisibility
divisible
division
divisional
divisions
divisive
divisor
divisors
divorce
divorced
divorcee
divulge
divulged
divulges
divulging
dizziness
dizzy
do
docile
dock
docked
docket
docks
dockside
dockyard
doctor
doctoral
doctorate
doctorates
doctored
doctors
doctrinaire
doctrinal
doctrine
doctrines
document
documentaries
documentary
documentation
documentations
documented
documenter
documenters
documenting
documents
dodecahedra
dodecahedral
dodecahedron
dodge
dodged
dodger
dodgers
dodging
doe
doer
doers
does
dog
dogged
doggedly
doggedness
dogging
doghouse
dogma
dogmas
dogmatic
dogmatism
dogs
doing
doings
doldrum
dole
doled
doleful
dolefully
doles
doll
dollar
dollars
dollies
dolls
dolly
dolphin
dolphins
domain
domains
dome
domed
domes
domestic
domestically
domesticate
domesticated
domesticates
domesticating
domestication
domicile
dominance
dominant
dominantly
dominate
dominated
dominates
dominating
domination
domineer
domineering
dominion
domino
don
donate
donated
donates
donating
donation
done
donkey
donkeys
donnybrook
donor
dons
doodle
doom
doomed
dooming
dooms
doomsday
door
doorbell
doorkeeper
doorman
doormen
doors
doorstep
doorsteps
doorway
doorways
dope
doped
doper
dopers
dopes
doping
dormant
dormitories
dormitory
dosage
dose
dosed
doses
dossier
dossiers
dot
dote
doted
dotes
doting
dotingly
dots
dotted
dotting
double
doubled
doubleheader
doubler
doublers
doubles
doublet
doubleton
doublets
doubling
doubloon
doubly
doubt
doubtable
doubted
doubter
doubters
doubtful
doubtfully
doubting
doubtless
doubtlessly
doubts
dough
doughnut
doughnuts
dove
dover
doves
dovetail
dowager
dowel
down
downcast
downed
downers
downfall
downfallen
downgrade
downhill
downlink
downlinks
download
downloaded
downloading
downloads
downplay
downplayed
downplaying
downplays
downpour
downright
downside
downstairs
downstream
downtown
downtowns
downtrodden
downturn
downward
downwards
downy
dowry
doze
dozed
dozen
dozens
dozenth
dozes
dozing
drab
draft
drafted
draftee
drafter
drafters
drafting
drafts
draftsman
draftsmen
drafty
drag
dragged
dragging
dragnet
dragon
dragonfly
dragonhead
dragons
dragoon
dragooned
dragoons
drags
drain
drainage
drained
drainer
draining
drains
drake
dram
drama
dramas
dramatic
dramatically
dramatics
dramatist
dramatists
drank
drape
draped
draper
draperies
drapers
drapery
drapes
drastic
drastically
draught
draughts
draw
drawback
drawbacks
drawbridge
drawbridges
drawer
drawers
drawing
drawings
drawl
drawled
drawling
drawls
drawn
drawnly
drawnness
draws
dread
dreaded
dreadful
dreadfully
dreading
dreadnought
dreads
dream
dreamboat
dreamed
dreamer
dreamers
dreamily
dreaming
dreamlike
dreams
dreamt
dreamy
dreariness
dreary
dredge
dregs
drench
drenched
drenches
drenching
dress
dressed
dresser
dressers
dresses
dressing
dressings
dressmaker
dressmakers
drew
dried
drier
driers
dries
driest
drift
drifted
drifter
drifters
drifting
drifts
drill
drilled
driller
drilling
drills
drily
drink
drinkable
drinker
drinkers
drinking
drinks
drip
dripping
drippy
drips
drive
driven
driver
drivers
drives
driveway
driveways
driving
drizzle
drizzly
droll
dromedary
drone
drones
drool
droop
drooped
drooping
droops
droopy
drop
droplet
dropout
dropped
dropper
droppers
dropping
droppings
drops
drosophila
drought
droughts
drove
drover
drovers
droves
drown
drowned
drowning
drownings
drowns
drowsiness
drowsy
drubbing
drudge
drudgery
drug
druggist
druggists
drugs
drugstore
drum
drumhead
drummed
drummer
drummers
drumming
drums
drunk
drunkard
drunkards
drunken
drunkenness
drunker
drunkly
drunks
dry
drying
dryly
dual
dualism
dualities
duality
dub
dubbed
dubious
dubiously
dubiousness
dubs
duchess
duchesses
duchy
duck
ducked
ducking
duckling
ducks
duct
ducts
dud
due
duel
dueling
duels
dues
duet
dug
duke
dukes
dull
dulled
duller
dullest
dulling
dullness
dulls
dully
duly
dumb
dumbbell
dumbbells
dumber
dumbest
dumbly
dumbness
dummies
dummy
dump
dumped
dumper
dumping
dumps
dunce
dunces
dune
dunes
dung
dungeon
dungeons
dunk
dupe
duplex
duplicable
duplicate
duplicated
duplicates
duplicating
duplication
duplications
duplicator
duplicators
duplicity
durabilities
durability
durable
durably
duration
durations
duress
during
dusk
duskiness
dusky
dust
dustbin
dusted
duster
dusters
dustier
dustiest
dusting
dusts
dusty
dutchess
duties
dutiful
dutifully
dutifulness
duty
dwarf
dwarfed
dwarfs
dwarves
dwell
dwelled
dweller
dwellers
dwelling
dwellings
dwells
dwelt
dwindle
dwindled
dwindling
dyad
dyadic
dye
dyed
dyeing
dyer
dyers
dyes
dying
dynamic
dynamically
dynamics
dynamism
dynamite
dynamited
dynamites
dynamiting
dynamo
dynastic
dynasties
dynasty
dyne
dysentery
dyspeptic
dystrophy
each
eager
eagerly
eagerness
eagle
eagles
ear
eardrum
eared
earl
earlier
earliest
earliness
earls
early
earmark
earmarked
earmarking
earmarkings
earmarks
earn
earned
earner
earners
earnest
earnestly
earnestness
earning
earnings
earns
earphone
earring
earrings
ears
earsplitting
earth
earthen
earthenware
earthliness
earthling
earthly
earthmover
earthquake
earthquakes
earths
earthworm
earthworms
earthy
ease
eased
easel
easement
easements
eases
easier
easiest
easily
easiness
easing
east
eastbound
easter
eastern
easterner
easterners
easternmost
eastward
eastwards
easy
easygoing
eat
eaten
eater
eaters
eating
eatings
eats
eaves
eavesdrop
eavesdropped
eavesdropper
eavesdroppers
eavesdropping
eavesdrops
ebb
ebbing
ebbs
ebony
eccentric
eccentricities
eccentricity
eccentrics
ecclesiastical
echelon
echo
echoed
echoes
echoing
eclectic
eclipse
eclipsed
eclipses
eclipsing
ecliptic
ecology
econometric
economic
economical
economically
economics
economies
economist
economists
economize
economized
economizer
economizers
economizes
economizing
economy
ecosystem
ecstasy
ecstatic
eddies
eddy
edge
edged
edges
edging
edible
edict
edicts
edifice
edifices
edit
edited
editing
edition
editions
editor
editorial
editorially
editorials
editors
edits
educable
educate
educated
educates
educating
education
educational
educationally
educations
educator
educators
eel
eelgrass
eels
eerie
eerily
effect
effected
effecting
effective
effectively
effectiveness
effector
effectors
effects
effectually
effectuate
effeminate
efficacy
efficiencies
efficiency
efficient
efficiently
effigy
effort
effortless
effortlessly
effortlessness
efforts
egalitarian
egg
egged
egghead
egging
eggplant
eggs
eggshell
ego
egocentric
egos
egotism
egotist
eigenfunction
eigenstate
eigenvalue
eigenvalues
eigenvector
eight
eighteen
eighteens
eighteenth
eightfold
eighth
eighthes
eighties
eightieth
eights
eighty
either
ejaculate
ejaculated
ejaculates
ejaculating
ejaculation
ejaculations
eject
ejected
ejecting
ejects
eke
eked
ekes
elaborate
elaborated
elaborately
elaborateness
elaborates
elaborating
elaboration
elaborations
elaborators
elapse
elapsed
elapses
elapsing
elastic
elastically
elasticity
elbow
elbowing
elbows
elder
elderly
elders
eldest
elect
elected
electing
election
elections
elective
electives
elector
electoral
electorate
electors
electric
electrical
electrically
electricalness
electrician
electricity
electrification
electrify
electrifying
electro
electrocardiogram
electrocardiograph
electrocute
electrocuted
electrocutes
electrocuting
electrocution
electrocutions
electrode
electrodes
electroencephalogram
electroencephalograph
electroencephalography
electrolysis
electrolyte
electrolytes
electrolytic
electromagnetic
electromechanical
electron
electronic
electronically
electronics
electrons
electrophoresis
electrophorus
elects
elegance
elegant
elegantly
elegy
element
elemental
elementals
elementary
elements
elephant
elephants
elevate
elevated
elevates
elevation
elevator
elevators
eleven
elevens
eleventh
elf
elicit
elicited
eliciting
elicits
elide
eligibility
eligible
eliminate
eliminated
eliminates
eliminating
elimination
eliminations
eliminator
eliminators
elision
elite
elitist
elk
elks
ellipse
ellipses
ellipsis
ellipsoid
ellipsoidal
ellipsoids
elliptic
elliptical
elliptically
elm
elms
elope
eloquence
eloquent
eloquently
else
elsewhere
elucidate
elucidated
elucidates
elucidating
elucidation
elude
eluded
eludes
eluding
elusive
elusively
elusiveness
elves
em
emaciate
emaciated
emacs
emanate
emanating
emancipate
emancipation
emasculate
embalm
embargo
embargoes
embark
embarked
embarks
embarrass
embarrassed
embarrasses
embarrassing
embarrassment
embassies
embassy
embed
embedded
embedding
embeds
embellish
embellished
embellishes
embellishing
embellishment
embellishments
ember
embezzle
emblem
embodied
embodies
embodiment
embodiments
embody
embodying
embolden
embrace
embraced
embraces
embracing
embroider
embroidered
embroideries
embroiders
embroidery
embroil
embryo
embryology
embryos
emerald
emeralds
emerge
emerged
emergence
emergencies
emergency
emergent
emerges
emerging
emeritus
emigrant
emigrants
emigrate
emigrated
emigrates
emigrating
emigration
eminence
eminent
eminently
emissary
emission
emit
emits
emitted
emitter
emitting
emotion
emotional
emotionally
emotions
empathy
emperor
emperors
emphases
emphasis
emphasize
emphasized
emphasizes
emphasizing
emphatic
emphatically
empire
empires
empirical
empirically
empiricist
empiricists
employ
employable
employed
employee
employees
employer
employers
employing
employment
employments
employs
emporium
empower
empowered
empowering
empowers
empress
emptied
emptier
empties
emptiest
emptily
emptiness
empty
emptying
emulate
emulated
emulates
emulating
emulation
emulations
emulator
emulators
en
enable
enabled
enabler
enablers
enables
enabling
enact
enacted
enacting
enactment
enacts
enamel
enameled
enameling
enamels
encamp
encamped
encamping
encamps
encapsulate
encapsulated
encapsulates
encapsulating
encapsulation
encased
enchant
enchanted
enchanter
enchanting
enchantment
enchantress
enchants
encipher
enciphered
enciphering
enciphers
encircle
encircled
encircles
enclose
enclosed
encloses
enclosing
enclosure
enclosures
encode
encoded
encoder
encoders
encodes
encoding
encodings
encompass
encompassed
encompasses
encompassing
encore
encounter
encountered
encountering
encounters
encourage
encouraged
encouragement
encouragements
encourages
encouraging
encouragingly
encroach
encrust
encrypt
encrypted
encrypting
encryption
encryptions
encrypts
encumber
encumbered
encumbering
encumbers
encyclopedia
encyclopedias
encyclopedic
end
endanger
endangered
endangering
endangers
endear
endeared
endearing
endears
endeavor
endeavored
endeavoring
endeavors
ended
endemic
ender
enders
endgame
ending
endings
endless
endlessly
endlessness
endorse
endorsed
endorsement
endorses
endorsing
endow
endowed
endowing
endowment
endowments
endows
endpoint
ends
endurable
endurably
endurance
endure
endured
endures
enduring
enduringly
enema
enemas
enemies
enemy
energetic
energies
energize
energy
enervate
enfeeble
enforce
enforceable
enforced
enforcement
enforcer
enforcers
enforces
enforcing
enfranchise
engage
engaged
engagement
engagements
engages
engaging
engagingly
engender
engendered
engendering
engenders
engine
engineer
engineered
engineering
engineers
engines
engrave
engraved
engraver
engraves
engraving
engravings
engross
engrossed
engrossing
engulf
enhance
enhanced
enhancement
enhancements
enhances
enhancing
enigma
enigmatic
enjoin
enjoined
enjoining
enjoins
enjoy
enjoyable
enjoyably
enjoyed
enjoying
enjoyment
enjoys
enlarge
enlarged
enlargement
enlargements
enlarger
enlargers
enlarges
enlarging
enlighten
enlightened
enlightening
enlightenment
enlist
enlisted
enlistment
enlists
enliven
enlivened
enlivening
enlivens
enmities
enmity
ennoble
ennobled
ennobles
ennobling
ennui
enormities
enormity
enormous
enormously
enough
enqueue
enqueued
enqueues
enquire
enquired
enquirer
enquires
enquiry
enrage
enraged
enrages
enraging
enrapture
enrich
enriched
enriches
enriching
enroll
enrolled
enrolling
enrollment
enrollments
enrolls
ensemble
ensembles
ensign
ensigns
enslave
enslaved
enslaves
enslaving
ensnare
ensnared
ensnares
ensnaring
ensue
ensued
ensues
ensuing
ensure
ensured
ensurer
ensurers
ensures
ensuring
entail
entailed
entailing
entails
entangle
enter
entered
entering
enterprise
enterprises
enterprising
enters
entertain
entertained
entertainer
entertainers
entertaining
entertainingly
entertainment
entertainments
entertains
enthusiasm
enthusiasms
enthusiast
enthusiastic
enthusiastically
enthusiasts
entice
enticed
enticer
enticers
entices
enticing
entire
entirely
entireties
entirety
entities
entitle
entitled
entitles
entitling
entity
entomb
entrance
entranced
entrances
entrap
entreat
entreated
entreaty
entree
entrench
entrenched
entrenches
entrenching
entrepreneur
entrepreneurial
entrepreneurs
entries
entropy
entrust
entrusted
entrusting
entrusts
entry
enumerable
enumerate
enumerated
enumerates
enumerating
enumeration
enumerative
enumerator
enumerators
enunciation
envelop
envelope
enveloped
enveloper
envelopes
enveloping
envelops
envied
envies
envious
enviously
enviousness
environ
environing
environment
environmental
environments
environs
envisage
envisaged
envisages
envision
envisioned
envisioning
envisions
envoy
envoys
envy
enzyme
epaulet
epaulets
ephemeral
epic
epicenter
epics
epidemic
epidemics
epidermis
epigram
epileptic
epilogue
episcopal
episode
episodes
epistemological
epistemology
epistle
epistles
epitaph
epitaphs
epitaxial
epitaxially
epithet
epithets
epitomize
epitomized
epitomizes
epitomizing
epoch
epochs
epsilon
equal
equaled
equaling
equalities
equality
equalization
equalize
equalized
equalizer
equalizers
equalizes
equalizing
equally
equals
equate
equated
equates
equating
equation
equations
equator
equatorial
equators
equestrian
equidistant
equilateral
equilibrate
equilibria
equilibrium
equilibriums
equinox
equip
equipment
equipoise
equipped
equipping
equips
equitable
equitably
equity
equivalence
equivalences
equivalent
equivalently
equivalents
equivocal
equivocally
era
eradicate
eradicated
eradicates
eradicating
eradication
eras
erasable
erase
erased
eraser
erasers
erases
erasing
erasure
ere
erect
erected
erecting
erection
erections
erector
erectors
erects
erg
ergo
ergodic
ermine
ermines
erode
erosion
erotic
erotica
err
errand
errant
errata
erratic
erratum
erred
erring
erringly
erroneous
erroneously
erroneousness
error
errors
errs
ersatz
erudite
erupt
eruption
escalate
escalated
escalates
escalating
escalation
escapable
escapade
escapades
escape
escaped
escapee
escapees
escapes
escaping
eschew
eschewed
eschewing
eschews
escort
escorted
escorting
escorts
escrow
esoteric
especial
especially
espionage
espouse
espoused
espouses
espousing
esprit
espy
esquire
esquires
essay
essayed
essays
essence
essences
essential
essentially
essentials
establish
established
establishes
establishing
establishment
establishments
estate
estates
esteem
esteemed
esteeming
esteems
esthetics
estimate
estimated
estimates
estimating
estimation
estimations
et
etch
etching
eternal
eternally
eternities
eternity
ether
ethereal
ethereally
ethers
ethic
ethical
ethically
ethics
ethnic
etiquette
etymology
eucalyptus
eunuch
eunuchs
euphemism
euphemisms
euphoria
euphoric
eureka
euthanasia
evacuate
evacuated
evacuation
evade
evaded
evades
evading
evaluate
evaluated
evaluates
evaluating
evaluation
evaluations
evaluative
evaluator
evaluators
evaporate
evaporated
evaporating
evaporation
evaporative
evasion
evasive
even
evened
evenhanded
evenhandedly
evenhandedness
evening
evenings
evenly
evenness
evens
event
eventful
eventfully
events
eventual
eventualities
eventuality
eventually
ever
evergreen
everlasting
everlastingly
evermore
every
everybody
everyday
everyone
everything
everywhere
evict
evicted
evicting
eviction
evictions
evicts
evidence
evidenced
evidences
evidencing
evident
evidently
evil
eviller
evilly
evils
evince
evinced
evinces
evoke
evoked
evokes
evoking
evolute
evolutes
evolution
evolutionary
evolutions
evolve
evolved
evolves
evolving
ewe
ewes
ex
exacerbate
exacerbated
exacerbates
exacerbating
exacerbation
exacerbations
exact
exacted
exacting
exactingly
exaction
exactions
exactitude
exactly
exactness
exacts
exaggerate
exaggerated
exaggerates
exaggerating
exaggeration
exaggerations
exalt
exaltation
exalted
exalting
exalts
exam
examination
examinations
examine
examined
examiner
examiners
examines
examining
example
examples
exams
exasperate
exasperated
exasperates
exasperating
exasperation
excavate
excavated
excavates
excavating
excavation
excavations
exceed
exceeded
exceeding
exceedingly
exceeds
excel
excelled
excellence
excellences
excellency
excellent
excellently
excelling
excels
except
excepted
excepting
exception
exceptionable
exceptional
exceptionally
exceptions
excepts
excerpt
excerpted
excerpts
excess
excesses
excessive
excessively
exchange
exchangeable
exchanged
exchanges
exchanging
exchequer
exchequers
excise
excised
excises
excising
excision
excitable
excitation
excitations
excite
excited
excitedly
excitement
excites
exciting
excitingly
exciton
exclaim
exclaimed
exclaimer
exclaimers
exclaiming
exclaims
exclamation
exclamations
exclamatory
exclude
excluded
excludes
excluding
exclusion
exclusionary
exclusions
exclusive
exclusively
exclusiveness
exclusivity
excommunicate
excommunicated
excommunicates
excommunicating
excommunication
excrete
excreted
excretes
excreting
excretion
excretions
excretory
excruciate
excursion
excursions
excusable
excusably
excuse
excused
excuses
excusing
exec
executable
execute
executed
executes
executing
execution
executional
executioner
executions
executive
executives
executor
executors
exemplar
exemplary
exemplification
exemplified
exemplifier
exemplifiers
exemplifies
exemplify
exemplifying
exempt
exempted
exempting
exemption
exempts
exercise
exercised
exerciser
exercisers
exercises
exercising
exert
exerted
exerting
exertion
exertions
exerts
exhale
exhaled
exhales
exhaling
exhaust
exhausted
exhaustedly
exhausting
exhaustion
exhaustive
exhaustively
exhausts
exhibit
exhibited
exhibiting
exhibition
exhibitions
exhibitor
exhibitors
exhibits
exhilarate
exhort
exhortation
exhortations
exhume
exigency
exile
exiled
exiles
exiling
exist
existed
existence
existent
existential
existentialism
existentialist
existentialists
existentially
existing
exists
exit
exited
exiting
exits
exodus
exorbitant
exorbitantly
exorcism
exorcist
exoskeleton
exotic
expand
expandable
expanded
expander
expanders
expanding
expands
expanse
expanses
expansible
expansion
expansionism
expansions
expansive
expect
expectancy
expectant
expectantly
expectation
expectations
expected
expectedly
expecting
expectingly
expects
expediency
expedient
expediently
expedite
expedited
expedites
expediting
expedition
expeditions
expeditious
expeditiously
expel
expelled
expelling
expels
expend
expendable
expended
expending
expenditure
expenditures
expends
expense
expenses
expensive
expensively
experience
experienced
experiences
experiencing
experiment
experimental
experimentally
experimentation
experimentations
experimented
experimenter
experimenters
experimenting
experiments
expert
expertise
expertly
expertness
experts
expiration
expirations
expire
expired
expires
expiring
explain
explainable
explained
explainer
explainers
explaining
explains
explanation
explanations
explanatory
expletive
explicit
explicitly
explicitness
explode
exploded
explodes
exploding
exploit
exploitable
exploitation
exploitations
exploited
exploiter
exploiters
exploiting
exploits
exploration
explorations
exploratory
explore
explored
explorer
explorers
explores
exploring
explosion
explosions
explosive
explosively
explosives
exponent
exponential
exponentially
exponentials
exponentiate
exponentiated
exponentiates
exponentiating
exponentiation
exponentiations
exponents
export
exportation
exported
exporter
exporters
exporting
exports
expose
exposed
exposer
exposers
exposes
exposing
exposition
expositions
expository
exposure
exposures
expound
expounded
expounder
expounding
expounds
express
expressed
expresses
expressibility
expressible
expressibly
expressing
expression
expressions
expressive
expressively
expressiveness
expressly
expulsion
expunge
expunged
expunges
expunging
expurgate
exquisite
exquisitely
exquisiteness
extant
extemporaneous
extend
extendable
extended
extending
extends
extensibility
extensible
extension
extensions
extensive
extensively
extent
extents
extenuate
extenuated
extenuating
extenuation
exterior
exteriors
exterminate
exterminated
exterminates
exterminating
extermination
external
externally
extinct
extinction
extinguish
extinguished
extinguisher
extinguishes
extinguishing
extirpate
extol
extort
extorted
extortion
extra
extract
extracted
extracting
extraction
extractions
extractor
extractors
extracts
extracurricular
extramarital
extraneous
extraneously
extraneousness
extraordinarily
extraordinariness
extraordinary
extrapolate
extrapolated
extrapolates
extrapolating
extrapolation
extrapolations
extras
extraterrestrial
extravagance
extravagant
extravagantly
extravaganza
extremal
extreme
extremely
extremes
extremist
extremists
extremities
extremity
extricate
extrinsic
extrovert
exuberance
exult
exultation
eye
eyeball
eyebrow
eyebrows
eyed
eyeful
eyeglass
eyeglasses
eyeing
eyelash
eyelid
eyelids
eyepiece
eyepieces
eyer
eyers
eyes
eyesight
eyewitness
eyewitnesses
eying
fable
fabled
fables
fabric
fabricate
fabricated
fabricates
fabricating
fabrication
fabrics
fabulous
fabulously
facade
facaded
facades
face
faced
faces
facet
faceted
facets
facial
facile
facilely
facilitate
facilitated
facilitates
facilitating
facilities
facility
facing
facings
facsimile
facsimiles
fact
faction
factions
factious
facto
factor
factored
factorial
factories
factoring
factorization
factorizations
factors
factory
facts
factual
factually
faculties
faculty
fade
faded
fadeout
fader
faders
fades
fading
fag
fags
fail
failed
failing
failings
fails
failsoft
failure
failures
fain
faint
fainted
fainter
faintest
fainting
faintly
faintness
faints
fair
fairer
fairest
fairies
fairing
fairly
fairness
fairs
fairy
fairyland
faith
faithful
faithfully
faithfulness
faithless
faithlessly
faithlessness
faiths
fake
faked
faker
fakes
faking
falcon
falconer
falcons
fall
fallacies
fallacious
fallacy
fallen
fallibility
fallible
falling
fallout
fallow
falls
false
falsehood
falsehoods
falsely
falseness
falsification
falsified
falsifies
falsify
falsifying
falsity
falter
faltered
falters
fame
famed
fames
familial
familiar
familiarities
familiarity
familiarization
familiarize
familiarized
familiarizes
familiarizing
familiarly
familiarness
families
familism
family
famine
famines
famish
famous
famously
fan
fanatic
fanaticism
fanatics
fancied
fancier
fanciers
fancies
fanciest
fanciful
fancifully
fancily
fanciness
fancy
fancying
fanfare
fanfold
fang
fangled
fangs
fanned
fanning
fanout
fans
fantasies
fantasize
fantastic
fantasy
far
farad
faraway
farce
farces
fare
fared
fares
farewell
farewells
farfetched
farina
faring
farm
farmed
farmer
farmers
farmhouse
farmhouses
farming
farmland
farms
farmyard
farmyards
farsighted
farther
farthest
farthing
fascicle
fascinate
fascinated
fascinates
fascinating
fascination
fascism
fascist
fashion
fashionable
fashionably
fashioned
fashioning
fashions
fast
fasted
fasten
fastened
fastener
fasteners
fastening
fastenings
fastens
faster
fastest
fastidious
fasting
fastness
fasts
fat
fatal
fatalities
fatality
fatally
fatals
fate
fated
fateful
fates
father
fathered
fatherland
fatherly
fathers
fathom
fathomed
fathoming
fathoms
fatigue
fatigued
fatigues
fatiguing
fatness
fats
fatten
fattened
fattener
fatteners
fattening
fattens
fatter
fattest
fatty
faucet
fault
faulted
faulting
faultless
faultlessly
faults
faulty
faun
fauna
favor
favorable
favorably
favored
favorer
favoring
favorite
favorites
favoritism
favors
fawn
fawned
fawning
fawns
faze
fear
feared
fearful
fearfully
fearing
fearless
fearlessly
fearlessness
fears
fearsome
feasibility
feasible
feast
feasted
feasting
feasts
feat
feather
featherbed
featherbedding
feathered
featherer
featherers
feathering
feathers
featherweight
feathery
feats
feature
featured
features
featuring
fecund
fed
federal
federalist
federally
federals
federation
fee
feeble
feebleness
feebler
feeblest
feebly
feed
feedback
feeder
feeders
feeding
feedings
feeds
feel
feeler
feelers
feeling
feelingly
feelings
feels
fees
feet
feign
feigned
feigning
felicities
felicity
feline
fell
fellatio
felled
felling
fellow
fellows
fellowship
fellowships
felon
felonious
felony
felt
felts
female
females
feminine
femininity
feminism
feminist
femur
femurs
fen
fence
fenced
fencer
fencers
fences
fencing
fend
ferment
fermentation
fermentations
fermented
fermenting
ferments
fern
ferns
ferocious
ferociously
ferociousness
ferocity
ferret
ferried
ferries
ferrite
ferry
fertile
fertilely
fertility
fertilization
fertilize
fertilized
fertilizer
fertilizers
fertilizes
fertilizing
fervent
fervently
fervor
fervors
festival
festivals
festive
festively
festivities
festivity
fetal
fetch
fetched
fetches
fetching
fetchingly
fetid
fetish
fetter
fettered
fetters
fettle
fetus
feud
feudal
feudalism
feuds
fever
fevered
feverish
feverishly
fevers
few
fewer
fewest
fewness
fiance
fiancee
fiasco
fiat
fib
fibbing
fiber
fibers
fibrosities
fibrosity
fibrous
fibrously
fickle
fickleness
fiction
fictional
fictionally
fictions
fictitious
fictitiously
fiddle
fiddled
fiddler
fiddles
fiddlestick
fiddlesticks
fiddling
fidelity
fidget
fiducial
fief
fiefdom
field
fielded
fielder
fielders
fielding
fieldwork
fiend
fiendish
fierce
fiercely
fierceness
fiercer
fiercest
fiery
fife
fifteen
fifteens
fifteenth
fifth
fifties
fiftieth
fifty
fig
fight
fighter
fighters
fighting
fights
figs
figurative
figuratively
figure
figured
figures
figuring
figurings
filament
filaments
file
filed
filename
filenames
filer
files
filial
filibuster
filing
filings
fill
fillable
filled
filler
fillers
filling
fillings
fills
filly
film
filmed
filming
films
filter
filtered
filtering
filters
filth
filthier
filthiest
filthiness
filthy
fin
final
finality
finalization
finalize
finalized
finalizes
finalizing
finally
finals
finance
financed
finances
financial
financially
financier
financiers
financing
find
finder
finders
finding
findings
finds
fine
fined
finely
fineness
finer
fines
finesse
finessed
finessing
finest
finger
fingered
fingering
fingerings
fingernail
fingerprint
fingerprints
fingers
fingertip
finicky
fining
finish
finished
finisher
finishers
finishes
finishing
finite
finitely
finiteness
fink
finny
fins
fir
fire
firearm
firearms
fireboat
firebreak
firebug
firecracker
fired
fireflies
firefly
firehouse
firelight
fireman
firemen
fireplace
fireplaces
firepower
fireproof
firer
firers
fires
fireside
firewall
firewood
fireworks
firing
firings
firm
firmament
firmed
firmer
firmest
firming
firmly
firmness
firms
firmware
first
firsthand
firstly
firsts
fiscal
fiscally
fish
fished
fisher
fisherman
fishermen
fishers
fishery
fishes
fishing
fishmonger
fishpond
fishy
fission
fissure
fissured
fist
fisted
fisticuff
fists
fit
fitful
fitfully
fitly
fitness
fits
fitted
fitter
fitters
fitting
fittingly
fittings
five
fivefold
fives
fix
fixate
fixated
fixates
fixating
fixation
fixations
fixed
fixedly
fixedness
fixer
fixers
fixes
fixing
fixings
fixture
fixtures
fizzle
fizzled
flabbergast
flabbergasted
flack
flag
flagellate
flagged
flagging
flagpole
flagrant
flagrantly
flags
flail
flair
flak
flake
flaked
flakes
flaking
flaky
flam
flamboyant
flame
flamed
flamer
flamers
flames
flaming
flammable
flank
flanked
flanker
flanking
flanks
flannel
flannels
flap
flaps
flare
flared
flares
flaring
flash
flashback
flashed
flasher
flashers
flashes
flashing
flashlight
flashlights
flashy
flask
flat
flatbed
flatly
flatness
flats
flatten
flattened
flattening
flatter
flattered
flatterer
flattering
flattery
flattest
flatulent
flatus
flatworm
flaunt
flaunted
flaunting
flaunts
flavor
flavored
flavoring
flavorings
flavors
flaw
flawed
flawless
flawlessly
flaws
flax
flaxen
flea
fleas
fled
fledged
fledgling
fledglings
flee
fleece
fleeces
fleecy
fleeing
flees
fleet
fleetest
fleeting
fleetly
fleetness
fleets
flesh
fleshed
fleshes
fleshing
fleshly
fleshy
flew
flex
flexibilities
flexibility
flexible
flexibly
flick
flicked
flicker
flickering
flicking
flicks
flier
fliers
flies
flight
flights
flimsy
flinch
flinched
flinches
flinching
fling
flings
flint
flinty
flip
flipflop
flipped
flips
flirt
flirtation
flirtatious
flirted
flirting
flirts
flit
flitting
float
floated
floater
floating
floats
flock
flocked
flocking
flocks
flog
flogging
flood
flooded
flooding
floodlight
floodlit
floods
floor
floored
flooring
floorings
floors
flop
floppies
floppily
flopping
floppy
flops
flora
floral
florid
florin
florist
floss
flossed
flosses
flossing
flotation
flotilla
flounder
floundered
floundering
flounders
flour
floured
flourish
flourished
flourishes
flourishing
flow
flowchart
flowcharting
flowcharts
flowed
flower
flowered
floweriness
flowering
flowerpot
flowers
flowery
flowing
flown
flows
flu
fluctuate
fluctuates
fluctuating
fluctuation
fluctuations
flue
fluency
fluent
fluently
fluff
fluffier
fluffiest
fluffy
fluid
fluidity
fluidly
fluids
fluke
flung
flunked
fluoresce
fluorescent
flurried
flurry
flush
flushed
flushes
flushing
flute
fluted
fluting
flutter
fluttered
fluttering
flutters
flux
fly
flyable
flyer
flyers
flying
foal
foam
foamed
foaming
foams
foamy
fob
fobbing
focal
focally
foci
focus
focused
focuses
focusing
focussed
fodder
foe
foes
fog
fogged
foggier
foggiest
foggily
fogging
foggy
fogs
fogy
foible
foil
foiled
foiling
foils
foist
fold
folded
folder
folders
folding
foldout
folds
foliage
folk
folklore
folks
folksong
folksy
follies
follow
followed
follower
followers
following
followings
follows
folly
fond
fonder
fondle
fondled
fondles
fondling
fondly
fondness
font
fonts
food
foods
foodstuff
foodstuffs
fool
fooled
foolhardy
fooling
foolish
foolishly
foolishness
foolproof
fools
foot
footage
football
footballs
footbridge
footed
footer
footers
footfall
foothill
foothold
footing
footman
footnote
footnotes
footpath
footprint
footprints
footstep
footsteps
for
forage
foraged
forages
foraging
foray
forays
forbade
forbear
forbearance
forbears
forbid
forbidden
forbidding
forbids
force
forced
forceful
forcefully
forcefulness
forcer
forces
forcible
forcibly
forcing
ford
fords
fore
forearm
forearms
foreboding
forecast
forecasted
forecaster
forecasters
forecasting
forecastle
forecasts
forefather
forefathers
forefinger
forefingers
forego
foregoes
foregoing
foregone
foreground
forehead
foreheads
foreign
foreigner
foreigners
foreigns
foreman
foremost
forenoon
forensic
forerunners
foresee
foreseeable
foreseen
foresees
foresight
foresighted
forest
forestall
forestalled
forestalling
forestallment
forestalls
forested
forester
foresters
forestry
forests
foretell
foretelling
foretells
foretold
forever
forewarn
forewarned
forewarning
forewarnings
forewarns
forfeit
forfeited
forfeiture
forgave
forge
forged
forger
forgeries
forgery
forges
forget
forgetful
forgetfulness
forgets
forgettable
forgettably
forgetting
forging
forgivable
forgivably
forgive
forgiven
forgiveness
forgives
forgiving
forgivingly
forgot
forgotten
fork
forked
forking
forklift
forks
forlorn
forlornly
form
formal
formalism
formalisms
formalities
formality
formalization
formalizations
formalize
formalized
formalizes
formalizing
formally
formant
formants
format
formation
formations
formative
formatively
formats
formatted
formatter
formatters
formatting
formed
former
formerly
formidable
forming
forms
formula
formulae
formulas
formulate
formulated
formulates
formulating
formulation
formulations
formulator
formulators
fornication
forsake
forsaken
forsakes
forsaking
fort
forte
forthcoming
forthright
forthwith
fortier
forties
fortieth
fortification
fortifications
fortified
fortifies
fortify
fortifying
fortiori
fortitude
fortnight
fortnightly
fortress
fortresses
forts
fortuitous
fortuitously
fortunate
fortunately
fortune
fortunes
forty
forum
forums
forward
forwarded
forwarder
forwarding
forwardness
forwards
fossil
foster
fostered
fostering
fosters
fought
foul
fouled
foulest
fouling
foully
foulmouth
foulness
fouls
found
foundation
foundations
founded
founder
foundered
founders
founding
foundling
foundries
foundry
founds
fount
fountain
fountains
founts
four
fourfold
fours
fourscore
foursome
foursquare
fourteen
fourteens
fourteenth
fourth
fowl
fowler
fowls
fox
foxes
fraction
fractional
fractionally
fractions
fracture
fractured
fractures
fracturing
fragile
fragment
fragmentary
fragmentation
fragmented
fragmenting
fragments
fragrance
fragrances
fragrant
fragrantly
frail
frailest
frailty
frame
framed
framer
frames
framework
frameworks
framing
franc
franchise
franchises
francs
frank
franked
franker
frankest
franking
frankly
frankness
franks
frantic
frantically
fraternal
fraternally
fraternities
fraternity
fraud
frauds
fraudulent
fraught
fray
frayed
fraying
frays
frazzle
freak
freakish
freaks
freckle
freckled
freckles
free
freed
freedom
freedoms
freeing
freeings
freely
freeman
freeness
freer
frees
freest
freestyle
freeway
freewheel
freeze
freezer
freezers
freezes
freezing
freight
freighted
freighter
freighters
freighting
freights
frenetic
frenzied
frenzy
freon
frequencies
frequency
frequent
frequented
frequenter
frequenters
frequenting
frequently
frequents
fresco
frescoes
fresh
freshen
freshened
freshener
fresheners
freshening
freshens
fresher
freshest
freshly
freshman
freshmen
freshness
freshwater
fret
fretful
fretfully
fretfulness
friar
friars
fricative
fricatives
friction
frictionless
frictions
fried
friend
friendless
friendlier
friendliest
friendliness
friendly
friends
friendship
friendships
fries
frieze
friezes
frigate
frigates
fright
frighten
frightened
frightening
frighteningly
frightens
frightful
frightfully
frightfulness
frigid
frill
frills
fringe
fringed
frisk
frisked
frisking
frisks
frisky
fritter
frivolity
frivolous
frivolously
fro
frock
frocks
frog
frogs
frolic
frolics
from
front
frontage
frontal
fronted
frontier
frontiers
frontiersman
frontiersmen
fronting
fronts
frost
frostbite
frostbitten
frosted
frosting
frosts
frosty
froth
frothing
frothy
frown
frowned
frowning
frowns
froze
frozen
frozenly
frugal
frugally
fruit
fruitful
fruitfully
fruitfulness
fruition
fruitless
fruitlessly
fruits
frustrate
frustrated
frustrates
frustrating
frustration
frustrations
fry
fudge
fuel
fueled
fueling
fuels
fugitive
fugitives
fugue
fulcrum
fulfill
fulfilled
fulfilling
fulfillment
fulfillments
fulfills
full
fuller
fullest
fullness
fully
fulminate
fumble
fumbled
fumbling
fume
fumed
fumes
fuming
fun
function
functional
functionalities
functionality
functionally
functionals
functionary
functioned
functioning
functions
functor
functors
fund
fundamental
fundamentally
fundamentals
funded
funder
funders
funding
funds
funeral
funerals
funereal
fungal
fungi
fungible
fungicide
fungus
funk
funnel
funneled
funneling
funnels
funnier
funniest
funnily
funniness
funny
fur
furies
furious
furiouser
furiously
furlong
furlough
furnace
furnaces
furnish
furnished
furnishes
furnishing
furnishings
furniture
furrier
furrow
furrowed
furrows
furry
furs
further
furthered
furthering
furthermore
furthermost
furthers
furthest
furtive
furtively
furtiveness
fury
fuse
fused
fuses
fusing
fusion
fuss
fussing
fussy
futile
futility
future
futures
futuristic
fuzz
fuzzier
fuzziness
fuzzy
gab
gabardine
gabbing
gable
gabled
gabler
gables
gad
gadfly
gadget
gadgetry
gadgets
gag
gagged
gagging
gaging
gags
gaieties
gaiety
gaily
gain
gained
gainer
gainers
gainful
gaining
gains
gait
gaited
gaiter
gaiters
galactic
galaxies
galaxy
gale
gall
gallant
gallantly
gallantry
gallants
galled
galleried
galleries
gallery
galley
galleys
galling
gallon
gallons
gallop
galloped
galloper
galloping
gallops
gallows
galls
gallstone
gambit
gamble
gambled
gambler
gamblers
gambles
gambling
gambol
game
gamed
gamely
gameness
games
gaming
gamma
gander
gang
gangland
gangling
gangplank
gangrene
gangs
gangster
gangsters
gantry
gap
gape
gaped
gapes
gaping
gaps
garage
garaged
garages
garb
garbage
garbages
garbed
garble
garbled
garden
gardened
gardener
gardeners
gardening
gardens
gargantuan
gargle
gargled
gargles
gargling
garland
garlanded
garlic
garment
garments
garner
garnered
garnish
garrison
garrisoned
garter
garters
gas
gaseous
gaseously
gases
gash
gashes
gasket
gaslight
gasoline
gasp
gasped
gasping
gasps
gassed
gasser
gassing
gassings
gassy
gastric
gastrointestinal
gastronome
gastronomy
gate
gated
gateway
gateways
gather
gathered
gatherer
gatherers
gathering
gatherings
gathers
gating
gator
gauche
gaudiness
gaudy
gauge
gauged
gauges
gaunt
gauntness
gauze
gave
gavel
gawk
gawky
gay
gayer
gayest
gayety
gayly
gayness
gaze
gazed
gazelle
gazer
gazers
gazes
gazette
gazing
gear
geared
gearing
gears
gecko
geese
geisha
gel
gelatin
gelatine
gelatinous
geld
gelled
gelling
gels
gem
gems
gender
genders
gene
genealogy
general
generalist
generalists
generalities
generality
generalization
generalizations
generalize
generalized
generalizer
generalizers
generalizes
generalizing
generally
generals
generate
generated
generates
generating
generation
generations
generative
generator
generators
generic
generically
generosities
generosity
generous
generously
generousness
genes
genesis
genetic
genetically
genial
genially
genie
genius
geniuses
genre
genres
gent
genteel
gentile
gentle
gentleman
gentlemanly
gentlemen
gentleness
gentler
gentlest
gentlewoman
gently
gentry
genuine
genuinely
genuineness
genus
geocentric
geodesic
geodesy
geodetic
geographer
geographic
geographical
geographically
geography
geological
geologist
geologists
geology
geometric
geometrical
geometrically
geometrician
geometries
geometry
geophysical
geophysics
geosynchronous
geranium
gerbil
geriatric
germ
germane
germicide
germinal
germinate
germinated
germinates
germinating
germination
germs
gerund
gesture
gestured
gestures
gesturing
get
getaway
gets
getter
getters
getting
geyser
ghastly
ghetto
ghost
ghosted
ghostly
ghosts
giant
giants
gibberish
giddiness
giddy
gift
gifted
gifts
gig
gigabit
gigabits
gigabyte
gigabytes
gigacycle
gigahertz
gigantic
gigavolt
gigawatt
giggle
giggled
giggles
giggling
gild
gilded
gilding
gilds
gill
gills
gilt
gimmick
gimmicks
gin
ginger
gingerbread
gingerly
gingham
ginghams
gins
giraffe
giraffes
gird
girder
girders
girdle
girl
girlfriend
girlie
girlish
girls
girt
girth
gist
give
giveaway
given
giver
givers
gives
giving
glacial
glacier
glaciers
glad
gladden
gladder
gladdest
glade
gladiator
gladly
gladness
glamor
glamorous
glamour
glance
glanced
glances
glancing
gland
glands
glandular
glare
glared
glares
glaring
glaringly
glass
glassed
glasses
glassy
glaucoma
glaze
glazed
glazer
glazes
glazing
gleam
gleamed
gleaming
gleams
glean
gleaned
gleaner
gleaning
gleanings
gleans
glee
gleeful
gleefully
glees
glen
glens
glide
glided
glider
gliders
glides
glimmer
glimmered
glimmering
glimmers
glimpse
glimpsed
glimpses
glint
glinted
glinting
glints
glisten
glistened
glistening
glistens
glitch
glitter
glittered
glittering
glitters
gloat
global
globally
globe
globes
globular
globularity
gloom
gloomily
gloomy
glories
glorification
glorified
glorifies
glorify
glorious
gloriously
glory
glorying
gloss
glossaries
glossary
glossed
glosses
glossing
glossy
glottal
glove
gloved
glover
glovers
gloves
gloving
glow
glowed
glower
glowers
glowing
glowingly
glows
glue
glued
glues
gluing
glut
glutton
gnash
gnat
gnats
gnaw
gnawed
gnawing
gnaws
gnome
gnomon
gnu
go
goad
goaded
goal
goals
goat
goatee
goatees
goats
gobble
gobbled
gobbler
gobblers
gobbles
goblet
goblets
goblin
goblins
god
goddess
goddesses
godfather
godhead
godlike
godly
godmother
godmothers
godparent
gods
godsend
godson
goes
goggles
going
goings
gold
golden
goldenly
goldenness
goldenrod
goldfish
golding
golds
goldsmith
golf
golfer
golfers
golfing
golly
gondola
gone
goner
gong
gongs
good
goodby
goodbye
goodies
goodly
goodness
goods
goodwill
goody
goof
goofed
goofs
goofy
goose
gopher
gore
gorge
gorgeous
gorgeously
gorges
gorging
gorilla
gorillas
gory
gosh
gospel
gospelers
gospels
gossip
gossiped
gossiping
gossips
got
gotten
gouge
gouged
gouges
gouging
gourd
gourmet
gout
govern
governance
governed
governess
governing
government
governmental
governmentally
governments
governor
governors
governs
gown
gowned
gowns
grab
grabbed
grabber
grabbers
grabbing
grabbings
grabs
grace
graced
graceful
gracefully
gracefulness
graces
gracing
gracious
graciously
graciousness
grad
gradation
gradations
grade
graded
grader
graders
grades
gradient
gradients
grading
gradings
gradual
gradually
graduate
graduated
graduates
graduating
graduation
graduations
graft
grafted
grafter
grafting
grafts
graham
grahams
grail
grain
grained
graining
grains
gram
grammar
grammarian
grammars
grammatic
grammatical
grammatically
grams
granaries
granary
grand
grandchild
grandchildren
granddaughter
grander
grandest
grandeur
grandfather
grandfathers
grandiose
grandly
grandma
grandmother
grandmothers
grandnephew
grandness
grandniece
grandpa
grandparent
grands
grandson
grandsons
grandstand
grange
granite
granny
granola
grant
granted
grantee
granter
granting
grantor
grants
granularity
granulate
granulated
granulates
granulating
grape
grapefruit
grapes
grapevine
graph
graphed
graphic
graphical
graphically
graphics
graphing
graphite
graphs
grapple
grappled
grappling
grasp
graspable
grasped
grasping
graspingly
grasps
grass
grassed
grassers
grasses
grassier
grassiest
grassland
grassy
grate
grated
grateful
gratefully
gratefulness
grater
grates
gratification
gratified
gratify
gratifying
grating
gratings
gratis
gratitude
gratuities
gratuitous
gratuitously
gratuitousness
gratuity
grave
gravel
gravelly
gravely
graven
graveness
graver
gravest
gravestone
graveyard
gravitate
gravitation
gravitational
gravity
gravy
gray
grayed
grayer
grayest
graying
grayness
graze
grazed
grazer
grazing
grease
greased
greases
greasy
great
greater
greatest
greatly
greatness
greed
greedily
greediness
greedy
green
greener
greenery
greenest
greengrocer
greenhouse
greenhouses
greening
greenish
greenly
greenness
greens
greenware
greet
greeted
greeter
greeting
greetings
greets
gregarious
grenade
grenades
grew
grey
greyest
greyhound
greying
grid
griddle
gridiron
grids
grief
griefs
grievance
grievances
grieve
grieved
griever
grievers
grieves
grieving
grievingly
grievous
grievously
grill
grilled
grilling
grills
grim
grimace
grime
grimed
grimly
grimness
grin
grind
grinder
grinders
grinding
grindings
grinds
grindstone
grindstones
grinning
grins
grip
gripe
griped
gripes
griping
gripped
gripping
grippingly
grips
grisly
grist
grit
grits
gritty
grizzly
groan
groaned
groaner
groaners
groaning
groans
grocer
groceries
grocers
grocery
groggy
groin
groom
groomed
grooming
grooms
groove
grooved
grooves
grope
groped
gropes
groping
gross
grossed
grosser
grosses
grossest
grossing
grossly
grossness
grotesque
grotesquely
grotesques
grotto
grottos
ground
grounded
grounder
grounders
grounding
grounds
groundwork
group
grouped
grouping
groupings
groups
grouse
grove
grovel
groveled
groveling
grovels
grovers
groves
grow
grower
growers
growing
growl
growled
growling
growls
grown
grownup
grownups
grows
growth
growths
grub
grubby
grubs
grudge
grudges
grudgingly
gruesome
gruff
gruffly
grumble
grumbled
grumbles
grumbling
grunt
grunted
grunting
grunts
guano
guarantee
guaranteed
guaranteeing
guaranteer
guaranteers
guarantees
guaranty
guard
guarded
guardedly
guardhouse
guardian
guardians
guardianship
guarding
guards
gubernatorial
guerrilla
guerrillas
guess
guessed
guesses
guessing
guesswork
guest
guests
guidance
guide
guidebook
guidebooks
guided
guideline
guidelines
guides
guiding
guild
guilder
guilders
guile
guilt
guiltier
guiltiest
guiltily
guiltiness
guiltless
guiltlessly
guilty
guinea
guise
guises
guitar
guitars
gulch
gulches
gulf
gulfs
gull
gulled
gullies
gulling
gulls
gully
gulp
gulped
gulps
gum
gumming
gumption
gums
gun
gunfire
gunman
gunmen
gunned
gunner
gunners
gunnery
gunning
gunny
gunplay
gunpowder
guns
gunshot
gurgle
guru
gush
gushed
gusher
gushes
gushing
gust
gusto
gusts
gusty
gut
guts
gutsy
gutter
guttered
gutters
gutting
guttural
guy
guyed
guyer
guyers
guying
guys
gymnasium
gymnasiums
gymnast
gymnastic
gymnastics
gymnasts
gypsies
gypsy
gyro
gyrocompass
gyroscope
gyroscopes
ha
habeas
habit
habitat
habitation
habitations
habitats
habits
habitual
habitually
habitualness
hack
hacked
hacker
hackers
hacking
hackneyed
hacks
hacksaw
had
haddock
hag
haggard
haggardly
haggle
hail
hailed
hailing
hails
hailstone
hailstorm
hair
haircut
haircuts
hairier
hairiness
hairless
hairpin
hairs
hairy
halcyon
hale
haler
half
halfhearted
halfway
hall
hallmark
hallmarks
hallow
hallowed
halls
hallucinate
hallway
hallways
halogen
halt
halted
halter
halters
halting
haltingly
halts
halve
halved
halvers
halves
halving
ham
hamburger
hamburgers
hamlet
hamlets
hammer
hammered
hammering
hammers
hamming
hammock
hammocks
hamper
hampered
hampers
hams
hamster
hand
handbag
handbags
handbook
handbooks
handcuff
handcuffed
handcuffing
handcuffs
handed
handful
handfuls
handgun
handicap
handicapped
handicaps
handier
handiest
handily
handiness
handing
handiwork
handkerchief
handkerchiefs
handle
handled
handler
handlers
handles
handling
handmaid
handout
hands
handshake
handshakes
handshaking
handsome
handsomely
handsomeness
handsomer
handsomest
handwriting
handwritten
handy
hang
hangar
hangars
hanged
hanger
hangers
hanging
hangman
hangmen
hangout
hangover
hangovers
hangs
hap
haphazard
haphazardly
haphazardness
hapless
haplessly
haplessness
haply
happen
happened
happening
happenings
happens
happier
happiest
happily
happiness
happy
harass
harassed
harasses
harassing
harassment
harbinger
harbor
harbored
harboring
harbors
hard
hardboiled
hardcopy
harden
harder
hardest
hardhat
hardiness
hardly
hardness
hardscrabble
hardship
hardships
hardware
hardwired
hardworking
hardy
hare
harelip
harem
hares
hark
harken
harlot
harlots
harm
harmed
harmful
harmfully
harmfulness
harming
harmless
harmlessly
harmlessness
harmonic
harmonics
harmonies
harmonious
harmoniously
harmoniousness
harmonize
harmony
harms
harness
harnessed
harnessing
harp
harper
harpers
harping
harried
harrier
harrow
harrowed
harrowing
harrows
harry
harsh
harsher
harshly
harshness
hart
harvest
harvested
harvester
harvesting
harvests
has
hash
hashed
hasher
hashes
hashing
hashish
hassle
haste
hasten
hastened
hastening
hastens
hastily
hastiness
hasty
hat
hatch
hatched
hatchet
hatchets
hatching
hate
hated
hateful
hatefully
hatefulness
hater
hates
hating
hatred
hats
haughtily
haughtiness
haughty
haul
hauled
hauler
hauling
hauls
haunch
haunches
haunt
haunted
haunter
haunting
haunts
have
haven
havens
haves
having
havoc
hawk
hawked
hawker
hawkers
hawks
hay
haying
haystack
hazard
hazardous
hazards
haze
hazel
hazes
haziness
hazy
he
head
headache
headaches
headed
header
headers
headgear
heading
headings
headland
headlands
headlight
headline
headlined
headlines
headlining
headlong
headmaster
headphone
headquarters
headroom
heads
headset
headway
heal
healed
healer
healers
healing
heals
health
healthful
healthfully
healthfulness
healthier
healthiest
healthily
healthiness
healthy
heap
heaped
heaping
heaps
hear
heard
hearer
hearers
hearing
hearings
hearken
hears
hearsay
heart
heartbeat
heartbreak
hearten
heartiest
heartily
heartiness
heartless
hearts
hearty
heat
heatable
heated
heatedly
heater
heaters
heath
heathen
heather
heating
heats
heave
heaved
heaven
heavenly
heavens
heaver
heavers
heaves
heavier
heaviest
heavily
heaviness
heaving
heavy
heavyweight
heck
heckle
hectic
hedge
hedged
hedgehog
hedgehogs
hedges
hedonism
hedonist
heed
heeded
heedless
heedlessly
heedlessness
heeds
heel
heeled
heelers
heeling
heels
hefty
hegemony
heifer
height
heighten
heightened
heightening
heightens
heights
heinous
heinously
heir
heiress
heiresses
heirs
held
helical
helicopter
heliocentric
helium
helix
hell
hellfire
hellish
hello
hells
helm
helmet
helmets
helmsman
help
helped
helper
helpers
helpful
helpfully
helpfulness
helping
helpless
helplessly
helplessness
helpmate
helps
hem
hemisphere
hemispheres
hemlock
hemlocks
hemoglobin
hemorrhoid
hemostat
hemostats
hemp
hempen
hems
hen
hence
henceforth
henchman
henchmen
henpeck
hens
hepatitis
her
herald
heralded
heralding
heralds
herb
herbivore
herbivorous
herbs
herd
herded
herder
herding
herds
here
hereabout
hereabouts
hereafter
hereby
hereditary
heredity
herein
hereinafter
hereof
heres
heresy
heretic
heretics
hereto
heretofore
hereunder
herewith
heritage
heritages
hermetic
hermetically
hermit
hermitian
hermits
hero
heroes
heroic
heroically
heroics
heroin
heroine
heroines
heroism
heron
herons
herpes
herring
herrings
hers
herself
hertz
hesitant
hesitantly
hesitate
hesitated
hesitates
hesitating
hesitatingly
hesitation
hesitations
heterogeneity
heterogeneous
heterogeneously
heterogeneousness
heterogenous
heterosexual
heuristic
heuristically
heuristics
hew
hewed
hewer
hews
hex
hexadecimal
hexagon
hexagonal
hexagonally
hexagons
hey
hi
hibernate
hick
hickory
hid
hidden
hide
hideous
hideously
hideousness
hideout
hideouts
hides
hiding
hierarchal
hierarchic
hierarchical
hierarchically
hierarchies
hierarchy
high
higher
highest
highland
highlander
highlands
highlight
highlighted
highlighting
highlights
highly
highness
highnesses
highway
highwayman
highwaymen
highways
hijack
hijacked
hike
hiked
hiker
hikes
hiking
hilarious
hilariously
hilarity
hill
hillbilly
hillock
hills
hillside
hillsides
hilltop
hilltops
hilt
hilts
him
himself
hind
hinder
hindered
hindering
hinders
hindrance
hindrances
hindsight
hinge
hinged
hinges
hint
hinted
hinting
hints
hip
hippo
hippopotamus
hips
hire
hired
hirer
hirers
hires
hiring
hirings
his
hiss
hissed
hisses
hissing
histogram
histograms
historian
historians
historic
historical
historically
histories
history
hit
hitch
hitched
hitchhike
hitchhiked
hitchhiker
hitchhikers
hitchhikes
hitchhiking
hitching
hither
hitherto
hits
hitter
hitters
hitting
hive
hoar
hoard
hoarder
hoarding
hoariness
hoarse
hoarsely
hoarseness
hoary
hobbies
hobble
hobbled
hobbles
hobbling
hobby
hobbyhorse
hobbyist
hobbyists
hockey
hodgepodge
hoe
hoes
hog
hogging
hogs
hoist
hoisted
hoisting
hoists
hold
holden
holder
holders
holding
holdings
holds
hole
holed
holes
holiday
holidays
holies
holiness
holistic
hollow
hollowed
hollowing
hollowly
hollowness
hollows
holly
holocaust
hologram
holograms
holy
homage
home
homed
homeless
homely
homemade
homemaker
homemakers
homeomorphic
homeomorphism
homeomorphisms
homeopath
homeowner
homer
homers
homes
homesick
homesickness
homespun
homestead
homesteader
homesteaders
homesteads
homeward
homewards
homework
homicidal
homicide
homing
homo
homogeneities
homogeneity
homogeneous
homogeneously
homogeneousness
homomorphic
homomorphism
homomorphisms
homosexual
hone
honed
honer
hones
honest
honestly
honesty
honey
honeybee
honeycomb
honeycombed
honeydew
honeymoon
honeymooned
honeymooner
honeymooners
honeymooning
honeymoons
honeysuckle
honing
honor
honorable
honorableness
honorably
honoraries
honorarium
honorary
honored
honorer
honoring
honors
hood
hooded
hoodlum
hoods
hoodwink
hoodwinked
hoodwinking
hoodwinks
hoof
hoofs
hook
hooked
hooker
hookers
hooking
hooks
hookup
hookups
hoop
hooper
hoops
hoot
hooted
hooter
hooting
hoots
hooves
hop
hope
hoped
hopeful
hopefully
hopefulness
hopefuls
hopeless
hopelessly
hopelessness
hopes
hoping
hopper
hoppers
hopping
hops
horde
hordes
horizon
horizons
horizontal
horizontally
hormone
hormones
horn
horned
hornet
hornets
horns
horny
horrendous
horrendously
horrible
horribleness
horribly
horrid
horridly
horrified
horrifies
horrify
horrifying
horror
horrors
horse
horseback
horseflesh
horsefly
horseman
horseplay
horsepower
horses
horseshoe
horseshoer
horticulture
hose
hoses
hospitable
hospitably
hospital
hospitality
hospitalize
hospitalized
hospitalizes
hospitalizing
hospitals
host
hostage
hostages
hosted
hostess
hostesses
hostile
hostilely
hostilities
hostility
hosting
hosts
hot
hotel
hotels
hotly
hotness
hotter
hottest
hound
hounded
hounding
hounds
hour
hourglass
hourly
hours
house
houseboat
housebroken
housed
houseflies
housefly
household
householder
householders
households
housekeeper
housekeepers
housekeeping
houses
housetop
housetops
housewife
housewifely
housewives
housework
housing
hovel
hovels
hover
hovered
hovering
hovers
how
however
howl
howled
howler
howling
howls
hub
hubris
hubs
huddle
huddled
huddling
hue
hues
hug
huge
hugely
hugeness
hugging
huh
hull
hulls
hum
human
humane
humanely
humaneness
humanitarian
humanities
humanity
humanly
humanness
humans
humble
humbled
humbleness
humbler
humblest
humbling
humbly
humbug
humerus
humid
humidification
humidified
humidifier
humidifiers
humidifies
humidify
humidifying
humidity
humidly
humiliate
humiliated
humiliates
humiliating
humiliation
humiliations
humility
hummed
humming
hummingbird
humor
humored
humorer
humorers
humoring
humorous
humorously
humorousness
humors
hump
humpback
humped
hums
hunch
hunched
hunches
hundred
hundredfold
hundreds
hundredth
hung
hunger
hungered
hungering
hungers
hungrier
hungriest
hungrily
hungry
hunk
hunks
hunt
hunted
hunters
hunting
hunts
huntsman
hurdle
hurl
hurled
hurler
hurlers
hurling
hurrah
hurricane
hurricanes
hurried
hurriedly
hurries
hurry
hurrying
hurt
hurting
hurtle
hurtling
hurts
husband
husbandry
husbands
hush
hushed
hushes
hushing
husk
husked
husker
huskiness
husking
husks
husky
hustle
hustled
hustler
hustles
hustling
hut
hutch
huts
hyacinth
hybrid
hydra
hydrant
hydraulic
hydro
hydrodynamic
hydrodynamics
hydrogen
hydrogens
hyena
hygiene
hymen
hymn
hymns
hyper
hyperbola
hyperbolic
hypertext
hyphen
hyphenate
hyphens
hypnosis
hypnotic
hypocrisies
hypocrisy
hypocrite
hypocrites
hypodermic
hypodermics
hypotheses
hypothesis
hypothesize
hypothesized
hypothesizer
hypothesizes
hypothesizing
hypothetical
hypothetically
hysteresis
hysterical
hysterically
ibex
ibid
ibis
ice
iceberg
icebergs
icebox
iced
ices
icicle
iciness
icing
icings
icon
iconoclasm
iconoclast
icons
icosahedra
icosahedral
icosahedron
icy
idea
ideal
idealism
idealistic
idealization
idealizations
idealize
idealized
idealizes
idealizing
ideally
ideals
ideas
idem
idempotency
idempotent
identical
identically
identifiable
identifiably
identification
identifications
identified
identifier
identifiers
identifies
identify
identifying
identities
identity
ideological
ideologically
ideology
idiocy
idiom
idiosyncrasies
idiosyncrasy
idiosyncratic
idiot
idiotic
idiots
idle
idled
idleness
idler
idlers
idles
idlest
idling
idly
idol
idolatry
idols
if
igloo
ignite
ignition
ignoble
ignominious
ignoramus
ignorance
ignorant
ignorantly
ignore
ignored
ignores
ignoring
ill
illegal
illegalities
illegality
illegally
illegitimate
illicit
illicitly
illiteracy
illiterate
illness
illnesses
illogical
illogically
ills
illuminate
illuminated
illuminates
illuminating
illumination
illuminations
illusion
illusions
illusive
illusively
illusory
illustrate
illustrated
illustrates
illustrating
illustration
illustrations
illustrative
illustratively
illustrator
illustrators
illustrious
illustriousness
illy
image
imagery
images
imaginable
imaginably
imaginary
imagination
imaginations
imaginative
imaginatively
imagine
imagined
imagines
imaging
imagining
imaginings
imbalance
imbalances
imbecile
imbibe
imitate
imitated
imitates
imitating
imitation
imitations
imitative
immaculate
immaculately
immaterial
immaterially
immature
immaturity
immediacies
immediacy
immediate
immediately
immemorial
immense
immensely
immerse
immersed
immerses
immersion
immigrant
immigrants
immigrate
immigrated
immigrates
immigrating
immigration
imminent
imminently
immoderate
immodest
immoral
immortal
immortality
immortally
immovability
immovable
immovably
immune
immunities
immunity
immunization
immutable
imp
impact
impacted
impacting
impaction
impactor
impactors
impacts
impair
impaired
impairing
impairs
impale
impart
imparted
impartial
impartially
imparts
impasse
impassive
impatience
impatient
impatiently
impeach
impeachable
impeached
impeachment
impeccable
impedance
impedances
impede
impeded
impedes
impediment
impediments
impeding
impel
impelled
impelling
impend
impending
impenetrability
impenetrable
impenetrably
imperative
imperatively
imperatives
imperceivable
imperceptible
imperfect
imperfection
imperfections
imperfectly
imperial
imperialism
imperialist
imperialists
imperil
imperiled
imperious
imperiously
impermanence
impermanent
impermeable
impermissible
impersonal
impersonally
impersonate
impersonated
impersonates
impersonating
impersonation
impersonations
impertinent
impertinently
impervious
imperviously
impetuous
impetuously
impetus
impinge
impinged
impinges
impinging
impious
implacable
implant
implanted
implanting
implants
implausible
implement
implementable
implementation
implementations
implemented
implementer
implementing
implementor
implementors
implements
implicant
implicants
implicate
implicated
implicates
implicating
implication
implications
implicit
implicitly
implicitness
implied
implies
implore
implored
imploring
imply
implying
impolite
import
importance
important
importantly
importation
imported
importer
importers
importing
imports
impose
imposed
imposes
imposing
imposition
impositions
impossibilities
impossibility
impossible
impossibly
impostor
impostors
impotence
impotency
impotent
impound
impoverish
impoverished
impoverishment
impracticable
impractical
impracticality
impractically
imprecise
imprecisely
imprecision
impregnable
impregnate
impress
impressed
impresser
impresses
impressible
impressing
impression
impressionable
impressionist
impressionistic
impressions
impressive
impressively
impressiveness
impressment
imprimatur
imprint
imprinted
imprinting
imprints
imprison
imprisoned
imprisoning
imprisonment
imprisonments
imprisons
improbability
improbable
impromptu
improper
improperly
impropriety
improve
improved
improvement
improvements
improves
improving
improvisation
improvisational
improvisations
improvise
improvised
improviser
improvisers
improvises
improvising
imprudent
imps
impudent
impudently
impugn
impulse
impulses
impulsion
impulsive
impunity
impure
impurities
impurity
impute
imputed
in
inability
inaccessible
inaccuracies
inaccuracy
inaccurate
inaction
inactivate
inactive
inactivity
inadequacies
inadequacy
inadequate
inadequately
inadequateness
inadmissibility
inadmissible
inadvertent
inadvertently
inadvisable
inalienable
inalterable
inane
inanimate
inanimately
inapplicable
inapproachable
inappropriate
inappropriateness
inasmuch
inattention
inaudible
inaugural
inaugurate
inaugurated
inaugurating
inauguration
inauspicious
inboard
inbound
inbreed
incalculable
incandescent
incantation
incapable
incapacitate
incapacitating
incarcerate
incarnation
incarnations
incendiaries
incendiary
incense
incensed
incenses
incentive
incentives
inception
incessant
incessantly
incest
incestuous
inch
inched
inches
inching
incidence
incident
incidental
incidentally
incidentals
incidents
incinerate
incipient
incisive
incite
incited
incitement
incites
inciting
inclement
inclination
inclinations
incline
inclined
inclines
inclining
inclose
inclosed
incloses
inclosing
include
included
includes
including
inclusion
inclusions
inclusive
inclusively
inclusiveness
incoherence
incoherent
incoherently
income
incomes
incoming
incommensurable
incommensurate
incommunicable
incomparable
incomparably
incompatibilities
incompatibility
incompatible
incompatibly
incompetence
incompetent
incompetents
incomplete
incompletely
incompleteness
incomprehensibility
incomprehensible
incomprehensibly
incomprehension
incompressible
incomputable
inconceivable
inconclusive
incongruity
incongruous
inconsequential
inconsequentially
inconsiderable
inconsiderate
inconsiderately
inconsiderateness
inconsistencies
inconsistency
inconsistent
inconsistently
inconspicuous
incontestable
incontrovertible
incontrovertibly
inconvenience
inconvenienced
inconveniences
inconveniencing
inconvenient
inconveniently
inconvertible
incorporate
incorporated
incorporates
incorporating
incorporation
incorrect
incorrectly
incorrectness
incorrigible
increase
increased
increases
increasing
increasingly
incredible
incredibly
incredulity
incredulous
incredulously
increment
incremental
incrementally
incremented
incrementer
incrementing
increments
incriminate
incubate
incubated
incubates
incubating
incubation
incubator
incubators
inculcate
incumbent
incur
incurable
incurred
incurring
incurs
incursion
indebted
indebtedness
indecent
indecipherable
indecision
indecisive
indeed
indefatigable
indefensible
indefinite
indefinitely
indefiniteness
indelible
indemnify
indemnity
indent
indentation
indentations
indented
indenting
indents
indenture
independence
independent
independently
indescribable
indestructible
indeterminacies
indeterminacy
indeterminate
indeterminately
index
indexable
indexed
indexes
indexing
indicate
indicated
indicates
indicating
indication
indications
indicative
indicator
indicators
indices
indict
indictment
indictments
indifference
indifferent
indifferently
indigenous
indigenously
indigenousness
indigestible
indigestion
indignant
indignantly
indignation
indignities
indignity
indigo
indirect
indirected
indirecting
indirection
indirections
indirectly
indirects
indiscreet
indiscretion
indiscriminate
indiscriminately
indispensability
indispensable
indispensably
indisputable
indistinct
indistinguishable
individual
individualism
individualistic
individuality
individualize
individualized
individualizes
individualizing
individually
individuals
indivisibility
indivisible
indoctrinate
indoctrinated
indoctrinates
indoctrinating
indoctrination
indolent
indolently
indomitable
indoor
indoors
indubitable
induce
induced
inducement
inducements
inducer
induces
inducing
induct
inductance
inductances
inducted
inductee
inducting
induction
inductions
inductive
inductively
inductor
inductors
inducts
indulge
indulged
indulgence
indulgences
indulgent
indulging
industrial
industrialism
industrialist
industrialists
industrialization
industrialized
industrially
industrials
industries
industrious
industriously
industriousness
industry
ineffective
ineffectively
ineffectiveness
ineffectual
inefficiencies
inefficiency
inefficient
inefficiently
inelegant
ineligible
inept
inequalities
inequality
inequitable
inequity
inert
inertia
inertial
inertly
inertness
inescapable
inescapably
inessential
inestimable
inevitabilities
inevitability
inevitable
inevitably
inexact
inexcusable
inexcusably
inexhaustible
inexorable
inexorably
inexpensive
inexpensively
inexperience
inexperienced
inexplicable
infallibility
infallible
infallibly
infamous
infamously
infamy
infancy
infant
infantile
infantry
infantryman
infantrymen
infants
infarct
infatuate
infeasible
infect
infected
infecting
infection
infections
infectious
infectiously
infective
infects
infer
inference
inferences
inferential
inferior
inferiority
inferiors
infernal
infernally
inferno
infernos
inferred
inferring
infers
infertile
infest
infested
infesting
infests
infidel
infidelity
infidels
infighting
infiltrate
infinite
infinitely
infiniteness
infinitesimal
infinitive
infinitives
infinitude
infinitum
infinity
infirm
infirmary
infirmity
infix
inflame
inflamed
inflammable
inflammation
inflammatory
inflatable
inflate
inflated
inflater
inflates
inflating
inflation
inflationary
inflexibility
inflexible
inflict
inflicted
inflicting
inflicts
inflow
influence
influenced
influences
influencing
influential
influentially
influenza
inform
informal
informality
informally
informant
informants
information
informational
informative
informatively
informed
informer
informers
informing
informs
infra
infrared
infrastructure
infrequent
infrequently
infringe
infringed
infringement
infringements
infringes
infringing
infuriate
infuriated
infuriates
infuriating
infuriation
infuse
infused
infuses
infusing
infusion
infusions
ingenious
ingeniously
ingeniousness
ingenuity
ingenuous
ingest
ingestion
inglorious
ingot
ingrate
ingratiate
ingratitude
ingredient
ingredients
ingrown
inhabit
inhabitable
inhabitance
inhabitant
inhabitants
inhabited
inhabiting
inhabits
inhale
inhaled
inhaler
inhales
inhaling
inhere
inherent
inherently
inheres
inherit
inheritable
inheritance
inheritances
inherited
inheriting
inheritor
inheritors
inheritress
inheritresses
inheritrices
inheritrix
inherits
inhibit
inhibited
inhibiting
inhibition
inhibitions
inhibitor
inhibitors
inhibitory
inhibits
inhomogeneities
inhomogeneity
inhomogeneous
inhospitable
inhuman
inhumane
inimical
inimitable
iniquities
iniquity
initial
initialed
initialing
initialization
initializations
initialize
initialized
initializer
initializers
initializes
initializing
initially
initials
initiate
initiated
initiates
initiating
initiation
initiations
initiative
initiatives
initiator
initiators
inject
injected
injecting
injection
injections
injective
injects
injudicious
injunction
injunctions
injure
injured
injures
injuries
injuring
injurious
injury
injustice
injustices
ink
inked
inker
inkers
inking
inkings
inkling
inklings
inks
inlaid
inland
inlay
inlet
inlets
inline
inmate
inmates
inn
innards
innate
innately
inner
innermost
inning
innings
innocence
innocent
innocently
innocents
innocuous
innocuously
innocuousness
innovate
innovation
innovations
innovative
inns
innuendo
innumerability
innumerable
innumerably
inoculate
inoperable
inoperative
inopportune
inordinate
inordinately
inorganic
input
inputs
inquest
inquire
inquired
inquirer
inquirers
inquires
inquiries
inquiring
inquiry
inquisition
inquisitions
inquisitive
inquisitively
inquisitiveness
inroad
inroads
insane
insanely
insanity
insatiable
inscribe
inscribed
inscribes
inscribing
inscription
inscriptions
inscrutable
insect
insecticide
insects
insecure
insecurely
inseminate
insensible
insensitive
insensitively
insensitivity
inseparable
insert
inserted
inserting
insertion
insertions
inserts
inset
inside
insider
insiders
insides
insidious
insidiously
insidiousness
insight
insightful
insights
insignia
insignificance
insignificant
insincere
insincerity
insinuate
insinuated
insinuates
insinuating
insinuation
insinuations
insipid
insist
insisted
insistence
insistent
insistently
insisting
insists
insofar
insolence
insolent
insolently
insoluble
insolvable
insolvent
insomnia
insomniac
inspect
inspected
inspecting
inspection
inspections
inspector
inspectors
inspects
inspiration
inspirations
inspire
inspired
inspirer
inspires
inspiring
instabilities
instability
install
installation
installations
installed
installer
installers
installing
installment
installments
installs
instance
instances
instant
instantaneous
instantaneously
instanter
instantiate
instantiated
instantiates
instantiating
instantiation
instantiations
instantly
instants
instead
instigate
instigated
instigates
instigating
instigator
instigators
instill
instinct
instinctive
instinctively
instincts
instinctual
institute
instituted
instituter
instituters
institutes
instituting
institution
institutional
institutionalize
institutionalized
institutionalizes
institutionalizing
institutionally
institutions
instruct
instructed
instructing
instruction
instructional
instructions
instructive
instructively
instructor
instructors
instructs
instrument
instrumental
instrumentalist
instrumentalists
instrumentally
instrumentals
instrumentation
instrumented
instrumenting
instruments
insubordinate
insufferable
insufficient
insufficiently
insular
insulate
insulated
insulates
insulating
insulation
insulator
insulators
insulin
insult
insulted
insulting
insults
insuperable
insupportable
insurance
insure
insured
insurer
insurers
insures
insurgent
insurgents
insuring
insurmountable
insurrection
insurrections
intact
intangible
intangibles
integer
integers
integrable
integral
integrals
integrand
integrate
integrated
integrates
integrating
integration
integrations
integrative
integrity
intellect
intellects
intellectual
intellectually
intellectuals
intelligence
intelligent
intelligently
intelligentsia
intelligibility
intelligible
intelligibly
intemperate
intend
intended
intending
intends
intense
intensely
intensification
intensified
intensifier
intensifiers
intensifies
intensify
intensifying
intensities
intensity
intensive
intensively
intent
intention
intentional
intentionally
intentioned
intentions
intently
intentness
intents
inter
interact
interacted
interacting
interaction
interactions
interactive
interactively
interactivity
interacts
intercept
intercepted
intercepting
interception
interceptor
intercepts
interchange
interchangeability
interchangeable
interchangeably
interchanged
interchanger
interchanges
interchanging
interchangings
interchannel
intercity
intercom
intercommunicate
intercommunicated
intercommunicates
intercommunicating
intercommunication
interconnect
interconnected
interconnecting
interconnection
interconnections
interconnects
intercontinental
intercourse
interdependence
interdependencies
interdependency
interdependent
interdict
interdiction
interdisciplinary
interest
interested
interesting
interestingly
interests
interface
interfaced
interfacer
interfaces
interfacing
interfere
interfered
interference
interferences
interferes
interfering
interferingly
interferometer
interferometric
interferometry
interframe
intergroup
interim
interior
interiors
interject
interlace
interlaced
interlaces
interlacing
interleave
interleaved
interleaves
interleaving
interlink
interlinked
interlinks
interlisp
intermediary
intermediate
intermediates
interminable
intermingle
intermingled
intermingles
intermingling
intermission
intermittent
intermittently
intermix
intermixed
intermodule
intern
internal
internalize
internalized
internalizes
internalizing
internally
internals
international
internationality
internationally
interned
internetwork
interning
interns
internship
interoffice
interpersonal
interplay
interpolate
interpolated
interpolates
interpolating
interpolation
interpolations
interpose
interposed
interposes
interposing
interpret
interpretable
interpretation
interpretations
interpreted
interpreter
interpreters
interpreting
interpretive
interpretively
interprets
interprocess
interrelate
interrelated
interrelates
interrelating
interrelation
interrelations
interrelationship
interrelationships
interrogate
interrogated
interrogates
interrogating
interrogation
interrogations
interrogative
interrupt
interrupted
interruptible
interrupting
interruption
interruptions
interruptive
interrupts
intersect
intersected
intersecting
intersection
intersections
intersects
intersperse
interspersed
intersperses
interspersing
interspersion
interstage
interstate
intertwine
intertwined
intertwines
intertwining
interval
intervals
intervene
intervened
intervenes
intervening
intervention
interventions
interview
interviewed
interviewee
interviewer
interviewers
interviewing
interviews
interwoven
intestate
intestinal
intestine
intestines
intimacy
intimate
intimated
intimately
intimating
intimation
intimations
intimidate
intimidated
intimidates
intimidating
intimidation
into
intolerable
intolerably
intolerance
intolerant
intonation
intonations
intone
intoxicant
intoxicate
intoxicated
intoxicating
intoxication
intractability
intractable
intractably
intragroup
intraline
intramural
intramuscular
intransigent
intransitive
intransitively
intraoffice
intraprocess
intrastate
intravenous
intrepid
intricacies
intricacy
intricate
intricately
intrigue
intrigued
intrigues
intriguing
intrinsic
intrinsically
introduce
introduced
introduces
introducing
introduction
introductions
introductory
introspect
introspection
introspections
introspective
introvert
introverted
intrude
intruded
intruder
intruders
intrudes
intruding
intrusion
intrusions
intrust
intubate
intubated
intubates
intubation
intuition
intuitionist
intuitions
intuitive
intuitively
inundate
invade
invaded
invader
invaders
invades
invading
invalid
invalidate
invalidated
invalidates
invalidating
invalidation
invalidations
invalidities
invalidity
invalidly
invalids
invaluable
invariable
invariably
invariance
invariant
invariantly
invariants
invasion
invasions
invective
invent
invented
inventing
invention
inventions
inventive
inventively
inventiveness
inventor
inventories
inventors
inventory
invents
inverse
inversely
inverses
inversion
inversions
invert
invertebrate
invertebrates
inverted
inverter
inverters
invertible
inverting
inverts
invest
invested
investigate
investigated
investigates
investigating
investigation
investigations
investigative
investigator
investigators
investigatory
investing
investment
investments
investor
investors
invests
inveterate
invigorate
invincible
invisibility
invisible
invisibly
invitation
invitations
invite
invited
invites
inviting
invocable
invocation
invocations
invoice
invoiced
invoices
invoicing
invoke
invoked
invoker
invokes
invoking
involuntarily
involuntary
involve
involved
involvement
involvements
involves
involving
inward
inwardly
inwardness
inwards
iodine
ion
ionosphere
ionospheric
ions
iota
irate
irately
irateness
ire
ires
iris
irk
irked
irking
irks
irksome
iron
ironed
ironic
ironical
ironically
ironies
ironing
ironings
irons
irony
irradiate
irrational
irrationally
irrationals
irreconcilable
irrecoverable
irreducible
irreducibly
irreflexive
irrefutable
irregular
irregularities
irregularity
irregularly
irregulars
irrelevance
irrelevances
irrelevant
irrelevantly
irreplaceable
irrepressible
irreproducibility
irreproducible
irresistible
irrespective
irrespectively
irresponsible
irresponsibly
irretrievably
irreverent
irreversibility
irreversible
irreversibly
irrevocable
irrevocably
irrigate
irrigated
irrigates
irrigating
irrigation
irritable
irritant
irritate
irritated
irritates
irritating
irritation
irritations
is
island
islander
islanders
islands
isle
isles
islet
islets
isolate
isolated
isolates
isolating
isolation
isolations
isometric
isomorphic
isomorphically
isomorphism
isomorphisms
isotope
isotopes
issuance
issue
issued
issuer
issuers
issues
issuing
isthmus
it
italic
italicize
italicized
italics
itch
itches
itching
item
itemization
itemizations
itemize
itemized
itemizes
itemizing
items
iterate
iterated
iterates
iterating
iteration
iterations
iterative
iteratively
iterator
iterators
itineraries
itinerary
its
itself
ivies
ivory
ivy
jab
jabbed
jabbing
jabs
jack
jackass
jacket
jacketed
jackets
jacking
jackknife
jackpot
jade
jaded
jaguar
jail
jailed
jailer
jailers
jailing
jails
jam
jammed
jamming
jams
janitor
janitors
jar
jargon
jarred
jarring
jarringly
jars
jaundice
jaunt
jauntiness
jaunts
jaunty
javelin
javelins
jaw
jawbone
jaws
jay
jazz
jazzy
jealous
jealousies
jealously
jealousy
jean
jeans
jeep
jeeps
jeer
jeers
jellies
jelly
jellyfish
jenny
jeopardize
jeopardized
jeopardizes
jeopardizing
jeopardy
jerk
jerked
jerkiness
jerking
jerkings
jerks
jerky
jersey
jerseys
jest
jested
jester
jesting
jests
jet
jetliner
jets
jetted
jetting
jewel
jeweled
jeweler
jewelries
jewelry
jewels
jiffy
jig
jigs
jigsaw
jingle
jingled
jingling
jitter
jitterbug
jittery
job
jobs
jockey
jockstrap
jocund
jog
jogging
jogs
join
joined
joiner
joiners
joining
joins
joint
jointly
joints
joke
joked
joker
jokers
jokes
joking
jokingly
jolly
jolt
jolted
jolting
jolts
jonquil
jostle
jostled
jostles
jostling
jot
jots
jotted
jotting
joule
journal
journalism
journalist
journalists
journalize
journalized
journalizes
journalizing
journals
journey
journeyed
journeying
journeyings
journeyman
journeymen
journeys
joust
jousted
jousting
jousts
jovial
joy
joyful
joyfully
joyous
joyously
joyousness
joyride
joys
joystick
jubilee
judge
judged
judges
judging
judgment
judgments
judicial
judiciary
judicious
judiciously
judo
jug
juggle
juggler
jugglers
juggles
juggling
jugs
juice
juices
juiciest
juicy
jumble
jumbled
jumbles
jumbo
jump
jumped
jumper
jumpers
jumping
jumps
jumpy
junction
junctions
juncture
junctures
jungle
jungles
junior
juniors
juniper
junk
junker
junkers
junks
junky
junta
jure
juries
jurisdiction
jurisdictions
jurisprudence
jurist
juror
jurors
jury
just
justice
justices
justifiable
justifiably
justification
justifications
justified
justifier
justifiers
justifies
justify
justifying
justly
justness
jut
jutting
juvenile
juveniles
juxtapose
juxtaposed
juxtaposes
juxtaposing
kangaroo
kanji
kappa
karate
keel
keeled
keeling
keels
keen
keener
keenest
keenly
keenness
keep
keeper
keepers
keeping
keeps
ken
kennel
kennels
kept
kerchief
kerchiefs
kern
kernel
kernels
kerosene
ketchup
kettle
kettles
key
keyboard
keyboards
keyed
keyhole
keying
keynote
keypad
keypads
keys
keystroke
keystrokes
keyword
keywords
kick
kicked
kicker
kickers
kicking
kickoff
kicks
kid
kidded
kiddie
kidding
kidnap
kidnapper
kidnappers
kidnapping
kidnappings
kidnaps
kidney
kidneys
kids
kill
killed
killer
killers
killing
killingly
killings
killjoy
kills
kilobit
kilobits
kiloblock
kilobyte
kilobytes
kilogram
kilograms
kilohertz
kilohm
kilojoule
kilometer
kilometers
kiloton
kilovolt
kilowatt
kiloword
kimono
kin
kind
kinder
kindergarten
kindest
kindhearted
kindle
kindled
kindles
kindling
kindly
kindness
kindred
kinds
kinetic
king
kingdom
kingdoms
kingly
kingpin
kings
kink
kinky
kinship
kinsman
kiosk
kiss
kissed
kisser
kissers
kisses
kissing
kit
kitchen
kitchenette
kitchens
kite
kited
kites
kiting
kits
kitten
kittenish
kittens
kitty
klaxon
kludge
kludges
klystron
knack
knapsack
knapsacks
knave
knaves
knead
kneads
knee
kneecap
kneed
kneeing
kneel
kneeled
kneeling
kneels
knees
knell
knells
knelt
knew
knife
knifed
knifes
knifing
knight
knighted
knighthood
knighting
knightly
knights
knit
knits
knives
knob
knobs
knock
knockdown
knocked
knocker
knockers
knocking
knockout
knocks
knoll
knolls
knot
knots
knotted
knotting
know
knowable
knower
knowhow
knowing
knowingly
knowledge
knowledgeable
known
knows
knuckle
knuckled
knuckles
koala
kosher
kudo
lab
label
labeled
labeling
labelled
labeller
labellers
labelling
labels
labor
laboratories
laboratory
labored
laborer
laborers
laboring
laborings
laborious
laboriously
labors
labs
labyrinth
labyrinths
lace
laced
lacerate
lacerated
lacerates
lacerating
laceration
lacerations
laces
lacing
lack
lacked
lackey
lacking
lacks
lacquer
lacquered
lacquers
lacrosse
lacy
lad
ladder
laden
ladies
lading
ladle
lads
lady
ladylike
lag
lager
lagers
lagoon
lagoons
lags
laid
lain
lair
lairs
laissez
lake
lakes
lamb
lambda
lambdas
lambert
lambs
lame
lamed
lamely
lameness
lament
lamentable
lamentation
lamentations
lamented
lamenting
laments
lames
laminar
laming
lamp
lamplight
lampoon
lamprey
lamps
lance
lanced
lancer
lances
land
landed
lander
landers
landfill
landing
landings
landladies
landlady
landlord
landlords
landmark
landmarks
landowner
landowners
lands
landscape
landscaped
landscapes
landscaping
landslide
lane
lanes
language
languages
languid
languidly
languidness
languish
languished
languishes
languishing
lantern
lanterns
lap
lapel
lapels
lapping
laps
lapse
lapsed
lapses
lapsing
lard
larder
large
largely
largeness
larger
largest
lark
larks
larva
larvae
larynx
lascivious
laser
lasers
lash
lashed
lashes
lashing
lashings
lass
lasses
lasso
last
lasted
lasting
lastly
lasts
latch
latched
latches
latching
late
lately
latency
lateness
latent
later
lateral
laterally
latest
lathe
latitude
latitudes
latrine
latrines
latter
latterly
lattice
lattices
laudable
laugh
laughable
laughably
laughed
laughing
laughingly
laughingstock
laughs
laughter
launch
launched
launcher
launches
launching
launchings
launder
laundered
launderer
laundering
launderings
launders
laundry
laureate
laurel
laurels
lava
lavatories
lavatory
lavender
lavish
lavished
lavishing
lavishly
law
lawbreaker
lawful
lawfully
lawgiver
lawless
lawlessness
lawn
lawns
laws
lawsuit
lawsuits
lawyer
lawyers
lax
laxative
lay
layer
layered
layering
layers
laying
layman
laymen
layoff
layoffs
layout
layouts
lays
lazed
lazier
laziest
lazily
laziness
lazing
lazy
lazybones
lead
leaded
leaden
leader
leaders
leadership
leaderships
leading
leadings
leads
leaf
leafed
leafiest
leafing
leafless
leaflet
leaflets
leafy
league
leagued
leaguer
leaguers
leagues
leak
leakage
leakages
leaked
leaking
leaks
leaky
lean
leaned
leaner
leanest
leaning
leanness
leans
leap
leaped
leapfrog
leaping
leaps
leapt
learn
learned
learner
learners
learning
learns
lease
leased
leases
leash
leashes
leasing
least
leather
leathered
leathern
leatherneck
leathers
leave
leaved
leaven
leavened
leavening
leaves
leaving
leavings
lechery
lecture
lectured
lecturer
lecturers
lectures
lecturing
led
ledge
ledger
ledgers
ledges
lee
leech
leeches
leek
leer
leery
lees
leeward
leeway
left
leftist
leftists
leftmost
leftover
leftovers
leftward
leg
legacies
legacy
legal
legality
legalization
legalize
legalized
legalizes
legalizing
legally
legend
legendary
legends
legged
leggings
legibility
legible
legibly
legion
legions
legislate
legislated
legislates
legislating
legislation
legislative
legislator
legislators
legislature
legislatures
legitimacy
legitimate
legitimately
legs
legume
leisure
leisurely
lemma
lemmas
lemming
lemmings
lemon
lemonade
lemons
lend
lender
lenders
lending
lends
length
lengthen
lengthened
lengthening
lengthens
lengthly
lengths
lengthwise
lengthy
leniency
lenient
leniently
lens
lenses
lent
lentil
lentils
leopard
leopards
leper
leprosy
less
lessen
lessened
lessening
lessens
lesser
lesson
lessons
lessor
lest
let
lethal
lets
letter
lettered
letterer
letterhead
lettering
letters
letting
lettuce
leukemia
levee
levees
level
leveled
leveler
leveling
levelled
leveller
levellest
levelling
levelly
levelness
levels
lever
leverage
levers
levied
levies
levity
levy
levying
lewd
lewdly
lewdness
lexical
lexically
lexicographic
lexicographical
lexicographically
lexicon
lexicons
liabilities
liability
liable
liaison
liaisons
liar
liars
libel
libelous
liberal
liberalize
liberalized
liberalizes
liberalizing
liberally
liberals
liberate
liberated
liberates
liberating
liberation
liberator
liberators
libertarian
liberties
liberty
libido
librarian
librarians
libraries
library
libretto
lice
license
licensed
licensee
licenses
licensing
licensor
licentious
lichen
lichens
lick
licked
licking
licks
licorice
lid
lids
lie
lied
liege
lien
liens
lies
lieu
lieutenant
lieutenants
life
lifeblood
lifeboat
lifeguard
lifeless
lifelessness
lifelike
lifelong
lifer
lifespan
lifestyle
lifestyles
lifetime
lifetimes
lift
lifted
lifter
lifters
lifting
lifts
ligament
ligature
light
lighted
lighten
lightens
lighter
lighters
lightest
lightface
lighthearted
lighthouse
lighthouses
lighting
lightly
lightness
lightning
lightnings
lights
lightweight
like
liked
likelier
likeliest
likelihood
likelihoods
likeliness
likely
liken
likened
likeness
likenesses
likening
likens
likes
likewise
liking
lilac
lilacs
lilies
lily
limb
limber
limbo
limbs
lime
limelight
limes
limestone
limit
limitability
limitably
limitation
limitations
limited
limiter
limiters
limiting
limitless
limits
limousine
limp
limped
limping
limply
limpness
limps
linden
line
linear
linearities
linearity
linearizable
linearize
linearized
linearizes
linearizing
linearly
lined
linen
linens
liner
liners
lines
lineup
linger
lingered
lingerie
lingering
lingers
lingo
lingua
linguist
linguistic
linguistically
linguistics
linguists
lining
linings
link
linkage
linkages
linked
linker
linkers
linking
links
linoleum
linseed
lint
lion
lioness
lionesses
lions
lip
lips
lipstick
liquid
liquidate
liquidation
liquidations
liquidity
liquids
liquor
liquors
lisp
lisped
lisping
lisps
list
listed
listen
listened
listener
listeners
listening
listens
listers
listing
listings
listless
lists
lit
litany
liter
literacy
literal
literally
literalness
literals
literary
literate
literature
literatures
liters
lithe
lithograph
lithography
litigant
litigate
litigation
litigious
litmus
litter
litterbug
littered
littering
litters
little
littleness
littler
littlest
livable
livably
live
lived
livelihood
lively
liveness
liver
liveried
livers
livery
lives
livestock
livid
living
lizard
lizards
load
loaded
loader
loaders
loading
loadings
loads
loaf
loafed
loafer
loan
loaned
loaning
loans
loath
loathe
loathed
loathing
loathly
loathsome
loaves
lobbied
lobbies
lobby
lobbying
lobe
lobes
lobster
lobsters
local
localities
locality
localization
localize
localized
localizes
localizing
locally
locals
locate
located
locates
locating
location
locations
locative
locatives
locator
locators
loci
lock
locked
locker
lockers
locking
lockings
lockout
lockouts
locks
locksmith
lockstep
lockup
lockups
locomotion
locomotive
locomotives
locus
locust
locusts
lodge
lodged
lodger
lodges
lodging
lodgings
loft
loftiness
lofts
lofty
logarithm
logarithmic
logarithmically
logarithms
logged
logger
loggers
logging
logic
logical
logically
logician
logicians
logics
login
logins
logistic
logistics
logjam
logo
logs
loin
loincloth
loins
loiter
loitered
loiterer
loitering
loiters
lone
lonelier
loneliest
loneliness
lonely
loner
loners
lonesome
long
longed
longer
longest
longevity
longhand
longing
longings
longitude
longitudes
longs
longstanding
look
lookahead
looked
looker
lookers
looking
lookout
looks
lookup
lookups
loom
loomed
looming
looms
loon
loop
looped
loophole
loopholes
looping
loops
loose
loosed
looseleaf
loosely
loosen
loosened
looseness
loosening
loosens
looser
looses
loosest
loosing
loot
looted
looter
looting
loots
lopsided
lord
lordly
lords
lordship
lore
lorry
lose
loser
losers
loses
losing
loss
losses
lossier
lossiest
lossy
lost
lot
lotion
lots
lottery
lotus
loud
louder
loudest
loudly
loudness
loudspeaker
loudspeakers
lounge
lounged
lounges
lounging
louse
lousy
lout
lovable
lovably
love
loved
lovelier
lovelies
loveliest
loveliness
lovelorn
lovely
lover
lovers
loves
loving
lovingly
low
lower
lowered
lowering
lowers
lowest
lowland
lowlands
lowliest
lowly
lowness
lows
loyal
loyally
loyalties
loyalty
lubricant
lubricate
lubrication
lucid
luck
lucked
luckier
luckiest
luckily
luckless
lucks
lucky
lucrative
ludicrous
ludicrously
ludicrousness
luggage
lukewarm
lull
lullaby
lulled
lulls
lumber
lumbered
lumbering
luminous
luminously
lummox
lump
lumped
lumping
lumps
lumpy
lunar
lunatic
lunch
lunched
luncheon
luncheons
lunches
lunching
lung
lunged
lungs
lurch
lurched
lurches
lurching
lure
lured
lures
luring
lurk
lurked
lurking
lurks
luscious
lusciously
lusciousness
lush
lust
luster
lustful
lustily
lustiness
lustrous
lusts
lusty
lute
lutes
luxuriant
luxuriantly
luxuries
luxurious
luxuriously
luxury
lying
lymph
lynch
lynched
lyncher
lynches
lynx
lynxes
lyre
lyric
lyrics
mace
maced
maces
machination
machine
machined
machinelike
machinery
machines
machining
macho
macintosh
mackerel
macro
macroeconomics
macromolecule
macromolecules
macrophage
macros
macroscopic
mad
madam
madden
maddening
madder
maddest
made
madhouse
madly
madman
madmen
madness
madras
maestro
magazine
magazines
magenta
maggot
maggots
magic
magical
magically
magician
magicians
magistrate
magistrates
magna
magnesium
magnet
magnetic
magnetically
magnetism
magnetisms
magnetizable
magnetized
magneto
magnification
magnificence
magnificent
magnificently
magnified
magnifier
magnifies
magnify
magnifying
magnitude
magnitudes
magnolia
magnum
magpie
mahogany
maid
maiden
maidens
maids
mail
mailable
mailbox
mailboxes
mailed
mailer
mailing
mailings
mailman
mailmen
mails
maim
maimed
maiming
maims
main
mainframe
mainframes
mainland
mainline
mainly
mains
mainstay
mainstream
maintain
maintainability
maintainable
maintained
maintainer
maintainers
maintaining
maintains
maintenance
maintenances
maize
majestic
majesties
majesty
major
majored
majoring
majorities
majority
majors
makable
make
maker
makers
makes
makeshift
makeup
makeups
making
makings
maladies
malady
malaria
malcontent
male
malefactor
malefactors
maleness
males
malevolent
malformed
malfunction
malfunctioned
malfunctioning
malfunctions
malice
malicious
maliciously
maliciousness
malign
malignant
malignantly
mall
mallard
mallet
mallets
malnutrition
malpractice
malt
malted
malts
mama
mamma
mammal
mammalian
mammals
mammas
mammoth
man
manage
manageable
manageableness
managed
management
managements
manager
managerial
managers
manages
managing
mandarin
mandate
mandated
mandates
mandating
mandatory
mandible
mane
manes
maneuver
maneuvered
maneuvering
maneuvers
manger
mangers
mangle
mangled
mangler
mangles
mangling
manhole
manhood
mania
maniac
maniacal
maniacs
manic
manicure
manicured
manicures
manicuring
manifest
manifestation
manifestations
manifested
manifesting
manifestly
manifests
manifold
manifolds
manipulability
manipulable
manipulatable
manipulate
manipulated
manipulates
manipulating
manipulation
manipulations
manipulative
manipulator
manipulators
manipulatory
mankind
manly
manned
manner
mannered
mannerly
manners
manning
manometer
manometers
manor
manors
manpower
mansion
mansions
manslaughter
mantel
mantels
mantis
mantissa
mantissas
mantle
mantlepiece
mantles
manual
manually
manuals
manufacture
manufactured
manufacturer
manufacturers
manufactures
manufacturing
manure
manuscript
manuscripts
many
map
maple
maples
mappable
mapped
mapping
mappings
maps
marathon
marble
marbles
marbling
march
marched
marcher
marches
marching
mare
mares
margarine
margin
marginal
marginally
margins
marigold
marijuana
marina
marinade
marinate
marine
mariner
marines
marionette
marital
maritime
mark
markable
marked
markedly
marker
markers
market
marketability
marketable
marketed
marketing
marketings
marketplace
marketplaces
markets
marking
markings
marmalade
marmot
maroon
marquis
marriage
marriageable
marriages
married
marries
marrow
marry
marrying
marsh
marshal
marshaled
marshaling
marshals
marshes
marshmallow
mart
marten
martial
martingale
martini
marts
martyr
martyrdom
martyrs
marvel
marveled
marvelled
marvelling
marvelous
marvelously
marvelousness
marvels
mascara
masculine
masculinely
masculinity
mash
mashed
mashes
mashing
mask
maskable
masked
masker
masking
maskings
masks
masochist
masochists
mason
masonry
masons
masquerade
masquerader
masquerades
masquerading
mass
massacre
massacred
massacres
massage
massages
massaging
massed
masses
massing
massive
mast
masted
master
mastered
masterful
masterfully
mastering
masterings
masterly
mastermind
masterpiece
masterpieces
masters
mastery
mastodon
masts
masturbate
masturbated
masturbates
masturbating
masturbation
mat
match
matchable
matched
matcher
matchers
matches
matching
matchings
matchless
mate
mated
mater
material
materialist
materialize
materialized
materializes
materializing
materially
materials
maternal
maternally
maternity
mates
math
mathematical
mathematically
mathematician
mathematicians
mathematics
mating
matings
matriarch
matriarchal
matrices
matriculate
matriculation
matrimonial
matrimony
matrix
matroid
matron
matronly
mats
matted
matter
mattered
matters
mattress
mattresses
maturation
mature
matured
maturely
matures
maturing
maturities
maturity
maul
mausoleum
maverick
maxim
maxima
maximal
maximally
maximize
maximized
maximizer
maximizers
maximizes
maximizing
maxims
maximum
maximums
maybe
mayhap
mayhem
mayonnaise
mayor
mayoral
mayors
maze
mazes
me
mead
meadow
meadows
meager
meagerly
meagerness
meal
meals
mealtime
mealy
mean
meander
meandered
meandering
meanders
meaner
meanest
meaning
meaningful
meaningfully
meaningfulness
meaningless
meaninglessly
meaninglessness
meanings
meanly
meanness
means
meant
meantime
meanwhile
measle
measles
measurable
measurably
measure
measured
measurement
measurements
measurer
measures
measuring
meat
meats
meaty
mechanic
mechanical
mechanically
mechanics
mechanism
mechanisms
mechanization
mechanizations
mechanize
mechanized
mechanizes
mechanizing
medal
medallion
medallions
medals
meddle
meddled
meddler
meddles
meddling
media
median
medians
mediate
mediated
mediates
mediating
mediation
mediations
mediator
medic
medical
medically
medicinal
medicinally
medicine
medicines
medics
medieval
mediocre
mediocrity
meditate
meditated
meditates
meditating
meditation
meditations
meditative
medium
mediums
medley
meek
meeker
meekest
meekly
meekness
meet
meeting
meetinghouse
meetings
meets
megabaud
megabit
megabits
megabyte
megabytes
megahertz
megalomania
megaton
megavolt
megawatt
megaword
megawords
megohm
melancholy
mellow
mellowed
mellowing
mellowness
mellows
melodies
melodious
melodiously
melodiousness
melodrama
melodramas
melodramatic
melody
melon
melons
melt
melted
melting
meltingly
melts
member
members
membership
memberships
membrane
memento
memo
memoir
memoirs
memorabilia
memorable
memorableness
memoranda
memorandum
memorial
memorially
memorials
memories
memorization
memorize
memorized
memorizer
memorizes
memorizing
memory
memoryless
memos
men
menace
menaced
menacing
menagerie
menarche
mend
mendacious
mendacity
mended
mender
mending
mends
menial
menials
mens
menstruate
mensurable
mensuration
mental
mentalities
mentality
mentally
mention
mentionable
mentioned
mentioner
mentioners
mentioning
mentions
mentor
mentors
menu
menus
mercantile
mercenaries
mercenariness
mercenary
merchandise
merchandiser
merchandising
merchant
merchants
merciful
mercifully
merciless
mercilessly
mercurial
mercury
mercy
mere
merely
merest
merge
merged
merger
mergers
merges
merging
meridian
meringue
merit
merited
meriting
meritorious
meritoriously
meritoriousness
merits
mermaid
merriest
merrily
merriment
merry
mescaline
mesh
meson
mesquite
mess
message
messages
messed
messenger
messengers
messes
messiahs
messier
messiest
messily
messiness
messing
messy
met
meta
metabolic
metabolism
metacircular
metacircularity
metal
metalanguage
metallic
metallization
metallizations
metallurgy
metals
metamathematical
metamorphosis
metaphor
metaphorical
metaphorically
metaphors
metaphysical
metaphysically
metaphysics
metavariable
mete
meted
meteor
meteoric
meteorite
meteoritic
meteorology
meteors
meter
metering
meters
metes
methane
method
methodical
methodically
methodicalness
methodists
methodological
methodologically
methodologies
methodologists
methodology
methods
meticulously
meting
metric
metrical
metrics
metro
metronome
metropolis
metropolitan
mets
mettle
mettlesome
mew
mewed
mews
miasma
mica
mice
micro
microarchitects
microarchitecture
microarchitectures
microbial
microbicidal
microbicide
microcode
microcoded
microcodes
microcoding
microcomputer
microcomputers
microcosm
microcycle
microcycles
microeconomics
microelectronics
microfilm
microfilms
microgramming
microinstruction
microinstructions
microjump
microjumps
microlevel
micron
microoperations
microphone
microphones
microphoning
microprocedure
microprocedures
microprocessing
microprocessor
microprocessors
microprogram
microprogrammable
microprogrammed
microprogrammer
microprogramming
microprograms
micros
microscope
microscopes
microscopic
microscopy
microsecond
microseconds
microstore
microsystems
microwave
microwaves
microword
microwords
mid
midday
middle
middleman
middlemen
middles
middling
midget
midnight
midnights
midpoint
midpoints
midrange
midscale
midsection
midshipman
midshipmen
midst
midstream
midsts
midsummer
midway
midweek
midwife
midwinter
midwives
mien
might
mightier
mightiest
mightily
mightiness
mighty
migrant
migrate
migrated
migrates
migrating
migration
migrations
migratory
mike
mild
milder
mildest
mildew
mildly
mildness
mile
mileage
milestone
milestones
militant
militantly
militarily
militarism
military
militia
milk
milked
milker
milkers
milkiness
milking
milkmaid
milkmaids
milks
milky
mill
milled
millennium
miller
millet
milliammeter
milliampere
millijoule
millimeter
millimeters
millinery
milling
million
millionaire
millionaires
millions
millionth
millipede
millipedes
millisecond
milliseconds
millivolt
millivoltmeter
milliwatt
millstone
millstones
mimeograph
mimic
mimicked
mimicking
mimics
minaret
mince
minced
mincemeat
minces
mincing
mind
minded
mindful
mindfully
mindfulness
minding
mindless
mindlessly
minds
mine
mined
minefield
miner
mineral
minerals
miners
mines
minesweeper
mingle
mingled
mingles
mingling
mini
miniature
miniatures
miniaturization
miniaturize
miniaturized
miniaturizes
miniaturizing
minicomputer
minicomputers
minima
minimal
minimally
minimax
minimization
minimizations
minimize
minimized
minimizer
minimizers
minimizes
minimizing
minimum
mining
minion
minis
minister
ministered
ministering
ministers
ministries
ministry
mink
minks
minnow
minnows
minor
minoring
minorities
minority
minors
minstrel
minstrels
mint
minted
minter
minting
mints
minuend
minuet
minus
minuscule
minute
minutely
minuteman
minutemen
minuteness
minuter
minutes
miracle
miracles
miraculous
miraculously
mirage
mire
mired
mires
mirror
mirrored
mirroring
mirrors
mirth
misanthrope
misbehaving
miscalculation
miscalculations
miscarriage
miscarry
miscegenation
miscellaneous
miscellaneously
miscellaneousness
mischief
mischievous
mischievously
mischievousness
misconception
misconceptions
misconduct
misconstrue
misconstrued
misconstrues
misdemeanors
miser
miserable
miserableness
miserably
miseries
miserly
misers
misery
misfit
misfits
misfortune
misfortunes
misgiving
misgivings
misguided
mishap
mishaps
misinformed
misjudged
misjudgment
mislead
misleading
misleads
misled
mismanagement
mismatch
mismatched
mismatches
mismatching
misnomer
misplace
misplaced
misplaces
misplacing
mispronunciation
misrepresentation
misrepresentations
miss
missed
misses
misshapen
missile
missiles
missing
mission
missionaries
missionary
missioner
missions
missive
misspell
misspelled
misspelling
misspellings
misspells
mist
mistakable
mistake
mistaken
mistakenly
mistakes
mistaking
misted
mister
misters
mistiness
misting
mistletoe
mistress
mistrust
mistrusted
mists
misty
mistype
mistyped
mistypes
mistyping
misunderstand
misunderstander
misunderstanders
misunderstanding
misunderstandings
misunderstood
misuse
misused
misuses
misusing
miter
mitigate
mitigated
mitigates
mitigating
mitigation
mitigative
mitten
mittens
mix
mixed
mixer
mixers
mixes
mixing
mixture
mixtures
mixup
mnemonic
mnemonically
mnemonics
moan
moaned
moans
moat
moats
mob
mobile
mobility
mobs
mobster
moccasin
moccasins
mock
mocked
mocker
mockery
mocking
mockingbird
mocks
mockup
modal
modalities
modality
modally
mode
model
modeled
modeling
modelings
models
modem
modems
moderate
moderated
moderately
moderateness
moderates
moderating
moderation
modern
modernity
modernize
modernized
modernizer
modernizing
modernly
modernness
moderns
modes
modest
modestly
modesty
modicum
modifiability
modifiable
modification
modifications
modified
modifier
modifiers
modifies
modify
modifying
modular
modularity
modularization
modularize
modularized
modularizes
modularizing
modularly
modulate
modulated
modulates
modulating
modulation
modulations
modulator
modulators
module
modules
moduli
modulo
modulus
modus
moist
moisten
moistly
moistness
moisture
molar
molasses
mold
molded
molder
molding
molds
mole
molecular
molecule
molecules
molehill
moles
molest
molested
molesting
molests
mollify
mollusk
mollycoddle
molten
moment
momentarily
momentariness
momentary
momentous
momentously
momentousness
moments
momentum
mommy
monadic
monarch
monarchies
monarchs
monarchy
monasteries
monastery
monastic
monetarism
monetary
money
moneyed
moneys
mongoose
monitor
monitored
monitoring
monitors
monk
monkey
monkeyed
monkeying
monkeys
monkish
monks
monoalphabetic
monochromatic
monochrome
monocotyledon
monocular
monogamous
monogamy
monogram
monograms
monograph
monographes
monographs
monolith
monolithic
monologue
monopolies
monopolize
monopolized
monopolizing
monopoly
monoprogrammed
monoprogramming
monostable
monotheism
monotone
monotonic
monotonically
monotonicity
monotonous
monotonously
monotonousness
monotony
monsoon
monster
monsters
monstrosity
monstrous
monstrously
month
monthly
months
monument
monumental
monumentally
monuments
moo
mood
moodiness
moods
moody
mooned
mooning
moonlight
moonlighter
moonlighting
moonlit
moons
moonshine
moored
mooring
moorings
moose
moot
mop
moped
mops
moraine
moral
morale
moralities
morality
morally
morals
morass
moratorium
morbid
morbidly
morbidness
more
moreover
mores
moribund
morn
morning
mornings
moron
morose
morphine
morphism
morphisms
morphological
morphology
morrow
morsel
morsels
mortal
mortality
mortally
mortals
mortar
mortared
mortaring
mortars
mortem
mortgage
mortgages
mortician
mortification
mortified
mortifies
mortify
mortifying
mosaic
mosaics
mosque
mosquito
mosquitoes
moss
mosses
mossy
most
mostly
motel
motels
moth
mothball
mothballs
mother
mothered
motherer
motherers
motherhood
mothering
motherland
motherly
mothers
motif
motifs
motion
motioned
motioning
motionless
motionlessly
motionlessness
motions
motivate
motivated
motivates
motivating
motivation
motivations
motive
motives
motley
motor
motorcar
motorcars
motorcycle
motorcycles
motoring
motorist
motorists
motorize
motorized
motorizes
motorizing
motors
motto
mottoes
mould
moulding
mound
mounded
mounds
mount
mountable
mountain
mountaineer
mountaineering
mountaineers
mountainous
mountainously
mountains
mounted
mounter
mounting
mountings
mounts
mourn
mourned
mourner
mourners
mournful
mournfully
mournfulness
mourning
mourns
mouse
mouser
mouses
mousetrap
mousy
mouth
mouthed
mouthes
mouthful
mouthing
mouthpiece
mouths
movable
move
moved
movement
movements
mover
movers
moves
movie
movies
moving
movings
mow
mowed
mower
mows
mu
much
muck
mucker
mucking
mucus
mud
muddied
muddiness
muddle
muddled
muddlehead
muddler
muddlers
muddles
muddling
muddy
muff
muffin
muffins
muffle
muffled
muffler
muffles
muffling
muffs
mug
mugging
mugs
mulatto
mulberries
mulberry
mule
mules
mull
mullah
multi
multibit
multibyte
multicast
multicasting
multicasts
multicellular
multicomputer
multidimensional
multilateral
multilayer
multilayered
multilevel
multimedia
multinational
multiple
multiples
multiplex
multiplexed
multiplexer
multiplexers
multiplexes
multiplexing
multiplexor
multiplexors
multiplicand
multiplicands
multiplication
multiplications
multiplicative
multiplicatives
multiplicity
multiplied
multiplier
multipliers
multiplies
multiply
multiplying
multiprocess
multiprocessing
multiprocessor
multiprocessors
multiprogram
multiprogrammed
multiprogramming
multistage
multitude
multitudes
multiuser
multivariate
multiword
mumble
mumbled
mumbler
mumblers
mumbles
mumbling
mumblings
mummies
mummy
munch
munched
munching
mundane
mundanely
mung
municipal
municipalities
municipality
municipally
munition
munitions
mural
murder
murdered
murderer
murderers
murdering
murderous
murderously
murders
murky
murmur
murmured
murmurer
murmuring
murmurs
muscle
muscled
muscles
muscling
muscular
musculature
muse
mused
muses
museum
museums
mush
mushroom
mushroomed
mushrooming
mushrooms
mushy
music
musical
musically
musicals
musician
musicianly
musicians
musicology
musing
musings
musk
musket
muskets
muskox
muskoxen
muskrat
muskrats
musks
muslin
mussel
mussels
must
mustache
mustached
mustaches
mustard
muster
mustiness
musts
musty
mutability
mutable
mutableness
mutandis
mutant
mutate
mutated
mutates
mutating
mutation
mutations
mutatis
mutative
mute
muted
mutely
muteness
mutilate
mutilated
mutilates
mutilating
mutilation
mutinies
mutiny
mutt
mutter
muttered
mutterer
mutterers
muttering
mutters
mutton
mutual
mutually
muzzle
muzzles
my
myriad
myrtle
myself
mysteries
mysterious
mysteriously
mysteriousness
mystery
mystic
mystical
mystics
mystify
myth
mythical
mythologies
mythology
nab
nabla
nablas
nadir
nag
nagged
nagging
nags
nail
nailed
nailing
nails
naive
naively
naiveness
naivete
naked
nakedly
nakedness
name
nameable
named
nameless
namelessly
namely
namer
namers
names
namesake
namesakes
naming
nanoinstruction
nanoinstructions
nanoprogram
nanoprogramming
nanosecond
nanoseconds
nanostore
nanostores
nap
napkin
napkins
naps
narcissus
narcotic
narcotics
narrate
narration
narrative
narratives
narrow
narrowed
narrower
narrowest
narrowing
narrowly
narrowness
narrows
nary
nasal
nasally
nastier
nastiest
nastily
nastiness
nasty
natal
nation
national
nationalist
nationalists
nationalities
nationality
nationalization
nationalize
nationalized
nationalizes
nationalizing
nationally
nationals
nationhood
nations
nationwide
native
natively
natives
nativity
natural
naturalism
naturalist
naturalization
naturally
naturalness
naturals
nature
natured
natures
naught
naughtier
naughtiness
naughty
nausea
nauseate
nauseum
naval
navally
navel
navies
navigable
navigate
navigated
navigates
navigating
navigation
navigator
navigators
navy
nay
near
nearby
neared
nearer
nearest
nearing
nearly
nearness
nears
nearsighted
neat
neater
neatest
neatly
neatness
nebula
nebular
nebulous
necessaries
necessarily
necessary
necessitate
necessitated
necessitates
necessitating
necessitation
necessities
necessity
neck
necking
necklace
necklaces
neckline
necks
necktie
neckties
necrosis
nectar
need
needed
needful
needing
needle
needled
needler
needlers
needles
needless
needlessly
needlessness
needlework
needling
needs
needy
negate
negated
negates
negating
negation
negations
negative
negatively
negatives
negator
negators
neglect
neglected
neglecting
neglects
negligee
negligence
negligent
negligible
negotiable
negotiate
negotiated
negotiates
negotiating
negotiation
negotiations
neigh
neighbor
neighborhood
neighborhoods
neighboring
neighborly
neighbors
neither
nemesis
neoclassic
neon
neonatal
neophyte
neophytes
nephew
nephews
nerve
nerves
nervous
nervously
nervousness
nest
nested
nester
nesting
nestle
nestled
nestles
nestling
nests
net
nether
nets
netted
netting
nettle
nettled
network
networked
networking
networks
neural
neuritis
neurological
neurologists
neuron
neurons
neuroses
neurosis
neurotic
neuter
neutral
neutralities
neutrality
neutralize
neutralized
neutralizing
neutrally
neutrino
neutrinos
neutron
never
nevertheless
new
newborn
newcomer
newcomers
newer
newest
newly
newlywed
newness
newscast
newsgroup
newsletter
newsletters
newsman
newsmen
newspaper
newspapers
newsstand
newt
next
nibble
nibbled
nibbler
nibblers
nibbles
nibbling
nice
nicely
niceness
nicer
nicest
niche
nick
nicked
nickel
nickels
nicker
nicking
nickname
nicknamed
nicknames
nicks
nicotine
niece
nieces
nifty
nigh
night
nightcap
nightclub
nightfall
nightgown
nightingale
nightingales
nightly
nightmare
nightmares
nightmarish
nights
nighttime
nihilism
nil
nimble
nimbleness
nimbler
nimbly
nimbus
nine
ninefold
nines
nineteen
nineteens
nineteenth
nineties
ninetieth
ninety
ninth
nip
nipple
nips
nitric
nitrogen
nitrous
nitty
no
nobility
noble
nobleman
nobleness
nobler
nobles
noblest
nobly
nobody
nocturnal
nocturnally
nod
nodal
nodded
nodding
node
nodes
nods
nodular
nodule
noise
noiseless
noiselessly
noises
noisier
noisily
noisiness
noisy
nomenclature
nominal
nominally
nominate
nominated
nominating
nomination
nominative
nominee
non
nonadaptive
nonbiodegradable
nonblocking
nonce
nonchalant
noncommercial
noncommunication
nonconsecutively
nonconservative
noncritical
noncyclic
nondecreasing
nondescript
nondescriptly
nondestructively
nondeterminacy
nondeterminate
nondeterminately
nondeterminism
nondeterministic
nondeterministically
none
nonempty
nonetheless
nonexistence
nonexistent
nonextensible
nonfunctional
nongovernmental
nonidempotent
noninteracting
noninterference
noninterleaved
nonintrusive
nonintuitive
noninverting
nonlinear
nonlinearities
nonlinearity
nonlinearly
nonlocal
nonmaskable
nonmathematical
nonmilitary
nonnegative
nonnegligible
nonnumerical
nonogenarian
nonorthogonal
nonorthogonality
nonperishable
nonpersistent
nonportable
nonprocedural
nonprocedurally
nonprofit
nonprogrammable
nonprogrammer
nonsegmented
nonsense
nonsensical
nonsequential
nonspecialist
nonspecialists
nonstandard
nonsynchronous
nontechnical
nonterminal
nonterminals
nonterminating
nontermination
nonthermal
nontransparent
nontrivial
nonuniform
nonuniformity
nonzero
noodle
nook
nooks
noon
noonday
noons
noontide
noontime
noose
nor
norm
normal
normalcy
normality
normalization
normalize
normalized
normalizes
normalizing
normally
normals
normative
norms
north
northbound
northeast
northeaster
northeastern
northerly
northern
northerner
northerners
northernly
northward
northwards
northwest
northwestern
nose
nosed
noses
nosing
nostalgia
nostalgic
nostril
nostrils
not
notable
notables
notably
notarize
notarized
notarizes
notarizing
notary
notation
notational
notations
notch
notched
notches
notching
note
notebook
notebooks
noted
notes
noteworthy
nothing
nothingness
nothings
notice
noticeable
noticeably
noticed
notices
noticing
notification
notifications
notified
notifier
notifiers
notifies
notify
notifying
noting
notion
notions
notoriety
notorious
notoriously
notwithstanding
noun
nouns
nourish
nourished
nourishes
nourishing
nourishment
novel
novelist
novelists
novels
novelties
novelty
novice
novices
now
nowadays
nowhere
noxious
nozzle
nu
nuance
nuances
nubile
nuclear
nuclei
nucleic
nucleotide
nucleotides
nucleus
nuclide
nude
nudge
nudged
nudity
nugget
nuisance
nuisances
null
nullary
nulled
nullified
nullifiers
nullifies
nullify
nullifying
nulls
numb
numbed
number
numbered
numberer
numbering
numberless
numbers
numbing
numbly
numbness
numbs
numerable
numeral
numerals
numerator
numerators
numeric
numerical
numerically
numerics
numerous
numismatic
numismatist
nun
nuns
nuptial
nurse
nursed
nurseries
nursery
nurses
nursing
nurture
nurtured
nurtures
nurturing
nut
nutate
nutria
nutrient
nutrition
nutritious
nuts
nutshell
nutshells
nuzzle
nylon
nymph
nymphomania
nymphomaniac
nymphs
oaf
oak
oaken
oaks
oar
oars
oases
oasis
oat
oaten
oath
oaths
oatmeal
oats
obedience
obediences
obedient
obediently
obelisk
obese
obey
obeyed
obeying
obeys
obfuscate
obfuscatory
obituary
object
objected
objecting
objection
objectionable
objections
objective
objectively
objectives
objector
objectors
objects
obligated
obligation
obligations
obligatory
oblige
obliged
obliges
obliging
obligingly
oblique
obliquely
obliqueness
obliterate
obliterated
obliterates
obliterating
obliteration
oblivion
oblivious
obliviously
obliviousness
oblong
obnoxious
oboe
obscene
obscure
obscured
obscurely
obscurer
obscures
obscuring
obscurities
obscurity
obsequious
observable
observance
observances
observant
observation
observations
observatory
observe
observed
observer
observers
observes
observing
obsession
obsessions
obsessive
obsolescence
obsolescent
obsolete
obsoleted
obsoletes
obsoleting
obstacle
obstacles
obstinacy
obstinate
obstinately
obstruct
obstructed
obstructing
obstruction
obstructions
obstructive
obtain
obtainable
obtainably
obtained
obtaining
obtains
obviate
obviated
obviates
obviating
obviation
obviations
obvious
obviously
obviousness
occasion
occasional
occasionally
occasioned
occasioning
occasionings
occasions
occipital
occlude
occluded
occludes
occlusion
occlusions
occult
occupancies
occupancy
occupant
occupants
occupation
occupational
occupationally
occupations
occupied
occupier
occupies
occupy
occupying
occur
occurred
occurrence
occurrences
occurring
occurs
ocean
oceanic
oceanography
oceans
octagon
octagonal
octahedra
octahedral
octahedron
octal
octane
octave
octaves
octet
octets
octogenarian
octopus
odd
odder
oddest
oddities
oddity
oddly
oddness
odds
ode
odes
odious
odiously
odiousness
odium
odor
odorous
odorously
odorousness
odors
of
off
offend
offended
offender
offenders
offending
offends
offense
offenses
offensive
offensively
offensiveness
offer
offered
offerer
offerers
offering
offerings
offers
offhand
office
officemate
officer
officers
offices
official
officialdom
officially
officials
officiate
officio
officious
officiously
officiousness
offing
offload
offs
offset
offsets
offsetting
offshore
offspring
oft
often
oftentimes
oh
ohm
ohmmeter
oil
oilcloth
oiled
oiler
oilers
oilier
oiliest
oiling
oils
oily
ointment
okay
old
olden
older
oldest
oldness
oldy
oleander
oleomargarine
oligarchy
olive
olives
omega
omelet
omen
omens
omicron
ominous
ominously
ominousness
omission
omissions
omit
omits
omitted
omitting
omnibus
omnidirectional
omnipotent
omnipresent
omniscient
omnisciently
omnivore
on
onanism
once
oncology
one
oneness
onerous
ones
oneself
onetime
ongoing
onion
onions
online
onlooker
only
onrush
onset
onsets
onslaught
onto
ontology
onus
onward
onwards
onyx
ooze
oozed
opacity
opal
opals
opaque
opaquely
opaqueness
opcode
open
opened
opener
openers
opening
openings
openly
openness
opens
opera
operable
operand
operandi
operands
operas
operate
operated
operates
operating
operation
operational
operationally
operations
operative
operatives
operator
operators
operetta
opiate
opinion
opinions
opium
opossum
opponent
opponents
opportune
opportunely
opportunism
opportunistic
opportunities
opportunity
opposable
oppose
opposed
opposes
opposing
opposite
oppositely
oppositeness
opposites
opposition
oppress
oppressed
oppresses
oppressing
oppression
oppressive
oppressor
oppressors
opprobrium
opt
opted
opthalmic
optic
optical
optically
optics
optima
optimal
optimality
optimally
optimism
optimist
optimistic
optimistically
optimization
optimizations
optimize
optimized
optimizer
optimizers
optimizes
optimizing
optimum
opting
option
optional
optionally
options
optoacoustic
optometrist
optometry
opts
opulence
opulent
opus
or
oracle
oracles
oral
orally
orange
oranges
orangutan
oration
orations
orator
oratories
orators
oratory
orb
orbit
orbital
orbitally
orbited
orbiter
orbiters
orbiting
orbits
orchard
orchards
orchestra
orchestral
orchestras
orchestrate
orchid
orchids
ordain
ordained
ordaining
ordains
ordeal
order
ordered
ordering
orderings
orderlies
orderly
orders
ordinal
ordinance
ordinances
ordinarily
ordinariness
ordinary
ordinate
ordinates
ordination
ore
oregano
ores
organ
organic
organism
organisms
organist
organists
organizable
organization
organizational
organizationally
organizations
organize
organized
organizer
organizers
organizes
organizing
organs
orgasm
orgiastic
orgies
orgy
orientation
orientations
oriented
orienting
orients
orifice
orifices
origin
original
originality
originally
originals
originate
originated
originates
originating
origination
originator
originators
origins
oriole
ornament
ornamental
ornamentally
ornamentation
ornamented
ornamenting
ornaments
ornate
ornery
orphan
orphanage
orphaned
orphans
orthant
orthodontist
orthodox
orthodoxy
orthogonal
orthogonality
orthogonally
orthopedic
oscillate
oscillated
oscillates
oscillating
oscillation
oscillations
oscillator
oscillators
oscillatory
oscilloscope
oscilloscopes
osmosis
osmotic
ossify
ostensible
ostensibly
ostentatious
osteopath
osteopathic
osteopathy
osteoporosis
ostracism
ostrich
ostriches
other
others
otherwise
otherworldly
otter
otters
ouch
ought
ounce
ounces
our
ours
ourself
ourselves
oust
out
outbound
outbreak
outbreaks
outburst
outbursts
outcast
outcasts
outcome
outcomes
outcries
outcry
outdated
outdo
outdoor
outdoors
outer
outermost
outfit
outfits
outfitted
outgoing
outgrew
outgrow
outgrowing
outgrown
outgrows
outgrowth
outing
outlandish
outlast
outlasts
outlaw
outlawed
outlawing
outlaws
outlay
outlays
outlet
outlets
outline
outlined
outlines
outlining
outlive
outlived
outlives
outliving
outlook
outlying
outnumbered
outperform
outperformed
outperforming
outperforms
outpost
outposts
output
outputs
outputting
outrage
outraged
outrageous
outrageously
outrages
outright
outrun
outruns
outs
outset
outside
outsider
outsiders
outskirts
outstanding
outstandingly
outstretched
outstrip
outstripped
outstripping
outstrips
outvote
outvoted
outvotes
outvoting
outward
outwardly
outweigh
outweighed
outweighing
outweighs
outwit
outwits
outwitted
outwitting
oval
ovals
ovaries
ovary
oven
ovens
over
overall
overalls
overboard
overcame
overcoat
overcoats
overcome
overcomes
overcoming
overcrowd
overcrowded
overcrowding
overcrowds
overdone
overdose
overdraft
overdrafts
overdue
overemphasis
overemphasized
overestimate
overestimated
overestimates
overestimating
overestimation
overflow
overflowed
overflowing
overflows
overgrown
overhang
overhanging
overhangs
overhaul
overhauling
overhead
overheads
overhear
overheard
overhearing
overhears
overjoy
overjoyed
overkill
overland
overlap
overlapped
overlapping
overlaps
overlay
overlaying
overlays
overload
overloaded
overloading
overloads
overlook
overlooked
overlooking
overlooks
overly
overnight
overnighter
overnighters
overpower
overpowered
overpowering
overpowers
overprint
overprinted
overprinting
overprints
overproduction
overridden
override
overrides
overriding
overrode
overrule
overruled
overrules
overrun
overrunning
overruns
overseas
oversee
overseeing
overseer
overseers
oversees
overshadow
overshadowed
overshadowing
overshadows
overshoot
overshot
oversight
oversights
oversimplified
oversimplifies
oversimplify
oversimplifying
oversized
overstate
overstated
overstatement
overstatements
overstates
overstating
overstocks
oversubscribed
overt
overtake
overtaken
overtaker
overtakers
overtakes
overtaking
overthrew
overthrow
overthrown
overtime
overtly
overtone
overtones
overtook
overture
overtures
overturn
overturned
overturning
overturns
overuse
overview
overviews
overwhelm
overwhelmed
overwhelming
overwhelmingly
overwhelms
overwork
overworked
overworking
overworks
overwrite
overwrites
overwriting
overwritten
overzealous
owe
owed
owes
owing
owl
owls
own
owned
owner
owners
ownership
ownerships
owning
owns
ox
oxen
oxide
oxides
oxidize
oxidized
oxygen
oyster
oysters
ozone
pace
paced
pacemaker
pacer
pacers
paces
pacific
pacification
pacified
pacifier
pacifies
pacifism
pacifist
pacify
pacing
pack
package
packaged
packager
packagers
packages
packaging
packagings
packed
packer
packers
packet
packets
packing
packs
pact
pacts
pad
padded
padding
paddle
paddock
paddy
padlock
pads
pagan
pagans
page
pageant
pageantry
pageants
paged
pager
pagers
pages
paginate
paginated
paginates
paginating
pagination
paging
pagoda
paid
pail
pails
pain
pained
painful
painfully
painless
pains
painstaking
painstakingly
paint
painted
painter
painters
painting
paintings
paints
pair
paired
pairing
pairings
pairs
pairwise
pajama
pajamas
pal
palace
palaces
palate
palates
pale
paled
palely
paleness
paler
pales
palest
palfrey
palindrome
palindromic
paling
pall
palladium
palliate
palliative
pallid
palm
palmed
palmer
palming
palms
palpable
pals
palsy
pamper
pamphlet
pamphlets
pan
panacea
panaceas
panama
pancake
pancakes
panda
pandas
pandemic
pandemonium
pander
pane
panel
paneled
paneling
panelist
panelists
panels
panes
pang
pangs
panic
panicked
panicking
panicky
panics
panned
panning
panorama
panoramic
pans
pansies
pansy
pant
panted
pantheism
pantheist
pantheon
panther
panthers
panties
panting
pantomime
pantries
pantry
pants
panty
pantyhose
papa
papal
paper
paperback
paperbacks
papered
paperer
paperers
papering
paperings
papers
paperweight
paperwork
papoose
papyrus
par
parabola
parabolic
paraboloid
paraboloidal
parachute
parachuted
parachutes
parade
paraded
parades
paradigm
paradigms
parading
paradise
paradox
paradoxes
paradoxical
paradoxically
paraffin
paragon
paragons
paragraph
paragraphing
paragraphs
parakeet
parallax
parallel
paralleled
paralleling
parallelism
parallelize
parallelized
parallelizes
parallelizing
parallelogram
parallelograms
parallels
paralysis
paralyze
paralyzed
paralyzes
paralyzing
parameter
parameterizable
parameterization
parameterizations
parameterize
parameterized
parameterizes
parameterizing
parameterless
parameters
parametric
parametrized
paramilitary
paramount
paranoia
paranoiac
paranoid
paranormal
parapet
parapets
paraphernalia
paraphrase
paraphrased
paraphrases
paraphrasing
parapsychology
parasite
parasites
parasitic
parasitics
parasol
parboil
parcel
parceled
parceling
parcels
parch
parched
parchment
pardon
pardonable
pardonably
pardoned
pardoner
pardoners
pardoning
pardons
pare
paregoric
parent
parentage
parental
parentheses
parenthesis
parenthesized
parenthesizes
parenthesizing
parenthetic
parenthetical
parenthetically
parenthood
parents
pares
pariah
parimutuel
paring
parings
parish
parishes
parishioner
parity
park
parked
parker
parkers
parking
parkland
parklike
parkway
parlay
parley
parliament
parliamentarian
parliamentary
parliaments
parlor
parlors
parochial
parody
parole
paroled
paroles
paroling
parried
parrot
parroting
parrots
parry
pars
parse
parsed
parser
parsers
parses
parsimony
parsing
parsings
parsley
parson
part
partake
partaker
partakes
partaking
parted
parter
parters
partial
partiality
partially
participant
participants
participate
participated
participates
participating
participation
participle
particle
particles
particular
particularly
particulars
particulate
parties
parting
partings
partisan
partisans
partition
partitioned
partitioning
partitions
partly
partner
partnered
partners
partnership
partook
partridge
partridges
parts
party
pass
passage
passages
passageway
passe
passed
passenger
passengers
passer
passers
passes
passing
passion
passionate
passionately
passions
passivate
passive
passively
passiveness
passivity
passport
passports
password
passwords
past
paste
pasted
pastel
pastes
pastime
pastimes
pasting
pastness
pastor
pastoral
pastors
pastry
pasts
pasture
pastures
pat
patch
patched
patches
patching
patchwork
patchy
pate
paten
patent
patentable
patented
patenter
patenters
patenting
patently
patents
paternal
paternally
paternoster
path
pathetic
pathname
pathnames
pathogen
pathogenesis
pathological
pathology
pathos
paths
pathway
pathways
patience
patient
patiently
patients
patina
patio
patriarch
patriarchal
patriarchs
patriarchy
patrician
patricians
patrimonial
patrimony
patriot
patriotic
patriotism
patriots
patrol
patrolled
patrolling
patrolman
patrolmen
patrols
patron
patronage
patronize
patronized
patronizes
patronizing
patrons
pats
patter
pattered
pattering
patterings
pattern
patterned
patterning
patterns
patters
patties
patty
paucity
paunch
paunchy
pauper
pause
paused
pauses
pausing
pave
paved
pavement
pavements
paves
pavilion
pavilions
paving
paw
pawing
pawn
pawns
pawnshop
paws
pay
payable
paycheck
paychecks
payed
payer
payers
paying
payment
payments
payoff
payoffs
payroll
pays
pea
peace
peaceable
peaceful
peacefully
peacefulness
peacetime
peach
peaches
peacock
peacocks
peak
peaked
peaks
peal
pealed
pealing
peals
peanut
peanuts
pear
pearl
pearls
pearly
pears
peas
peasant
peasantry
peasants
peat
pebble
pebbles
peccary
peck
pecked
pecking
pecks
pectoral
peculiar
peculiarities
peculiarity
peculiarly
pecuniary
pedagogic
pedagogical
pedagogically
pedagogy
pedal
pedant
pedantic
pedantry
peddle
peddler
peddlers
pedestal
pedestrian
pedestrians
pediatric
pediatrician
pediatrics
pedigree
peek
peeked
peeking
peeks
peel
peeled
peeling
peels
peep
peeped
peeper
peephole
peeping
peeps
peer
peered
peering
peerless
peers
peg
pegboard
pegs
pejorative
pelican
pellagra
pelt
pelting
pelts
pelvic
pelvis
pen
penal
penalize
penalized
penalizes
penalizing
penalties
penalty
penance
pence
penchant
pencil
penciled
pencils
pend
pendant
pended
pending
pends
pendulum
pendulums
penetrable
penetrate
penetrated
penetrates
penetrating
penetratingly
penetration
penetrations
penetrative
penetrator
penetrators
penguin
penguins
penicillin
peninsula
peninsulas
penis
penises
penitent
penitentiary
penned
pennies
penniless
penning
penny
pens
pension
pensioner
pensions
pensive
pent
pentagon
pentagons
pentecostal
penthouse
penultimate
penumbra
peony
people
peopled
peoples
pep
pepper
peppered
peppering
peppermint
pepperoni
peppers
peppery
peppy
peptide
per
perceivable
perceivably
perceive
perceived
perceiver
perceivers
perceives
perceiving
percent
percentage
percentages
percentile
percentiles
percents
perceptible
perceptibly
perception
perceptions
perceptive
perceptively
perceptual
perceptually
perch
perchance
perched
perches
perching
percussion
percutaneous
peremptory
perennial
perennially
perfect
perfected
perfectible
perfecting
perfection
perfectionist
perfectionists
perfectly
perfectness
perfects
perforce
perform
performance
performances
performed
performer
performers
performing
performs
perfume
perfumed
perfumes
perfuming
perfunctory
perhaps
perihelion
peril
perilous
perilously
perils
perimeter
period
periodic
periodical
periodically
periodicals
periods
peripheral
peripherally
peripherals
peripheries
periphery
periscope
perish
perishable
perishables
perished
perisher
perishers
perishes
perishing
perjure
perjury
perk
perky
permanence
permanent
permanently
permeable
permeate
permeated
permeates
permeating
permeation
permissibility
permissible
permissibly
permission
permissions
permissive
permissively
permit
permits
permitted
permitting
permutation
permutations
permute
permuted
permutes
permuting
pernicious
peroxide
perpendicular
perpendicularly
perpendiculars
perpetrate
perpetrated
perpetrates
perpetrating
perpetration
perpetrations
perpetrator
perpetrators
perpetual
perpetually
perpetuate
perpetuated
perpetuates
perpetuating
perpetuation
perpetuity
perplex
perplexed
perplexing
perplexity
persecute
persecuted
persecutes
persecuting
persecution
persecutor
persecutors
perseverance
persevere
persevered
perseveres
persevering
persist
persisted
persistence
persistent
persistently
persisting
persists
person
personage
personages
personal
personalities
personality
personalization
personalize
personalized
personalizes
personalizing
personally
personification
personified
personifies
personify
personifying
personnel
persons
perspective
perspectives
perspicuous
perspicuously
perspiration
perspire
persuadable
persuade
persuaded
persuader
persuaders
persuades
persuading
persuasion
persuasions
persuasive
persuasively
persuasiveness
pertain
pertained
pertaining
pertains
pertinent
perturb
perturbation
perturbations
perturbed
perusal
peruse
perused
peruser
perusers
peruses
perusing
pervade
pervaded
pervades
pervading
pervasive
pervasively
perversion
pervert
perverted
perverts
pessimism
pessimist
pessimistic
pest
pester
pesticide
pestilence
pestilent
pests
pet
petal
petals
petition
petitioned
petitioner
petitioning
petitions
petri
petroleum
pets
petted
petter
petters
petticoat
petticoats
pettiness
petting
petty
petulance
petulant
pew
pews
pewter
phantom
phantoms
pharmaceutic
pharmacist
pharmacology
pharmacopoeia
pharmacy
phase
phased
phaser
phasers
phases
phasing
pheasant
pheasants
phenomena
phenomenal
phenomenally
phenomenological
phenomenologically
phenomenologies
phenomenology
phenomenon
phi
philanthropy
philharmonic
philosopher
philosophers
philosophic
philosophical
philosophically
philosophies
philosophize
philosophized
philosophizer
philosophizers
philosophizes
philosophizing
philosophy
phoenix
phone
phoned
phoneme
phonemes
phonemic
phones
phonetic
phonetics
phoning
phonograph
phonographs
phony
phosgene
phosphate
phosphates
phosphor
phosphorescent
phosphoric
phosphorus
photo
photocopied
photocopier
photocopiers
photocopies
photocopy
photocopying
photodiode
photodiodes
photogenic
photograph
photographed
photographer
photographers
photographic
photographing
photographs
photography
photon
photos
photosensitive
phototypesetter
phototypesetters
phrase
phrased
phraseology
phrases
phrasing
phrasings
phyla
phylum
physic
physical
physically
physicalness
physicals
physician
physicians
physicist
physicists
physics
physiological
physiologically
physiology
physiotherapist
physiotherapy
physique
phytoplankton
pi
pianist
piano
pianos
pica
picas
picayune
piccolo
pick
pickaxe
picked
picker
pickers
picket
picketed
picketer
picketers
picketing
pickets
picking
pickings
pickle
pickled
pickles
pickling
picks
pickup
pickups
picky
picnic
picnicked
picnicking
picnics
picofarad
picojoule
picosecond
pictorial
pictorially
picture
pictured
pictures
picturesque
picturesqueness
picturing
piddle
pidgin
pie
piece
pieced
piecemeal
pieces
piecewise
piecing
pier
pierce
pierced
pierces
piercing
piers
pies
piety
piezoelectric
pig
pigeon
pigeonhole
pigeons
piggish
piggy
piggyback
piggybacked
piggybacking
piggybacks
pigment
pigmentation
pigmented
pigments
pigpen
pigs
pigskin
pigtail
pike
piker
pikes
pile
piled
pilers
piles
pilfer
pilferage
pilgrim
pilgrimage
pilgrimages
pilgrims
piling
pilings
pill
pillage
pillaged
pillar
pillared
pillars
pillory
pillow
pillows
pills
pilot
piloting
pilots
pimp
pimple
pin
pinafore
pinball
pinch
pinched
pinches
pinching
pincushion
pine
pineapple
pineapples
pined
pines
ping
pinhead
pinhole
pining
pinion
pink
pinker
pinkest
pinkie
pinkish
pinkly
pinkness
pinks
pinnacle
pinnacles
pinned
pinning
pinnings
pinochle
pinpoint
pinpointing
pinpoints
pins
pinscher
pint
pinto
pints
pinwheel
pion
pioneer
pioneered
pioneering
pioneers
pious
piously
pip
pipe
piped
pipeline
pipelined
pipelines
pipelining
pipers
pipes
pipette
piping
pique
piracy
pirate
pirates
piss
pistachio
pistil
pistils
pistol
pistols
piston
pistons
pit
pitch
pitched
pitcher
pitchers
pitches
pitchfork
pitching
piteous
piteously
pitfall
pitfalls
pith
pithed
pithes
pithier
pithiest
pithiness
pithing
pithy
pitiable
pitied
pitier
pitiers
pities
pitiful
pitifully
pitiless
pitilessly
pits
pitted
pituitary
pity
pitying
pityingly
pivot
pivotal
pivoting
pivots
pixel
pixels
pizza
placard
placards
placate
place
placebo
placed
placeholder
placement
placements
placenta
placental
placer
places
placid
placidly
placing
plagiarism
plagiarist
plague
plagued
plagues
plaguing
plaid
plaids
plain
plainer
plainest
plainly
plainness
plains
plaintext
plaintexts
plaintiff
plaintiffs
plaintive
plaintively
plaintiveness
plait
plaits
plan
planar
planarity
plane
planed
planeload
planer
planers
planes
planet
planetaria
planetarium
planetary
planetesimal
planetoid
planets
planing
plank
planking
planks
plankton
planned
planner
planners
planning
planoconcave
planoconvex
plans
plant
plantation
plantations
planted
planter
planters
planting
plantings
plants
plaque
plasma
plaster
plastered
plasterer
plastering
plasters
plastic
plasticity
plastics
plate
plateau
plateaus
plated
platelet
platelets
platen
platens
plates
platform
platforms
plating
platinum
platitude
platonic
platoon
platter
platters
plausibility
plausible
play
playable
playback
playboy
played
player
players
playful
playfully
playfulness
playground
playgrounds
playhouse
playing
playmate
playmates
playoff
playroom
plays
plaything
playthings
playtime
playwright
playwrights
playwriting
plaza
plea
plead
pleaded
pleader
pleading
pleads
pleas
pleasant
pleasantly
pleasantness
please
pleased
pleases
pleasing
pleasingly
pleasure
pleasures
pleat
plebeian
plebian
plebiscite
plebiscites
pledge
pledged
pledges
plenary
plenipotentiary
plenteous
plentiful
plentifully
plenty
plethora
pleurisy
pliable
pliant
plied
pliers
plies
plight
plod
plodding
plot
plots
plotted
plotter
plotters
plotting
plow
plowed
plower
plowing
plowman
plows
plowshare
ploy
ploys
pluck
plucked
plucking
plucks
plucky
plug
pluggable
plugged
plugging
plugs
plum
plumage
plumb
plumbed
plumbing
plumbs
plume
plumed
plumes
plummet
plummeting
plump
plumped
plumpness
plums
plunder
plundered
plunderer
plunderers
plundering
plunders
plunge
plunged
plunger
plungers
plunges
plunging
plunk
plural
plurality
plurals
plus
pluses
plush
plutonium
ply
plywood
pneumatic
pneumonia
poach
poacher
poaches
pocket
pocketbook
pocketbooks
pocketed
pocketful
pocketing
pockets
pod
podia
podium
pods
poem
poems
poet
poetic
poetical
poetically
poetics
poetries
poetry
poets
pogo
pogrom
poignancy
poignant
point
pointed
pointedly
pointer
pointers
pointing
pointless
points
pointy
poise
poised
poises
poison
poisoned
poisoner
poisoning
poisonous
poisonousness
poisons
poke
poked
poker
pokerface
pokes
poking
polar
polarities
polarity
pole
polecat
poled
polemic
polemics
poles
police
policed
policeman
policemen
polices
policies
policing
policy
poling
polio
polish
polished
polisher
polishers
polishes
polishing
polite
politely
politeness
politer
politest
politic
political
politically
politician
politicians
politicking
politics
polka
poll
polled
pollen
polling
polloi
polls
pollutant
pollute
polluted
pollutes
polluting
pollution
polo
polyalphabetic
polygon
polygons
polymer
polymers
polymorphic
polynomial
polynomials
polytechnic
polytheist
pomp
pompadour
pomposity
pompous
pompously
pompousness
poncho
pond
ponder
pondered
pondering
ponderous
ponders
ponds
pong
ponies
pontiff
pontific
pontificate
pony
pooch
poodle
pool
pooled
pooling
pools
poor
poorer
poorest
poorly
poorness
pop
popcorn
popish
poplar
poplin
popped
poppies
popping
poppy
pops
populace
popular
popularity
popularization
popularize
popularized
popularizes
popularizing
popularly
populate
populated
populates
populating
population
populations
populous
populousness
porcelain
porch
porches
porcine
porcupine
porcupines
pore
pored
pores
poring
pork
porker
pornographer
pornographic
pornography
porous
porpoise
porridge
port
portability
portable
portage
portal
portals
ported
portend
portended
portending
portends
portent
portentous
porter
porterhouse
porters
portfolio
portfolios
portico
porting
portion
portions
portly
portmanteau
portrait
portraits
portray
portrayal
portrayed
portraying
portrays
ports
pose
posed
poser
posers
poses
posh
posing
posit
posited
positing
position
positional
positioned
positioning
positions
positive
positively
positiveness
positives
positron
posits
posse
possess
possessed
possesses
possessing
possession
possessional
possessions
possessive
possessively
possessiveness
possessor
possessors
possibilities
possibility
possible
possibly
possum
possums
post
postage
postal
postcard
postcondition
postdoctoral
posted
poster
posterior
posteriori
posterity
posters
postfix
postgraduate
posting
postlude
postman
postmark
postmaster
postmasters
postmortem
postoperative
postorder
postpone
postponed
postponing
postprocess
postprocessor
posts
postscript
postscripts
postulate
postulated
postulates
postulating
postulation
postulations
posture
postures
pot
potable
potash
potassium
potato
potatoes
potbelly
potent
potentate
potentates
potential
potentialities
potentiality
potentially
potentials
potentiating
potentiometer
potentiometers
pothole
potion
potlatch
potpourri
pots
potted
potter
potters
pottery
potting
pouch
pouches
poultice
poultry
pounce
pounced
pounces
pouncing
pound
pounded
pounder
pounders
pounding
pounds
pour
poured
pourer
pourers
pouring
pours
pout
pouted
pouting
pouts
poverty
powder
powdered
powdering
powderpuff
powders
powdery
power
powered
powerful
powerfully
powerfulness
powering
powerless
powerlessly
powerlessness
pox
practicable
practicably
practical
practicality
practically
practice
practiced
practices
practicing
practitioner
practitioners
pragmatic
pragmatically
pragmatics
pragmatism
pragmatist
prairie
praise
praised
praiser
praisers
praises
praiseworthy
praising
praisingly
prance
pranced
prancer
prancing
prank
pranks
prate
pray
prayed
prayer
prayers
praying
preach
preached
preacher
preachers
preaches
preaching
preallocate
preallocated
preallocating
preamble
preambles
preassign
preassigned
preassigning
preassigns
precarious
precariously
precariousness
precaution
precautions
precede
preceded
precedence
precedences
precedent
precedented
precedents
precedes
preceding
precept
precepts
precess
precession
precinct
precincts
precious
preciously
preciousness
precipice
precipitable
precipitate
precipitated
precipitately
precipitateness
precipitates
precipitating
precipitation
precipitous
precipitously
precise
precisely
preciseness
precision
precisions
preclude
precluded
precludes
precluding
precocious
precociously
precocity
precompute
precomputed
precomputing
preconceive
preconceived
preconception
preconceptions
precondition
preconditioned
preconditions
precursor
precursors
predate
predated
predates
predating
predatory
predecessor
predecessors
predefine
predefined
predefines
predefining
predefinition
predefinitions
predetermination
predetermine
predetermined
predetermines
predetermining
predicament
predicate
predicated
predicates
predicating
predication
predications
predict
predictability
predictable
predictably
predicted
predicting
prediction
predictions
predictive
predictor
predicts
predilection
predilections
predisposition
predominant
predominantly
predominate
predominated
predominately
predominates
predominating
predomination
preeminence
preeminent
preempt
preempted
preempting
preemption
preemptive
preemptor
preempts
preen
preexisting
prefab
prefabricate
preface
prefaced
prefaces
prefacing
prefer
preferable
preferably
preference
preferences
preferential
preferentially
preferred
preferring
prefers
prefix
prefixed
prefixes
prefixing
pregnancy
pregnant
prehistoric
preinitialize
preinitialized
preinitializes
preinitializing
prejudge
prejudged
prejudice
prejudiced
prejudices
prejudicial
prelate
preliminaries
preliminary
prelude
preludes
premature
prematurely
prematurity
premeditated
premeditation
premier
premiers
premise
premises
premium
premiums
premonition
prenatal
preoccupation
preoccupied
preoccupies
preoccupy
prep
preparation
preparations
preparative
preparatives
preparatory
prepare
prepared
prepares
preparing
prepend
prepended
prepending
preposition
prepositional
prepositions
preposterous
preposterously
preprocessed
preprocessing
preprocessor
preprocessors
preproduction
preprogrammed
prerequisite
prerequisites
prerogative
prerogatives
prescribe
prescribed
prescribes
prescription
prescriptions
prescriptive
preselect
preselected
preselecting
preselects
presence
presences
present
presentation
presentations
presented
presenter
presenting
presently
presentness
presents
preservation
preservations
preserve
preserved
preserver
preservers
preserves
preserving
preset
preside
presided
presidency
president
presidential
presidents
presides
presiding
press
pressed
presser
presses
pressing
pressings
pressure
pressured
pressures
pressuring
pressurize
pressurized
prestidigitate
prestige
prestigious
presumably
presume
presumed
presumes
presuming
presumption
presumptions
presumptive
presumptuous
presumptuousness
presuppose
presupposed
presupposes
presupposing
presupposition
pretend
pretended
pretender
pretenders
pretending
pretends
pretense
pretenses
pretension
pretensions
pretentious
pretentiously
pretentiousness
pretext
pretexts
prettier
prettiest
prettily
prettiness
pretty
prevail
prevailed
prevailing
prevailingly
prevails
prevalence
prevalent
prevalently
prevent
preventable
preventably
prevented
preventing
prevention
preventive
preventives
prevents
preview
previewed
previewing
previews
previous
previously
prey
preyed
preying
preys
price
priced
priceless
pricer
pricers
prices
pricing
prick
pricked
pricking
prickly
pricks
pride
prided
prides
priding
priest
priggish
prim
prima
primacy
primal
primaries
primarily
primary
primate
prime
primed
primeness
primer
primers
primes
primeval
priming
primitive
primitively
primitiveness
primitives
primrose
prince
princely
princes
princess
princesses
principal
principalities
principality
principally
principals
principle
principled
principles
print
printable
printably
printed
printer
printers
printing
printout
prints
prior
priori
priorities
priority
priory
prism
prisms
prison
prisoner
prisoners
prisons
pristine
privacies
privacy
private
privately
privates
privation
privations
privies
privilege
privileged
privileges
privy
prize
prized
prizer
prizers
prizes
prizewinning
prizing
pro
probabilistic
probabilistically
probabilities
probability
probable
probably
probate
probated
probates
probating
probation
probative
probe
probed
probes
probing
probings
probity
problem
problematic
problematical
problematically
problems
procaine
procedural
procedurally
procedure
procedures
proceed
proceeded
proceeding
proceedings
proceeds
process
processed
processes
processing
procession
processor
processors
proclaim
proclaimed
proclaimer
proclaimers
proclaiming
proclaims
proclamation
proclamations
proclivities
proclivity
procotols
procrastinate
procrastinated
procrastinates
procrastinating
procrastination
procreate
procure
procured
procurement
procurements
procurer
procurers
procures
procuring
prod
prodigal
prodigally
prodigious
prodigy
produce
produced
producer
producers
produces
producible
producing
product
production
productions
productive
productively
productivity
products
profane
profanely
profess
professed
professes
professing
profession
professional
professionalism
professionally
professionals
professions
professor
professorial
professors
proffer
proffered
proffers
proficiency
proficient
proficiently
profile
profiled
profiles
profiling
profit
profitability
profitable
profitably
profited
profiteer
profiteers
profiting
profits
profitted
profligate
profound
profoundest
profoundly
profundity
profuse
profusion
progenitor
progeny
prognosis
prognosticate
program
programmability
programmable
programmed
programmer
programmers
programming
programs
progress
progressed
progresses
progressing
progression
progressions
progressive
progressively
prohibit
prohibited
prohibiting
prohibition
prohibitions
prohibitive
prohibitively
prohibitory
prohibits
project
projected
projectile
projecting
projection
projections
projective
projectively
projector
projectors
projects
prolate
prolegomena
proletariat
proliferate
proliferated
proliferates
proliferating
proliferation
prolific
prolix
prolog
prologue
prolong
prolongate
prolonged
prolonging
prolongs
promenade
promenades
prominence
prominent
prominently
promiscuous
promise
promised
promises
promising
promontory
promote
promoted
promoter
promoters
promotes
promoting
promotion
promotional
promotions
prompt
prompted
prompter
promptest
prompting
promptings
promptly
promptness
prompts
promulgate
promulgated
promulgates
promulgating
promulgation
prone
proneness
prong
pronged
prongs
pronoun
pronounce
pronounceable
pronounced
pronouncement
pronouncements
pronounces
pronouncing
pronouns
pronunciation
pronunciations
proof
proofread
proofreader
proofs
prop
propaganda
propagandist
propagate
propagated
propagates
propagating
propagation
propagations
propane
propel
propellant
propelled
propeller
propellers
propelling
propels
propensity
proper
properly
properness
propertied
properties
property
prophecies
prophecy
prophesied
prophesier
prophesies
prophesy
prophet
prophetic
prophets
propitious
proponent
proponents
proportion
proportional
proportionally
proportionately
proportioned
proportioning
proportionment
proportions
propos
proposal
proposals
propose
proposed
proposer
proposes
proposing
proposition
propositional
propositionally
propositioned
propositioning
propositions
propound
propounded
propounding
propounds
proprietary
proprietor
proprietors
propriety
props
propulsion
propulsions
prorate
prorated
prorates
pros
proscenium
proscribe
proscription
prose
prosecute
prosecuted
prosecutes
prosecuting
prosecution
prosecutions
prosecutor
proselytize
proselytized
proselytizes
proselytizing
prosodic
prosodics
prospect
prospected
prospecting
prospection
prospections
prospective
prospectively
prospectives
prospector
prospectors
prospects
prospectus
prosper
prospered
prospering
prosperity
prosperous
prospers
prostate
prosthetic
prostitute
prostitution
prostrate
prostration
protagonist
protean
protect
protected
protecting
protection
protections
protective
protectively
protectiveness
protector
protectorate
protectors
protects
protege
proteges
protein
proteins
protest
protestant
protestation
protestations
protested
protesting
protestingly
protestor
protests
protocol
protocols
proton
protons
protoplasm
prototype
prototyped
prototypes
prototypical
prototypically
prototyping
protozoan
protract
protrude
protruded
protrudes
protruding
protrusion
protrusions
protuberant
proud
prouder
proudest
proudly
provability
provable
provably
prove
proved
proven
provenance
prover
proverb
proverbial
proverbs
provers
proves
provide
provided
providence
provident
provider
providers
provides
providing
province
provinces
provincial
proving
provision
provisional
provisionally
provisioned
provisioning
provisions
proviso
provocation
provoke
provoked
provokes
provost
prow
prowess
prowl
prowled
prowler
prowlers
prowling
prows
proximal
proximate
proximity
proxy
prudence
prudent
prudential
prudently
prune
pruned
pruner
pruners
prunes
pruning
prurient
pry
prying
psalm
psalms
pseudo
pseudofiles
pseudoinstruction
pseudoinstructions
pseudonym
pseudoparallelism
psilocybin
psych
psyche
psychedelic
psyches
psychiatric
psychiatrist
psychiatrists
psychiatry
psychic
psycho
psychoanalysis
psychoanalyst
psychoanalytic
psychobiology
psychological
psychologically
psychologist
psychologists
psychology
psychopath
psychopathic
psychophysic
psychoses
psychosis
psychosocial
psychosomatic
psychotherapeutic
psychotherapist
psychotherapy
psychotic
pub
puberty
public
publication
publications
publicity
publicize
publicized
publicizes
publicizing
publicly
publish
published
publisher
publishers
publishes
publishing
pubs
pucker
puckered
puckering
puckers
pudding
puddings
puddle
puddles
puddling
puff
puffed
puffin
puffing
puffs
puke
pull
pulled
puller
pulley
pulleys
pulling
pullings
pullover
pulls
pulmonary
pulp
pulping
pulpit
pulpits
pulsar
pulsate
pulsation
pulsations
pulse
pulsed
pulses
pulsing
puma
pumice
pummel
pump
pumped
pumping
pumpkin
pumpkins
pumps
pun
punch
punched
puncher
punches
punching
punctual
punctually
punctuation
puncture
punctured
punctures
puncturing
pundit
pungent
punish
punishable
punished
punishes
punishing
punishment
punishments
punitive
puns
punt
punted
punting
punts
puny
pup
pupa
pupil
pupils
puppet
puppeteer
puppets
puppies
puppy
pups
purchase
purchased
purchaser
purchasers
purchases
purchasing
pure
purely
purer
purest
purgatory
purge
purged
purges
purging
purification
purifications
purified
purifier
purifiers
purifies
purify
purifying
purist
puritanic
purity
purple
purpler
purplest
purport
purported
purportedly
purporter
purporters
purporting
purports
purpose
purposed
purposeful
purposefully
purposely
purposes
purposive
purr
purred
purring
purrs
purse
pursed
purser
purses
pursuant
pursue
pursued
pursuer
pursuers
pursues
pursuing
pursuit
pursuits
purveyor
purview
pus
push
pushbutton
pushdown
pushed
pusher
pushers
pushes
pushing
puss
pussy
pussycat
put
puts
putt
putter
puttering
putters
putting
putty
puzzle
puzzled
puzzlement
puzzler
puzzlers
puzzles
puzzling
puzzlings
pygmies
pygmy
pyramid
pyramids
pyre
python
qua
quack
quacked
quackery
quacks
quad
quadrangle
quadrangular
quadrant
quadrants
quadratic
quadratical
quadratically
quadratics
quadrature
quadratures
quadrennial
quadrilateral
quadrillion
quadruple
quadrupled
quadruples
quadrupling
quadrupole
quaff
quagmire
quagmires
quahog
quail
quails
quaint
quaintly
quaintness
quake
quaked
quaker
quakers
quakes
quaking
qualification
qualifications
qualified
qualifier
qualifiers
qualifies
qualify
qualifying
qualitative
qualitatively
qualities
quality
qualm
quandaries
quandary
quanta
quantifiable
quantification
quantifications
quantified
quantifier
quantifiers
quantifies
quantify
quantifying
quantile
quantitative
quantitatively
quantities
quantity
quantization
quantize
quantized
quantizes
quantizing
quantum
quarantine
quarantines
quarantining
quark
quarrel
quarreled
quarreling
quarrels
quarrelsome
quarries
quarry
quart
quarter
quarterback
quartered
quartering
quarterly
quartermaster
quarters
quartet
quartets
quartile
quarts
quartz
quartzite
quasar
quash
quashed
quashes
quashing
quasi
quaternary
quaver
quavered
quavering
quavers
quay
queasy
queen
queenly
queens
queer
queerer
queerest
queerly
queerness
quell
quelling
quench
quenched
quenches
quenching
queried
queries
query
querying
quest
quested
quester
questers
questing
question
questionable
questionably
questioned
questioner
questioners
questioning
questioningly
questionings
questionnaire
questionnaires
questions
quests
queue
queued
queueing
queuer
queuers
queues
queuing
quibble
quick
quicken
quickened
quickening
quickens
quicker
quickest
quickie
quicklime
quickly
quickness
quicksand
quicksilver
quiescent
quiet
quieted
quieter
quietest
quieting
quietly
quietness
quiets
quietude
quill
quilt
quilted
quilting
quilts
quince
quinine
quint
quintet
quintillion
quip
quirk
quirky
quit
quite
quits
quitter
quitters
quitting
quiver
quivered
quivering
quivers
quixotic
quiz
quizzed
quizzes
quizzical
quizzing
quo
quonset
quorum
quota
quotas
quotation
quotations
quote
quoted
quotes
quoth
quotient
quotients
quoting
rabbi
rabbit
rabbits
rabble
rabid
rabies
raccoon
raccoons
race
raced
racer
racers
races
racetrack
racial
racially
racing
rack
racked
racket
racketeer
racketeering
racketeers
rackets
racking
racks
radar
radars
radial
radially
radian
radiance
radiant
radiantly
radiate
radiated
radiates
radiating
radiation
radiations
radiator
radiators
radical
radically
radicals
radices
radii
radio
radioactive
radioastronomy
radioed
radiography
radioing
radiology
radios
radish
radishes
radium
radius
radix
radon
raft
rafter
rafters
rafts
rag
rage
raged
rages
ragged
raggedly
raggedness
raging
rags
ragweed
raid
raided
raider
raiders
raiding
raids
rail
railed
railer
railers
railing
railroad
railroaded
railroader
railroaders
railroading
railroads
rails
railway
railways
raiment
rain
rainbow
raincoat
raincoats
raindrop
raindrops
rained
rainfall
rainier
rainiest
raining
rains
rainstorm
rainy
raise
raised
raiser
raisers
raises
raisin
raising
rake
raked
rakes
raking
rallied
rallies
rally
rallying
ram
ramble
rambler
rambles
rambling
ramblings
ramification
ramifications
ramp
rampage
rampant
rampart
ramps
ramrod
rams
ran
ranch
ranched
rancher
ranchers
ranches
ranching
rancid
random
randomization
randomize
randomized
randomizes
randomly
randomness
randy
rang
range
ranged
rangeland
ranger
rangers
ranges
ranging
rangy
rank
ranked
ranker
rankers
rankest
ranking
rankings
rankle
rankly
rankness
ranks
ransack
ransacked
ransacking
ransacks
ransom
ransomer
ransoming
ransoms
rant
ranted
ranter
ranters
ranting
rants
rap
rapacious
rape
raped
raper
rapes
rapid
rapidity
rapidly
rapids
rapier
raping
rapport
rapprochement
raps
rapt
raptly
rapture
raptures
rapturous
rare
rarely
rareness
rarer
rarest
rarity
rascal
rascally
rascals
rash
rasher
rashly
rashness
rasp
raspberry
rasped
rasping
rasps
raster
rat
rate
rated
rater
raters
rates
rather
ratification
ratified
ratifies
ratify
ratifying
rating
ratings
ratio
ration
rational
rationale
rationales
rationalities
rationality
rationalization
rationalizations
rationalize
rationalized
rationalizes
rationalizing
rationally
rationals
rationing
rations
ratios
rats
rattle
rattled
rattler
rattlers
rattles
rattlesnake
rattlesnakes
rattling
raucous
ravage
ravaged
ravager
ravagers
ravages
ravaging
rave
raved
raven
ravening
ravenous
ravenously
ravens
raves
ravine
ravines
raving
ravings
raw
rawer
rawest
rawly
rawness
ray
rays
raze
razor
razors
re
reabbreviate
reabbreviated
reabbreviates
reabbreviating
reach
reachability
reachable
reachably
reached
reacher
reaches
reaching
reacquired
react
reacted
reacting
reaction
reactionaries
reactionary
reactions
reactivate
reactivated
reactivates
reactivating
reactivation
reactive
reactively
reactivity
reactor
reactors
reacts
read
readability
readable
reader
readers
readied
readier
readies
readiest
readily
readiness
reading
readings
readjusted
readout
readouts
reads
ready
readying
real
realest
realign
realigned
realigning
realigns
realism
realist
realistic
realistically
realists
realities
reality
realizable
realizably
realization
realizations
realize
realized
realizes
realizing
reallocate
really
realm
realms
realness
reals
realtor
ream
reanalyze
reanalyzes
reanalyzing
reap
reaped
reaper
reaping
reappear
reappeared
reappearing
reappears
reappraisal
reappraisals
reaps
rear
reared
rearing
rearrange
rearrangeable
rearranged
rearrangement
rearrangements
rearranges
rearranging
rearrest
rearrested
rears
reason
reasonable
reasonableness
reasonably
reasoned
reasoner
reasoning
reasonings
reasons
reassemble
reassembled
reassembles
reassembling
reassembly
reassessment
reassessments
reassign
reassigned
reassigning
reassignment
reassignments
reassigns
reassure
reassured
reassures
reassuring
reawaken
reawakened
reawakening
reawakens
rebate
rebates
rebel
rebelled
rebelling
rebellion
rebellions
rebellious
rebelliously
rebelliousness
rebels
rebind
rebinding
rebinds
reboot
rebooted
rebooting
reboots
rebound
rebounded
rebounding
rebounds
rebroadcast
rebroadcasting
rebroadcasts
rebuff
rebuffed
rebuild
rebuilding
rebuilds
rebuilt
rebuke
rebuked
rebukes
rebuking
rebuttal
rebutted
rebutting
recalcitrant
recalculate
recalculated
recalculates
recalculating
recalculation
recalculations
recalibrate
recalibrated
recalibrates
recalibrating
recall
recalled
recalling
recalls
recant
recapitulate
recapitulated
recapitulates
recapitulation
recapture
recaptured
recaptures
recapturing
recast
recasting
recasts
recede
receded
recedes
receding
receipt
receipts
receivable
receive
received
receiver
receivers
receives
receiving
recent
recently
recentness
receptacle
receptacles
reception
receptionist
receptions
receptive
receptively
receptiveness
receptivity
receptor
recess
recessed
recesses
recession
recessive
recipe
recipes
recipient
recipients
reciprocal
reciprocally
reciprocate
reciprocated
reciprocates
reciprocating
reciprocation
reciprocity
recirculate
recirculated
recirculates
recirculating
recital
recitals
recitation
recitations
recite
recited
reciter
recites
reciting
reckless
recklessly
recklessness
reckon
reckoned
reckoner
reckoning
reckonings
reckons
reclaim
reclaimable
reclaimed
reclaimer
reclaimers
reclaiming
reclaims
reclamation
reclamations
reclassification
reclassified
reclassifies
reclassify
reclassifying
recline
reclining
recode
recoded
recodes
recoding
recognition
recognitions
recognizability
recognizable
recognizably
recognize
recognized
recognizer
recognizers
recognizes
recognizing
recoil
recoiled
recoiling
recoils
recollect
recollected
recollecting
recollection
recollections
recombination
recombine
recombined
recombines
recombining
recommend
recommendation
recommendations
recommended
recommender
recommending
recommends
recompense
recompile
recompiled
recompiles
recompiling
recompute
recomputed
recomputes
recomputing
reconcile
reconciled
reconciler
reconciles
reconciliation
reconciling
reconfigurable
reconfiguration
reconfigurations
reconfigure
reconfigured
reconfigurer
reconfigures
reconfiguring
reconnect
reconnected
reconnecting
reconnection
reconnects
reconsider
reconsideration
reconsidered
reconsidering
reconsiders
reconstituted
reconstruct
reconstructed
reconstructing
reconstruction
reconstructs
reconverted
reconverts
record
recorded
recorder
recorders
recording
recordings
records
recount
recounted
recounting
recounts
recourse
recover
recoverable
recovered
recoveries
recovering
recovers
recovery
recreate
recreated
recreates
recreating
recreation
recreational
recreations
recreative
recruit
recruited
recruiter
recruiting
recruits
recta
rectangle
rectangles
rectangular
rectify
rector
rectors
rectum
rectums
recuperate
recur
recurrence
recurrences
recurrent
recurrently
recurring
recurs
recurse
recursed
recurses
recursing
recursion
recursions
recursive
recursively
recyclable
recycle
recycled
recycles
recycling
red
redbreast
redcoat
redden
reddened
redder
reddest
reddish
reddishness
redeclare
redeclared
redeclares
redeclaring
redeem
redeemed
redeemer
redeemers
redeeming
redeems
redefine
redefined
redefines
redefining
redefinition
redefinitions
redemption
redesign
redesigned
redesigning
redesigns
redevelopment
redhead
redirect
redirected
redirecting
redirection
redirections
redisplay
redisplayed
redisplaying
redisplays
redistribute
redistributed
redistributes
redistributing
redly
redneck
redness
redo
redone
redouble
redoubled
redraw
redrawn
redress
redressed
redresses
redressing
reds
reduce
reduced
reducer
reducers
reduces
reducibility
reducible
reducibly
reducing
reduction
reductions
redundancies
redundancy
redundant
redundantly
redwood
reed
reeds
reeducation
reef
reefer
reefs
reel
reelect
reelected
reelecting
reelects
reeled
reeler
reeling
reels
reemphasize
reemphasized
reemphasizes
reemphasizing
reenabled
reenforcement
reenter
reentered
reentering
reenters
reentrant
reestablish
reestablished
reestablishes
reestablishing
reevaluate
reevaluated
reevaluates
reevaluating
reevaluation
reexamine
reexamined
reexamines
reexamining
reexecuted
refer
referee
refereed
refereeing
referees
reference
referenced
referencer
references
referencing
referenda
referendum
referendums
referent
referential
referentiality
referentially
referents
referral
referrals
referred
referring
refers
refill
refillable
refilled
refilling
refills
refine
refined
refinement
refinements
refiner
refinery
refines
refining
reflect
reflected
reflecting
reflection
reflections
reflective
reflectively
reflectivity
reflector
reflectors
reflects
reflex
reflexes
reflexive
reflexively
reflexiveness
reflexivity
reforestation
reform
reformable
reformat
reformation
reformatory
reformats
reformatted
reformatting
reformed
reformer
reformers
reforming
reforms
reformulate
reformulated
reformulates
reformulating
reformulation
refract
refracted
refraction
refractory
refragment
refrain
refrained
refraining
refrains
refresh
refreshed
refresher
refreshers
refreshes
refreshing
refreshingly
refreshment
refreshments
refrigerate
refrigerator
refrigerators
refuel
refueled
refueling
refuels
refuge
refugee
refugees
refusal
refuse
refused
refuses
refusing
refutable
refutation
refute
refuted
refuter
refutes
refuting
regain
regained
regaining
regains
regal
regaled
regally
regard
regarded
regarding
regardless
regards
regatta
regenerate
regenerated
regenerates
regenerating
regeneration
regenerative
regenerator
regenerators
regent
regents
regime
regimen
regiment
regimentation
regimented
regiments
regimes
region
regional
regionally
regions
register
registered
registering
registers
registrar
registration
registrations
registry
regress
regressed
regresses
regressing
regression
regressions
regressive
regret
regretful
regretfully
regrets
regrettable
regrettably
regretted
regretting
regroup
regrouped
regrouping
regular
regularities
regularity
regularly
regulars
regulate
regulated
regulates
regulating
regulation
regulations
regulative
regulator
regulators
regulatory
rehabilitate
rehearsal
rehearsals
rehearse
rehearsed
rehearser
rehearses
rehearsing
reign
reigned
reigning
reigns
reimbursable
reimburse
reimbursed
reimbursement
reimbursements
rein
reincarnate
reincarnated
reincarnation
reindeer
reined
reinforce
reinforced
reinforcement
reinforcements
reinforcer
reinforces
reinforcing
reinitialize
reinitialized
reinitializing
reins
reinsert
reinserted
reinserting
reinserts
reinstate
reinstated
reinstatement
reinstates
reinstating
reinterpret
reinterpreted
reinterpreting
reinterprets
reintroduce
reintroduced
reintroduces
reintroducing
reinvent
reinvented
reinventing
reinvents
reiterate
reiterated
reiterates
reiterating
reiteration
reject
rejected
rejecting
rejection
rejections
rejector
rejectors
rejects
rejoice
rejoiced
rejoicer
rejoices
rejoicing
rejoin
rejoinder
rejoined
rejoining
rejoins
relabel
relabeled
relabeling
relabelled
relabelling
relabels
relapse
relate
related
relater
relates
relating
relation
relational
relationally
relations
relationship
relationships
relative
relatively
relativeness
relatives
relativism
relativistic
relativistically
relativity
relax
relaxation
relaxations
relaxed
relaxer
relaxes
relaxing
relay
relayed
relaying
relays
release
released
releases
releasing
relegate
relegated
relegates
relegating
relent
relented
relenting
relentless
relentlessly
relentlessness
relents
relevance
relevances
relevant
relevantly
reliability
reliable
reliably
reliance
reliant
relic
relics
relied
relief
relies
relieve
relieved
reliever
relievers
relieves
relieving
religion
religions
religious
religiously
religiousness
relink
relinquish
relinquished
relinquishes
relinquishing
relish
relished
relishes
relishing
relive
relives
reliving
reload
reloaded
reloader
reloading
reloads
relocatable
relocate
relocated
relocates
relocating
relocation
relocations
reluctance
reluctant
reluctantly
rely
relying
remain
remainder
remainders
remained
remaining
remains
remark
remarkable
remarkableness
remarkably
remarked
remarking
remarks
remedial
remedied
remedies
remedy
remedying
remember
remembered
remembering
remembers
remembrance
remembrances
remind
reminded
reminder
reminders
reminding
reminds
reminiscence
reminiscences
reminiscent
reminiscently
remiss
remission
remit
remittance
remnant
remnants
remodel
remodeled
remodeling
remodels
remonstrate
remonstrated
remonstrates
remonstrating
remonstration
remonstrative
remorse
remorseful
remote
remotely
remoteness
remotest
removable
removal
removals
remove
removed
remover
removes
removing
remunerate
remuneration
renaissance
renal
rename
renamed
renames
renaming
rend
render
rendered
rendering
renderings
renders
rendezvous
rending
rendition
renditions
rends
renegade
renegotiable
renew
renewable
renewal
renewed
renewer
renewing
renews
renounce
renounces
renouncing
renovate
renovated
renovation
renown
renowned
rent
rental
rentals
rented
renting
rents
renumber
renumbering
renumbers
renunciate
renunciation
reoccur
reopen
reopened
reopening
reopens
reorder
reordered
reordering
reorders
reorganization
reorganizations
reorganize
reorganized
reorganizes
reorganizing
repackage
repaid
repair
repaired
repairer
repairing
repairman
repairmen
repairs
reparation
reparations
repartee
repartition
repast
repasts
repay
repaying
repays
repeal
repealed
repealer
repealing
repeals
repeat
repeatable
repeated
repeatedly
repeater
repeaters
repeating
repeats
repel
repelled
repellent
repels
repent
repentance
repented
repenting
repents
repercussion
repercussions
repertoire
repertory
repetition
repetitions
repetitious
repetitive
repetitively
repetitiveness
rephrase
rephrased
rephrases
rephrasing
repine
replace
replaceable
replaced
replacement
replacements
replacer
replaces
replacing
replay
replayed
replaying
replays
replenish
replenished
replenishes
replenishing
replete
repleteness
repletion
replica
replicas
replicate
replicated
replicates
replicating
replication
replications
replied
replies
reply
replying
report
reported
reportedly
reporter
reporters
reporting
reports
repose
reposed
reposes
reposing
reposition
repositioned
repositioning
repositions
repositories
repository
reprehensible
represent
representable
representably
representation
representational
representationally
representations
representative
representatively
representativeness
representatives
represented
representing
represents
repress
repressed
represses
repressing
repression
repressions
repressive
reprieve
reprieved
reprieves
reprieving
reprimand
reprint
reprinted
reprinting
reprints
reprisal
reprisals
reproach
reproached
reproaches
reproaching
reprobate
reproduce
reproduced
reproducer
reproducers
reproduces
reproducibilities
reproducibility
reproducible
reproducibly
reproducing
reproduction
reproductions
reprogram
reprogrammed
reprogramming
reprograms
reproof
reprove
reprover
reptile
reptiles
reptilian
republic
republican
republicans
republics
repudiate
repudiated
repudiates
repudiating
repudiation
repudiations
repugnant
repulse
repulsed
repulses
repulsing
repulsion
repulsions
repulsive
reputable
reputably
reputation
reputations
repute
reputed
reputedly
reputes
request
requested
requester
requesters
requesting
requests
require
required
requirement
requirements
requires
requiring
requisite
requisites
requisition
requisitioned
requisitioning
requisitions
reread
reregister
reroute
rerouted
reroutes
rerouting
rerun
reruns
reschedule
rescind
rescue
rescued
rescuer
rescuers
rescues
rescuing
research
researched
researcher
researchers
researches
researching
reselect
reselected
reselecting
reselects
resell
reselling
resemblance
resemblances
resemble
resembled
resembles
resembling
resent
resented
resentful
resentfully
resenting
resentment
resents
reserpine
reservation
reservations
reserve
reserved
reserver
reserves
reserving
reservoir
reservoirs
reset
resets
resetting
resettings
reside
resided
residence
residences
resident
residential
residentially
residents
resides
residing
residual
residue
residues
resign
resignation
resignations
resigned
resigning
resigns
resilient
resin
resins
resist
resistable
resistance
resistances
resistant
resistantly
resisted
resistible
resisting
resistive
resistivity
resistor
resistors
resists
resolute
resolutely
resoluteness
resolution
resolutions
resolvable
resolve
resolved
resolver
resolvers
resolves
resolving
resonance
resonances
resonant
resonate
resort
resorted
resorting
resorts
resound
resounding
resounds
resource
resourceful
resourcefully
resourcefulness
resources
respect
respectability
respectable
respectably
respected
respecter
respectful
respectfully
respectfulness
respecting
respective
respectively
respects
respiration
respirator
respiratory
respite
resplendent
resplendently
respond
responded
respondent
respondents
responder
responding
responds
response
responses
responsibilities
responsibility
responsible
responsibleness
responsibly
responsive
responsively
responsiveness
rest
restart
restarted
restarting
restarts
restate
restated
restatement
restates
restating
restaurant
restaurants
restaurateur
rested
restful
restfully
restfulness
resting
restitution
restive
restless
restlessly
restlessness
restoration
restorations
restore
restored
restorer
restorers
restores
restoring
restrain
restrained
restrainer
restrainers
restraining
restrains
restraint
restraints
restrict
restricted
restricting
restriction
restrictions
restrictive
restrictively
restricts
restroom
restructure
restructured
restructures
restructuring
rests
result
resultant
resultantly
resultants
resulted
resulting
results
resumable
resume
resumed
resumes
resuming
resumption
resumptions
resurgent
resurrect
resurrected
resurrecting
resurrection
resurrections
resurrector
resurrectors
resurrects
resuscitate
resynchronization
resynchronize
resynchronized
resynchronizing
retail
retailer
retailers
retailing
retain
retained
retainer
retainers
retaining
retainment
retains
retaliate
retaliation
retaliatory
retard
retarded
retarder
retarding
retch
retention
retentions
retentive
retentively
retentiveness
reticle
reticles
reticular
reticulate
reticulated
reticulately
reticulates
reticulating
reticulation
retina
retinal
retinas
retinue
retire
retired
retiree
retirement
retirements
retires
retiring
retort
retorted
retorts
retrace
retraced
retraces
retracing
retract
retracted
retracting
retraction
retractions
retracts
retrain
retrained
retraining
retrains
retranslate
retranslated
retransmission
retransmissions
retransmit
retransmits
retransmitted
retransmitting
retreat
retreated
retreating
retreats
retribution
retried
retrier
retriers
retries
retrievable
retrieval
retrievals
retrieve
retrieved
retriever
retrievers
retrieves
retrieving
retroactive
retroactively
retrofit
retrofitting
retrograde
retrospect
retrospection
retrospective
retry
retrying
return
returnable
returned
returner
returning
returns
retype
retyped
retypes
retyping
reunion
reunions
reunite
reunited
reuniting
reusable
reuse
reused
reuses
reusing
revamp
revamped
revamping
revamps
reveal
revealed
revealing
reveals
revel
revelation
revelations
reveled
reveler
reveling
revelry
revels
revenge
revenger
revenue
revenuers
revenues
reverberate
revere
revered
reverence
reverend
reverends
reverent
reverently
reveres
reverie
reverified
reverifies
reverify
reverifying
revering
reversal
reversals
reverse
reversed
reversely
reverser
reverses
reversible
reversing
reversion
revert
reverted
reverting
reverts
review
reviewed
reviewer
reviewers
reviewing
reviews
revile
reviled
reviler
reviling
revise
revised
reviser
revises
revising
revision
revisionary
revisions
revisit
revisited
revisiting
revisits
revival
revivals
revive
revived
reviver
revives
reviving
revocable
revocation
revoke
revoked
revoker
revokes
revoking
revolt
revolted
revolter
revolting
revoltingly
revolts
revolution
revolutionaries
revolutionary
revolutionize
revolutionized
revolutionizer
revolutions
revolve
revolved
revolver
revolvers
revolves
revolving
revulsion
reward
rewarded
rewarding
rewardingly
rewards
rewind
rewinding
rewinds
rewire
rework
reworked
reworking
reworks
rewound
rewrite
rewrites
rewriting
rewritten
rhapsody
rhesus
rhetoric
rheumatic
rheumatism
rhinestone
rhino
rhinoceros
rho
rhododendron
rhombic
rhombus
rhubarb
rhyme
rhymed
rhymes
rhyming
rhythm
rhythmic
rhythmically
rhythms
rib
ribald
ribbed
ribbing
ribbon
ribbons
riboflavin
ribonucleic
ribs
rice
rich
richer
riches
richest
richly
richness
rickets
rickety
rickshaw
rickshaws
ricochet
rid
riddance
ridden
ridding
riddle
riddled
riddles
riddling
ride
rider
riders
rides
ridge
ridgepole
ridges
ridicule
ridiculed
ridicules
ridiculing
ridiculous
ridiculously
ridiculousness
riding
rids
rifle
rifled
rifleman
rifler
rifles
rifling
rift
rig
rigging
right
righted
righteous
righteously
righteousness
righter
rightful
rightfully
rightfulness
righting
rightly
rightmost
rightness
rights
rightward
rigid
rigidity
rigidly
rigor
rigorous
rigorously
rigors
rigs
rill
rim
rime
rims
rind
rinds
ring
ringed
ringer
ringers
ringing
ringingly
ringings
rings
ringside
rink
rinse
rinsed
rinser
rinses
rinsing
riot
rioted
rioter
rioters
rioting
riotous
riots
rip
ripe
ripely
ripen
ripeness
ripoff
ripped
ripping
ripple
rippled
ripples
rippling
rips
rise
risen
riser
risers
rises
rising
risings
risk
risked
risking
risks
risky
rite
rites
ritual
ritually
rituals
rival
rivaled
rivalled
rivalling
rivalries
rivalry
rivals
river
riverbank
riverfront
rivers
riverside
rivet
riveter
rivets
rivulet
rivulets
roach
road
roadbed
roadblock
roads
roadside
roadster
roadsters
roadway
roadways
roam
roamed
roaming
roams
roar
roared
roarer
roaring
roars
roast
roasted
roaster
roasting
roasts
rob
robbed
robber
robberies
robbers
robbery
robbing
robe
robed
robes
robin
robing
robins
robot
robotic
robotics
robots
robs
robust
robustly
robustness
rock
rockabye
rocked
rocker
rockers
rocket
rocketed
rocketing
rockets
rocking
rocks
rocky
rod
rode
rodent
rodents
rodeo
rods
roe
rogue
rogues
role
roles
roll
rollback
rolled
roller
rollers
rolling
rolls
romance
romancer
romancers
romances
romancing
romantic
romantics
romp
romped
romper
romping
romps
roof
roofed
roofer
roofing
roofs
rooftop
rook
rookie
room
roomed
roomer
roomers
roomful
rooming
roommate
rooms
roomy
roost
rooster
roosters
root
rooted
rooter
rooting
roots
rope
roped
roper
ropers
ropes
roping
rosary
rosebud
rosebuds
rosebush
rosemary
roses
rosette
rosiness
roster
rostrum
rosy
rot
rotary
rotate
rotated
rotates
rotating
rotation
rotational
rotations
rotator
rotor
rots
rotten
rottenness
rotting
rotund
rotunda
rouge
rough
roughed
roughen
rougher
roughest
roughly
roughneck
roughness
roulette
round
roundabout
rounded
roundedness
rounder
roundest
roundhead
roundhouse
rounding
roundly
roundness
roundoff
rounds
roundtable
roundup
roundworm
rouse
roused
rouses
rousing
roustabout
rout
route
routed
router
routers
routes
routine
routinely
routines
routing
routings
rove
roved
rover
roves
roving
row
rowboat
rowdy
rowed
rower
rowing
rows
royal
royalist
royalists
royally
royalties
royalty
rub
rubbed
rubber
rubbers
rubbery
rubbing
rubbish
rubble
rubdown
rubies
ruble
rubles
rubout
rubs
ruby
rudder
rudders
ruddiness
ruddy
rude
rudely
rudeness
rudiment
rudimentary
rudiments
rue
ruefully
ruffian
ruffianly
ruffians
ruffle
ruffled
ruffles
rug
rugged
ruggedly
ruggedness
rugs
ruin
ruination
ruinations
ruined
ruining
ruinous
ruinously
ruins
rule
ruled
ruler
rulers
rules
ruling
rulings
rum
rumble
rumbled
rumbler
rumbles
rumbling
rumen
rummage
rummy
rumor
rumored
rumors
rump
rumple
rumpled
rumply
rumpus
run
runaway
rundown
rung
rungs
runnable
runner
runners
running
runoff
runs
runt
runtime
rupee
rupture
ruptured
ruptures
rupturing
rural
rurally
rush
rushed
rusher
rushes
rushing
russet
rust
rusted
rustic
rusticate
rusticated
rusticates
rusticating
rustication
rusting
rustle
rustled
rustler
rustlers
rustling
rusts
rusty
rut
ruthless
ruthlessly
ruthlessness
ruts
rye
sabbath
sabbatical
saber
sabers
sable
sables
sabotage
sack
sacker
sacking
sacks
sacrament
sacred
sacredly
sacredness
sacrifice
sacrificed
sacrificer
sacrificers
sacrifices
sacrificial
sacrificially
sacrificing
sacrilege
sacrilegious
sacrosanct
sad
sadden
saddened
saddens
sadder
saddest
saddle
saddlebag
saddled
saddles
sadism
sadist
sadistic
sadistically
sadists
sadly
sadness
safari
safe
safeguard
safeguarded
safeguarding
safeguards
safekeeping
safely
safeness
safer
safes
safest
safeties
safety
saffron
sag
saga
sagacious
sagacity
sage
sagebrush
sagely
sages
sagging
sagittal
sags
saguaro
said
sail
sailboat
sailed
sailfish
sailing
sailor
sailorly
sailors
sails
saint
sainted
sainthood
saintly
saints
sake
sakes
salable
salad
salads
salamander
salami
salaried
salaries
salary
sale
sales
salesgirl
saleslady
salesman
salesmen
salesperson
salient
saline
saliva
salivary
salivate
sallies
sallow
sallying
salmon
salon
salons
saloon
saloons
salt
salted
salter
salters
saltier
saltiest
saltiness
salting
salts
salty
salutary
salutation
salutations
salute
saluted
salutes
saluting
salvage
salvaged
salvager
salvages
salvaging
salvation
salve
salver
salves
same
sameness
sample
sampled
sampler
samplers
samples
sampling
samplings
sanatoria
sanatorium
sanctification
sanctified
sanctify
sanctimonious
sanction
sanctioned
sanctioning
sanctions
sanctity
sanctuaries
sanctuary
sanctum
sand
sandal
sandals
sandbag
sanded
sander
sanding
sandman
sandpaper
sands
sandstone
sandwich
sandwiches
sandy
sane
sanely
saner
sanest
sang
sanguine
sanitarium
sanitary
sanitation
sanity
sank
sap
sapiens
sapling
saplings
sapphire
saps
sapsucker
sarcasm
sarcasms
sarcastic
sardine
sardonic
sari
sash
sat
satanic
satchel
satchels
sate
sated
satellite
satellites
sates
satin
sating
satire
satires
satiric
satisfaction
satisfactions
satisfactorily
satisfactory
satisfiability
satisfiable
satisfied
satisfies
satisfy
satisfying
saturate
saturated
saturates
saturating
saturation
satyr
sauce
saucepan
saucepans
saucer
saucers
sauces
saucy
saunter
sausage
sausages
savage
savaged
savagely
savageness
savager
savagers
savages
savaging
save
saved
saver
savers
saves
saving
savings
savior
saviors
savor
savored
savoring
savors
savory
saw
sawdust
sawed
sawfish
sawing
sawmill
sawmills
saws
sawtooth
sax
saxophone
say
sayer
sayers
saying
sayings
says
scab
scabbard
scabbards
scabrous
scaffold
scaffolding
scaffoldings
scaffolds
scalable
scalar
scalars
scald
scalded
scalding
scale
scaled
scales
scaling
scalings
scallop
scalloped
scallops
scalp
scalps
scaly
scamper
scampering
scampers
scan
scandal
scandalous
scandals
scanned
scanner
scanners
scanning
scans
scant
scantier
scantiest
scantily
scantiness
scantly
scanty
scapegoat
scar
scarce
scarcely
scarceness
scarcer
scarcity
scare
scarecrow
scared
scares
scarf
scaring
scarlet
scars
scarves
scary
scatter
scatterbrain
scattered
scattering
scatters
scenario
scenarios
scene
scenery
scenes
scenic
scent
scented
scents
scepter
scepters
schedulable
schedule
scheduled
scheduler
schedulers
schedules
scheduling
schema
schemas
schemata
schematic
schematically
schematics
scheme
schemed
schemer
schemers
schemes
scheming
schism
schizophrenia
scholar
scholarly
scholars
scholarship
scholarships
scholastic
scholastically
scholastics
school
schoolboy
schoolboys
schooled
schooler
schoolers
schoolhouse
schoolhouses
schooling
schoolmaster
schoolmasters
schoolroom
schoolrooms
schools
schooner
science
sciences
scientific
scientifically
scientist
scientists
scissor
scissored
scissoring
scissors
sclerosis
sclerotic
scoff
scoffed
scoffer
scoffing
scoffs
scold
scolded
scolding
scolds
scoop
scooped
scooping
scoops
scoot
scope
scoped
scopes
scoping
scorch
scorched
scorcher
scorches
scorching
score
scoreboard
scorecard
scored
scorer
scorers
scores
scoring
scorings
scorn
scorned
scorner
scornful
scornfully
scorning
scorns
scorpion
scorpions
scotch
scoundrel
scoundrels
scour
scoured
scourge
scouring
scours
scout
scouted
scouting
scouts
scow
scowl
scowled
scowling
scowls
scram
scramble
scrambled
scrambler
scrambles
scrambling
scrap
scrape
scraped
scraper
scrapers
scrapes
scraping
scrapings
scrapped
scraps
scratch
scratched
scratcher
scratchers
scratches
scratching
scratchy
scrawl
scrawled
scrawling
scrawls
scrawny
scream
screamed
screamer
screamers
screaming
screams
screech
screeched
screeches
screeching
screen
screened
screening
screenings
screenplay
screens
screw
screwball
screwdriver
screwed
screwing
screws
scribble
scribbled
scribbler
scribbles
scribe
scribes
scribing
scrimmage
script
scripts
scripture
scriptures
scroll
scrolled
scrolling
scrolls
scrounge
scrub
scrumptious
scruple
scrupulous
scrupulously
scrutinize
scrutinized
scrutinizing
scrutiny
scuba
scud
scuffle
scuffled
scuffles
scuffling
sculpt
sculpted
sculptor
sculptors
sculpts
sculpture
sculptured
sculptures
scurried
scurry
scurvy
scuttle
scuttled
scuttles
scuttling
scythe
scythes
sea
seaboard
seacoast
seacoasts
seafood
seagull
seahorse
seal
sealed
sealer
sealing
seals
sealy
seam
seaman
seamed
seamen
seaming
seams
seamy
seaport
seaports
sear
search
searched
searcher
searchers
searches
searching
searchingly
searchings
searchlight
seared
searing
searingly
seas
seashore
seashores
seaside
season
seasonable
seasonably
seasonal
seasonally
seasoned
seasoner
seasoners
seasoning
seasonings
seasons
seat
seated
seating
seats
seaward
seaweed
secant
secede
seceded
secedes
seceding
secession
seclude
secluded
seclusion
second
secondaries
secondarily
secondary
seconded
seconder
seconders
secondhand
seconding
secondly
seconds
secrecy
secret
secretarial
secretariat
secretaries
secretary
secrete
secreted
secretes
secreting
secretion
secretions
secretive
secretively
secretly
secrets
sect
sectarian
section
sectional
sectioned
sectioning
sections
sector
sectors
sects
secular
secure
secured
securely
secures
securing
securings
securities
security
sedan
sedate
sedge
sediment
sedimentary
sediments
sedition
seditious
seduce
seduced
seducer
seducers
seduces
seducing
seduction
seductive
see
seed
seeded
seeder
seeders
seeding
seedings
seedling
seedlings
seeds
seedy
seeing
seek
seeker
seekers
seeking
seeks
seem
seemed
seeming
seemingly
seemly
seems
seen
seep
seepage
seeped
seeping
seeps
seer
seers
seersucker
sees
seethe
seethed
seethes
seething
segment
segmentation
segmentations
segmented
segmenting
segments
segregate
segregated
segregates
segregating
segregation
seismic
seismograph
seismology
seize
seized
seizes
seizing
seizure
seizures
seldom
select
selected
selecting
selection
selections
selective
selectively
selectivity
selectman
selectmen
selector
selectors
selects
selenium
self
selfish
selfishly
selfishness
selfsame
sell
seller
sellers
selling
sellout
sells
seltzer
selves
semantic
semantical
semantically
semanticist
semanticists
semantics
semaphore
semaphores
semblance
semester
semesters
semi
semiautomated
semicolon
semicolons
semiconductor
semiconductors
seminal
seminar
seminarian
seminaries
seminars
seminary
semipermanent
semipermanently
senate
senates
senator
senatorial
senators
send
sender
senders
sending
sends
senile
senior
seniority
seniors
sensation
sensational
sensationally
sensations
sense
sensed
senseless
senselessly
senselessness
senses
sensibilities
sensibility
sensible
sensibly
sensing
sensitive
sensitively
sensitiveness
sensitives
sensitivities
sensitivity
sensor
sensors
sensory
sensual
sensuous
sent
sentence
sentenced
sentences
sentencing
sentential
sentiment
sentimental
sentimentally
sentiments
sentinel
sentinels
sentries
sentry
separable
separate
separated
separately
separateness
separates
separating
separation
separations
separator
separators
sepia
sept
sepulcher
sepulchers
sequel
sequels
sequence
sequenced
sequencer
sequencers
sequences
sequencing
sequencings
sequential
sequentiality
sequentialize
sequentialized
sequentializes
sequentializing
sequentially
sequester
serendipitous
serendipity
serene
serenely
serenity
serf
serfs
sergeant
sergeants
serial
serializability
serializable
serialization
serializations
serialize
serialized
serializes
serializing
serially
serials
series
serif
serious
seriously
seriousness
sermon
sermons
serpent
serpentine
serpents
serum
serums
servant
servants
serve
served
server
servers
serves
service
serviceability
serviceable
serviced
serviceman
servicemen
services
servicing
servile
serving
servings
servitude
servo
servomechanism
sesame
session
sessions
set
setback
sets
settable
setter
setters
setting
settings
settle
settled
settlement
settlements
settler
settlers
settles
settling
setup
setups
seven
sevenfold
sevens
seventeen
seventeens
seventeenth
seventh
seventies
seventieth
seventy
sever
several
severalfold
severally
severance
severe
severed
severely
severer
severest
severing
severities
severity
severs
sew
sewage
sewed
sewer
sewers
sewing
sews
sex
sexed
sexes
sexist
sextet
sextillion
sexton
sextuple
sextuplet
sexual
sexuality
sexually
sexy
shabby
shack
shacked
shackle
shackled
shackles
shackling
shacks
shade
shaded
shades
shadier
shadiest
shadily
shadiness
shading
shadings
shadow
shadowed
shadowing
shadows
shadowy
shady
shaft
shafts
shaggy
shakable
shakably
shake
shakedown
shaken
shaker
shakers
shakes
shakiness
shaking
shaky
shale
shall
shallow
shallower
shallowly
shallowness
sham
shambles
shame
shamed
shameful
shamefully
shameless
shamelessly
shames
shaming
shampoo
shamrock
shams
shanties
shanty
shape
shaped
shapeless
shapelessly
shapelessness
shapely
shaper
shapers
shapes
shaping
sharable
shard
share
shareable
sharecropper
sharecroppers
shared
shareholder
shareholders
sharer
sharers
shares
sharing
shark
sharks
sharp
sharpen
sharpened
sharpening
sharpens
sharper
sharpest
sharply
sharpness
sharpshoot
shatter
shattered
shattering
shatterproof
shatters
shave
shaved
shaven
shaves
shaving
shavings
shawl
shawls
she
sheaf
shear
sheared
shearing
shears
sheath
sheathing
sheaths
sheaves
shed
shedding
sheds
sheen
sheep
sheepskin
sheer
sheered
sheet
sheeted
sheeting
sheets
sheik
shelf
shell
shelled
sheller
shelling
shells
shelter
sheltered
sheltering
shelters
shelve
shelved
shelves
shelving
shenanigan
shepherd
shepherds
sherbet
sheriff
sheriffs
sherry
shibboleth
shied
shield
shielded
shielding
shies
shift
shifted
shifter
shifters
shiftier
shiftiest
shiftily
shiftiness
shifting
shifts
shifty
shill
shilling
shillings
shimmer
shimmering
shin
shinbone
shine
shined
shiner
shiners
shines
shingle
shingles
shining
shiningly
shiny
ship
shipboard
shipbuilding
shipmate
shipment
shipments
shipped
shipper
shippers
shipping
ships
shipshape
shipwreck
shipwrecked
shipwrecks
shipyard
shire
shirk
shirker
shirking
shirks
shirt
shirting
shirts
shit
shiver
shivered
shiverer
shivering
shivers
shoal
shoals
shock
shocked
shocker
shockers
shocking
shockingly
shocks
shod
shoddy
shoe
shoed
shoehorn
shoeing
shoelace
shoemaker
shoes
shoestring
shone
shook
shoot
shooter
shooters
shooting
shootings
shoots
shop
shopkeeper
shopkeepers
shopped
shopper
shoppers
shopping
shops
shopworn
shore
shoreline
shores
shorn
short
shortage
shortages
shortcoming
shortcomings
shortcut
shortcuts
shorted
shorten
shortened
shortening
shortens
shorter
shortest
shortfall
shorthand
shorthanded
shorting
shortish
shortly
shortness
shorts
shortsighted
shortstop
shot
shotgun
shotguns
shots
should
shoulder
shouldered
shouldering
shoulders
shout
shouted
shouter
shouters
shouting
shouts
shove
shoved
shovel
shoveled
shovels
shoves
shoving
show
showboat
showcase
showdown
showed
shower
showered
showering
showers
showing
showings
shown
showpiece
showroom
shows
showy
shrank
shrapnel
shred
shredder
shredding
shreds
shrew
shrewd
shrewdest
shrewdly
shrewdness
shrews
shriek
shrieked
shrieking
shrieks
shrill
shrilled
shrilling
shrillness
shrilly
shrimp
shrine
shrines
shrink
shrinkable
shrinkage
shrinking
shrinks
shrivel
shriveled
shroud
shrouded
shrub
shrubbery
shrubs
shrug
shrugs
shrunk
shrunken
shudder
shuddered
shuddering
shudders
shuffle
shuffleboard
shuffled
shuffles
shuffling
shun
shuns
shunt
shut
shutdown
shutdowns
shutoff
shutout
shuts
shutter
shuttered
shutters
shutting
shuttle
shuttlecock
shuttled
shuttles
shuttling
shy
shyly
shyness
sibling
siblings
sick
sicken
sicker
sickest
sickle
sickly
sickness
sicknesses
sickroom
side
sidearm
sideband
sideboard
sideboards
sideburns
sidecar
sided
sidelight
sidelights
sideline
sidereal
sides
sidesaddle
sideshow
sidestep
sidetrack
sidewalk
sidewalks
sideways
sidewise
siding
sidings
siege
sieges
sierra
sieve
sieves
sift
sifted
sifter
sifting
sigh
sighed
sighing
sighs
sight
sighted
sighting
sightings
sightly
sights
sightseeing
sigma
sign
signal
signaled
signaling
signalled
signalling
signally
signals
signature
signatures
signed
signer
signers
signet
significance
significant
significantly
significants
signification
signified
signifies
signify
signifying
signing
signs
silence
silenced
silencer
silencers
silences
silencing
silent
silently
silhouette
silhouetted
silhouettes
silica
silicate
silicon
silicone
silk
silken
silkier
silkiest
silkily
silks
silky
sill
silliest
silliness
sills
silly
silo
silt
silted
silting
silts
silver
silvered
silvering
silvers
silversmith
silverware
silvery
similar
similarities
similarity
similarly
simile
similitude
simmer
simmered
simmering
simmers
simple
simpleminded
simpleness
simpler
simplest
simpleton
simplex
simplicities
simplicity
simplification
simplifications
simplified
simplifier
simplifiers
simplifies
simplify
simplifying
simplistic
simply
simulate
simulated
simulates
simulating
simulation
simulations
simulator
simulators
simulcast
simultaneity
simultaneous
simultaneously
since
sincere
sincerely
sincerest
sincerity
sine
sines
sinew
sinews
sinewy
sinful
sinfully
sinfulness
sing
singable
singe
singed
singer
singers
singing
singingly
single
singled
singlehanded
singleness
singles
singlet
singleton
singletons
singling
singly
sings
singsong
singular
singularities
singularity
singularly
sinister
sink
sinked
sinker
sinkers
sinkhole
sinking
sinks
sinned
sinner
sinners
sinning
sins
sinuous
sinus
sinusoid
sinusoidal
sinusoids
sip
siphon
siphoning
sipping
sips
sir
sire
sired
siren
sirens
sires
sirs
sirup
sister
sisterly
sisters
sit
site
sited
sites
siting
sits
sitter
sitters
sitting
sittings
situ
situate
situated
situates
situating
situation
situational
situationally
situations
six
sixes
sixfold
sixgun
sixpence
sixteen
sixteens
sixteenth
sixth
sixties
sixtieth
sixty
sizable
size
sized
sizes
sizing
sizings
sizzle
skate
skated
skater
skaters
skates
skating
skeletal
skeleton
skeletons
skeptic
skeptical
skeptically
skepticism
skeptics
sketch
sketchbook
sketched
sketches
sketchily
sketching
sketchpad
sketchy
skew
skewed
skewer
skewers
skewing
skews
ski
skid
skidding
skied
skies
skiff
skiing
skill
skilled
skillet
skillful
skillfully
skillfulness
skills
skim
skimmed
skimming
skimp
skimped
skimping
skimps
skimpy
skims
skin
skindive
skinned
skinner
skinners
skinning
skinny
skins
skip
skipped
skipper
skippers
skipping
skips
skirmish
skirmished
skirmisher
skirmishers
skirmishes
skirmishing
skirt
skirted
skirting
skirts
skis
skit
skulk
skulked
skulker
skulking
skulks
skull
skullcap
skullduggery
skulls
skunk
skunks
sky
skyhook
skyjack
skylark
skylarking
skylarks
skylight
skylights
skyline
skyrockets
skyscraper
skyscrapers
slab
slack
slacken
slacker
slacking
slackly
slackness
slacks
slain
slam
slammed
slamming
slams
slander
slanderer
slanderous
slanders
slang
slant
slanted
slanting
slants
slap
slapped
slapping
slaps
slapstick
slash
slashed
slashes
slashing
slat
slate
slated
slater
slates
slats
slaughter
slaughtered
slaughterhouse
slaughtering
slaughters
slave
slaver
slavery
slaves
slavish
slay
slayer
slayers
slaying
slays
sled
sledding
sledge
sledgehammer
sledges
sleds
sleek
sleep
sleeper
sleepers
sleepily
sleepiness
sleeping
sleepless
sleeplessly
sleeplessness
sleeps
sleepwalk
sleepy
sleet
sleeve
sleeves
sleigh
sleighs
sleight
slender
slenderer
slept
sleuth
slew
slewing
slice
sliced
slicer
slicers
slices
slicing
slick
slicker
slickers
slicks
slid
slide
slider
sliders
slides
sliding
slight
slighted
slighter
slightest
slighting
slightly
slightness
slights
slim
slime
slimed
slimly
slimy
sling
slinging
slings
slingshot
slip
slippage
slipped
slipper
slipperiness
slippers
slippery
slipping
slips
slit
slither
slits
sliver
slob
slogan
slogans
sloop
slop
slope
sloped
sloper
slopers
slopes
sloping
slopped
sloppiness
slopping
sloppy
slops
slot
sloth
slothful
sloths
slots
slotted
slotting
slouch
slouched
slouches
slouching
slow
slowdown
slowed
slower
slowest
slowing
slowly
slowness
slows
sludge
slug
sluggish
sluggishly
sluggishness
slugs
sluice
slum
slumber
slumbered
slumming
slump
slumped
slumps
slums
slung
slur
slurp
slurring
slurry
slurs
sly
slyly
smack
smacked
smacking
smacks
small
smaller
smallest
smallish
smallness
smallpox
smalltime
smart
smarted
smarter
smartest
smartly
smartness
smash
smashed
smasher
smashers
smashes
smashing
smashingly
smattering
smear
smeared
smearing
smears
smell
smelled
smelling
smells
smelly
smelt
smelter
smelts
smile
smiled
smiles
smiling
smilingly
smirk
smite
smith
smithereens
smiths
smithy
smitten
smock
smocking
smocks
smog
smokable
smoke
smoked
smoker
smokers
smokes
smokescreen
smokestack
smokies
smoking
smoky
smolder
smoldered
smoldering
smolders
smooch
smooth
smoothbore
smoothed
smoother
smoothes
smoothest
smoothing
smoothly
smoothness
smote
smother
smothered
smothering
smothers
smudge
smug
smuggle
smuggled
smuggler
smugglers
smuggles
smuggling
smut
smutty
snack
snafu
snag
snail
snails
snake
snaked
snakelike
snakes
snap
snapdragon
snapped
snapper
snappers
snappily
snapping
snappy
snaps
snapshot
snapshots
snare
snared
snares
snaring
snark
snarl
snarled
snarling
snatch
snatched
snatches
snatching
snazzy
sneak
sneaked
sneaker
sneakers
sneakier
sneakiest
sneakily
sneakiness
sneaking
sneaks
sneaky
sneer
sneered
sneering
sneers
sneeze
sneezed
sneezes
sneezing
sniff
sniffed
sniffing
sniffle
sniffs
snifter
snigger
snip
snipe
snippet
snivel
snob
snobbery
snobbish
snoop
snooped
snooping
snoops
snoopy
snore
snored
snores
snoring
snorkel
snort
snorted
snorting
snorts
snotty
snout
snouts
snow
snowball
snowed
snowfall
snowflake
snowier
snowiest
snowily
snowing
snowman
snowmen
snows
snowshoe
snowshoes
snowstorm
snowy
snub
snuff
snuffed
snuffer
snuffing
snuffs
snug
snuggle
snuggled
snuggles
snuggling
snugly
snugness
so
soak
soaked
soaking
soaks
soap
soaped
soaping
soaps
soapy
soar
soared
soaring
soars
sob
sobbing
sober
sobered
sobering
soberly
soberness
sobers
sobriety
sobs
soccer
sociability
sociable
sociably
social
socialism
socialist
socialists
socialize
socialized
socializes
socializing
socially
societal
societies
society
socioeconomic
sociological
sociologically
sociologist
sociologists
sociology
sock
socked
socket
sockets
socking
socks
sod
soda
sodium
sodomy
sods
sofa
sofas
soft
softball
soften
softened
softening
softens
softer
softest
softly
softness
software
softwares
soggy
soil
soiled
soiling
soils
soiree
sojourn
sojourner
sojourners
solace
solaced
solar
sold
solder
soldered
soldier
soldiering
soldierly
soldiers
sole
solely
solemn
solemnity
solemnly
solemnness
solenoid
soles
solicit
solicitation
solicited
soliciting
solicitor
solicitous
solicits
solicitude
solid
solidarity
solidification
solidified
solidifies
solidify
solidifying
solidity
solidly
solidness
solids
soliloquy
solitaire
solitary
solitude
solitudes
solo
solos
solstice
solubility
soluble
solution
solutions
solvable
solve
solved
solvent
solvents
solver
solvers
solves
solving
somatic
somber
somberly
some
somebody
someday
somehow
someone
someplace
somersault
something
sometime
sometimes
somewhat
somewhere
sommelier
somnolent
son
sonar
sonata
song
songbook
songs
sonic
sonnet
sonnets
sonny
sons
soon
sooner
soonest
soot
sooth
soothe
soothed
soother
soothes
soothing
soothsayer
sophisticated
sophistication
sophistry
sophomore
sophomores
soprano
sorcerer
sorcerers
sorcery
sordid
sordidly
sordidness
sore
sorely
soreness
sorer
sores
sorest
sorghum
sorority
sorrel
sorrier
sorriest
sorrow
sorrowful
sorrowfully
sorrows
sorry
sort
sorted
sorter
sorters
sortie
sorting
sorts
sought
soul
soulful
souls
sound
sounded
sounder
soundest
sounding
soundings
soundly
soundness
soundproof
sounds
soup
souped
soups
sour
source
sources
sourdough
soured
sourer
sourest
souring
sourly
sourness
sours
south
southbound
southeast
southeastern
southern
southerner
southerners
southernmost
southland
southpaw
southward
southwest
southwestern
souvenir
sovereign
sovereigns
sovereignty
soviet
soviets
sow
sown
soy
soya
soybean
spa
space
spacecraft
spaced
spacer
spacers
spaces
spaceship
spaceships
spacesuit
spacing
spacings
spacious
spaded
spades
spading
span
spandrel
spaniel
spank
spanked
spanking
spanks
spanned
spanner
spanners
spanning
spans
spare
spared
sparely
spareness
sparer
spares
sparest
sparing
sparingly
spark
sparked
sparking
sparkle
sparkling
sparks
sparring
sparrow
sparrows
sparse
sparsely
sparseness
sparser
sparsest
spasm
spastic
spat
spate
spates
spatial
spatially
spatter
spattered
spatula
spawn
spawned
spawning
spawns
spayed
speak
speakable
speakeasy
speaker
speakers
speaking
speaks
spear
speared
spearmint
spears
spec
special
specialist
specialists
specialization
specializations
specialize
specialized
specializes
specializing
specially
specials
specialties
specialty
specie
species
specifiable
specific
specifically
specification
specifications
specificity
specifics
specified
specifier
specifiers
specifies
specify
specifying
specimen
specimens
specious
speck
speckle
speckled
speckles
specks
spectacle
spectacled
spectacles
spectacular
spectacularly
spectator
spectators
specter
specters
spectra
spectral
spectrogram
spectrograms
spectrograph
spectrographic
spectrography
spectrometer
spectrophotometer
spectrophotometry
spectroscope
spectroscopic
spectroscopy
spectrum
speculate
speculated
speculates
speculating
speculation
speculations
speculative
speculator
speculators
sped
speech
speeches
speechless
speechlessness
speed
speedboat
speeded
speeder
speeders
speedily
speeding
speedometer
speeds
speedup
speedups
speedy
spell
spellbound
spelled
speller
spellers
spelling
spellings
spells
spend
spender
spenders
spending
spends
spent
sperm
sphere
spheres
spherical
spherically
spheroid
spheroidal
sphinx
spice
spiced
spices
spiciness
spicy
spider
spiders
spidery
spies
spigot
spike
spiked
spikes
spill
spilled
spiller
spilling
spills
spilt
spin
spinach
spinal
spinally
spindle
spindled
spindling
spine
spinnaker
spinner
spinners
spinning
spinoff
spins
spinster
spiny
spiral
spiraled
spiraling
spirally
spire
spires
spirit
spirited
spiritedly
spiriting
spirits
spiritual
spiritually
spirituals
spit
spite
spited
spiteful
spitefully
spitefulness
spites
spitfire
spiting
spits
spitting
spittle
splash
splashed
splashes
splashing
splashy
spleen
splendid
splendidly
splendor
splenetic
splice
spliced
splicer
splicers
splices
splicing
splicings
spline
splines
splint
splinter
splintered
splinters
splintery
split
splits
splitter
splitters
splitting
splurge
spoil
spoilage
spoiled
spoiler
spoilers
spoiling
spoils
spoke
spoked
spoken
spokes
spokesman
spokesmen
sponge
sponged
sponger
spongers
sponges
sponging
spongy
sponsor
sponsored
sponsoring
sponsors
sponsorship
spontaneity
spontaneous
spontaneously
spoof
spook
spooky
spool
spooled
spooler
spoolers
spooling
spools
spoon
spooned
spoonful
spooning
spoons
sporadic
spore
spores
sport
sported
sporting
sportingly
sportive
sports
sportsman
sportsmen
sportswear
sportswriter
sportswriting
sporty
spot
spotless
spotlessly
spotlight
spots
spotted
spotter
spotters
spotting
spotty
spouse
spouses
spout
spouted
spouting
spouts
sprain
sprang
sprawl
sprawled
sprawling
sprawls
spray
sprayed
sprayer
spraying
sprays
spread
spreader
spreaders
spreading
spreadings
spreads
spreadsheet
spree
sprees
sprig
sprightly
spring
springboard
springer
springers
springier
springiest
springiness
springing
springs
springtime
springy
sprinkle
sprinkled
sprinkler
sprinkles
sprinkling
sprint
sprinted
sprinter
sprinters
sprinting
sprints
sprite
sprocket
sprout
sprouted
sprouting
spruce
spruced
sprung
spun
spunk
spur
spurious
spurn
spurned
spurning
spurns
spurs
spurt
spurted
spurting
spurts
sputter
sputtered
spy
spyglass
spying
squabble
squabbled
squabbles
squabbling
squad
squadron
squadrons
squads
squalid
squall
squalls
squander
square
squared
squarely
squareness
squarer
squares
squarest
squaring
squash
squashed
squashing
squat
squats
squatting
squaw
squawk
squawked
squawking
squawks
squeak
squeaked
squeaking
squeaks
squeaky
squeal
squealed
squealing
squeals
squeamish
squeeze
squeezed
squeezer
squeezes
squeezing
squelch
squid
squint
squinted
squinting
squire
squires
squirm
squirmed
squirms
squirmy
squirrel
squirreled
squirreling
squirrels
squirt
squishy
stab
stabbed
stabbing
stabile
stabilities
stability
stabilize
stabilized
stabilizer
stabilizers
stabilizes
stabilizing
stable
stabled
stabler
stables
stabling
stably
stabs
stack
stacked
stacking
stacks
stadia
stadium
staff
staffed
staffer
staffers
staffing
staffs
stag
stage
stagecoach
stagecoaches
staged
stager
stagers
stages
stagger
staggered
staggering
staggers
staging
stagnant
stagnate
stagnation
stags
staid
stain
stained
staining
stainless
stains
stair
staircase
staircases
stairs
stairway
stairways
stairwell
stake
staked
stakes
stalactite
stale
stalemate
stalk
stalked
stalking
stall
stalled
stalling
stallings
stallion
stalls
stalwart
stalwartly
stamen
stamens
stamina
stammer
stammered
stammerer
stammering
stammers
stamp
stamped
stampede
stampeded
stampedes
stampeding
stamper
stampers
stamping
stamps
stanch
stanchest
stanchion
stand
standard
standardization
standardize
standardized
standardizes
standardizing
standardly
standards
standby
standing
standings
standoff
standpoint
standpoints
stands
standstill
stanza
stanzas
staphylococcus
staple
stapler
staples
stapling
star
starboard
starch
starched
stardom
stare
stared
starer
stares
starfish
staring
stark
starkly
starlet
starlight
starling
starred
starring
starry
stars
start
started
starter
starters
starting
startle
startled
startles
startling
starts
startup
startups
starvation
starve
starved
starves
starving
state
stated
stately
statement
statements
states
statesman
statesmanlike
statesmen
statewide
static
statically
stating
station
stationary
stationed
stationer
stationery
stationing
stationmaster
stations
statistic
statistical
statistically
statistician
statisticians
statistics
statue
statues
statuesque
statuesquely
statuesqueness
statuette
stature
status
statuses
statute
statutes
statutorily
statutoriness
statutory
staunch
staunchest
staunchly
stave
staved
staves
stay
stayed
staying
stays
stead
steadfast
steadfastly
steadfastness
steadied
steadier
steadies
steadiest
steadily
steadiness
steady
steadying
steak
steaks
steal
stealer
stealing
steals
stealth
stealthily
stealthy
steam
steamboat
steamboats
steamed
steamer
steamers
steaming
steams
steamship
steamships
steamy
steed
steel
steeled
steelers
steeling
steelmaker
steels
steely
steep
steeped
steeper
steepest
steeping
steeple
steeples
steeply
steepness
steeps
steer
steerable
steered
steering
steers
stellar
stem
stemmed
stemming
stems
stench
stenches
stencil
stencils
stenographer
stenographers
stenotype
step
stepchild
stepmother
stepmothers
stepped
stepper
stepping
steps
stepson
stepwise
stereo
stereos
stereoscopic
stereotype
stereotyped
stereotypes
stereotypical
sterile
sterilization
sterilizations
sterilize
sterilized
sterilizer
sterilizes
sterilizing
sterling
stern
sternly
sternness
sterns
stethoscope
stevedore
stew
steward
stewardess
stewards
stewed
stews
stick
sticker
stickers
stickier
stickiest
stickily
stickiness
sticking
stickleback
sticks
sticky
stiff
stiffen
stiffens
stiffer
stiffest
stiffly
stiffness
stiffs
stifle
stifled
stifles
stifling
stigma
stigmata
stile
stiles
stiletto
still
stillbirth
stillborn
stilled
stiller
stillest
stilling
stillness
stills
stilt
stilts
stimulant
stimulants
stimulate
stimulated
stimulates
stimulating
stimulation
stimulations
stimulative
stimuli
stimulus
sting
stinging
stings
stingy
stink
stinker
stinkers
stinking
stinks
stint
stipend
stipends
stipulate
stipulated
stipulates
stipulating
stipulation
stipulations
stir
stirred
stirrer
stirrers
stirring
stirringly
stirrings
stirrup
stirs
stitch
stitched
stitches
stitching
stochastic
stochastically
stock
stockade
stockades
stockbroker
stocked
stocker
stockers
stockholder
stockholders
stocking
stockings
stockpile
stockroom
stocks
stocky
stodgy
stoichiometry
stoke
stole
stolen
stoles
stolid
stomach
stomached
stomacher
stomaches
stomaching
stomp
stoned
stones
stoning
stony
stood
stooge
stool
stoop
stooped
stooping
stoops
stop
stopcock
stopcocks
stopgap
stopover
stoppable
stoppage
stopped
stopper
stoppers
stopping
stops
stopwatch
storage
storages
store
stored
storehouse
storehouses
storekeeper
storeroom
stores
storied
stories
storing
stork
storks
storm
stormed
stormier
stormiest
storminess
storming
storms
stormy
story
storyboard
storyteller
stout
stouter
stoutest
stoutly
stoutness
stove
stoves
stow
stowed
straddle
strafe
straggle
straggled
straggler
stragglers
straggles
straggling
straight
straightaway
straighten
straightened
straightens
straighter
straightest
straightforward
straightforwardly
straightforwardness
straightness
straightway
strain
strained
strainer
strainers
straining
strains
strait
straiten
straits
strand
stranded
stranding
strands
strange
strangely
strangeness
stranger
strangers
strangest
strangle
strangled
strangler
stranglers
strangles
strangling
stranglings
strangulation
strangulations
strap
straps
stratagem
stratagems
strategic
strategies
strategist
strategy
stratification
stratifications
stratified
stratifies
stratify
stratosphere
stratospheric
stratum
straw
strawberries
strawberry
straws
stray
strayed
strays
streak
streaked
streaks
stream
streamed
streamer
streamers
streaming
streamline
streamlined
streamliner
streamlines
streamlining
streams
street
streetcar
streetcars
streeters
streets
strength
strengthen
strengthened
strengthener
strengthening
strengthens
strengths
strenuous
strenuously
streptococcus
stress
stressed
stresses
stressful
stressing
stretch
stretched
stretcher
stretchers
stretches
stretching
strew
strewn
strews
stricken
strict
stricter
strictest
strictly
strictness
stricture
stride
strider
strides
striding
strife
strike
strikebreaker
striker
strikers
strikes
striking
strikingly
string
stringed
stringent
stringently
stringer
stringers
stringier
stringiest
stringiness
stringing
strings
stringy
strip
stripe
striped
stripes
stripped
stripper
strippers
stripping
strips
striptease
strive
striven
strives
striving
strivings
strobe
strobed
strobes
stroboscopic
strode
stroke
stroked
stroker
strokers
strokes
stroking
stroll
strolled
stroller
strolling
strolls
strong
stronger
strongest
stronghold
strongly
strontium
strove
struck
structural
structurally
structure
structured
structurer
structures
structuring
struggle
struggled
struggles
struggling
strung
strut
struts
strutting
strychnine
stub
stubble
stubborn
stubbornly
stubbornness
stubby
stubs
stucco
stuck
stud
student
students
studied
studies
studio
studios
studious
studiously
studs
study
studying
stuff
stuffed
stuffier
stuffiest
stuffing
stuffs
stuffy
stumble
stumbled
stumbles
stumbling
stump
stumped
stumping
stumps
stun
stung
stunning
stunningly
stunt
stunts
stupefy
stupefying
stupendous
stupendously
stupid
stupidest
stupidities
stupidity
stupidly
stupor
sturdiness
sturdy
sturgeon
stutter
style
styled
styler
stylers
styles
styli
styling
stylish
stylishly
stylishness
stylistic
stylistically
stylized
stylus
suave
sub
subatomic
subchannel
subchannels
subclass
subclasses
subcommittees
subcomponent
subcomponents
subcomputation
subcomputations
subconscious
subconsciously
subculture
subcultures
subcycle
subcycles
subdirectories
subdirectory
subdivide
subdivided
subdivides
subdividing
subdivision
subdivisions
subdomains
subdue
subdued
subdues
subduing
subexpression
subexpressions
subfield
subfields
subfile
subfiles
subgoal
subgoals
subgraph
subgraphs
subgroup
subgroups
subinterval
subintervals
subject
subjected
subjecting
subjection
subjective
subjectively
subjectivity
subjects
sublanguage
sublanguages
sublayer
sublayers
sublimation
sublimations
sublime
sublimed
sublist
sublists
submarine
submariner
submariners
submarines
submerge
submerged
submerges
submerging
submission
submissions
submissive
submit
submits
submittal
submitted
submitting
submode
submodes
submodule
submodules
submultiplexed
subnet
subnets
subnetwork
subnetworks
suboptimal
subordinate
subordinated
subordinates
subordination
subparts
subphases
subpoena
subproblem
subproblems
subprocesses
subprogram
subprograms
subproject
subproof
subproofs
subrange
subranges
subroutine
subroutines
subs
subschema
subschemas
subscribe
subscribed
subscriber
subscribers
subscribes
subscribing
subscript
subscripted
subscripting
subscription
subscriptions
subscripts
subsection
subsections
subsegment
subsegments
subsequence
subsequences
subsequent
subsequently
subservient
subset
subsets
subside
subsided
subsides
subsidiaries
subsidiary
subsidies
subsiding
subsidize
subsidized
subsidizes
subsidizing
subsidy
subsist
subsisted
subsistence
subsistent
subsisting
subsists
subslot
subslots
subspace
subspaces
substance
substances
substantial
substantially
substantiate
substantiated
substantiates
substantiating
substantiation
substantiations
substantive
substantively
substantivity
substation
substations
substitutability
substitutable
substitute
substituted
substitutes
substituting
substitution
substitutions
substrate
substrates
substring
substrings
substructure
substructures
subsume
subsumed
subsumes
subsuming
subsystem
subsystems
subtask
subtasks
subterfuge
subterranean
subtitle
subtitled
subtitles
subtle
subtleness
subtler
subtlest
subtleties
subtlety
subtly
subtotal
subtract
subtracted
subtracting
subtraction
subtractions
subtractor
subtractors
subtracts
subtrahend
subtrahends
subtree
subtrees
subunit
subunits
suburb
suburban
suburbia
suburbs
subversion
subversive
subvert
subverted
subverter
subverting
subverts
subway
subways
succeed
succeeded
succeeding
succeeds
success
successes
successful
successfully
succession
successions
successive
successively
successor
successors
succinct
succinctly
succinctness
succor
succumb
succumbed
succumbing
succumbs
such
suck
sucked
sucker
suckers
sucking
suckle
suckling
sucks
suction
sudden
suddenly
suddenness
suds
sudsing
sue
sued
sues
suffer
sufferance
suffered
sufferer
sufferers
suffering
sufferings
suffers
suffice
sufficed
suffices
sufficiency
sufficient
sufficiently
sufficing
suffix
suffixed
suffixer
suffixes
suffixing
suffocate
suffocated
suffocates
suffocating
suffocation
suffrage
suffragette
sugar
sugared
sugaring
sugarings
sugars
suggest
suggested
suggestible
suggesting
suggestion
suggestions
suggestive
suggestively
suggests
suicidal
suicidally
suicide
suicides
suing
suit
suitability
suitable
suitableness
suitably
suitcase
suitcases
suite
suited
suiters
suites
suiting
suitor
suitors
suits
sulfa
sulfur
sulfuric
sulfurous
sulk
sulked
sulkiness
sulking
sulks
sulky
sullen
sullenly
sullenness
sulphate
sulphur
sulphured
sulphuric
sultan
sultans
sultry
sum
sumac
summand
summands
summaries
summarily
summarization
summarizations
summarize
summarized
summarizes
summarizing
summary
summation
summations
summed
summertime
summing
summit
summitry
summon
summoned
summoner
summoners
summoning
summons
summonses
sumptuous
sums
sun
sunbeam
sunbeams
sunbonnet
sunburn
sunburnt
sunder
sundial
sundown
sundries
sundry
sunflower
sung
sunglass
sunglasses
sunk
sunken
sunlight
sunlit
sunned
sunning
sunny
sunrise
suns
sunset
sunshine
sunspot
suntan
suntanned
suntanning
super
superb
superblock
superbly
supercomputer
supercomputers
superego
superegos
superficial
superficially
superfluities
superfluity
superfluous
superfluously
supergroup
supergroups
superhuman
superhumanly
superimpose
superimposed
superimposes
superimposing
superintend
superintendent
superintendents
superior
superiority
superiors
superlative
superlatively
superlatives
supermarket
supermarkets
supermini
superminis
supernatural
superpose
superposed
superposes
superposing
superposition
superscript
superscripted
superscripting
superscripts
supersede
superseded
supersedes
superseding
superset
supersets
superstition
superstitions
superstitious
superuser
supervise
supervised
supervises
supervising
supervision
supervisor
supervisors
supervisory
supine
supper
suppers
supplant
supplanted
supplanting
supplants
supple
supplement
supplemental
supplementary
supplemented
supplementing
supplements
suppleness
supplication
supplied
supplier
suppliers
supplies
supply
supplying
support
supportable
supported
supporter
supporters
supporting
supportingly
supportive
supportively
supports
suppose
supposed
supposedly
supposes
supposing
supposition
suppositions
suppress
suppressed
suppresses
suppressing
suppression
suppressor
suppressors
supranational
supremacy
supreme
supremely
surcharge
sure
surely
sureness
sureties
surety
surf
surface
surfaced
surfaceness
surfaces
surfacing
surge
surged
surgeon
surgeons
surgery
surges
surgical
surgically
surging
surliness
surly
surmise
surmised
surmises
surmount
surmounted
surmounting
surmounts
surname
surnames
surpass
surpassed
surpasses
surpassing
surplus
surpluses
surprise
surprised
surprises
surprising
surprisingly
surreal
surrender
surrendered
surrendering
surrenders
surreptitious
surrey
surrogate
surrogates
surround
surrounded
surrounding
surroundings
surrounds
surtax
survey
surveyed
surveying
surveyor
surveyors
surveys
survival
survivals
survive
survived
survives
surviving
survivor
survivors
susceptible
suspect
suspected
suspecting
suspects
suspend
suspended
suspender
suspenders
suspending
suspends
suspense
suspenses
suspension
suspensions
suspicion
suspicions
suspicious
suspiciously
sustain
sustained
sustaining
sustains
sustenance
suture
sutures
suzerainty
svelte
swab
swabbing
swagger
swaggered
swaggering
swain
swains
swallow
swallowed
swallowing
swallows
swallowtail
swam
swami
swamp
swamped
swamping
swamps
swampy
swan
swank
swanky
swanlike
swans
swap
swapped
swapping
swaps
swarm
swarmed
swarming
swarms
swarthy
swastika
swat
swatted
sway
swayed
swaying
swear
swearer
swearing
swears
sweat
sweated
sweater
sweaters
sweating
sweats
sweatshirt
sweaty
sweep
sweeper
sweepers
sweeping
sweepings
sweeps
sweepstakes
sweet
sweeten
sweetened
sweetener
sweeteners
sweetening
sweetenings
sweetens
sweeter
sweetest
sweetheart
sweethearts
sweetish
sweetly
sweetness
sweets
swell
swelled
swelling
swellings
swells
swelter
swept
swerve
swerved
swerves
swerving
swift
swifter
swiftest
swiftly
swiftness
swim
swimmer
swimmers
swimming
swimmingly
swims
swimsuit
swindle
swine
swing
swinger
swingers
swinging
swings
swipe
swirl
swirled
swirling
swish
swished
swiss
switch
switchblade
switchboard
switchboards
switched
switcher
switchers
switches
switching
switchings
switchman
swivel
swizzle
swollen
swoon
swoop
swooped
swooping
swoops
sword
swordfish
swords
swore
sworn
swum
swung
sycamore
sycophant
sycophantic
syllable
syllables
syllogism
syllogisms
syllogistic
sylvan
symbiosis
symbiotic
symbol
symbolic
symbolically
symbolics
symbolism
symbolization
symbolize
symbolized
symbolizes
symbolizing
symbols
symmetric
symmetrical
symmetrically
symmetries
symmetry
sympathetic
sympathies
sympathize
sympathized
sympathizer
sympathizers
sympathizes
sympathizing
sympathizingly
sympathy
symphonic
symphonies
symphony
symposia
symposium
symposiums
symptom
symptomatic
symptoms
synagogue
synapse
synapses
synaptic
synchronism
synchronization
synchronize
synchronized
synchronizer
synchronizers
synchronizes
synchronizing
synchronous
synchronously
synchrony
synchrotron
syncopate
syndicate
syndicated
syndicates
syndication
syndrome
syndromes
synergism
synergistic
synergy
synod
synonym
synonymous
synonymously
synonyms
synopses
synopsis
syntactic
syntactical
syntactically
syntax
syntaxes
synthesis
synthesize
synthesized
synthesizer
synthesizers
synthesizes
synthesizing
synthetic
synthetics
syringe
syringes
syrup
syrupy
system
systematic
systematically
systematize
systematized
systematizes
systematizing
systemic
systems
systemwide
tab
tabernacle
tabernacles
table
tableau
tableaus
tablecloth
tablecloths
tabled
tables
tablespoon
tablespoonful
tablespoonfuls
tablespoons
tablet
tablets
tabling
taboo
taboos
tabs
tabular
tabulate
tabulated
tabulates
tabulating
tabulation
tabulations
tabulator
tabulators
tachometer
tachometers
tacit
tacitly
tack
tacked
tacking
tackle
tackles
tact
tactic
tactics
tactile
tag
tagged
tagging
tags
tail
tailed
tailing
tailor
tailored
tailoring
tailors
tails
taint
tainted
take
taken
taker
takers
takes
taking
takings
tale
talent
talented
talents
tales
talk
talkative
talkatively
talkativeness
talked
talker
talkers
talkie
talking
talks
tall
taller
tallest
tallness
tallow
tally
tame
tamed
tamely
tameness
tamer
tames
taming
tamper
tampered
tampering
tampers
tan
tandem
tang
tangent
tangential
tangents
tangible
tangibly
tangle
tangled
tangy
tank
tanker
tankers
tanks
tanner
tanners
tantalizing
tantalizingly
tantamount
tantrum
tantrums
tap
tape
taped
taper
tapered
tapering
tapers
tapes
tapestries
tapestry
taping
tapings
tapped
tapper
tappers
tapping
taproot
taproots
taps
tar
tardiness
tardy
target
targeted
targeting
targets
tariff
tariffs
tarry
tart
tartly
tartness
task
tasked
tasking
tasks
tassel
tassels
taste
tasted
tasteful
tastefully
tastefulness
tasteless
tastelessly
taster
tasters
tastes
tasting
tatter
tattered
tattoo
tattooed
tattoos
tau
taught
taunt
taunted
taunter
taunting
taunts
taut
tautly
tautness
tautological
tautologically
tautologies
tautology
tavern
taverns
tawny
tax
taxable
taxation
taxed
taxes
taxi
taxicab
taxicabs
taxied
taxiing
taxing
taxis
taxonomic
taxonomically
taxonomy
taxpayer
taxpayers
tea
teach
teachable
teacher
teachers
teaches
teaching
teachings
teacup
team
teamed
teaming
teams
tear
teared
tearful
tearfully
tearing
tears
teas
tease
teased
teases
teasing
teaspoon
teaspoonful
teaspoonfuls
teaspoons
technical
technicalities
technicality
technically
technician
technicians
technique
techniques
technological
technologically
technologies
technologist
technologists
technology
tedious
tediously
tediousness
tedium
teem
teemed
teeming
teems
teen
teenage
teenaged
teenager
teenagers
teens
teeth
teethe
teethed
teethes
teething
telecommunication
telecommunications
telegram
telegrams
telegraph
telegraphed
telegrapher
telegraphers
telegraphic
telegraphing
telegraphs
telemetry
teleological
teleologically
teleology
telepathy
telephone
telephoned
telephoner
telephoners
telephones
telephonic
telephoning
telephony
teleprocessing
telescope
telescoped
telescopes
telescoping
teletype
teletypes
televise
televised
televises
televising
television
televisions
televisor
televisors
tell
teller
tellers
telling
tells
temper
temperament
temperamental
temperaments
temperance
temperate
temperately
temperateness
temperature
temperatures
tempered
tempering
tempers
tempest
tempestuous
tempestuously
template
templates
temple
temples
temporal
temporally
temporaries
temporarily
temporary
tempt
temptation
temptations
tempted
tempter
tempters
tempting
temptingly
tempts
ten
tenacious
tenaciously
tenant
tenants
tend
tended
tendencies
tendency
tender
tenderly
tenderness
tenders
tending
tends
tenement
tenements
tenfold
tennis
tenor
tenors
tens
tense
tensed
tensely
tenseness
tenser
tenses
tensest
tensing
tension
tensions
tent
tentacle
tentacled
tentacles
tentative
tentatively
tented
tenth
tenting
tents
tenure
term
termed
terminal
terminally
terminals
terminate
terminated
terminates
terminating
termination
terminations
terminator
terminators
terming
terminologies
terminology
terminus
terms
termwise
ternary
terrace
terraced
terraces
terrain
terrains
terrestrial
terrestrials
terrible
terribly
terrier
terriers
terrific
terrified
terrifies
terrify
terrifying
territorial
territories
territory
terror
terrorism
terrorist
terroristic
terrorists
terrorize
terrorized
terrorizes
terrorizing
terrors
tertiary
test
testability
testable
testament
testaments
tested
tester
testers
testicle
testicles
testified
testifier
testifiers
testifies
testify
testifying
testimonies
testimony
testing
testings
tests
text
textbook
textbooks
textile
textiles
texts
textual
textually
texture
textured
textures
than
thank
thanked
thankful
thankfully
thankfulness
thanking
thankless
thanklessly
thanklessness
thanks
thanksgiving
thanksgivings
that
thatch
thatches
thats
thaw
thawed
thawing
thaws
the
theater
theaters
theatrical
theatrically
theatricals
theft
thefts
their
theirs
them
thematic
theme
themes
themselves
then
thence
thenceforth
theological
theology
theorem
theorems
theoretic
theoretical
theoretically
theoreticians
theories
theorist
theorists
theorization
theorizations
theorize
theorized
theorizer
theorizers
theorizes
theorizing
theory
therapeutic
therapies
therapist
therapists
therapy
there
thereabouts
thereafter
thereby
therefore
therein
thereof
thereon
thereto
thereupon
therewith
thermal
thermodynamic
thermodynamics
thermometer
thermometers
thermostat
thermostats
these
theses
thesis
they
thick
thicken
thickens
thicker
thickest
thicket
thickets
thickly
thickness
thief
thieve
thieves
thieving
thigh
thighs
thimble
thimbles
thin
thing
things
think
thinkable
thinkably
thinker
thinkers
thinking
thinks
thinly
thinner
thinness
thinnest
third
thirdly
thirds
thirst
thirsted
thirsts
thirsty
thirteen
thirteens
thirteenth
thirties
thirtieth
thirty
this
thistle
thong
thorn
thorns
thorny
thorough
thoroughfare
thoroughfares
thoroughly
thoroughness
those
though
thought
thoughtful
thoughtfully
thoughtfulness
thoughtless
thoughtlessly
thoughtlessness
thoughts
thousand
thousands
thousandth
thrash
thrashed
thrasher
thrashes
thrashing
thread
threaded
threader
threaders
threading
threads
threat
threaten
threatened
threatening
threatens
threats
three
threefold
threes
threescore
threshold
thresholds
threw
thrice
thrift
thrifty
thrill
thrilled
thriller
thrillers
thrilling
thrillingly
thrills
thrive
thrived
thrives
thriving
throat
throated
throats
throb
throbbed
throbbing
throbs
throne
thrones
throng
throngs
throttle
throttled
throttles
throttling
through
throughout
throughput
throw
thrower
throwing
thrown
throws
thrush
thrust
thruster
thrusters
thrusting
thrusts
thud
thuds
thug
thugs
thumb
thumbed
thumbing
thumbs
thump
thumped
thumping
thunder
thunderbolt
thunderbolts
thundered
thunderer
thunderers
thundering
thunders
thunderstorm
thunderstorms
thus
thusly
thwart
thwarted
thwarting
thwarts
thyself
tick
ticked
ticker
tickers
ticket
tickets
ticking
tickle
tickled
tickles
tickling
ticklish
ticks
tidal
tidally
tide
tided
tides
tidied
tidiness
tiding
tidings
tidy
tidying
tie
tied
tier
tiers
ties
tiger
tigers
tight
tighten
tightened
tightener
tighteners
tightening
tightenings
tightens
tighter
tightest
tightly
tightness
tilde
tile
tiled
tiles
tiling
till
tillable
tilled
tiller
tillers
tilling
tills
tilt
tilted
tilting
tilts
timber
timbered
timbering
timbers
time
timed
timeless
timelessly
timelessness
timely
timeout
timeouts
timer
timers
times
timeshare
timeshares
timesharing
timestamp
timestamps
timetable
timetables
timid
timidity
timidly
timing
timings
tin
tincture
tinge
tinged
tingle
tingled
tingles
tingling
tinier
tiniest
tinily
tininess
tinker
tinkered
tinkering
tinkers
tinkle
tinkled
tinkles
tinkling
tinnier
tinniest
tinnily
tinniness
tinny
tins
tint
tinted
tinting
tints
tiny
tip
tipped
tipper
tippers
tipping
tips
tiptoe
tire
tired
tiredly
tireless
tirelessly
tirelessness
tires
tiresome
tiresomely
tiresomeness
tiring
tissue
tissues
tit
tithe
tither
tithes
tithing
title
titled
titles
tits
titter
titters
to
toad
toads
toast
toasted
toaster
toasting
toasts
tobacco
today
todays
toe
toes
together
togetherness
toggle
toggled
toggles
toggling
toil
toiled
toiler
toilet
toilets
toiling
toils
token
tokens
told
tolerability
tolerable
tolerably
tolerance
tolerances
tolerant
tolerantly
tolerate
tolerated
tolerates
tolerating
toleration
toll
tolled
tolls
tomahawk
tomahawks
tomato
tomatoes
tomb
tombs
tomography
tomorrow
tomorrows
ton
tone
toned
toner
tones
tongs
tongue
tongued
tongues
tonic
tonics
tonight
toning
tonnage
tons
tonsil
too
took
tool
tooled
tooler
toolers
tooling
tools
tooth
toothbrush
toothbrushes
toothpaste
toothpick
toothpicks
top
toper
topic
topical
topically
topics
topmost
topography
topological
topologies
topology
topple
toppled
topples
toppling
tops
torch
torches
tore
torment
tormented
tormenter
tormenters
tormenting
torn
tornado
tornadoes
torpedo
torpedoes
torque
torrent
torrents
torrid
tortoise
tortoises
torture
tortured
torturer
torturers
tortures
torturing
torus
toruses
toss
tossed
tosses
tossing
total
totaled
totaling
totalities
totality
totalled
totaller
totallers
totalling
totally
totals
totter
tottered
tottering
totters
touch
touchable
touched
touches
touchier
touchiest
touchily
touchiness
touching
touchingly
touchy
tough
toughen
tougher
toughest
toughly
toughness
tour
toured
touring
tourist
tourists
tournament
tournaments
tours
tow
toward
towards
towed
towel
toweling
towelled
towelling
towels
tower
towered
towering
towers
town
towns
township
townships
toy
toyed
toying
toys
trace
traceable
traced
tracer
tracers
traces
tracing
tracings
track
tracked
tracker
trackers
tracking
tracks
tract
tractability
tractable
tractive
tractor
tractors
tracts
trade
traded
trademark
trademarks
tradeoff
tradeoffs
trader
traders
trades
tradesman
trading
tradition
traditional
traditionally
traditions
traffic
trafficked
trafficker
traffickers
trafficking
traffics
tragedies
tragedy
tragic
tragically
trail
trailed
trailer
trailers
trailing
trailings
trails
train
trained
trainee
trainees
trainer
trainers
training
trains
trait
traitor
traitors
traits
trajectories
trajectory
tramp
tramped
tramping
trample
trampled
trampler
tramples
trampling
tramps
trance
trances
tranquil
tranquility
tranquilly
transact
transaction
transactions
transatlantic
transceive
transceiver
transceivers
transcend
transcended
transcendent
transcending
transcends
transcontinental
transcribe
transcribed
transcriber
transcribers
transcribes
transcribing
transcript
transcription
transcriptions
transcripts
transfer
transferability
transferable
transferal
transferals
transference
transferred
transferrer
transferrers
transferring
transfers
transfinite
transform
transformable
transformation
transformational
transformations
transformed
transformer
transformers
transforming
transforms
transgress
transgressed
transgression
transgressions
transience
transiency
transient
transiently
transients
transistor
transistorize
transistorized
transistorizing
transistors
transit
transition
transitional
transitioned
transitions
transitive
transitively
transitiveness
transitivity
transitory
translatability
translatable
translate
translated
translates
translating
translation
translational
translations
translator
translators
translucent
transmission
transmissions
transmit
transmits
transmittal
transmitted
transmitter
transmitters
transmitting
transmogrification
transmogrify
transpacific
transparencies
transparency
transparent
transparently
transpire
transpired
transpires
transpiring
transplant
transplanted
transplanting
transplants
transponder
transponders
transport
transportability
transportation
transported
transporter
transporters
transporting
transports
transpose
transposed
transposes
transposing
transposition
trap
trapezoid
trapezoidal
trapezoids
trapped
trapper
trappers
trapping
trappings
traps
trash
trauma
traumatic
travail
travel
traveled
traveler
travelers
traveling
travelings
travels
traversal
traversals
traverse
traversed
traverses
traversing
travesties
travesty
tray
trays
treacheries
treacherous
treacherously
treachery
tread
treading
treads
treason
treasure
treasured
treasurer
treasures
treasuries
treasuring
treasury
treat
treated
treaties
treating
treatise
treatises
treatment
treatments
treats
treaty
treble
tree
trees
treetop
treetops
trek
treks
tremble
trembled
trembles
trembling
tremendous
tremendously
tremor
tremors
trench
trencher
trenches
trend
trending
trends
trespass
trespassed
trespasser
trespassers
trespasses
tress
tresses
trial
trials
triangle
triangles
triangular
triangularly
tribal
tribe
tribes
tribunal
tribunals
tribune
tribunes
tributary
tribute
tributes
trichotomy
trick
tricked
trickier
trickiest
trickiness
tricking
trickle
trickled
trickles
trickling
tricks
tricky
tried
trier
triers
tries
trifle
trifler
trifles
trifling
trigger
triggered
triggering
triggers
trigonometric
trigonometry
trigram
trigrams
trihedral
trilateral
trill
trilled
trillion
trillions
trillionth
trim
trimly
trimmed
trimmer
trimmest
trimming
trimmings
trimness
trims
trinket
trinkets
trio
trip
triple
tripled
triples
triplet
triplets
tripling
tripod
trips
triumph
triumphal
triumphant
triumphantly
triumphed
triumphing
triumphs
trivia
trivial
trivialities
triviality
trivially
trod
troll
trolley
trolleys
trolls
troop
trooper
troopers
troops
trophies
trophy
tropic
tropical
tropics
trot
trots
trouble
troubled
troublemaker
troublemakers
troubles
troubleshoot
troubleshooter
troubleshooters
troubleshooting
troubleshoots
troublesome
troublesomely
troubling
trough
trouser
trousers
trout
trowel
trowels
truant
truants
truce
truck
trucked
trucker
truckers
trucking
trucks
trudge
trudged
true
trued
truer
trues
truest
truing
truism
truisms
truly
trump
trumped
trumpet
trumpeter
trumps
truncate
truncated
truncates
truncating
truncation
truncations
trunk
trunks
trust
trusted
trustee
trustees
trustful
trustfully
trustfulness
trusting
trustingly
trusts
trustworthiness
trustworthy
trusty
truth
truthful
truthfully
truthfulness
truths
try
trying
tub
tube
tuber
tuberculosis
tubers
tubes
tubing
tubs
tuck
tucked
tucking
tucks
tuft
tufts
tug
tugs
tuition
tulip
tulips
tumble
tumbled
tumbler
tumblers
tumbles
tumbling
tumor
tumors
tumult
tumults
tumultuous
tunable
tune
tuned
tuner
tuners
tunes
tunic
tunics
tuning
tunnel
tunneled
tunnels
tuple
tuples
turban
turbans
turbulence
turbulent
turbulently
turf
turgid
turgidly
turkey
turkeys
turmoil
turmoils
turn
turnable
turnaround
turned
turner
turners
turning
turnings
turnip
turnips
turnover
turns
turpentine
turquoise
turret
turrets
turtle
turtleneck
turtles
tutor
tutored
tutorial
tutorials
tutoring
tutors
twain
twang
twas
tweed
twelfth
twelve
twelves
twenties
twentieth
twenty
twice
twig
twigs
twilight
twilights
twill
twin
twine
twined
twiner
twinkle
twinkled
twinkler
twinkles
twinkling
twins
twirl
twirled
twirler
twirling
twirls
twist
twisted
twister
twisters
twisting
twists
twitch
twitched
twitching
twitter
twittered
twittering
two
twofold
twos
tying
type
typed
typeout
types
typesetter
typewriter
typewriters
typhoid
typical
typically
typicalness
typified
typifies
typify
typifying
typing
typist
typists
typo
typographic
typographical
typographically
typography
tyrannical
tyranny
tyrant
tyrants
ubiquitous
ubiquitously
ubiquity
ugh
uglier
ugliest
ugliness
ugly
ulcer
ulcers
ultimate
ultimately
ultra
ultrasonic
umbrage
umbrella
umbrellas
umpire
umpires
unabated
unabbreviated
unable
unacceptability
unacceptable
unacceptably
unaccountable
unaccustomed
unachievable
unacknowledged
unadulterated
unaesthetically
unaffected
unaffectedly
unaffectedness
unaided
unalienability
unalienable
unalterably
unaltered
unambiguous
unambiguously
unambitious
unanalyzable
unanimity
unanimous
unanimously
unanswerable
unanswered
unanticipated
unarmed
unary
unassailable
unassigned
unassisted
unattainability
unattainable
unattended
unattractive
unattractively
unauthorized
unavailability
unavailable
unavoidable
unavoidably
unaware
unawareness
unawares
unbalanced
unbearable
unbecoming
unbelievable
unbiased
unbind
unblock
unblocked
unblocking
unblocks
unborn
unbound
unbounded
unbreakable
unbridled
unbroken
unbuffered
uncancelled
uncanny
uncapitalized
uncaught
uncertain
uncertainly
uncertainties
uncertainty
unchangeable
unchanged
unchanging
unclaimed
unclassified
uncle
unclean
uncleanly
uncleanness
unclear
uncleared
uncles
unclosed
uncomfortable
uncomfortably
uncommitted
uncommon
uncommonly
uncompromising
uncomputable
unconcerned
unconcernedly
unconditional
unconditionally
unconnected
unconscionable
unconscious
unconsciously
unconsciousness
unconstitutional
unconstrained
uncontrollability
uncontrollable
uncontrollably
uncontrolled
unconventional
unconventionally
unconvinced
unconvincing
uncoordinated
uncorrectable
uncorrected
uncountable
uncountably
uncouth
uncover
uncovered
uncovering
uncovers
undamaged
undaunted
undauntedly
undecidable
undecided
undeclared
undecomposable
undefinability
undefined
undeleted
undeniable
undeniably
under
underbrush
underdone
underestimate
underestimated
underestimates
underestimating
underestimation
underflow
underflowed
underflowing
underflows
underfoot
undergo
undergoes
undergoing
undergone
undergraduate
undergraduates
underground
underlie
underlies
underline
underlined
underlines
underling
underlings
underlining
underlinings
underloaded
underlying
undermine
undermined
undermines
undermining
underneath
underpinning
underpinnings
underplay
underplayed
underplaying
underplays
underscore
underscored
underscores
understand
understandability
understandable
understandably
understanding
understandingly
understandings
understands
understated
understood
undertake
undertaken
undertaker
undertakers
undertakes
undertaking
undertakings
undertook
underwater
underway
underwear
underwent
underworld
underwrite
underwriter
underwriters
underwrites
underwriting
undesirability
undesirable
undetectable
undetected
undetermined
undeveloped
undid
undiminished
undirected
undisciplined
undiscovered
undisturbed
undivided
undo
undocumented
undoes
undoing
undoings
undone
undoubtedly
undress
undressed
undresses
undressing
undue
unduly
uneasily
uneasiness
uneasy
uneconomic
uneconomical
unembellished
unemployed
unemployment
unencrypted
unending
unenlightening
unequal
unequaled
unequally
unequivocal
unequivocally
unessential
unevaluated
uneven
unevenly
unevenness
uneventful
unexcused
unexpanded
unexpected
unexpectedly
unexplained
unexplored
unextended
unfair
unfairly
unfairness
unfaithful
unfaithfully
unfaithfulness
unfamiliar
unfamiliarity
unfamiliarly
unfavorable
unfettered
unfinished
unfit
unfitness
unflagging
unfold
unfolded
unfolding
unfolds
unforeseen
unforgeable
unforgiving
unformatted
unfortunate
unfortunately
unfortunates
unfounded
unfriendliness
unfriendly
unfulfilled
ungrammatical
ungrateful
ungratefully
ungratefulness
ungrounded
unguarded
unguided
unhappier
unhappiest
unhappily
unhappiness
unhappy
unharmed
unhealthy
unheard
unheeded
unicorn
unicorns
unicycle
unidentified
unidirectional
unidirectionality
unidirectionally
unification
unifications
unified
unifier
unifiers
unifies
uniform
uniformed
uniformity
uniformly
uniforms
unify
unifying
unilluminating
unimaginable
unimpeded
unimplemented
unimportant
unindented
uninitialized
uninsulated
unintelligible
unintended
unintentional
unintentionally
uninteresting
uninterestingly
uninterpreted
uninterrupted
uninterruptedly
union
unionization
unionize
unionized
unionizer
unionizers
unionizes
unionizing
unions
uniprocessor
unique
uniquely
uniqueness
unison
unit
unite
united
unites
unities
uniting
units
unity
univalve
univalves
universal
universality
universally
universals
universe
universes
universities
university
unjust
unjustifiable
unjustified
unjustly
unkind
unkindly
unkindness
unknowable
unknowing
unknowingly
unknown
unknowns
unlabelled
unlawful
unlawfully
unleash
unleashed
unleashes
unleashing
unless
unlike
unlikely
unlikeness
unlimited
unlink
unlinked
unlinking
unlinks
unload
unloaded
unloading
unloads
unlock
unlocked
unlocking
unlocks
unlucky
unmanageable
unmanageably
unmanned
unmarked
unmarried
unmask
unmasked
unmatched
unmentionable
unmerciful
unmercifully
unmistakable
unmistakably
unmodified
unmoved
unnamed
unnatural
unnaturally
unnaturalness
unnecessarily
unnecessary
unneeded
unnerve
unnerved
unnerves
unnerving
unnoticed
unobservable
unobserved
unobtainable
unoccupied
unofficial
unofficially
unopened
unordered
unpack
unpacked
unpacking
unpacks
unpaid
unparalleled
unparsed
unplanned
unpleasant
unpleasantly
unpleasantness
unplug
unpopular
unpopularity
unprecedented
unpredictable
unpredictably
unprescribed
unpreserved
unprimed
unprofitable
unprojected
unprotected
unprovability
unprovable
unproven
unpublished
unqualified
unqualifiedly
unquestionably
unquestioned
unquoted
unravel
unraveled
unraveling
unravels
unreachable
unreal
unrealistic
unrealistically
unreasonable
unreasonableness
unreasonably
unrecognizable
unrecognized
unregulated
unrelated
unreliability
unreliable
unreported
unrepresentable
unresolved
unresponsive
unrest
unrestrained
unrestricted
unrestrictedly
unrestrictive
unroll
unrolled
unrolling
unrolls
unruly
unsafe
unsafely
unsanitary
unsatisfactory
unsatisfiability
unsatisfiable
unsatisfied
unsatisfying
unscrupulous
unseeded
unseen
unselected
unselfish
unselfishly
unselfishness
unsent
unsettled
unsettling
unshaken
unshared
unsigned
unskilled
unslotted
unsolvable
unsolved
unsophisticated
unsound
unspeakable
unspecified
unstable
unsteadiness
unsteady
unstructured
unsuccessful
unsuccessfully
unsuitable
unsuited
unsupported
unsure
unsurprising
unsurprisingly
unsynchronized
untagged
untapped
untenable
unterminated
untested
unthinkable
unthinking
untidiness
untidy
untie
untied
unties
until
untimely
unto
untold
untouchable
untouchables
untouched
untoward
untrained
untranslated
untreated
untried
untrue
untruthful
untruthfulness
untying
unusable
unused
unusual
unusually
unvarying
unveil
unveiled
unveiling
unveils
unwanted
unwelcome
unwholesome
unwieldiness
unwieldy
unwilling
unwillingly
unwillingness
unwind
unwinder
unwinders
unwinding
unwinds
unwise
unwisely
unwiser
unwisest
unwitting
unwittingly
unworthiness
unworthy
unwound
unwrap
unwrapped
unwrapping
unwraps
unwritten
up
upbraid
upcoming
update
updated
updater
updates
updating
upgrade
upgraded
upgrades
upgrading
upheld
uphill
uphold
upholder
upholders
upholding
upholds
upholster
upholstered
upholsterer
upholstering
upholsters
upkeep
upland
uplands
uplift
uplink
uplinks
upload
upon
upper
uppermost
upright
uprightly
uprightness
uprising
uprisings
uproar
uproot
uprooted
uprooting
uproots
upset
upsets
upshot
upshots
upside
upstairs
upstream
upturn
upturned
upturning
upturns
upward
upwards
urban
urchin
urchins
urge
urged
urgent
urgently
urges
urging
urgings
urinate
urinated
urinates
urinating
urination
urine
urn
urns
us
usability
usable
usably
usage
usages
use
used
useful
usefully
usefulness
useless
uselessly
uselessness
user
users
uses
usher
ushered
ushering
ushers
using
usual
usually
usurp
usurped
usurper
utensil
utensils
utilities
utility
utilization
utilizations
utilize
utilized
utilizes
utilizing
utmost
utopia
utopian
utopians
utter
utterance
utterances
uttered
uttering
utterly
uttermost
utters
vacancies
vacancy
vacant
vacantly
vacate
vacated
vacates
vacating
vacation
vacationed
vacationer
vacationers
vacationing
vacations
vacuo
vacuous
vacuously
vacuum
vacuumed
vacuuming
vagabond
vagabonds
vagaries
vagary
vagina
vaginas
vagrant
vagrantly
vague
vaguely
vagueness
vaguer
vaguest
vain
vainly
vale
valence
valences
valentine
valentines
vales
valet
valets
valiant
valiantly
valid
validate
validated
validates
validating
validation
validity
validly
validness
valley
valleys
valor
valuable
valuables
valuably
valuation
valuations
value
valued
valuer
valuers
values
valuing
valve
valves
vampire
van
vandalize
vandalized
vandalizes
vandalizing
vane
vanes
vanguard
vanilla
vanish
vanished
vanisher
vanishes
vanishing
vanishingly
vanities
vanity
vanquish
vanquished
vanquishes
vanquishing
vans
vantage
vapor
vaporing
vapors
variability
variable
variableness
variables
variably
variance
variances
variant
variantly
variants
variation
variations
varied
varies
varieties
variety
various
variously
varnish
varnishes
vary
varying
varyings
vase
vases
vassal
vast
vaster
vastest
vastly
vastness
vat
vats
vaudeville
vault
vaulted
vaulter
vaulting
vaults
vaunt
vaunted
veal
vector
vectorization
vectorizing
vectors
veer
veered
veering
veers
vegetable
vegetables
vegetarian
vegetarians
vegetate
vegetated
vegetates
vegetating
vegetation
vegetative
vehemence
vehement
vehemently
vehicle
vehicles
vehicular
veil
veiled
veiling
veils
vein
veined
veining
veins
velocities
velocity
velvet
vendor
vendors
venerable
veneration
vengeance
venial
venison
venom
venomous
venomously
vent
vented
ventilate
ventilated
ventilates
ventilating
ventilation
ventricle
ventricles
vents
venture
ventured
venturer
venturers
ventures
venturing
venturings
veracity
veranda
verandas
verb
verbal
verbalize
verbalized
verbalizes
verbalizing
verbally
verbose
verbs
verdict
verdure
verge
verger
verges
verifiability
verifiable
verification
verifications
verified
verifier
verifiers
verifies
verify
verifying
verily
veritable
vermin
vernacular
versa
versatile
versatility
verse
versed
verses
versing
version
versions
versus
vertebrate
vertebrates
vertex
vertical
vertically
verticalness
vertices
very
vessel
vessels
vest
vested
vestige
vestiges
vestigial
vests
veteran
veterans
veterinarian
veterinarians
veterinary
veto
vetoed
vetoer
vetoes
vex
vexation
vexed
vexes
vexing
via
viability
viable
viably
vial
vials
vibrate
vibrated
vibrating
vibration
vibrations
vibrator
vice
viceroy
vices
vicinity
vicious
viciously
viciousness
vicissitude
vicissitudes
victim
victimize
victimized
victimizer
victimizers
victimizes
victimizing
victims
victor
victories
victorious
victoriously
victors
victory
victual
victualer
victuals
video
videotape
videotapes
vie
vied
vier
vies
view
viewable
viewed
viewer
viewers
viewing
viewpoint
viewpoints
views
vigilance
vigilant
vigilante
vigilantes
vigilantly
vignette
vignettes
vigor
vigorous
vigorously
vile
vilely
vileness
vilification
vilifications
vilified
vilifies
vilify
vilifying
villa
village
villager
villagers
villages
villain
villainous
villainously
villainousness
villains
villainy
villas
vindicate
vindicated
vindication
vindictive
vindictively
vindictiveness
vine
vinegar
vines
vineyard
vineyards
vintage
violate
violated
violates
violating
violation
violations
violator
violators
violence
violent
violently
violet
violets
violin
violinist
violinists
violins
viper
vipers
virgin
virginity
virgins
virtual
virtually
virtue
virtues
virtuoso
virtuosos
virtuous
virtuously
virulent
virus
viruses
visa
visage
visas
viscount
viscounts
viscous
visibility
visible
visibly
vision
visionary
visions
visit
visitation
visitations
visited
visiting
visitor
visitors
visits
visor
visors
vista
vistas
visual
visualize
visualized
visualizer
visualizes
visualizing
visually
vita
vitae
vital
vitality
vitally
vitals
vivid
vividly
vividness
vizier
vocabularies
vocabulary
vocal
vocally
vocals
vocation
vocational
vocationally
vocations
vogue
voice
voiced
voicer
voicers
voices
voicing
void
voided
voider
voiding
voids
volatile
volatilities
volatility
volcanic
volcano
volcanos
volition
volley
volleyball
volleyballs
volt
voltage
voltages
volts
volume
volumes
voluntarily
voluntary
volunteer
volunteered
volunteering
volunteers
vomit
vomited
vomiting
vomits
vortex
vote
voted
voter
voters
votes
voting
votive
vouch
voucher
vouchers
vouches
vouching
vow
vowed
vowel
vowels
vower
vowing
vows
voyage
voyaged
voyager
voyagers
voyages
voyaging
voyagings
vulgar
vulgarly
vulnerabilities
vulnerability
vulnerable
vulture
vultures
wacky
wade
waded
wader
wades
wading
wafer
wafers
waffle
waffles
waft
wag
wage
waged
wager
wagers
wages
waging
wagon
wagoner
wagons
wags
wail
wailed
wailing
wails
waist
waistcoat
waistcoats
waists
wait
waited
waiter
waiters
waiting
waitress
waitresses
waits
waive
waived
waiver
waiverable
waives
waiving
wake
waked
waken
wakened
wakening
wakes
wakeup
waking
wales
walk
walked
walker
walkers
walking
walks
wall
walled
wallet
wallets
walling
wallow
wallowed
wallowing
wallows
walnut
walnuts
walrus
walruses
waltz
waltzed
waltzes
waltzing
wan
wand
wander
wandered
wanderer
wanderers
wandering
wanderings
wanders
wane
waned
wanes
waning
wanly
want
wanted
wanting
wanton
wantonly
wantonness
wants
war
warble
warbled
warbler
warbles
warbling
ward
warden
wardens
warder
wardrobe
wardrobes
wards
ware
warehouse
warehouses
warehousing
wares
warfare
warily
wariness
warlike
warm
warmed
warmer
warmers
warmest
warming
warmly
warms
warmth
warn
warned
warner
warning
warningly
warnings
warns
warp
warped
warping
warps
warrant
warranted
warranties
warranting
warrants
warranty
warred
warring
warrior
warriors
wars
warship
warships
wart
wartime
warts
wary
was
wash
washed
washer
washers
washes
washing
washings
wasp
wasps
waste
wasted
wasteful
wastefully
wastefulness
wastes
wasting
watch
watched
watcher
watchers
watches
watchful
watchfully
watchfulness
watching
watchings
watchman
watchword
watchwords
water
watered
waterfall
waterfalls
watering
waterings
waterproof
waterproofing
waterway
waterways
watery
wave
waved
waveform
waveforms
wavefront
wavefronts
waveguides
wavelength
wavelengths
waver
wavers
waves
waving
wax
waxed
waxen
waxer
waxers
waxes
waxing
waxy
way
ways
wayside
wayward
we
weak
weaken
weakened
weakening
weakens
weaker
weakest
weakly
weakness
weaknesses
wealth
wealthiest
wealths
wealthy
wean
weaned
weaning
weapon
weapons
wear
wearable
wearer
wearied
wearier
weariest
wearily
weariness
wearing
wearisome
wearisomely
wears
weary
wearying
weasel
weasels
weather
weathercock
weathercocks
weathered
weathering
weathers
weave
weaver
weaves
weaving
web
webs
wedded
wedding
weddings
wedge
wedged
wedges
wedging
wedlock
weds
wee
weed
weeds
week
weekend
weekends
weekly
weep
weeper
weeping
weeps
weigh
weighed
weighing
weighings
weighs
weight
weighted
weighting
weights
weighty
weird
weirdly
welcome
welcomed
welcomes
welcoming
weld
welded
welder
welding
welds
welfare
well
welled
welling
welsh
wench
wenches
went
wept
were
west
westbound
western
westerner
westerners
westward
westwards
wet
wetly
wetness
wets
wetted
wetter
wettest
wetting
whack
whacked
whacking
whacks
whale
whaler
whales
whaling
wharf
wharves
what
whatever
whatsoever
wheat
wheaten
wheel
wheeled
wheeler
wheelers
wheeling
wheelings
wheels
whelp
when
whence
whenever
where
whereabouts
whereas
whereby
wherein
whereupon
wherever
whether
which
whichever
while
whim
whimper
whimpered
whimpering
whimpers
whims
whimsical
whimsically
whimsies
whimsy
whine
whined
whines
whining
whip
whipped
whipper
whippers
whipping
whippings
whips
whirl
whirled
whirling
whirlpool
whirlpools
whirls
whirlwind
whirr
whirring
whisk
whisked
whisker
whiskers
whiskey
whisking
whisks
whisper
whispered
whispering
whisperings
whispers
whistle
whistled
whistler
whistlers
whistles
whistling
whit
white
whitely
whiten
whitened
whitener
whiteners
whiteness
whitening
whitens
whiter
whites
whitespace
whitest
whitewash
whitewashed
whiting
whittle
whittled
whittles
whittling
whiz
whizzed
whizzes
whizzing
who
whoever
whole
wholehearted
wholeheartedly
wholeness
wholes
wholesale
wholesaler
wholesalers
wholesome
wholesomeness
wholly
whom
whomever
whoop
whooped
whooping
whoops
whore
whores
whorl
whorls
whose
why
wick
wicked
wickedly
wickedness
wicker
wicks
wide
wideband
widely
widen
widened
widener
widening
widens
wider
widespread
widest
widget
widow
widowed
widower
widowers
widows
width
widths
wield
wielded
wielder
wielding
wields
wife
wifely
wig
wigs
wigwam
wild
wildcat
wildcats
wilder
wilderness
wildest
wildly
wildness
wile
wiles
wiliness
will
willed
willful
willfully
willing
willingly
willingness
willow
willows
wilt
wilted
wilting
wilts
wily
win
wince
winced
winces
wincing
wind
winded
winder
winders
winding
windmill
windmills
window
windows
winds
windy
wine
wined
winer
winers
wines
wing
winged
winging
wings
wining
wink
winked
winker
winking
winks
winner
winners
winning
winningly
winnings
wins
winter
wintered
wintering
wintry
wipe
wiped
wiper
wipers
wipes
wiping
wire
wired
wireless
wires
wiretap
wiretappers
wiretapping
wiretaps
wiriness
wiring
wiry
wisdom
wisdoms
wise
wised
wisely
wiser
wisest
wish
wished
wisher
wishers
wishes
wishful
wishing
wisp
wisps
wistful
wistfully
wistfulness
wit
witch
witchcraft
witches
witching
with
withal
withdraw
withdrawal
withdrawals
withdrawing
withdrawn
withdraws
withdrew
wither
withers
withheld
withhold
withholder
withholders
withholding
withholdings
withholds
within
without
withstand
withstanding
withstands
withstood
witness
witnessed
witnesses
witnessing
wits
witty
wives
wizard
wizards
woe
woeful
woefully
woke
wolf
wolves
woman
womanhood
womanly
womb
wombs
women
won
wonder
wondered
wonderful
wonderfully
wonderfulness
wondering
wonderingly
wonderment
wonders
wondrous
wondrously
wont
wonted
woo
wood
woodchuck
woodchucks
woodcock
woodcocks
wooded
wooden
woodenly
woodenness
woodland
woodman
woodpecker
woodpeckers
woodwork
woodworking
woody
wooed
wooer
woof
woofed
woofer
woofers
woofing
woofs
wooing
wool
woolen
woolly
wools
woos
word
worded
wordily
wordiness
wording
words
wordy
wore
work
workable
workably
workbench
workbenches
workbook
workbooks
worked
worker
workers
workhorse
workhorses
working
workingman
workings
workload
workman
workmanship
workmen
works
workshop
workshops
workspace
workstation
workstations
world
worldliness
worldly
worlds
worldwide
worm
wormed
worming
worms
worn
worried
worrier
worriers
worries
worrisome
worry
worrying
worryingly
worse
worship
worshiped
worshiper
worshipful
worshiping
worships
worst
worsted
worth
worthiest
worthiness
worthless
worthlessness
worths
worthwhile
worthwhileness
worthy
would
wound
wounded
wounding
wounds
wove
woven
wrangle
wrangled
wrangler
wrap
wraparound
wrapped
wrapper
wrappers
wrapping
wrappings
wraps
wrath
wreak
wreaks
wreath
wreathed
wreathes
wreck
wreckage
wrecked
wrecker
wreckers
wrecking
wrecks
wren
wrench
wrenched
wrenches
wrenching
wrens
wrest
wrestle
wrestler
wrestles
wrestling
wrestlings
wretch
wretched
wretchedness
wretches
wriggle
wriggled
wriggler
wriggles
wriggling
wring
wringer
wrings
wrinkle
wrinkled
wrinkles
wrist
wrists
wristwatch
wristwatches
writ
writable
write
writer
writers
writes
writhe
writhed
writhes
writhing
writing
writings
writs
written
wrong
wronged
wronging
wrongly
wrongs
wrote
wrought
wrung
yank
yanked
yanking
yanks
yard
yards
yardstick
yardsticks
yarn
yarns
yawn
yawner
yawning
yea
year
yearly
yearn
yearned
yearning
yearnings
years
yeas
yeast
yeasts
yell
yelled
yeller
yelling
yellow
yellowed
yellower
yellowest
yellowing
yellowish
yellowness
yellows
yelp
yelped
yelping
yelps
yeoman
yeomen
yes
yesterday
yesterdays
yet
yield
yielded
yielding
yields
yoke
yokes
yon
yonder
you
young
younger
youngest
youngly
youngster
youngsters
your
yours
yourself
yourselves
youth
youthes
youthful
youthfully
youthfulness
zeal
zealous
zealously
zealousness
zebra
zebras
zenith
zero
zeroed
zeroes
zeroing
zeros
zeroth
zest
zigzag
zillions
zinc
zodiac
zonal
zonally
zone
zoned
zones
zoning
zoo
zoological
zoologically
zoom
zooms
zoos
zuul
EOF

rm -f $input; touch $input

i=0
while test $i -lt 10;do cat $temp >>$input; i=`expr $i + 1`;done

cat <<'EOF' >$temp
aback
abaft
abandon
abandoned
abandoning
abandonment
abandons
abase
abased
abasement
abasements
abases
abash
abashed
abashes
abashing
abasing
abate
abated
abatement
abatements
abater
abates
abating
abbe
abbey
abbeys
abbot
abbots
abbreviate
abbreviated
abbreviates
abbreviating
abbreviation
abbreviations
abdomen
abdomens
abdominal
abduct
abducted
abduction
abductions
abductor
abductors
abducts
abed
aberrant
aberration
aberrations
abet
abets
abetted
abetter
abetting
abeyance
abhor
abhorred
abhorrent
abhorrer
abhorring
abhors
abide
abided
abides
abiding
abilities
ability
abject
abjection
abjections
abjectly
abjectness
abjure
abjured
abjures
abjuring
ablate
ablated
ablates
ablating
ablation
ablative
ablaze
able
abler
ablest
ably
abnormal
abnormalities
abnormality
abnormally
aboard
abode
abodes
abolish
abolished
abolisher
abolishers
abolishes
abolishing
abolishment
abolishments
abolition
abolitionist
abolitionists
abominable
abominate
aboriginal
aborigine
aborigines
abort
aborted
aborting
abortion
abortions
abortive
abortively
aborts
abound
abounded
abounding
abounds
about
above
aboveboard
aboveground
abovementioned
abrade
abraded
abrades
abrading
abrasion
abrasions
abrasive
abreaction
abreactions
abreast
abridge
abridged
abridges
abridging
abridgment
abroad
abrogate
abrogated
abrogates
abrogating
abrupt
abruptly
abruptness
abscess
abscessed
abscesses
abscissa
abscissas
abscond
absconded
absconding
absconds
absence
absences
absent
absented
absentee
absenteeism
absentees
absentia
absenting
absently
absentminded
absents
absinthe
absolute
absolutely
absoluteness
absolutes
absolution
absolve
absolved
absolves
absolving
absorb
absorbed
absorbency
absorbent
absorber
absorbing
absorbs
absorption
absorptions
absorptive
abstain
abstained
abstainer
abstaining
abstains
abstention
abstentions
abstinence
abstract
abstracted
abstracting
abstraction
abstractionism
abstractionist
abstractions
abstractly
abstractness
abstractor
abstractors
abstracts
abstruse
abstruseness
absurd
absurdities
absurdity
absurdly
abundance
abundant
abundantly
abuse
abused
abuses
abusing
abusive
abut
abutment
abuts
abutted
abutter
abutters
abutting
abysmal
abysmally
abyss
abysses
acacia
academia
academic
academically
academics
academies
academy
accede
acceded
accedes
accelerate
accelerated
accelerates
accelerating
acceleration
accelerations
accelerator
accelerators
accelerometer
accelerometers
accent
accented
accenting
accents
accentual
accentuate
accentuated
accentuates
accentuating
accentuation
accept
acceptability
acceptable
acceptably
acceptance
acceptances
accepted
accepter
accepters
accepting
acceptor
acceptors
accepts
access
accessed
accesses
accessibility
accessible
accessibly
accessing
accession
accessions
accessories
accessors
accessory
accident
accidental
accidentally
accidently
accidents
acclaim
acclaimed
acclaiming
acclaims
acclamation
acclimate
acclimated
acclimates
acclimating
acclimatization
acclimatized
accolade
accolades
accommodate
accommodated
accommodates
accommodating
accommodation
accommodations
accompanied
accompanies
accompaniment
accompaniments
accompanist
accompanists
accompany
accompanying
accomplice
accomplices
accomplish
accomplished
accomplisher
accomplishers
accomplishes
accomplishing
accomplishment
accomplishments
accord
accordance
accorded
accorder
accorders
according
accordingly
accordion
accordions
accords
accost
accosted
accosting
accosts
account
accountability
accountable
accountably
accountancy
accountant
accountants
accounted
accounting
accounts
accredit
accreditation
accreditations
accredited
accretion
accretions
accrue
accrued
accrues
accruing
acculturate
acculturated
acculturates
acculturating
acculturation
accumulate
accumulated
accumulates
accumulating
accumulation
accumulations
accumulator
accumulators
accuracies
accuracy
accurate
accurately
accurateness
accursed
accusal
accusation
accusations
accusative
accuse
accused
accuser
accuses
accusing
accusingly
accustom
accustomed
accustoming
accustoms
ace
aces
acetate
acetone
acetylene
ache
ached
aches
achievable
achieve
achieved
achievement
achievements
achiever
achievers
achieves
achieving
aching
acid
acidic
acidities
acidity
acidly
acids
acidulous
acknowledge
acknowledgeable
acknowledged
acknowledgement
acknowledgements
acknowledger
acknowledgers
acknowledges
acknowledging
acknowledgment
acknowledgments
acme
acne
acolyte
acolytes
acorn
acorns
acoustic
acoustical
acoustically
acoustician
acoustics
acquaint
acquaintance
acquaintances
acquainted
acquainting
acquaints
acquiesce
acquiesced
acquiescence
acquiescent
acquiesces
acquiescing
acquirable
acquire
acquired
acquires
acquiring
acquisition
acquisitions
acquisitive
acquisitiveness
acquit
acquits
acquittal
acquitted
acquitter
acquitting
acre
acreage
acres
acrid
acrimonious
acrimony
acrobat
acrobatic
acrobatics
acrobats
acronym
acronyms
acropolis
across
acrylic
act
acted
acting
actinium
actinometer
actinometers
action
actions
activate
activated
activates
activating
activation
activations
activator
activators
active
actively
activism
activist
activists
activities
activity
actor
actors
actress
actresses
actual
actualities
actuality
actualization
actually
actuals
actuarial
actuarially
actuate
actuated
actuates
actuating
actuator
actuators
acuity
acumen
acute
acutely
acuteness
acyclic
acyclically
ad
adage
adages
adagio
adagios
adamant
adamantly
adapt
adaptability
adaptable
adaptation
adaptations
adapted
adapter
adapters
adapting
adaptive
adaptively
adaptor
adaptors
adapts
add
added
addend
addenda
addendum
adder
adders
addict
addicted
addicting
addiction
addictions
addicts
adding
addition
additional
additionally
additions
additive
additives
additivity
address
addressability
addressable
addressed
addressee
addressees
addresser
addressers
addresses
addressing
adds
adduce
adduced
adduces
adducible
adducing
adduct
adducted
adducting
adduction
adductor
adducts
adept
adequacies
adequacy
adequate
adequately
adhere
adhered
adherence
adherent
adherents
adherer
adherers
adheres
adhering
adhesion
adhesions
adhesive
adhesives
adiabatic
adiabatically
adieu
adjacency
adjacent
adjective
adjectives
adjoin
adjoined
adjoining
adjoins
adjourn
adjourned
adjourning
adjournment
adjourns
adjudge
adjudged
adjudges
adjudging
adjudicate
adjudicated
adjudicates
adjudicating
adjudication
adjudications
adjunct
adjuncts
adjure
adjured
adjures
adjuring
adjust
adjustable
adjustably
adjusted
adjuster
adjusters
adjusting
adjustment
adjustments
adjustor
adjustors
adjusts
adjutant
adjutants
administer
administered
administering
administerings
administers
administrable
administrate
administration
administrations
administrative
administratively
administrator
administrators
admirable
admirably
admiral
admirals
admiralty
admiration
admirations
admire
admired
admirer
admirers
admires
admiring
admiringly
admissibility
admissible
admission
admissions
admit
admits
admittance
admitted
admittedly
admitter
admitters
admitting
admix
admixed
admixes
admixture
admonish
admonished
admonishes
admonishing
admonishment
admonishments
admonition
admonitions
ado
adobe
adolescence
adolescent
adolescents
adopt
adopted
adopter
adopters
adopting
adoption
adoptions
adoptive
adopts
adorable
adoration
adore
adored
adores
adorn
adorned
adornment
adornments
adorns
adrenal
adrenaline
adrift
adroit
adroitness
ads
adsorb
adsorbed
adsorbing
adsorbs
adsorption
adulate
adulating
adulation
adult
adulterate
adulterated
adulterates
adulterating
adulterer
adulterers
adulterous
adulterously
adultery
adulthood
adults
adumbrate
adumbrated
adumbrates
adumbrating
adumbration
advance
advanced
advancement
advancements
advances
advancing
advantage
advantaged
advantageous
advantageously
advantages
advent
adventist
adventists
adventitious
adventure
adventured
adventurer
adventurers
adventures
adventuring
adventurous
adverb
adverbial
adverbs
adversaries
adversary
adverse
adversely
adversities
adversity
advert
advertise
advertised
advertisement
advertisements
advertiser
advertisers
advertises
advertising
advice
advisability
advisable
advisably
advise
advised
advisedly
advisee
advisees
advisement
advisements
adviser
advisers
advises
advising
advisor
advisors
advisory
advocacy
advocate
advocated
advocates
advocating
aegis
aerate
aerated
aerates
aerating
aeration
aerator
aerators
aerial
aerials
aeroacoustic
aerobic
aerobics
aerodynamic
aerodynamics
aeronautic
aeronautical
aeronautics
aerosol
aerosolize
aerosols
aerospace
aesthetic
aesthetically
aesthetics
afar
affable
affair
affairs
affect
affectation
affectations
affected
affecting
affectingly
affection
affectionate
affectionately
affections
affective
affects
afferent
affianced
affidavit
affidavits
affiliate
affiliated
affiliates
affiliating
affiliation
affiliations
affinities
affinity
affirm
affirmation
affirmations
affirmative
affirmatively
affirmed
affirming
affirms
affix
affixed
affixes
affixing
afflict
afflicted
afflicting
affliction
afflictions
afflictive
afflicts
affluence
affluent
afford
affordable
afforded
affording
affords
affricate
affricates
affright
affront
affronted
affronting
affronts
aficionado
afield
afire
aflame
afloat
afoot
afore
aforementioned
aforesaid
aforethought
afoul
afraid
afresh
aft
after
aftereffect
afterglow
afterimage
afterlife
aftermath
aftermost
afternoon
afternoons
aftershock
aftershocks
afterthought
afterthoughts
afterward
afterwards
again
against
agape
agar
agate
agates
age
aged
ageless
agencies
agency
agenda
agendas
agent
agents
ager
agers
ages
agglomerate
agglomerated
agglomerates
agglomeration
agglutinate
agglutinated
agglutinates
agglutinating
agglutination
agglutinin
agglutinins
aggrandize
aggravate
aggravated
aggravates
aggravation
aggregate
aggregated
aggregately
aggregates
aggregating
aggregation
aggregations
aggression
aggressions
aggressive
aggressively
aggressiveness
aggressor
aggressors
aggrieve
aggrieved
aggrieves
aggrieving
aghast
agile
agilely
agility
aging
agitate
agitated
agitates
agitating
agitation
agitations
agitator
agitators
agleam
aglow
agnostic
agnostics
ago
agog
agonies
agonize
agonized
agonizes
agonizing
agonizingly
agony
agrarian
agree
agreeable
agreeably
agreed
agreeing
agreement
agreements
agreer
agreers
agrees
agricultural
agriculturally
agriculture
ague
ah
ahead
aid
aide
aided
aiding
aids
ail
aileron
ailerons
ailing
ailment
ailments
aim
aimed
aimer
aimers
aiming
aimless
aimlessly
aims
air
airbag
airbags
airborne
aircraft
airdrop
airdrops
aired
airer
airers
airfare
airfield
airfields
airflow
airfoil
airfoils
airframe
airframes
airily
airing
airings
airless
airlift
airlifts
airline
airliner
airlines
airlock
airlocks
airmail
airmails
airman
airmen
airplane
airplanes
airport
airports
airs
airship
airships
airspace
airspeed
airstrip
airstrips
airtight
airway
airways
airy
aisle
ajar
akimbo
akin
alabaster
alacrity
alarm
alarmed
alarming
alarmingly
alarmist
alarms
alas
alba
albacore
albatross
albeit
album
albumin
albums
alchemy
alcohol
alcoholic
alcoholics
alcoholism
alcohols
alcove
alcoves
alder
alderman
aldermen
ale
alee
alert
alerted
alertedly
alerter
alerters
alerting
alertly
alertness
alerts
alfalfa
alfresco
alga
algae
algaecide
algebra
algebraic
algebraically
algebras
alginate
algorithm
algorithmic
algorithmically
algorithms
alias
aliased
aliases
aliasing
alibi
alibis
alien
alienate
alienated
alienates
alienating
alienation
aliens
alight
align
aligned
aligning
alignment
alignments
aligns
alike
aliment
aliments
alimony
alive
alkali
alkaline
alkalis
alkaloid
alkaloids
alkyl
all
allay
allayed
allaying
allays
allegation
allegations
allege
alleged
allegedly
alleges
allegiance
allegiances
alleging
allegoric
allegorical
allegorically
allegories
allegory
allegretto
allegrettos
allele
alleles
allemande
allergic
allergies
allergy
alleviate
alleviated
alleviates
alleviating
alleviation
alley
alleys
alleyway
alleyways
alliance
alliances
allied
allies
alligator
alligators
alliteration
alliterations
alliterative
allocatable
allocate
allocated
allocates
allocating
allocation
allocations
allocator
allocators
allophone
allophones
allophonic
allot
allotment
allotments
allots
allotted
allotter
allotting
allow
allowable
allowably
allowance
allowances
allowed
allowing
allows
alloy
alloys
allude
alluded
alludes
alluding
allure
allurement
alluring
allusion
allusions
allusive
allusiveness
ally
allying
alma
almanac
almanacs
almighty
almond
almonds
almoner
almost
alms
almsman
alnico
aloe
aloes
aloft
aloha
alone
aloneness
along
alongside
aloof
aloofness
aloud
alpha
alphabet
alphabetic
alphabetical
alphabetically
alphabetics
alphabetize
alphabetized
alphabetizes
alphabetizing
alphabets
alphanumeric
alpine
already
also
altar
altars
alter
alterable
alteration
alterations
altercation
altercations
altered
alterer
alterers
altering
alternate
alternated
alternately
alternates
alternating
alternation
alternations
alternative
alternatively
alternatives
alternator
alternators
alters
although
altitude
altitudes
altogether
altruism
altruist
altruistic
altruistically
alum
aluminum
alumna
alumnae
alumni
alumnus
alundum
alveolar
alveoli
alveolus
always
am
amain
amalgam
amalgamate
amalgamated
amalgamates
amalgamating
amalgamation
amalgams
amanuensis
amaretto
amass
amassed
amasses
amassing
amateur
amateurish
amateurishness
amateurism
amateurs
amatory
amaze
amazed
amazedly
amazement
amazer
amazers
amazes
amazing
amazingly
ambassador
ambassadors
amber
ambiance
ambidextrous
ambidextrously
ambient
ambiguities
ambiguity
ambiguous
ambiguously
ambition
ambitions
ambitious
ambitiously
ambivalence
ambivalent
ambivalently
amble
ambled
ambler
ambles
ambling
ambrosial
ambulance
ambulances
ambulatory
ambuscade
ambush
ambushed
ambushes
ameliorate
ameliorated
ameliorating
amelioration
amen
amenable
amend
amended
amending
amendment
amendments
amends
amenities
amenity
amenorrhea
americium
amiable
amicable
amicably
amid
amide
amidst
amigo
amino
amiss
amity
ammo
ammonia
ammoniac
ammonium
ammunition
amnesty
amoeba
amoebae
amoebas
amok
among
amongst
amoral
amorality
amorist
amorous
amorphous
amorphously
amortize
amortized
amortizes
amortizing
amount
amounted
amounter
amounters
amounting
amounts
amour
amperage
ampere
amperes
ampersand
ampersands
amphetamine
amphetamines
amphibian
amphibians
amphibious
amphibiously
amphibology
amphitheater
amphitheaters
ample
amplification
amplified
amplifier
amplifiers
amplifies
amplify
amplifying
amplitude
amplitudes
amply
ampoule
ampoules
amputate
amputated
amputates
amputating
amulet
amulets
amuse
amused
amusedly
amusement
amusements
amuser
amusers
amuses
amusing
amusingly
amyl
an
anachronism
anachronisms
anachronistically
anaconda
anacondas
anaerobic
anagram
anagrams
anal
analog
analogical
analogies
analogous
analogously
analogue
analogues
analogy
analyses
analysis
analyst
analysts
analytic
analytical
analytically
analyticities
analyticity
analyzable
analyze
analyzed
analyzer
analyzers
analyzes
analyzing
anaphora
anaphoric
anaphorically
anaplasmosis
anarchic
anarchical
anarchism
anarchist
anarchists
anarchy
anastomoses
anastomosis
anastomotic
anathema
anatomic
anatomical
anatomically
anatomy
ancestor
ancestors
ancestral
ancestry
anchor
anchorage
anchorages
anchored
anchoring
anchorite
anchoritism
anchors
anchovies
anchovy
ancient
anciently
ancients
ancillary
and
anders
anding
anecdotal
anecdote
anecdotes
anechoic
anemia
anemic
anemometer
anemometers
anemometry
anemone
anesthesia
anesthetic
anesthetically
anesthetics
anesthetize
anesthetized
anesthetizes
anesthetizing
anew
angel
angelic
angels
anger
angered
angering
angers
angiography
angle
angled
angler
anglers
angling
angrier
angriest
angrily
angry
angst
angstrom
anguish
anguished
angular
angularly
anhydrous
anhydrously
aniline
animal
animals
animate
animated
animatedly
animately
animateness
animates
animating
animation
animations
animator
animators
animism
animized
animosity
anion
anionic
anions
anise
aniseikonic
anisotropic
anisotropy
ankle
ankles
annal
annals
annex
annexation
annexed
annexes
annexing
annihilate
annihilated
annihilates
annihilating
annihilation
anniversaries
anniversary
annotate
annotated
annotates
annotating
annotation
annotations
announce
announced
announcement
announcements
announcer
announcers
announces
announcing
annoy
annoyance
annoyances
annoyed
annoyer
annoyers
annoying
annoyingly
annoys
annual
annually
annuals
annuity
annul
annular
annuli
annulled
annulling
annulment
annulments
annuls
annulus
annum
annunciate
annunciated
annunciates
annunciating
annunciator
annunciators
anode
anodes
anodize
anodized
anodizes
anoint
anointed
anointing
anoints
anomalies
anomalous
anomalously
anomaly
anomic
anomie
anon
anonymity
anonymous
anonymously
anorexia
another
answer
answerable
answered
answerer
answerers
answering
answers
ant
antagonism
antagonisms
antagonist
antagonistic
antagonistically
antagonists
antagonize
antagonized
antagonizes
antagonizing
antarctic
ante
anteater
anteaters
antecedent
antecedents
antedate
antelope
antelopes
antenna
antennae
antennas
anterior
anthem
anthems
anther
anthologies
anthology
anthracite
anthropological
anthropologically
anthropologist
anthropologists
anthropology
anthropomorphic
anthropomorphically
anti
antibacterial
antibiotic
antibiotics
antibodies
antibody
antic
anticipate
anticipated
anticipates
anticipating
anticipation
anticipations
anticipatory
anticoagulation
anticompetitive
antics
antidisestablishmentarianism
antidote
antidotes
antiformant
antifundamentalist
antigen
antigens
antihistorical
antimicrobial
antimony
antinomian
antinomy
antipathy
antiphonal
antipode
antipodes
antiquarian
antiquarians
antiquate
antiquated
antique
antiques
antiquities
antiquity
antiredeposition
antiresonance
antiresonator
antisemitic
antisemitism
antiseptic
antisera
antiserum
antislavery
antisocial
antisubmarine
antisymmetric
antisymmetry
antithesis
antithetical
antithyroid
antitoxin
antitoxins
antitrust
antler
antlered
ants
anus
anvil
anvils
anxieties
anxiety
anxious
anxiously
any
anybody
anyhow
anymore
anyone
anyplace
anything
anytime
anyway
anywhere
aorta
apace
apart
apartment
apartments
apathetic
apathy
ape
aped
aperiodic
aperiodicity
aperture
apes
apex
aphasia
aphasic
aphelion
aphid
aphids
aphonic
aphorism
aphorisms
apiaries
apiary
apical
apiece
aping
apish
aplenty
aplomb
apocalypse
apocalyptic
apocryphal
apogee
apogees
apologetic
apologetically
apologia
apologies
apologist
apologists
apologize
apologized
apologizes
apologizing
apology
apostate
apostle
apostles
apostolic
apostrophe
apostrophes
apothecary
apothegm
apotheoses
apotheosis
appall
appalled
appalling
appallingly
appanage
apparatus
apparel
appareled
apparent
apparently
apparition
apparitions
appeal
appealed
appealer
appealers
appealing
appealingly
appeals
appear
appearance
appearances
appeared
appearer
appearers
appearing
appears
appease
appeased
appeasement
appeases
appeasing
appellant
appellants
appellate
appellation
append
appendage
appendages
appended
appender
appenders
appendices
appendicitis
appending
appendix
appendixes
appends
appertain
appertains
appetite
appetites
appetizer
appetizing
applaud
applauded
applauding
applauds
applause
apple
applejack
apples
appliance
appliances
applicability
applicable
applicant
applicants
application
applications
applicative
applicatively
applicator
applicators
applied
applier
appliers
applies
applique
apply
applying
appoint
appointed
appointee
appointees
appointer
appointers
appointing
appointive
appointment
appointments
appoints
apportion
apportioned
apportioning
apportionment
apportionments
apportions
apposite
appraisal
appraisals
appraise
appraised
appraiser
appraisers
appraises
appraising
appraisingly
appreciable
appreciably
appreciate
appreciated
appreciates
appreciating
appreciation
appreciations
appreciative
appreciatively
apprehend
apprehended
apprehensible
apprehension
apprehensions
apprehensive
apprehensively
apprehensiveness
apprentice
apprenticed
apprentices
apprenticeship
apprise
apprised
apprises
apprising
approach
approachability
approachable
approached
approacher
approachers
approaches
approaching
approbate
approbation
appropriate
appropriated
appropriately
appropriateness
appropriates
appropriating
appropriation
appropriations
appropriator
appropriators
approval
approvals
approve
approved
approver
approvers
approves
approving
approvingly
approximate
approximated
approximately
approximates
approximating
approximation
approximations
appurtenance
appurtenances
apricot
apricots
apron
aprons
apropos
apse
apsis
apt
aptitude
aptitudes
aptly
aptness
aqua
aquaria
aquarium
aquatic
aqueduct
aqueducts
aqueous
aquifer
aquifers
arabesque
arable
arachnid
arachnids
arbiter
arbiters
arbitrarily
arbitrariness
arbitrary
arbitrate
arbitrated
arbitrates
arbitrating
arbitration
arbitrator
arbitrators
arbor
arboreal
arbors
arc
arcade
arcaded
arcades
arcane
arced
arch
archaic
archaically
archaicness
archaism
archaize
archangel
archangels
archbishop
archdiocese
archdioceses
arched
archenemy
archeological
archeologist
archeology
archers
archery
arches
archetype
archfool
arching
archipelago
archipelagoes
architect
architectonic
architects
architectural
architecturally
architecture
architectures
archival
archive
archived
archiver
archivers
archives
archiving
archivist
archly
arcing
arclike
arcs
arcsine
arctangent
arctic
ardent
ardently
ardor
arduous
arduously
arduousness
are
area
areas
arena
arenas
argon
argonauts
argot
arguable
arguably
argue
argued
arguer
arguers
argues
arguing
argument
argumentation
argumentative
arguments
arid
aridity
aright
arise
arisen
ariser
arises
arising
arisings
aristocracy
aristocrat
aristocratic
aristocratically
aristocrats
arithmetic
arithmetical
arithmetically
arithmetics
arithmetize
arithmetized
arithmetizes
ark
arm
armadillo
armadillos
armament
armaments
armchair
armchairs
armed
armer
armers
armful
armhole
armies
arming
armistice
armload
armor
armored
armorer
armory
armpit
armpits
arms
army
aroma
aromas
aromatic
arose
around
arousal
arouse
aroused
arouses
arousing
arpeggio
arpeggios
arrack
arraign
arraigned
arraigning
arraignment
arraignments
arraigns
arrange
arranged
arrangement
arrangements
arranger
arrangers
arranges
arranging
arrant
array
arrayed
arrays
arrears
arrest
arrested
arrester
arresters
arresting
arrestingly
arrestor
arrestors
arrests
arrival
arrivals
arrive
arrived
arrives
arriving
arrogance
arrogant
arrogantly
arrogate
arrogated
arrogates
arrogating
arrogation
arrow
arrowed
arrowhead
arrowheads
arrows
arroyo
arroyos
arsenal
arsenals
arsenic
arsine
arson
art
arterial
arteries
arteriolar
arteriole
arterioles
arteriosclerosis
artery
artful
artfully
artfulness
arthritis
arthropod
arthropods
artichoke
artichokes
article
articles
articulate
articulated
articulately
articulateness
articulates
articulating
articulation
articulations
articulator
articulators
articulatory
artifact
artifacts
artifice
artificer
artifices
artificial
artificialities
artificiality
artificially
artificialness
artillerist
artillery
artisan
artisans
artist
artistic
artistically
artistry
artists
artless
arts
artwork
as
asbestos
ascend
ascendancy
ascendant
ascended
ascendency
ascendent
ascender
ascenders
ascending
ascends
ascension
ascensions
ascent
ascertain
ascertainable
ascertained
ascertaining
ascertains
ascetic
asceticism
ascetics
ascot
ascribable
ascribe
ascribed
ascribes
ascribing
ascription
aseptic
ash
ashamed
ashamedly
ashen
ashes
ashman
ashore
ashtray
ashtrays
aside
asinine
ask
askance
asked
asker
askers
askew
asking
asks
asleep
asocial
asp
asparagus
aspect
aspects
aspen
aspersion
aspersions
asphalt
asphyxia
aspic
aspirant
aspirants
aspirate
aspirated
aspirates
aspirating
aspiration
aspirations
aspirator
aspirators
aspire
aspired
aspires
aspirin
aspiring
aspirins
ass
assail
assailant
assailants
assailed
assailing
assails
assassin
assassinate
assassinated
assassinates
assassinating
assassination
assassinations
assassins
assault
assaulted
assaulting
assaults
assay
assayed
assaying
assemblage
assemblages
assemble
assembled
assembler
assemblers
assembles
assemblies
assembling
assembly
assent
assented
assenter
assenting
assents
assert
asserted
asserter
asserters
asserting
assertion
assertions
assertive
assertively
assertiveness
asserts
asses
assess
assessed
assesses
assessing
assessment
assessments
assessor
assessors
asset
assets
assiduity
assiduous
assiduously
assign
assignable
assigned
assignee
assignees
assigner
assigners
assigning
assignment
assignments
assigns
assimilate
assimilated
assimilates
assimilating
assimilation
assimilations
assist
assistance
assistances
assistant
assistants
assistantship
assistantships
assisted
assisting
assists
associate
associated
associates
associating
association
associational
associations
associative
associatively
associativity
associator
associators
assonance
assonant
assort
assorted
assortment
assortments
assorts
assuage
assuaged
assuages
assume
assumed
assumes
assuming
assumption
assumptions
assurance
assurances
assure
assured
assuredly
assurer
assurers
assures
assuring
assuringly
astatine
aster
asterisk
asterisks
asteroid
asteroidal
asteroids
asters
asthma
astonish
astonished
astonishes
astonishing
astonishingly
astonishment
astound
astounded
astounding
astounds
astral
astray
astride
astringency
astringent
astrology
astronaut
astronautics
astronauts
astronomer
astronomers
astronomical
astronomically
astronomy
astrophysical
astrophysics
astute
astutely
astuteness
asunder
asylum
asymmetric
asymmetrically
asymmetry
asymptomatically
asymptote
asymptotes
asymptotic
asymptotically
asynchronism
asynchronous
asynchronously
asynchrony
at
atavistic
ate
atemporal
atheism
atheist
atheistic
atheists
atherosclerosis
athlete
athletes
athletic
athleticism
athletics
atlas
atmosphere
atmospheres
atmospheric
atoll
atolls
atom
atomic
atomically
atomics
atomization
atomize
atomized
atomizes
atomizing
atoms
atonal
atonally
atone
atoned
atonement
atones
atop
atrocious
atrociously
atrocities
atrocity
atrophic
atrophied
atrophies
atrophy
atrophying
attach
attache
attached
attacher
attachers
attaches
attaching
attachment
attachments
attack
attackable
attacked
attacker
attackers
attacking
attacks
attain
attainable
attainably
attained
attainer
attainers
attaining
attainment
attainments
attains
attempt
attempted
attempter
attempters
attempting
attempts
attend
attendance
attendances
attendant
attendants
attended
attendee
attendees
attender
attenders
attending
attends
attention
attentional
attentionality
attentions
attentive
attentively
attentiveness
attenuate
attenuated
attenuates
attenuating
attenuation
attenuator
attenuators
attest
attested
attesting
attests
attic
attics
attire
attired
attires
attiring
attitude
attitudes
attitudinal
attorney
attorneys
attract
attracted
attracting
attraction
attractions
attractive
attractively
attractiveness
attractor
attractors
attracts
attributable
attribute
attributed
attributes
attributing
attribution
attributions
attributive
attributively
attrition
attune
attuned
attunes
attuning
atypical
atypically
auburn
auction
auctioneer
auctioneers
audacious
audaciously
audaciousness
audacity
audible
audibly
audience
audiences
audio
audiogram
audiograms
audiological
audiologist
audiologists
audiology
audiometer
audiometers
audiometric
audiometry
audit
audited
auditing
audition
auditioned
auditioning
auditions
auditor
auditorium
auditors
auditory
audits
auger
augers
aught
augment
augmentation
augmented
augmenting
augments
augur
augurs
august
augustly
augustness
aunt
aunts
aura
aural
aurally
auras
aureole
aureomycin
aurora
auscultate
auscultated
auscultates
auscultating
auscultation
auscultations
auspice
auspices
auspicious
auspiciously
austere
austerely
austerity
authentic
authentically
authenticate
authenticated
authenticates
authenticating
authentication
authentications
authenticator
authenticators
authenticity
author
authored
authoring
authoritarian
authoritarianism
authoritative
authoritatively
authorities
authority
authorization
authorizations
authorize
authorized
authorizer
authorizers
authorizes
authorizing
authors
authorship
autism
autistic
auto
autobiographic
autobiographical
autobiographies
autobiography
autocollimator
autocorrelate
autocorrelation
autocracies
autocracy
autocrat
autocratic
autocratically
autocrats
autodecrement
autodecremented
autodecrements
autodialer
autofluorescence
autograph
autographed
autographing
autographs
autoincrement
autoincremented
autoincrements
autoindex
autoindexing
automata
automate
automated
automates
automatic
automatically
automating
automation
automaton
automobile
automobiles
automotive
autonavigator
autonavigators
autonomic
autonomous
autonomously
autonomy
autopilot
autopilots
autopsied
autopsies
autopsy
autoregressive
autos
autosuggestibility
autotransformer
autumn
autumnal
autumns
auxiliaries
auxiliary
avail
availabilities
availability
available
availably
availed
availer
availers
availing
avails
avalanche
avalanched
avalanches
avalanching
avant
avarice
avaricious
avariciously
avenge
avenged
avenger
avenges
avenging
avenue
avenues
aver
average
averaged
averages
averaging
averred
averrer
averring
avers
averse
aversion
aversions
avert
averted
averting
averts
avian
aviaries
aviary
aviation
aviator
aviators
avid
avidity
avidly
avionic
avionics
avocado
avocados
avocation
avocations
avoid
avoidable
avoidably
avoidance
avoided
avoider
avoiders
avoiding
avoids
avouch
avow
avowal
avowed
avows
await
awaited
awaiting
awaits
awake
awaken
awakened
awakening
awakens
awakes
awaking
award
awarded
awarder
awarders
awarding
awards
aware
awareness
awash
away
awe
awed
awesome
awful
awfully
awfulness
awhile
awkward
awkwardly
awkwardness
awl
awls
awning
awnings
awoke
awry
ax
axed
axer
axers
axes
axial
axially
axing
axiological
axiom
axiomatic
axiomatically
axiomatization
axiomatizations
axiomatize
axiomatized
axiomatizes
axiomatizing
axioms
axis
axle
axles
axolotl
axolotls
axon
axons
aye
ayes
azalea
azaleas
azimuth
azimuths
azure
babble
babbled
babbles
babbling
babe
babes
babied
babies
baboon
baboons
baby
babyhood
babying
babyish
babysit
babysitting
baccalaureate
bachelor
bachelors
bacilli
bacillus
back
backache
backaches
backarrow
backbend
backbends
backboard
backbone
backbones
backdrop
backdrops
backed
backer
backers
backfill
backfiring
background
backgrounds
backhand
backing
backlash
backlog
backlogged
backlogs
backorder
backpack
backpacks
backplane
backplanes
backplate
backs
backscatter
backscattered
backscattering
backscatters
backside
backslash
backslashes
backspace
backspaced
backspaces
backspacing
backstage
backstairs
backstitch
backstitched
backstitches
backstitching
backstop
backtrack
backtracked
backtracker
backtrackers
backtracking
backtracks
backup
backups
backward
backwardness
backwards
backwater
backwaters
backwoods
backyard
backyards
bacon
bacteria
bacterial
bacterium
bad
bade
badge
badger
badgered
badgering
badgers
badges
badlands
badly
badminton
badness
baffle
baffled
baffler
bafflers
baffling
bag
bagatelle
bagatelles
bagel
bagels
baggage
bagged
bagger
baggers
bagging
baggy
bagpipe
bagpipes
bags
bah
bail
bailiff
bailiffs
bailing
bait
baited
baiter
baiting
baits
bake
baked
baker
bakeries
bakers
bakery
bakes
baking
baklava
balalaika
balalaikas
balance
balanced
balancer
balancers
balances
balancing
balconies
balcony
bald
balding
baldly
baldness
bale
baleful
baler
bales
balk
balkanized
balkanizing
balked
balkiness
balking
balks
balky
ball
ballad
ballads
ballast
ballasts
balled
baller
ballerina
ballerinas
ballers
ballet
ballets
ballgown
balling
ballistic
ballistics
balloon
ballooned
ballooner
ballooners
ballooning
balloons
ballot
ballots
ballpark
ballparks
ballplayer
ballplayers
ballroom
ballrooms
balls
ballyhoo
balm
balms
balmy
balsa
balsam
balustrade
balustrades
bamboo
ban
banal
banally
banana
bananas
band
bandage
bandaged
bandages
bandaging
banded
bandied
bandies
banding
bandit
bandits
bandpass
bands
bandstand
bandstands
bandwagon
bandwagons
bandwidth
bandwidths
bandy
bandying
bane
baneful
bang
banged
banging
bangle
bangles
bangs
banish
banished
banishes
banishing
banishment
banister
banisters
banjo
banjos
bank
banked
banker
bankers
banking
bankrupt
bankruptcies
bankruptcy
bankrupted
bankrupting
bankrupts
banned
banner
banners
banning
banquet
banqueting
banquetings
banquets
bans
banshee
banshees
bantam
banter
bantered
bantering
banters
baptism
baptismal
baptisms
baptistery
baptistries
baptistry
baptize
baptized
baptizes
baptizing
bar
barb
barbarian
barbarians
barbaric
barbarism
barbarities
barbarity
barbarous
barbarously
barbecue
barbecued
barbecues
barbed
barbell
barbells
barber
barbital
barbiturate
barbiturates
barbs
bard
bards
bare
bared
barefaced
barefoot
barefooted
barely
bareness
barer
bares
barest
barflies
barfly
bargain
bargained
bargaining
bargains
barge
barges
barging
baring
baritone
baritones
barium
bark
barked
barker
barkers
barking
barks
barley
barn
barns
barnstorm
barnstormed
barnstorming
barnstorms
barnyard
barnyards
barometer
barometers
barometric
baron
baroness
baronial
baronies
barons
barony
baroque
baroqueness
barrack
barracks
barrage
barrages
barred
barrel
barrelled
barrelling
barrels
barren
barrenness
barricade
barricades
barrier
barriers
barring
barringer
barrow
bars
bartender
bartenders
barter
bartered
bartering
barters
basal
basalt
base
baseball
baseballs
baseband
baseboard
baseboards
based
baseless
baseline
baselines
basely
baseman
basement
basements
baseness
baser
bases
bash
bashed
bashes
bashful
bashfulness
bashing
basic
basically
basics
basil
basin
basing
basins
basis
bask
basked
basket
basketball
basketballs
baskets
basking
bass
basses
basset
bassinet
bassinets
bastard
bastards
baste
basted
bastes
basting
bastion
bastions
bat
batch
batched
batches
bath
bathe
bathed
bather
bathers
bathes
bathing
bathos
bathrobe
bathrobes
bathroom
bathrooms
baths
bathtub
bathtubs
baton
batons
bats
battalion
battalions
batted
batten
battens
batter
battered
batteries
battering
batters
battery
batting
battle
battled
battlefield
battlefields
battlefront
battlefronts
battleground
battlegrounds
battlement
battlements
battler
battlers
battles
battleship
battleships
battling
bauble
baubles
baud
bauxite
bawdy
bawl
bawled
bawling
bawls
bay
bayed
baying
bayonet
bayonets
bayou
bayous
bays
bazaar
bazaars
be
beach
beached
beaches
beachhead
beachheads
beaching
beacon
beacons
bead
beaded
beading
beadle
beadles
beads
beady
beagle
beagles
beak
beaked
beaker
beakers
beaks
beam
beamed
beamer
beamers
beaming
beams
bean
beanbag
beaned
beaner
beaners
beaning
beans
bear
bearable
bearably
beard
bearded
beardless
beards
bearer
bearers
bearing
bearings
bearish
bears
beast
beastly
beasts
beat
beatable
beatably
beaten
beater
beaters
beatific
beatification
beatify
beating
beatings
beatitude
beatitudes
beatnik
beatniks
beats
beau
beaus
beauteous
beauteously
beauties
beautifications
beautified
beautifier
beautifiers
beautifies
beautiful
beautifully
beautify
beautifying
beauty
beaver
beavers
becalm
becalmed
becalming
becalms
became
because
beck
beckon
beckoned
beckoning
beckons
become
becomes
becoming
becomingly
bed
bedazzle
bedazzled
bedazzlement
bedazzles
bedazzling
bedbug
bedbugs
bedded
bedder
bedders
bedding
bedevil
bedeviled
bedeviling
bedevils
bedfast
bedlam
bedpost
bedposts
bedraggle
bedraggled
bedridden
bedrock
bedroom
bedrooms
beds
bedside
bedspread
bedspreads
bedspring
bedsprings
bedstead
bedsteads
bedtime
bee
beech
beechen
beecher
beef
beefed
beefer
beefers
beefing
beefs
beefsteak
beefy
beehive
beehives
been
beep
beeps
beer
beers
bees
beet
beetle
beetled
beetles
beetling
beets
befall
befallen
befalling
befalls
befell
befit
befits
befitted
befitting
befog
befogged
befogging
before
beforehand
befoul
befouled
befouling
befouls
befriend
befriended
befriending
befriends
befuddle
befuddled
befuddles
befuddling
beg
began
beget
begets
begetting
beggar
beggarly
beggars
beggary
begged
begging
begin
beginner
beginners
beginning
beginnings
begins
begot
begotten
begrudge
begrudged
begrudges
begrudging
begrudgingly
begs
beguile
beguiled
beguiles
beguiling
begun
behalf
behave
behaved
behaves
behaving
behavior
behavioral
behaviorally
behaviorism
behavioristic
behaviors
behead
beheading
beheld
behemoth
behemoths
behest
behind
behold
beholden
beholder
beholders
beholding
beholds
behoove
behooves
beige
being
beings
belabor
belabored
belaboring
belabors
belated
belatedly
belay
belayed
belaying
belays
belch
belched
belches
belching
belfries
belfry
belie
belied
belief
beliefs
belies
believable
believably
believe
believed
believer
believers
believes
believing
belittle
belittled
belittles
belittling
bell
bellboy
bellboys
belle
belles
bellhop
bellhops
bellicose
bellicosity
bellies
belligerence
belligerent
belligerently
belligerents
bellman
bellmen
bellow
bellowed
bellowing
bellows
bells
bellum
bellwether
bellwethers
belly
bellyache
bellyfull
belong
belonged
belonging
belongings
belongs
beloved
below
belt
belted
belting
belts
bely
belying
bemoan
bemoaned
bemoaning
bemoans
bench
benched
benches
benchmark
benchmarking
benchmarks
bend
bendable
benders
bending
bends
beneath
benediction
benedictions
benefactor
benefactors
beneficence
beneficences
beneficent
beneficial
beneficially
beneficiaries
beneficiary
benefit
benefited
benefiting
benefits
benefitted
benefitting
benevolence
benevolent
benighted
benign
benignly
bent
benzene
bequeath
bequeathal
bequeathed
bequeathing
bequeaths
bequest
bequests
berate
berated
berates
berating
bereave
bereaved
bereavement
bereavements
bereaves
bereaving
bereft
beret
berets
beribboned
beriberi
berkelium
berne
berries
berry
berserk
berth
berths
beryl
beryllium
beseech
beseeches
beseeching
beset
besets
besetting
beside
besides
besiege
besieged
besieger
besiegers
besieging
besmirch
besmirched
besmirches
besmirching
besotted
besotter
besotting
besought
bespeak
bespeaks
bespectacled
bespoke
best
bested
bestial
besting
bestir
bestirring
bestow
bestowal
bestowed
bests
bestseller
bestsellers
bestselling
bet
beta
betatron
betel
betide
betray
betrayal
betrayed
betrayer
betraying
betrays
betroth
betrothal
betrothed
bets
better
bettered
bettering
betterment
betterments
betters
betting
between
betwixt
bevel
beveled
beveling
bevels
beverage
beverages
bevy
bewail
bewailed
bewailing
bewails
beware
bewhiskered
bewilder
bewildered
bewildering
bewilderingly
bewilderment
bewilders
bewitch
bewitched
bewitches
bewitching
beyond
biannual
bias
biased
biases
biasing
bib
bibbed
bibbing
bibles
biblical
biblically
bibliographic
bibliographical
bibliographies
bibliography
bibliophile
bibs
bicameral
bicarbonate
bicentennial
bicep
biceps
bicker
bickered
bickering
bickers
biconcave
biconnected
biconvex
bicycle
bicycled
bicycler
bicyclers
bicycles
bicycling
bid
biddable
bidden
bidder
bidders
biddies
bidding
biddy
bide
bidirectional
bids
biennial
biennium
bier
bifocal
bifocals
bifurcate
big
bigger
biggest
bight
bights
bigness
bigot
bigoted
bigotry
bigots
biharmonic
bijection
bijections
bijective
bijectively
bike
bikes
biking
bikini
bikinis
bilabial
bilateral
bilaterally
bile
bilge
bilges
bilinear
bilingual
bilk
bilked
bilking
bilks
bill
billboard
billboards
billed
biller
billers
billet
billeted
billeting
billets
billiard
billiards
billing
billion
billions
billionth
billow
billowed
billows
bills
bimetallic
bimetallism
bimodal
bimolecular
bimonthlies
bimonthly
bin
binaries
binary
binaural
bind
binder
binders
binding
bindings
binds
bing
binge
binges
bingo
binocular
binoculars
binomial
bins
binuclear
biochemical
biochemist
biochemistry
biofeedback
biographer
biographers
biographic
biographical
biographically
biographies
biography
biological
biologically
biologist
biologists
biology
biomedical
biomedicine
biophysical
biophysicist
biophysics
biopsies
biopsy
bioscience
biosphere
biostatistic
biosynthesize
biota
biotic
bipartisan
bipartite
biped
bipeds
biplane
biplanes
bipolar
biracial
birch
birchen
birches
bird
birdbath
birdbaths
birdie
birdied
birdies
birdlike
birds
birefringence
birefringent
birth
birthday
birthdays
birthed
birthplace
birthplaces
birthright
birthrights
births
biscuit
biscuits
bisect
bisected
bisecting
bisection
bisections
bisector
bisectors
bisects
bishop
bishops
bismuth
bison
bisons
bisque
bisques
bistable
bistate
bit
bitch
bitches
bite
biter
biters
bites
biting
bitingly
bitmap
bits
bitten
bitter
bitterer
bitterest
bitterly
bitterness
bitternut
bitterroot
bitters
bittersweet
bitumen
bituminous
bitwise
bivalve
bivalves
bivariate
bivouac
bivouacs
biweekly
bizarre
blab
blabbed
blabbermouth
blabbermouths
blabbing
blabs
black
blackberries
blackberry
blackbird
blackbirds
blackboard
blackboards
blacked
blacken
blackened
blackening
blackens
blacker
blackest
blacking
blackjack
blackjacks
blacklist
blacklisted
blacklisting
blacklists
blackly
blackmail
blackmailed
blackmailer
blackmailers
blackmailing
blackmails
blackness
blackout
blackouts
blacks
blacksmith
blacksmiths
bladder
bladders
blade
blades
blamable
blame
blamed
blameless
blamelessness
blamer
blamers
blames
blameworthy
blaming
blanch
blanched
blanches
blanching
bland
blandly
blandness
blank
blanked
blanker
blankest
blanket
blanketed
blanketer
blanketers
blanketing
blankets
blanking
blankly
blankness
blanks
blare
blared
blares
blaring
blase
blaspheme
blasphemed
blasphemes
blasphemies
blaspheming
blasphemous
blasphemously
blasphemousness
blasphemy
blast
blasted
blaster
blasters
blasting
blasts
blatant
blatantly
blaze
blazed
blazer
blazers
blazes
blazing
bleach
bleached
bleacher
bleachers
bleaches
bleaching
bleak
bleaker
bleakly
bleakness
blear
bleary
bleat
bleating
bleats
bled
bleed
bleeder
bleeding
bleedings
bleeds
blemish
blemishes
blend
blended
blender
blending
blends
bless
blessed
blessing
blessings
blew
blight
blighted
blimp
blimps
blind
blinded
blinder
blinders
blindfold
blindfolded
blindfolding
blindfolds
blinding
blindingly
blindly
blindness
blinds
blink
blinked
blinker
blinkers
blinking
blinks
blip
blips
bliss
blissful
blissfully
blister
blistered
blistering
blisters
blithe
blithely
blitz
blitzes
blitzkrieg
blizzard
blizzards
bloat
bloated
bloater
bloating
bloats
blob
blobs
bloc
block
blockade
blockaded
blockades
blockading
blockage
blockages
blocked
blocker
blockers
blockhouse
blockhouses
blocking
blocks
blocs
bloke
blokes
blond
blonde
blondes
blonds
blood
bloodbath
blooded
bloodhound
bloodhounds
bloodied
bloodiest
bloodless
bloods
bloodshed
bloodshot
bloodstain
bloodstained
bloodstains
bloodstream
bloody
bloom
bloomed
bloomers
blooming
blooms
blooper
blossom
blossomed
blossoms
blot
blots
blotted
blotting
blouse
blouses
blow
blower
blowers
blowfish
blowing
blown
blowout
blows
blowup
blubber
bludgeon
bludgeoned
bludgeoning
bludgeons
blue
blueberries
blueberry
bluebird
bluebirds
bluebonnet
bluebonnets
bluefish
blueness
blueprint
blueprints
bluer
blues
bluest
bluestocking
bluff
bluffing
bluffs
bluing
bluish
blunder
blunderbuss
blundered
blundering
blunderings
blunders
blunt
blunted
blunter
bluntest
blunting
bluntly
bluntness
blunts
blur
blurb
blurred
blurring
blurry
blurs
blurt
blurted
blurting
blurts
blush
blushed
blushes
blushing
bluster
blustered
blustering
blusters
blustery
boa
boar
board
boarded
boarder
boarders
boarding
boardinghouse
boardinghouses
boards
boast
boasted
boaster
boasters
boastful
boastfully
boasting
boastings
boasts
boat
boater
boaters
boathouse
boathouses
boating
boatload
boatloads
boatman
boatmen
boats
boatsman
boatsmen
boatswain
boatswains
boatyard
boatyards
bob
bobbed
bobbin
bobbing
bobbins
bobby
bobolink
bobolinks
bobs
bobwhite
bobwhites
bode
bodes
bodice
bodied
bodies
bodily
body
bodybuilder
bodybuilders
bodybuilding
bodyguard
bodyguards
bodyweight
bog
bogeymen
bogged
boggle
boggled
boggles
boggling
bogs
bogus
boil
boiled
boiler
boilerplate
boilers
boiling
boils
boisterous
boisterously
bold
bolder
boldest
boldface
boldly
boldness
boll
bolster
bolstered
bolstering
bolsters
bolt
bolted
bolting
bolts
bomb
bombard
bombarded
bombarding
bombardment
bombards
bombast
bombastic
bombed
bomber
bombers
bombing
bombings
bombproof
bombs
bonanza
bonanzas
bond
bondage
bonded
bonder
bonders
bonding
bonds
bondsman
bondsmen
bone
boned
boner
boners
bones
bonfire
bonfires
bong
boning
bonnet
bonneted
bonnets
bonny
bonus
bonuses
bony
boo
boob
booboo
booby
book
bookcase
bookcases
booked
booker
bookers
bookie
bookies
booking
bookings
bookish
bookkeeper
bookkeepers
bookkeeping
booklet
booklets
bookmark
books
bookseller
booksellers
bookshelf
bookshelves
bookstore
bookstores
bookworm
boolean
boom
boomed
boomerang
boomerangs
booming
booms
boon
boor
boorish
boors
boos
boost
boosted
booster
boosting
boosts
boot
bootable
booted
booth
booths
booting
bootleg
bootlegged
bootlegger
bootleggers
bootlegging
bootlegs
boots
bootstrap
bootstrapped
bootstrapping
bootstraps
booty
booze
borate
borates
borax
bordello
bordellos
border
bordered
bordering
borderings
borderland
borderlands
borderline
borders
bore
bored
boredom
borer
bores
boric
boring
born
borne
boron
borough
boroughs
borrow
borrowed
borrower
borrowers
borrowing
borrows
bosom
bosoms
boss
bossed
bosses
bosun
botanical
botanist
botanists
botany
botch
botched
botcher
botchers
botches
botching
both
bother
bothered
bothering
bothers
bothersome
bottle
bottled
bottleneck
bottlenecks
bottler
bottlers
bottles
bottling
bottom
bottomed
bottoming
bottomless
bottoms
botulinus
botulism
bouffant
bough
boughs
bought
boulder
boulders
boulevard
boulevards
bounce
bounced
bouncer
bounces
bouncing
bouncy
bound
boundaries
boundary
bounded
bounden
bounding
boundless
boundlessness
bounds
bounteous
bounteously
bounties
bountiful
bounty
bouquet
bouquets
bourbon
bourgeois
bourgeoisie
boustrophedon
boustrophedonic
bout
boutique
bouts
bovine
bovines
bow
bowdlerize
bowdlerized
bowdlerizes
bowdlerizing
bowed
bowel
bowels
bower
bowers
bowing
bowl
bowled
bowler
bowlers
bowline
bowlines
bowling
bowls
bowman
bows
bowstring
bowstrings
box
boxcar
boxcars
boxed
boxer
boxers
boxes
boxing
boxtop
boxtops
boxwood
boy
boycott
boycotted
boycotts
boyfriend
boyfriends
boyhood
boyish
boyishness
boys
bra
brace
braced
bracelet
bracelets
braces
bracing
bracket
bracketed
bracketing
brackets
brackish
brae
braes
brag
bragged
bragger
bragging
brags
braid
braided
braiding
braids
brain
brainchild
brained
braining
brains
brainstem
brainstems
brainstorm
brainstorms
brainwash
brainwashed
brainwashes
brainwashing
brainy
brake
braked
brakeman
brakes
braking
bramble
brambles
brambly
bran
branch
branched
branches
branching
branchings
brand
branded
branding
brandish
brandishes
brandishing
brands
brandy
brandywine
bras
brash
brashly
brashness
brass
brasses
brassiere
brassy
brat
brats
bravado
brave
braved
bravely
braveness
braver
bravery
braves
bravest
braving
bravo
bravos
brawl
brawler
brawling
brawn
bray
brayed
brayer
braying
brays
braze
brazed
brazen
brazenly
brazenness
brazes
brazier
braziers
brazing
breach
breached
breacher
breachers
breaches
breaching
bread
breadboard
breadboards
breadbox
breadboxes
breaded
breading
breads
breadth
breadwinner
breadwinners
break
breakable
breakables
breakage
breakaway
breakdown
breakdowns
breaker
breakers
breakfast
breakfasted
breakfaster
breakfasters
breakfasting
breakfasts
breaking
breakpoint
breakpoints
breaks
breakthrough
breakthroughes
breakthroughs
breakup
breakwater
breakwaters
breast
breasted
breasts
breastwork
breastworks
breath
breathable
breathe
breathed
breather
breathers
breathes
breathing
breathless
breathlessly
breaths
breathtaking
breathtakingly
breathy
bred
breech
breeches
breed
breeder
breeding
breeds
breeze
breezes
breezily
breezy
bremsstrahlung
brethren
breve
brevet
breveted
breveting
brevets
brevity
brew
brewed
brewer
breweries
brewers
brewery
brewing
brews
briar
briars
bribe
bribed
briber
bribers
bribery
bribes
bribing
brick
brickbat
bricked
bricker
bricklayer
bricklayers
bricklaying
bricks
bridal
bride
bridegroom
brides
bridesmaid
bridesmaids
bridge
bridgeable
bridged
bridgehead
bridgeheads
bridges
bridgework
bridging
bridle
bridled
bridles
bridling
brief
briefcase
briefcases
briefed
briefer
briefest
briefing
briefings
briefly
briefness
briefs
brier
brig
brigade
brigades
brigadier
brigadiers
brigantine
bright
brighten
brightened
brightener
brighteners
brightening
brightens
brighter
brightest
brightly
brightness
brigs
brilliance
brilliancy
brilliant
brilliantly
brim
brimful
brimmed
brimming
brimstone
brindle
brindled
brine
bring
bringer
bringers
bringing
brings
brink
brinkmanship
briny
brisk
brisker
briskly
briskness
bristle
bristled
bristles
bristling
britches
brittle
brittleness
broach
broached
broaches
broaching
broad
broadband
broadcast
broadcasted
broadcaster
broadcasters
broadcasting
broadcastings
broadcasts
broaden
broadened
broadener
broadeners
broadening
broadenings
broadens
broader
broadest
broadly
broadness
broadside
brocade
brocaded
broccoli
brochure
brochures
broil
broiled
broiler
broilers
broiling
broils
broke
broken
brokenly
brokenness
broker
brokerage
brokers
bromide
bromides
bromine
bronchi
bronchial
bronchiole
bronchioles
bronchitis
bronchus
bronze
bronzed
bronzes
brooch
brooches
brood
brooder
brooding
broods
brook
brooked
brooks
broom
brooms
broomstick
broomsticks
broth
brothel
brothels
brother
brotherhood
brotherliness
brotherly
brothers
brought
brow
browbeat
browbeaten
browbeating
browbeats
brown
browned
browner
brownest
brownie
brownies
browning
brownish
brownness
browns
brows
browse
browsing
bruise
bruised
bruises
bruising
brunch
brunches
brunette
brunt
brush
brushed
brushes
brushfire
brushfires
brushing
brushlike
brushy
brusque
brusquely
brutal
brutalities
brutality
brutalize
brutalized
brutalizes
brutalizing
brutally
brute
brutes
brutish
bubble
bubbled
bubbles
bubbling
bubbly
buck
buckboard
buckboards
bucked
bucket
buckets
bucking
buckle
buckled
buckler
buckles
buckling
bucks
buckshot
buckskin
buckskins
buckwheat
bucolic
bud
budded
buddies
budding
buddy
budge
budged
budges
budget
budgetary
budgeted
budgeter
budgeters
budgeting
budgets
budging
buds
buff
buffalo
buffaloes
buffer
buffered
buffering
buffers
buffet
buffeted
buffeting
buffetings
buffets
buffoon
buffoons
buffs
bug
bugaboo
bugeyed
bugged
bugger
buggers
buggies
bugging
buggy
bugle
bugled
bugler
bugles
bugling
bugs
build
builder
builders
building
buildings
builds
buildup
buildups
built
builtin
bulb
bulbs
bulge
bulged
bulging
bulk
bulked
bulkhead
bulkheads
bulks
bulky
bull
bulldog
bulldogs
bulldoze
bulldozed
bulldozer
bulldozes
bulldozing
bulled
bullet
bulletin
bulletins
bullets
bullfrog
bullied
bullies
bulling
bullion
bullish
bullock
bulls
bullseye
bully
bullying
bulwark
bum
bumble
bumblebee
bumblebees
bumbled
bumbler
bumblers
bumbles
bumbling
bummed
bumming
bump
bumped
bumper
bumpers
bumping
bumps
bumptious
bumptiously
bumptiousness
bums
bun
bunch
bunched
bunches
bunching
bundle
bundled
bundles
bundling
bungalow
bungalows
bungle
bungled
bungler
bunglers
bungles
bungling
bunion
bunions
bunk
bunker
bunkered
bunkers
bunkhouse
bunkhouses
bunkmate
bunkmates
bunks
bunnies
bunny
buns
bunt
bunted
bunter
bunters
bunting
bunts
buoy
buoyancy
buoyant
buoyed
buoys
burden
burdened
burdening
burdens
burdensome
bureau
bureaucracies
bureaucracy
bureaucrat
bureaucratic
bureaucrats
bureaus
burgeon
burgeoned
burgeoning
burgess
burgesses
burgher
burghers
burglar
burglaries
burglarize
burglarized
burglarizes
burglarizing
burglarproof
burglarproofed
burglarproofing
burglarproofs
burglars
burglary
burial
buried
buries
burl
burlesque
burlesques
burly
burn
burned
burner
burners
burning
burningly
burnings
burnish
burnished
burnishes
burnishing
burns
burnt
burntly
burntness
burp
burped
burping
burps
burrow
burrowed
burrower
burrowing
burrows
burrs
bursa
bursitis
burst
burstiness
bursting
bursts
bursty
bury
burying
bus
busboy
busboys
bused
buses
bush
bushel
bushels
bushes
bushing
bushwhack
bushwhacked
bushwhacking
bushwhacks
bushy
busied
busier
busiest
busily
business
businesses
businesslike
businessman
businessmen
busing
buss
bussed
busses
bussing
bust
bustard
bustards
busted
buster
bustle
bustling
busts
busy
but
butane
butcher
butchered
butchers
butchery
butler
butlers
butt
butte
butted
butter
butterball
buttercup
buttered
butterer
butterers
butterfat
butterflies
butterfly
buttering
buttermilk
butternut
butters
buttery
buttes
butting
buttock
buttocks
button
buttoned
buttonhole
buttonholes
buttoning
buttons
buttress
buttressed
buttresses
buttressing
butts
butyl
butyrate
buxom
buy
buyer
buyers
buying
buys
buzz
buzzards
buzzed
buzzer
buzzes
buzzing
buzzword
buzzwords
buzzy
by
bye
bygone
bylaw
bylaws
byline
bylines
bypass
bypassed
bypasses
bypassing
byproduct
byproducts
bystander
bystanders
byte
bytes
byway
byways
byword
bywords
cab
cabal
cabana
cabaret
cabbage
cabbages
cabdriver
cabin
cabinet
cabinets
cabins
cable
cabled
cables
cabling
caboose
cabs
cache
cached
caches
caching
cackle
cackled
cackler
cackles
cackling
cacti
cactus
cadaver
cadence
cadenced
cadres
cafe
cafes
cafeteria
cage
caged
cager
cagers
cages
caging
caiman
cairn
cajole
cajoled
cajoles
cajoling
cake
caked
cakes
caking
calamities
calamitous
calamity
calcify
calcium
calculate
calculated
calculates
calculating
calculation
calculations
calculative
calculator
calculators
calculi
calculus
caldera
calendar
calendars
calf
calfskin
caliber
calibers
calibrate
calibrated
calibrates
calibrating
calibration
calibrations
calico
caliph
caliphs
call
callable
called
caller
callers
calling
calliope
callous
calloused
callously
callousness
calls
callus
calm
calmed
calmer
calmest
calming
calmingly
calmly
calmness
calms
caloric
calorie
calories
calorimeter
calorimetric
calorimetry
calumny
calve
calves
calypso
cam
came
camel
camels
camera
cameraman
cameramen
cameras
camouflage
camouflaged
camouflages
camouflaging
camp
campaign
campaigned
campaigner
campaigners
campaigning
campaigns
camped
camper
campers
campfire
campground
camping
camps
campsite
campus
campuses
can
canal
canals
canaries
canary
cancel
canceled
canceling
cancellation
cancellations
cancels
cancer
cancerous
cancers
candid
candidacy
candidate
candidates
candidly
candidness
candied
candies
candle
candlelight
candler
candles
candlestick
candlesticks
candor
candy
cane
caner
canine
canister
canker
cankerworm
cannabis
canned
cannel
canner
canners
cannery
cannibal
cannibalize
cannibalized
cannibalizes
cannibalizing
cannibals
canning
cannister
cannisters
cannon
cannonball
cannons
cannot
canny
canoe
canoes
canon
canonic
canonical
canonicalization
canonicalize
canonicalized
canonicalizes
canonicalizing
canonically
canonicals
canons
canopy
cans
cant
cantaloupe
cantankerous
cantankerously
canteen
cantilever
canto
canton
cantons
cantor
cantors
canvas
canvases
canvass
canvassed
canvasser
canvassers
canvasses
canvassing
canyon
canyons
cap
capabilities
capability
capable
capably
capacious
capaciously
capaciousness
capacitance
capacitances
capacities
capacitive
capacitor
capacitors
capacity
cape
caper
capers
capes
capillary
capita
capital
capitalism
capitalist
capitalists
capitalization
capitalizations
capitalize
capitalized
capitalizer
capitalizers
capitalizes
capitalizing
capitally
capitals
capitol
capitols
capped
capping
caprice
capricious
capriciously
capriciousness
caps
capstan
capstone
capsule
captain
captained
captaining
captains
caption
captions
captivate
captivated
captivates
captivating
captivation
captive
captives
captivity
captor
captors
capture
captured
capturer
capturers
captures
capturing
capybara
car
caramel
caravan
caravans
caraway
carbohydrate
carbolic
carbon
carbonate
carbonates
carbonation
carbonic
carbonization
carbonize
carbonized
carbonizer
carbonizers
carbonizes
carbonizing
carbons
carborundum
carbuncle
carcass
carcasses
carcinogen
carcinogenic
carcinoma
card
cardboard
carder
cardiac
cardinal
cardinalities
cardinality
cardinally
cardinals
cardiology
cardiovascular
cards
care
cared
careen
career
careers
carefree
careful
carefully
carefulness
careless
carelessly
carelessness
cares
caress
caressed
caresser
caresses
caressing
caret
caretaker
cargo
cargoes
caribou
caricature
caring
carload
carnage
carnal
carnation
carnival
carnivals
carnivorous
carnivorously
carol
carols
carp
carpenter
carpenters
carpentry
carpet
carpeted
carpeting
carpets
carport
carriage
carriages
carried
carrier
carriers
carries
carrion
carrot
carrots
carry
carrying
carryover
carryovers
cars
cart
carted
cartel
carter
carters
cartilage
carting
cartographer
cartographic
cartography
carton
cartons
cartoon
cartoons
cartridge
cartridges
carts
cartwheel
carve
carved
carver
carves
carving
carvings
cascadable
cascade
cascaded
cascades
cascading
case
cased
casement
casements
cases
casework
cash
cashed
casher
cashers
cashes
cashew
cashier
cashiers
cashing
cashmere
casing
casings
casino
cask
casket
caskets
casks
casserole
casseroles
cassette
cassock
cast
caste
caster
casters
castes
castigate
casting
castle
castled
castles
castor
casts
casual
casually
casualness
casuals
casualties
casualty
cat
cataclysmic
catalog
cataloged
cataloger
cataloging
catalogs
catalyst
catalysts
catalytic
catapult
cataract
catastrophe
catastrophes
catastrophic
catch
catchable
catcher
catchers
catches
catching
categorical
categorically
categories
categorization
categorize
categorized
categorizer
categorizers
categorizes
categorizing
category
cater
catered
caterer
catering
caterpillar
caterpillars
caters
cathedral
cathedrals
catheter
catheters
cathode
cathodes
catlike
catnip
cats
catsup
cattail
cattle
cattleman
cattlemen
caucus
caught
cauldron
cauldrons
cauliflower
caulk
causal
causality
causally
causation
causations
cause
caused
causer
causes
causeway
causeways
causing
caustic
causticly
caustics
caution
cautioned
cautioner
cautioners
cautioning
cautionings
cautions
cautious
cautiously
cautiousness
cavalier
cavalierly
cavalierness
cavalry
cave
caveat
caveats
caved
caveman
cavemen
cavern
cavernous
caverns
caves
caviar
cavil
caving
cavities
cavity
caw
cawing
cease
ceased
ceaseless
ceaselessly
ceaselessness
ceases
ceasing
cedar
cede
ceded
ceding
ceiling
ceilings
celebrate
celebrated
celebrates
celebrating
celebration
celebrations
celebrities
celebrity
celerity
celery
celestial
celestially
cell
cellar
cellars
celled
cellist
cellists
cellophane
cells
cellular
cellulose
cement
cemented
cementing
cements
cemeteries
cemetery
censor
censored
censoring
censors
censorship
censure
censured
censurer
censures
census
censuses
cent
centaur
centenary
centennial
center
centered
centering
centerpiece
centerpieces
centers
centigrade
centimeter
centimeters
centipede
centipedes
central
centralism
centralist
centralization
centralize
centralized
centralizes
centralizing
centrally
centrifugal
centrifuge
centripetal
centrist
centroid
cents
centuries
century
ceramic
cereal
cereals
cerebellum
cerebral
ceremonial
ceremonially
ceremonialness
ceremonies
ceremony
certain
certainly
certainties
certainty
certifiable
certificate
certificates
certification
certifications
certified
certifier
certifiers
certifies
certify
certifying
cessation
cessations
chafe
chafer
chaff
chaffer
chaffing
chafing
chagrin
chain
chained
chaining
chains
chair
chaired
chairing
chairlady
chairman
chairmen
chairperson
chairpersons
chairs
chairwoman
chairwomen
chalice
chalices
chalk
chalked
chalking
chalks
challenge
challenged
challenger
challengers
challenges
challenging
chamber
chambered
chamberlain
chamberlains
chambermaid
chameleon
champagne
champion
championed
championing
champions
championship
championships
chance
chanced
chancellor
chancery
chances
chancing
chandelier
chandeliers
change
changeability
changeable
changeably
changed
changeover
changer
changers
changes
changing
channel
channeled
channeling
channelled
channeller
channellers
channelling
channels
chant
chanted
chanter
chanticleer
chanticleers
chanting
chants
chaos
chaotic
chap
chapel
chapels
chaperon
chaperone
chaperoned
chaplain
chaplains
chaps
chapter
chapters
char
character
characteristic
characteristically
characteristics
characterizable
characterization
characterizations
characterize
characterized
characterizer
characterizers
characterizes
characterizing
characters
charcoal
charcoaled
charge
chargeable
charged
charger
chargers
charges
charging
chariot
chariots
charisma
charismatic
charitable
charitableness
charities
charity
charm
charmed
charmer
charmers
charming
charmingly
charms
chars
chart
chartable
charted
charter
chartered
chartering
charters
charting
chartings
chartreuse
charts
chase
chased
chaser
chasers
chases
chasing
chasm
chasms
chassis
chaste
chastely
chasteness
chastise
chastised
chastiser
chastisers
chastises
chastising
chastity
chat
chateau
chateaus
chattel
chatter
chattered
chatterer
chattering
chatters
chatting
chatty
chauffeur
chauffeured
cheap
cheapen
cheapened
cheapening
cheapens
cheaper
cheapest
cheaply
cheapness
cheat
cheated
cheater
cheaters
cheating
cheats
check
checkable
checkbook
checkbooks
checked
checker
checkerboard
checkerboarded
checkerboarding
checkers
checking
checklist
checkout
checkpoint
checkpoints
checks
checksum
checksummed
checksumming
checksums
checkup
cheek
cheekbone
cheeks
cheeky
cheer
cheered
cheerer
cheerful
cheerfully
cheerfulness
cheerily
cheeriness
cheering
cheerleader
cheerless
cheerlessly
cheerlessness
cheers
cheery
cheese
cheesecloth
cheeses
cheesy
cheetah
chef
chefs
chemical
chemically
chemicals
chemise
chemist
chemistries
chemistry
chemists
cherish
cherished
cherishes
cherishing
cherries
cherry
cherub
cherubim
cherubs
chess
chest
chestnut
chestnuts
chests
chew
chewed
chewer
chewers
chewing
chews
chic
chicanery
chick
chickadee
chickadees
chicken
chickens
chicks
chide
chided
chides
chiding
chief
chiefly
chiefs
chieftain
chieftains
chiffon
child
childbirth
childhood
childish
childishly
childishness
childlike
children
chili
chill
chilled
chiller
chillers
chillier
chilliness
chilling
chillingly
chills
chilly
chime
chimera
chimes
chimney
chimneys
chimpanzee
chin
chink
chinked
chinks
chinned
chinner
chinners
chinning
chins
chintz
chip
chipmunk
chipmunks
chips
chiropractor
chirp
chirped
chirping
chirps
chisel
chiseled
chiseler
chisels
chit
chivalrous
chivalrously
chivalrousness
chivalry
chlorine
chloroform
chlorophyll
chloroplast
chloroplasts
chock
chocks
chocolate
chocolates
choice
choices
choicest
choir
choirs
choke
choked
choker
chokers
chokes
choking
cholera
choose
chooser
choosers
chooses
choosing
chop
chopped
chopper
choppers
chopping
choppy
chops
choral
chord
chordate
chorded
chording
chords
chore
choreograph
choreography
chores
choring
chortle
chorus
chorused
choruses
chose
chosen
chowder
christen
christened
christening
christens
chromatogram
chromatograph
chromatography
chrome
chromium
chromosphere
chronic
chronicle
chronicled
chronicler
chroniclers
chronicles
chronograph
chronography
chronological
chronologically
chronologies
chronology
chrysanthemum
chubbier
chubbiest
chubbiness
chubby
chuck
chuckle
chuckled
chuckles
chucks
chum
chunk
chunks
chunky
church
churches
churchgoer
churchgoing
churchly
churchman
churchmen
churchwoman
churchwomen
churchyard
churchyards
churn
churned
churning
churns
chute
chutes
chutzpah
cicada
cider
cigar
cigarette
cigarettes
cigars
cilia
cinder
cinders
cinema
cinematic
cinnamon
cipher
ciphers
ciphertext
ciphertexts
circa
circle
circled
circles
circlet
circling
circuit
circuitous
circuitously
circuitry
circuits
circulant
circular
circularity
circularly
circulate
circulated
circulates
circulating
circulation
circumcise
circumcision
circumference
circumflex
circumlocution
circumlocutions
circumnavigate
circumnavigated
circumnavigates
circumpolar
circumscribe
circumscribed
circumscribing
circumscription
circumspect
circumspection
circumspectly
circumstance
circumstanced
circumstances
circumstantial
circumstantially
circumvent
circumventable
circumvented
circumventing
circumvents
circus
circuses
cistern
cisterns
citadel
citadels
citation
citations
cite
cited
cites
cities
citing
citizen
citizens
citizenship
citrus
city
cityscape
citywide
civet
civic
civics
civil
civilian
civilians
civility
civilization
civilizations
civilize
civilized
civilizes
civilizing
civilly
clad
cladding
claim
claimable
claimant
claimants
claimed
claiming
claims
clairvoyant
clairvoyantly
clam
clamber
clambered
clambering
clambers
clamor
clamored
clamoring
clamorous
clamors
clamp
clamped
clamping
clamps
clams
clan
clandestine
clang
clanged
clanging
clangs
clank
clannish
clap
clapboard
clapping
claps
clarification
clarifications
clarified
clarifies
clarify
clarifying
clarinet
clarity
clash
clashed
clashes
clashing
clasp
clasped
clasping
clasps
class
classed
classes
classic
classical
classically
classics
classifiable
classification
classifications
classified
classifier
classifiers
classifies
classify
classifying
classmate
classmates
classroom
classrooms
classy
clatter
clattered
clattering
clause
clauses
claustrophobia
claustrophobic
claw
clawed
clawing
claws
clay
clays
clean
cleaned
cleaner
cleaners
cleanest
cleaning
cleanliness
cleanly
cleanness
cleans
cleanse
cleansed
cleanser
cleansers
cleanses
cleansing
cleanup
clear
clearance
clearances
cleared
clearer
clearest
clearing
clearings
clearly
clearness
clears
cleavage
cleave
cleaved
cleaver
cleavers
cleaves
cleaving
cleft
clefts
clemency
clement
clench
clenched
clenches
clergy
clergyman
clergymen
clerical
clerk
clerked
clerking
clerks
clever
cleverer
cleverest
cleverly
cleverness
cliche
cliches
click
clicked
clicking
clicks
client
clientele
clients
cliff
cliffs
climate
climates
climatic
climatically
climatology
climax
climaxed
climaxes
climb
climbed
climber
climbers
climbing
climbs
clime
climes
clinch
clinched
clincher
clinches
cling
clinging
clings
clinic
clinical
clinically
clinician
clinics
clink
clinked
clinker
clip
clipboard
clipped
clipper
clippers
clipping
clippings
clips
clique
cliques
clitoris
cloak
cloakroom
cloaks
clobber
clobbered
clobbering
clobbers
clock
clocked
clocker
clockers
clocking
clockings
clocks
clockwatcher
clockwise
clockwork
clod
clods
clog
clogged
clogging
clogs
cloister
cloisters
clone
cloned
clones
cloning
close
closed
closely
closeness
closenesses
closer
closers
closes
closest
closet
closeted
closets
closeup
closing
closure
closures
clot
cloth
clothe
clothed
clothes
clotheshorse
clothesline
clothing
clotting
cloture
cloud
cloudburst
clouded
cloudier
cloudiest
cloudiness
clouding
cloudless
clouds
cloudy
clout
clove
clover
cloves
clown
clowning
clowns
club
clubbed
clubbing
clubhouse
clubroom
clubs
cluck
clucked
clucking
clucks
clue
clues
clump
clumped
clumping
clumps
clumsily
clumsiness
clumsy
clung
cluster
clustered
clustering
clusterings
clusters
clutch
clutched
clutches
clutching
clutter
cluttered
cluttering
clutters
coach
coached
coacher
coaches
coaching
coachman
coachmen
coagulate
coal
coalesce
coalesced
coalesces
coalescing
coalition
coals
coarse
coarsely
coarsen
coarsened
coarseness
coarser
coarsest
coast
coastal
coasted
coaster
coasters
coasting
coastline
coasts
coat
coated
coating
coatings
coats
coattail
coauthor
coax
coaxed
coaxer
coaxes
coaxial
coaxing
cobalt
cobble
cobbler
cobblers
cobblestone
cobra
cobweb
cobwebs
coca
cocaine
cock
cocked
cocking
cockpit
cockroach
cocks
cocktail
cocktails
cocky
coco
cocoa
coconut
coconuts
cocoon
cocoons
cod
coddle
code
coded
codeine
coder
coders
codes
codeword
codewords
codfish
codicil
codification
codifications
codified
codifier
codifiers
codifies
codify
codifying
coding
codings
codpiece
coed
coeditor
coeducation
coefficient
coefficients
coequal
coerce
coerced
coerces
coercible
coercing
coercion
coercive
coexist
coexisted
coexistence
coexisting
coexists
cofactor
coffee
coffeecup
coffeepot
coffees
coffer
coffers
coffin
coffins
cog
cogent
cogently
cogitate
cogitated
cogitates
cogitating
cogitation
cognac
cognition
cognitive
cognitively
cognizance
cognizant
cogs
cohabitation
cohabitations
cohere
cohered
coherence
coherent
coherently
coheres
cohering
cohesion
cohesive
cohesively
cohesiveness
cohort
coil
coiled
coiling
coils
coin
coinage
coincide
coincided
coincidence
coincidences
coincident
coincidental
coincides
coinciding
coined
coiner
coining
coins
coke
cokes
colander
cold
colder
coldest
coldly
coldness
colds
colicky
coliform
coliseum
collaborate
collaborated
collaborates
collaborating
collaboration
collaborations
collaborative
collaborator
collaborators
collagen
collapse
collapsed
collapses
collapsible
collapsing
collar
collarbone
collared
collaring
collars
collate
collateral
colleague
colleagues
collect
collected
collectible
collecting
collection
collections
collective
collectively
collectives
collector
collectors
collects
college
colleges
collegian
collegiate
collide
collided
collides
colliding
collie
collies
collision
collisions
colloidal
colloquia
colloquial
colloquium
colloquy
collusion
colon
colonel
colonels
colonial
colonially
colonials
colonies
colonist
colonists
colonization
colonize
colonized
colonizer
colonizers
colonizes
colonizing
colons
colony
color
colored
colorer
colorers
colorful
coloring
colorings
colorless
colors
colossal
colt
colts
column
columnize
columnized
columnizes
columnizing
columns
comb
combat
combatant
combatants
combated
combating
combative
combats
combed
comber
combers
combination
combinational
combinations
combinator
combinatorial
combinatorially
combinatoric
combinatorics
combinators
combine
combined
combines
combing
combings
combining
combs
combustible
combustion
come
comeback
comedian
comedians
comedic
comedies
comedy
comeliness
comely
comer
comers
comes
comestible
comet
cometary
comets
comfort
comfortabilities
comfortability
comfortable
comfortably
comforted
comforter
comforters
comforting
comfortingly
comforts
comic
comical
comically
comics
coming
comings
comma
command
commandant
commandants
commanded
commandeer
commander
commanders
commanding
commandingly
commandment
commandments
commando
commands
commas
commemorate
commemorated
commemorates
commemorating
commemoration
commemorative
commence
commenced
commencement
commencements
commences
commencing
commend
commendation
commendations
commended
commending
commends
commensurate
comment
commentaries
commentary
commentator
commentators
commented
commenting
comments
commerce
commercial
commercially
commercialness
commercials
commission
commissioned
commissioner
commissioners
commissioning
commissions
commit
commitment
commitments
commits
committed
committee
committeeman
committeemen
committees
committeewoman
committeewomen
committing
commodities
commodity
commodore
commodores
common
commonalities
commonality
commoner
commoners
commonest
commonly
commonness
commonplace
commonplaces
commons
commonwealth
commonwealths
commotion
communal
communally
commune
communes
communicant
communicants
communicate
communicated
communicates
communicating
communication
communications
communicative
communicator
communicators
communion
communist
communists
communities
community
commutative
commutativity
commute
commuted
commuter
commuters
commutes
commuting
compact
compacted
compacter
compactest
compacting
compaction
compactly
compactness
compactor
compactors
compacts
companies
companion
companionable
companions
companionship
company
comparability
comparable
comparably
comparative
comparatively
comparatives
comparator
comparators
compare
compared
compares
comparing
comparison
comparisons
compartment
compartmentalize
compartmentalized
compartmentalizes
compartmentalizing
compartmented
compartments
compass
compassion
compassionate
compassionately
compatibilities
compatibility
compatible
compatibles
compatibly
compel
compelled
compelling
compellingly
compels
compendium
compensate
compensated
compensates
compensating
compensation
compensations
compensatory
compete
competed
competence
competency
competent
competently
competes
competing
competition
competitions
competitive
competitively
competitor
competitors
compilation
compilations
compile
compiled
compiler
compilers
compiles
compiling
complacency
complain
complained
complainer
complainers
complaining
complains
complaint
complaints
complement
complementary
complemented
complementer
complementers
complementing
complements
complete
completed
completely
completeness
completes
completing
completion
completions
complex
complexes
complexion
complexities
complexity
complexly
compliance
compliant
complicate
complicated
complicates
complicating
complication
complications
complicator
complicators
complicity
complied
compliment
complimentary
complimented
complimenter
complimenters
complimenting
compliments
comply
complying
component
componentry
components
componentwise
compose
composed
composedly
composer
composers
composes
composing
composite
composites
composition
compositional
compositions
compost
composure
compound
compounded
compounding
compounds
comprehend
comprehended
comprehending
comprehends
comprehensibility
comprehensible
comprehension
comprehensive
comprehensively
compress
compressed
compresses
compressible
compressing
compression
compressive
compressor
comprise
comprised
comprises
comprising
compromise
compromised
compromiser
compromisers
compromises
compromising
compromisingly
comptroller
comptrollers
compulsion
compulsions
compulsive
compulsory
compunction
computability
computable
computation
computational
computationally
computations
compute
computed
computer
computerize
computerized
computerizes
computerizing
computers
computes
computing
comrade
comradely
comrades
comradeship
con
concatenate
concatenated
concatenates
concatenating
concatenation
concatenations
concave
conceal
concealed
concealer
concealers
concealing
concealment
conceals
concede
conceded
concedes
conceding
conceit
conceited
conceits
conceivable
conceivably
conceive
conceived
conceives
conceiving
concentrate
concentrated
concentrates
concentrating
concentration
concentrations
concentrator
concentrators
concentric
concept
conception
conceptions
concepts
conceptual
conceptualization
conceptualizations
conceptualize
conceptualized
conceptualizes
conceptualizing
conceptually
concern
concerned
concernedly
concerning
concerns
concert
concerted
concertmaster
concerto
concerts
concession
concessions
conciliate
conciliatory
concise
concisely
conciseness
conclave
conclude
concluded
concludes
concluding
conclusion
conclusions
conclusive
conclusively
concoct
concomitant
concord
concordant
concourse
concrete
concretely
concreteness
concretes
concretion
concubine
concur
concurred
concurrence
concurrencies
concurrency
concurrent
concurrently
concurring
concurs
concussion
condemn
condemnation
condemnations
condemned
condemner
condemners
condemning
condemns
condensation
condense
condensed
condenser
condenses
condensing
condescend
condescending
condition
conditional
conditionally
conditionals
conditioned
conditioner
conditioners
conditioning
conditions
condom
condone
condoned
condones
condoning
conduce
conducive
conduciveness
conduct
conductance
conducted
conducting
conduction
conductive
conductivity
conductor
conductors
conducts
conduit
cone
cones
confectionery
confederacy
confederate
confederates
confederation
confederations
confer
conferee
conference
conferences
conferred
conferrer
conferrers
conferring
confers
confess
confessed
confesses
confessing
confession
confessions
confessor
confessors
confidant
confidants
confide
confided
confidence
confidences
confident
confidential
confidentiality
confidentially
confidently
confides
confiding
confidingly
configurable
configuration
configurations
configure
configured
configures
configuring
confine
confined
confinement
confinements
confiner
confines
confining
confirm
confirmation
confirmations
confirmatory
confirmed
confirming
confirms
confiscate
confiscated
confiscates
confiscating
confiscation
confiscations
conflagration
conflict
conflicted
conflicting
conflicts
confluent
confocal
conform
conformal
conformance
conformed
conforming
conformity
conforms
confound
confounded
confounding
confounds
confront
confrontation
confrontations
confronted
confronter
confronters
confronting
confronts
confuse
confused
confuser
confusers
confuses
confusing
confusingly
confusion
confusions
congenial
congenially
congenital
congest
congested
congestion
congestive
conglomerate
congratulate
congratulated
congratulation
congratulations
congratulatory
congregate
congregated
congregates
congregating
congregation
congregations
congress
congresses
congressional
congressionally
congressman
congressmen
congresswoman
congresswomen
congruence
congruent
conic
conifer
coniferous
conjecture
conjectured
conjectures
conjecturing
conjoined
conjugal
conjugate
conjunct
conjuncted
conjunction
conjunctions
conjunctive
conjunctively
conjuncts
conjuncture
conjure
conjured
conjurer
conjures
conjuring
connect
connected
connectedness
connecting
connection
connectionless
connections
connective
connectives
connectivity
connector
connectors
connects
connivance
connive
connoisseur
connoisseurs
connotation
connotative
connote
connoted
connotes
connoting
connubial
conquer
conquerable
conquered
conquerer
conquerers
conquering
conqueror
conquerors
conquers
conquest
conquests
conscience
consciences
conscientious
conscientiously
conscious
consciously
consciousness
conscript
conscription
consecrate
consecration
consecutive
consecutively
consensual
consensus
consent
consented
consenter
consenters
consenting
consents
consequence
consequences
consequent
consequential
consequentialities
consequentiality
consequently
consequents
conservation
conservationist
conservationists
conservations
conservatism
conservative
conservatively
conservatives
conservator
conserve
conserved
conserves
conserving
consider
considerable
considerably
considerate
considerately
consideration
considerations
considered
considering
considers
consign
consigned
consigning
consigns
consist
consisted
consistency
consistent
consistently
consisting
consists
consolable
consolation
consolations
console
consoled
consoler
consolers
consoles
consolidate
consolidated
consolidates
consolidating
consolidation
consoling
consolingly
consonant
consonants
consort
consorted
consorting
consortium
consorts
conspicuous
conspicuously
conspiracies
conspiracy
conspirator
conspirators
conspire
conspired
conspires
conspiring
constable
constables
constancy
constant
constantly
constants
constellation
constellations
consternation
constituencies
constituency
constituent
constituents
constitute
constituted
constitutes
constituting
constitution
constitutional
constitutionality
constitutionally
constitutions
constitutive
constrain
constrained
constraining
constrains
constraint
constraints
constrict
construct
constructed
constructibility
constructible
constructing
construction
constructions
constructive
constructively
constructor
constructors
constructs
construe
construed
construing
consul
consular
consulate
consulates
consuls
consult
consultant
consultants
consultation
consultations
consultative
consulted
consulting
consults
consumable
consume
consumed
consumer
consumers
consumes
consuming
consummate
consummated
consummately
consummation
consumption
consumptions
consumptive
consumptively
contact
contacted
contacting
contacts
contagion
contagious
contagiously
contain
containable
contained
container
containers
containing
containment
containments
contains
contaminate
contaminated
contaminates
contaminating
contamination
contemplate
contemplated
contemplates
contemplating
contemplation
contemplations
contemplative
contemporaries
contemporariness
contemporary
contempt
contemptible
contemptuous
contemptuously
contend
contended
contender
contenders
contending
contends
content
contented
contenting
contention
contentions
contently
contentment
contents
contest
contestable
contestant
contested
contester
contesters
contesting
contests
context
contexts
contextual
contextually
contiguity
contiguous
contiguously
continent
continental
continentally
continents
contingencies
contingency
contingent
contingents
continual
continually
continuance
continuances
continuation
continuations
continue
continued
continues
continuing
continuities
continuity
continuous
continuously
continuum
contortions
contour
contoured
contouring
contours
contraband
contraception
contraceptive
contract
contracted
contracting
contraction
contractions
contractor
contractors
contracts
contractual
contractually
contradict
contradicted
contradicting
contradiction
contradictions
contradictory
contradicts
contradistinction
contradistinctions
contrapositive
contrapositives
contraption
contraptions
contrariness
contrary
contrast
contrasted
contraster
contrasters
contrasting
contrastingly
contrasts
contribute
contributed
contributes
contributing
contribution
contributions
contributor
contributorily
contributors
contributory
contrite
contrition
contrivance
contrivances
contrive
contrived
contriver
contrives
contriving
control
controllability
controllable
controllably
controlled
controller
controllers
controlling
controls
controversial
controversies
controversy
controvertible
contumacious
contumacy
conundrum
conundrums
convalescent
convect
convene
convened
convenes
convenience
conveniences
convenient
conveniently
convening
convent
convention
conventional
conventionally
conventions
convents
converge
converged
convergence
convergent
converges
converging
conversant
conversantly
conversation
conversational
conversationally
conversations
converse
conversed
conversely
converses
conversing
conversion
conversions
convert
converted
converter
converters
convertibility
convertible
converting
converts
convex
convey
conveyance
conveyances
conveyed
conveyer
conveyers
conveying
conveyor
conveys
convict
convicted
convicting
conviction
convictions
convicts
convince
convinced
convincer
convincers
convinces
convincing
convincingly
convivial
convoke
convoluted
convolution
convoy
convoyed
convoying
convoys
convulse
convulsion
convulsions
coo
cooing
cook
cookbook
cooked
cookery
cookie
cookies
cooking
cooks
cooky
cool
cooled
cooler
coolers
coolest
coolie
coolies
cooling
coolly
coolness
cools
coon
coons
coop
cooped
cooper
cooperate
cooperated
cooperates
cooperating
cooperation
cooperations
cooperative
cooperatively
cooperatives
cooperator
cooperators
coopers
coops
coordinate
coordinated
coordinates
coordinating
coordination
coordinations
coordinator
coordinators
cop
cope
coped
copes
copied
copier
copiers
copies
coping
copings
copious
copiously
copiousness
coplanar
copper
copperhead
coppers
copra
coprocessor
cops
copse
copy
copying
copyright
copyrightable
copyrighted
copyrights
copywriter
coquette
coral
cord
corded
corder
cordial
cordiality
cordially
cords
core
cored
corer
corers
cores
coriander
coring
cork
corked
corker
corkers
corking
corks
corkscrew
cormorant
corn
cornea
corner
cornered
corners
cornerstone
cornerstones
cornet
cornfield
cornfields
corning
cornmeal
corns
cornstarch
cornucopia
corny
corollaries
corollary
coronaries
coronary
coronation
coroner
coronet
coronets
coroutine
coroutines
corporal
corporals
corporate
corporately
corporation
corporations
corps
corpse
corpses
corpulent
corpus
corpuscular
corral
correct
correctable
corrected
correcting
correction
corrections
corrective
correctively
correctives
correctly
correctness
corrector
corrects
correlate
correlated
correlates
correlating
correlation
correlations
correlative
correspond
corresponded
correspondence
correspondences
correspondent
correspondents
corresponding
correspondingly
corresponds
corridor
corridors
corrigenda
corrigendum
corrigible
corroborate
corroborated
corroborates
corroborating
corroboration
corroborations
corroborative
corrode
corrosion
corrosive
corrugate
corrupt
corrupted
corrupter
corruptible
corrupting
corruption
corruptions
corrupts
corset
cortex
cortical
cosine
cosines
cosmetic
cosmetics
cosmic
cosmology
cosmopolitan
cosmos
cosponsor
cost
costed
costing
costly
costs
costume
costumed
costumer
costumes
costuming
cosy
cot
cotangent
cotillion
cots
cottage
cottager
cottages
cotton
cottonmouth
cottons
cottonseed
cottonwood
cotyledon
cotyledons
couch
couched
couches
couching
cougar
cough
coughed
coughing
coughs
could
coulomb
council
councillor
councillors
councilman
councilmen
councils
councilwoman
councilwomen
counsel
counseled
counseling
counselled
counselling
counsellor
counsellors
counselor
counselors
counsels
count
countable
countably
counted
countenance
counter
counteract
counteracted
counteracting
counteractive
counterargument
counterattack
counterbalance
counterclockwise
countered
counterexample
counterexamples
counterfeit
counterfeited
counterfeiter
counterfeiting
counterflow
countering
counterintuitive
counterman
countermeasure
countermeasures
countermen
counterpart
counterparts
counterpoint
counterpointing
counterpoise
counterproductive
counterproposal
counterrevolution
counters
countersink
countersunk
countess
counties
counting
countless
countries
country
countryman
countrymen
countryside
countrywide
counts
county
countywide
couple
coupled
coupler
couplers
couples
coupling
couplings
coupon
coupons
courage
courageous
courageously
courier
couriers
course
coursed
courser
courses
coursing
court
courted
courteous
courteously
courter
courters
courtesan
courtesies
courtesy
courthouse
courthouses
courtier
courtiers
courting
courtly
courtroom
courtrooms
courts
courtship
courtyard
courtyards
cousin
cousins
covalent
covariant
cove
covenant
covenants
cover
coverable
coverage
covered
covering
coverings
coverlet
coverlets
covers
covert
covertly
coves
covet
coveted
coveting
covetous
covetousness
covets
cow
coward
cowardice
cowardly
cowboy
cowboys
cowed
cower
cowered
cowerer
cowerers
cowering
coweringly
cowers
cowherd
cowhide
cowing
cowl
cowlick
cowling
cowls
coworker
cows
cowslip
cowslips
coyote
coyotes
coypu
cozier
coziness
cozy
crab
crabapple
crabs
crack
cracked
cracker
crackers
cracking
crackle
crackled
crackles
crackling
crackpot
cracks
cradle
cradled
cradles
craft
crafted
crafter
craftiness
crafting
crafts
craftsman
craftsmen
craftspeople
craftsperson
crafty
crag
craggy
crags
cram
cramming
cramp
cramps
crams
cranberries
cranberry
crane
cranes
crania
cranium
crank
crankcase
cranked
crankier
crankiest
crankily
cranking
cranks
crankshaft
cranky
cranny
crash
crashed
crasher
crashers
crashes
crashing
crass
crate
crater
craters
crates
cravat
cravats
crave
craved
craven
craves
craving
crawl
crawled
crawler
crawlers
crawling
crawls
crayon
craze
crazed
crazes
crazier
craziest
crazily
craziness
crazing
crazy
creak
creaked
creaking
creaks
creaky
cream
creamed
creamer
creamers
creamery
creaming
creams
creamy
crease
creased
creases
creasing
create
created
creates
creating
creation
creations
creative
creatively
creativeness
creativity
creator
creators
creature
creatures
credence
credential
credibility
credible
credibly
credit
creditable
creditably
credited
crediting
creditor
creditors
credits
credulity
credulous
credulousness
creed
creeds
creek
creeks
creep
creeper
creepers
creeping
creeps
creepy
cremate
cremated
cremates
cremating
cremation
cremations
crematory
crepe
crept
crescent
crescents
crest
crested
crestfallen
crests
cretin
crevice
crevices
crew
crewcut
crewed
crewing
crews
crib
cribs
cricket
crickets
cried
crier
criers
cries
crime
crimes
criminal
criminally
criminals
criminate
crimson
crimsoning
cringe
cringed
cringes
cringing
cripple
crippled
cripples
crippling
crises
crisis
crisp
crisply
crispness
crisscross
criteria
criterion
critic
critical
critically
criticism
criticisms
criticize
criticized
criticizes
criticizing
critics
critique
critiques
critiquing
critter
croak
croaked
croaking
croaks
crochet
crochets
crock
crockery
crocks
crocodile
crocus
croft
crook
crooked
crooks
crop
cropped
cropper
croppers
cropping
crops
cross
crossable
crossbar
crossbars
crossed
crosser
crossers
crosses
crossing
crossings
crossly
crossover
crossovers
crosspoint
crossroad
crosstalk
crosswalk
crossword
crosswords
crotch
crotchety
crouch
crouched
crouching
crow
crowd
crowded
crowder
crowding
crowds
crowed
crowing
crown
crowned
crowning
crowns
crows
crucial
crucially
crucible
crucified
crucifies
crucifix
crucifixion
crucify
crucifying
crud
cruddy
crude
crudely
crudeness
cruder
crudest
cruel
crueler
cruelest
cruelly
cruelty
cruise
cruiser
cruisers
cruises
cruising
crumb
crumble
crumbled
crumbles
crumbling
crumbly
crumbs
crummy
crumple
crumpled
crumples
crumpling
crunch
crunched
crunches
crunchier
crunchiest
crunching
crunchy
crusade
crusader
crusaders
crusades
crusading
crush
crushable
crushed
crusher
crushers
crushes
crushing
crushingly
crust
crustacean
crustaceans
crusts
crutch
crutches
crux
cruxes
cry
crying
cryogenic
crypt
cryptanalysis
cryptanalyst
cryptanalytic
cryptic
cryptogram
cryptographer
cryptographic
cryptographically
cryptography
cryptologist
cryptology
crystal
crystalline
crystallize
crystallized
crystallizes
crystallizing
crystals
cub
cubbyhole
cube
cubed
cubes
cubic
cubs
cuckoo
cuckoos
cucumber
cucumbers
cuddle
cuddled
cuddly
cudgel
cudgels
cue
cued
cues
cuff
cufflink
cuffs
cuisine
culinary
cull
culled
culler
culling
culls
culminate
culminated
culminates
culminating
culmination
culpa
culpable
culprit
culprits
cult
cultivable
cultivate
cultivated
cultivates
cultivating
cultivation
cultivations
cultivator
cultivators
cults
cultural
culturally
culture
cultured
cultures
culturing
cumbersome
cumulative
cumulatively
cunnilingus
cunning
cunningly
cup
cupboard
cupboards
cupful
cupped
cupping
cups
curable
curably
curb
curbing
curbs
curd
curdle
cure
cured
cures
curfew
curfews
curing
curiosities
curiosity
curious
curiouser
curiousest
curiously
curl
curled
curler
curlers
curlicue
curling
curls
curly
currant
currants
currencies
currency
current
currently
currentness
currents
curricular
curriculum
curriculums
curried
curries
curry
currying
curs
curse
cursed
curses
cursing
cursive
cursor
cursorily
cursors
cursory
curt
curtail
curtailed
curtails
curtain
curtained
curtains
curtate
curtly
curtness
curtsies
curtsy
curvaceous
curvature
curve
curved
curves
curvilinear
curving
cushion
cushioned
cushioning
cushions
cusp
cusps
custard
custodial
custodian
custodians
custody
custom
customarily
customary
customer
customers
customizable
customization
customizations
customize
customized
customizer
customizers
customizes
customizing
customs
cut
cutaneous
cutback
cute
cutest
cutlass
cutlet
cutoff
cutout
cutover
cuts
cutter
cutters
cutthroat
cutting
cuttingly
cuttings
cuttlefish
cyanide
cybernetic
cybernetics
cyberspace
cycle
cycled
cycles
cyclic
cyclically
cycling
cycloid
cycloidal
cycloids
cyclone
cyclones
cyclotron
cyclotrons
cylinder
cylinders
cylindrical
cymbal
cymbals
cynic
cynical
cynically
cypress
cyst
cysts
cytology
cytoplasm
czar
dabble
dabbled
dabbler
dabbles
dabbling
dactyl
dactylic
dad
daddy
dads
daemon
daemons
daffodil
daffodils
dagger
dahlia
dailies
daily
daintily
daintiness
dainty
dairy
daisies
daisy
dale
dales
dam
damage
damaged
damager
damagers
damages
damaging
damask
dame
damming
damn
damnation
damned
damning
damns
damp
dampen
dampens
damper
damping
dampness
dams
damsel
damsels
dance
danced
dancer
dancers
dances
dancing
dandelion
dandelions
dandy
danger
dangerous
dangerously
dangers
dangle
dangled
dangles
dangling
dare
dared
darer
darers
dares
daresay
daring
daringly
dark
darken
darker
darkest
darkly
darkness
darkroom
darling
darlings
darn
darned
darner
darning
darns
dart
darted
darter
darting
darts
dash
dashboard
dashed
dasher
dashers
dashes
dashing
dashingly
data
database
databases
datagram
datagrams
date
dated
dateline
dater
dates
dating
dative
datum
daughter
daughterly
daughters
daunt
daunted
dauntless
dawn
dawned
dawning
dawns
day
daybreak
daydream
daydreaming
daydreams
daylight
daylights
days
daytime
daze
dazed
dazzle
dazzled
dazzler
dazzles
dazzling
dazzlingly
deacon
deacons
deactivate
dead
deaden
deadline
deadlines
deadlock
deadlocked
deadlocking
deadlocks
deadly
deadness
deadwood
deaf
deafen
deafer
deafest
deafness
deal
dealer
dealers
dealership
dealing
dealings
deallocate
deallocated
deallocating
deallocation
deallocations
deals
dealt
dean
deans
dear
dearer
dearest
dearly
dearness
dearth
dearths
death
deathbed
deathly
deaths
debacle
debar
debase
debatable
debate
debated
debater
debaters
debates
debating
debauch
debauchery
debilitate
debilitated
debilitates
debilitating
debility
debit
debited
debrief
debris
debt
debtor
debts
debug
debugged
debugger
debuggers
debugging
debugs
debunk
debutante
decade
decadence
decadent
decadently
decades
decal
decathlon
decay
decayed
decaying
decays
decease
deceased
deceases
deceasing
decedent
deceit
deceitful
deceitfully
deceitfulness
deceive
deceived
deceiver
deceivers
deceives
deceiving
decelerate
decelerated
decelerates
decelerating
deceleration
decencies
decency
decennial
decent
decently
decentralization
decentralized
deception
deceptions
deceptive
deceptively
decertify
decibel
decidability
decidable
decide
decided
decidedly
decides
deciding
deciduous
decimal
decimals
decimate
decimated
decimates
decimating
decimation
decipher
deciphered
decipherer
deciphering
deciphers
decision
decisions
decisive
decisively
decisiveness
deck
decked
decking
deckings
decks
declaration
declarations
declarative
declaratively
declaratives
declarator
declaratory
declare
declared
declarer
declarers
declares
declaring
declassify
declination
declinations
decline
declined
decliner
decliners
declines
declining
decode
decoded
decoder
decoders
decodes
decoding
decodings
decolletage
decollimate
decompile
decomposability
decomposable
decompose
decomposed
decomposes
decomposing
decomposition
decompositions
decompress
decompression
decorate
decorated
decorates
decorating
decoration
decorations
decorative
decorum
decouple
decoupled
decouples
decoupling
decoy
decoys
decrease
decreased
decreases
decreasing
decreasingly
decree
decreed
decreeing
decrees
decrement
decremented
decrementing
decrements
decrypt
decrypted
decrypting
decryption
decrypts
dedicate
dedicated
dedicates
dedicating
dedication
deduce
deduced
deducer
deduces
deducible
deducing
deduct
deducted
deductible
deducting
deduction
deductions
deductive
deed
deeded
deeding
deeds
deem
deemed
deeming
deemphasize
deemphasized
deemphasizes
deemphasizing
deems
deep
deepen
deepened
deepening
deepens
deeper
deepest
deeply
deeps
deer
deface
default
defaulted
defaulter
defaulting
defaults
defeat
defeated
defeating
defeats
defecate
defect
defected
defecting
defection
defections
defective
defects
defend
defendant
defendants
defended
defender
defenders
defending
defends
defenestrate
defenestrated
defenestrates
defenestrating
defenestration
defense
defenseless
defenses
defensible
defensive
defer
deference
deferment
deferments
deferrable
deferred
deferrer
deferrers
deferring
defers
defiance
defiant
defiantly
deficiencies
deficiency
deficient
deficit
deficits
defied
defies
defile
defiling
definable
define
defined
definer
defines
defining
definite
definitely
definiteness
definition
definitional
definitions
definitive
deflate
deflater
deflect
defocus
deforest
deforestation
deform
deformation
deformations
deformed
deformities
deformity
defraud
defray
defrost
deftly
defunct
defy
defying
degeneracy
degenerate
degenerated
degenerates
degenerating
degeneration
degenerative
degradable
degradation
degradations
degrade
degraded
degrades
degrading
degree
degrees
dehumidify
dehydrate
deify
deign
deigned
deigning
deigns
deities
deity
dejected
dejectedly
delay
delayed
delaying
delays
delegate
delegated
delegates
delegating
delegation
delegations
delete
deleted
deleter
deleterious
deletes
deleting
deletion
deletions
deliberate
deliberated
deliberately
deliberateness
deliberates
deliberating
deliberation
deliberations
deliberative
deliberator
deliberators
delicacies
delicacy
delicate
delicately
delicatessen
delicious
deliciously
delight
delighted
delightedly
delightful
delightfully
delighting
delights
delimit
delimitation
delimited
delimiter
delimiters
delimiting
delimits
delineament
delineate
delineated
delineates
delineating
delineation
delinquency
delinquent
delirious
deliriously
delirium
deliver
deliverable
deliverables
deliverance
delivered
deliverer
deliverers
deliveries
delivering
delivers
delivery
dell
dells
delta
deltas
delude
deluded
deludes
deluding
deluge
deluged
deluges
delusion
delusions
deluxe
delve
delves
delving
demagnify
demagogue
demand
demanded
demander
demanding
demandingly
demands
demarcate
demeanor
demented
demerit
demigod
demise
demo
democracies
democracy
democrat
democratic
democratically
democrats
demodulate
demodulator
demographic
demolish
demolished
demolishes
demolition
demon
demoniac
demonic
demons
demonstrable
demonstrate
demonstrated
demonstrates
demonstrating
demonstration
demonstrations
demonstrative
demonstratively
demonstrator
demonstrators
demoralize
demoralized
demoralizes
demoralizing
demote
demountable
demultiplex
demultiplexed
demultiplexer
demultiplexers
demultiplexing
demur
demythologize
den
denature
deniable
denial
denials
denied
denier
denies
denigrate
denigrated
denigrates
denigrating
denizen
denominate
denomination
denominations
denominator
denominators
denotable
denotation
denotational
denotationally
denotations
denotative
denote
denoted
denotes
denoting
denounce
denounced
denounces
denouncing
dens
dense
densely
denseness
denser
densest
densities
density
dent
dental
dentally
dented
denting
dentist
dentistry
dentists
dents
denture
denude
denumerable
denunciate
denunciation
deny
denying
deodorant
deoxyribonucleic
depart
departed
departing
department
departmental
departments
departs
departure
departures
depend
dependability
dependable
dependably
depended
dependence
dependencies
dependency
dependent
dependently
dependents
depending
depends
depict
depicted
depicting
depicts
deplete
depleted
depletes
depleting
depletion
depletions
deplorable
deplore
deplored
deplores
deploring
deploy
deployed
deploying
deployment
deployments
deploys
deport
deportation
deportee
deportment
depose
deposed
deposes
deposit
depositary
deposited
depositing
deposition
depositions
depositor
depositors
depository
deposits
depot
depots
deprave
depraved
depravity
deprecate
depreciate
depreciated
depreciates
depreciation
depress
depressed
depresses
depressing
depression
depressions
deprivation
deprivations
deprive
deprived
deprives
depriving
depth
depths
deputies
deputy
dequeue
dequeued
dequeues
dequeuing
derail
derailed
derailing
derails
derby
dereference
deregulate
deregulated
deride
derision
derivable
derivation
derivations
derivative
derivatives
derive
derived
derives
deriving
derogatory
derrick
derriere
dervish
descend
descendant
descendants
descended
descendent
descender
descenders
descending
descends
descent
descents
describable
describe
described
describer
describes
describing
description
descriptions
descriptive
descriptively
descriptives
descriptor
descriptors
descry
desecrate
desegregate
desert
deserted
deserter
deserters
deserting
desertion
desertions
deserts
deserve
deserved
deserves
deserving
deservingly
deservings
desiderata
desideratum
design
designate
designated
designates
designating
designation
designations
designator
designators
designed
designer
designers
designing
designs
desirability
desirable
desirably
desire
desired
desires
desiring
desirous
desist
desk
desks
desktop
desolate
desolately
desolation
desolations
despair
despaired
despairing
despairingly
despairs
despatch
despatched
desperado
desperate
desperately
desperation
despicable
despise
despised
despises
despising
despite
despoil
despondent
despot
despotic
despotism
despots
dessert
desserts
desiccate
destabilize
destination
destinations
destine
destined
destinies
destiny
destitute
destitution
destroy
destroyed
destroyer
destroyers
destroying
destroys
destruct
destruction
destructions
destructive
destructively
destructiveness
destructor
destuff
destuffing
destuffs
desuetude
desultory
desynchronize
detach
detached
detacher
detaches
detaching
detachment
detachments
detail
detailed
detailing
details
detain
detained
detaining
detains
detect
detectable
detectably
detected
detecting
detection
detections
detective
detectives
detector
detectors
detects
detente
detention
deter
detergent
deteriorate
deteriorated
deteriorates
deteriorating
deterioration
determinable
determinacy
determinant
determinants
determinate
determinately
determination
determinations
determinative
determine
determined
determiner
determiners
determines
determining
determinism
deterministic
deterministically
deterred
deterrent
deterring
detest
detestable
detested
detour
detract
detractor
detractors
detracts
detriment
detrimental
deuce
deus
deuterium
devastate
devastated
devastates
devastating
devastation
develop
developed
developer
developers
developing
development
developmental
developments
develops
deviant
deviants
deviate
deviated
deviates
deviating
deviation
deviations
device
devices
devil
devilish
devilishly
devils
devious
devise
devised
devises
devising
devisings
devoid
devolve
devote
devoted
devotedly
devotee
devotees
devotes
devoting
devotion
devotions
devour
devoured
devourer
devours
devout
devoutly
devoutness
dew
dewdrop
dewdrops
dewy
dexterity
diabetes
diabetic
diabolic
diachronic
diacritical
diadem
diagnosable
diagnose
diagnosed
diagnoses
diagnosing
diagnosis
diagnostic
diagnostician
diagnostics
diagonal
diagonally
diagonals
diagram
diagrammable
diagrammatic
diagrammatically
diagrammed
diagrammer
diagrammers
diagramming
diagrams
dial
dialect
dialectic
dialects
dialed
dialer
dialers
dialing
dialog
dialogs
dialogue
dialogues
dials
dialup
dialysis
diamagnetic
diameter
diameters
diametric
diametrically
diamond
diamonds
diaper
diapers
diaphragm
diaphragms
diaries
diarrhea
diary
diatribe
diatribes
dibble
dice
dichotomize
dichotomy
dickens
dicky
dictate
dictated
dictates
dictating
dictation
dictations
dictator
dictatorial
dictators
dictatorship
diction
dictionaries
dictionary
dictum
dictums
did
didactic
diddle
die
died
diehard
dielectric
dielectrics
diem
dies
diesel
diet
dietary
dieter
dieters
dietetic
dietician
dietitian
dietitians
diets
differ
differed
difference
differences
different
differentiable
differential
differentials
differentiate
differentiated
differentiates
differentiating
differentiation
differentiations
differentiators
differently
differer
differers
differing
differs
difficult
difficulties
difficultly
difficulty
diffract
diffuse
diffused
diffusely
diffuser
diffusers
diffuses
diffusible
diffusing
diffusion
diffusions
diffusive
dig
digest
digested
digestible
digesting
digestion
digestive
digests
digger
diggers
digging
diggings
digit
digital
digitalis
digitally
digitization
digitize
digitized
digitizes
digitizing
digits
dignified
dignify
dignitary
dignities
dignity
digram
digress
digressed
digresses
digressing
digression
digressions
digressive
digs
dihedral
dike
dikes
dilapidate
dilatation
dilate
dilated
dilates
dilating
dilation
dildo
dilemma
dilemmas
diligence
diligent
diligently
dill
dilogarithm
dilute
diluted
dilutes
diluting
dilution
dim
dime
dimension
dimensional
dimensionality
dimensionally
dimensioned
dimensioning
dimensions
dimes
diminish
diminished
diminishes
diminishing
diminution
diminutive
dimly
dimmed
dimmer
dimmers
dimmest
dimming
dimness
dimple
dims
din
dine
dined
diner
diners
dines
ding
dinghy
dinginess
dingo
dingy
dining
dinner
dinners
dinnertime
dinnerware
dinosaur
dint
diode
diodes
diopter
diorama
dioxide
dip
diphtheria
diphthong
diploma
diplomacy
diplomas
diplomat
diplomatic
diplomats
dipole
dipped
dipper
dippers
dipping
dippings
dips
dire
direct
directed
directing
direction
directional
directionality
directionally
directions
directive
directives
directly
directness
director
directorate
directories
directors
directory
directrices
directrix
directs
dirge
dirges
dirt
dirtier
dirtiest
dirtily
dirtiness
dirts
dirty
disabilities
disability
disable
disabled
disabler
disablers
disables
disabling
disadvantage
disadvantageous
disadvantages
disaffected
disaffection
disagree
disagreeable
disagreed
disagreeing
disagreement
disagreements
disagrees
disallow
disallowed
disallowing
disallows
disambiguate
disambiguated
disambiguates
disambiguating
disambiguation
disambiguations
disappear
disappearance
disappearances
disappeared
disappearing
disappears
disappoint
disappointed
disappointing
disappointment
disappointments
disapproval
disapprove
disapproved
disapproves
disarm
disarmament
disarmed
disarming
disarms
disassemble
disassembled
disassembles
disassembling
disassembly
disaster
disasters
disastrous
disastrously
disband
disbanded
disbanding
disbands
disburse
disbursed
disbursement
disbursements
disburses
disbursing
disc
discard
discarded
discarding
discards
discern
discerned
discernibility
discernible
discernibly
discerning
discerningly
discernment
discerns
discharge
discharged
discharges
discharging
disciple
disciples
disciplinary
discipline
disciplined
disciplines
disciplining
disclaim
disclaimed
disclaimer
disclaims
disclose
disclosed
discloses
disclosing
disclosure
disclosures
discomfort
disconcert
disconcerting
disconcertingly
disconnect
disconnected
disconnecting
disconnection
disconnects
discontent
discontented
discontinuance
discontinue
discontinued
discontinues
discontinuities
discontinuity
discontinuous
discord
discordant
discount
discounted
discounting
discounts
discourage
discouraged
discouragement
discourages
discouraging
discourse
discourses
discover
discovered
discoverer
discoverers
discoveries
discovering
discovers
discovery
discredit
discredited
discreet
discreetly
discrepancies
discrepancy
discrete
discretely
discreteness
discretion
discretionary
discriminant
discriminate
discriminated
discriminates
discriminating
discrimination
discriminatory
discs
discuss
discussant
discussed
discusses
discussing
discussion
discussions
disdain
disdaining
disdains
disease
diseased
diseases
disembowel
disengage
disengaged
disengages
disengaging
disentangle
disentangling
disfigure
disfigured
disfigures
disfiguring
disgorge
disgrace
disgraced
disgraceful
disgracefully
disgraces
disgruntle
disgruntled
disguise
disguised
disguises
disgust
disgusted
disgustedly
disgustful
disgusting
disgustingly
disgusts
dish
dishearten
disheartening
dished
dishes
dishevel
dishing
dishonest
dishonestly
dishonesty
dishonor
dishonorable
dishonored
dishonoring
dishonors
dishwasher
dishwashers
dishwashing
dishwater
disillusion
disillusioned
disillusioning
disillusionment
disillusionments
disinclined
disingenuous
disinterested
disinterestedness
disjoint
disjointed
disjointly
disjointness
disjunct
disjunction
disjunctions
disjunctive
disjunctively
disjuncts
disk
diskette
diskettes
disks
dislike
disliked
dislikes
disliking
dislocate
dislocated
dislocates
dislocating
dislocation
dislocations
dislodge
dislodged
dismal
dismally
dismay
dismayed
dismaying
dismember
dismembered
dismemberment
dismembers
dismiss
dismissal
dismissals
dismissed
dismisser
dismissers
dismisses
dismissing
dismount
dismounted
dismounting
dismounts
disobedience
disobedient
disobey
disobeyed
disobeying
disobeys
disorder
disordered
disorderly
disorders
disorganized
disown
disowned
disowning
disowns
disparage
disparate
disparities
disparity
dispassionate
dispatch
dispatched
dispatcher
dispatchers
dispatches
dispatching
dispel
dispell
dispelled
dispelling
dispels
dispensary
dispensation
dispense
dispensed
dispenser
dispensers
dispenses
dispensing
dispersal
disperse
dispersed
disperses
dispersing
dispersion
dispersions
displace
displaced
displacement
displacements
displaces
displacing
display
displayable
displayed
displayer
displaying
displays
displease
displeased
displeases
displeasing
displeasure
disposable
disposal
disposals
dispose
disposed
disposer
disposes
disposing
disposition
dispositions
dispossessed
disproportionate
disprove
disproved
disproves
disproving
dispute
disputed
disputer
disputers
disputes
disputing
disqualification
disqualified
disqualifies
disqualify
disqualifying
disquiet
disquieting
disregard
disregarded
disregarding
disregards
disrespectful
disrupt
disrupted
disrupting
disruption
disruptions
disruptive
disrupts
dissatisfaction
dissatisfactions
dissatisfactory
dissatisfied
dissect
dissects
dissemble
disseminate
disseminated
disseminates
disseminating
dissemination
dissension
dissensions
dissent
dissented
dissenter
dissenters
dissenting
dissents
dissertation
dissertations
disservice
dissident
dissidents
dissimilar
dissimilarities
dissimilarity
dissipate
dissipated
dissipates
dissipating
dissipation
dissociate
dissociated
dissociates
dissociating
dissociation
dissolution
dissolutions
dissolve
dissolved
dissolves
dissolving
dissonant
dissuade
distaff
distal
distally
distance
distances
distant
distantly
distaste
distasteful
distastefully
distastes
distemper
distempered
distempers
distill
distillation
distilled
distiller
distillers
distillery
distilling
distills
distinct
distinction
distinctions
distinctive
distinctively
distinctiveness
distinctly
distinctness
distinguish
distinguishable
distinguished
distinguishes
distinguishing
distort
distorted
distorting
distortion
distortions
distorts
distract
distracted
distracting
distraction
distractions
distracts
distraught
distress
distressed
distresses
distressing
distribute
distributed
distributes
distributing
distribution
distributional
distributions
distributive
distributivity
distributor
distributors
district
districts
distrust
distrusted
disturb
disturbance
disturbances
disturbed
disturber
disturbing
disturbingly
disturbs
disuse
ditch
ditches
dither
ditto
ditty
diurnal
divan
divans
dive
dived
diver
diverge
diverged
divergence
divergences
divergent
diverges
diverging
divers
diverse
diversely
diversification
diversified
diversifies
diversify
diversifying
diversion
diversionary
diversions
diversities
diversity
divert
diverted
diverting
diverts
dives
divest
divested
divesting
divestiture
divests
divide
divided
dividend
dividends
divider
dividers
divides
dividing
divine
divinely
diviner
diving
divining
divinities
divinity
divisibility
divisible
division
divisional
divisions
divisive
divisor
divisors
divorce
divorced
divorcee
divulge
divulged
divulges
divulging
dizziness
dizzy
do
docile
dock
docked
docket
docks
dockside
dockyard
doctor
doctoral
doctorate
doctorates
doctored
doctors
doctrinaire
doctrinal
doctrine
doctrines
document
documentaries
documentary
documentation
documentations
documented
documenter
documenters
documenting
documents
dodecahedra
dodecahedral
dodecahedron
dodge
dodged
dodger
dodgers
dodging
doe
doer
doers
does
dog
dogged
doggedly
doggedness
dogging
doghouse
dogma
dogmas
dogmatic
dogmatism
dogs
doing
doings
doldrum
dole
doled
doleful
dolefully
doles
doll
dollar
dollars
dollies
dolls
dolly
dolphin
dolphins
domain
domains
dome
domed
domes
domestic
domestically
domesticate
domesticated
domesticates
domesticating
domestication
domicile
dominance
dominant
dominantly
dominate
dominated
dominates
dominating
domination
domineer
domineering
dominion
domino
don
donate
donated
donates
donating
donation
done
donkey
donkeys
donnybrook
donor
dons
doodle
doom
doomed
dooming
dooms
doomsday
door
doorbell
doorkeeper
doorman
doormen
doors
doorstep
doorsteps
doorway
doorways
dope
doped
doper
dopers
dopes
doping
dormant
dormitories
dormitory
dosage
dose
dosed
doses
dossier
dossiers
dot
dote
doted
dotes
doting
dotingly
dots
dotted
dotting
double
doubled
doubleheader
doubler
doublers
doubles
doublet
doubleton
doublets
doubling
doubloon
doubly
doubt
doubtable
doubted
doubter
doubters
doubtful
doubtfully
doubting
doubtless
doubtlessly
doubts
dough
doughnut
doughnuts
dove
dover
doves
dovetail
dowager
dowel
down
downcast
downed
downers
downfall
downfallen
downgrade
downhill
downlink
downlinks
download
downloaded
downloading
downloads
downplay
downplayed
downplaying
downplays
downpour
downright
downside
downstairs
downstream
downtown
downtowns
downtrodden
downturn
downward
downwards
downy
dowry
doze
dozed
dozen
dozens
dozenth
dozes
dozing
drab
draft
drafted
draftee
drafter
drafters
drafting
drafts
draftsman
draftsmen
drafty
drag
dragged
dragging
dragnet
dragon
dragonfly
dragonhead
dragons
dragoon
dragooned
dragoons
drags
drain
drainage
drained
drainer
draining
drains
drake
dram
drama
dramas
dramatic
dramatically
dramatics
dramatist
dramatists
drank
drape
draped
draper
draperies
drapers
drapery
drapes
drastic
drastically
draught
draughts
draw
drawback
drawbacks
drawbridge
drawbridges
drawer
drawers
drawing
drawings
drawl
drawled
drawling
drawls
drawn
drawnly
drawnness
draws
dread
dreaded
dreadful
dreadfully
dreading
dreadnought
dreads
dream
dreamboat
dreamed
dreamer
dreamers
dreamily
dreaming
dreamlike
dreams
dreamt
dreamy
dreariness
dreary
dredge
dregs
drench
drenched
drenches
drenching
dress
dressed
dresser
dressers
dresses
dressing
dressings
dressmaker
dressmakers
drew
dried
drier
driers
dries
driest
drift
drifted
drifter
drifters
drifting
drifts
drill
drilled
driller
drilling
drills
drily
drink
drinkable
drinker
drinkers
drinking
drinks
drip
dripping
drippy
drips
drive
driven
driver
drivers
drives
driveway
driveways
driving
drizzle
drizzly
droll
dromedary
drone
drones
drool
droop
drooped
drooping
droops
droopy
drop
droplet
dropout
dropped
dropper
droppers
dropping
droppings
drops
drosophila
drought
droughts
drove
drover
drovers
droves
drown
drowned
drowning
drownings
drowns
drowsiness
drowsy
drubbing
drudge
drudgery
drug
druggist
druggists
drugs
drugstore
drum
drumhead
drummed
drummer
drummers
drumming
drums
drunk
drunkard
drunkards
drunken
drunkenness
drunker
drunkly
drunks
dry
drying
dryly
dual
dualism
dualities
duality
dub
dubbed
dubious
dubiously
dubiousness
dubs
duchess
duchesses
duchy
duck
ducked
ducking
duckling
ducks
duct
ducts
dud
due
duel
dueling
duels
dues
duet
dug
duke
dukes
dull
dulled
duller
dullest
dulling
dullness
dulls
dully
duly
dumb
dumbbell
dumbbells
dumber
dumbest
dumbly
dumbness
dummies
dummy
dump
dumped
dumper
dumping
dumps
dunce
dunces
dune
dunes
dung
dungeon
dungeons
dunk
dupe
duplex
duplicable
duplicate
duplicated
duplicates
duplicating
duplication
duplications
duplicator
duplicators
duplicity
durabilities
durability
durable
durably
duration
durations
duress
during
dusk
duskiness
dusky
dust
dustbin
dusted
duster
dusters
dustier
dustiest
dusting
dusts
dusty
dutchess
duties
dutiful
dutifully
dutifulness
duty
dwarf
dwarfed
dwarfs
dwarves
dwell
dwelled
dweller
dwellers
dwelling
dwellings
dwells
dwelt
dwindle
dwindled
dwindling
dyad
dyadic
dye
dyed
dyeing
dyer
dyers
dyes
dying
dynamic
dynamically
dynamics
dynamism
dynamite
dynamited
dynamites
dynamiting
dynamo
dynastic
dynasties
dynasty
dyne
dysentery
dyspeptic
dystrophy
each
eager
eagerly
eagerness
eagle
eagles
ear
eardrum
eared
earl
earlier
earliest
earliness
earls
early
earmark
earmarked
earmarking
earmarkings
earmarks
earn
earned
earner
earners
earnest
earnestly
earnestness
earning
earnings
earns
earphone
earring
earrings
ears
earsplitting
earth
earthen
earthenware
earthliness
earthling
earthly
earthmover
earthquake
earthquakes
earths
earthworm
earthworms
earthy
ease
eased
easel
easement
easements
eases
easier
easiest
easily
easiness
easing
east
eastbound
easter
eastern
easterner
easterners
easternmost
eastward
eastwards
easy
easygoing
eat
eaten
eater
eaters
eating
eatings
eats
eaves
eavesdrop
eavesdropped
eavesdropper
eavesdroppers
eavesdropping
eavesdrops
ebb
ebbing
ebbs
ebony
eccentric
eccentricities
eccentricity
eccentrics
ecclesiastical
echelon
echo
echoed
echoes
echoing
eclectic
eclipse
eclipsed
eclipses
eclipsing
ecliptic
ecology
econometric
economic
economical
economically
economics
economies
economist
economists
economize
economized
economizer
economizers
economizes
economizing
economy
ecosystem
ecstasy
ecstatic
eddies
eddy
edge
edged
edges
edging
edible
edict
edicts
edifice
edifices
edit
edited
editing
edition
editions
editor
editorial
editorially
editorials
editors
edits
educable
educate
educated
educates
educating
education
educational
educationally
educations
educator
educators
eel
eelgrass
eels
eerie
eerily
effect
effected
effecting
effective
effectively
effectiveness
effector
effectors
effects
effectually
effectuate
effeminate
efficacy
efficiencies
efficiency
efficient
efficiently
effigy
effort
effortless
effortlessly
effortlessness
efforts
egalitarian
egg
egged
egghead
egging
eggplant
eggs
eggshell
ego
egocentric
egos
egotism
egotist
eigenfunction
eigenstate
eigenvalue
eigenvalues
eigenvector
eight
eighteen
eighteens
eighteenth
eightfold
eighth
eighthes
eighties
eightieth
eights
eighty
either
ejaculate
ejaculated
ejaculates
ejaculating
ejaculation
ejaculations
eject
ejected
ejecting
ejects
eke
eked
ekes
elaborate
elaborated
elaborately
elaborateness
elaborates
elaborating
elaboration
elaborations
elaborators
elapse
elapsed
elapses
elapsing
elastic
elastically
elasticity
elbow
elbowing
elbows
elder
elderly
elders
eldest
elect
elected
electing
election
elections
elective
electives
elector
electoral
electorate
electors
electric
electrical
electrically
electricalness
electrician
electricity
electrification
electrify
electrifying
electro
electrocardiogram
electrocardiograph
electrocute
electrocuted
electrocutes
electrocuting
electrocution
electrocutions
electrode
electrodes
electroencephalogram
electroencephalograph
electroencephalography
electrolysis
electrolyte
electrolytes
electrolytic
electromagnetic
electromechanical
electron
electronic
electronically
electronics
electrons
electrophoresis
electrophorus
elects
elegance
elegant
elegantly
elegy
element
elemental
elementals
elementary
elements
elephant
elephants
elevate
elevated
elevates
elevation
elevator
elevators
eleven
elevens
eleventh
elf
elicit
elicited
eliciting
elicits
elide
eligibility
eligible
eliminate
eliminated
eliminates
eliminating
elimination
eliminations
eliminator
eliminators
elision
elite
elitist
elk
elks
ellipse
ellipses
ellipsis
ellipsoid
ellipsoidal
ellipsoids
elliptic
elliptical
elliptically
elm
elms
elope
eloquence
eloquent
eloquently
else
elsewhere
elucidate
elucidated
elucidates
elucidating
elucidation
elude
eluded
eludes
eluding
elusive
elusively
elusiveness
elves
em
emaciate
emaciated
emacs
emanate
emanating
emancipate
emancipation
emasculate
embalm
embargo
embargoes
embark
embarked
embarks
embarrass
embarrassed
embarrasses
embarrassing
embarrassment
embassies
embassy
embed
embedded
embedding
embeds
embellish
embellished
embellishes
embellishing
embellishment
embellishments
ember
embezzle
emblem
embodied
embodies
embodiment
embodiments
embody
embodying
embolden
embrace
embraced
embraces
embracing
embroider
embroidered
embroideries
embroiders
embroidery
embroil
embryo
embryology
embryos
emerald
emeralds
emerge
emerged
emergence
emergencies
emergency
emergent
emerges
emerging
emeritus
emigrant
emigrants
emigrate
emigrated
emigrates
emigrating
emigration
eminence
eminent
eminently
emissary
emission
emit
emits
emitted
emitter
emitting
emotion
emotional
emotionally
emotions
empathy
emperor
emperors
emphases
emphasis
emphasize
emphasized
emphasizes
emphasizing
emphatic
emphatically
empire
empires
empirical
empirically
empiricist
empiricists
employ
employable
employed
employee
employees
employer
employers
employing
employment
employments
employs
emporium
empower
empowered
empowering
empowers
empress
emptied
emptier
empties
emptiest
emptily
emptiness
empty
emptying
emulate
emulated
emulates
emulating
emulation
emulations
emulator
emulators
en
enable
enabled
enabler
enablers
enables
enabling
enact
enacted
enacting
enactment
enacts
enamel
enameled
enameling
enamels
encamp
encamped
encamping
encamps
encapsulate
encapsulated
encapsulates
encapsulating
encapsulation
encased
enchant
enchanted
enchanter
enchanting
enchantment
enchantress
enchants
encipher
enciphered
enciphering
enciphers
encircle
encircled
encircles
enclose
enclosed
encloses
enclosing
enclosure
enclosures
encode
encoded
encoder
encoders
encodes
encoding
encodings
encompass
encompassed
encompasses
encompassing
encore
encounter
encountered
encountering
encounters
encourage
encouraged
encouragement
encouragements
encourages
encouraging
encouragingly
encroach
encrust
encrypt
encrypted
encrypting
encryption
encryptions
encrypts
encumber
encumbered
encumbering
encumbers
encyclopedia
encyclopedias
encyclopedic
end
endanger
endangered
endangering
endangers
endear
endeared
endearing
endears
endeavor
endeavored
endeavoring
endeavors
ended
endemic
ender
enders
endgame
ending
endings
endless
endlessly
endlessness
endorse
endorsed
endorsement
endorses
endorsing
endow
endowed
endowing
endowment
endowments
endows
endpoint
ends
endurable
endurably
endurance
endure
endured
endures
enduring
enduringly
enema
enemas
enemies
enemy
energetic
energies
energize
energy
enervate
enfeeble
enforce
enforceable
enforced
enforcement
enforcer
enforcers
enforces
enforcing
enfranchise
engage
engaged
engagement
engagements
engages
engaging
engagingly
engender
engendered
engendering
engenders
engine
engineer
engineered
engineering
engineers
engines
engrave
engraved
engraver
engraves
engraving
engravings
engross
engrossed
engrossing
engulf
enhance
enhanced
enhancement
enhancements
enhances
enhancing
enigma
enigmatic
enjoin
enjoined
enjoining
enjoins
enjoy
enjoyable
enjoyably
enjoyed
enjoying
enjoyment
enjoys
enlarge
enlarged
enlargement
enlargements
enlarger
enlargers
enlarges
enlarging
enlighten
enlightened
enlightening
enlightenment
enlist
enlisted
enlistment
enlists
enliven
enlivened
enlivening
enlivens
enmities
enmity
ennoble
ennobled
ennobles
ennobling
ennui
enormities
enormity
enormous
enormously
enough
enqueue
enqueued
enqueues
enquire
enquired
enquirer
enquires
enquiry
enrage
enraged
enrages
enraging
enrapture
enrich
enriched
enriches
enriching
enroll
enrolled
enrolling
enrollment
enrollments
enrolls
ensemble
ensembles
ensign
ensigns
enslave
enslaved
enslaves
enslaving
ensnare
ensnared
ensnares
ensnaring
ensue
ensued
ensues
ensuing
ensure
ensured
ensurer
ensurers
ensures
ensuring
entail
entailed
entailing
entails
entangle
enter
entered
entering
enterprise
enterprises
enterprising
enters
entertain
entertained
entertainer
entertainers
entertaining
entertainingly
entertainment
entertainments
entertains
enthusiasm
enthusiasms
enthusiast
enthusiastic
enthusiastically
enthusiasts
entice
enticed
enticer
enticers
entices
enticing
entire
entirely
entireties
entirety
entities
entitle
entitled
entitles
entitling
entity
entomb
entrance
entranced
entrances
entrap
entreat
entreated
entreaty
entree
entrench
entrenched
entrenches
entrenching
entrepreneur
entrepreneurial
entrepreneurs
entries
entropy
entrust
entrusted
entrusting
entrusts
entry
enumerable
enumerate
enumerated
enumerates
enumerating
enumeration
enumerative
enumerator
enumerators
enunciation
envelop
envelope
enveloped
enveloper
envelopes
enveloping
envelops
envied
envies
envious
enviously
enviousness
environ
environing
environment
environmental
environments
environs
envisage
envisaged
envisages
envision
envisioned
envisioning
envisions
envoy
envoys
envy
enzyme
epaulet
epaulets
ephemeral
epic
epicenter
epics
epidemic
epidemics
epidermis
epigram
epileptic
epilogue
episcopal
episode
episodes
epistemological
epistemology
epistle
epistles
epitaph
epitaphs
epitaxial
epitaxially
epithet
epithets
epitomize
epitomized
epitomizes
epitomizing
epoch
epochs
epsilon
equal
equaled
equaling
equalities
equality
equalization
equalize
equalized
equalizer
equalizers
equalizes
equalizing
equally
equals
equate
equated
equates
equating
equation
equations
equator
equatorial
equators
equestrian
equidistant
equilateral
equilibrate
equilibria
equilibrium
equilibriums
equinox
equip
equipment
equipoise
equipped
equipping
equips
equitable
equitably
equity
equivalence
equivalences
equivalent
equivalently
equivalents
equivocal
equivocally
era
eradicate
eradicated
eradicates
eradicating
eradication
eras
erasable
erase
erased
eraser
erasers
erases
erasing
erasure
ere
erect
erected
erecting
erection
erections
erector
erectors
erects
erg
ergo
ergodic
ermine
ermines
erode
erosion
erotic
erotica
err
errand
errant
errata
erratic
erratum
erred
erring
erringly
erroneous
erroneously
erroneousness
error
errors
errs
ersatz
erudite
erupt
eruption
escalate
escalated
escalates
escalating
escalation
escapable
escapade
escapades
escape
escaped
escapee
escapees
escapes
escaping
eschew
eschewed
eschewing
eschews
escort
escorted
escorting
escorts
escrow
esoteric
especial
especially
espionage
espouse
espoused
espouses
espousing
esprit
espy
esquire
esquires
essay
essayed
essays
essence
essences
essential
essentially
essentials
establish
established
establishes
establishing
establishment
establishments
estate
estates
esteem
esteemed
esteeming
esteems
esthetics
estimate
estimated
estimates
estimating
estimation
estimations
et
etch
etching
eternal
eternally
eternities
eternity
ether
ethereal
ethereally
ethers
ethic
ethical
ethically
ethics
ethnic
etiquette
etymology
eucalyptus
eunuch
eunuchs
euphemism
euphemisms
euphoria
euphoric
eureka
euthanasia
evacuate
evacuated
evacuation
evade
evaded
evades
evading
evaluate
evaluated
evaluates
evaluating
evaluation
evaluations
evaluative
evaluator
evaluators
evaporate
evaporated
evaporating
evaporation
evaporative
evasion
evasive
even
evened
evenhanded
evenhandedly
evenhandedness
evening
evenings
evenly
evenness
evens
event
eventful
eventfully
events
eventual
eventualities
eventuality
eventually
ever
evergreen
everlasting
everlastingly
evermore
every
everybody
everyday
everyone
everything
everywhere
evict
evicted
evicting
eviction
evictions
evicts
evidence
evidenced
evidences
evidencing
evident
evidently
evil
eviller
evilly
evils
evince
evinced
evinces
evoke
evoked
evokes
evoking
evolute
evolutes
evolution
evolutionary
evolutions
evolve
evolved
evolves
evolving
ewe
ewes
ex
exacerbate
exacerbated
exacerbates
exacerbating
exacerbation
exacerbations
exact
exacted
exacting
exactingly
exaction
exactions
exactitude
exactly
exactness
exacts
exaggerate
exaggerated
exaggerates
exaggerating
exaggeration
exaggerations
exalt
exaltation
exalted
exalting
exalts
exam
examination
examinations
examine
examined
examiner
examiners
examines
examining
example
examples
exams
exasperate
exasperated
exasperates
exasperating
exasperation
excavate
excavated
excavates
excavating
excavation
excavations
exceed
exceeded
exceeding
exceedingly
exceeds
excel
excelled
excellence
excellences
excellency
excellent
excellently
excelling
excels
except
excepted
excepting
exception
exceptionable
exceptional
exceptionally
exceptions
excepts
excerpt
excerpted
excerpts
excess
excesses
excessive
excessively
exchange
exchangeable
exchanged
exchanges
exchanging
exchequer
exchequers
excise
excised
excises
excising
excision
excitable
excitation
excitations
excite
excited
excitedly
excitement
excites
exciting
excitingly
exciton
exclaim
exclaimed
exclaimer
exclaimers
exclaiming
exclaims
exclamation
exclamations
exclamatory
exclude
excluded
excludes
excluding
exclusion
exclusionary
exclusions
exclusive
exclusively
exclusiveness
exclusivity
excommunicate
excommunicated
excommunicates
excommunicating
excommunication
excrete
excreted
excretes
excreting
excretion
excretions
excretory
excruciate
excursion
excursions
excusable
excusably
excuse
excused
excuses
excusing
exec
executable
execute
executed
executes
executing
execution
executional
executioner
executions
executive
executives
executor
executors
exemplar
exemplary
exemplification
exemplified
exemplifier
exemplifiers
exemplifies
exemplify
exemplifying
exempt
exempted
exempting
exemption
exempts
exercise
exercised
exerciser
exercisers
exercises
exercising
exert
exerted
exerting
exertion
exertions
exerts
exhale
exhaled
exhales
exhaling
exhaust
exhausted
exhaustedly
exhausting
exhaustion
exhaustive
exhaustively
exhausts
exhibit
exhibited
exhibiting
exhibition
exhibitions
exhibitor
exhibitors
exhibits
exhilarate
exhort
exhortation
exhortations
exhume
exigency
exile
exiled
exiles
exiling
exist
existed
existence
existent
existential
existentialism
existentialist
existentialists
existentially
existing
exists
exit
exited
exiting
exits
exodus
exorbitant
exorbitantly
exorcism
exorcist
exoskeleton
exotic
expand
expandable
expanded
expander
expanders
expanding
expands
expanse
expanses
expansible
expansion
expansionism
expansions
expansive
expect
expectancy
expectant
expectantly
expectation
expectations
expected
expectedly
expecting
expectingly
expects
expediency
expedient
expediently
expedite
expedited
expedites
expediting
expedition
expeditions
expeditious
expeditiously
expel
expelled
expelling
expels
expend
expendable
expended
expending
expenditure
expenditures
expends
expense
expenses
expensive
expensively
experience
experienced
experiences
experiencing
experiment
experimental
experimentally
experimentation
experimentations
experimented
experimenter
experimenters
experimenting
experiments
expert
expertise
expertly
expertness
experts
expiration
expirations
expire
expired
expires
expiring
explain
explainable
explained
explainer
explainers
explaining
explains
explanation
explanations
explanatory
expletive
explicit
explicitly
explicitness
explode
exploded
explodes
exploding
exploit
exploitable
exploitation
exploitations
exploited
exploiter
exploiters
exploiting
exploits
exploration
explorations
exploratory
explore
explored
explorer
explorers
explores
exploring
explosion
explosions
explosive
explosively
explosives
exponent
exponential
exponentially
exponentials
exponentiate
exponentiated
exponentiates
exponentiating
exponentiation
exponentiations
exponents
export
exportation
exported
exporter
exporters
exporting
exports
expose
exposed
exposer
exposers
exposes
exposing
exposition
expositions
expository
exposure
exposures
expound
expounded
expounder
expounding
expounds
express
expressed
expresses
expressibility
expressible
expressibly
expressing
expression
expressions
expressive
expressively
expressiveness
expressly
expulsion
expunge
expunged
expunges
expunging
expurgate
exquisite
exquisitely
exquisiteness
extant
extemporaneous
extend
extendable
extended
extending
extends
extensibility
extensible
extension
extensions
extensive
extensively
extent
extents
extenuate
extenuated
extenuating
extenuation
exterior
exteriors
exterminate
exterminated
exterminates
exterminating
extermination
external
externally
extinct
extinction
extinguish
extinguished
extinguisher
extinguishes
extinguishing
extirpate
extol
extort
extorted
extortion
extra
extract
extracted
extracting
extraction
extractions
extractor
extractors
extracts
extracurricular
extramarital
extraneous
extraneously
extraneousness
extraordinarily
extraordinariness
extraordinary
extrapolate
extrapolated
extrapolates
extrapolating
extrapolation
extrapolations
extras
extraterrestrial
extravagance
extravagant
extravagantly
extravaganza
extremal
extreme
extremely
extremes
extremist
extremists
extremities
extremity
extricate
extrinsic
extrovert
exuberance
exult
exultation
eye
eyeball
eyebrow
eyebrows
eyed
eyeful
eyeglass
eyeglasses
eyeing
eyelash
eyelid
eyelids
eyepiece
eyepieces
eyer
eyers
eyes
eyesight
eyewitness
eyewitnesses
eying
fable
fabled
fables
fabric
fabricate
fabricated
fabricates
fabricating
fabrication
fabrics
fabulous
fabulously
facade
facaded
facades
face
faced
faces
facet
faceted
facets
facial
facile
facilely
facilitate
facilitated
facilitates
facilitating
facilities
facility
facing
facings
facsimile
facsimiles
fact
faction
factions
factious
facto
factor
factored
factorial
factories
factoring
factorization
factorizations
factors
factory
facts
factual
factually
faculties
faculty
fade
faded
fadeout
fader
faders
fades
fading
fag
fags
fail
failed
failing
failings
fails
failsoft
failure
failures
fain
faint
fainted
fainter
faintest
fainting
faintly
faintness
faints
fair
fairer
fairest
fairies
fairing
fairly
fairness
fairs
fairy
fairyland
faith
faithful
faithfully
faithfulness
faithless
faithlessly
faithlessness
faiths
fake
faked
faker
fakes
faking
falcon
falconer
falcons
fall
fallacies
fallacious
fallacy
fallen
fallibility
fallible
falling
fallout
fallow
falls
false
falsehood
falsehoods
falsely
falseness
falsification
falsified
falsifies
falsify
falsifying
falsity
falter
faltered
falters
fame
famed
fames
familial
familiar
familiarities
familiarity
familiarization
familiarize
familiarized
familiarizes
familiarizing
familiarly
familiarness
families
familism
family
famine
famines
famish
famous
famously
fan
fanatic
fanaticism
fanatics
fancied
fancier
fanciers
fancies
fanciest
fanciful
fancifully
fancily
fanciness
fancy
fancying
fanfare
fanfold
fang
fangled
fangs
fanned
fanning
fanout
fans
fantasies
fantasize
fantastic
fantasy
far
farad
faraway
farce
farces
fare
fared
fares
farewell
farewells
farfetched
farina
faring
farm
farmed
farmer
farmers
farmhouse
farmhouses
farming
farmland
farms
farmyard
farmyards
farsighted
farther
farthest
farthing
fascicle
fascinate
fascinated
fascinates
fascinating
fascination
fascism
fascist
fashion
fashionable
fashionably
fashioned
fashioning
fashions
fast
fasted
fasten
fastened
fastener
fasteners
fastening
fastenings
fastens
faster
fastest
fastidious
fasting
fastness
fasts
fat
fatal
fatalities
fatality
fatally
fatals
fate
fated
fateful
fates
father
fathered
fatherland
fatherly
fathers
fathom
fathomed
fathoming
fathoms
fatigue
fatigued
fatigues
fatiguing
fatness
fats
fatten
fattened
fattener
fatteners
fattening
fattens
fatter
fattest
fatty
faucet
fault
faulted
faulting
faultless
faultlessly
faults
faulty
faun
fauna
favor
favorable
favorably
favored
favorer
favoring
favorite
favorites
favoritism
favors
fawn
fawned
fawning
fawns
faze
fear
feared
fearful
fearfully
fearing
fearless
fearlessly
fearlessness
fears
fearsome
feasibility
feasible
feast
feasted
feasting
feasts
feat
feather
featherbed
featherbedding
feathered
featherer
featherers
feathering
feathers
featherweight
feathery
feats
feature
featured
features
featuring
fecund
fed
federal
federalist
federally
federals
federation
fee
feeble
feebleness
feebler
feeblest
feebly
feed
feedback
feeder
feeders
feeding
feedings
feeds
feel
feeler
feelers
feeling
feelingly
feelings
feels
fees
feet
feign
feigned
feigning
felicities
felicity
feline
fell
fellatio
felled
felling
fellow
fellows
fellowship
fellowships
felon
felonious
felony
felt
felts
female
females
feminine
femininity
feminism
feminist
femur
femurs
fen
fence
fenced
fencer
fencers
fences
fencing
fend
ferment
fermentation
fermentations
fermented
fermenting
ferments
fern
ferns
ferocious
ferociously
ferociousness
ferocity
ferret
ferried
ferries
ferrite
ferry
fertile
fertilely
fertility
fertilization
fertilize
fertilized
fertilizer
fertilizers
fertilizes
fertilizing
fervent
fervently
fervor
fervors
festival
festivals
festive
festively
festivities
festivity
fetal
fetch
fetched
fetches
fetching
fetchingly
fetid
fetish
fetter
fettered
fetters
fettle
fetus
feud
feudal
feudalism
feuds
fever
fevered
feverish
feverishly
fevers
few
fewer
fewest
fewness
fiance
fiancee
fiasco
fiat
fib
fibbing
fiber
fibers
fibrosities
fibrosity
fibrous
fibrously
fickle
fickleness
fiction
fictional
fictionally
fictions
fictitious
fictitiously
fiddle
fiddled
fiddler
fiddles
fiddlestick
fiddlesticks
fiddling
fidelity
fidget
fiducial
fief
fiefdom
field
fielded
fielder
fielders
fielding
fieldwork
fiend
fiendish
fierce
fiercely
fierceness
fiercer
fiercest
fiery
fife
fifteen
fifteens
fifteenth
fifth
fifties
fiftieth
fifty
fig
fight
fighter
fighters
fighting
fights
figs
figurative
figuratively
figure
figured
figures
figuring
figurings
filament
filaments
file
filed
filename
filenames
filer
files
filial
filibuster
filing
filings
fill
fillable
filled
filler
fillers
filling
fillings
fills
filly
film
filmed
filming
films
filter
filtered
filtering
filters
filth
filthier
filthiest
filthiness
filthy
fin
final
finality
finalization
finalize
finalized
finalizes
finalizing
finally
finals
finance
financed
finances
financial
financially
financier
financiers
financing
find
finder
finders
finding
findings
finds
fine
fined
finely
fineness
finer
fines
finesse
finessed
finessing
finest
finger
fingered
fingering
fingerings
fingernail
fingerprint
fingerprints
fingers
fingertip
finicky
fining
finish
finished
finisher
finishers
finishes
finishing
finite
finitely
finiteness
fink
finny
fins
fir
fire
firearm
firearms
fireboat
firebreak
firebug
firecracker
fired
fireflies
firefly
firehouse
firelight
fireman
firemen
fireplace
fireplaces
firepower
fireproof
firer
firers
fires
fireside
firewall
firewood
fireworks
firing
firings
firm
firmament
firmed
firmer
firmest
firming
firmly
firmness
firms
firmware
first
firsthand
firstly
firsts
fiscal
fiscally
fish
fished
fisher
fisherman
fishermen
fishers
fishery
fishes
fishing
fishmonger
fishpond
fishy
fission
fissure
fissured
fist
fisted
fisticuff
fists
fit
fitful
fitfully
fitly
fitness
fits
fitted
fitter
fitters
fitting
fittingly
fittings
five
fivefold
fives
fix
fixate
fixated
fixates
fixating
fixation
fixations
fixed
fixedly
fixedness
fixer
fixers
fixes
fixing
fixings
fixture
fixtures
fizzle
fizzled
flabbergast
flabbergasted
flack
flag
flagellate
flagged
flagging
flagpole
flagrant
flagrantly
flags
flail
flair
flak
flake
flaked
flakes
flaking
flaky
flam
flamboyant
flame
flamed
flamer
flamers
flames
flaming
flammable
flank
flanked
flanker
flanking
flanks
flannel
flannels
flap
flaps
flare
flared
flares
flaring
flash
flashback
flashed
flasher
flashers
flashes
flashing
flashlight
flashlights
flashy
flask
flat
flatbed
flatly
flatness
flats
flatten
flattened
flattening
flatter
flattered
flatterer
flattering
flattery
flattest
flatulent
flatus
flatworm
flaunt
flaunted
flaunting
flaunts
flavor
flavored
flavoring
flavorings
flavors
flaw
flawed
flawless
flawlessly
flaws
flax
flaxen
flea
fleas
fled
fledged
fledgling
fledglings
flee
fleece
fleeces
fleecy
fleeing
flees
fleet
fleetest
fleeting
fleetly
fleetness
fleets
flesh
fleshed
fleshes
fleshing
fleshly
fleshy
flew
flex
flexibilities
flexibility
flexible
flexibly
flick
flicked
flicker
flickering
flicking
flicks
flier
fliers
flies
flight
flights
flimsy
flinch
flinched
flinches
flinching
fling
flings
flint
flinty
flip
flipflop
flipped
flips
flirt
flirtation
flirtatious
flirted
flirting
flirts
flit
flitting
float
floated
floater
floating
floats
flock
flocked
flocking
flocks
flog
flogging
flood
flooded
flooding
floodlight
floodlit
floods
floor
floored
flooring
floorings
floors
flop
floppies
floppily
flopping
floppy
flops
flora
floral
florid
florin
florist
floss
flossed
flosses
flossing
flotation
flotilla
flounder
floundered
floundering
flounders
flour
floured
flourish
flourished
flourishes
flourishing
flow
flowchart
flowcharting
flowcharts
flowed
flower
flowered
floweriness
flowering
flowerpot
flowers
flowery
flowing
flown
flows
flu
fluctuate
fluctuates
fluctuating
fluctuation
fluctuations
flue
fluency
fluent
fluently
fluff
fluffier
fluffiest
fluffy
fluid
fluidity
fluidly
fluids
fluke
flung
flunked
fluoresce
fluorescent
flurried
flurry
flush
flushed
flushes
flushing
flute
fluted
fluting
flutter
fluttered
fluttering
flutters
flux
fly
flyable
flyer
flyers
flying
foal
foam
foamed
foaming
foams
foamy
fob
fobbing
focal
focally
foci
focus
focused
focuses
focusing
focussed
fodder
foe
foes
fog
fogged
foggier
foggiest
foggily
fogging
foggy
fogs
fogy
foible
foil
foiled
foiling
foils
foist
fold
folded
folder
folders
folding
foldout
folds
foliage
folk
folklore
folks
folksong
folksy
follies
follow
followed
follower
followers
following
followings
follows
folly
fond
fonder
fondle
fondled
fondles
fondling
fondly
fondness
font
fonts
food
foods
foodstuff
foodstuffs
fool
fooled
foolhardy
fooling
foolish
foolishly
foolishness
foolproof
fools
foot
footage
football
footballs
footbridge
footed
footer
footers
footfall
foothill
foothold
footing
footman
footnote
footnotes
footpath
footprint
footprints
footstep
footsteps
for
forage
foraged
forages
foraging
foray
forays
forbade
forbear
forbearance
forbears
forbid
forbidden
forbidding
forbids
force
forced
forceful
forcefully
forcefulness
forcer
forces
forcible
forcibly
forcing
ford
fords
fore
forearm
forearms
foreboding
forecast
forecasted
forecaster
forecasters
forecasting
forecastle
forecasts
forefather
forefathers
forefinger
forefingers
forego
foregoes
foregoing
foregone
foreground
forehead
foreheads
foreign
foreigner
foreigners
foreigns
foreman
foremost
forenoon
forensic
forerunners
foresee
foreseeable
foreseen
foresees
foresight
foresighted
forest
forestall
forestalled
forestalling
forestallment
forestalls
forested
forester
foresters
forestry
forests
foretell
foretelling
foretells
foretold
forever
forewarn
forewarned
forewarning
forewarnings
forewarns
forfeit
forfeited
forfeiture
forgave
forge
forged
forger
forgeries
forgery
forges
forget
forgetful
forgetfulness
forgets
forgettable
forgettably
forgetting
forging
forgivable
forgivably
forgive
forgiven
forgiveness
forgives
forgiving
forgivingly
forgot
forgotten
fork
forked
forking
forklift
forks
forlorn
forlornly
form
formal
formalism
formalisms
formalities
formality
formalization
formalizations
formalize
formalized
formalizes
formalizing
formally
formant
formants
format
formation
formations
formative
formatively
formats
formatted
formatter
formatters
formatting
formed
former
formerly
formidable
forming
forms
formula
formulae
formulas
formulate
formulated
formulates
formulating
formulation
formulations
formulator
formulators
fornication
forsake
forsaken
forsakes
forsaking
fort
forte
forthcoming
forthright
forthwith
fortier
forties
fortieth
fortification
fortifications
fortified
fortifies
fortify
fortifying
fortiori
fortitude
fortnight
fortnightly
fortress
fortresses
forts
fortuitous
fortuitously
fortunate
fortunately
fortune
fortunes
forty
forum
forums
forward
forwarded
forwarder
forwarding
forwardness
forwards
fossil
foster
fostered
fostering
fosters
fought
foul
fouled
foulest
fouling
foully
foulmouth
foulness
fouls
found
foundation
foundations
founded
founder
foundered
founders
founding
foundling
foundries
foundry
founds
fount
fountain
fountains
founts
four
fourfold
fours
fourscore
foursome
foursquare
fourteen
fourteens
fourteenth
fourth
fowl
fowler
fowls
fox
foxes
fraction
fractional
fractionally
fractions
fracture
fractured
fractures
fracturing
fragile
fragment
fragmentary
fragmentation
fragmented
fragmenting
fragments
fragrance
fragrances
fragrant
fragrantly
frail
frailest
frailty
frame
framed
framer
frames
framework
frameworks
framing
franc
franchise
franchises
francs
frank
franked
franker
frankest
franking
frankly
frankness
franks
frantic
frantically
fraternal
fraternally
fraternities
fraternity
fraud
frauds
fraudulent
fraught
fray
frayed
fraying
frays
frazzle
freak
freakish
freaks
freckle
freckled
freckles
free
freed
freedom
freedoms
freeing
freeings
freely
freeman
freeness
freer
frees
freest
freestyle
freeway
freewheel
freeze
freezer
freezers
freezes
freezing
freight
freighted
freighter
freighters
freighting
freights
frenetic
frenzied
frenzy
freon
frequencies
frequency
frequent
frequented
frequenter
frequenters
frequenting
frequently
frequents
fresco
frescoes
fresh
freshen
freshened
freshener
fresheners
freshening
freshens
fresher
freshest
freshly
freshman
freshmen
freshness
freshwater
fret
fretful
fretfully
fretfulness
friar
friars
fricative
fricatives
friction
frictionless
frictions
fried
friend
friendless
friendlier
friendliest
friendliness
friendly
friends
friendship
friendships
fries
frieze
friezes
frigate
frigates
fright
frighten
frightened
frightening
frighteningly
frightens
frightful
frightfully
frightfulness
frigid
frill
frills
fringe
fringed
frisk
frisked
frisking
frisks
frisky
fritter
frivolity
frivolous
frivolously
fro
frock
frocks
frog
frogs
frolic
frolics
from
front
frontage
frontal
fronted
frontier
frontiers
frontiersman
frontiersmen
fronting
fronts
frost
frostbite
frostbitten
frosted
frosting
frosts
frosty
froth
frothing
frothy
frown
frowned
frowning
frowns
froze
frozen
frozenly
frugal
frugally
fruit
fruitful
fruitfully
fruitfulness
fruition
fruitless
fruitlessly
fruits
frustrate
frustrated
frustrates
frustrating
frustration
frustrations
fry
fudge
fuel
fueled
fueling
fuels
fugitive
fugitives
fugue
fulcrum
fulfill
fulfilled
fulfilling
fulfillment
fulfillments
fulfills
full
fuller
fullest
fullness
fully
fulminate
fumble
fumbled
fumbling
fume
fumed
fumes
fuming
fun
function
functional
functionalities
functionality
functionally
functionals
functionary
functioned
functioning
functions
functor
functors
fund
fundamental
fundamentally
fundamentals
funded
funder
funders
funding
funds
funeral
funerals
funereal
fungal
fungi
fungible
fungicide
fungus
funk
funnel
funneled
funneling
funnels
funnier
funniest
funnily
funniness
funny
fur
furies
furious
furiouser
furiously
furlong
furlough
furnace
furnaces
furnish
furnished
furnishes
furnishing
furnishings
furniture
furrier
furrow
furrowed
furrows
furry
furs
further
furthered
furthering
furthermore
furthermost
furthers
furthest
furtive
furtively
furtiveness
fury
fuse
fused
fuses
fusing
fusion
fuss
fussing
fussy
futile
futility
future
futures
futuristic
fuzz
fuzzier
fuzziness
fuzzy
gab
gabardine
gabbing
gable
gabled
gabler
gables
gad
gadfly
gadget
gadgetry
gadgets
gag
gagged
gagging
gaging
gags
gaieties
gaiety
gaily
gain
gained
gainer
gainers
gainful
gaining
gains
gait
gaited
gaiter
gaiters
galactic
galaxies
galaxy
gale
gall
gallant
gallantly
gallantry
gallants
galled
galleried
galleries
gallery
galley
galleys
galling
gallon
gallons
gallop
galloped
galloper
galloping
gallops
gallows
galls
gallstone
gambit
gamble
gambled
gambler
gamblers
gambles
gambling
gambol
game
gamed
gamely
gameness
games
gaming
gamma
gander
gang
gangland
gangling
gangplank
gangrene
gangs
gangster
gangsters
gantry
gap
gape
gaped
gapes
gaping
gaps
garage
garaged
garages
garb
garbage
garbages
garbed
garble
garbled
garden
gardened
gardener
gardeners
gardening
gardens
gargantuan
gargle
gargled
gargles
gargling
garland
garlanded
garlic
garment
garments
garner
garnered
garnish
garrison
garrisoned
garter
garters
gas
gaseous
gaseously
gases
gash
gashes
gasket
gaslight
gasoline
gasp
gasped
gasping
gasps
gassed
gasser
gassing
gassings
gassy
gastric
gastrointestinal
gastronome
gastronomy
gate
gated
gateway
gateways
gather
gathered
gatherer
gatherers
gathering
gatherings
gathers
gating
gator
gauche
gaudiness
gaudy
gauge
gauged
gauges
gaunt
gauntness
gauze
gave
gavel
gawk
gawky
gay
gayer
gayest
gayety
gayly
gayness
gaze
gazed
gazelle
gazer
gazers
gazes
gazette
gazing
gear
geared
gearing
gears
gecko
geese
geisha
gel
gelatin
gelatine
gelatinous
geld
gelled
gelling
gels
gem
gems
gender
genders
gene
genealogy
general
generalist
generalists
generalities
generality
generalization
generalizations
generalize
generalized
generalizer
generalizers
generalizes
generalizing
generally
generals
generate
generated
generates
generating
generation
generations
generative
generator
generators
generic
generically
generosities
generosity
generous
generously
generousness
genes
genesis
genetic
genetically
genial
genially
genie
genius
geniuses
genre
genres
gent
genteel
gentile
gentle
gentleman
gentlemanly
gentlemen
gentleness
gentler
gentlest
gentlewoman
gently
gentry
genuine
genuinely
genuineness
genus
geocentric
geodesic
geodesy
geodetic
geographer
geographic
geographical
geographically
geography
geological
geologist
geologists
geology
geometric
geometrical
geometrically
geometrician
geometries
geometry
geophysical
geophysics
geosynchronous
geranium
gerbil
geriatric
germ
germane
germicide
germinal
germinate
germinated
germinates
germinating
germination
germs
gerund
gesture
gestured
gestures
gesturing
get
getaway
gets
getter
getters
getting
geyser
ghastly
ghetto
ghost
ghosted
ghostly
ghosts
giant
giants
gibberish
giddiness
giddy
gift
gifted
gifts
gig
gigabit
gigabits
gigabyte
gigabytes
gigacycle
gigahertz
gigantic
gigavolt
gigawatt
giggle
giggled
giggles
giggling
gild
gilded
gilding
gilds
gill
gills
gilt
gimmick
gimmicks
gin
ginger
gingerbread
gingerly
gingham
ginghams
gins
giraffe
giraffes
gird
girder
girders
girdle
girl
girlfriend
girlie
girlish
girls
girt
girth
gist
give
giveaway
given
giver
givers
gives
giving
glacial
glacier
glaciers
glad
gladden
gladder
gladdest
glade
gladiator
gladly
gladness
glamor
glamorous
glamour
glance
glanced
glances
glancing
gland
glands
glandular
glare
glared
glares
glaring
glaringly
glass
glassed
glasses
glassy
glaucoma
glaze
glazed
glazer
glazes
glazing
gleam
gleamed
gleaming
gleams
glean
gleaned
gleaner
gleaning
gleanings
gleans
glee
gleeful
gleefully
glees
glen
glens
glide
glided
glider
gliders
glides
glimmer
glimmered
glimmering
glimmers
glimpse
glimpsed
glimpses
glint
glinted
glinting
glints
glisten
glistened
glistening
glistens
glitch
glitter
glittered
glittering
glitters
gloat
global
globally
globe
globes
globular
globularity
gloom
gloomily
gloomy
glories
glorification
glorified
glorifies
glorify
glorious
gloriously
glory
glorying
gloss
glossaries
glossary
glossed
glosses
glossing
glossy
glottal
glove
gloved
glover
glovers
gloves
gloving
glow
glowed
glower
glowers
glowing
glowingly
glows
glue
glued
glues
gluing
glut
glutton
gnash
gnat
gnats
gnaw
gnawed
gnawing
gnaws
gnome
gnomon
gnu
go
goad
goaded
goal
goals
goat
goatee
goatees
goats
gobble
gobbled
gobbler
gobblers
gobbles
goblet
goblets
goblin
goblins
god
goddess
goddesses
godfather
godhead
godlike
godly
godmother
godmothers
godparent
gods
godsend
godson
goes
goggles
going
goings
gold
golden
goldenly
goldenness
goldenrod
goldfish
golding
golds
goldsmith
golf
golfer
golfers
golfing
golly
gondola
gone
goner
gong
gongs
good
goodby
goodbye
goodies
goodly
goodness
goods
goodwill
goody
goof
goofed
goofs
goofy
goose
gopher
gore
gorge
gorgeous
gorgeously
gorges
gorging
gorilla
gorillas
gory
gosh
gospel
gospelers
gospels
gossip
gossiped
gossiping
gossips
got
gotten
gouge
gouged
gouges
gouging
gourd
gourmet
gout
govern
governance
governed
governess
governing
government
governmental
governmentally
governments
governor
governors
governs
gown
gowned
gowns
grab
grabbed
grabber
grabbers
grabbing
grabbings
grabs
grace
graced
graceful
gracefully
gracefulness
graces
gracing
gracious
graciously
graciousness
grad
gradation
gradations
grade
graded
grader
graders
grades
gradient
gradients
grading
gradings
gradual
gradually
graduate
graduated
graduates
graduating
graduation
graduations
graft
grafted
grafter
grafting
grafts
graham
grahams
grail
grain
grained
graining
grains
gram
grammar
grammarian
grammars
grammatic
grammatical
grammatically
grams
granaries
granary
grand
grandchild
grandchildren
granddaughter
grander
grandest
grandeur
grandfather
grandfathers
grandiose
grandly
grandma
grandmother
grandmothers
grandnephew
grandness
grandniece
grandpa
grandparent
grands
grandson
grandsons
grandstand
grange
granite
granny
granola
grant
granted
grantee
granter
granting
grantor
grants
granularity
granulate
granulated
granulates
granulating
grape
grapefruit
grapes
grapevine
graph
graphed
graphic
graphical
graphically
graphics
graphing
graphite
graphs
grapple
grappled
grappling
grasp
graspable
grasped
grasping
graspingly
grasps
grass
grassed
grassers
grasses
grassier
grassiest
grassland
grassy
grate
grated
grateful
gratefully
gratefulness
grater
grates
gratification
gratified
gratify
gratifying
grating
gratings
gratis
gratitude
gratuities
gratuitous
gratuitously
gratuitousness
gratuity
grave
gravel
gravelly
gravely
graven
graveness
graver
gravest
gravestone
graveyard
gravitate
gravitation
gravitational
gravity
gravy
gray
grayed
grayer
grayest
graying
grayness
graze
grazed
grazer
grazing
grease
greased
greases
greasy
great
greater
greatest
greatly
greatness
greed
greedily
greediness
greedy
green
greener
greenery
greenest
greengrocer
greenhouse
greenhouses
greening
greenish
greenly
greenness
greens
greenware
greet
greeted
greeter
greeting
greetings
greets
gregarious
grenade
grenades
grew
grey
greyest
greyhound
greying
grid
griddle
gridiron
grids
grief
griefs
grievance
grievances
grieve
grieved
griever
grievers
grieves
grieving
grievingly
grievous
grievously
grill
grilled
grilling
grills
grim
grimace
grime
grimed
grimly
grimness
grin
grind
grinder
grinders
grinding
grindings
grinds
grindstone
grindstones
grinning
grins
grip
gripe
griped
gripes
griping
gripped
gripping
grippingly
grips
grisly
grist
grit
grits
gritty
grizzly
groan
groaned
groaner
groaners
groaning
groans
grocer
groceries
grocers
grocery
groggy
groin
groom
groomed
grooming
grooms
groove
grooved
grooves
grope
groped
gropes
groping
gross
grossed
grosser
grosses
grossest
grossing
grossly
grossness
grotesque
grotesquely
grotesques
grotto
grottos
ground
grounded
grounder
grounders
grounding
grounds
groundwork
group
grouped
grouping
groupings
groups
grouse
grove
grovel
groveled
groveling
grovels
grovers
groves
grow
grower
growers
growing
growl
growled
growling
growls
grown
grownup
grownups
grows
growth
growths
grub
grubby
grubs
grudge
grudges
grudgingly
gruesome
gruff
gruffly
grumble
grumbled
grumbles
grumbling
grunt
grunted
grunting
grunts
guano
guarantee
guaranteed
guaranteeing
guaranteer
guaranteers
guarantees
guaranty
guard
guarded
guardedly
guardhouse
guardian
guardians
guardianship
guarding
guards
gubernatorial
guerrilla
guerrillas
guess
guessed
guesses
guessing
guesswork
guest
guests
guidance
guide
guidebook
guidebooks
guided
guideline
guidelines
guides
guiding
guild
guilder
guilders
guile
guilt
guiltier
guiltiest
guiltily
guiltiness
guiltless
guiltlessly
guilty
guinea
guise
guises
guitar
guitars
gulch
gulches
gulf
gulfs
gull
gulled
gullies
gulling
gulls
gully
gulp
gulped
gulps
gum
gumming
gumption
gums
gun
gunfire
gunman
gunmen
gunned
gunner
gunners
gunnery
gunning
gunny
gunplay
gunpowder
guns
gunshot
gurgle
guru
gush
gushed
gusher
gushes
gushing
gust
gusto
gusts
gusty
gut
guts
gutsy
gutter
guttered
gutters
gutting
guttural
guy
guyed
guyer
guyers
guying
guys
gymnasium
gymnasiums
gymnast
gymnastic
gymnastics
gymnasts
gypsies
gypsy
gyro
gyrocompass
gyroscope
gyroscopes
ha
habeas
habit
habitat
habitation
habitations
habitats
habits
habitual
habitually
habitualness
hack
hacked
hacker
hackers
hacking
hackneyed
hacks
hacksaw
had
haddock
hag
haggard
haggardly
haggle
hail
hailed
hailing
hails
hailstone
hailstorm
hair
haircut
haircuts
hairier
hairiness
hairless
hairpin
hairs
hairy
halcyon
hale
haler
half
halfhearted
halfway
hall
hallmark
hallmarks
hallow
hallowed
halls
hallucinate
hallway
hallways
halogen
halt
halted
halter
halters
halting
haltingly
halts
halve
halved
halvers
halves
halving
ham
hamburger
hamburgers
hamlet
hamlets
hammer
hammered
hammering
hammers
hamming
hammock
hammocks
hamper
hampered
hampers
hams
hamster
hand
handbag
handbags
handbook
handbooks
handcuff
handcuffed
handcuffing
handcuffs
handed
handful
handfuls
handgun
handicap
handicapped
handicaps
handier
handiest
handily
handiness
handing
handiwork
handkerchief
handkerchiefs
handle
handled
handler
handlers
handles
handling
handmaid
handout
hands
handshake
handshakes
handshaking
handsome
handsomely
handsomeness
handsomer
handsomest
handwriting
handwritten
handy
hang
hangar
hangars
hanged
hanger
hangers
hanging
hangman
hangmen
hangout
hangover
hangovers
hangs
hap
haphazard
haphazardly
haphazardness
hapless
haplessly
haplessness
haply
happen
happened
happening
happenings
happens
happier
happiest
happily
happiness
happy
harass
harassed
harasses
harassing
harassment
harbinger
harbor
harbored
harboring
harbors
hard
hardboiled
hardcopy
harden
harder
hardest
hardhat
hardiness
hardly
hardness
hardscrabble
hardship
hardships
hardware
hardwired
hardworking
hardy
hare
harelip
harem
hares
hark
harken
harlot
harlots
harm
harmed
harmful
harmfully
harmfulness
harming
harmless
harmlessly
harmlessness
harmonic
harmonics
harmonies
harmonious
harmoniously
harmoniousness
harmonize
harmony
harms
harness
harnessed
harnessing
harp
harper
harpers
harping
harried
harrier
harrow
harrowed
harrowing
harrows
harry
harsh
harsher
harshly
harshness
hart
harvest
harvested
harvester
harvesting
harvests
has
hash
hashed
hasher
hashes
hashing
hashish
hassle
haste
hasten
hastened
hastening
hastens
hastily
hastiness
hasty
hat
hatch
hatched
hatchet
hatchets
hatching
hate
hated
hateful
hatefully
hatefulness
hater
hates
hating
hatred
hats
haughtily
haughtiness
haughty
haul
hauled
hauler
hauling
hauls
haunch
haunches
haunt
haunted
haunter
haunting
haunts
have
haven
havens
haves
having
havoc
hawk
hawked
hawker
hawkers
hawks
hay
haying
haystack
hazard
hazardous
hazards
haze
hazel
hazes
haziness
hazy
he
head
headache
headaches
headed
header
headers
headgear
heading
headings
headland
headlands
headlight
headline
headlined
headlines
headlining
headlong
headmaster
headphone
headquarters
headroom
heads
headset
headway
heal
healed
healer
healers
healing
heals
health
healthful
healthfully
healthfulness
healthier
healthiest
healthily
healthiness
healthy
heap
heaped
heaping
heaps
hear
heard
hearer
hearers
hearing
hearings
hearken
hears
hearsay
heart
heartbeat
heartbreak
hearten
heartiest
heartily
heartiness
heartless
hearts
hearty
heat
heatable
heated
heatedly
heater
heaters
heath
heathen
heather
heating
heats
heave
heaved
heaven
heavenly
heavens
heaver
heavers
heaves
heavier
heaviest
heavily
heaviness
heaving
heavy
heavyweight
heck
heckle
hectic
hedge
hedged
hedgehog
hedgehogs
hedges
hedonism
hedonist
heed
heeded
heedless
heedlessly
heedlessness
heeds
heel
heeled
heelers
heeling
heels
hefty
hegemony
heifer
height
heighten
heightened
heightening
heightens
heights
heinous
heinously
heir
heiress
heiresses
heirs
held
helical
helicopter
heliocentric
helium
helix
hell
hellfire
hellish
hello
hells
helm
helmet
helmets
helmsman
help
helped
helper
helpers
helpful
helpfully
helpfulness
helping
helpless
helplessly
helplessness
helpmate
helps
hem
hemisphere
hemispheres
hemlock
hemlocks
hemoglobin
hemorrhoid
hemostat
hemostats
hemp
hempen
hems
hen
hence
henceforth
henchman
henchmen
henpeck
hens
hepatitis
her
herald
heralded
heralding
heralds
herb
herbivore
herbivorous
herbs
herd
herded
herder
herding
herds
here
hereabout
hereabouts
hereafter
hereby
hereditary
heredity
herein
hereinafter
hereof
heres
heresy
heretic
heretics
hereto
heretofore
hereunder
herewith
heritage
heritages
hermetic
hermetically
hermit
hermitian
hermits
hero
heroes
heroic
heroically
heroics
heroin
heroine
heroines
heroism
heron
herons
herpes
herring
herrings
hers
herself
hertz
hesitant
hesitantly
hesitate
hesitated
hesitates
hesitating
hesitatingly
hesitation
hesitations
heterogeneity
heterogeneous
heterogeneously
heterogeneousness
heterogenous
heterosexual
heuristic
heuristically
heuristics
hew
hewed
hewer
hews
hex
hexadecimal
hexagon
hexagonal
hexagonally
hexagons
hey
hi
hibernate
hick
hickory
hid
hidden
hide
hideous
hideously
hideousness
hideout
hideouts
hides
hiding
hierarchal
hierarchic
hierarchical
hierarchically
hierarchies
hierarchy
high
higher
highest
highland
highlander
highlands
highlight
highlighted
highlighting
highlights
highly
highness
highnesses
highway
highwayman
highwaymen
highways
hijack
hijacked
hike
hiked
hiker
hikes
hiking
hilarious
hilariously
hilarity
hill
hillbilly
hillock
hills
hillside
hillsides
hilltop
hilltops
hilt
hilts
him
himself
hind
hinder
hindered
hindering
hinders
hindrance
hindrances
hindsight
hinge
hinged
hinges
hint
hinted
hinting
hints
hip
hippo
hippopotamus
hips
hire
hired
hirer
hirers
hires
hiring
hirings
his
hiss
hissed
hisses
hissing
histogram
histograms
historian
historians
historic
historical
historically
histories
history
hit
hitch
hitched
hitchhike
hitchhiked
hitchhiker
hitchhikers
hitchhikes
hitchhiking
hitching
hither
hitherto
hits
hitter
hitters
hitting
hive
hoar
hoard
hoarder
hoarding
hoariness
hoarse
hoarsely
hoarseness
hoary
hobbies
hobble
hobbled
hobbles
hobbling
hobby
hobbyhorse
hobbyist
hobbyists
hockey
hodgepodge
hoe
hoes
hog
hogging
hogs
hoist
hoisted
hoisting
hoists
hold
holden
holder
holders
holding
holdings
holds
hole
holed
holes
holiday
holidays
holies
holiness
holistic
hollow
hollowed
hollowing
hollowly
hollowness
hollows
holly
holocaust
hologram
holograms
holy
homage
home
homed
homeless
homely
homemade
homemaker
homemakers
homeomorphic
homeomorphism
homeomorphisms
homeopath
homeowner
homer
homers
homes
homesick
homesickness
homespun
homestead
homesteader
homesteaders
homesteads
homeward
homewards
homework
homicidal
homicide
homing
homo
homogeneities
homogeneity
homogeneous
homogeneously
homogeneousness
homomorphic
homomorphism
homomorphisms
homosexual
hone
honed
honer
hones
honest
honestly
honesty
honey
honeybee
honeycomb
honeycombed
honeydew
honeymoon
honeymooned
honeymooner
honeymooners
honeymooning
honeymoons
honeysuckle
honing
honor
honorable
honorableness
honorably
honoraries
honorarium
honorary
honored
honorer
honoring
honors
hood
hooded
hoodlum
hoods
hoodwink
hoodwinked
hoodwinking
hoodwinks
hoof
hoofs
hook
hooked
hooker
hookers
hooking
hooks
hookup
hookups
hoop
hooper
hoops
hoot
hooted
hooter
hooting
hoots
hooves
hop
hope
hoped
hopeful
hopefully
hopefulness
hopefuls
hopeless
hopelessly
hopelessness
hopes
hoping
hopper
hoppers
hopping
hops
horde
hordes
horizon
horizons
horizontal
horizontally
hormone
hormones
horn
horned
hornet
hornets
horns
horny
horrendous
horrendously
horrible
horribleness
horribly
horrid
horridly
horrified
horrifies
horrify
horrifying
horror
horrors
horse
horseback
horseflesh
horsefly
horseman
horseplay
horsepower
horses
horseshoe
horseshoer
horticulture
hose
hoses
hospitable
hospitably
hospital
hospitality
hospitalize
hospitalized
hospitalizes
hospitalizing
hospitals
host
hostage
hostages
hosted
hostess
hostesses
hostile
hostilely
hostilities
hostility
hosting
hosts
hot
hotel
hotels
hotly
hotness
hotter
hottest
hound
hounded
hounding
hounds
hour
hourglass
hourly
hours
house
houseboat
housebroken
housed
houseflies
housefly
household
householder
householders
households
housekeeper
housekeepers
housekeeping
houses
housetop
housetops
housewife
housewifely
housewives
housework
housing
hovel
hovels
hover
hovered
hovering
hovers
how
however
howl
howled
howler
howling
howls
hub
hubris
hubs
huddle
huddled
huddling
hue
hues
hug
huge
hugely
hugeness
hugging
huh
hull
hulls
hum
human
humane
humanely
humaneness
humanitarian
humanities
humanity
humanly
humanness
humans
humble
humbled
humbleness
humbler
humblest
humbling
humbly
humbug
humerus
humid
humidification
humidified
humidifier
humidifiers
humidifies
humidify
humidifying
humidity
humidly
humiliate
humiliated
humiliates
humiliating
humiliation
humiliations
humility
hummed
humming
hummingbird
humor
humored
humorer
humorers
humoring
humorous
humorously
humorousness
humors
hump
humpback
humped
hums
hunch
hunched
hunches
hundred
hundredfold
hundreds
hundredth
hung
hunger
hungered
hungering
hungers
hungrier
hungriest
hungrily
hungry
hunk
hunks
hunt
hunted
hunters
hunting
hunts
huntsman
hurdle
hurl
hurled
hurler
hurlers
hurling
hurrah
hurricane
hurricanes
hurried
hurriedly
hurries
hurry
hurrying
hurt
hurting
hurtle
hurtling
hurts
husband
husbandry
husbands
hush
hushed
hushes
hushing
husk
husked
husker
huskiness
husking
husks
husky
hustle
hustled
hustler
hustles
hustling
hut
hutch
huts
hyacinth
hybrid
hydra
hydrant
hydraulic
hydro
hydrodynamic
hydrodynamics
hydrogen
hydrogens
hyena
hygiene
hymen
hymn
hymns
hyper
hyperbola
hyperbolic
hypertext
hyphen
hyphenate
hyphens
hypnosis
hypnotic
hypocrisies
hypocrisy
hypocrite
hypocrites
hypodermic
hypodermics
hypotheses
hypothesis
hypothesize
hypothesized
hypothesizer
hypothesizes
hypothesizing
hypothetical
hypothetically
hysteresis
hysterical
hysterically
ibex
ibid
ibis
ice
iceberg
icebergs
icebox
iced
ices
icicle
iciness
icing
icings
icon
iconoclasm
iconoclast
icons
icosahedra
icosahedral
icosahedron
icy
idea
ideal
idealism
idealistic
idealization
idealizations
idealize
idealized
idealizes
idealizing
ideally
ideals
ideas
idem
idempotency
idempotent
identical
identically
identifiable
identifiably
identification
identifications
identified
identifier
identifiers
identifies
identify
identifying
identities
identity
ideological
ideologically
ideology
idiocy
idiom
idiosyncrasies
idiosyncrasy
idiosyncratic
idiot
idiotic
idiots
idle
idled
idleness
idler
idlers
idles
idlest
idling
idly
idol
idolatry
idols
if
igloo
ignite
ignition
ignoble
ignominious
ignoramus
ignorance
ignorant
ignorantly
ignore
ignored
ignores
ignoring
ill
illegal
illegalities
illegality
illegally
illegitimate
illicit
illicitly
illiteracy
illiterate
illness
illnesses
illogical
illogically
ills
illuminate
illuminated
illuminates
illuminating
illumination
illuminations
illusion
illusions
illusive
illusively
illusory
illustrate
illustrated
illustrates
illustrating
illustration
illustrations
illustrative
illustratively
illustrator
illustrators
illustrious
illustriousness
illy
image
imagery
images
imaginable
imaginably
imaginary
imagination
imaginations
imaginative
imaginatively
imagine
imagined
imagines
imaging
imagining
imaginings
imbalance
imbalances
imbecile
imbibe
imitate
imitated
imitates
imitating
imitation
imitations
imitative
immaculate
immaculately
immaterial
immaterially
immature
immaturity
immediacies
immediacy
immediate
immediately
immemorial
immense
immensely
immerse
immersed
immerses
immersion
immigrant
immigrants
immigrate
immigrated
immigrates
immigrating
immigration
imminent
imminently
immoderate
immodest
immoral
immortal
immortality
immortally
immovability
immovable
immovably
immune
immunities
immunity
immunization
immutable
imp
impact
impacted
impacting
impaction
impactor
impactors
impacts
impair
impaired
impairing
impairs
impale
impart
imparted
impartial
impartially
imparts
impasse
impassive
impatience
impatient
impatiently
impeach
impeachable
impeached
impeachment
impeccable
impedance
impedances
impede
impeded
impedes
impediment
impediments
impeding
impel
impelled
impelling
impend
impending
impenetrability
impenetrable
impenetrably
imperative
imperatively
imperatives
imperceivable
imperceptible
imperfect
imperfection
imperfections
imperfectly
imperial
imperialism
imperialist
imperialists
imperil
imperiled
imperious
imperiously
impermanence
impermanent
impermeable
impermissible
impersonal
impersonally
impersonate
impersonated
impersonates
impersonating
impersonation
impersonations
impertinent
impertinently
impervious
imperviously
impetuous
impetuously
impetus
impinge
impinged
impinges
impinging
impious
implacable
implant
implanted
implanting
implants
implausible
implement
implementable
implementation
implementations
implemented
implementer
implementing
implementor
implementors
implements
implicant
implicants
implicate
implicated
implicates
implicating
implication
implications
implicit
implicitly
implicitness
implied
implies
implore
implored
imploring
imply
implying
impolite
import
importance
important
importantly
importation
imported
importer
importers
importing
imports
impose
imposed
imposes
imposing
imposition
impositions
impossibilities
impossibility
impossible
impossibly
impostor
impostors
impotence
impotency
impotent
impound
impoverish
impoverished
impoverishment
impracticable
impractical
impracticality
impractically
imprecise
imprecisely
imprecision
impregnable
impregnate
impress
impressed
impresser
impresses
impressible
impressing
impression
impressionable
impressionist
impressionistic
impressions
impressive
impressively
impressiveness
impressment
imprimatur
imprint
imprinted
imprinting
imprints
imprison
imprisoned
imprisoning
imprisonment
imprisonments
imprisons
improbability
improbable
impromptu
improper
improperly
impropriety
improve
improved
improvement
improvements
improves
improving
improvisation
improvisational
improvisations
improvise
improvised
improviser
improvisers
improvises
improvising
imprudent
imps
impudent
impudently
impugn
impulse
impulses
impulsion
impulsive
impunity
impure
impurities
impurity
impute
imputed
in
inability
inaccessible
inaccuracies
inaccuracy
inaccurate
inaction
inactivate
inactive
inactivity
inadequacies
inadequacy
inadequate
inadequately
inadequateness
inadmissibility
inadmissible
inadvertent
inadvertently
inadvisable
inalienable
inalterable
inane
inanimate
inanimately
inapplicable
inapproachable
inappropriate
inappropriateness
inasmuch
inattention
inaudible
inaugural
inaugurate
inaugurated
inaugurating
inauguration
inauspicious
inboard
inbound
inbreed
incalculable
incandescent
incantation
incapable
incapacitate
incapacitating
incarcerate
incarnation
incarnations
incendiaries
incendiary
incense
incensed
incenses
incentive
incentives
inception
incessant
incessantly
incest
incestuous
inch
inched
inches
inching
incidence
incident
incidental
incidentally
incidentals
incidents
incinerate
incipient
incisive
incite
incited
incitement
incites
inciting
inclement
inclination
inclinations
incline
inclined
inclines
inclining
inclose
inclosed
incloses
inclosing
include
included
includes
including
inclusion
inclusions
inclusive
inclusively
inclusiveness
incoherence
incoherent
incoherently
income
incomes
incoming
incommensurable
incommensurate
incommunicable
incomparable
incomparably
incompatibilities
incompatibility
incompatible
incompatibly
incompetence
incompetent
incompetents
incomplete
incompletely
incompleteness
incomprehensibility
incomprehensible
incomprehensibly
incomprehension
incompressible
incomputable
inconceivable
inconclusive
incongruity
incongruous
inconsequential
inconsequentially
inconsiderable
inconsiderate
inconsiderately
inconsiderateness
inconsistencies
inconsistency
inconsistent
inconsistently
inconspicuous
incontestable
incontrovertible
incontrovertibly
inconvenience
inconvenienced
inconveniences
inconveniencing
inconvenient
inconveniently
inconvertible
incorporate
incorporated
incorporates
incorporating
incorporation
incorrect
incorrectly
incorrectness
incorrigible
increase
increased
increases
increasing
increasingly
incredible
incredibly
incredulity
incredulous
incredulously
increment
incremental
incrementally
incremented
incrementer
incrementing
increments
incriminate
incubate
incubated
incubates
incubating
incubation
incubator
incubators
inculcate
incumbent
incur
incurable
incurred
incurring
incurs
incursion
indebted
indebtedness
indecent
indecipherable
indecision
indecisive
indeed
indefatigable
indefensible
indefinite
indefinitely
indefiniteness
indelible
indemnify
indemnity
indent
indentation
indentations
indented
indenting
indents
indenture
independence
independent
independently
indescribable
indestructible
indeterminacies
indeterminacy
indeterminate
indeterminately
index
indexable
indexed
indexes
indexing
indicate
indicated
indicates
indicating
indication
indications
indicative
indicator
indicators
indices
indict
indictment
indictments
indifference
indifferent
indifferently
indigenous
indigenously
indigenousness
indigestible
indigestion
indignant
indignantly
indignation
indignities
indignity
indigo
indirect
indirected
indirecting
indirection
indirections
indirectly
indirects
indiscreet
indiscretion
indiscriminate
indiscriminately
indispensability
indispensable
indispensably
indisputable
indistinct
indistinguishable
individual
individualism
individualistic
individuality
individualize
individualized
individualizes
individualizing
individually
individuals
indivisibility
indivisible
indoctrinate
indoctrinated
indoctrinates
indoctrinating
indoctrination
indolent
indolently
indomitable
indoor
indoors
indubitable
induce
induced
inducement
inducements
inducer
induces
inducing
induct
inductance
inductances
inducted
inductee
inducting
induction
inductions
inductive
inductively
inductor
inductors
inducts
indulge
indulged
indulgence
indulgences
indulgent
indulging
industrial
industrialism
industrialist
industrialists
industrialization
industrialized
industrially
industrials
industries
industrious
industriously
industriousness
industry
ineffective
ineffectively
ineffectiveness
ineffectual
inefficiencies
inefficiency
inefficient
inefficiently
inelegant
ineligible
inept
inequalities
inequality
inequitable
inequity
inert
inertia
inertial
inertly
inertness
inescapable
inescapably
inessential
inestimable
inevitabilities
inevitability
inevitable
inevitably
inexact
inexcusable
inexcusably
inexhaustible
inexorable
inexorably
inexpensive
inexpensively
inexperience
inexperienced
inexplicable
infallibility
infallible
infallibly
infamous
infamously
infamy
infancy
infant
infantile
infantry
infantryman
infantrymen
infants
infarct
infatuate
infeasible
infect
infected
infecting
infection
infections
infectious
infectiously
infective
infects
infer
inference
inferences
inferential
inferior
inferiority
inferiors
infernal
infernally
inferno
infernos
inferred
inferring
infers
infertile
infest
infested
infesting
infests
infidel
infidelity
infidels
infighting
infiltrate
infinite
infinitely
infiniteness
infinitesimal
infinitive
infinitives
infinitude
infinitum
infinity
infirm
infirmary
infirmity
infix
inflame
inflamed
inflammable
inflammation
inflammatory
inflatable
inflate
inflated
inflater
inflates
inflating
inflation
inflationary
inflexibility
inflexible
inflict
inflicted
inflicting
inflicts
inflow
influence
influenced
influences
influencing
influential
influentially
influenza
inform
informal
informality
informally
informant
informants
information
informational
informative
informatively
informed
informer
informers
informing
informs
infra
infrared
infrastructure
infrequent
infrequently
infringe
infringed
infringement
infringements
infringes
infringing
infuriate
infuriated
infuriates
infuriating
infuriation
infuse
infused
infuses
infusing
infusion
infusions
ingenious
ingeniously
ingeniousness
ingenuity
ingenuous
ingest
ingestion
inglorious
ingot
ingrate
ingratiate
ingratitude
ingredient
ingredients
ingrown
inhabit
inhabitable
inhabitance
inhabitant
inhabitants
inhabited
inhabiting
inhabits
inhale
inhaled
inhaler
inhales
inhaling
inhere
inherent
inherently
inheres
inherit
inheritable
inheritance
inheritances
inherited
inheriting
inheritor
inheritors
inheritress
inheritresses
inheritrices
inheritrix
inherits
inhibit
inhibited
inhibiting
inhibition
inhibitions
inhibitor
inhibitors
inhibitory
inhibits
inhomogeneities
inhomogeneity
inhomogeneous
inhospitable
inhuman
inhumane
inimical
inimitable
iniquities
iniquity
initial
initialed
initialing
initialization
initializations
initialize
initialized
initializer
initializers
initializes
initializing
initially
initials
initiate
initiated
initiates
initiating
initiation
initiations
initiative
initiatives
initiator
initiators
inject
injected
injecting
injection
injections
injective
injects
injudicious
injunction
injunctions
injure
injured
injures
injuries
injuring
injurious
injury
injustice
injustices
ink
inked
inker
inkers
inking
inkings
inkling
inklings
inks
inlaid
inland
inlay
inlet
inlets
inline
inmate
inmates
inn
innards
innate
innately
inner
innermost
inning
innings
innocence
innocent
innocently
innocents
innocuous
innocuously
innocuousness
innovate
innovation
innovations
innovative
inns
innuendo
innumerability
innumerable
innumerably
inoculate
inoperable
inoperative
inopportune
inordinate
inordinately
inorganic
input
inputs
inquest
inquire
inquired
inquirer
inquirers
inquires
inquiries
inquiring
inquiry
inquisition
inquisitions
inquisitive
inquisitively
inquisitiveness
inroad
inroads
insane
insanely
insanity
insatiable
inscribe
inscribed
inscribes
inscribing
inscription
inscriptions
inscrutable
insect
insecticide
insects
insecure
insecurely
inseminate
insensible
insensitive
insensitively
insensitivity
inseparable
insert
inserted
inserting
insertion
insertions
inserts
inset
inside
insider
insiders
insides
insidious
insidiously
insidiousness
insight
insightful
insights
insignia
insignificance
insignificant
insincere
insincerity
insinuate
insinuated
insinuates
insinuating
insinuation
insinuations
insipid
insist
insisted
insistence
insistent
insistently
insisting
insists
insofar
insolence
insolent
insolently
insoluble
insolvable
insolvent
insomnia
insomniac
inspect
inspected
inspecting
inspection
inspections
inspector
inspectors
inspects
inspiration
inspirations
inspire
inspired
inspirer
inspires
inspiring
instabilities
instability
install
installation
installations
installed
installer
installers
installing
installment
installments
installs
instance
instances
instant
instantaneous
instantaneously
instanter
instantiate
instantiated
instantiates
instantiating
instantiation
instantiations
instantly
instants
instead
instigate
instigated
instigates
instigating
instigator
instigators
instill
instinct
instinctive
instinctively
instincts
instinctual
institute
instituted
instituter
instituters
institutes
instituting
institution
institutional
institutionalize
institutionalized
institutionalizes
institutionalizing
institutionally
institutions
instruct
instructed
instructing
instruction
instructional
instructions
instructive
instructively
instructor
instructors
instructs
instrument
instrumental
instrumentalist
instrumentalists
instrumentally
instrumentals
instrumentation
instrumented
instrumenting
instruments
insubordinate
insufferable
insufficient
insufficiently
insular
insulate
insulated
insulates
insulating
insulation
insulator
insulators
insulin
insult
insulted
insulting
insults
insuperable
insupportable
insurance
insure
insured
insurer
insurers
insures
insurgent
insurgents
insuring
insurmountable
insurrection
insurrections
intact
intangible
intangibles
integer
integers
integrable
integral
integrals
integrand
integrate
integrated
integrates
integrating
integration
integrations
integrative
integrity
intellect
intellects
intellectual
intellectually
intellectuals
intelligence
intelligent
intelligently
intelligentsia
intelligibility
intelligible
intelligibly
intemperate
intend
intended
intending
intends
intense
intensely
intensification
intensified
intensifier
intensifiers
intensifies
intensify
intensifying
intensities
intensity
intensive
intensively
intent
intention
intentional
intentionally
intentioned
intentions
intently
intentness
intents
inter
interact
interacted
interacting
interaction
interactions
interactive
interactively
interactivity
interacts
intercept
intercepted
intercepting
interception
interceptor
intercepts
interchange
interchangeability
interchangeable
interchangeably
interchanged
interchanger
interchanges
interchanging
interchangings
interchannel
intercity
intercom
intercommunicate
intercommunicated
intercommunicates
intercommunicating
intercommunication
interconnect
interconnected
interconnecting
interconnection
interconnections
interconnects
intercontinental
intercourse
interdependence
interdependencies
interdependency
interdependent
interdict
interdiction
interdisciplinary
interest
interested
interesting
interestingly
interests
interface
interfaced
interfacer
interfaces
interfacing
interfere
interfered
interference
interferences
interferes
interfering
interferingly
interferometer
interferometric
interferometry
interframe
intergroup
interim
interior
interiors
interject
interlace
interlaced
interlaces
interlacing
interleave
interleaved
interleaves
interleaving
interlink
interlinked
interlinks
interlisp
intermediary
intermediate
intermediates
interminable
intermingle
intermingled
intermingles
intermingling
intermission
intermittent
intermittently
intermix
intermixed
intermodule
intern
internal
internalize
internalized
internalizes
internalizing
internally
internals
international
internationality
internationally
interned
internetwork
interning
interns
internship
interoffice
interpersonal
interplay
interpolate
interpolated
interpolates
interpolating
interpolation
interpolations
interpose
interposed
interposes
interposing
interpret
interpretable
interpretation
interpretations
interpreted
interpreter
interpreters
interpreting
interpretive
interpretively
interprets
interprocess
interrelate
interrelated
interrelates
interrelating
interrelation
interrelations
interrelationship
interrelationships
interrogate
interrogated
interrogates
interrogating
interrogation
interrogations
interrogative
interrupt
interrupted
interruptible
interrupting
interruption
interruptions
interruptive
interrupts
intersect
intersected
intersecting
intersection
intersections
intersects
intersperse
interspersed
intersperses
interspersing
interspersion
interstage
interstate
intertwine
intertwined
intertwines
intertwining
interval
intervals
intervene
intervened
intervenes
intervening
intervention
interventions
interview
interviewed
interviewee
interviewer
interviewers
interviewing
interviews
interwoven
intestate
intestinal
intestine
intestines
intimacy
intimate
intimated
intimately
intimating
intimation
intimations
intimidate
intimidated
intimidates
intimidating
intimidation
into
intolerable
intolerably
intolerance
intolerant
intonation
intonations
intone
intoxicant
intoxicate
intoxicated
intoxicating
intoxication
intractability
intractable
intractably
intragroup
intraline
intramural
intramuscular
intransigent
intransitive
intransitively
intraoffice
intraprocess
intrastate
intravenous
intrepid
intricacies
intricacy
intricate
intricately
intrigue
intrigued
intrigues
intriguing
intrinsic
intrinsically
introduce
introduced
introduces
introducing
introduction
introductions
introductory
introspect
introspection
introspections
introspective
introvert
introverted
intrude
intruded
intruder
intruders
intrudes
intruding
intrusion
intrusions
intrust
intubate
intubated
intubates
intubation
intuition
intuitionist
intuitions
intuitive
intuitively
inundate
invade
invaded
invader
invaders
invades
invading
invalid
invalidate
invalidated
invalidates
invalidating
invalidation
invalidations
invalidities
invalidity
invalidly
invalids
invaluable
invariable
invariably
invariance
invariant
invariantly
invariants
invasion
invasions
invective
invent
invented
inventing
invention
inventions
inventive
inventively
inventiveness
inventor
inventories
inventors
inventory
invents
inverse
inversely
inverses
inversion
inversions
invert
invertebrate
invertebrates
inverted
inverter
inverters
invertible
inverting
inverts
invest
invested
investigate
investigated
investigates
investigating
investigation
investigations
investigative
investigator
investigators
investigatory
investing
investment
investments
investor
investors
invests
inveterate
invigorate
invincible
invisibility
invisible
invisibly
invitation
invitations
invite
invited
invites
inviting
invocable
invocation
invocations
invoice
invoiced
invoices
invoicing
invoke
invoked
invoker
invokes
invoking
involuntarily
involuntary
involve
involved
involvement
involvements
involves
involving
inward
inwardly
inwardness
inwards
iodine
ion
ionosphere
ionospheric
ions
iota
irate
irately
irateness
ire
ires
iris
irk
irked
irking
irks
irksome
iron
ironed
ironic
ironical
ironically
ironies
ironing
ironings
irons
irony
irradiate
irrational
irrationally
irrationals
irreconcilable
irrecoverable
irreducible
irreducibly
irreflexive
irrefutable
irregular
irregularities
irregularity
irregularly
irregulars
irrelevance
irrelevances
irrelevant
irrelevantly
irreplaceable
irrepressible
irreproducibility
irreproducible
irresistible
irrespective
irrespectively
irresponsible
irresponsibly
irretrievably
irreverent
irreversibility
irreversible
irreversibly
irrevocable
irrevocably
irrigate
irrigated
irrigates
irrigating
irrigation
irritable
irritant
irritate
irritated
irritates
irritating
irritation
irritations
is
island
islander
islanders
islands
isle
isles
islet
islets
isolate
isolated
isolates
isolating
isolation
isolations
isometric
isomorphic
isomorphically
isomorphism
isomorphisms
isotope
isotopes
issuance
issue
issued
issuer
issuers
issues
issuing
isthmus
it
italic
italicize
italicized
italics
itch
itches
itching
item
itemization
itemizations
itemize
itemized
itemizes
itemizing
items
iterate
iterated
iterates
iterating
iteration
iterations
iterative
iteratively
iterator
iterators
itineraries
itinerary
its
itself
ivies
ivory
ivy
jab
jabbed
jabbing
jabs
jack
jackass
jacket
jacketed
jackets
jacking
jackknife
jackpot
jade
jaded
jaguar
jail
jailed
jailer
jailers
jailing
jails
jam
jammed
jamming
jams
janitor
janitors
jar
jargon
jarred
jarring
jarringly
jars
jaundice
jaunt
jauntiness
jaunts
jaunty
javelin
javelins
jaw
jawbone
jaws
jay
jazz
jazzy
jealous
jealousies
jealously
jealousy
jean
jeans
jeep
jeeps
jeer
jeers
jellies
jelly
jellyfish
jenny
jeopardize
jeopardized
jeopardizes
jeopardizing
jeopardy
jerk
jerked
jerkiness
jerking
jerkings
jerks
jerky
jersey
jerseys
jest
jested
jester
jesting
jests
jet
jetliner
jets
jetted
jetting
jewel
jeweled
jeweler
jewelries
jewelry
jewels
jiffy
jig
jigs
jigsaw
jingle
jingled
jingling
jitter
jitterbug
jittery
job
jobs
jockey
jockstrap
jocund
jog
jogging
jogs
join
joined
joiner
joiners
joining
joins
joint
jointly
joints
joke
joked
joker
jokers
jokes
joking
jokingly
jolly
jolt
jolted
jolting
jolts
jonquil
jostle
jostled
jostles
jostling
jot
jots
jotted
jotting
joule
journal
journalism
journalist
journalists
journalize
journalized
journalizes
journalizing
journals
journey
journeyed
journeying
journeyings
journeyman
journeymen
journeys
joust
jousted
jousting
jousts
jovial
joy
joyful
joyfully
joyous
joyously
joyousness
joyride
joys
joystick
jubilee
judge
judged
judges
judging
judgment
judgments
judicial
judiciary
judicious
judiciously
judo
jug
juggle
juggler
jugglers
juggles
juggling
jugs
juice
juices
juiciest
juicy
jumble
jumbled
jumbles
jumbo
jump
jumped
jumper
jumpers
jumping
jumps
jumpy
junction
junctions
juncture
junctures
jungle
jungles
junior
juniors
juniper
junk
junker
junkers
junks
junky
junta
jure
juries
jurisdiction
jurisdictions
jurisprudence
jurist
juror
jurors
jury
just
justice
justices
justifiable
justifiably
justification
justifications
justified
justifier
justifiers
justifies
justify
justifying
justly
justness
jut
jutting
juvenile
juveniles
juxtapose
juxtaposed
juxtaposes
juxtaposing
kangaroo
kanji
kappa
karate
keel
keeled
keeling
keels
keen
keener
keenest
keenly
keenness
keep
keeper
keepers
keeping
keeps
ken
kennel
kennels
kept
kerchief
kerchiefs
kern
kernel
kernels
kerosene
ketchup
kettle
kettles
key
keyboard
keyboards
keyed
keyhole
keying
keynote
keypad
keypads
keys
keystroke
keystrokes
keyword
keywords
kick
kicked
kicker
kickers
kicking
kickoff
kicks
kid
kidded
kiddie
kidding
kidnap
kidnapper
kidnappers
kidnapping
kidnappings
kidnaps
kidney
kidneys
kids
kill
killed
killer
killers
killing
killingly
killings
killjoy
kills
kilobit
kilobits
kiloblock
kilobyte
kilobytes
kilogram
kilograms
kilohertz
kilohm
kilojoule
kilometer
kilometers
kiloton
kilovolt
kilowatt
kiloword
kimono
kin
kind
kinder
kindergarten
kindest
kindhearted
kindle
kindled
kindles
kindling
kindly
kindness
kindred
kinds
kinetic
king
kingdom
kingdoms
kingly
kingpin
kings
kink
kinky
kinship
kinsman
kiosk
kiss
kissed
kisser
kissers
kisses
kissing
kit
kitchen
kitchenette
kitchens
kite
kited
kites
kiting
kits
kitten
kittenish
kittens
kitty
klaxon
kludge
kludges
klystron
knack
knapsack
knapsacks
knave
knaves
knead
kneads
knee
kneecap
kneed
kneeing
kneel
kneeled
kneeling
kneels
knees
knell
knells
knelt
knew
knife
knifed
knifes
knifing
knight
knighted
knighthood
knighting
knightly
knights
knit
knits
knives
knob
knobs
knock
knockdown
knocked
knocker
knockers
knocking
knockout
knocks
knoll
knolls
knot
knots
knotted
knotting
know
knowable
knower
knowhow
knowing
knowingly
knowledge
knowledgeable
known
knows
knuckle
knuckled
knuckles
koala
kosher
kudo
lab
label
labeled
labeling
labelled
labeller
labellers
labelling
labels
labor
laboratories
laboratory
labored
laborer
laborers
laboring
laborings
laborious
laboriously
labors
labs
labyrinth
labyrinths
lace
laced
lacerate
lacerated
lacerates
lacerating
laceration
lacerations
laces
lacing
lack
lacked
lackey
lacking
lacks
lacquer
lacquered
lacquers
lacrosse
lacy
lad
ladder
laden
ladies
lading
ladle
lads
lady
ladylike
lag
lager
lagers
lagoon
lagoons
lags
laid
lain
lair
lairs
laissez
lake
lakes
lamb
lambda
lambdas
lambert
lambs
lame
lamed
lamely
lameness
lament
lamentable
lamentation
lamentations
lamented
lamenting
laments
lames
laminar
laming
lamp
lamplight
lampoon
lamprey
lamps
lance
lanced
lancer
lances
land
landed
lander
landers
landfill
landing
landings
landladies
landlady
landlord
landlords
landmark
landmarks
landowner
landowners
lands
landscape
landscaped
landscapes
landscaping
landslide
lane
lanes
language
languages
languid
languidly
languidness
languish
languished
languishes
languishing
lantern
lanterns
lap
lapel
lapels
lapping
laps
lapse
lapsed
lapses
lapsing
lard
larder
large
largely
largeness
larger
largest
lark
larks
larva
larvae
larynx
lascivious
laser
lasers
lash
lashed
lashes
lashing
lashings
lass
lasses
lasso
last
lasted
lasting
lastly
lasts
latch
latched
latches
latching
late
lately
latency
lateness
latent
later
lateral
laterally
latest
lathe
latitude
latitudes
latrine
latrines
latter
latterly
lattice
lattices
laudable
laugh
laughable
laughably
laughed
laughing
laughingly
laughingstock
laughs
laughter
launch
launched
launcher
launches
launching
launchings
launder
laundered
launderer
laundering
launderings
launders
laundry
laureate
laurel
laurels
lava
lavatories
lavatory
lavender
lavish
lavished
lavishing
lavishly
law
lawbreaker
lawful
lawfully
lawgiver
lawless
lawlessness
lawn
lawns
laws
lawsuit
lawsuits
lawyer
lawyers
lax
laxative
lay
layer
layered
layering
layers
laying
layman
laymen
layoff
layoffs
layout
layouts
lays
lazed
lazier
laziest
lazily
laziness
lazing
lazy
lazybones
lead
leaded
leaden
leader
leaders
leadership
leaderships
leading
leadings
leads
leaf
leafed
leafiest
leafing
leafless
leaflet
leaflets
leafy
league
leagued
leaguer
leaguers
leagues
leak
leakage
leakages
leaked
leaking
leaks
leaky
lean
leaned
leaner
leanest
leaning
leanness
leans
leap
leaped
leapfrog
leaping
leaps
leapt
learn
learned
learner
learners
learning
learns
lease
leased
leases
leash
leashes
leasing
least
leather
leathered
leathern
leatherneck
leathers
leave
leaved
leaven
leavened
leavening
leaves
leaving
leavings
lechery
lecture
lectured
lecturer
lecturers
lectures
lecturing
led
ledge
ledger
ledgers
ledges
lee
leech
leeches
leek
leer
leery
lees
leeward
leeway
left
leftist
leftists
leftmost
leftover
leftovers
leftward
leg
legacies
legacy
legal
legality
legalization
legalize
legalized
legalizes
legalizing
legally
legend
legendary
legends
legged
leggings
legibility
legible
legibly
legion
legions
legislate
legislated
legislates
legislating
legislation
legislative
legislator
legislators
legislature
legislatures
legitimacy
legitimate
legitimately
legs
legume
leisure
leisurely
lemma
lemmas
lemming
lemmings
lemon
lemonade
lemons
lend
lender
lenders
lending
lends
length
lengthen
lengthened
lengthening
lengthens
lengthly
lengths
lengthwise
lengthy
leniency
lenient
leniently
lens
lenses
lent
lentil
lentils
leopard
leopards
leper
leprosy
less
lessen
lessened
lessening
lessens
lesser
lesson
lessons
lessor
lest
let
lethal
lets
letter
lettered
letterer
letterhead
lettering
letters
letting
lettuce
leukemia
levee
levees
level
leveled
leveler
leveling
levelled
leveller
levellest
levelling
levelly
levelness
levels
lever
leverage
levers
levied
levies
levity
levy
levying
lewd
lewdly
lewdness
lexical
lexically
lexicographic
lexicographical
lexicographically
lexicon
lexicons
liabilities
liability
liable
liaison
liaisons
liar
liars
libel
libelous
liberal
liberalize
liberalized
liberalizes
liberalizing
liberally
liberals
liberate
liberated
liberates
liberating
liberation
liberator
liberators
libertarian
liberties
liberty
libido
librarian
librarians
libraries
library
libretto
lice
license
licensed
licensee
licenses
licensing
licensor
licentious
lichen
lichens
lick
licked
licking
licks
licorice
lid
lids
lie
lied
liege
lien
liens
lies
lieu
lieutenant
lieutenants
life
lifeblood
lifeboat
lifeguard
lifeless
lifelessness
lifelike
lifelong
lifer
lifespan
lifestyle
lifestyles
lifetime
lifetimes
lift
lifted
lifter
lifters
lifting
lifts
ligament
ligature
light
lighted
lighten
lightens
lighter
lighters
lightest
lightface
lighthearted
lighthouse
lighthouses
lighting
lightly
lightness
lightning
lightnings
lights
lightweight
like
liked
likelier
likeliest
likelihood
likelihoods
likeliness
likely
liken
likened
likeness
likenesses
likening
likens
likes
likewise
liking
lilac
lilacs
lilies
lily
limb
limber
limbo
limbs
lime
limelight
limes
limestone
limit
limitability
limitably
limitation
limitations
limited
limiter
limiters
limiting
limitless
limits
limousine
limp
limped
limping
limply
limpness
limps
linden
line
linear
linearities
linearity
linearizable
linearize
linearized
linearizes
linearizing
linearly
lined
linen
linens
liner
liners
lines
lineup
linger
lingered
lingerie
lingering
lingers
lingo
lingua
linguist
linguistic
linguistically
linguistics
linguists
lining
linings
link
linkage
linkages
linked
linker
linkers
linking
links
linoleum
linseed
lint
lion
lioness
lionesses
lions
lip
lips
lipstick
liquid
liquidate
liquidation
liquidations
liquidity
liquids
liquor
liquors
lisp
lisped
lisping
lisps
list
listed
listen
listened
listener
listeners
listening
listens
listers
listing
listings
listless
lists
lit
litany
liter
literacy
literal
literally
literalness
literals
literary
literate
literature
literatures
liters
lithe
lithograph
lithography
litigant
litigate
litigation
litigious
litmus
litter
litterbug
littered
littering
litters
little
littleness
littler
littlest
livable
livably
live
lived
livelihood
lively
liveness
liver
liveried
livers
livery
lives
livestock
livid
living
lizard
lizards
load
loaded
loader
loaders
loading
loadings
loads
loaf
loafed
loafer
loan
loaned
loaning
loans
loath
loathe
loathed
loathing
loathly
loathsome
loaves
lobbied
lobbies
lobby
lobbying
lobe
lobes
lobster
lobsters
local
localities
locality
localization
localize
localized
localizes
localizing
locally
locals
locate
located
locates
locating
location
locations
locative
locatives
locator
locators
loci
lock
locked
locker
lockers
locking
lockings
lockout
lockouts
locks
locksmith
lockstep
lockup
lockups
locomotion
locomotive
locomotives
locus
locust
locusts
lodge
lodged
lodger
lodges
lodging
lodgings
loft
loftiness
lofts
lofty
logarithm
logarithmic
logarithmically
logarithms
logged
logger
loggers
logging
logic
logical
logically
logician
logicians
logics
login
logins
logistic
logistics
logjam
logo
logs
loin
loincloth
loins
loiter
loitered
loiterer
loitering
loiters
lone
lonelier
loneliest
loneliness
lonely
loner
loners
lonesome
long
longed
longer
longest
longevity
longhand
longing
longings
longitude
longitudes
longs
longstanding
look
lookahead
looked
looker
lookers
looking
lookout
looks
lookup
lookups
loom
loomed
looming
looms
loon
loop
looped
loophole
loopholes
looping
loops
loose
loosed
looseleaf
loosely
loosen
loosened
looseness
loosening
loosens
looser
looses
loosest
loosing
loot
looted
looter
looting
loots
lopsided
lord
lordly
lords
lordship
lore
lorry
lose
loser
losers
loses
losing
loss
losses
lossier
lossiest
lossy
lost
lot
lotion
lots
lottery
lotus
loud
louder
loudest
loudly
loudness
loudspeaker
loudspeakers
lounge
lounged
lounges
lounging
louse
lousy
lout
lovable
lovably
love
loved
lovelier
lovelies
loveliest
loveliness
lovelorn
lovely
lover
lovers
loves
loving
lovingly
low
lower
lowered
lowering
lowers
lowest
lowland
lowlands
lowliest
lowly
lowness
lows
loyal
loyally
loyalties
loyalty
lubricant
lubricate
lubrication
lucid
luck
lucked
luckier
luckiest
luckily
luckless
lucks
lucky
lucrative
ludicrous
ludicrously
ludicrousness
luggage
lukewarm
lull
lullaby
lulled
lulls
lumber
lumbered
lumbering
luminous
luminously
lummox
lump
lumped
lumping
lumps
lumpy
lunar
lunatic
lunch
lunched
luncheon
luncheons
lunches
lunching
lung
lunged
lungs
lurch
lurched
lurches
lurching
lure
lured
lures
luring
lurk
lurked
lurking
lurks
luscious
lusciously
lusciousness
lush
lust
luster
lustful
lustily
lustiness
lustrous
lusts
lusty
lute
lutes
luxuriant
luxuriantly
luxuries
luxurious
luxuriously
luxury
lying
lymph
lynch
lynched
lyncher
lynches
lynx
lynxes
lyre
lyric
lyrics
mace
maced
maces
machination
machine
machined
machinelike
machinery
machines
machining
macho
macintosh
mackerel
macro
macroeconomics
macromolecule
macromolecules
macrophage
macros
macroscopic
mad
madam
madden
maddening
madder
maddest
made
madhouse
madly
madman
madmen
madness
madras
maestro
magazine
magazines
magenta
maggot
maggots
magic
magical
magically
magician
magicians
magistrate
magistrates
magna
magnesium
magnet
magnetic
magnetically
magnetism
magnetisms
magnetizable
magnetized
magneto
magnification
magnificence
magnificent
magnificently
magnified
magnifier
magnifies
magnify
magnifying
magnitude
magnitudes
magnolia
magnum
magpie
mahogany
maid
maiden
maidens
maids
mail
mailable
mailbox
mailboxes
mailed
mailer
mailing
mailings
mailman
mailmen
mails
maim
maimed
maiming
maims
main
mainframe
mainframes
mainland
mainline
mainly
mains
mainstay
mainstream
maintain
maintainability
maintainable
maintained
maintainer
maintainers
maintaining
maintains
maintenance
maintenances
maize
majestic
majesties
majesty
major
majored
majoring
majorities
majority
majors
makable
make
maker
makers
makes
makeshift
makeup
makeups
making
makings
maladies
malady
malaria
malcontent
male
malefactor
malefactors
maleness
males
malevolent
malformed
malfunction
malfunctioned
malfunctioning
malfunctions
malice
malicious
maliciously
maliciousness
malign
malignant
malignantly
mall
mallard
mallet
mallets
malnutrition
malpractice
malt
malted
malts
mama
mamma
mammal
mammalian
mammals
mammas
mammoth
man
manage
manageable
manageableness
managed
management
managements
manager
managerial
managers
manages
managing
mandarin
mandate
mandated
mandates
mandating
mandatory
mandible
mane
manes
maneuver
maneuvered
maneuvering
maneuvers
manger
mangers
mangle
mangled
mangler
mangles
mangling
manhole
manhood
mania
maniac
maniacal
maniacs
manic
manicure
manicured
manicures
manicuring
manifest
manifestation
manifestations
manifested
manifesting
manifestly
manifests
manifold
manifolds
manipulability
manipulable
manipulatable
manipulate
manipulated
manipulates
manipulating
manipulation
manipulations
manipulative
manipulator
manipulators
manipulatory
mankind
manly
manned
manner
mannered
mannerly
manners
manning
manometer
manometers
manor
manors
manpower
mansion
mansions
manslaughter
mantel
mantels
mantis
mantissa
mantissas
mantle
mantlepiece
mantles
manual
manually
manuals
manufacture
manufactured
manufacturer
manufacturers
manufactures
manufacturing
manure
manuscript
manuscripts
many
map
maple
maples
mappable
mapped
mapping
mappings
maps
marathon
marble
marbles
marbling
march
marched
marcher
marches
marching
mare
mares
margarine
margin
marginal
marginally
margins
marigold
marijuana
marina
marinade
marinate
marine
mariner
marines
marionette
marital
maritime
mark
markable
marked
markedly
marker
markers
market
marketability
marketable
marketed
marketing
marketings
marketplace
marketplaces
markets
marking
markings
marmalade
marmot
maroon
marquis
marriage
marriageable
marriages
married
marries
marrow
marry
marrying
marsh
marshal
marshaled
marshaling
marshals
marshes
marshmallow
mart
marten
martial
martingale
martini
marts
martyr
martyrdom
martyrs
marvel
marveled
marvelled
marvelling
marvelous
marvelously
marvelousness
marvels
mascara
masculine
masculinely
masculinity
mash
mashed
mashes
mashing
mask
maskable
masked
masker
masking
maskings
masks
masochist
masochists
mason
masonry
masons
masquerade
masquerader
masquerades
masquerading
mass
massacre
massacred
massacres
massage
massages
massaging
massed
masses
massing
massive
mast
masted
master
mastered
masterful
masterfully
mastering
masterings
masterly
mastermind
masterpiece
masterpieces
masters
mastery
mastodon
masts
masturbate
masturbated
masturbates
masturbating
masturbation
mat
match
matchable
matched
matcher
matchers
matches
matching
matchings
matchless
mate
mated
mater
material
materialist
materialize
materialized
materializes
materializing
materially
materials
maternal
maternally
maternity
mates
math
mathematical
mathematically
mathematician
mathematicians
mathematics
mating
matings
matriarch
matriarchal
matrices
matriculate
matriculation
matrimonial
matrimony
matrix
matroid
matron
matronly
mats
matted
matter
mattered
matters
mattress
mattresses
maturation
mature
matured
maturely
matures
maturing
maturities
maturity
maul
mausoleum
maverick
maxim
maxima
maximal
maximally
maximize
maximized
maximizer
maximizers
maximizes
maximizing
maxims
maximum
maximums
maybe
mayhap
mayhem
mayonnaise
mayor
mayoral
mayors
maze
mazes
me
mead
meadow
meadows
meager
meagerly
meagerness
meal
meals
mealtime
mealy
mean
meander
meandered
meandering
meanders
meaner
meanest
meaning
meaningful
meaningfully
meaningfulness
meaningless
meaninglessly
meaninglessness
meanings
meanly
meanness
means
meant
meantime
meanwhile
measle
measles
measurable
measurably
measure
measured
measurement
measurements
measurer
measures
measuring
meat
meats
meaty
mechanic
mechanical
mechanically
mechanics
mechanism
mechanisms
mechanization
mechanizations
mechanize
mechanized
mechanizes
mechanizing
medal
medallion
medallions
medals
meddle
meddled
meddler
meddles
meddling
media
median
medians
mediate
mediated
mediates
mediating
mediation
mediations
mediator
medic
medical
medically
medicinal
medicinally
medicine
medicines
medics
medieval
mediocre
mediocrity
meditate
meditated
meditates
meditating
meditation
meditations
meditative
medium
mediums
medley
meek
meeker
meekest
meekly
meekness
meet
meeting
meetinghouse
meetings
meets
megabaud
megabit
megabits
megabyte
megabytes
megahertz
megalomania
megaton
megavolt
megawatt
megaword
megawords
megohm
melancholy
mellow
mellowed
mellowing
mellowness
mellows
melodies
melodious
melodiously
melodiousness
melodrama
melodramas
melodramatic
melody
melon
melons
melt
melted
melting
meltingly
melts
member
members
membership
memberships
membrane
memento
memo
memoir
memoirs
memorabilia
memorable
memorableness
memoranda
memorandum
memorial
memorially
memorials
memories
memorization
memorize
memorized
memorizer
memorizes
memorizing
memory
memoryless
memos
men
menace
menaced
menacing
menagerie
menarche
mend
mendacious
mendacity
mended
mender
mending
mends
menial
menials
mens
menstruate
mensurable
mensuration
mental
mentalities
mentality
mentally
mention
mentionable
mentioned
mentioner
mentioners
mentioning
mentions
mentor
mentors
menu
menus
mercantile
mercenaries
mercenariness
mercenary
merchandise
merchandiser
merchandising
merchant
merchants
merciful
mercifully
merciless
mercilessly
mercurial
mercury
mercy
mere
merely
merest
merge
merged
merger
mergers
merges
merging
meridian
meringue
merit
merited
meriting
meritorious
meritoriously
meritoriousness
merits
mermaid
merriest
merrily
merriment
merry
mescaline
mesh
meson
mesquite
mess
message
messages
messed
messenger
messengers
messes
messiahs
messier
messiest
messily
messiness
messing
messy
met
meta
metabolic
metabolism
metacircular
metacircularity
metal
metalanguage
metallic
metallization
metallizations
metallurgy
metals
metamathematical
metamorphosis
metaphor
metaphorical
metaphorically
metaphors
metaphysical
metaphysically
metaphysics
metavariable
mete
meted
meteor
meteoric
meteorite
meteoritic
meteorology
meteors
meter
metering
meters
metes
methane
method
methodical
methodically
methodicalness
methodists
methodological
methodologically
methodologies
methodologists
methodology
methods
meticulously
meting
metric
metrical
metrics
metro
metronome
metropolis
metropolitan
mets
mettle
mettlesome
mew
mewed
mews
miasma
mica
mice
micro
microarchitects
microarchitecture
microarchitectures
microbial
microbicidal
microbicide
microcode
microcoded
microcodes
microcoding
microcomputer
microcomputers
microcosm
microcycle
microcycles
microeconomics
microelectronics
microfilm
microfilms
microgramming
microinstruction
microinstructions
microjump
microjumps
microlevel
micron
microoperations
microphone
microphones
microphoning
microprocedure
microprocedures
microprocessing
microprocessor
microprocessors
microprogram
microprogrammable
microprogrammed
microprogrammer
microprogramming
microprograms
micros
microscope
microscopes
microscopic
microscopy
microsecond
microseconds
microstore
microsystems
microwave
microwaves
microword
microwords
mid
midday
middle
middleman
middlemen
middles
middling
midget
midnight
midnights
midpoint
midpoints
midrange
midscale
midsection
midshipman
midshipmen
midst
midstream
midsts
midsummer
midway
midweek
midwife
midwinter
midwives
mien
might
mightier
mightiest
mightily
mightiness
mighty
migrant
migrate
migrated
migrates
migrating
migration
migrations
migratory
mike
mild
milder
mildest
mildew
mildly
mildness
mile
mileage
milestone
milestones
militant
militantly
militarily
militarism
military
militia
milk
milked
milker
milkers
milkiness
milking
milkmaid
milkmaids
milks
milky
mill
milled
millennium
miller
millet
milliammeter
milliampere
millijoule
millimeter
millimeters
millinery
milling
million
millionaire
millionaires
millions
millionth
millipede
millipedes
millisecond
milliseconds
millivolt
millivoltmeter
milliwatt
millstone
millstones
mimeograph
mimic
mimicked
mimicking
mimics
minaret
mince
minced
mincemeat
minces
mincing
mind
minded
mindful
mindfully
mindfulness
minding
mindless
mindlessly
minds
mine
mined
minefield
miner
mineral
minerals
miners
mines
minesweeper
mingle
mingled
mingles
mingling
mini
miniature
miniatures
miniaturization
miniaturize
miniaturized
miniaturizes
miniaturizing
minicomputer
minicomputers
minima
minimal
minimally
minimax
minimization
minimizations
minimize
minimized
minimizer
minimizers
minimizes
minimizing
minimum
mining
minion
minis
minister
ministered
ministering
ministers
ministries
ministry
mink
minks
minnow
minnows
minor
minoring
minorities
minority
minors
minstrel
minstrels
mint
minted
minter
minting
mints
minuend
minuet
minus
minuscule
minute
minutely
minuteman
minutemen
minuteness
minuter
minutes
miracle
miracles
miraculous
miraculously
mirage
mire
mired
mires
mirror
mirrored
mirroring
mirrors
mirth
misanthrope
misbehaving
miscalculation
miscalculations
miscarriage
miscarry
miscegenation
miscellaneous
miscellaneously
miscellaneousness
mischief
mischievous
mischievously
mischievousness
misconception
misconceptions
misconduct
misconstrue
misconstrued
misconstrues
misdemeanors
miser
miserable
miserableness
miserably
miseries
miserly
misers
misery
misfit
misfits
misfortune
misfortunes
misgiving
misgivings
misguided
mishap
mishaps
misinformed
misjudged
misjudgment
mislead
misleading
misleads
misled
mismanagement
mismatch
mismatched
mismatches
mismatching
misnomer
misplace
misplaced
misplaces
misplacing
mispronunciation
misrepresentation
misrepresentations
miss
missed
misses
misshapen
missile
missiles
missing
mission
missionaries
missionary
missioner
missions
missive
misspell
misspelled
misspelling
misspellings
misspells
mist
mistakable
mistake
mistaken
mistakenly
mistakes
mistaking
misted
mister
misters
mistiness
misting
mistletoe
mistress
mistrust
mistrusted
mists
misty
mistype
mistyped
mistypes
mistyping
misunderstand
misunderstander
misunderstanders
misunderstanding
misunderstandings
misunderstood
misuse
misused
misuses
misusing
miter
mitigate
mitigated
mitigates
mitigating
mitigation
mitigative
mitten
mittens
mix
mixed
mixer
mixers
mixes
mixing
mixture
mixtures
mixup
mnemonic
mnemonically
mnemonics
moan
moaned
moans
moat
moats
mob
mobile
mobility
mobs
mobster
moccasin
moccasins
mock
mocked
mocker
mockery
mocking
mockingbird
mocks
mockup
modal
modalities
modality
modally
mode
model
modeled
modeling
modelings
models
modem
modems
moderate
moderated
moderately
moderateness
moderates
moderating
moderation
modern
modernity
modernize
modernized
modernizer
modernizing
modernly
modernness
moderns
modes
modest
modestly
modesty
modicum
modifiability
modifiable
modification
modifications
modified
modifier
modifiers
modifies
modify
modifying
modular
modularity
modularization
modularize
modularized
modularizes
modularizing
modularly
modulate
modulated
modulates
modulating
modulation
modulations
modulator
modulators
module
modules
moduli
modulo
modulus
modus
moist
moisten
moistly
moistness
moisture
molar
molasses
mold
molded
molder
molding
molds
mole
molecular
molecule
molecules
molehill
moles
molest
molested
molesting
molests
mollify
mollusk
mollycoddle
molten
moment
momentarily
momentariness
momentary
momentous
momentously
momentousness
moments
momentum
mommy
monadic
monarch
monarchies
monarchs
monarchy
monasteries
monastery
monastic
monetarism
monetary
money
moneyed
moneys
mongoose
monitor
monitored
monitoring
monitors
monk
monkey
monkeyed
monkeying
monkeys
monkish
monks
monoalphabetic
monochromatic
monochrome
monocotyledon
monocular
monogamous
monogamy
monogram
monograms
monograph
monographes
monographs
monolith
monolithic
monologue
monopolies
monopolize
monopolized
monopolizing
monopoly
monoprogrammed
monoprogramming
monostable
monotheism
monotone
monotonic
monotonically
monotonicity
monotonous
monotonously
monotonousness
monotony
monsoon
monster
monsters
monstrosity
monstrous
monstrously
month
monthly
months
monument
monumental
monumentally
monuments
moo
mood
moodiness
moods
moody
mooned
mooning
moonlight
moonlighter
moonlighting
moonlit
moons
moonshine
moored
mooring
moorings
moose
moot
mop
moped
mops
moraine
moral
morale
moralities
morality
morally
morals
morass
moratorium
morbid
morbidly
morbidness
more
moreover
mores
moribund
morn
morning
mornings
moron
morose
morphine
morphism
morphisms
morphological
morphology
morrow
morsel
morsels
mortal
mortality
mortally
mortals
mortar
mortared
mortaring
mortars
mortem
mortgage
mortgages
mortician
mortification
mortified
mortifies
mortify
mortifying
mosaic
mosaics
mosque
mosquito
mosquitoes
moss
mosses
mossy
most
mostly
motel
motels
moth
mothball
mothballs
mother
mothered
motherer
motherers
motherhood
mothering
motherland
motherly
mothers
motif
motifs
motion
motioned
motioning
motionless
motionlessly
motionlessness
motions
motivate
motivated
motivates
motivating
motivation
motivations
motive
motives
motley
motor
motorcar
motorcars
motorcycle
motorcycles
motoring
motorist
motorists
motorize
motorized
motorizes
motorizing
motors
motto
mottoes
mould
moulding
mound
mounded
mounds
mount
mountable
mountain
mountaineer
mountaineering
mountaineers
mountainous
mountainously
mountains
mounted
mounter
mounting
mountings
mounts
mourn
mourned
mourner
mourners
mournful
mournfully
mournfulness
mourning
mourns
mouse
mouser
mouses
mousetrap
mousy
mouth
mouthed
mouthes
mouthful
mouthing
mouthpiece
mouths
movable
move
moved
movement
movements
mover
movers
moves
movie
movies
moving
movings
mow
mowed
mower
mows
mu
much
muck
mucker
mucking
mucus
mud
muddied
muddiness
muddle
muddled
muddlehead
muddler
muddlers
muddles
muddling
muddy
muff
muffin
muffins
muffle
muffled
muffler
muffles
muffling
muffs
mug
mugging
mugs
mulatto
mulberries
mulberry
mule
mules
mull
mullah
multi
multibit
multibyte
multicast
multicasting
multicasts
multicellular
multicomputer
multidimensional
multilateral
multilayer
multilayered
multilevel
multimedia
multinational
multiple
multiples
multiplex
multiplexed
multiplexer
multiplexers
multiplexes
multiplexing
multiplexor
multiplexors
multiplicand
multiplicands
multiplication
multiplications
multiplicative
multiplicatives
multiplicity
multiplied
multiplier
multipliers
multiplies
multiply
multiplying
multiprocess
multiprocessing
multiprocessor
multiprocessors
multiprogram
multiprogrammed
multiprogramming
multistage
multitude
multitudes
multiuser
multivariate
multiword
mumble
mumbled
mumbler
mumblers
mumbles
mumbling
mumblings
mummies
mummy
munch
munched
munching
mundane
mundanely
mung
municipal
municipalities
municipality
municipally
munition
munitions
mural
murder
murdered
murderer
murderers
murdering
murderous
murderously
murders
murky
murmur
murmured
murmurer
murmuring
murmurs
muscle
muscled
muscles
muscling
muscular
musculature
muse
mused
muses
museum
museums
mush
mushroom
mushroomed
mushrooming
mushrooms
mushy
music
musical
musically
musicals
musician
musicianly
musicians
musicology
musing
musings
musk
musket
muskets
muskox
muskoxen
muskrat
muskrats
musks
muslin
mussel
mussels
must
mustache
mustached
mustaches
mustard
muster
mustiness
musts
musty
mutability
mutable
mutableness
mutandis
mutant
mutate
mutated
mutates
mutating
mutation
mutations
mutatis
mutative
mute
muted
mutely
muteness
mutilate
mutilated
mutilates
mutilating
mutilation
mutinies
mutiny
mutt
mutter
muttered
mutterer
mutterers
muttering
mutters
mutton
mutual
mutually
muzzle
muzzles
my
myriad
myrtle
myself
mysteries
mysterious
mysteriously
mysteriousness
mystery
mystic
mystical
mystics
mystify
myth
mythical
mythologies
mythology
nab
nabla
nablas
nadir
nag
nagged
nagging
nags
nail
nailed
nailing
nails
naive
naively
naiveness
naivete
naked
nakedly
nakedness
name
nameable
named
nameless
namelessly
namely
namer
namers
names
namesake
namesakes
naming
nanoinstruction
nanoinstructions
nanoprogram
nanoprogramming
nanosecond
nanoseconds
nanostore
nanostores
nap
napkin
napkins
naps
narcissus
narcotic
narcotics
narrate
narration
narrative
narratives
narrow
narrowed
narrower
narrowest
narrowing
narrowly
narrowness
narrows
nary
nasal
nasally
nastier
nastiest
nastily
nastiness
nasty
natal
nation
national
nationalist
nationalists
nationalities
nationality
nationalization
nationalize
nationalized
nationalizes
nationalizing
nationally
nationals
nationhood
nations
nationwide
native
natively
natives
nativity
natural
naturalism
naturalist
naturalization
naturally
naturalness
naturals
nature
natured
natures
naught
naughtier
naughtiness
naughty
nausea
nauseate
nauseum
naval
navally
navel
navies
navigable
navigate
navigated
navigates
navigating
navigation
navigator
navigators
navy
nay
near
nearby
neared
nearer
nearest
nearing
nearly
nearness
nears
nearsighted
neat
neater
neatest
neatly
neatness
nebula
nebular
nebulous
necessaries
necessarily
necessary
necessitate
necessitated
necessitates
necessitating
necessitation
necessities
necessity
neck
necking
necklace
necklaces
neckline
necks
necktie
neckties
necrosis
nectar
need
needed
needful
needing
needle
needled
needler
needlers
needles
needless
needlessly
needlessness
needlework
needling
needs
needy
negate
negated
negates
negating
negation
negations
negative
negatively
negatives
negator
negators
neglect
neglected
neglecting
neglects
negligee
negligence
negligent
negligible
negotiable
negotiate
negotiated
negotiates
negotiating
negotiation
negotiations
neigh
neighbor
neighborhood
neighborhoods
neighboring
neighborly
neighbors
neither
nemesis
neoclassic
neon
neonatal
neophyte
neophytes
nephew
nephews
nerve
nerves
nervous
nervously
nervousness
nest
nested
nester
nesting
nestle
nestled
nestles
nestling
nests
net
nether
nets
netted
netting
nettle
nettled
network
networked
networking
networks
neural
neuritis
neurological
neurologists
neuron
neurons
neuroses
neurosis
neurotic
neuter
neutral
neutralities
neutrality
neutralize
neutralized
neutralizing
neutrally
neutrino
neutrinos
neutron
never
nevertheless
new
newborn
newcomer
newcomers
newer
newest
newly
newlywed
newness
newscast
newsgroup
newsletter
newsletters
newsman
newsmen
newspaper
newspapers
newsstand
newt
next
nibble
nibbled
nibbler
nibblers
nibbles
nibbling
nice
nicely
niceness
nicer
nicest
niche
nick
nicked
nickel
nickels
nicker
nicking
nickname
nicknamed
nicknames
nicks
nicotine
niece
nieces
nifty
nigh
night
nightcap
nightclub
nightfall
nightgown
nightingale
nightingales
nightly
nightmare
nightmares
nightmarish
nights
nighttime
nihilism
nil
nimble
nimbleness
nimbler
nimbly
nimbus
nine
ninefold
nines
nineteen
nineteens
nineteenth
nineties
ninetieth
ninety
ninth
nip
nipple
nips
nitric
nitrogen
nitrous
nitty
no
nobility
noble
nobleman
nobleness
nobler
nobles
noblest
nobly
nobody
nocturnal
nocturnally
nod
nodal
nodded
nodding
node
nodes
nods
nodular
nodule
noise
noiseless
noiselessly
noises
noisier
noisily
noisiness
noisy
nomenclature
nominal
nominally
nominate
nominated
nominating
nomination
nominative
nominee
non
nonadaptive
nonbiodegradable
nonblocking
nonce
nonchalant
noncommercial
noncommunication
nonconsecutively
nonconservative
noncritical
noncyclic
nondecreasing
nondescript
nondescriptly
nondestructively
nondeterminacy
nondeterminate
nondeterminately
nondeterminism
nondeterministic
nondeterministically
none
nonempty
nonetheless
nonexistence
nonexistent
nonextensible
nonfunctional
nongovernmental
nonidempotent
noninteracting
noninterference
noninterleaved
nonintrusive
nonintuitive
noninverting
nonlinear
nonlinearities
nonlinearity
nonlinearly
nonlocal
nonmaskable
nonmathematical
nonmilitary
nonnegative
nonnegligible
nonnumerical
nonogenarian
nonorthogonal
nonorthogonality
nonperishable
nonpersistent
nonportable
nonprocedural
nonprocedurally
nonprofit
nonprogrammable
nonprogrammer
nonsegmented
nonsense
nonsensical
nonsequential
nonspecialist
nonspecialists
nonstandard
nonsynchronous
nontechnical
nonterminal
nonterminals
nonterminating
nontermination
nonthermal
nontransparent
nontrivial
nonuniform
nonuniformity
nonzero
noodle
nook
nooks
noon
noonday
noons
noontide
noontime
noose
nor
norm
normal
normalcy
normality
normalization
normalize
normalized
normalizes
normalizing
normally
normals
normative
norms
north
northbound
northeast
northeaster
northeastern
northerly
northern
northerner
northerners
northernly
northward
northwards
northwest
northwestern
nose
nosed
noses
nosing
nostalgia
nostalgic
nostril
nostrils
not
notable
notables
notably
notarize
notarized
notarizes
notarizing
notary
notation
notational
notations
notch
notched
notches
notching
note
notebook
notebooks
noted
notes
noteworthy
nothing
nothingness
nothings
notice
noticeable
noticeably
noticed
notices
noticing
notification
notifications
notified
notifier
notifiers
notifies
notify
notifying
noting
notion
notions
notoriety
notorious
notoriously
notwithstanding
noun
nouns
nourish
nourished
nourishes
nourishing
nourishment
novel
novelist
novelists
novels
novelties
novelty
novice
novices
now
nowadays
nowhere
noxious
nozzle
nu
nuance
nuances
nubile
nuclear
nuclei
nucleic
nucleotide
nucleotides
nucleus
nuclide
nude
nudge
nudged
nudity
nugget
nuisance
nuisances
null
nullary
nulled
nullified
nullifiers
nullifies
nullify
nullifying
nulls
numb
numbed
number
numbered
numberer
numbering
numberless
numbers
numbing
numbly
numbness
numbs
numerable
numeral
numerals
numerator
numerators
numeric
numerical
numerically
numerics
numerous
numismatic
numismatist
nun
nuns
nuptial
nurse
nursed
nurseries
nursery
nurses
nursing
nurture
nurtured
nurtures
nurturing
nut
nutate
nutria
nutrient
nutrition
nutritious
nuts
nutshell
nutshells
nuzzle
nylon
nymph
nymphomania
nymphomaniac
nymphs
oaf
oak
oaken
oaks
oar
oars
oases
oasis
oat
oaten
oath
oaths
oatmeal
oats
obedience
obediences
obedient
obediently
obelisk
obese
obey
obeyed
obeying
obeys
obfuscate
obfuscatory
obituary
object
objected
objecting
objection
objectionable
objections
objective
objectively
objectives
objector
objectors
objects
obligated
obligation
obligations
obligatory
oblige
obliged
obliges
obliging
obligingly
oblique
obliquely
obliqueness
obliterate
obliterated
obliterates
obliterating
obliteration
oblivion
oblivious
obliviously
obliviousness
oblong
obnoxious
oboe
obscene
obscure
obscured
obscurely
obscurer
obscures
obscuring
obscurities
obscurity
obsequious
observable
observance
observances
observant
observation
observations
observatory
observe
observed
observer
observers
observes
observing
obsession
obsessions
obsessive
obsolescence
obsolescent
obsolete
obsoleted
obsoletes
obsoleting
obstacle
obstacles
obstinacy
obstinate
obstinately
obstruct
obstructed
obstructing
obstruction
obstructions
obstructive
obtain
obtainable
obtainably
obtained
obtaining
obtains
obviate
obviated
obviates
obviating
obviation
obviations
obvious
obviously
obviousness
occasion
occasional
occasionally
occasioned
occasioning
occasionings
occasions
occipital
occlude
occluded
occludes
occlusion
occlusions
occult
occupancies
occupancy
occupant
occupants
occupation
occupational
occupationally
occupations
occupied
occupier
occupies
occupy
occupying
occur
occurred
occurrence
occurrences
occurring
occurs
ocean
oceanic
oceanography
oceans
octagon
octagonal
octahedra
octahedral
octahedron
octal
octane
octave
octaves
octet
octets
octogenarian
octopus
odd
odder
oddest
oddities
oddity
oddly
oddness
odds
ode
odes
odious
odiously
odiousness
odium
odor
odorous
odorously
odorousness
odors
of
off
offend
offended
offender
offenders
offending
offends
offense
offenses
offensive
offensively
offensiveness
offer
offered
offerer
offerers
offering
offerings
offers
offhand
office
officemate
officer
officers
offices
official
officialdom
officially
officials
officiate
officio
officious
officiously
officiousness
offing
offload
offs
offset
offsets
offsetting
offshore
offspring
oft
often
oftentimes
oh
ohm
ohmmeter
oil
oilcloth
oiled
oiler
oilers
oilier
oiliest
oiling
oils
oily
ointment
okay
old
olden
older
oldest
oldness
oldy
oleander
oleomargarine
oligarchy
olive
olives
omega
omelet
omen
omens
omicron
ominous
ominously
ominousness
omission
omissions
omit
omits
omitted
omitting
omnibus
omnidirectional
omnipotent
omnipresent
omniscient
omnisciently
omnivore
on
onanism
once
oncology
one
oneness
onerous
ones
oneself
onetime
ongoing
onion
onions
online
onlooker
only
onrush
onset
onsets
onslaught
onto
ontology
onus
onward
onwards
onyx
ooze
oozed
opacity
opal
opals
opaque
opaquely
opaqueness
opcode
open
opened
opener
openers
opening
openings
openly
openness
opens
opera
operable
operand
operandi
operands
operas
operate
operated
operates
operating
operation
operational
operationally
operations
operative
operatives
operator
operators
operetta
opiate
opinion
opinions
opium
opossum
opponent
opponents
opportune
opportunely
opportunism
opportunistic
opportunities
opportunity
opposable
oppose
opposed
opposes
opposing
opposite
oppositely
oppositeness
opposites
opposition
oppress
oppressed
oppresses
oppressing
oppression
oppressive
oppressor
oppressors
opprobrium
opt
opted
opthalmic
optic
optical
optically
optics
optima
optimal
optimality
optimally
optimism
optimist
optimistic
optimistically
optimization
optimizations
optimize
optimized
optimizer
optimizers
optimizes
optimizing
optimum
opting
option
optional
optionally
options
optoacoustic
optometrist
optometry
opts
opulence
opulent
opus
or
oracle
oracles
oral
orally
orange
oranges
orangutan
oration
orations
orator
oratories
orators
oratory
orb
orbit
orbital
orbitally
orbited
orbiter
orbiters
orbiting
orbits
orchard
orchards
orchestra
orchestral
orchestras
orchestrate
orchid
orchids
ordain
ordained
ordaining
ordains
ordeal
order
ordered
ordering
orderings
orderlies
orderly
orders
ordinal
ordinance
ordinances
ordinarily
ordinariness
ordinary
ordinate
ordinates
ordination
ore
oregano
ores
organ
organic
organism
organisms
organist
organists
organizable
organization
organizational
organizationally
organizations
organize
organized
organizer
organizers
organizes
organizing
organs
orgasm
orgiastic
orgies
orgy
orientation
orientations
oriented
orienting
orients
orifice
orifices
origin
original
originality
originally
originals
originate
originated
originates
originating
origination
originator
originators
origins
oriole
ornament
ornamental
ornamentally
ornamentation
ornamented
ornamenting
ornaments
ornate
ornery
orphan
orphanage
orphaned
orphans
orthant
orthodontist
orthodox
orthodoxy
orthogonal
orthogonality
orthogonally
orthopedic
oscillate
oscillated
oscillates
oscillating
oscillation
oscillations
oscillator
oscillators
oscillatory
oscilloscope
oscilloscopes
osmosis
osmotic
ossify
ostensible
ostensibly
ostentatious
osteopath
osteopathic
osteopathy
osteoporosis
ostracism
ostrich
ostriches
other
others
otherwise
otherworldly
otter
otters
ouch
ought
ounce
ounces
our
ours
ourself
ourselves
oust
out
outbound
outbreak
outbreaks
outburst
outbursts
outcast
outcasts
outcome
outcomes
outcries
outcry
outdated
outdo
outdoor
outdoors
outer
outermost
outfit
outfits
outfitted
outgoing
outgrew
outgrow
outgrowing
outgrown
outgrows
outgrowth
outing
outlandish
outlast
outlasts
outlaw
outlawed
outlawing
outlaws
outlay
outlays
outlet
outlets
outline
outlined
outlines
outlining
outlive
outlived
outlives
outliving
outlook
outlying
outnumbered
outperform
outperformed
outperforming
outperforms
outpost
outposts
output
outputs
outputting
outrage
outraged
outrageous
outrageously
outrages
outright
outrun
outruns
outs
outset
outside
outsider
outsiders
outskirts
outstanding
outstandingly
outstretched
outstrip
outstripped
outstripping
outstrips
outvote
outvoted
outvotes
outvoting
outward
outwardly
outweigh
outweighed
outweighing
outweighs
outwit
outwits
outwitted
outwitting
oval
ovals
ovaries
ovary
oven
ovens
over
overall
overalls
overboard
overcame
overcoat
overcoats
overcome
overcomes
overcoming
overcrowd
overcrowded
overcrowding
overcrowds
overdone
overdose
overdraft
overdrafts
overdue
overemphasis
overemphasized
overestimate
overestimated
overestimates
overestimating
overestimation
overflow
overflowed
overflowing
overflows
overgrown
overhang
overhanging
overhangs
overhaul
overhauling
overhead
overheads
overhear
overheard
overhearing
overhears
overjoy
overjoyed
overkill
overland
overlap
overlapped
overlapping
overlaps
overlay
overlaying
overlays
overload
overloaded
overloading
overloads
overlook
overlooked
overlooking
overlooks
overly
overnight
overnighter
overnighters
overpower
overpowered
overpowering
overpowers
overprint
overprinted
overprinting
overprints
overproduction
overridden
override
overrides
overriding
overrode
overrule
overruled
overrules
overrun
overrunning
overruns
overseas
oversee
overseeing
overseer
overseers
oversees
overshadow
overshadowed
overshadowing
overshadows
overshoot
overshot
oversight
oversights
oversimplified
oversimplifies
oversimplify
oversimplifying
oversized
overstate
overstated
overstatement
overstatements
overstates
overstating
overstocks
oversubscribed
overt
overtake
overtaken
overtaker
overtakers
overtakes
overtaking
overthrew
overthrow
overthrown
overtime
overtly
overtone
overtones
overtook
overture
overtures
overturn
overturned
overturning
overturns
overuse
overview
overviews
overwhelm
overwhelmed
overwhelming
overwhelmingly
overwhelms
overwork
overworked
overworking
overworks
overwrite
overwrites
overwriting
overwritten
overzealous
owe
owed
owes
owing
owl
owls
own
owned
owner
owners
ownership
ownerships
owning
owns
ox
oxen
oxide
oxides
oxidize
oxidized
oxygen
oyster
oysters
ozone
pace
paced
pacemaker
pacer
pacers
paces
pacific
pacification
pacified
pacifier
pacifies
pacifism
pacifist
pacify
pacing
pack
package
packaged
packager
packagers
packages
packaging
packagings
packed
packer
packers
packet
packets
packing
packs
pact
pacts
pad
padded
padding
paddle
paddock
paddy
padlock
pads
pagan
pagans
page
pageant
pageantry
pageants
paged
pager
pagers
pages
paginate
paginated
paginates
paginating
pagination
paging
pagoda
paid
pail
pails
pain
pained
painful
painfully
painless
pains
painstaking
painstakingly
paint
painted
painter
painters
painting
paintings
paints
pair
paired
pairing
pairings
pairs
pairwise
pajama
pajamas
pal
palace
palaces
palate
palates
pale
paled
palely
paleness
paler
pales
palest
palfrey
palindrome
palindromic
paling
pall
palladium
palliate
palliative
pallid
palm
palmed
palmer
palming
palms
palpable
pals
palsy
pamper
pamphlet
pamphlets
pan
panacea
panaceas
panama
pancake
pancakes
panda
pandas
pandemic
pandemonium
pander
pane
panel
paneled
paneling
panelist
panelists
panels
panes
pang
pangs
panic
panicked
panicking
panicky
panics
panned
panning
panorama
panoramic
pans
pansies
pansy
pant
panted
pantheism
pantheist
pantheon
panther
panthers
panties
panting
pantomime
pantries
pantry
pants
panty
pantyhose
papa
papal
paper
paperback
paperbacks
papered
paperer
paperers
papering
paperings
papers
paperweight
paperwork
papoose
papyrus
par
parabola
parabolic
paraboloid
paraboloidal
parachute
parachuted
parachutes
parade
paraded
parades
paradigm
paradigms
parading
paradise
paradox
paradoxes
paradoxical
paradoxically
paraffin
paragon
paragons
paragraph
paragraphing
paragraphs
parakeet
parallax
parallel
paralleled
paralleling
parallelism
parallelize
parallelized
parallelizes
parallelizing
parallelogram
parallelograms
parallels
paralysis
paralyze
paralyzed
paralyzes
paralyzing
parameter
parameterizable
parameterization
parameterizations
parameterize
parameterized
parameterizes
parameterizing
parameterless
parameters
parametric
parametrized
paramilitary
paramount
paranoia
paranoiac
paranoid
paranormal
parapet
parapets
paraphernalia
paraphrase
paraphrased
paraphrases
paraphrasing
parapsychology
parasite
parasites
parasitic
parasitics
parasol
parboil
parcel
parceled
parceling
parcels
parch
parched
parchment
pardon
pardonable
pardonably
pardoned
pardoner
pardoners
pardoning
pardons
pare
paregoric
parent
parentage
parental
parentheses
parenthesis
parenthesized
parenthesizes
parenthesizing
parenthetic
parenthetical
parenthetically
parenthood
parents
pares
pariah
parimutuel
paring
parings
parish
parishes
parishioner
parity
park
parked
parker
parkers
parking
parkland
parklike
parkway
parlay
parley
parliament
parliamentarian
parliamentary
parliaments
parlor
parlors
parochial
parody
parole
paroled
paroles
paroling
parried
parrot
parroting
parrots
parry
pars
parse
parsed
parser
parsers
parses
parsimony
parsing
parsings
parsley
parson
part
partake
partaker
partakes
partaking
parted
parter
parters
partial
partiality
partially
participant
participants
participate
participated
participates
participating
participation
participle
particle
particles
particular
particularly
particulars
particulate
parties
parting
partings
partisan
partisans
partition
partitioned
partitioning
partitions
partly
partner
partnered
partners
partnership
partook
partridge
partridges
parts
party
pass
passage
passages
passageway
passe
passed
passenger
passengers
passer
passers
passes
passing
passion
passionate
passionately
passions
passivate
passive
passively
passiveness
passivity
passport
passports
password
passwords
past
paste
pasted
pastel
pastes
pastime
pastimes
pasting
pastness
pastor
pastoral
pastors
pastry
pasts
pasture
pastures
pat
patch
patched
patches
patching
patchwork
patchy
pate
paten
patent
patentable
patented
patenter
patenters
patenting
patently
patents
paternal
paternally
paternoster
path
pathetic
pathname
pathnames
pathogen
pathogenesis
pathological
pathology
pathos
paths
pathway
pathways
patience
patient
patiently
patients
patina
patio
patriarch
patriarchal
patriarchs
patriarchy
patrician
patricians
patrimonial
patrimony
patriot
patriotic
patriotism
patriots
patrol
patrolled
patrolling
patrolman
patrolmen
patrols
patron
patronage
patronize
patronized
patronizes
patronizing
patrons
pats
patter
pattered
pattering
patterings
pattern
patterned
patterning
patterns
patters
patties
patty
paucity
paunch
paunchy
pauper
pause
paused
pauses
pausing
pave
paved
pavement
pavements
paves
pavilion
pavilions
paving
paw
pawing
pawn
pawns
pawnshop
paws
pay
payable
paycheck
paychecks
payed
payer
payers
paying
payment
payments
payoff
payoffs
payroll
pays
pea
peace
peaceable
peaceful
peacefully
peacefulness
peacetime
peach
peaches
peacock
peacocks
peak
peaked
peaks
peal
pealed
pealing
peals
peanut
peanuts
pear
pearl
pearls
pearly
pears
peas
peasant
peasantry
peasants
peat
pebble
pebbles
peccary
peck
pecked
pecking
pecks
pectoral
peculiar
peculiarities
peculiarity
peculiarly
pecuniary
pedagogic
pedagogical
pedagogically
pedagogy
pedal
pedant
pedantic
pedantry
peddle
peddler
peddlers
pedestal
pedestrian
pedestrians
pediatric
pediatrician
pediatrics
pedigree
peek
peeked
peeking
peeks
peel
peeled
peeling
peels
peep
peeped
peeper
peephole
peeping
peeps
peer
peered
peering
peerless
peers
peg
pegboard
pegs
pejorative
pelican
pellagra
pelt
pelting
pelts
pelvic
pelvis
pen
penal
penalize
penalized
penalizes
penalizing
penalties
penalty
penance
pence
penchant
pencil
penciled
pencils
pend
pendant
pended
pending
pends
pendulum
pendulums
penetrable
penetrate
penetrated
penetrates
penetrating
penetratingly
penetration
penetrations
penetrative
penetrator
penetrators
penguin
penguins
penicillin
peninsula
peninsulas
penis
penises
penitent
penitentiary
penned
pennies
penniless
penning
penny
pens
pension
pensioner
pensions
pensive
pent
pentagon
pentagons
pentecostal
penthouse
penultimate
penumbra
peony
people
peopled
peoples
pep
pepper
peppered
peppering
peppermint
pepperoni
peppers
peppery
peppy
peptide
per
perceivable
perceivably
perceive
perceived
perceiver
perceivers
perceives
perceiving
percent
percentage
percentages
percentile
percentiles
percents
perceptible
perceptibly
perception
perceptions
perceptive
perceptively
perceptual
perceptually
perch
perchance
perched
perches
perching
percussion
percutaneous
peremptory
perennial
perennially
perfect
perfected
perfectible
perfecting
perfection
perfectionist
perfectionists
perfectly
perfectness
perfects
perforce
perform
performance
performances
performed
performer
performers
performing
performs
perfume
perfumed
perfumes
perfuming
perfunctory
perhaps
perihelion
peril
perilous
perilously
perils
perimeter
period
periodic
periodical
periodically
periodicals
periods
peripheral
peripherally
peripherals
peripheries
periphery
periscope
perish
perishable
perishables
perished
perisher
perishers
perishes
perishing
perjure
perjury
perk
perky
permanence
permanent
permanently
permeable
permeate
permeated
permeates
permeating
permeation
permissibility
permissible
permissibly
permission
permissions
permissive
permissively
permit
permits
permitted
permitting
permutation
permutations
permute
permuted
permutes
permuting
pernicious
peroxide
perpendicular
perpendicularly
perpendiculars
perpetrate
perpetrated
perpetrates
perpetrating
perpetration
perpetrations
perpetrator
perpetrators
perpetual
perpetually
perpetuate
perpetuated
perpetuates
perpetuating
perpetuation
perpetuity
perplex
perplexed
perplexing
perplexity
persecute
persecuted
persecutes
persecuting
persecution
persecutor
persecutors
perseverance
persevere
persevered
perseveres
persevering
persist
persisted
persistence
persistent
persistently
persisting
persists
person
personage
personages
personal
personalities
personality
personalization
personalize
personalized
personalizes
personalizing
personally
personification
personified
personifies
personify
personifying
personnel
persons
perspective
perspectives
perspicuous
perspicuously
perspiration
perspire
persuadable
persuade
persuaded
persuader
persuaders
persuades
persuading
persuasion
persuasions
persuasive
persuasively
persuasiveness
pertain
pertained
pertaining
pertains
pertinent
perturb
perturbation
perturbations
perturbed
perusal
peruse
perused
peruser
perusers
peruses
perusing
pervade
pervaded
pervades
pervading
pervasive
pervasively
perversion
pervert
perverted
perverts
pessimism
pessimist
pessimistic
pest
pester
pesticide
pestilence
pestilent
pests
pet
petal
petals
petition
petitioned
petitioner
petitioning
petitions
petri
petroleum
pets
petted
petter
petters
petticoat
petticoats
pettiness
petting
petty
petulance
petulant
pew
pews
pewter
phantom
phantoms
pharmaceutic
pharmacist
pharmacology
pharmacopoeia
pharmacy
phase
phased
phaser
phasers
phases
phasing
pheasant
pheasants
phenomena
phenomenal
phenomenally
phenomenological
phenomenologically
phenomenologies
phenomenology
phenomenon
phi
philanthropy
philharmonic
philosopher
philosophers
philosophic
philosophical
philosophically
philosophies
philosophize
philosophized
philosophizer
philosophizers
philosophizes
philosophizing
philosophy
phoenix
phone
phoned
phoneme
phonemes
phonemic
phones
phonetic
phonetics
phoning
phonograph
phonographs
phony
phosgene
phosphate
phosphates
phosphor
phosphorescent
phosphoric
phosphorus
photo
photocopied
photocopier
photocopiers
photocopies
photocopy
photocopying
photodiode
photodiodes
photogenic
photograph
photographed
photographer
photographers
photographic
photographing
photographs
photography
photon
photos
photosensitive
phototypesetter
phototypesetters
phrase
phrased
phraseology
phrases
phrasing
phrasings
phyla
phylum
physic
physical
physically
physicalness
physicals
physician
physicians
physicist
physicists
physics
physiological
physiologically
physiology
physiotherapist
physiotherapy
physique
phytoplankton
pi
pianist
piano
pianos
pica
picas
picayune
piccolo
pick
pickaxe
picked
picker
pickers
picket
picketed
picketer
picketers
picketing
pickets
picking
pickings
pickle
pickled
pickles
pickling
picks
pickup
pickups
picky
picnic
picnicked
picnicking
picnics
picofarad
picojoule
picosecond
pictorial
pictorially
picture
pictured
pictures
picturesque
picturesqueness
picturing
piddle
pidgin
pie
piece
pieced
piecemeal
pieces
piecewise
piecing
pier
pierce
pierced
pierces
piercing
piers
pies
piety
piezoelectric
pig
pigeon
pigeonhole
pigeons
piggish
piggy
piggyback
piggybacked
piggybacking
piggybacks
pigment
pigmentation
pigmented
pigments
pigpen
pigs
pigskin
pigtail
pike
piker
pikes
pile
piled
pilers
piles
pilfer
pilferage
pilgrim
pilgrimage
pilgrimages
pilgrims
piling
pilings
pill
pillage
pillaged
pillar
pillared
pillars
pillory
pillow
pillows
pills
pilot
piloting
pilots
pimp
pimple
pin
pinafore
pinball
pinch
pinched
pinches
pinching
pincushion
pine
pineapple
pineapples
pined
pines
ping
pinhead
pinhole
pining
pinion
pink
pinker
pinkest
pinkie
pinkish
pinkly
pinkness
pinks
pinnacle
pinnacles
pinned
pinning
pinnings
pinochle
pinpoint
pinpointing
pinpoints
pins
pinscher
pint
pinto
pints
pinwheel
pion
pioneer
pioneered
pioneering
pioneers
pious
piously
pip
pipe
piped
pipeline
pipelined
pipelines
pipelining
pipers
pipes
pipette
piping
pique
piracy
pirate
pirates
piss
pistachio
pistil
pistils
pistol
pistols
piston
pistons
pit
pitch
pitched
pitcher
pitchers
pitches
pitchfork
pitching
piteous
piteously
pitfall
pitfalls
pith
pithed
pithes
pithier
pithiest
pithiness
pithing
pithy
pitiable
pitied
pitier
pitiers
pities
pitiful
pitifully
pitiless
pitilessly
pits
pitted
pituitary
pity
pitying
pityingly
pivot
pivotal
pivoting
pivots
pixel
pixels
pizza
placard
placards
placate
place
placebo
placed
placeholder
placement
placements
placenta
placental
placer
places
placid
placidly
placing
plagiarism
plagiarist
plague
plagued
plagues
plaguing
plaid
plaids
plain
plainer
plainest
plainly
plainness
plains
plaintext
plaintexts
plaintiff
plaintiffs
plaintive
plaintively
plaintiveness
plait
plaits
plan
planar
planarity
plane
planed
planeload
planer
planers
planes
planet
planetaria
planetarium
planetary
planetesimal
planetoid
planets
planing
plank
planking
planks
plankton
planned
planner
planners
planning
planoconcave
planoconvex
plans
plant
plantation
plantations
planted
planter
planters
planting
plantings
plants
plaque
plasma
plaster
plastered
plasterer
plastering
plasters
plastic
plasticity
plastics
plate
plateau
plateaus
plated
platelet
platelets
platen
platens
plates
platform
platforms
plating
platinum
platitude
platonic
platoon
platter
platters
plausibility
plausible
play
playable
playback
playboy
played
player
players
playful
playfully
playfulness
playground
playgrounds
playhouse
playing
playmate
playmates
playoff
playroom
plays
plaything
playthings
playtime
playwright
playwrights
playwriting
plaza
plea
plead
pleaded
pleader
pleading
pleads
pleas
pleasant
pleasantly
pleasantness
please
pleased
pleases
pleasing
pleasingly
pleasure
pleasures
pleat
plebeian
plebian
plebiscite
plebiscites
pledge
pledged
pledges
plenary
plenipotentiary
plenteous
plentiful
plentifully
plenty
plethora
pleurisy
pliable
pliant
plied
pliers
plies
plight
plod
plodding
plot
plots
plotted
plotter
plotters
plotting
plow
plowed
plower
plowing
plowman
plows
plowshare
ploy
ploys
pluck
plucked
plucking
plucks
plucky
plug
pluggable
plugged
plugging
plugs
plum
plumage
plumb
plumbed
plumbing
plumbs
plume
plumed
plumes
plummet
plummeting
plump
plumped
plumpness
plums
plunder
plundered
plunderer
plunderers
plundering
plunders
plunge
plunged
plunger
plungers
plunges
plunging
plunk
plural
plurality
plurals
plus
pluses
plush
plutonium
ply
plywood
pneumatic
pneumonia
poach
poacher
poaches
pocket
pocketbook
pocketbooks
pocketed
pocketful
pocketing
pockets
pod
podia
podium
pods
poem
poems
poet
poetic
poetical
poetically
poetics
poetries
poetry
poets
pogo
pogrom
poignancy
poignant
point
pointed
pointedly
pointer
pointers
pointing
pointless
points
pointy
poise
poised
poises
poison
poisoned
poisoner
poisoning
poisonous
poisonousness
poisons
poke
poked
poker
pokerface
pokes
poking
polar
polarities
polarity
pole
polecat
poled
polemic
polemics
poles
police
policed
policeman
policemen
polices
policies
policing
policy
poling
polio
polish
polished
polisher
polishers
polishes
polishing
polite
politely
politeness
politer
politest
politic
political
politically
politician
politicians
politicking
politics
polka
poll
polled
pollen
polling
polloi
polls
pollutant
pollute
polluted
pollutes
polluting
pollution
polo
polyalphabetic
polygon
polygons
polymer
polymers
polymorphic
polynomial
polynomials
polytechnic
polytheist
pomp
pompadour
pomposity
pompous
pompously
pompousness
poncho
pond
ponder
pondered
pondering
ponderous
ponders
ponds
pong
ponies
pontiff
pontific
pontificate
pony
pooch
poodle
pool
pooled
pooling
pools
poor
poorer
poorest
poorly
poorness
pop
popcorn
popish
poplar
poplin
popped
poppies
popping
poppy
pops
populace
popular
popularity
popularization
popularize
popularized
popularizes
popularizing
popularly
populate
populated
populates
populating
population
populations
populous
populousness
porcelain
porch
porches
porcine
porcupine
porcupines
pore
pored
pores
poring
pork
porker
pornographer
pornographic
pornography
porous
porpoise
porridge
port
portability
portable
portage
portal
portals
ported
portend
portended
portending
portends
portent
portentous
porter
porterhouse
porters
portfolio
portfolios
portico
porting
portion
portions
portly
portmanteau
portrait
portraits
portray
portrayal
portrayed
portraying
portrays
ports
pose
posed
poser
posers
poses
posh
posing
posit
posited
positing
position
positional
positioned
positioning
positions
positive
positively
positiveness
positives
positron
posits
posse
possess
possessed
possesses
possessing
possession
possessional
possessions
possessive
possessively
possessiveness
possessor
possessors
possibilities
possibility
possible
possibly
possum
possums
post
postage
postal
postcard
postcondition
postdoctoral
posted
poster
posterior
posteriori
posterity
posters
postfix
postgraduate
posting
postlude
postman
postmark
postmaster
postmasters
postmortem
postoperative
postorder
postpone
postponed
postponing
postprocess
postprocessor
posts
postscript
postscripts
postulate
postulated
postulates
postulating
postulation
postulations
posture
postures
pot
potable
potash
potassium
potato
potatoes
potbelly
potent
potentate
potentates
potential
potentialities
potentiality
potentially
potentials
potentiating
potentiometer
potentiometers
pothole
potion
potlatch
potpourri
pots
potted
potter
potters
pottery
potting
pouch
pouches
poultice
poultry
pounce
pounced
pounces
pouncing
pound
pounded
pounder
pounders
pounding
pounds
pour
poured
pourer
pourers
pouring
pours
pout
pouted
pouting
pouts
poverty
powder
powdered
powdering
powderpuff
powders
powdery
power
powered
powerful
powerfully
powerfulness
powering
powerless
powerlessly
powerlessness
pox
practicable
practicably
practical
practicality
practically
practice
practiced
practices
practicing
practitioner
practitioners
pragmatic
pragmatically
pragmatics
pragmatism
pragmatist
prairie
praise
praised
praiser
praisers
praises
praiseworthy
praising
praisingly
prance
pranced
prancer
prancing
prank
pranks
prate
pray
prayed
prayer
prayers
praying
preach
preached
preacher
preachers
preaches
preaching
preallocate
preallocated
preallocating
preamble
preambles
preassign
preassigned
preassigning
preassigns
precarious
precariously
precariousness
precaution
precautions
precede
preceded
precedence
precedences
precedent
precedented
precedents
precedes
preceding
precept
precepts
precess
precession
precinct
precincts
precious
preciously
preciousness
precipice
precipitable
precipitate
precipitated
precipitately
precipitateness
precipitates
precipitating
precipitation
precipitous
precipitously
precise
precisely
preciseness
precision
precisions
preclude
precluded
precludes
precluding
precocious
precociously
precocity
precompute
precomputed
precomputing
preconceive
preconceived
preconception
preconceptions
precondition
preconditioned
preconditions
precursor
precursors
predate
predated
predates
predating
predatory
predecessor
predecessors
predefine
predefined
predefines
predefining
predefinition
predefinitions
predetermination
predetermine
predetermined
predetermines
predetermining
predicament
predicate
predicated
predicates
predicating
predication
predications
predict
predictability
predictable
predictably
predicted
predicting
prediction
predictions
predictive
predictor
predicts
predilection
predilections
predisposition
predominant
predominantly
predominate
predominated
predominately
predominates
predominating
predomination
preeminence
preeminent
preempt
preempted
preempting
preemption
preemptive
preemptor
preempts
preen
preexisting
prefab
prefabricate
preface
prefaced
prefaces
prefacing
prefer
preferable
preferably
preference
preferences
preferential
preferentially
preferred
preferring
prefers
prefix
prefixed
prefixes
prefixing
pregnancy
pregnant
prehistoric
preinitialize
preinitialized
preinitializes
preinitializing
prejudge
prejudged
prejudice
prejudiced
prejudices
prejudicial
prelate
preliminaries
preliminary
prelude
preludes
premature
prematurely
prematurity
premeditated
premeditation
premier
premiers
premise
premises
premium
premiums
premonition
prenatal
preoccupation
preoccupied
preoccupies
preoccupy
prep
preparation
preparations
preparative
preparatives
preparatory
prepare
prepared
prepares
preparing
prepend
prepended
prepending
preposition
prepositional
prepositions
preposterous
preposterously
preprocessed
preprocessing
preprocessor
preprocessors
preproduction
preprogrammed
prerequisite
prerequisites
prerogative
prerogatives
prescribe
prescribed
prescribes
prescription
prescriptions
prescriptive
preselect
preselected
preselecting
preselects
presence
presences
present
presentation
presentations
presented
presenter
presenting
presently
presentness
presents
preservation
preservations
preserve
preserved
preserver
preservers
preserves
preserving
preset
preside
presided
presidency
president
presidential
presidents
presides
presiding
press
pressed
presser
presses
pressing
pressings
pressure
pressured
pressures
pressuring
pressurize
pressurized
prestidigitate
prestige
prestigious
presumably
presume
presumed
presumes
presuming
presumption
presumptions
presumptive
presumptuous
presumptuousness
presuppose
presupposed
presupposes
presupposing
presupposition
pretend
pretended
pretender
pretenders
pretending
pretends
pretense
pretenses
pretension
pretensions
pretentious
pretentiously
pretentiousness
pretext
pretexts
prettier
prettiest
prettily
prettiness
pretty
prevail
prevailed
prevailing
prevailingly
prevails
prevalence
prevalent
prevalently
prevent
preventable
preventably
prevented
preventing
prevention
preventive
preventives
prevents
preview
previewed
previewing
previews
previous
previously
prey
preyed
preying
preys
price
priced
priceless
pricer
pricers
prices
pricing
prick
pricked
pricking
prickly
pricks
pride
prided
prides
priding
priest
priggish
prim
prima
primacy
primal
primaries
primarily
primary
primate
prime
primed
primeness
primer
primers
primes
primeval
priming
primitive
primitively
primitiveness
primitives
primrose
prince
princely
princes
princess
princesses
principal
principalities
principality
principally
principals
principle
principled
principles
print
printable
printably
printed
printer
printers
printing
printout
prints
prior
priori
priorities
priority
priory
prism
prisms
prison
prisoner
prisoners
prisons
pristine
privacies
privacy
private
privately
privates
privation
privations
privies
privilege
privileged
privileges
privy
prize
prized
prizer
prizers
prizes
prizewinning
prizing
pro
probabilistic
probabilistically
probabilities
probability
probable
probably
probate
probated
probates
probating
probation
probative
probe
probed
probes
probing
probings
probity
problem
problematic
problematical
problematically
problems
procaine
procedural
procedurally
procedure
procedures
proceed
proceeded
proceeding
proceedings
proceeds
process
processed
processes
processing
procession
processor
processors
proclaim
proclaimed
proclaimer
proclaimers
proclaiming
proclaims
proclamation
proclamations
proclivities
proclivity
procotols
procrastinate
procrastinated
procrastinates
procrastinating
procrastination
procreate
procure
procured
procurement
procurements
procurer
procurers
procures
procuring
prod
prodigal
prodigally
prodigious
prodigy
produce
produced
producer
producers
produces
producible
producing
product
production
productions
productive
productively
productivity
products
profane
profanely
profess
professed
professes
professing
profession
professional
professionalism
professionally
professionals
professions
professor
professorial
professors
proffer
proffered
proffers
proficiency
proficient
proficiently
profile
profiled
profiles
profiling
profit
profitability
profitable
profitably
profited
profiteer
profiteers
profiting
profits
profitted
profligate
profound
profoundest
profoundly
profundity
profuse
profusion
progenitor
progeny
prognosis
prognosticate
program
programmability
programmable
programmed
programmer
programmers
programming
programs
progress
progressed
progresses
progressing
progression
progressions
progressive
progressively
prohibit
prohibited
prohibiting
prohibition
prohibitions
prohibitive
prohibitively
prohibitory
prohibits
project
projected
projectile
projecting
projection
projections
projective
projectively
projector
projectors
projects
prolate
prolegomena
proletariat
proliferate
proliferated
proliferates
proliferating
proliferation
prolific
prolix
prolog
prologue
prolong
prolongate
prolonged
prolonging
prolongs
promenade
promenades
prominence
prominent
prominently
promiscuous
promise
promised
promises
promising
promontory
promote
promoted
promoter
promoters
promotes
promoting
promotion
promotional
promotions
prompt
prompted
prompter
promptest
prompting
promptings
promptly
promptness
prompts
promulgate
promulgated
promulgates
promulgating
promulgation
prone
proneness
prong
pronged
prongs
pronoun
pronounce
pronounceable
pronounced
pronouncement
pronouncements
pronounces
pronouncing
pronouns
pronunciation
pronunciations
proof
proofread
proofreader
proofs
prop
propaganda
propagandist
propagate
propagated
propagates
propagating
propagation
propagations
propane
propel
propellant
propelled
propeller
propellers
propelling
propels
propensity
proper
properly
properness
propertied
properties
property
prophecies
prophecy
prophesied
prophesier
prophesies
prophesy
prophet
prophetic
prophets
propitious
proponent
proponents
proportion
proportional
proportionally
proportionately
proportioned
proportioning
proportionment
proportions
propos
proposal
proposals
propose
proposed
proposer
proposes
proposing
proposition
propositional
propositionally
propositioned
propositioning
propositions
propound
propounded
propounding
propounds
proprietary
proprietor
proprietors
propriety
props
propulsion
propulsions
prorate
prorated
prorates
pros
proscenium
proscribe
proscription
prose
prosecute
prosecuted
prosecutes
prosecuting
prosecution
prosecutions
prosecutor
proselytize
proselytized
proselytizes
proselytizing
prosodic
prosodics
prospect
prospected
prospecting
prospection
prospections
prospective
prospectively
prospectives
prospector
prospectors
prospects
prospectus
prosper
prospered
prospering
prosperity
prosperous
prospers
prostate
prosthetic
prostitute
prostitution
prostrate
prostration
protagonist
protean
protect
protected
protecting
protection
protections
protective
protectively
protectiveness
protector
protectorate
protectors
protects
protege
proteges
protein
proteins
protest
protestant
protestation
protestations
protested
protesting
protestingly
protestor
protests
protocol
protocols
proton
protons
protoplasm
prototype
prototyped
prototypes
prototypical
prototypically
prototyping
protozoan
protract
protrude
protruded
protrudes
protruding
protrusion
protrusions
protuberant
proud
prouder
proudest
proudly
provability
provable
provably
prove
proved
proven
provenance
prover
proverb
proverbial
proverbs
provers
proves
provide
provided
providence
provident
provider
providers
provides
providing
province
provinces
provincial
proving
provision
provisional
provisionally
provisioned
provisioning
provisions
proviso
provocation
provoke
provoked
provokes
provost
prow
prowess
prowl
prowled
prowler
prowlers
prowling
prows
proximal
proximate
proximity
proxy
prudence
prudent
prudential
prudently
prune
pruned
pruner
pruners
prunes
pruning
prurient
pry
prying
psalm
psalms
pseudo
pseudofiles
pseudoinstruction
pseudoinstructions
pseudonym
pseudoparallelism
psilocybin
psych
psyche
psychedelic
psyches
psychiatric
psychiatrist
psychiatrists
psychiatry
psychic
psycho
psychoanalysis
psychoanalyst
psychoanalytic
psychobiology
psychological
psychologically
psychologist
psychologists
psychology
psychopath
psychopathic
psychophysic
psychoses
psychosis
psychosocial
psychosomatic
psychotherapeutic
psychotherapist
psychotherapy
psychotic
pub
puberty
public
publication
publications
publicity
publicize
publicized
publicizes
publicizing
publicly
publish
published
publisher
publishers
publishes
publishing
pubs
pucker
puckered
puckering
puckers
pudding
puddings
puddle
puddles
puddling
puff
puffed
puffin
puffing
puffs
puke
pull
pulled
puller
pulley
pulleys
pulling
pullings
pullover
pulls
pulmonary
pulp
pulping
pulpit
pulpits
pulsar
pulsate
pulsation
pulsations
pulse
pulsed
pulses
pulsing
puma
pumice
pummel
pump
pumped
pumping
pumpkin
pumpkins
pumps
pun
punch
punched
puncher
punches
punching
punctual
punctually
punctuation
puncture
punctured
punctures
puncturing
pundit
pungent
punish
punishable
punished
punishes
punishing
punishment
punishments
punitive
puns
punt
punted
punting
punts
puny
pup
pupa
pupil
pupils
puppet
puppeteer
puppets
puppies
puppy
pups
purchase
purchased
purchaser
purchasers
purchases
purchasing
pure
purely
purer
purest
purgatory
purge
purged
purges
purging
purification
purifications
purified
purifier
purifiers
purifies
purify
purifying
purist
puritanic
purity
purple
purpler
purplest
purport
purported
purportedly
purporter
purporters
purporting
purports
purpose
purposed
purposeful
purposefully
purposely
purposes
purposive
purr
purred
purring
purrs
purse
pursed
purser
purses
pursuant
pursue
pursued
pursuer
pursuers
pursues
pursuing
pursuit
pursuits
purveyor
purview
pus
push
pushbutton
pushdown
pushed
pusher
pushers
pushes
pushing
puss
pussy
pussycat
put
puts
putt
putter
puttering
putters
putting
putty
puzzle
puzzled
puzzlement
puzzler
puzzlers
puzzles
puzzling
puzzlings
pygmies
pygmy
pyramid
pyramids
pyre
python
qua
quack
quacked
quackery
quacks
quad
quadrangle
quadrangular
quadrant
quadrants
quadratic
quadratical
quadratically
quadratics
quadrature
quadratures
quadrennial
quadrilateral
quadrillion
quadruple
quadrupled
quadruples
quadrupling
quadrupole
quaff
quagmire
quagmires
quahog
quail
quails
quaint
quaintly
quaintness
quake
quaked
quaker
quakers
quakes
quaking
qualification
qualifications
qualified
qualifier
qualifiers
qualifies
qualify
qualifying
qualitative
qualitatively
qualities
quality
qualm
quandaries
quandary
quanta
quantifiable
quantification
quantifications
quantified
quantifier
quantifiers
quantifies
quantify
quantifying
quantile
quantitative
quantitatively
quantities
quantity
quantization
quantize
quantized
quantizes
quantizing
quantum
quarantine
quarantines
quarantining
quark
quarrel
quarreled
quarreling
quarrels
quarrelsome
quarries
quarry
quart
quarter
quarterback
quartered
quartering
quarterly
quartermaster
quarters
quartet
quartets
quartile
quarts
quartz
quartzite
quasar
quash
quashed
quashes
quashing
quasi
quaternary
quaver
quavered
quavering
quavers
quay
queasy
queen
queenly
queens
queer
queerer
queerest
queerly
queerness
quell
quelling
quench
quenched
quenches
quenching
queried
queries
query
querying
quest
quested
quester
questers
questing
question
questionable
questionably
questioned
questioner
questioners
questioning
questioningly
questionings
questionnaire
questionnaires
questions
quests
queue
queued
queueing
queuer
queuers
queues
queuing
quibble
quick
quicken
quickened
quickening
quickens
quicker
quickest
quickie
quicklime
quickly
quickness
quicksand
quicksilver
quiescent
quiet
quieted
quieter
quietest
quieting
quietly
quietness
quiets
quietude
quill
quilt
quilted
quilting
quilts
quince
quinine
quint
quintet
quintillion
quip
quirk
quirky
quit
quite
quits
quitter
quitters
quitting
quiver
quivered
quivering
quivers
quixotic
quiz
quizzed
quizzes
quizzical
quizzing
quo
quonset
quorum
quota
quotas
quotation
quotations
quote
quoted
quotes
quoth
quotient
quotients
quoting
rabbi
rabbit
rabbits
rabble
rabid
rabies
raccoon
raccoons
race
raced
racer
racers
races
racetrack
racial
racially
racing
rack
racked
racket
racketeer
racketeering
racketeers
rackets
racking
racks
radar
radars
radial
radially
radian
radiance
radiant
radiantly
radiate
radiated
radiates
radiating
radiation
radiations
radiator
radiators
radical
radically
radicals
radices
radii
radio
radioactive
radioastronomy
radioed
radiography
radioing
radiology
radios
radish
radishes
radium
radius
radix
radon
raft
rafter
rafters
rafts
rag
rage
raged
rages
ragged
raggedly
raggedness
raging
rags
ragweed
raid
raided
raider
raiders
raiding
raids
rail
railed
railer
railers
railing
railroad
railroaded
railroader
railroaders
railroading
railroads
rails
railway
railways
raiment
rain
rainbow
raincoat
raincoats
raindrop
raindrops
rained
rainfall
rainier
rainiest
raining
rains
rainstorm
rainy
raise
raised
raiser
raisers
raises
raisin
raising
rake
raked
rakes
raking
rallied
rallies
rally
rallying
ram
ramble
rambler
rambles
rambling
ramblings
ramification
ramifications
ramp
rampage
rampant
rampart
ramps
ramrod
rams
ran
ranch
ranched
rancher
ranchers
ranches
ranching
rancid
random
randomization
randomize
randomized
randomizes
randomly
randomness
randy
rang
range
ranged
rangeland
ranger
rangers
ranges
ranging
rangy
rank
ranked
ranker
rankers
rankest
ranking
rankings
rankle
rankly
rankness
ranks
ransack
ransacked
ransacking
ransacks
ransom
ransomer
ransoming
ransoms
rant
ranted
ranter
ranters
ranting
rants
rap
rapacious
rape
raped
raper
rapes
rapid
rapidity
rapidly
rapids
rapier
raping
rapport
rapprochement
raps
rapt
raptly
rapture
raptures
rapturous
rare
rarely
rareness
rarer
rarest
rarity
rascal
rascally
rascals
rash
rasher
rashly
rashness
rasp
raspberry
rasped
rasping
rasps
raster
rat
rate
rated
rater
raters
rates
rather
ratification
ratified
ratifies
ratify
ratifying
rating
ratings
ratio
ration
rational
rationale
rationales
rationalities
rationality
rationalization
rationalizations
rationalize
rationalized
rationalizes
rationalizing
rationally
rationals
rationing
rations
ratios
rats
rattle
rattled
rattler
rattlers
rattles
rattlesnake
rattlesnakes
rattling
raucous
ravage
ravaged
ravager
ravagers
ravages
ravaging
rave
raved
raven
ravening
ravenous
ravenously
ravens
raves
ravine
ravines
raving
ravings
raw
rawer
rawest
rawly
rawness
ray
rays
raze
razor
razors
re
reabbreviate
reabbreviated
reabbreviates
reabbreviating
reach
reachability
reachable
reachably
reached
reacher
reaches
reaching
reacquired
react
reacted
reacting
reaction
reactionaries
reactionary
reactions
reactivate
reactivated
reactivates
reactivating
reactivation
reactive
reactively
reactivity
reactor
reactors
reacts
read
readability
readable
reader
readers
readied
readier
readies
readiest
readily
readiness
reading
readings
readjusted
readout
readouts
reads
ready
readying
real
realest
realign
realigned
realigning
realigns
realism
realist
realistic
realistically
realists
realities
reality
realizable
realizably
realization
realizations
realize
realized
realizes
realizing
reallocate
really
realm
realms
realness
reals
realtor
ream
reanalyze
reanalyzes
reanalyzing
reap
reaped
reaper
reaping
reappear
reappeared
reappearing
reappears
reappraisal
reappraisals
reaps
rear
reared
rearing
rearrange
rearrangeable
rearranged
rearrangement
rearrangements
rearranges
rearranging
rearrest
rearrested
rears
reason
reasonable
reasonableness
reasonably
reasoned
reasoner
reasoning
reasonings
reasons
reassemble
reassembled
reassembles
reassembling
reassembly
reassessment
reassessments
reassign
reassigned
reassigning
reassignment
reassignments
reassigns
reassure
reassured
reassures
reassuring
reawaken
reawakened
reawakening
reawakens
rebate
rebates
rebel
rebelled
rebelling
rebellion
rebellions
rebellious
rebelliously
rebelliousness
rebels
rebind
rebinding
rebinds
reboot
rebooted
rebooting
reboots
rebound
rebounded
rebounding
rebounds
rebroadcast
rebroadcasting
rebroadcasts
rebuff
rebuffed
rebuild
rebuilding
rebuilds
rebuilt
rebuke
rebuked
rebukes
rebuking
rebuttal
rebutted
rebutting
recalcitrant
recalculate
recalculated
recalculates
recalculating
recalculation
recalculations
recalibrate
recalibrated
recalibrates
recalibrating
recall
recalled
recalling
recalls
recant
recapitulate
recapitulated
recapitulates
recapitulation
recapture
recaptured
recaptures
recapturing
recast
recasting
recasts
recede
receded
recedes
receding
receipt
receipts
receivable
receive
received
receiver
receivers
receives
receiving
recent
recently
recentness
receptacle
receptacles
reception
receptionist
receptions
receptive
receptively
receptiveness
receptivity
receptor
recess
recessed
recesses
recession
recessive
recipe
recipes
recipient
recipients
reciprocal
reciprocally
reciprocate
reciprocated
reciprocates
reciprocating
reciprocation
reciprocity
recirculate
recirculated
recirculates
recirculating
recital
recitals
recitation
recitations
recite
recited
reciter
recites
reciting
reckless
recklessly
recklessness
reckon
reckoned
reckoner
reckoning
reckonings
reckons
reclaim
reclaimable
reclaimed
reclaimer
reclaimers
reclaiming
reclaims
reclamation
reclamations
reclassification
reclassified
reclassifies
reclassify
reclassifying
recline
reclining
recode
recoded
recodes
recoding
recognition
recognitions
recognizability
recognizable
recognizably
recognize
recognized
recognizer
recognizers
recognizes
recognizing
recoil
recoiled
recoiling
recoils
recollect
recollected
recollecting
recollection
recollections
recombination
recombine
recombined
recombines
recombining
recommend
recommendation
recommendations
recommended
recommender
recommending
recommends
recompense
recompile
recompiled
recompiles
recompiling
recompute
recomputed
recomputes
recomputing
reconcile
reconciled
reconciler
reconciles
reconciliation
reconciling
reconfigurable
reconfiguration
reconfigurations
reconfigure
reconfigured
reconfigurer
reconfigures
reconfiguring
reconnect
reconnected
reconnecting
reconnection
reconnects
reconsider
reconsideration
reconsidered
reconsidering
reconsiders
reconstituted
reconstruct
reconstructed
reconstructing
reconstruction
reconstructs
reconverted
reconverts
record
recorded
recorder
recorders
recording
recordings
records
recount
recounted
recounting
recounts
recourse
recover
recoverable
recovered
recoveries
recovering
recovers
recovery
recreate
recreated
recreates
recreating
recreation
recreational
recreations
recreative
recruit
recruited
recruiter
recruiting
recruits
recta
rectangle
rectangles
rectangular
rectify
rector
rectors
rectum
rectums
recuperate
recur
recurrence
recurrences
recurrent
recurrently
recurring
recurs
recurse
recursed
recurses
recursing
recursion
recursions
recursive
recursively
recyclable
recycle
recycled
recycles
recycling
red
redbreast
redcoat
redden
reddened
redder
reddest
reddish
reddishness
redeclare
redeclared
redeclares
redeclaring
redeem
redeemed
redeemer
redeemers
redeeming
redeems
redefine
redefined
redefines
redefining
redefinition
redefinitions
redemption
redesign
redesigned
redesigning
redesigns
redevelopment
redhead
redirect
redirected
redirecting
redirection
redirections
redisplay
redisplayed
redisplaying
redisplays
redistribute
redistributed
redistributes
redistributing
redly
redneck
redness
redo
redone
redouble
redoubled
redraw
redrawn
redress
redressed
redresses
redressing
reds
reduce
reduced
reducer
reducers
reduces
reducibility
reducible
reducibly
reducing
reduction
reductions
redundancies
redundancy
redundant
redundantly
redwood
reed
reeds
reeducation
reef
reefer
reefs
reel
reelect
reelected
reelecting
reelects
reeled
reeler
reeling
reels
reemphasize
reemphasized
reemphasizes
reemphasizing
reenabled
reenforcement
reenter
reentered
reentering
reenters
reentrant
reestablish
reestablished
reestablishes
reestablishing
reevaluate
reevaluated
reevaluates
reevaluating
reevaluation
reexamine
reexamined
reexamines
reexamining
reexecuted
refer
referee
refereed
refereeing
referees
reference
referenced
referencer
references
referencing
referenda
referendum
referendums
referent
referential
referentiality
referentially
referents
referral
referrals
referred
referring
refers
refill
refillable
refilled
refilling
refills
refine
refined
refinement
refinements
refiner
refinery
refines
refining
reflect
reflected
reflecting
reflection
reflections
reflective
reflectively
reflectivity
reflector
reflectors
reflects
reflex
reflexes
reflexive
reflexively
reflexiveness
reflexivity
reforestation
reform
reformable
reformat
reformation
reformatory
reformats
reformatted
reformatting
reformed
reformer
reformers
reforming
reforms
reformulate
reformulated
reformulates
reformulating
reformulation
refract
refracted
refraction
refractory
refragment
refrain
refrained
refraining
refrains
refresh
refreshed
refresher
refreshers
refreshes
refreshing
refreshingly
refreshment
refreshments
refrigerate
refrigerator
refrigerators
refuel
refueled
refueling
refuels
refuge
refugee
refugees
refusal
refuse
refused
refuses
refusing
refutable
refutation
refute
refuted
refuter
refutes
refuting
regain
regained
regaining
regains
regal
regaled
regally
regard
regarded
regarding
regardless
regards
regatta
regenerate
regenerated
regenerates
regenerating
regeneration
regenerative
regenerator
regenerators
regent
regents
regime
regimen
regiment
regimentation
regimented
regiments
regimes
region
regional
regionally
regions
register
registered
registering
registers
registrar
registration
registrations
registry
regress
regressed
regresses
regressing
regression
regressions
regressive
regret
regretful
regretfully
regrets
regrettable
regrettably
regretted
regretting
regroup
regrouped
regrouping
regular
regularities
regularity
regularly
regulars
regulate
regulated
regulates
regulating
regulation
regulations
regulative
regulator
regulators
regulatory
rehabilitate
rehearsal
rehearsals
rehearse
rehearsed
rehearser
rehearses
rehearsing
reign
reigned
reigning
reigns
reimbursable
reimburse
reimbursed
reimbursement
reimbursements
rein
reincarnate
reincarnated
reincarnation
reindeer
reined
reinforce
reinforced
reinforcement
reinforcements
reinforcer
reinforces
reinforcing
reinitialize
reinitialized
reinitializing
reins
reinsert
reinserted
reinserting
reinserts
reinstate
reinstated
reinstatement
reinstates
reinstating
reinterpret
reinterpreted
reinterpreting
reinterprets
reintroduce
reintroduced
reintroduces
reintroducing
reinvent
reinvented
reinventing
reinvents
reiterate
reiterated
reiterates
reiterating
reiteration
reject
rejected
rejecting
rejection
rejections
rejector
rejectors
rejects
rejoice
rejoiced
rejoicer
rejoices
rejoicing
rejoin
rejoinder
rejoined
rejoining
rejoins
relabel
relabeled
relabeling
relabelled
relabelling
relabels
relapse
relate
related
relater
relates
relating
relation
relational
relationally
relations
relationship
relationships
relative
relatively
relativeness
relatives
relativism
relativistic
relativistically
relativity
relax
relaxation
relaxations
relaxed
relaxer
relaxes
relaxing
relay
relayed
relaying
relays
release
released
releases
releasing
relegate
relegated
relegates
relegating
relent
relented
relenting
relentless
relentlessly
relentlessness
relents
relevance
relevances
relevant
relevantly
reliability
reliable
reliably
reliance
reliant
relic
relics
relied
relief
relies
relieve
relieved
reliever
relievers
relieves
relieving
religion
religions
religious
religiously
religiousness
relink
relinquish
relinquished
relinquishes
relinquishing
relish
relished
relishes
relishing
relive
relives
reliving
reload
reloaded
reloader
reloading
reloads
relocatable
relocate
relocated
relocates
relocating
relocation
relocations
reluctance
reluctant
reluctantly
rely
relying
remain
remainder
remainders
remained
remaining
remains
remark
remarkable
remarkableness
remarkably
remarked
remarking
remarks
remedial
remedied
remedies
remedy
remedying
remember
remembered
remembering
remembers
remembrance
remembrances
remind
reminded
reminder
reminders
reminding
reminds
reminiscence
reminiscences
reminiscent
reminiscently
remiss
remission
remit
remittance
remnant
remnants
remodel
remodeled
remodeling
remodels
remonstrate
remonstrated
remonstrates
remonstrating
remonstration
remonstrative
remorse
remorseful
remote
remotely
remoteness
remotest
removable
removal
removals
remove
removed
remover
removes
removing
remunerate
remuneration
renaissance
renal
rename
renamed
renames
renaming
rend
render
rendered
rendering
renderings
renders
rendezvous
rending
rendition
renditions
rends
renegade
renegotiable
renew
renewable
renewal
renewed
renewer
renewing
renews
renounce
renounces
renouncing
renovate
renovated
renovation
renown
renowned
rent
rental
rentals
rented
renting
rents
renumber
renumbering
renumbers
renunciate
renunciation
reoccur
reopen
reopened
reopening
reopens
reorder
reordered
reordering
reorders
reorganization
reorganizations
reorganize
reorganized
reorganizes
reorganizing
repackage
repaid
repair
repaired
repairer
repairing
repairman
repairmen
repairs
reparation
reparations
repartee
repartition
repast
repasts
repay
repaying
repays
repeal
repealed
repealer
repealing
repeals
repeat
repeatable
repeated
repeatedly
repeater
repeaters
repeating
repeats
repel
repelled
repellent
repels
repent
repentance
repented
repenting
repents
repercussion
repercussions
repertoire
repertory
repetition
repetitions
repetitious
repetitive
repetitively
repetitiveness
rephrase
rephrased
rephrases
rephrasing
repine
replace
replaceable
replaced
replacement
replacements
replacer
replaces
replacing
replay
replayed
replaying
replays
replenish
replenished
replenishes
replenishing
replete
repleteness
repletion
replica
replicas
replicate
replicated
replicates
replicating
replication
replications
replied
replies
reply
replying
report
reported
reportedly
reporter
reporters
reporting
reports
repose
reposed
reposes
reposing
reposition
repositioned
repositioning
repositions
repositories
repository
reprehensible
represent
representable
representably
representation
representational
representationally
representations
representative
representatively
representativeness
representatives
represented
representing
represents
repress
repressed
represses
repressing
repression
repressions
repressive
reprieve
reprieved
reprieves
reprieving
reprimand
reprint
reprinted
reprinting
reprints
reprisal
reprisals
reproach
reproached
reproaches
reproaching
reprobate
reproduce
reproduced
reproducer
reproducers
reproduces
reproducibilities
reproducibility
reproducible
reproducibly
reproducing
reproduction
reproductions
reprogram
reprogrammed
reprogramming
reprograms
reproof
reprove
reprover
reptile
reptiles
reptilian
republic
republican
republicans
republics
repudiate
repudiated
repudiates
repudiating
repudiation
repudiations
repugnant
repulse
repulsed
repulses
repulsing
repulsion
repulsions
repulsive
reputable
reputably
reputation
reputations
repute
reputed
reputedly
reputes
request
requested
requester
requesters
requesting
requests
require
required
requirement
requirements
requires
requiring
requisite
requisites
requisition
requisitioned
requisitioning
requisitions
reread
reregister
reroute
rerouted
reroutes
rerouting
rerun
reruns
reschedule
rescind
rescue
rescued
rescuer
rescuers
rescues
rescuing
research
researched
researcher
researchers
researches
researching
reselect
reselected
reselecting
reselects
resell
reselling
resemblance
resemblances
resemble
resembled
resembles
resembling
resent
resented
resentful
resentfully
resenting
resentment
resents
reserpine
reservation
reservations
reserve
reserved
reserver
reserves
reserving
reservoir
reservoirs
reset
resets
resetting
resettings
reside
resided
residence
residences
resident
residential
residentially
residents
resides
residing
residual
residue
residues
resign
resignation
resignations
resigned
resigning
resigns
resilient
resin
resins
resist
resistable
resistance
resistances
resistant
resistantly
resisted
resistible
resisting
resistive
resistivity
resistor
resistors
resists
resolute
resolutely
resoluteness
resolution
resolutions
resolvable
resolve
resolved
resolver
resolvers
resolves
resolving
resonance
resonances
resonant
resonate
resort
resorted
resorting
resorts
resound
resounding
resounds
resource
resourceful
resourcefully
resourcefulness
resources
respect
respectability
respectable
respectably
respected
respecter
respectful
respectfully
respectfulness
respecting
respective
respectively
respects
respiration
respirator
respiratory
respite
resplendent
resplendently
respond
responded
respondent
respondents
responder
responding
responds
response
responses
responsibilities
responsibility
responsible
responsibleness
responsibly
responsive
responsively
responsiveness
rest
restart
restarted
restarting
restarts
restate
restated
restatement
restates
restating
restaurant
restaurants
restaurateur
rested
restful
restfully
restfulness
resting
restitution
restive
restless
restlessly
restlessness
restoration
restorations
restore
restored
restorer
restorers
restores
restoring
restrain
restrained
restrainer
restrainers
restraining
restrains
restraint
restraints
restrict
restricted
restricting
restriction
restrictions
restrictive
restrictively
restricts
restroom
restructure
restructured
restructures
restructuring
rests
result
resultant
resultantly
resultants
resulted
resulting
results
resumable
resume
resumed
resumes
resuming
resumption
resumptions
resurgent
resurrect
resurrected
resurrecting
resurrection
resurrections
resurrector
resurrectors
resurrects
resuscitate
resynchronization
resynchronize
resynchronized
resynchronizing
retail
retailer
retailers
retailing
retain
retained
retainer
retainers
retaining
retainment
retains
retaliate
retaliation
retaliatory
retard
retarded
retarder
retarding
retch
retention
retentions
retentive
retentively
retentiveness
reticle
reticles
reticular
reticulate
reticulated
reticulately
reticulates
reticulating
reticulation
retina
retinal
retinas
retinue
retire
retired
retiree
retirement
retirements
retires
retiring
retort
retorted
retorts
retrace
retraced
retraces
retracing
retract
retracted
retracting
retraction
retractions
retracts
retrain
retrained
retraining
retrains
retranslate
retranslated
retransmission
retransmissions
retransmit
retransmits
retransmitted
retransmitting
retreat
retreated
retreating
retreats
retribution
retried
retrier
retriers
retries
retrievable
retrieval
retrievals
retrieve
retrieved
retriever
retrievers
retrieves
retrieving
retroactive
retroactively
retrofit
retrofitting
retrograde
retrospect
retrospection
retrospective
retry
retrying
return
returnable
returned
returner
returning
returns
retype
retyped
retypes
retyping
reunion
reunions
reunite
reunited
reuniting
reusable
reuse
reused
reuses
reusing
revamp
revamped
revamping
revamps
reveal
revealed
revealing
reveals
revel
revelation
revelations
reveled
reveler
reveling
revelry
revels
revenge
revenger
revenue
revenuers
revenues
reverberate
revere
revered
reverence
reverend
reverends
reverent
reverently
reveres
reverie
reverified
reverifies
reverify
reverifying
revering
reversal
reversals
reverse
reversed
reversely
reverser
reverses
reversible
reversing
reversion
revert
reverted
reverting
reverts
review
reviewed
reviewer
reviewers
reviewing
reviews
revile
reviled
reviler
reviling
revise
revised
reviser
revises
revising
revision
revisionary
revisions
revisit
revisited
revisiting
revisits
revival
revivals
revive
revived
reviver
revives
reviving
revocable
revocation
revoke
revoked
revoker
revokes
revoking
revolt
revolted
revolter
revolting
revoltingly
revolts
revolution
revolutionaries
revolutionary
revolutionize
revolutionized
revolutionizer
revolutions
revolve
revolved
revolver
revolvers
revolves
revolving
revulsion
reward
rewarded
rewarding
rewardingly
rewards
rewind
rewinding
rewinds
rewire
rework
reworked
reworking
reworks
rewound
rewrite
rewrites
rewriting
rewritten
rhapsody
rhesus
rhetoric
rheumatic
rheumatism
rhinestone
rhino
rhinoceros
rho
rhododendron
rhombic
rhombus
rhubarb
rhyme
rhymed
rhymes
rhyming
rhythm
rhythmic
rhythmically
rhythms
rib
ribald
ribbed
ribbing
ribbon
ribbons
riboflavin
ribonucleic
ribs
rice
rich
richer
riches
richest
richly
richness
rickets
rickety
rickshaw
rickshaws
ricochet
rid
riddance
ridden
ridding
riddle
riddled
riddles
riddling
ride
rider
riders
rides
ridge
ridgepole
ridges
ridicule
ridiculed
ridicules
ridiculing
ridiculous
ridiculously
ridiculousness
riding
rids
rifle
rifled
rifleman
rifler
rifles
rifling
rift
rig
rigging
right
righted
righteous
righteously
righteousness
righter
rightful
rightfully
rightfulness
righting
rightly
rightmost
rightness
rights
rightward
rigid
rigidity
rigidly
rigor
rigorous
rigorously
rigors
rigs
rill
rim
rime
rims
rind
rinds
ring
ringed
ringer
ringers
ringing
ringingly
ringings
rings
ringside
rink
rinse
rinsed
rinser
rinses
rinsing
riot
rioted
rioter
rioters
rioting
riotous
riots
rip
ripe
ripely
ripen
ripeness
ripoff
ripped
ripping
ripple
rippled
ripples
rippling
rips
rise
risen
riser
risers
rises
rising
risings
risk
risked
risking
risks
risky
rite
rites
ritual
ritually
rituals
rival
rivaled
rivalled
rivalling
rivalries
rivalry
rivals
river
riverbank
riverfront
rivers
riverside
rivet
riveter
rivets
rivulet
rivulets
roach
road
roadbed
roadblock
roads
roadside
roadster
roadsters
roadway
roadways
roam
roamed
roaming
roams
roar
roared
roarer
roaring
roars
roast
roasted
roaster
roasting
roasts
rob
robbed
robber
robberies
robbers
robbery
robbing
robe
robed
robes
robin
robing
robins
robot
robotic
robotics
robots
robs
robust
robustly
robustness
rock
rockabye
rocked
rocker
rockers
rocket
rocketed
rocketing
rockets
rocking
rocks
rocky
rod
rode
rodent
rodents
rodeo
rods
roe
rogue
rogues
role
roles
roll
rollback
rolled
roller
rollers
rolling
rolls
romance
romancer
romancers
romances
romancing
romantic
romantics
romp
romped
romper
romping
romps
roof
roofed
roofer
roofing
roofs
rooftop
rook
rookie
room
roomed
roomer
roomers
roomful
rooming
roommate
rooms
roomy
roost
rooster
roosters
root
rooted
rooter
rooting
roots
rope
roped
roper
ropers
ropes
roping
rosary
rosebud
rosebuds
rosebush
rosemary
roses
rosette
rosiness
roster
rostrum
rosy
rot
rotary
rotate
rotated
rotates
rotating
rotation
rotational
rotations
rotator
rotor
rots
rotten
rottenness
rotting
rotund
rotunda
rouge
rough
roughed
roughen
rougher
roughest
roughly
roughneck
roughness
roulette
round
roundabout
rounded
roundedness
rounder
roundest
roundhead
roundhouse
rounding
roundly
roundness
roundoff
rounds
roundtable
roundup
roundworm
rouse
roused
rouses
rousing
roustabout
rout
route
routed
router
routers
routes
routine
routinely
routines
routing
routings
rove
roved
rover
roves
roving
row
rowboat
rowdy
rowed
rower
rowing
rows
royal
royalist
royalists
royally
royalties
royalty
rub
rubbed
rubber
rubbers
rubbery
rubbing
rubbish
rubble
rubdown
rubies
ruble
rubles
rubout
rubs
ruby
rudder
rudders
ruddiness
ruddy
rude
rudely
rudeness
rudiment
rudimentary
rudiments
rue
ruefully
ruffian
ruffianly
ruffians
ruffle
ruffled
ruffles
rug
rugged
ruggedly
ruggedness
rugs
ruin
ruination
ruinations
ruined
ruining
ruinous
ruinously
ruins
rule
ruled
ruler
rulers
rules
ruling
rulings
rum
rumble
rumbled
rumbler
rumbles
rumbling
rumen
rummage
rummy
rumor
rumored
rumors
rump
rumple
rumpled
rumply
rumpus
run
runaway
rundown
rung
rungs
runnable
runner
runners
running
runoff
runs
runt
runtime
rupee
rupture
ruptured
ruptures
rupturing
rural
rurally
rush
rushed
rusher
rushes
rushing
russet
rust
rusted
rustic
rusticate
rusticated
rusticates
rusticating
rustication
rusting
rustle
rustled
rustler
rustlers
rustling
rusts
rusty
rut
ruthless
ruthlessly
ruthlessness
ruts
rye
sabbath
sabbatical
saber
sabers
sable
sables
sabotage
sack
sacker
sacking
sacks
sacrament
sacred
sacredly
sacredness
sacrifice
sacrificed
sacrificer
sacrificers
sacrifices
sacrificial
sacrificially
sacrificing
sacrilege
sacrilegious
sacrosanct
sad
sadden
saddened
saddens
sadder
saddest
saddle
saddlebag
saddled
saddles
sadism
sadist
sadistic
sadistically
sadists
sadly
sadness
safari
safe
safeguard
safeguarded
safeguarding
safeguards
safekeeping
safely
safeness
safer
safes
safest
safeties
safety
saffron
sag
saga
sagacious
sagacity
sage
sagebrush
sagely
sages
sagging
sagittal
sags
saguaro
said
sail
sailboat
sailed
sailfish
sailing
sailor
sailorly
sailors
sails
saint
sainted
sainthood
saintly
saints
sake
sakes
salable
salad
salads
salamander
salami
salaried
salaries
salary
sale
sales
salesgirl
saleslady
salesman
salesmen
salesperson
salient
saline
saliva
salivary
salivate
sallies
sallow
sallying
salmon
salon
salons
saloon
saloons
salt
salted
salter
salters
saltier
saltiest
saltiness
salting
salts
salty
salutary
salutation
salutations
salute
saluted
salutes
saluting
salvage
salvaged
salvager
salvages
salvaging
salvation
salve
salver
salves
same
sameness
sample
sampled
sampler
samplers
samples
sampling
samplings
sanatoria
sanatorium
sanctification
sanctified
sanctify
sanctimonious
sanction
sanctioned
sanctioning
sanctions
sanctity
sanctuaries
sanctuary
sanctum
sand
sandal
sandals
sandbag
sanded
sander
sanding
sandman
sandpaper
sands
sandstone
sandwich
sandwiches
sandy
sane
sanely
saner
sanest
sang
sanguine
sanitarium
sanitary
sanitation
sanity
sank
sap
sapiens
sapling
saplings
sapphire
saps
sapsucker
sarcasm
sarcasms
sarcastic
sardine
sardonic
sari
sash
sat
satanic
satchel
satchels
sate
sated
satellite
satellites
sates
satin
sating
satire
satires
satiric
satisfaction
satisfactions
satisfactorily
satisfactory
satisfiability
satisfiable
satisfied
satisfies
satisfy
satisfying
saturate
saturated
saturates
saturating
saturation
satyr
sauce
saucepan
saucepans
saucer
saucers
sauces
saucy
saunter
sausage
sausages
savage
savaged
savagely
savageness
savager
savagers
savages
savaging
save
saved
saver
savers
saves
saving
savings
savior
saviors
savor
savored
savoring
savors
savory
saw
sawdust
sawed
sawfish
sawing
sawmill
sawmills
saws
sawtooth
sax
saxophone
say
sayer
sayers
saying
sayings
says
scab
scabbard
scabbards
scabrous
scaffold
scaffolding
scaffoldings
scaffolds
scalable
scalar
scalars
scald
scalded
scalding
scale
scaled
scales
scaling
scalings
scallop
scalloped
scallops
scalp
scalps
scaly
scamper
scampering
scampers
scan
scandal
scandalous
scandals
scanned
scanner
scanners
scanning
scans
scant
scantier
scantiest
scantily
scantiness
scantly
scanty
scapegoat
scar
scarce
scarcely
scarceness
scarcer
scarcity
scare
scarecrow
scared
scares
scarf
scaring
scarlet
scars
scarves
scary
scatter
scatterbrain
scattered
scattering
scatters
scenario
scenarios
scene
scenery
scenes
scenic
scent
scented
scents
scepter
scepters
schedulable
schedule
scheduled
scheduler
schedulers
schedules
scheduling
schema
schemas
schemata
schematic
schematically
schematics
scheme
schemed
schemer
schemers
schemes
scheming
schism
schizophrenia
scholar
scholarly
scholars
scholarship
scholarships
scholastic
scholastically
scholastics
school
schoolboy
schoolboys
schooled
schooler
schoolers
schoolhouse
schoolhouses
schooling
schoolmaster
schoolmasters
schoolroom
schoolrooms
schools
schooner
science
sciences
scientific
scientifically
scientist
scientists
scissor
scissored
scissoring
scissors
sclerosis
sclerotic
scoff
scoffed
scoffer
scoffing
scoffs
scold
scolded
scolding
scolds
scoop
scooped
scooping
scoops
scoot
scope
scoped
scopes
scoping
scorch
scorched
scorcher
scorches
scorching
score
scoreboard
scorecard
scored
scorer
scorers
scores
scoring
scorings
scorn
scorned
scorner
scornful
scornfully
scorning
scorns
scorpion
scorpions
scotch
scoundrel
scoundrels
scour
scoured
scourge
scouring
scours
scout
scouted
scouting
scouts
scow
scowl
scowled
scowling
scowls
scram
scramble
scrambled
scrambler
scrambles
scrambling
scrap
scrape
scraped
scraper
scrapers
scrapes
scraping
scrapings
scrapped
scraps
scratch
scratched
scratcher
scratchers
scratches
scratching
scratchy
scrawl
scrawled
scrawling
scrawls
scrawny
scream
screamed
screamer
screamers
screaming
screams
screech
screeched
screeches
screeching
screen
screened
screening
screenings
screenplay
screens
screw
screwball
screwdriver
screwed
screwing
screws
scribble
scribbled
scribbler
scribbles
scribe
scribes
scribing
scrimmage
script
scripts
scripture
scriptures
scroll
scrolled
scrolling
scrolls
scrounge
scrub
scrumptious
scruple
scrupulous
scrupulously
scrutinize
scrutinized
scrutinizing
scrutiny
scuba
scud
scuffle
scuffled
scuffles
scuffling
sculpt
sculpted
sculptor
sculptors
sculpts
sculpture
sculptured
sculptures
scurried
scurry
scurvy
scuttle
scuttled
scuttles
scuttling
scythe
scythes
sea
seaboard
seacoast
seacoasts
seafood
seagull
seahorse
seal
sealed
sealer
sealing
seals
sealy
seam
seaman
seamed
seamen
seaming
seams
seamy
seaport
seaports
sear
search
searched
searcher
searchers
searches
searching
searchingly
searchings
searchlight
seared
searing
searingly
seas
seashore
seashores
seaside
season
seasonable
seasonably
seasonal
seasonally
seasoned
seasoner
seasoners
seasoning
seasonings
seasons
seat
seated
seating
seats
seaward
seaweed
secant
secede
seceded
secedes
seceding
secession
seclude
secluded
seclusion
second
secondaries
secondarily
secondary
seconded
seconder
seconders
secondhand
seconding
secondly
seconds
secrecy
secret
secretarial
secretariat
secretaries
secretary
secrete
secreted
secretes
secreting
secretion
secretions
secretive
secretively
secretly
secrets
sect
sectarian
section
sectional
sectioned
sectioning
sections
sector
sectors
sects
secular
secure
secured
securely
secures
securing
securings
securities
security
sedan
sedate
sedge
sediment
sedimentary
sediments
sedition
seditious
seduce
seduced
seducer
seducers
seduces
seducing
seduction
seductive
see
seed
seeded
seeder
seeders
seeding
seedings
seedling
seedlings
seeds
seedy
seeing
seek
seeker
seekers
seeking
seeks
seem
seemed
seeming
seemingly
seemly
seems
seen
seep
seepage
seeped
seeping
seeps
seer
seers
seersucker
sees
seethe
seethed
seethes
seething
segment
segmentation
segmentations
segmented
segmenting
segments
segregate
segregated
segregates
segregating
segregation
seismic
seismograph
seismology
seize
seized
seizes
seizing
seizure
seizures
seldom
select
selected
selecting
selection
selections
selective
selectively
selectivity
selectman
selectmen
selector
selectors
selects
selenium
self
selfish
selfishly
selfishness
selfsame
sell
seller
sellers
selling
sellout
sells
seltzer
selves
semantic
semantical
semantically
semanticist
semanticists
semantics
semaphore
semaphores
semblance
semester
semesters
semi
semiautomated
semicolon
semicolons
semiconductor
semiconductors
seminal
seminar
seminarian
seminaries
seminars
seminary
semipermanent
semipermanently
senate
senates
senator
senatorial
senators
send
sender
senders
sending
sends
senile
senior
seniority
seniors
sensation
sensational
sensationally
sensations
sense
sensed
senseless
senselessly
senselessness
senses
sensibilities
sensibility
sensible
sensibly
sensing
sensitive
sensitively
sensitiveness
sensitives
sensitivities
sensitivity
sensor
sensors
sensory
sensual
sensuous
sent
sentence
sentenced
sentences
sentencing
sentential
sentiment
sentimental
sentimentally
sentiments
sentinel
sentinels
sentries
sentry
separable
separate
separated
separately
separateness
separates
separating
separation
separations
separator
separators
sepia
sept
sepulcher
sepulchers
sequel
sequels
sequence
sequenced
sequencer
sequencers
sequences
sequencing
sequencings
sequential
sequentiality
sequentialize
sequentialized
sequentializes
sequentializing
sequentially
sequester
serendipitous
serendipity
serene
serenely
serenity
serf
serfs
sergeant
sergeants
serial
serializability
serializable
serialization
serializations
serialize
serialized
serializes
serializing
serially
serials
series
serif
serious
seriously
seriousness
sermon
sermons
serpent
serpentine
serpents
serum
serums
servant
servants
serve
served
server
servers
serves
service
serviceability
serviceable
serviced
serviceman
servicemen
services
servicing
servile
serving
servings
servitude
servo
servomechanism
sesame
session
sessions
set
setback
sets
settable
setter
setters
setting
settings
settle
settled
settlement
settlements
settler
settlers
settles
settling
setup
setups
seven
sevenfold
sevens
seventeen
seventeens
seventeenth
seventh
seventies
seventieth
seventy
sever
several
severalfold
severally
severance
severe
severed
severely
severer
severest
severing
severities
severity
severs
sew
sewage
sewed
sewer
sewers
sewing
sews
sex
sexed
sexes
sexist
sextet
sextillion
sexton
sextuple
sextuplet
sexual
sexuality
sexually
sexy
shabby
shack
shacked
shackle
shackled
shackles
shackling
shacks
shade
shaded
shades
shadier
shadiest
shadily
shadiness
shading
shadings
shadow
shadowed
shadowing
shadows
shadowy
shady
shaft
shafts
shaggy
shakable
shakably
shake
shakedown
shaken
shaker
shakers
shakes
shakiness
shaking
shaky
shale
shall
shallow
shallower
shallowly
shallowness
sham
shambles
shame
shamed
shameful
shamefully
shameless
shamelessly
shames
shaming
shampoo
shamrock
shams
shanties
shanty
shape
shaped
shapeless
shapelessly
shapelessness
shapely
shaper
shapers
shapes
shaping
sharable
shard
share
shareable
sharecropper
sharecroppers
shared
shareholder
shareholders
sharer
sharers
shares
sharing
shark
sharks
sharp
sharpen
sharpened
sharpening
sharpens
sharper
sharpest
sharply
sharpness
sharpshoot
shatter
shattered
shattering
shatterproof
shatters
shave
shaved
shaven
shaves
shaving
shavings
shawl
shawls
she
sheaf
shear
sheared
shearing
shears
sheath
sheathing
sheaths
sheaves
shed
shedding
sheds
sheen
sheep
sheepskin
sheer
sheered
sheet
sheeted
sheeting
sheets
sheik
shelf
shell
shelled
sheller
shelling
shells
shelter
sheltered
sheltering
shelters
shelve
shelved
shelves
shelving
shenanigan
shepherd
shepherds
sherbet
sheriff
sheriffs
sherry
shibboleth
shied
shield
shielded
shielding
shies
shift
shifted
shifter
shifters
shiftier
shiftiest
shiftily
shiftiness
shifting
shifts
shifty
shill
shilling
shillings
shimmer
shimmering
shin
shinbone
shine
shined
shiner
shiners
shines
shingle
shingles
shining
shiningly
shiny
ship
shipboard
shipbuilding
shipmate
shipment
shipments
shipped
shipper
shippers
shipping
ships
shipshape
shipwreck
shipwrecked
shipwrecks
shipyard
shire
shirk
shirker
shirking
shirks
shirt
shirting
shirts
shit
shiver
shivered
shiverer
shivering
shivers
shoal
shoals
shock
shocked
shocker
shockers
shocking
shockingly
shocks
shod
shoddy
shoe
shoed
shoehorn
shoeing
shoelace
shoemaker
shoes
shoestring
shone
shook
shoot
shooter
shooters
shooting
shootings
shoots
shop
shopkeeper
shopkeepers
shopped
shopper
shoppers
shopping
shops
shopworn
shore
shoreline
shores
shorn
short
shortage
shortages
shortcoming
shortcomings
shortcut
shortcuts
shorted
shorten
shortened
shortening
shortens
shorter
shortest
shortfall
shorthand
shorthanded
shorting
shortish
shortly
shortness
shorts
shortsighted
shortstop
shot
shotgun
shotguns
shots
should
shoulder
shouldered
shouldering
shoulders
shout
shouted
shouter
shouters
shouting
shouts
shove
shoved
shovel
shoveled
shovels
shoves
shoving
show
showboat
showcase
showdown
showed
shower
showered
showering
showers
showing
showings
shown
showpiece
showroom
shows
showy
shrank
shrapnel
shred
shredder
shredding
shreds
shrew
shrewd
shrewdest
shrewdly
shrewdness
shrews
shriek
shrieked
shrieking
shrieks
shrill
shrilled
shrilling
shrillness
shrilly
shrimp
shrine
shrines
shrink
shrinkable
shrinkage
shrinking
shrinks
shrivel
shriveled
shroud
shrouded
shrub
shrubbery
shrubs
shrug
shrugs
shrunk
shrunken
shudder
shuddered
shuddering
shudders
shuffle
shuffleboard
shuffled
shuffles
shuffling
shun
shuns
shunt
shut
shutdown
shutdowns
shutoff
shutout
shuts
shutter
shuttered
shutters
shutting
shuttle
shuttlecock
shuttled
shuttles
shuttling
shy
shyly
shyness
sibling
siblings
sick
sicken
sicker
sickest
sickle
sickly
sickness
sicknesses
sickroom
side
sidearm
sideband
sideboard
sideboards
sideburns
sidecar
sided
sidelight
sidelights
sideline
sidereal
sides
sidesaddle
sideshow
sidestep
sidetrack
sidewalk
sidewalks
sideways
sidewise
siding
sidings
siege
sieges
sierra
sieve
sieves
sift
sifted
sifter
sifting
sigh
sighed
sighing
sighs
sight
sighted
sighting
sightings
sightly
sights
sightseeing
sigma
sign
signal
signaled
signaling
signalled
signalling
signally
signals
signature
signatures
signed
signer
signers
signet
significance
significant
significantly
significants
signification
signified
signifies
signify
signifying
signing
signs
silence
silenced
silencer
silencers
silences
silencing
silent
silently
silhouette
silhouetted
silhouettes
silica
silicate
silicon
silicone
silk
silken
silkier
silkiest
silkily
silks
silky
sill
silliest
silliness
sills
silly
silo
silt
silted
silting
silts
silver
silvered
silvering
silvers
silversmith
silverware
silvery
similar
similarities
similarity
similarly
simile
similitude
simmer
simmered
simmering
simmers
simple
simpleminded
simpleness
simpler
simplest
simpleton
simplex
simplicities
simplicity
simplification
simplifications
simplified
simplifier
simplifiers
simplifies
simplify
simplifying
simplistic
simply
simulate
simulated
simulates
simulating
simulation
simulations
simulator
simulators
simulcast
simultaneity
simultaneous
simultaneously
since
sincere
sincerely
sincerest
sincerity
sine
sines
sinew
sinews
sinewy
sinful
sinfully
sinfulness
sing
singable
singe
singed
singer
singers
singing
singingly
single
singled
singlehanded
singleness
singles
singlet
singleton
singletons
singling
singly
sings
singsong
singular
singularities
singularity
singularly
sinister
sink
sinked
sinker
sinkers
sinkhole
sinking
sinks
sinned
sinner
sinners
sinning
sins
sinuous
sinus
sinusoid
sinusoidal
sinusoids
sip
siphon
siphoning
sipping
sips
sir
sire
sired
siren
sirens
sires
sirs
sirup
sister
sisterly
sisters
sit
site
sited
sites
siting
sits
sitter
sitters
sitting
sittings
situ
situate
situated
situates
situating
situation
situational
situationally
situations
six
sixes
sixfold
sixgun
sixpence
sixteen
sixteens
sixteenth
sixth
sixties
sixtieth
sixty
sizable
size
sized
sizes
sizing
sizings
sizzle
skate
skated
skater
skaters
skates
skating
skeletal
skeleton
skeletons
skeptic
skeptical
skeptically
skepticism
skeptics
sketch
sketchbook
sketched
sketches
sketchily
sketching
sketchpad
sketchy
skew
skewed
skewer
skewers
skewing
skews
ski
skid
skidding
skied
skies
skiff
skiing
skill
skilled
skillet
skillful
skillfully
skillfulness
skills
skim
skimmed
skimming
skimp
skimped
skimping
skimps
skimpy
skims
skin
skindive
skinned
skinner
skinners
skinning
skinny
skins
skip
skipped
skipper
skippers
skipping
skips
skirmish
skirmished
skirmisher
skirmishers
skirmishes
skirmishing
skirt
skirted
skirting
skirts
skis
skit
skulk
skulked
skulker
skulking
skulks
skull
skullcap
skullduggery
skulls
skunk
skunks
sky
skyhook
skyjack
skylark
skylarking
skylarks
skylight
skylights
skyline
skyrockets
skyscraper
skyscrapers
slab
slack
slacken
slacker
slacking
slackly
slackness
slacks
slain
slam
slammed
slamming
slams
slander
slanderer
slanderous
slanders
slang
slant
slanted
slanting
slants
slap
slapped
slapping
slaps
slapstick
slash
slashed
slashes
slashing
slat
slate
slated
slater
slates
slats
slaughter
slaughtered
slaughterhouse
slaughtering
slaughters
slave
slaver
slavery
slaves
slavish
slay
slayer
slayers
slaying
slays
sled
sledding
sledge
sledgehammer
sledges
sleds
sleek
sleep
sleeper
sleepers
sleepily
sleepiness
sleeping
sleepless
sleeplessly
sleeplessness
sleeps
sleepwalk
sleepy
sleet
sleeve
sleeves
sleigh
sleighs
sleight
slender
slenderer
slept
sleuth
slew
slewing
slice
sliced
slicer
slicers
slices
slicing
slick
slicker
slickers
slicks
slid
slide
slider
sliders
slides
sliding
slight
slighted
slighter
slightest
slighting
slightly
slightness
slights
slim
slime
slimed
slimly
slimy
sling
slinging
slings
slingshot
slip
slippage
slipped
slipper
slipperiness
slippers
slippery
slipping
slips
slit
slither
slits
sliver
slob
slogan
slogans
sloop
slop
slope
sloped
sloper
slopers
slopes
sloping
slopped
sloppiness
slopping
sloppy
slops
slot
sloth
slothful
sloths
slots
slotted
slotting
slouch
slouched
slouches
slouching
slow
slowdown
slowed
slower
slowest
slowing
slowly
slowness
slows
sludge
slug
sluggish
sluggishly
sluggishness
slugs
sluice
slum
slumber
slumbered
slumming
slump
slumped
slumps
slums
slung
slur
slurp
slurring
slurry
slurs
sly
slyly
smack
smacked
smacking
smacks
small
smaller
smallest
smallish
smallness
smallpox
smalltime
smart
smarted
smarter
smartest
smartly
smartness
smash
smashed
smasher
smashers
smashes
smashing
smashingly
smattering
smear
smeared
smearing
smears
smell
smelled
smelling
smells
smelly
smelt
smelter
smelts
smile
smiled
smiles
smiling
smilingly
smirk
smite
smith
smithereens
smiths
smithy
smitten
smock
smocking
smocks
smog
smokable
smoke
smoked
smoker
smokers
smokes
smokescreen
smokestack
smokies
smoking
smoky
smolder
smoldered
smoldering
smolders
smooch
smooth
smoothbore
smoothed
smoother
smoothes
smoothest
smoothing
smoothly
smoothness
smote
smother
smothered
smothering
smothers
smudge
smug
smuggle
smuggled
smuggler
smugglers
smuggles
smuggling
smut
smutty
snack
snafu
snag
snail
snails
snake
snaked
snakelike
snakes
snap
snapdragon
snapped
snapper
snappers
snappily
snapping
snappy
snaps
snapshot
snapshots
snare
snared
snares
snaring
snark
snarl
snarled
snarling
snatch
snatched
snatches
snatching
snazzy
sneak
sneaked
sneaker
sneakers
sneakier
sneakiest
sneakily
sneakiness
sneaking
sneaks
sneaky
sneer
sneered
sneering
sneers
sneeze
sneezed
sneezes
sneezing
sniff
sniffed
sniffing
sniffle
sniffs
snifter
snigger
snip
snipe
snippet
snivel
snob
snobbery
snobbish
snoop
snooped
snooping
snoops
snoopy
snore
snored
snores
snoring
snorkel
snort
snorted
snorting
snorts
snotty
snout
snouts
snow
snowball
snowed
snowfall
snowflake
snowier
snowiest
snowily
snowing
snowman
snowmen
snows
snowshoe
snowshoes
snowstorm
snowy
snub
snuff
snuffed
snuffer
snuffing
snuffs
snug
snuggle
snuggled
snuggles
snuggling
snugly
snugness
so
soak
soaked
soaking
soaks
soap
soaped
soaping
soaps
soapy
soar
soared
soaring
soars
sob
sobbing
sober
sobered
sobering
soberly
soberness
sobers
sobriety
sobs
soccer
sociability
sociable
sociably
social
socialism
socialist
socialists
socialize
socialized
socializes
socializing
socially
societal
societies
society
socioeconomic
sociological
sociologically
sociologist
sociologists
sociology
sock
socked
socket
sockets
socking
socks
sod
soda
sodium
sodomy
sods
sofa
sofas
soft
softball
soften
softened
softening
softens
softer
softest
softly
softness
software
softwares
soggy
soil
soiled
soiling
soils
soiree
sojourn
sojourner
sojourners
solace
solaced
solar
sold
solder
soldered
soldier
soldiering
soldierly
soldiers
sole
solely
solemn
solemnity
solemnly
solemnness
solenoid
soles
solicit
solicitation
solicited
soliciting
solicitor
solicitous
solicits
solicitude
solid
solidarity
solidification
solidified
solidifies
solidify
solidifying
solidity
solidly
solidness
solids
soliloquy
solitaire
solitary
solitude
solitudes
solo
solos
solstice
solubility
soluble
solution
solutions
solvable
solve
solved
solvent
solvents
solver
solvers
solves
solving
somatic
somber
somberly
some
somebody
someday
somehow
someone
someplace
somersault
something
sometime
sometimes
somewhat
somewhere
sommelier
somnolent
son
sonar
sonata
song
songbook
songs
sonic
sonnet
sonnets
sonny
sons
soon
sooner
soonest
soot
sooth
soothe
soothed
soother
soothes
soothing
soothsayer
sophisticated
sophistication
sophistry
sophomore
sophomores
soprano
sorcerer
sorcerers
sorcery
sordid
sordidly
sordidness
sore
sorely
soreness
sorer
sores
sorest
sorghum
sorority
sorrel
sorrier
sorriest
sorrow
sorrowful
sorrowfully
sorrows
sorry
sort
sorted
sorter
sorters
sortie
sorting
sorts
sought
soul
soulful
souls
sound
sounded
sounder
soundest
sounding
soundings
soundly
soundness
soundproof
sounds
soup
souped
soups
sour
source
sources
sourdough
soured
sourer
sourest
souring
sourly
sourness
sours
south
southbound
southeast
southeastern
southern
southerner
southerners
southernmost
southland
southpaw
southward
southwest
southwestern
souvenir
sovereign
sovereigns
sovereignty
soviet
soviets
sow
sown
soy
soya
soybean
spa
space
spacecraft
spaced
spacer
spacers
spaces
spaceship
spaceships
spacesuit
spacing
spacings
spacious
spaded
spades
spading
span
spandrel
spaniel
spank
spanked
spanking
spanks
spanned
spanner
spanners
spanning
spans
spare
spared
sparely
spareness
sparer
spares
sparest
sparing
sparingly
spark
sparked
sparking
sparkle
sparkling
sparks
sparring
sparrow
sparrows
sparse
sparsely
sparseness
sparser
sparsest
spasm
spastic
spat
spate
spates
spatial
spatially
spatter
spattered
spatula
spawn
spawned
spawning
spawns
spayed
speak
speakable
speakeasy
speaker
speakers
speaking
speaks
spear
speared
spearmint
spears
spec
special
specialist
specialists
specialization
specializations
specialize
specialized
specializes
specializing
specially
specials
specialties
specialty
specie
species
specifiable
specific
specifically
specification
specifications
specificity
specifics
specified
specifier
specifiers
specifies
specify
specifying
specimen
specimens
specious
speck
speckle
speckled
speckles
specks
spectacle
spectacled
spectacles
spectacular
spectacularly
spectator
spectators
specter
specters
spectra
spectral
spectrogram
spectrograms
spectrograph
spectrographic
spectrography
spectrometer
spectrophotometer
spectrophotometry
spectroscope
spectroscopic
spectroscopy
spectrum
speculate
speculated
speculates
speculating
speculation
speculations
speculative
speculator
speculators
sped
speech
speeches
speechless
speechlessness
speed
speedboat
speeded
speeder
speeders
speedily
speeding
speedometer
speeds
speedup
speedups
speedy
spell
spellbound
spelled
speller
spellers
spelling
spellings
spells
spend
spender
spenders
spending
spends
spent
sperm
sphere
spheres
spherical
spherically
spheroid
spheroidal
sphinx
spice
spiced
spices
spiciness
spicy
spider
spiders
spidery
spies
spigot
spike
spiked
spikes
spill
spilled
spiller
spilling
spills
spilt
spin
spinach
spinal
spinally
spindle
spindled
spindling
spine
spinnaker
spinner
spinners
spinning
spinoff
spins
spinster
spiny
spiral
spiraled
spiraling
spirally
spire
spires
spirit
spirited
spiritedly
spiriting
spirits
spiritual
spiritually
spirituals
spit
spite
spited
spiteful
spitefully
spitefulness
spites
spitfire
spiting
spits
spitting
spittle
splash
splashed
splashes
splashing
splashy
spleen
splendid
splendidly
splendor
splenetic
splice
spliced
splicer
splicers
splices
splicing
splicings
spline
splines
splint
splinter
splintered
splinters
splintery
split
splits
splitter
splitters
splitting
splurge
spoil
spoilage
spoiled
spoiler
spoilers
spoiling
spoils
spoke
spoked
spoken
spokes
spokesman
spokesmen
sponge
sponged
sponger
spongers
sponges
sponging
spongy
sponsor
sponsored
sponsoring
sponsors
sponsorship
spontaneity
spontaneous
spontaneously
spoof
spook
spooky
spool
spooled
spooler
spoolers
spooling
spools
spoon
spooned
spoonful
spooning
spoons
sporadic
spore
spores
sport
sported
sporting
sportingly
sportive
sports
sportsman
sportsmen
sportswear
sportswriter
sportswriting
sporty
spot
spotless
spotlessly
spotlight
spots
spotted
spotter
spotters
spotting
spotty
spouse
spouses
spout
spouted
spouting
spouts
sprain
sprang
sprawl
sprawled
sprawling
sprawls
spray
sprayed
sprayer
spraying
sprays
spread
spreader
spreaders
spreading
spreadings
spreads
spreadsheet
spree
sprees
sprig
sprightly
spring
springboard
springer
springers
springier
springiest
springiness
springing
springs
springtime
springy
sprinkle
sprinkled
sprinkler
sprinkles
sprinkling
sprint
sprinted
sprinter
sprinters
sprinting
sprints
sprite
sprocket
sprout
sprouted
sprouting
spruce
spruced
sprung
spun
spunk
spur
spurious
spurn
spurned
spurning
spurns
spurs
spurt
spurted
spurting
spurts
sputter
sputtered
spy
spyglass
spying
squabble
squabbled
squabbles
squabbling
squad
squadron
squadrons
squads
squalid
squall
squalls
squander
square
squared
squarely
squareness
squarer
squares
squarest
squaring
squash
squashed
squashing
squat
squats
squatting
squaw
squawk
squawked
squawking
squawks
squeak
squeaked
squeaking
squeaks
squeaky
squeal
squealed
squealing
squeals
squeamish
squeeze
squeezed
squeezer
squeezes
squeezing
squelch
squid
squint
squinted
squinting
squire
squires
squirm
squirmed
squirms
squirmy
squirrel
squirreled
squirreling
squirrels
squirt
squishy
stab
stabbed
stabbing
stabile
stabilities
stability
stabilize
stabilized
stabilizer
stabilizers
stabilizes
stabilizing
stable
stabled
stabler
stables
stabling
stably
stabs
stack
stacked
stacking
stacks
stadia
stadium
staff
staffed
staffer
staffers
staffing
staffs
stag
stage
stagecoach
stagecoaches
staged
stager
stagers
stages
stagger
staggered
staggering
staggers
staging
stagnant
stagnate
stagnation
stags
staid
stain
stained
staining
stainless
stains
stair
staircase
staircases
stairs
stairway
stairways
stairwell
stake
staked
stakes
stalactite
stale
stalemate
stalk
stalked
stalking
stall
stalled
stalling
stallings
stallion
stalls
stalwart
stalwartly
stamen
stamens
stamina
stammer
stammered
stammerer
stammering
stammers
stamp
stamped
stampede
stampeded
stampedes
stampeding
stamper
stampers
stamping
stamps
stanch
stanchest
stanchion
stand
standard
standardization
standardize
standardized
standardizes
standardizing
standardly
standards
standby
standing
standings
standoff
standpoint
standpoints
stands
standstill
stanza
stanzas
staphylococcus
staple
stapler
staples
stapling
star
starboard
starch
starched
stardom
stare
stared
starer
stares
starfish
staring
stark
starkly
starlet
starlight
starling
starred
starring
starry
stars
start
started
starter
starters
starting
startle
startled
startles
startling
starts
startup
startups
starvation
starve
starved
starves
starving
state
stated
stately
statement
statements
states
statesman
statesmanlike
statesmen
statewide
static
statically
stating
station
stationary
stationed
stationer
stationery
stationing
stationmaster
stations
statistic
statistical
statistically
statistician
statisticians
statistics
statue
statues
statuesque
statuesquely
statuesqueness
statuette
stature
status
statuses
statute
statutes
statutorily
statutoriness
statutory
staunch
staunchest
staunchly
stave
staved
staves
stay
stayed
staying
stays
stead
steadfast
steadfastly
steadfastness
steadied
steadier
steadies
steadiest
steadily
steadiness
steady
steadying
steak
steaks
steal
stealer
stealing
steals
stealth
stealthily
stealthy
steam
steamboat
steamboats
steamed
steamer
steamers
steaming
steams
steamship
steamships
steamy
steed
steel
steeled
steelers
steeling
steelmaker
steels
steely
steep
steeped
steeper
steepest
steeping
steeple
steeples
steeply
steepness
steeps
steer
steerable
steered
steering
steers
stellar
stem
stemmed
stemming
stems
stench
stenches
stencil
stencils
stenographer
stenographers
stenotype
step
stepchild
stepmother
stepmothers
stepped
stepper
stepping
steps
stepson
stepwise
stereo
stereos
stereoscopic
stereotype
stereotyped
stereotypes
stereotypical
sterile
sterilization
sterilizations
sterilize
sterilized
sterilizer
sterilizes
sterilizing
sterling
stern
sternly
sternness
sterns
stethoscope
stevedore
stew
steward
stewardess
stewards
stewed
stews
stick
sticker
stickers
stickier
stickiest
stickily
stickiness
sticking
stickleback
sticks
sticky
stiff
stiffen
stiffens
stiffer
stiffest
stiffly
stiffness
stiffs
stifle
stifled
stifles
stifling
stigma
stigmata
stile
stiles
stiletto
still
stillbirth
stillborn
stilled
stiller
stillest
stilling
stillness
stills
stilt
stilts
stimulant
stimulants
stimulate
stimulated
stimulates
stimulating
stimulation
stimulations
stimulative
stimuli
stimulus
sting
stinging
stings
stingy
stink
stinker
stinkers
stinking
stinks
stint
stipend
stipends
stipulate
stipulated
stipulates
stipulating
stipulation
stipulations
stir
stirred
stirrer
stirrers
stirring
stirringly
stirrings
stirrup
stirs
stitch
stitched
stitches
stitching
stochastic
stochastically
stock
stockade
stockades
stockbroker
stocked
stocker
stockers
stockholder
stockholders
stocking
stockings
stockpile
stockroom
stocks
stocky
stodgy
stoichiometry
stoke
stole
stolen
stoles
stolid
stomach
stomached
stomacher
stomaches
stomaching
stomp
stoned
stones
stoning
stony
stood
stooge
stool
stoop
stooped
stooping
stoops
stop
stopcock
stopcocks
stopgap
stopover
stoppable
stoppage
stopped
stopper
stoppers
stopping
stops
stopwatch
storage
storages
store
stored
storehouse
storehouses
storekeeper
storeroom
stores
storied
stories
storing
stork
storks
storm
stormed
stormier
stormiest
storminess
storming
storms
stormy
story
storyboard
storyteller
stout
stouter
stoutest
stoutly
stoutness
stove
stoves
stow
stowed
straddle
strafe
straggle
straggled
straggler
stragglers
straggles
straggling
straight
straightaway
straighten
straightened
straightens
straighter
straightest
straightforward
straightforwardly
straightforwardness
straightness
straightway
strain
strained
strainer
strainers
straining
strains
strait
straiten
straits
strand
stranded
stranding
strands
strange
strangely
strangeness
stranger
strangers
strangest
strangle
strangled
strangler
stranglers
strangles
strangling
stranglings
strangulation
strangulations
strap
straps
stratagem
stratagems
strategic
strategies
strategist
strategy
stratification
stratifications
stratified
stratifies
stratify
stratosphere
stratospheric
stratum
straw
strawberries
strawberry
straws
stray
strayed
strays
streak
streaked
streaks
stream
streamed
streamer
streamers
streaming
streamline
streamlined
streamliner
streamlines
streamlining
streams
street
streetcar
streetcars
streeters
streets
strength
strengthen
strengthened
strengthener
strengthening
strengthens
strengths
strenuous
strenuously
streptococcus
stress
stressed
stresses
stressful
stressing
stretch
stretched
stretcher
stretchers
stretches
stretching
strew
strewn
strews
stricken
strict
stricter
strictest
strictly
strictness
stricture
stride
strider
strides
striding
strife
strike
strikebreaker
striker
strikers
strikes
striking
strikingly
string
stringed
stringent
stringently
stringer
stringers
stringier
stringiest
stringiness
stringing
strings
stringy
strip
stripe
striped
stripes
stripped
stripper
strippers
stripping
strips
striptease
strive
striven
strives
striving
strivings
strobe
strobed
strobes
stroboscopic
strode
stroke
stroked
stroker
strokers
strokes
stroking
stroll
strolled
stroller
strolling
strolls
strong
stronger
strongest
stronghold
strongly
strontium
strove
struck
structural
structurally
structure
structured
structurer
structures
structuring
struggle
struggled
struggles
struggling
strung
strut
struts
strutting
strychnine
stub
stubble
stubborn
stubbornly
stubbornness
stubby
stubs
stucco
stuck
stud
student
students
studied
studies
studio
studios
studious
studiously
studs
study
studying
stuff
stuffed
stuffier
stuffiest
stuffing
stuffs
stuffy
stumble
stumbled
stumbles
stumbling
stump
stumped
stumping
stumps
stun
stung
stunning
stunningly
stunt
stunts
stupefy
stupefying
stupendous
stupendously
stupid
stupidest
stupidities
stupidity
stupidly
stupor
sturdiness
sturdy
sturgeon
stutter
style
styled
styler
stylers
styles
styli
styling
stylish
stylishly
stylishness
stylistic
stylistically
stylized
stylus
suave
sub
subatomic
subchannel
subchannels
subclass
subclasses
subcommittees
subcomponent
subcomponents
subcomputation
subcomputations
subconscious
subconsciously
subculture
subcultures
subcycle
subcycles
subdirectories
subdirectory
subdivide
subdivided
subdivides
subdividing
subdivision
subdivisions
subdomains
subdue
subdued
subdues
subduing
subexpression
subexpressions
subfield
subfields
subfile
subfiles
subgoal
subgoals
subgraph
subgraphs
subgroup
subgroups
subinterval
subintervals
subject
subjected
subjecting
subjection
subjective
subjectively
subjectivity
subjects
sublanguage
sublanguages
sublayer
sublayers
sublimation
sublimations
sublime
sublimed
sublist
sublists
submarine
submariner
submariners
submarines
submerge
submerged
submerges
submerging
submission
submissions
submissive
submit
submits
submittal
submitted
submitting
submode
submodes
submodule
submodules
submultiplexed
subnet
subnets
subnetwork
subnetworks
suboptimal
subordinate
subordinated
subordinates
subordination
subparts
subphases
subpoena
subproblem
subproblems
subprocesses
subprogram
subprograms
subproject
subproof
subproofs
subrange
subranges
subroutine
subroutines
subs
subschema
subschemas
subscribe
subscribed
subscriber
subscribers
subscribes
subscribing
subscript
subscripted
subscripting
subscription
subscriptions
subscripts
subsection
subsections
subsegment
subsegments
subsequence
subsequences
subsequent
subsequently
subservient
subset
subsets
subside
subsided
subsides
subsidiaries
subsidiary
subsidies
subsiding
subsidize
subsidized
subsidizes
subsidizing
subsidy
subsist
subsisted
subsistence
subsistent
subsisting
subsists
subslot
subslots
subspace
subspaces
substance
substances
substantial
substantially
substantiate
substantiated
substantiates
substantiating
substantiation
substantiations
substantive
substantively
substantivity
substation
substations
substitutability
substitutable
substitute
substituted
substitutes
substituting
substitution
substitutions
substrate
substrates
substring
substrings
substructure
substructures
subsume
subsumed
subsumes
subsuming
subsystem
subsystems
subtask
subtasks
subterfuge
subterranean
subtitle
subtitled
subtitles
subtle
subtleness
subtler
subtlest
subtleties
subtlety
subtly
subtotal
subtract
subtracted
subtracting
subtraction
subtractions
subtractor
subtractors
subtracts
subtrahend
subtrahends
subtree
subtrees
subunit
subunits
suburb
suburban
suburbia
suburbs
subversion
subversive
subvert
subverted
subverter
subverting
subverts
subway
subways
succeed
succeeded
succeeding
succeeds
success
successes
successful
successfully
succession
successions
successive
successively
successor
successors
succinct
succinctly
succinctness
succor
succumb
succumbed
succumbing
succumbs
such
suck
sucked
sucker
suckers
sucking
suckle
suckling
sucks
suction
sudden
suddenly
suddenness
suds
sudsing
sue
sued
sues
suffer
sufferance
suffered
sufferer
sufferers
suffering
sufferings
suffers
suffice
sufficed
suffices
sufficiency
sufficient
sufficiently
sufficing
suffix
suffixed
suffixer
suffixes
suffixing
suffocate
suffocated
suffocates
suffocating
suffocation
suffrage
suffragette
sugar
sugared
sugaring
sugarings
sugars
suggest
suggested
suggestible
suggesting
suggestion
suggestions
suggestive
suggestively
suggests
suicidal
suicidally
suicide
suicides
suing
suit
suitability
suitable
suitableness
suitably
suitcase
suitcases
suite
suited
suiters
suites
suiting
suitor
suitors
suits
sulfa
sulfur
sulfuric
sulfurous
sulk
sulked
sulkiness
sulking
sulks
sulky
sullen
sullenly
sullenness
sulphate
sulphur
sulphured
sulphuric
sultan
sultans
sultry
sum
sumac
summand
summands
summaries
summarily
summarization
summarizations
summarize
summarized
summarizes
summarizing
summary
summation
summations
summed
summertime
summing
summit
summitry
summon
summoned
summoner
summoners
summoning
summons
summonses
sumptuous
sums
sun
sunbeam
sunbeams
sunbonnet
sunburn
sunburnt
sunder
sundial
sundown
sundries
sundry
sunflower
sung
sunglass
sunglasses
sunk
sunken
sunlight
sunlit
sunned
sunning
sunny
sunrise
suns
sunset
sunshine
sunspot
suntan
suntanned
suntanning
super
superb
superblock
superbly
supercomputer
supercomputers
superego
superegos
superficial
superficially
superfluities
superfluity
superfluous
superfluously
supergroup
supergroups
superhuman
superhumanly
superimpose
superimposed
superimposes
superimposing
superintend
superintendent
superintendents
superior
superiority
superiors
superlative
superlatively
superlatives
supermarket
supermarkets
supermini
superminis
supernatural
superpose
superposed
superposes
superposing
superposition
superscript
superscripted
superscripting
superscripts
supersede
superseded
supersedes
superseding
superset
supersets
superstition
superstitions
superstitious
superuser
supervise
supervised
supervises
supervising
supervision
supervisor
supervisors
supervisory
supine
supper
suppers
supplant
supplanted
supplanting
supplants
supple
supplement
supplemental
supplementary
supplemented
supplementing
supplements
suppleness
supplication
supplied
supplier
suppliers
supplies
supply
supplying
support
supportable
supported
supporter
supporters
supporting
supportingly
supportive
supportively
supports
suppose
supposed
supposedly
supposes
supposing
supposition
suppositions
suppress
suppressed
suppresses
suppressing
suppression
suppressor
suppressors
supranational
supremacy
supreme
supremely
surcharge
sure
surely
sureness
sureties
surety
surf
surface
surfaced
surfaceness
surfaces
surfacing
surge
surged
surgeon
surgeons
surgery
surges
surgical
surgically
surging
surliness
surly
surmise
surmised
surmises
surmount
surmounted
surmounting
surmounts
surname
surnames
surpass
surpassed
surpasses
surpassing
surplus
surpluses
surprise
surprised
surprises
surprising
surprisingly
surreal
surrender
surrendered
surrendering
surrenders
surreptitious
surrey
surrogate
surrogates
surround
surrounded
surrounding
surroundings
surrounds
surtax
survey
surveyed
surveying
surveyor
surveyors
surveys
survival
survivals
survive
survived
survives
surviving
survivor
survivors
susceptible
suspect
suspected
suspecting
suspects
suspend
suspended
suspender
suspenders
suspending
suspends
suspense
suspenses
suspension
suspensions
suspicion
suspicions
suspicious
suspiciously
sustain
sustained
sustaining
sustains
sustenance
suture
sutures
suzerainty
svelte
swab
swabbing
swagger
swaggered
swaggering
swain
swains
swallow
swallowed
swallowing
swallows
swallowtail
swam
swami
swamp
swamped
swamping
swamps
swampy
swan
swank
swanky
swanlike
swans
swap
swapped
swapping
swaps
swarm
swarmed
swarming
swarms
swarthy
swastika
swat
swatted
sway
swayed
swaying
swear
swearer
swearing
swears
sweat
sweated
sweater
sweaters
sweating
sweats
sweatshirt
sweaty
sweep
sweeper
sweepers
sweeping
sweepings
sweeps
sweepstakes
sweet
sweeten
sweetened
sweetener
sweeteners
sweetening
sweetenings
sweetens
sweeter
sweetest
sweetheart
sweethearts
sweetish
sweetly
sweetness
sweets
swell
swelled
swelling
swellings
swells
swelter
swept
swerve
swerved
swerves
swerving
swift
swifter
swiftest
swiftly
swiftness
swim
swimmer
swimmers
swimming
swimmingly
swims
swimsuit
swindle
swine
swing
swinger
swingers
swinging
swings
swipe
swirl
swirled
swirling
swish
swished
swiss
switch
switchblade
switchboard
switchboards
switched
switcher
switchers
switches
switching
switchings
switchman
swivel
swizzle
swollen
swoon
swoop
swooped
swooping
swoops
sword
swordfish
swords
swore
sworn
swum
swung
sycamore
sycophant
sycophantic
syllable
syllables
syllogism
syllogisms
syllogistic
sylvan
symbiosis
symbiotic
symbol
symbolic
symbolically
symbolics
symbolism
symbolization
symbolize
symbolized
symbolizes
symbolizing
symbols
symmetric
symmetrical
symmetrically
symmetries
symmetry
sympathetic
sympathies
sympathize
sympathized
sympathizer
sympathizers
sympathizes
sympathizing
sympathizingly
sympathy
symphonic
symphonies
symphony
symposia
symposium
symposiums
symptom
symptomatic
symptoms
synagogue
synapse
synapses
synaptic
synchronism
synchronization
synchronize
synchronized
synchronizer
synchronizers
synchronizes
synchronizing
synchronous
synchronously
synchrony
synchrotron
syncopate
syndicate
syndicated
syndicates
syndication
syndrome
syndromes
synergism
synergistic
synergy
synod
synonym
synonymous
synonymously
synonyms
synopses
synopsis
syntactic
syntactical
syntactically
syntax
syntaxes
synthesis
synthesize
synthesized
synthesizer
synthesizers
synthesizes
synthesizing
synthetic
synthetics
syringe
syringes
syrup
syrupy
system
systematic
systematically
systematize
systematized
systematizes
systematizing
systemic
systems
systemwide
tab
tabernacle
tabernacles
table
tableau
tableaus
tablecloth
tablecloths
tabled
tables
tablespoon
tablespoonful
tablespoonfuls
tablespoons
tablet
tablets
tabling
taboo
taboos
tabs
tabular
tabulate
tabulated
tabulates
tabulating
tabulation
tabulations
tabulator
tabulators
tachometer
tachometers
tacit
tacitly
tack
tacked
tacking
tackle
tackles
tact
tactic
tactics
tactile
tag
tagged
tagging
tags
tail
tailed
tailing
tailor
tailored
tailoring
tailors
tails
taint
tainted
take
taken
taker
takers
takes
taking
takings
tale
talent
talented
talents
tales
talk
talkative
talkatively
talkativeness
talked
talker
talkers
talkie
talking
talks
tall
taller
tallest
tallness
tallow
tally
tame
tamed
tamely
tameness
tamer
tames
taming
tamper
tampered
tampering
tampers
tan
tandem
tang
tangent
tangential
tangents
tangible
tangibly
tangle
tangled
tangy
tank
tanker
tankers
tanks
tanner
tanners
tantalizing
tantalizingly
tantamount
tantrum
tantrums
tap
tape
taped
taper
tapered
tapering
tapers
tapes
tapestries
tapestry
taping
tapings
tapped
tapper
tappers
tapping
taproot
taproots
taps
tar
tardiness
tardy
target
targeted
targeting
targets
tariff
tariffs
tarry
tart
tartly
tartness
task
tasked
tasking
tasks
tassel
tassels
taste
tasted
tasteful
tastefully
tastefulness
tasteless
tastelessly
taster
tasters
tastes
tasting
tatter
tattered
tattoo
tattooed
tattoos
tau
taught
taunt
taunted
taunter
taunting
taunts
taut
tautly
tautness
tautological
tautologically
tautologies
tautology
tavern
taverns
tawny
tax
taxable
taxation
taxed
taxes
taxi
taxicab
taxicabs
taxied
taxiing
taxing
taxis
taxonomic
taxonomically
taxonomy
taxpayer
taxpayers
tea
teach
teachable
teacher
teachers
teaches
teaching
teachings
teacup
team
teamed
teaming
teams
tear
teared
tearful
tearfully
tearing
tears
teas
tease
teased
teases
teasing
teaspoon
teaspoonful
teaspoonfuls
teaspoons
technical
technicalities
technicality
technically
technician
technicians
technique
techniques
technological
technologically
technologies
technologist
technologists
technology
tedious
tediously
tediousness
tedium
teem
teemed
teeming
teems
teen
teenage
teenaged
teenager
teenagers
teens
teeth
teethe
teethed
teethes
teething
telecommunication
telecommunications
telegram
telegrams
telegraph
telegraphed
telegrapher
telegraphers
telegraphic
telegraphing
telegraphs
telemetry
teleological
teleologically
teleology
telepathy
telephone
telephoned
telephoner
telephoners
telephones
telephonic
telephoning
telephony
teleprocessing
telescope
telescoped
telescopes
telescoping
teletype
teletypes
televise
televised
televises
televising
television
televisions
televisor
televisors
tell
teller
tellers
telling
tells
temper
temperament
temperamental
temperaments
temperance
temperate
temperately
temperateness
temperature
temperatures
tempered
tempering
tempers
tempest
tempestuous
tempestuously
template
templates
temple
temples
temporal
temporally
temporaries
temporarily
temporary
tempt
temptation
temptations
tempted
tempter
tempters
tempting
temptingly
tempts
ten
tenacious
tenaciously
tenant
tenants
tend
tended
tendencies
tendency
tender
tenderly
tenderness
tenders
tending
tends
tenement
tenements
tenfold
tennis
tenor
tenors
tens
tense
tensed
tensely
tenseness
tenser
tenses
tensest
tensing
tension
tensions
tent
tentacle
tentacled
tentacles
tentative
tentatively
tented
tenth
tenting
tents
tenure
term
termed
terminal
terminally
terminals
terminate
terminated
terminates
terminating
termination
terminations
terminator
terminators
terming
terminologies
terminology
terminus
terms
termwise
ternary
terrace
terraced
terraces
terrain
terrains
terrestrial
terrestrials
terrible
terribly
terrier
terriers
terrific
terrified
terrifies
terrify
terrifying
territorial
territories
territory
terror
terrorism
terrorist
terroristic
terrorists
terrorize
terrorized
terrorizes
terrorizing
terrors
tertiary
test
testability
testable
testament
testaments
tested
tester
testers
testicle
testicles
testified
testifier
testifiers
testifies
testify
testifying
testimonies
testimony
testing
testings
tests
text
textbook
textbooks
textile
textiles
texts
textual
textually
texture
textured
textures
than
thank
thanked
thankful
thankfully
thankfulness
thanking
thankless
thanklessly
thanklessness
thanks
thanksgiving
thanksgivings
that
thatch
thatches
thats
thaw
thawed
thawing
thaws
the
theater
theaters
theatrical
theatrically
theatricals
theft
thefts
their
theirs
them
thematic
theme
themes
themselves
then
thence
thenceforth
theological
theology
theorem
theorems
theoretic
theoretical
theoretically
theoreticians
theories
theorist
theorists
theorization
theorizations
theorize
theorized
theorizer
theorizers
theorizes
theorizing
theory
therapeutic
therapies
therapist
therapists
therapy
there
thereabouts
thereafter
thereby
therefore
therein
thereof
thereon
thereto
thereupon
therewith
thermal
thermodynamic
thermodynamics
thermometer
thermometers
thermostat
thermostats
these
theses
thesis
they
thick
thicken
thickens
thicker
thickest
thicket
thickets
thickly
thickness
thief
thieve
thieves
thieving
thigh
thighs
thimble
thimbles
thin
thing
things
think
thinkable
thinkably
thinker
thinkers
thinking
thinks
thinly
thinner
thinness
thinnest
third
thirdly
thirds
thirst
thirsted
thirsts
thirsty
thirteen
thirteens
thirteenth
thirties
thirtieth
thirty
this
thistle
thong
thorn
thorns
thorny
thorough
thoroughfare
thoroughfares
thoroughly
thoroughness
those
though
thought
thoughtful
thoughtfully
thoughtfulness
thoughtless
thoughtlessly
thoughtlessness
thoughts
thousand
thousands
thousandth
thrash
thrashed
thrasher
thrashes
thrashing
thread
threaded
threader
threaders
threading
threads
threat
threaten
threatened
threatening
threatens
threats
three
threefold
threes
threescore
threshold
thresholds
threw
thrice
thrift
thrifty
thrill
thrilled
thriller
thrillers
thrilling
thrillingly
thrills
thrive
thrived
thrives
thriving
throat
throated
throats
throb
throbbed
throbbing
throbs
throne
thrones
throng
throngs
throttle
throttled
throttles
throttling
through
throughout
throughput
throw
thrower
throwing
thrown
throws
thrush
thrust
thruster
thrusters
thrusting
thrusts
thud
thuds
thug
thugs
thumb
thumbed
thumbing
thumbs
thump
thumped
thumping
thunder
thunderbolt
thunderbolts
thundered
thunderer
thunderers
thundering
thunders
thunderstorm
thunderstorms
thus
thusly
thwart
thwarted
thwarting
thwarts
thyself
tick
ticked
ticker
tickers
ticket
tickets
ticking
tickle
tickled
tickles
tickling
ticklish
ticks
tidal
tidally
tide
tided
tides
tidied
tidiness
tiding
tidings
tidy
tidying
tie
tied
tier
tiers
ties
tiger
tigers
tight
tighten
tightened
tightener
tighteners
tightening
tightenings
tightens
tighter
tightest
tightly
tightness
tilde
tile
tiled
tiles
tiling
till
tillable
tilled
tiller
tillers
tilling
tills
tilt
tilted
tilting
tilts
timber
timbered
timbering
timbers
time
timed
timeless
timelessly
timelessness
timely
timeout
timeouts
timer
timers
times
timeshare
timeshares
timesharing
timestamp
timestamps
timetable
timetables
timid
timidity
timidly
timing
timings
tin
tincture
tinge
tinged
tingle
tingled
tingles
tingling
tinier
tiniest
tinily
tininess
tinker
tinkered
tinkering
tinkers
tinkle
tinkled
tinkles
tinkling
tinnier
tinniest
tinnily
tinniness
tinny
tins
tint
tinted
tinting
tints
tiny
tip
tipped
tipper
tippers
tipping
tips
tiptoe
tire
tired
tiredly
tireless
tirelessly
tirelessness
tires
tiresome
tiresomely
tiresomeness
tiring
tissue
tissues
tit
tithe
tither
tithes
tithing
title
titled
titles
tits
titter
titters
to
toad
toads
toast
toasted
toaster
toasting
toasts
tobacco
today
todays
toe
toes
together
togetherness
toggle
toggled
toggles
toggling
toil
toiled
toiler
toilet
toilets
toiling
toils
token
tokens
told
tolerability
tolerable
tolerably
tolerance
tolerances
tolerant
tolerantly
tolerate
tolerated
tolerates
tolerating
toleration
toll
tolled
tolls
tomahawk
tomahawks
tomato
tomatoes
tomb
tombs
tomography
tomorrow
tomorrows
ton
tone
toned
toner
tones
tongs
tongue
tongued
tongues
tonic
tonics
tonight
toning
tonnage
tons
tonsil
too
took
tool
tooled
tooler
toolers
tooling
tools
tooth
toothbrush
toothbrushes
toothpaste
toothpick
toothpicks
top
toper
topic
topical
topically
topics
topmost
topography
topological
topologies
topology
topple
toppled
topples
toppling
tops
torch
torches
tore
torment
tormented
tormenter
tormenters
tormenting
torn
tornado
tornadoes
torpedo
torpedoes
torque
torrent
torrents
torrid
tortoise
tortoises
torture
tortured
torturer
torturers
tortures
torturing
torus
toruses
toss
tossed
tosses
tossing
total
totaled
totaling
totalities
totality
totalled
totaller
totallers
totalling
totally
totals
totter
tottered
tottering
totters
touch
touchable
touched
touches
touchier
touchiest
touchily
touchiness
touching
touchingly
touchy
tough
toughen
tougher
toughest
toughly
toughness
tour
toured
touring
tourist
tourists
tournament
tournaments
tours
tow
toward
towards
towed
towel
toweling
towelled
towelling
towels
tower
towered
towering
towers
town
towns
township
townships
toy
toyed
toying
toys
trace
traceable
traced
tracer
tracers
traces
tracing
tracings
track
tracked
tracker
trackers
tracking
tracks
tract
tractability
tractable
tractive
tractor
tractors
tracts
trade
traded
trademark
trademarks
tradeoff
tradeoffs
trader
traders
trades
tradesman
trading
tradition
traditional
traditionally
traditions
traffic
trafficked
trafficker
traffickers
trafficking
traffics
tragedies
tragedy
tragic
tragically
trail
trailed
trailer
trailers
trailing
trailings
trails
train
trained
trainee
trainees
trainer
trainers
training
trains
trait
traitor
traitors
traits
trajectories
trajectory
tramp
tramped
tramping
trample
trampled
trampler
tramples
trampling
tramps
trance
trances
tranquil
tranquility
tranquilly
transact
transaction
transactions
transatlantic
transceive
transceiver
transceivers
transcend
transcended
transcendent
transcending
transcends
transcontinental
transcribe
transcribed
transcriber
transcribers
transcribes
transcribing
transcript
transcription
transcriptions
transcripts
transfer
transferability
transferable
transferal
transferals
transference
transferred
transferrer
transferrers
transferring
transfers
transfinite
transform
transformable
transformation
transformational
transformations
transformed
transformer
transformers
transforming
transforms
transgress
transgressed
transgression
transgressions
transience
transiency
transient
transiently
transients
transistor
transistorize
transistorized
transistorizing
transistors
transit
transition
transitional
transitioned
transitions
transitive
transitively
transitiveness
transitivity
transitory
translatability
translatable
translate
translated
translates
translating
translation
translational
translations
translator
translators
translucent
transmission
transmissions
transmit
transmits
transmittal
transmitted
transmitter
transmitters
transmitting
transmogrification
transmogrify
transpacific
transparencies
transparency
transparent
transparently
transpire
transpired
transpires
transpiring
transplant
transplanted
transplanting
transplants
transponder
transponders
transport
transportability
transportation
transported
transporter
transporters
transporting
transports
transpose
transposed
transposes
transposing
transposition
trap
trapezoid
trapezoidal
trapezoids
trapped
trapper
trappers
trapping
trappings
traps
trash
trauma
traumatic
travail
travel
traveled
traveler
travelers
traveling
travelings
travels
traversal
traversals
traverse
traversed
traverses
traversing
travesties
travesty
tray
trays
treacheries
treacherous
treacherously
treachery
tread
treading
treads
treason
treasure
treasured
treasurer
treasures
treasuries
treasuring
treasury
treat
treated
treaties
treating
treatise
treatises
treatment
treatments
treats
treaty
treble
tree
trees
treetop
treetops
trek
treks
tremble
trembled
trembles
trembling
tremendous
tremendously
tremor
tremors
trench
trencher
trenches
trend
trending
trends
trespass
trespassed
trespasser
trespassers
trespasses
tress
tresses
trial
trials
triangle
triangles
triangular
triangularly
tribal
tribe
tribes
tribunal
tribunals
tribune
tribunes
tributary
tribute
tributes
trichotomy
trick
tricked
trickier
trickiest
trickiness
tricking
trickle
trickled
trickles
trickling
tricks
tricky
tried
trier
triers
tries
trifle
trifler
trifles
trifling
trigger
triggered
triggering
triggers
trigonometric
trigonometry
trigram
trigrams
trihedral
trilateral
trill
trilled
trillion
trillions
trillionth
trim
trimly
trimmed
trimmer
trimmest
trimming
trimmings
trimness
trims
trinket
trinkets
trio
trip
triple
tripled
triples
triplet
triplets
tripling
tripod
trips
triumph
triumphal
triumphant
triumphantly
triumphed
triumphing
triumphs
trivia
trivial
trivialities
triviality
trivially
trod
troll
trolley
trolleys
trolls
troop
trooper
troopers
troops
trophies
trophy
tropic
tropical
tropics
trot
trots
trouble
troubled
troublemaker
troublemakers
troubles
troubleshoot
troubleshooter
troubleshooters
troubleshooting
troubleshoots
troublesome
troublesomely
troubling
trough
trouser
trousers
trout
trowel
trowels
truant
truants
truce
truck
trucked
trucker
truckers
trucking
trucks
trudge
trudged
true
trued
truer
trues
truest
truing
truism
truisms
truly
trump
trumped
trumpet
trumpeter
trumps
truncate
truncated
truncates
truncating
truncation
truncations
trunk
trunks
trust
trusted
trustee
trustees
trustful
trustfully
trustfulness
trusting
trustingly
trusts
trustworthiness
trustworthy
trusty
truth
truthful
truthfully
truthfulness
truths
try
trying
tub
tube
tuber
tuberculosis
tubers
tubes
tubing
tubs
tuck
tucked
tucking
tucks
tuft
tufts
tug
tugs
tuition
tulip
tulips
tumble
tumbled
tumbler
tumblers
tumbles
tumbling
tumor
tumors
tumult
tumults
tumultuous
tunable
tune
tuned
tuner
tuners
tunes
tunic
tunics
tuning
tunnel
tunneled
tunnels
tuple
tuples
turban
turbans
turbulence
turbulent
turbulently
turf
turgid
turgidly
turkey
turkeys
turmoil
turmoils
turn
turnable
turnaround
turned
turner
turners
turning
turnings
turnip
turnips
turnover
turns
turpentine
turquoise
turret
turrets
turtle
turtleneck
turtles
tutor
tutored
tutorial
tutorials
tutoring
tutors
twain
twang
twas
tweed
twelfth
twelve
twelves
twenties
twentieth
twenty
twice
twig
twigs
twilight
twilights
twill
twin
twine
twined
twiner
twinkle
twinkled
twinkler
twinkles
twinkling
twins
twirl
twirled
twirler
twirling
twirls
twist
twisted
twister
twisters
twisting
twists
twitch
twitched
twitching
twitter
twittered
twittering
two
twofold
twos
tying
type
typed
typeout
types
typesetter
typewriter
typewriters
typhoid
typical
typically
typicalness
typified
typifies
typify
typifying
typing
typist
typists
typo
typographic
typographical
typographically
typography
tyrannical
tyranny
tyrant
tyrants
ubiquitous
ubiquitously
ubiquity
ugh
uglier
ugliest
ugliness
ugly
ulcer
ulcers
ultimate
ultimately
ultra
ultrasonic
umbrage
umbrella
umbrellas
umpire
umpires
unabated
unabbreviated
unable
unacceptability
unacceptable
unacceptably
unaccountable
unaccustomed
unachievable
unacknowledged
unadulterated
unaesthetically
unaffected
unaffectedly
unaffectedness
unaided
unalienability
unalienable
unalterably
unaltered
unambiguous
unambiguously
unambitious
unanalyzable
unanimity
unanimous
unanimously
unanswerable
unanswered
unanticipated
unarmed
unary
unassailable
unassigned
unassisted
unattainability
unattainable
unattended
unattractive
unattractively
unauthorized
unavailability
unavailable
unavoidable
unavoidably
unaware
unawareness
unawares
unbalanced
unbearable
unbecoming
unbelievable
unbiased
unbind
unblock
unblocked
unblocking
unblocks
unborn
unbound
unbounded
unbreakable
unbridled
unbroken
unbuffered
uncancelled
uncanny
uncapitalized
uncaught
uncertain
uncertainly
uncertainties
uncertainty
unchangeable
unchanged
unchanging
unclaimed
unclassified
uncle
unclean
uncleanly
uncleanness
unclear
uncleared
uncles
unclosed
uncomfortable
uncomfortably
uncommitted
uncommon
uncommonly
uncompromising
uncomputable
unconcerned
unconcernedly
unconditional
unconditionally
unconnected
unconscionable
unconscious
unconsciously
unconsciousness
unconstitutional
unconstrained
uncontrollability
uncontrollable
uncontrollably
uncontrolled
unconventional
unconventionally
unconvinced
unconvincing
uncoordinated
uncorrectable
uncorrected
uncountable
uncountably
uncouth
uncover
uncovered
uncovering
uncovers
undamaged
undaunted
undauntedly
undecidable
undecided
undeclared
undecomposable
undefinability
undefined
undeleted
undeniable
undeniably
under
underbrush
underdone
underestimate
underestimated
underestimates
underestimating
underestimation
underflow
underflowed
underflowing
underflows
underfoot
undergo
undergoes
undergoing
undergone
undergraduate
undergraduates
underground
underlie
underlies
underline
underlined
underlines
underling
underlings
underlining
underlinings
underloaded
underlying
undermine
undermined
undermines
undermining
underneath
underpinning
underpinnings
underplay
underplayed
underplaying
underplays
underscore
underscored
underscores
understand
understandability
understandable
understandably
understanding
understandingly
understandings
understands
understated
understood
undertake
undertaken
undertaker
undertakers
undertakes
undertaking
undertakings
undertook
underwater
underway
underwear
underwent
underworld
underwrite
underwriter
underwriters
underwrites
underwriting
undesirability
undesirable
undetectable
undetected
undetermined
undeveloped
undid
undiminished
undirected
undisciplined
undiscovered
undisturbed
undivided
undo
undocumented
undoes
undoing
undoings
undone
undoubtedly
undress
undressed
undresses
undressing
undue
unduly
uneasily
uneasiness
uneasy
uneconomic
uneconomical
unembellished
unemployed
unemployment
unencrypted
unending
unenlightening
unequal
unequaled
unequally
unequivocal
unequivocally
unessential
unevaluated
uneven
unevenly
unevenness
uneventful
unexcused
unexpanded
unexpected
unexpectedly
unexplained
unexplored
unextended
unfair
unfairly
unfairness
unfaithful
unfaithfully
unfaithfulness
unfamiliar
unfamiliarity
unfamiliarly
unfavorable
unfettered
unfinished
unfit
unfitness
unflagging
unfold
unfolded
unfolding
unfolds
unforeseen
unforgeable
unforgiving
unformatted
unfortunate
unfortunately
unfortunates
unfounded
unfriendliness
unfriendly
unfulfilled
ungrammatical
ungrateful
ungratefully
ungratefulness
ungrounded
unguarded
unguided
unhappier
unhappiest
unhappily
unhappiness
unhappy
unharmed
unhealthy
unheard
unheeded
unicorn
unicorns
unicycle
unidentified
unidirectional
unidirectionality
unidirectionally
unification
unifications
unified
unifier
unifiers
unifies
uniform
uniformed
uniformity
uniformly
uniforms
unify
unifying
unilluminating
unimaginable
unimpeded
unimplemented
unimportant
unindented
uninitialized
uninsulated
unintelligible
unintended
unintentional
unintentionally
uninteresting
uninterestingly
uninterpreted
uninterrupted
uninterruptedly
union
unionization
unionize
unionized
unionizer
unionizers
unionizes
unionizing
unions
uniprocessor
unique
uniquely
uniqueness
unison
unit
unite
united
unites
unities
uniting
units
unity
univalve
univalves
universal
universality
universally
universals
universe
universes
universities
university
unjust
unjustifiable
unjustified
unjustly
unkind
unkindly
unkindness
unknowable
unknowing
unknowingly
unknown
unknowns
unlabelled
unlawful
unlawfully
unleash
unleashed
unleashes
unleashing
unless
unlike
unlikely
unlikeness
unlimited
unlink
unlinked
unlinking
unlinks
unload
unloaded
unloading
unloads
unlock
unlocked
unlocking
unlocks
unlucky
unmanageable
unmanageably
unmanned
unmarked
unmarried
unmask
unmasked
unmatched
unmentionable
unmerciful
unmercifully
unmistakable
unmistakably
unmodified
unmoved
unnamed
unnatural
unnaturally
unnaturalness
unnecessarily
unnecessary
unneeded
unnerve
unnerved
unnerves
unnerving
unnoticed
unobservable
unobserved
unobtainable
unoccupied
unofficial
unofficially
unopened
unordered
unpack
unpacked
unpacking
unpacks
unpaid
unparalleled
unparsed
unplanned
unpleasant
unpleasantly
unpleasantness
unplug
unpopular
unpopularity
unprecedented
unpredictable
unpredictably
unprescribed
unpreserved
unprimed
unprofitable
unprojected
unprotected
unprovability
unprovable
unproven
unpublished
unqualified
unqualifiedly
unquestionably
unquestioned
unquoted
unravel
unraveled
unraveling
unravels
unreachable
unreal
unrealistic
unrealistically
unreasonable
unreasonableness
unreasonably
unrecognizable
unrecognized
unregulated
unrelated
unreliability
unreliable
unreported
unrepresentable
unresolved
unresponsive
unrest
unrestrained
unrestricted
unrestrictedly
unrestrictive
unroll
unrolled
unrolling
unrolls
unruly
unsafe
unsafely
unsanitary
unsatisfactory
unsatisfiability
unsatisfiable
unsatisfied
unsatisfying
unscrupulous
unseeded
unseen
unselected
unselfish
unselfishly
unselfishness
unsent
unsettled
unsettling
unshaken
unshared
unsigned
unskilled
unslotted
unsolvable
unsolved
unsophisticated
unsound
unspeakable
unspecified
unstable
unsteadiness
unsteady
unstructured
unsuccessful
unsuccessfully
unsuitable
unsuited
unsupported
unsure
unsurprising
unsurprisingly
unsynchronized
untagged
untapped
untenable
unterminated
untested
unthinkable
unthinking
untidiness
untidy
untie
untied
unties
until
untimely
unto
untold
untouchable
untouchables
untouched
untoward
untrained
untranslated
untreated
untried
untrue
untruthful
untruthfulness
untying
unusable
unused
unusual
unusually
unvarying
unveil
unveiled
unveiling
unveils
unwanted
unwelcome
unwholesome
unwieldiness
unwieldy
unwilling
unwillingly
unwillingness
unwind
unwinder
unwinders
unwinding
unwinds
unwise
unwisely
unwiser
unwisest
unwitting
unwittingly
unworthiness
unworthy
unwound
unwrap
unwrapped
unwrapping
unwraps
unwritten
up
upbraid
upcoming
update
updated
updater
updates
updating
upgrade
upgraded
upgrades
upgrading
upheld
uphill
uphold
upholder
upholders
upholding
upholds
upholster
upholstered
upholsterer
upholstering
upholsters
upkeep
upland
uplands
uplift
uplink
uplinks
upload
upon
upper
uppermost
upright
uprightly
uprightness
uprising
uprisings
uproar
uproot
uprooted
uprooting
uproots
upset
upsets
upshot
upshots
upside
upstairs
upstream
upturn
upturned
upturning
upturns
upward
upwards
urban
urchin
urchins
urge
urged
urgent
urgently
urges
urging
urgings
urinate
urinated
urinates
urinating
urination
urine
urn
urns
us
usability
usable
usably
usage
usages
use
used
useful
usefully
usefulness
useless
uselessly
uselessness
user
users
uses
usher
ushered
ushering
ushers
using
usual
usually
usurp
usurped
usurper
utensil
utensils
utilities
utility
utilization
utilizations
utilize
utilized
utilizes
utilizing
utmost
utopia
utopian
utopians
utter
utterance
utterances
uttered
uttering
utterly
uttermost
utters
vacancies
vacancy
vacant
vacantly
vacate
vacated
vacates
vacating
vacation
vacationed
vacationer
vacationers
vacationing
vacations
vacuo
vacuous
vacuously
vacuum
vacuumed
vacuuming
vagabond
vagabonds
vagaries
vagary
vagina
vaginas
vagrant
vagrantly
vague
vaguely
vagueness
vaguer
vaguest
vain
vainly
vale
valence
valences
valentine
valentines
vales
valet
valets
valiant
valiantly
valid
validate
validated
validates
validating
validation
validity
validly
validness
valley
valleys
valor
valuable
valuables
valuably
valuation
valuations
value
valued
valuer
valuers
values
valuing
valve
valves
vampire
van
vandalize
vandalized
vandalizes
vandalizing
vane
vanes
vanguard
vanilla
vanish
vanished
vanisher
vanishes
vanishing
vanishingly
vanities
vanity
vanquish
vanquished
vanquishes
vanquishing
vans
vantage
vapor
vaporing
vapors
variability
variable
variableness
variables
variably
variance
variances
variant
variantly
variants
variation
variations
varied
varies
varieties
variety
various
variously
varnish
varnishes
vary
varying
varyings
vase
vases
vassal
vast
vaster
vastest
vastly
vastness
vat
vats
vaudeville
vault
vaulted
vaulter
vaulting
vaults
vaunt
vaunted
veal
vector
vectorization
vectorizing
vectors
veer
veered
veering
veers
vegetable
vegetables
vegetarian
vegetarians
vegetate
vegetated
vegetates
vegetating
vegetation
vegetative
vehemence
vehement
vehemently
vehicle
vehicles
vehicular
veil
veiled
veiling
veils
vein
veined
veining
veins
velocities
velocity
velvet
vendor
vendors
venerable
veneration
vengeance
venial
venison
venom
venomous
venomously
vent
vented
ventilate
ventilated
ventilates
ventilating
ventilation
ventricle
ventricles
vents
venture
ventured
venturer
venturers
ventures
venturing
venturings
veracity
veranda
verandas
verb
verbal
verbalize
verbalized
verbalizes
verbalizing
verbally
verbose
verbs
verdict
verdure
verge
verger
verges
verifiability
verifiable
verification
verifications
verified
verifier
verifiers
verifies
verify
verifying
verily
veritable
vermin
vernacular
versa
versatile
versatility
verse
versed
verses
versing
version
versions
versus
vertebrate
vertebrates
vertex
vertical
vertically
verticalness
vertices
very
vessel
vessels
vest
vested
vestige
vestiges
vestigial
vests
veteran
veterans
veterinarian
veterinarians
veterinary
veto
vetoed
vetoer
vetoes
vex
vexation
vexed
vexes
vexing
via
viability
viable
viably
vial
vials
vibrate
vibrated
vibrating
vibration
vibrations
vibrator
vice
viceroy
vices
vicinity
vicious
viciously
viciousness
vicissitude
vicissitudes
victim
victimize
victimized
victimizer
victimizers
victimizes
victimizing
victims
victor
victories
victorious
victoriously
victors
victory
victual
victualer
victuals
video
videotape
videotapes
vie
vied
vier
vies
view
viewable
viewed
viewer
viewers
viewing
viewpoint
viewpoints
views
vigilance
vigilant
vigilante
vigilantes
vigilantly
vignette
vignettes
vigor
vigorous
vigorously
vile
vilely
vileness
vilification
vilifications
vilified
vilifies
vilify
vilifying
villa
village
villager
villagers
villages
villain
villainous
villainously
villainousness
villains
villainy
villas
vindicate
vindicated
vindication
vindictive
vindictively
vindictiveness
vine
vinegar
vines
vineyard
vineyards
vintage
violate
violated
violates
violating
violation
violations
violator
violators
violence
violent
violently
violet
violets
violin
violinist
violinists
violins
viper
vipers
virgin
virginity
virgins
virtual
virtually
virtue
virtues
virtuoso
virtuosos
virtuous
virtuously
virulent
virus
viruses
visa
visage
visas
viscount
viscounts
viscous
visibility
visible
visibly
vision
visionary
visions
visit
visitation
visitations
visited
visiting
visitor
visitors
visits
visor
visors
vista
vistas
visual
visualize
visualized
visualizer
visualizes
visualizing
visually
vita
vitae
vital
vitality
vitally
vitals
vivid
vividly
vividness
vizier
vocabularies
vocabulary
vocal
vocally
vocals
vocation
vocational
vocationally
vocations
vogue
voice
voiced
voicer
voicers
voices
voicing
void
voided
voider
voiding
voids
volatile
volatilities
volatility
volcanic
volcano
volcanos
volition
volley
volleyball
volleyballs
volt
voltage
voltages
volts
volume
volumes
voluntarily
voluntary
volunteer
volunteered
volunteering
volunteers
vomit
vomited
vomiting
vomits
vortex
vote
voted
voter
voters
votes
voting
votive
vouch
voucher
vouchers
vouches
vouching
vow
vowed
vowel
vowels
vower
vowing
vows
voyage
voyaged
voyager
voyagers
voyages
voyaging
voyagings
vulgar
vulgarly
vulnerabilities
vulnerability
vulnerable
vulture
vultures
wacky
wade
waded
wader
wades
wading
wafer
wafers
waffle
waffles
waft
wag
wage
waged
wager
wagers
wages
waging
wagon
wagoner
wagons
wags
wail
wailed
wailing
wails
waist
waistcoat
waistcoats
waists
wait
waited
waiter
waiters
waiting
waitress
waitresses
waits
waive
waived
waiver
waiverable
waives
waiving
wake
waked
waken
wakened
wakening
wakes
wakeup
waking
wales
walk
walked
walker
walkers
walking
walks
wall
walled
wallet
wallets
walling
wallow
wallowed
wallowing
wallows
walnut
walnuts
walrus
walruses
waltz
waltzed
waltzes
waltzing
wan
wand
wander
wandered
wanderer
wanderers
wandering
wanderings
wanders
wane
waned
wanes
waning
wanly
want
wanted
wanting
wanton
wantonly
wantonness
wants
war
warble
warbled
warbler
warbles
warbling
ward
warden
wardens
warder
wardrobe
wardrobes
wards
ware
warehouse
warehouses
warehousing
wares
warfare
warily
wariness
warlike
warm
warmed
warmer
warmers
warmest
warming
warmly
warms
warmth
warn
warned
warner
warning
warningly
warnings
warns
warp
warped
warping
warps
warrant
warranted
warranties
warranting
warrants
warranty
warred
warring
warrior
warriors
wars
warship
warships
wart
wartime
warts
wary
was
wash
washed
washer
washers
washes
washing
washings
wasp
wasps
waste
wasted
wasteful
wastefully
wastefulness
wastes
wasting
watch
watched
watcher
watchers
watches
watchful
watchfully
watchfulness
watching
watchings
watchman
watchword
watchwords
water
watered
waterfall
waterfalls
watering
waterings
waterproof
waterproofing
waterway
waterways
watery
wave
waved
waveform
waveforms
wavefront
wavefronts
waveguides
wavelength
wavelengths
waver
wavers
waves
waving
wax
waxed
waxen
waxer
waxers
waxes
waxing
waxy
way
ways
wayside
wayward
we
weak
weaken
weakened
weakening
weakens
weaker
weakest
weakly
weakness
weaknesses
wealth
wealthiest
wealths
wealthy
wean
weaned
weaning
weapon
weapons
wear
wearable
wearer
wearied
wearier
weariest
wearily
weariness
wearing
wearisome
wearisomely
wears
weary
wearying
weasel
weasels
weather
weathercock
weathercocks
weathered
weathering
weathers
weave
weaver
weaves
weaving
web
webs
wedded
wedding
weddings
wedge
wedged
wedges
wedging
wedlock
weds
wee
weed
weeds
week
weekend
weekends
weekly
weep
weeper
weeping
weeps
weigh
weighed
weighing
weighings
weighs
weight
weighted
weighting
weights
weighty
weird
weirdly
welcome
welcomed
welcomes
welcoming
weld
welded
welder
welding
welds
welfare
well
welled
welling
welsh
wench
wenches
went
wept
were
west
westbound
western
westerner
westerners
westward
westwards
wet
wetly
wetness
wets
wetted
wetter
wettest
wetting
whack
whacked
whacking
whacks
whale
whaler
whales
whaling
wharf
wharves
what
whatever
whatsoever
wheat
wheaten
wheel
wheeled
wheeler
wheelers
wheeling
wheelings
wheels
whelp
when
whence
whenever
where
whereabouts
whereas
whereby
wherein
whereupon
wherever
whether
which
whichever
while
whim
whimper
whimpered
whimpering
whimpers
whims
whimsical
whimsically
whimsies
whimsy
whine
whined
whines
whining
whip
whipped
whipper
whippers
whipping
whippings
whips
whirl
whirled
whirling
whirlpool
whirlpools
whirls
whirlwind
whirr
whirring
whisk
whisked
whisker
whiskers
whiskey
whisking
whisks
whisper
whispered
whispering
whisperings
whispers
whistle
whistled
whistler
whistlers
whistles
whistling
whit
white
whitely
whiten
whitened
whitener
whiteners
whiteness
whitening
whitens
whiter
whites
whitespace
whitest
whitewash
whitewashed
whiting
whittle
whittled
whittles
whittling
whiz
whizzed
whizzes
whizzing
who
whoever
whole
wholehearted
wholeheartedly
wholeness
wholes
wholesale
wholesaler
wholesalers
wholesome
wholesomeness
wholly
whom
whomever
whoop
whooped
whooping
whoops
whore
whores
whorl
whorls
whose
why
wick
wicked
wickedly
wickedness
wicker
wicks
wide
wideband
widely
widen
widened
widener
widening
widens
wider
widespread
widest
widget
widow
widowed
widower
widowers
widows
width
widths
wield
wielded
wielder
wielding
wields
wife
wifely
wig
wigs
wigwam
wild
wildcat
wildcats
wilder
wilderness
wildest
wildly
wildness
wile
wiles
wiliness
will
willed
willful
willfully
willing
willingly
willingness
willow
willows
wilt
wilted
wilting
wilts
wily
win
wince
winced
winces
wincing
wind
winded
winder
winders
winding
windmill
windmills
window
windows
winds
windy
wine
wined
winer
winers
wines
wing
winged
winging
wings
wining
wink
winked
winker
winking
winks
winner
winners
winning
winningly
winnings
wins
winter
wintered
wintering
wintry
wipe
wiped
wiper
wipers
wipes
wiping
wire
wired
wireless
wires
wiretap
wiretappers
wiretapping
wiretaps
wiriness
wiring
wiry
wisdom
wisdoms
wise
wised
wisely
wiser
wisest
wish
wished
wisher
wishers
wishes
wishful
wishing
wisp
wisps
wistful
wistfully
wistfulness
wit
witch
witchcraft
witches
witching
with
withal
withdraw
withdrawal
withdrawals
withdrawing
withdrawn
withdraws
withdrew
wither
withers
withheld
withhold
withholder
withholders
withholding
withholdings
withholds
within
without
withstand
withstanding
withstands
withstood
witness
witnessed
witnesses
witnessing
wits
witty
wives
wizard
wizards
woe
woeful
woefully
woke
wolf
wolves
woman
womanhood
womanly
womb
wombs
women
won
wonder
wondered
wonderful
wonderfully
wonderfulness
wondering
wonderingly
wonderment
wonders
wondrous
wondrously
wont
wonted
woo
wood
woodchuck
woodchucks
woodcock
woodcocks
wooded
wooden
woodenly
woodenness
woodland
woodman
woodpecker
woodpeckers
woodwork
woodworking
woody
wooed
wooer
woof
woofed
woofer
woofers
woofing
woofs
wooing
wool
woolen
woolly
wools
woos
word
worded
wordily
wordiness
wording
words
wordy
wore
work
workable
workably
workbench
workbenches
workbook
workbooks
worked
worker
workers
workhorse
workhorses
working
workingman
workings
workload
workman
workmanship
workmen
works
workshop
workshops
workspace
workstation
workstations
world
worldliness
worldly
worlds
worldwide
worm
wormed
worming
worms
worn
worried
worrier
worriers
worries
worrisome
worry
worrying
worryingly
worse
worship
worshiped
worshiper
worshipful
worshiping
worships
worst
worsted
worth
worthiest
worthiness
worthless
worthlessness
worths
worthwhile
worthwhileness
worthy
would
wound
wounded
wounding
wounds
wove
woven
wrangle
wrangled
wrangler
wrap
wraparound
wrapped
wrapper
wrappers
wrapping
wrappings
wraps
wrath
wreak
wreaks
wreath
wreathed
wreathes
wreck
wreckage
wrecked
wrecker
wreckers
wrecking
wrecks
wren
wrench
wrenched
wrenches
wrenching
wrens
wrest
wrestle
wrestler
wrestles
wrestling
wrestlings
wretch
wretched
wretchedness
wretches
wriggle
wriggled
wriggler
wriggles
wriggling
wring
wringer
wrings
wrinkle
wrinkled
wrinkles
wrist
wrists
wristwatch
wristwatches
writ
writable
write
writer
writers
writes
writhe
writhed
writhes
writhing
writing
writings
writs
written
wrong
wronged
wronging
wrongly
wrongs
wrote
wrought
wrung
yank
yanked
yanking
yanks
yard
yards
yardstick
yardsticks
yarn
yarns
yawn
yawner
yawning
yea
year
yearly
yearn
yearned
yearning
yearnings
years
yeas
yeast
yeasts
yell
yelled
yeller
yelling
yellow
yellowed
yellower
yellowest
yellowing
yellowish
yellowness
yellows
yelp
yelped
yelping
yelps
yeoman
yeomen
yes
yesterday
yesterdays
yet
yield
yielded
yielding
yields
yoke
yokes
yon
yonder
you
young
younger
youngest
youngly
youngster
youngsters
your
yours
yourself
yourselves
youth
youthes
youthful
youthfully
youthfulness
zeal
zealous
zealously
zealousness
zebra
zebras
zenith
zero
zeroed
zeroes
zeroing
zeros
zeroth
zest
zigzag
zillions
zinc
zodiac
zonal
zonally
zone
zoned
zones
zoning
zoo
zoological
zoologically
zoom
zooms
zoos
EOF

if test x$PERL != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/perl
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

use strict;

# read dictionary
my %dict = ();
open(DICT, "<__temp") or
    die "Error, unable to open __temp\n";
while (<DICT>) {
    chomp;
    $dict{$_} = 1;
}
close(DICT);

while (<STDIN>) {
    chomp;
    print "$_\n" if (!$dict{$_});
}
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/python
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/
# From Fred Bremmer

import sys

def main():
    dict = {}
    dict_has_key = dict.has_key
    for line in open("__temp").xreadlines():
        word = line[:-1]
        if word: dict[word] = 1

    for line in sys.stdin.xreadlines():
        word = line[:-1]
        if word:
            if not dict_has_key(word): print word

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/tclsh
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/
# from: Miguel Sofer
# some modifications suggested by Kristoffer Lawson

proc main {} {
    set 1 [open "__temp" r]
    foreach  2 [read $1 [file size "__temp"]] {
	set $2 1
    }
    close $1

    fconfigure stdout -buffering full
    while {[gets stdin 1] >= 0} {
        if {[catch {set $1}]} {
            puts $1
        }
    }
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

BEGIN {
    delete ARGV;
    while (getline < "__temp") {
	dict[$0] = 1;
    }
}
{
    if (!dict[$1]) {
	print $1
    }
}
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
func main () {
  var f, i, words, dict = {};
  
  f = open ("__temp", "r");
  words = fgetf (f, 1);
  close (f);
  for (i = 0; i < #words; i++)
    dict {words [i]} = 1;
  
  words = getf (1);
  for (i = 0; i < #words; i++)
    if (!(words [i] in dict))
      putln (words[i]);
}
main ();
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 22.
if test $start_test_number -le 22; then

######################################################
echo 
echo '+++++ Test #22: Statistical Moments (N=150):  +++++'

cat <<'EOF' >$temp
1
2
3
4
5
6
7
8
9
10
11
12
13
14
15
16
17
18
19
20
21
22
23
24
25
26
27
28
29
30
31
32
33
34
35
36
37
38
39
40
41
42
43
44
45
46
47
48
49
50
51
52
53
54
55
56
57
58
59
60
61
62
63
64
65
66
67
68
69
70
71
72
73
74
75
76
77
78
79
80
81
82
83
84
85
86
87
88
89
90
91
92
93
94
95
96
97
98
99
100
101
102
103
104
105
106
107
108
109
110
111
112
113
114
115
116
117
118
119
120
121
122
123
124
125
126
127
128
129
130
131
132
133
134
135
136
137
138
139
140
141
142
143
144
145
146
147
148
149
150
151
152
153
154
155
156
157
158
159
160
161
162
163
164
165
166
167
168
169
170
171
172
173
174
175
176
177
178
179
180
181
182
183
184
185
186
187
188
189
190
191
192
193
194
195
196
197
198
199
200
201
202
203
204
205
206
207
208
209
210
211
212
213
214
215
216
217
218
219
220
221
222
223
224
225
226
227
228
229
230
231
232
233
234
235
236
237
238
239
240
241
242
243
244
245
246
247
248
249
250
251
252
253
254
255
256
257
258
259
260
261
262
263
264
265
266
267
268
269
270
271
272
273
274
275
276
277
278
279
280
281
282
283
284
285
286
287
288
289
290
291
292
293
294
295
296
297
298
299
300
301
302
303
304
305
306
307
308
309
310
311
312
313
314
315
316
317
318
319
320
321
322
323
324
325
326
327
328
329
330
331
332
333
334
335
336
337
338
339
340
341
342
343
344
345
346
347
348
349
350
351
352
353
354
355
356
357
358
359
360
361
362
363
364
365
366
367
368
369
370
371
372
373
374
375
376
377
378
379
380
381
382
383
384
385
386
387
388
389
390
391
392
393
394
395
396
397
398
399
400
401
402
403
404
405
406
407
408
409
410
411
412
413
414
415
416
417
418
419
420
421
422
423
424
425
426
427
428
429
430
431
432
433
434
435
436
437
438
439
440
441
442
443
444
445
446
447
448
449
450
451
452
453
454
455
456
457
458
459
460
461
462
463
464
465
466
467
468
469
470
471
472
473
474
475
476
477
478
479
480
481
482
483
484
485
486
487
488
489
490
491
492
493
494
495
496
497
498
499
500
EOF

rm -f $input; touch $input

i=0
while test $i -lt 150;do cat $temp >>$input; i=`expr $i + 1`;done

if test x$PERL != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/perl
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

use strict;

my @nums = <STDIN>;
my $sum = 0;
foreach (@nums) { $sum += $_ }
my $n = scalar(@nums);
my $mean = $sum/$n;
my $average_deviation = 0;
my $standard_deviation = 0;
my $variance = 0;
my $skew = 0;
my $kurtosis = 0;
foreach (@nums) {
    my $deviation = $_ - $mean;
    $average_deviation += abs($deviation);
    $variance += $deviation**2;
    $skew += $deviation**3;
    $kurtosis += $deviation**4;
}
$average_deviation /= $n;
$variance /= ($n - 1);
$standard_deviation = sqrt($variance);

if ($variance) {
    $skew /= ($n * $variance * $standard_deviation);
    $kurtosis = $kurtosis/($n * $variance * $variance) - 3.0;
}

@nums = sort { $a <=> $b } @nums;
my $mid = int($n/2);
my $median = ($n % 2) ? $nums[$mid] : ($nums[$mid] + $nums[$mid-1])/2;

printf("n:                  %d\n", $n);
printf("median:             %f\n", $median);
printf("mean:               %f\n", $mean);
printf("average_deviation:  %f\n", $average_deviation);
printf("standard_deviation: %f\n", $standard_deviation);
printf("variance:           %f\n", $variance);
printf("skew:               %f\n", $skew);
printf("kurtosis:           %f\n", $kurtosis);
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/python
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

import sys, string, math, operator

def main():
    sum = 0
    nums = []
    atof = string.atof

    nums = map(atof, sys.stdin.readlines())
    sum = reduce(operator.add, nums)

    n = len(nums)
    mean = sum/n
    average_deviation = 0
    standard_deviation = 0
    variance = 0
    skew = 0
    kurtosis = 0

    for num in nums:
        deviation = num - mean
        average_deviation += abs(deviation)
        variance += deviation**2;
        skew += deviation**3;
        kurtosis += deviation**4

    average_deviation /= n
    variance /= (n - 1)
    standard_deviation = math.sqrt(variance)

    if variance > 0.0:
        skew /= (n * variance * standard_deviation)
        kurtosis = kurtosis/(n * variance * variance) - 3.0

    nums.sort()
    mid = n / 2

    if (n % 2) == 0:
        median = (nums[mid] + nums[mid-1])/2
    else:
        median = nums[mid]

    print "n:                  %d" % n
    print "median:             %f" % median
    print "mean:               %f" % mean
    print "average_deviation:  %f" % average_deviation
    print "standard_deviation: %f" % standard_deviation
    print "variance:           %f" % variance
    print "skew:               %f" % skew
    print "kurtosis:           %f" % kurtosis

main()

EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
#!/usr/local/bin/tclsh
# $Id: compare.tst,v 1.9 2002/05/17 19:39:26 vmakarov Exp $
# http://www.bagley.org/~doug/shootout/

proc main {} {
    set sum 0.0
    set nums [read stdin]
    foreach num $nums {
        set sum [expr {$sum + $num}]
    }
    set n [llength $nums]
    set mean [expr {$sum / $n}]
    set average_deviation 0.0
    set standard_deviation 0.0
    set variance 0.0
    set skew 0.0
    set kurtosis 0.0
    
    foreach num $nums {
	set deviation [expr {$num - $mean}]
	set average_deviation [expr {$average_deviation + abs($deviation)}]
	set variance [expr {$variance + pow($deviation, 2)}]
	set skew [expr {$skew + pow($skew, 3)}]
	set kurtosis [expr {$kurtosis + pow($deviation, 4)}]
    }

    set average_deviation [expr {$average_deviation / $n}]
    set variance [expr {$variance / ($n - 1)}]
    set standard_deviation [expr {sqrt($variance)}]

    if {$variance} {
	set skew [expr {$skew / ($n * $variance * $standard_deviation)}]
	set kurtosis [expr {$kurtosis / ($n * $variance * $variance) - 3.0}]
    }

    set nums [lsort -integer $nums]
    set mid [expr {int($n / 2)}]
    if [expr {$n % 2}] {
	set median [lindex $nums $mid]
    } else {
	set a [lindex $nums $mid]
	set b [lindex $nums [expr {$mid - 1}]]
	set median [expr {($a + $b) / 2.0}]
    }
	
    puts [format "n:                  %d" $n]
    puts [format "median:             %f" $median]
    puts [format "mean:               %f" $mean]
    puts [format "average_deviation:  %f" $average_deviation]
    puts [format "standard_deviation: %f" $standard_deviation]
    puts [format "variance:           %f" $variance]
    puts [format "skew:               %f" $skew]
    puts [format "kurtosis:           %f" $kurtosis]
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
func main () {
  var i, sum = 0.0;
  var lns = getf (1);
  var nums = [#lns:0];
  
  for (i = 0; i < #lns; i++) {
    nums [i] = lns [i] + 0.0;
    sum += nums [i];
  }
  
  var n = #nums;
  var mean = sum / n;
  var average_deviation = 0.0;
  var standard_deviation = 0.0;
  var variance = 0.0;
  var skew = 0.0;
  var kurtosis = 0.0;
  var deviation, pow2;
  
  for (i = 0; i < n; i++) {
    deviation = nums [i] - mean;
    average_deviation += deviation < 0.0 ? -deviation : deviation;
    pow2 = deviation * deviation;
    variance += pow2;
    skew += deviation * pow2;
    kurtosis += pow2 * pow2;
  }
  average_deviation /= n;
  variance /= (n - 1);
  standard_deviation = sqrt (variance);
  
  if (variance > 0.0) {
    skew /= (n * variance * standard_deviation);
    kurtosis = kurtosis / (n * variance * variance) - 3.0;
  }
  sort (nums);
  
  var mid = n / 2;
  
  var median = (n % 2 == 0 ? (nums [mid] + nums [mid - 1]) / 2 : nums [mid]);
  
  putln ("n:                  ", n);
  putln ("median:             ", vector (median, "%f"));
  putln ("mean:               ", vector (mean, "%f"));
  putln ("average_deviation:  ", vector (average_deviation, "%f"));
  putln ("standard_deviation: ", vector (standard_deviation, "%f"));
  putln ("variance:           ", vector (variance, "%f"));
  putln ("skew:               ", vector (skew, "%f"));
  putln ("kurtosis:           ", vector (kurtosis, "%f"));
}
main();
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 23.
if test $start_test_number -le 23; then

######################################################
echo 
echo '+++++ Test #23: String Concatenation:  +++++'

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

use strict;

my $NUM = $ARGV[0];
$NUM = 1 if ($NUM < 1);

my $str = "";
$str .= "hello\n" foreach (1..$NUM);
print length($str),"\n";

EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest 500000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# from Brad Knotwell

import sys,cStringIO

def main():
    n = int(sys.argv[1])
    str = cStringIO.StringIO()
    for i in xrange(0,n):
        str.write('hello\n')

    print str.tell()

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest 500000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# from: Kristoffer Lawson

proc main {n} {
    incr n
    while {[incr n -1]} {
        append str "hello\n"
    }
    puts [string length $str]
}

main [lindex $argv 0]
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest 500000 >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

# AWK is too slow therefore it is disqualified.

cat <<'EOF' >$ftest
var i, n = argv [0] < 1 ? 1 : int (argv [0]);

var str = new "";
var add = "hello\n";
for (i = 0; i < n; i++)
  insv (str, add, -1);

putln (#str);
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest 500000 </dev/null >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 24.
if test $start_test_number -le 24; then

######################################################
echo 
echo '+++++ Test #24: Sum a File of Numbers (N=1000):  +++++'

cat <<'EOF' >$temp
1
2
3
4
5
6
7
8
9
10
11
12
13
14
15
16
17
18
19
20
21
22
23
24
25
26
27
28
29
30
31
32
33
34
35
36
37
38
39
40
41
42
43
44
45
46
47
48
49
50
51
52
53
54
55
56
57
58
59
60
61
62
63
64
65
66
67
68
69
70
71
72
73
74
75
76
77
78
79
80
81
82
83
84
85
86
87
88
89
90
91
92
93
94
95
96
97
98
99
100
101
102
103
104
105
106
107
108
109
110
111
112
113
114
115
116
117
118
119
120
121
122
123
124
125
126
127
128
129
130
131
132
133
134
135
136
137
138
139
140
141
142
143
144
145
146
147
148
149
150
151
152
153
154
155
156
157
158
159
160
161
162
163
164
165
166
167
168
169
170
171
172
173
174
175
176
177
178
179
180
181
182
183
184
185
186
187
188
189
190
191
192
193
194
195
196
197
198
199
200
201
202
203
204
205
206
207
208
209
210
211
212
213
214
215
216
217
218
219
220
221
222
223
224
225
226
227
228
229
230
231
232
233
234
235
236
237
238
239
240
241
242
243
244
245
246
247
248
249
250
251
252
253
254
255
256
257
258
259
260
261
262
263
264
265
266
267
268
269
270
271
272
273
274
275
276
277
278
279
280
281
282
283
284
285
286
287
288
289
290
291
292
293
294
295
296
297
298
299
300
301
302
303
304
305
306
307
308
309
310
311
312
313
314
315
316
317
318
319
320
321
322
323
324
325
326
327
328
329
330
331
332
333
334
335
336
337
338
339
340
341
342
343
344
345
346
347
348
349
350
351
352
353
354
355
356
357
358
359
360
361
362
363
364
365
366
367
368
369
370
371
372
373
374
375
376
377
378
379
380
381
382
383
384
385
386
387
388
389
390
391
392
393
394
395
396
397
398
399
400
401
402
403
404
405
406
407
408
409
410
411
412
413
414
415
416
417
418
419
420
421
422
423
424
425
426
427
428
429
430
431
432
433
434
435
436
437
438
439
440
441
442
443
444
445
446
447
448
449
450
451
452
453
454
455
456
457
458
459
460
461
462
463
464
465
466
467
468
469
470
471
472
473
474
475
476
477
478
479
480
481
482
483
484
485
486
487
488
489
490
491
492
493
494
495
496
497
498
499
500
501
502
503
504
505
506
507
508
509
510
511
512
513
514
515
516
517
518
519
520
521
522
523
524
525
526
527
528
529
530
531
532
533
534
535
536
537
538
539
540
541
542
543
544
545
546
547
548
549
550
551
552
553
554
555
556
557
558
559
560
561
562
563
564
565
566
567
568
569
570
571
572
573
574
575
576
577
578
579
580
581
582
583
584
585
586
587
588
589
590
591
592
593
594
595
596
597
598
599
600
601
602
603
604
605
606
607
608
609
610
611
612
613
614
615
616
617
618
619
620
621
622
623
624
625
626
627
628
629
630
631
632
633
634
635
636
637
638
639
640
641
642
643
644
645
646
647
648
649
650
651
652
653
654
655
656
657
658
659
660
661
662
663
664
665
666
667
668
669
670
671
672
673
674
675
676
677
678
679
680
681
682
683
684
685
686
687
688
689
690
691
692
693
694
695
696
697
698
699
700
701
702
703
704
705
706
707
708
709
710
711
712
713
714
715
716
717
718
719
720
721
722
723
724
725
726
727
728
729
730
731
732
733
734
735
736
737
738
739
740
741
742
743
744
745
746
747
748
749
750
751
752
753
754
755
756
757
758
759
760
761
762
763
764
765
766
767
768
769
770
771
772
773
774
775
776
777
778
779
780
781
782
783
784
785
786
787
788
789
790
791
792
793
794
795
796
797
798
799
800
801
802
803
804
805
806
807
808
809
810
811
812
813
814
815
816
817
818
819
820
821
822
823
824
825
826
827
828
829
830
831
832
833
834
835
836
837
838
839
840
841
842
843
844
845
846
847
848
849
850
851
852
853
854
855
856
857
858
859
860
861
862
863
864
865
866
867
868
869
870
871
872
873
874
875
876
877
878
879
880
881
882
883
884
885
886
887
888
889
890
891
892
893
894
895
896
897
898
899
900
901
902
903
904
905
906
907
908
909
910
911
912
913
914
915
916
917
918
919
920
921
922
923
924
925
926
927
928
929
930
931
932
933
934
935
936
937
938
939
940
941
942
943
944
945
946
947
948
949
950
951
952
953
954
955
956
957
958
959
960
961
962
963
964
965
966
967
968
969
970
971
972
973
974
975
976
977
978
979
980
981
982
983
984
985
986
987
988
989
990
991
992
993
994
995
996
997
998
999
1000
EOF

rm -f $input; touch $input

i=0
while test $i -lt 1000;do cat $temp >>$input; i=`expr $i + 1`;done

if test x$PERL != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

use integer;
shift;
while (<>) { $tot += $_ }
print "$tot\n";
EOF
  echo PERL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PERL $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$PYTHON != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# with help from Mark Baker

import sys

def main():
    count = 0
    for line in sys.stdin.xreadlines():
        count = count + int(line)
    print count

main()
EOF
  echo PYTHON:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $PYTHON $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$TCLSH != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/
# from: Miguel Sofer

proc main {} {
    set sum 0
    while {[gets stdin line]> 0} {
	incr sum $line
    }
    puts $sum
}

main
EOF
  echo TCL:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $TCLSH $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

if test x$AWK != x; then
  cat <<'EOF' >$ftest
# http://www.bagley.org/~doug/shootout/

BEGIN { delete ARGV; tot = 0 }
{ tot += $1 }
END { print tot }
EOF
  echo AWK:
  if test "x$NECHO" != x;then $NECHO "   ";fi
  if time $AWK -f $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi
fi

cat <<'EOF' >$ftest
func main () {
  var count = 0;
  
  for (;;)
    try {
      count += scan ();
    } catch (invcalls.eof) {
      break;
    }
  putln (count);
}

main ();
EOF
echo DINO:
if test "x$NECHO" != x;then $NECHO "   ";fi
if time $DINO $ftest <$input >$temp2 2>&1;then fgrep user $temp2;else echo FAILED;fi

fi

# Test 25.
if test $start_test_number -le 25; then

######################################################
echo 
echo '+++++ Test #25: Word Frequency (N=20):  +++++'

cat <<'EOF' >$temp

                                      1515
                                   THE PRINCE
                             by Nicolo Machiavelli
                          translated by W. K. Marriott
                              CHAPTER I
             HOW MANY KINDS OF PRINCIPALITIES THERE ARE,
                 AND BY WHAT MEANS THEY ARE ACQUIRED

  ALL STATES, all powers, that have held and hold rule over men have
been and are either republics or principalities.
  Principalities are either hereditary, in which the family has been
long established; or they are new.
  The new are either entirely new, as was Milan to Francesco Sforza,
or they are, as it were, members annexed to the hereditary state of
the prince who has acquired them, as was the kingdom of Naples to that
of the King of Spain.
  Such dominions thus acquired are either accustomed to live under a
prince, or to live in freedom; and are acquired either by the arms
of the prince himself, or of others, or else by fortune or by ability.
                              CHAPTER II
                 CONCERNING HEREDITARY PRINCIPALITIES

  I WILL leave out all discussion on republics, inasmuch as in another
place I have written of them at length, and will address myself only
to principalities. In doing so I will keep to the order indicated
above, and discuss how such principalities are to be ruled and
preserved.
  I say at once there are fewer difficulties in holding hereditary
states, and those long accustomed to the family of their prince,
than new ones; for it is sufficient only not to transgress the customs
of his ancestors, and to deal prudently with circumstances as they
arise, for a prince of average powers to maintain himself in his
state, unless he be deprived of it by some extraordinary and excessive
force; and if he should be so deprived of it, whenever anything
sinister happens to the usurper, he will regain it.
  We have in Italy, for example, the Duke of Ferrara, who could not
have withstood the attacks of the Venetians in '84, nor those of
Pope Julius in '10, unless he had been long established in his
dominions. For the hereditary prince has less cause and less necessity
to offend; hence it happens that he will be more loved; and unless
extraordinary vices cause him to be hated, it is reasonable to
expect that his subjects will be naturally well disposed towards
him; and in the antiquity and duration of his rule the memories and
motives that make for change are lost, for one change always leaves
the toothing for another.
                             CHAPTER III
                   CONCERNING MIXED PRINCIPALITIES

  BUT the difficulties occur in a new principality. And firstly, if it
be not entirely new, but is, as it were, a member of a state which,
taken collectively, may be called composite, the changes arise chiefly
from an inherent difficulty which there is in all new
principalities; for men change their rulers willingly, hoping to
better themselves, and this hope induces them to take up arms
against him who rules: wherein they are deceived, because they
afterwards find by experience they have gone from bad to worse. This
follows also on another natural and common necessity, which always
causes a new prince to burden those who have submitted to him with his
soldiery and with infinite other hardships which he must put upon
his new acquisition.
  In this way you have enemies in all those whom you have injured in
seizing that principality, and you are not able to keep those
friends who put you there because of your not being able to satisfy
them in the way they expected, and you cannot take strong measures
against them, feeling bound to them. For, although one may be very
strong in armed forces, yet in entering a province one has always need
of the goodwill of the natives.
  For these reasons Louis XII, King of France, quickly occupied Milan,
and as quickly lost it; and to turn him out the first time it only
needed Lodovico's own forces; because those who had opened the gates
to him, finding themselves deceived in their hopes of future
benefit, would not endure the ill-treatment of the new prince. It is
very true that, after acquiring rebellious provinces a second time,
they are not so lightly lost afterwards, because the prince, with
little reluctance, takes the opportunity of the rebellion to punish
the delinquents, to clear out the suspects, and to strengthen
himself in the weakest places. Thus to cause France to lose Milan
the first time it was enough for the Duke Lodovico to raise
insurrections on the borders; but to cause him to lose it a second
time it was necessary to bring the whole world against him, and that
his armies should be defeated and driven out of Italy; which
followed from the causes above mentioned.
  Nevertheless Milan was taken from France both the first and the
second time. The general reasons for the first have been discussed; it
remains to name those for the second, and to see what resources he
had, and what any one in his situation would have had for
maintaining himself more securely in his acquisition than did the King
of France.
  Now I say that those dominions which, when acquired, are added to an
ancient state by him who acquires them, are either of the same country
and language, or they are not. When they are, it is easier to hold
them, especially when they have not been accustomed to
self-government; and to hold them securely it is enough to have
destroyed the family of the prince who was ruling them; because the
two peoples, preserving in other things the old conditions, and not
being unlike in customs, will live quietly together, as one has seen
in Brittany, Burgundy, Gascony, and Normandy, which have been bound to
France for so long a time: and, although there may be some
difference in language, nevertheless the customs are alike, and the
people will easily be able to get on amongst themselves. He who has
annexed them, if he wishes to hold them, has only to bear in mind
two considerations: the one, that the family of their former lord is
extinguished; the other, that neither their laws nor their taxes are
altered, so that in a very short time they will become entirely one
body with the old principality.
  But when states are acquired in a country differing in language,
customs, or laws, there are difficulties, and good fortune and great
energy are needed to hold them, and one of the greatest and most
real helps would be that he who has acquired them should go and reside
there. This would make his position more secure and durable, as it has
made that of the Turk in Greece, who, notwithstanding all the other
measures taken by him for holding that state, if he had not settled
there, would not have been able to keep it. Because, if one is on
the spot, disorders are seen as they spring up, and one can quickly
remedy them; but if one is not at hand, they heard of only when they
are one can no longer remedy them. Besides this, the country is not
pillaged by your officials; the subjects are satisfied by prompt
recourse to the prince; thus, wishing to be good, they have more cause
to love him, and wishing to be otherwise, to fear him. He who would
attack that state from the outside must have the utmost caution; as
long as the prince resides there it can only be wrested from him
with the greatest difficulty.
  The other and better course is to send colonies to one or two
places, which may be as keys to that state, for it necessary either to
do this or else to keep there a great number of cavalry and
infantry. A prince does not spend much on colonies, for with little or
no expense he can send them out and keep them there, and he offends
a minority only of the citizens from whom he takes lands and houses to
give them to the new inhabitants; and those whom he offends, remaining
poor and scattered, are never able to injure him; whilst the rest
being uninjured are easily kept quiet, and at the same time are
anxious not to err for fear it should happen to them as it has to
those who have been despoiled. In conclusion, I say that these
colonies are not costly, they are more faithful, they injure less, and
the injured, as has been said, I being poor and scattered, cannot
hurt. Upon this, one has to remark that men ought either to be well
treated or crushed, because they can avenge themselves of lighter
injuries, of more serious ones they cannot; therefore the injury
that is to be done to a man ought to be of such a kind that one does
not stand in fear of revenge.
  But in maintaining armed men there in place of colonies one spends
much more, having to consume on the garrison all income from the
state, so that the acquisition turns into a loss, and many more are
exasperated, because the whole state is injured; through the
shifting of the garrison up and down all become acquainted with
hardship, and all become hostile, and they are enemies who, whilst
beaten on their own ground, are yet able to do hurt. For every reason,
therefore, such guards are as useless as a colony is useful.
  Again, the prince who holds a country differing in the above
respects ought to make himself the head and defender of his powerful
neighbours, and to weaken the more powerful amongst them, taking
care that no foreigner as powerful as himself shall, by any
accident, get a footing there; for it will always happen that such a
one will be introduced by those who are discontented, either through
excess of ambition or through fear, as one has seen already. The
Romans were brought into Greece by the Aetolians; and in every other
country where they obtained a footing they were brought in by the
inhabitants. And the usual course of affairs is that, as soon as a
powerful foreigner enters a country, all the subject states are
drawn to him, moved by the hatred which they feel against the ruling
power. So that in respect to these subject states he has not to take
any trouble to gain them over to himself, for the whole of them
quickly rally to the state which he has acquired there. He has only to
take care that they do not get hold of too much power and too much
authority, and then with his own forces, and with their goodwill, he
can easily keep down the more powerful of them, so as to remain
entirely master in the country. And he who does not properly manage
this business will soon lose what he has acquired, and whilst he
does hold it he will have endless difficulties and troubles.
  The Romans, in the countries which they annexed, observed closely
these measures; they sent colonies and maintained friendly relations
with the minor powers, without increasing their strength; they kept
down the greater, and did not allow any strong foreign powers to
gain authority. Greece appears to me sufficient for an example. The
Achaeans and Aetolians were kept friendly by them, the kingdom of
Macedonia was humbled, Antiochus was driven out; yet the merits of the
Achaeans and Aetolians never secured for them permission to increase
their power, nor did the persuasions of Philip ever induce the
Romans to be his friends without first humbling him, nor did the
influence of Antiochus make them agree that he should retain any
lordship over the country. Because the Romans did in these instances
what all prudent princes ought to do, who have to regard not only
present troubles, but also future ones, for which they must prepare
with every energy, because, when foreseen, it is easy to remedy
them; but if you wait until they approach, the medicine is no longer
in time because the malady has become incurable; for it happens in
this, as the physicians say it happens in hectic fever, that in the
beginning of the malady it is easy to cure but difficult to detect,
but in the course of time, not having been either detected or
treated in the beginning, it becomes easy to detect but difficult to
cure. Thus it happens in affairs of state, for when the evils that
arise have been foreseen (which it is only given to a wise man to
see), they can be quickly redressed, but when, through not having been
foreseen, they have been permitted to grow in a way that every one can
see them. there is no longer a remedy. Therefore, the Romans,
foreseeing troubles, dealt with them at once, and, even to avoid a
war, would not let them come to a head, for they knew that war is
not to be avoided, but is only put off to the advantage of others;
moreover they wished to fight with Philip and Antiochus in Greece so
as not to have to do it in Italy; they could have avoided both, but
this they did not wish; nor did that ever please them which is for
ever in the mouths of the wise ones of our time:- Let us enjoy the
benefits of the time- but rather the benefits of their own valour
and prudence, for time drives everything before it, and is able to
bring with it good as well as evil, and evil as well as good.
  But let us turn to France and inquire whether she has done any of
the things mentioned. I will speak of Louis [XII] (and not of
Charles [VIII]) as the one whose conduct is the better to be observed,
he having held possession of Italy for the longest period; and you
will see that he has done the opposite to those things which ought
to be done to retain a state composed of divers elements.
  King Louis was brought into Italy by the ambition of the
Venetians, who desired to obtain half the state of Lombardy by his
intervention. I will not blame the course taken by the king,
because, wishing to get a foothold in Italy, and having no friends
there- seeing rather that every door was shut to him owing to the
conduct of Charles- he was forced to accept those friendships which he
could get, and he would have succeeded very quickly in his design if
in other matters he had not made some mistakes. The king, however,
having acquired Lombardy, regained at once the authority which Charles
had lost: Genoa yielded; the Florentines became his friends; the
Marquess of Mantua, the Duke of Ferrara, the Bentivoglio, my lady of
Forli, the Lords of Faenza, of Pesaro, of Rimini, of Camerino, of
Piombino, the Lucchesi, the Pisans, the Sienese- everybody made
advances to him to become his friend. Then could the Venetians realize
the rashness of the course taken by them, which, in order that they
might secure two towns in Lombardy, had made the king master of
two-thirds of Italy.
  Let any one now consider with what little difficulty the king
could have maintained his position in Italy had he observed the
rules above laid down, and kept all his friends secure and
protected; for although they were numerous they were both weak and
timid, some afraid of the Church, some of the Venetians, and thus they
would always have been forced to stand in with him, and by their means
he could easily have made himself secure against those who remained
powerful. But he was no sooner in Milan than he did the contrary by
assisting Pope Alexander to occupy the Romagna. It never occurred to
him that by this action he was weakening himself, depriving himself of
friends and those who had thrown themselves into his lap, whilst he
aggrandized the Church by adding much temporal power to the spiritual,
thus giving it great authority. And having committed this prime error,
he was obliged to follow it up, so much so that, to put an end to
the ambition of Alexander, and to prevent his becoming the master of
Tuscany, he was himself forced to come into Italy.
  And as if it were not enough to have aggrandized the Church, and
deprived himself friends, he, wishing to have the kingdom of Naples,
divides it with the King of Spain, and where he was the prime
arbiter of Italy he takes an associate, so that the ambitious of
that country and the malcontents of his own should have where to
shelter; and whereas he could have left in the kingdom his own
pensioner as king, he drove him out, to put one there who was able
to drive him, Louis, out in turn.
  The wish to acquire is in truth very natural and common, and men
always do so when they can, and for this they will be praised not
blamed; but when they cannot do so, yet wish to do so by any means,
then there is folly and blame. Therefore, if France could have
attacked Naples with her own forces she ought to have done so; if
she could not, then she ought not to have divided it. And if the
partition which she made with the Venetians in Lombardy was
justified by the excuse that by it she got a foothold in Italy, this
other partition merited blame, for it had not the excuse of that
necessity.
  Therefore Louis made these five errors: he destroyed the minor
powers, he increased the strength of one of the greater powers in
Italy, he brought in a foreign power, he did not settle in the
country, he did not send colonies. Which errors, if he had lived, were
not enough to injure him had he not made a sixth by taking away
their dominions from the Venetians; because, had he not aggrandized
the Church, nor brought Spain into Italy, it would have been very
reasonable and necessary to humble them; but having first taken
these steps, he ought never to have consented to their ruin, for they,
being powerful, would always have kept off others from designs on
Lombardy, to which the Venetians would never have consented except
to become masters themselves there; also because the others would
not wish to take Lombardy from France in order to give it to the
Venetians, and to run counter to both they would not have had the
courage.
  And if any one should say: King Louis yielded the Romagna to
Alexander and the kingdom to Spain to avoid war, I answer for the
reasons given above that a blunder ought never be perpetrated to avoid
war, because it is not to be avoided, but is only deferred to your
disadvantage. And if another should allege the pledge which the king
had given to the Pope that he would assist him in the enterprise, in
exchange for the dissolution of his marriage and for the hat to Rouen,
to that I reply what I shall write later on concerning the faith of
princes, and how it ought to be kept.
  Thus King Louis lost Lombardy by not having followed any of the
conditions observed by those who have taken possession of countries
and wished to retain them. Nor is there any miracle in this, but
much that is reasonable and quite natural. And on these matters I
spoke at Nantes with Rouen, when Valentino,* as Cesare Borgia, the son
of Pope Alexander, was usually called, occupied the Romagna, and on
Cardinal Rouen observing to me that the Italians did not understand
war, I replied to him that the French did not understand statecraft,
meaning that otherwise they would not have allowed the Church to reach
such greatness. And in fact it has been seen that the greatness of the
Church and of Spain in Italy has been caused by France, and her ruin
may be attributed to them. From this a general rule is drawn which
never or rarely fails: that he who is the cause of another becoming
powerful is ruined; because that predominancy has been brought about
either by astuteness or else by force, and both are distrusted by
him who has been raised to power.

  * So called- in Italian- from the duchy of Valentinois, conferred on
him by Louis XII.
                              CHAPTER IV
          WHY THE KINGDOM OF DARIUS, CONQUERED BY ALEXANDER,
    DID NOT REBEL AGAINST THE SUCCESSORS OF ALEXANDER AT HIS DEATH

  CONSIDERING the difficulties which men have had to hold a newly
acquired state, some might wonder how, seeing that Alexander the Great
became the master of Asia in a few years, and died whilst it was yet
scarcely settled (whence it might appear reasonable that the whole
empire would have rebelled), nevertheless his successors maintained
themselves, and had to meet no other difficulty than that which
arose among themselves from their own ambitions.
  I answer that the principalities of which one has record are found
to be governed in two different ways: either by a prince, with a
body of servants, who assist him to govern the kingdom as ministers by
his favour and permission; or by a prince and barons, who hold that
dignity by antiquity of blood and not by the grace of the prince. Such
barons have states and their own subjects, who recognize them as lords
and hold them in natural affection. Those states that are governed
by a prince and his servants hold their prince in more
consideration, because in all the country there is no one who is
recognized as superior to him, and if they yield obedience to
another they do it as to a minister and official, and they do not bear
him any particular affection.
  The examples of these two governments in our time are the Turk and
the King of France. The entire monarchy of the Turk is governed by one
lord, the others are his servants; and, dividing his kingdom into
sanjaks, he sends there different administrators, and shifts and
changes them as he chooses. But the King of France is placed in the
midst of an ancient body of lords, acknowledged by their own subjects,
and beloved by them; they have their own prerogatives, nor can the
king take these away except at his peril. Therefore, he who
considers both of these states will recognize great difficulties in
seizing the state of the Turk, but, once it is conquered, great ease
in holding it. The causes of the difficulties in seizing the kingdom
of the Turk are that the usurper cannot be called in by the princes of
the kingdom, nor can he hope to be assisted in his designs by the
revolt of those whom the lord has around him. This arises from the
reasons given above; for his ministers, being all slaves and
bondmen, can only be corrupted with great difficulty, and one can
expect little advantage from them when they have been corrupted, as
they cannot carry the people with them, for the reasons assigned.
Hence, he who attacks the Turk must bear in mind that he will find him
united, and he will have to rely more on his own strength than on
the revolt of others; but, if once the Turk has been conquered, and
routed in the field in such a way that he cannot replace his armies,
there is nothing to fear but the family of the prince, and, this being
exterminated, there remains no one to fear, the others having no
credit with the people; and as the conqueror did not rely on them
before his victory, so he ought not to fear them after it.
  The contrary happens in kingdoms governed like that of France,
because one can easily enter there by gaining over some baron of the
kingdom, for one always finds malcontents and such as desire a change.
Such men, for the reasons given, can open the way into the state and
render the victory easy; but if you wish to hold it afterwards, you
meet with infinite difficulties, both from those who have assisted you
and from those you have crushed. Nor is it enough for you to have
exterminated the family of the prince, because the lords that remain
make themselves the heads of fresh movements against you, and as you
are unable either to satisfy or exterminate them, that state is lost
whenever time brings the opportunity.
  Now if you will consider what was the nature of the government of
Darius, you will find it similar to the kingdom of the Turk, and
therefore it was only necessary for Alexander, first to overthrow
him in the field, and then to take the country from him. After which
victory, Darius being killed, the state remained secure to
Alexander, for the above reasons. And if his successors had been
united they would have enjoyed it securely and at their ease, for
there were no tumults raised in the kingdom except those they provoked
themselves.
  But it is impossible to hold with such tranquillity states
constituted like that of France. Hence arose those frequent rebellions
against the Romans in Spain, France, and Greece, owing to the many
principalities there were in these states, of which, as long as the
memory of them endured, the Romans always held an insecure possession;
but with the power and long continuance of the empire the memory of
them passed away, and the Romans then became secure possessors. And
when fighting afterwards amongst themselves, each one was able to
attach to himself his own parts of the country, according to the
authority he had assumed there; and the family of the former lord
being exterminated, none other than the Romans were acknowledged.
  When these things are remembered no one will marvel at the ease with
which Alexander held the Empire of Asia, or at the difficulties
which others have had to keep an acquisition, such as Pyrrhus and many
more; this is not occasioned by the little or abundance of ability
in the conqueror, but by the want of uniformity in the subject state.
                              CHAPTER V
     CONCERNING THE WAY TO GOVERN CITIES OR PRINCIPALITIES WHICH
         LIVED UNDER THEIR OWN LAWS BEFORE THEY WERE ANNEXED

  WHENEVER those states which have been acquired as stated have been
accustomed to live under their own laws and in freedom, there are
three courses for those who wish to hold them: the first is to ruin
them, the next is to reside there in person, the third is to permit
them to live under their own laws, drawing a tribute, and establishing
within it an oligarchy which will keep it friendly to you. Because
such a government, being created by the prince, knows that it cannot
stand without his friendship and interest, and does its utmost to
support him; and therefore he who would keep a city accustomed to
freedom will hold it more easily by the means of its own citizens than
in any other way.
  There are, for example, the Spartans and the Romans. The Spartans
held Athens and Thebes, establishing there an oligarchy,
nevertheless they lost them. The Romans, in order to hold Capua,
Carthage, and Numantia, dismantled them, and did not lose them. They
wished to hold Greece as the Spartans held it, making it free and
permitting its laws, and did not succeed. So to hold it they were
compelled to dismantle many cities in the country, for in truth
there is no safe way to retain them otherwise than by ruining them.
And he who becomes master of a city accustomed to freedom and does not
destroy it, may expect to be destroyed by it, for in rebellion it
has always the watch-word of liberty and its ancient privileges as a
rallying point, which neither time nor benefits will ever cause it
to forget. And what ever you may do or provide against, they never
forget that name or their privileges unless they are disunited or
dispersed but at every chance they immediately rally to them, as
Pisa after the hundred years she had been held in bondage by the
Florentines.
  But when cities or countries are accustomed to live under a
prince, and his family is exterminated, they, being on the one hand
accustomed to obey and on the other hand not having the old prince,
cannot agree in making one from amongst themselves, and they do not
know how to govern themselves. For this reason they are very slow to
take up arms, and a prince can gain them to himself and secure them
much more easily. But in republics there is more vitality, greater
hatred, and more desire for vengeance, which will never permit them to
allow the memory of their former liberty to rest; so that the safest
way is to destroy them or to reside there.
                              CHAPTER VI
           CONCERNING NEW PRINCIPALITIES WHICH ARE ACQUIRED
                    BY ONE'S OWN ARMS AND ABILITY

  LET no one be surprised if, in speaking of entirely new
principalities as I shall do, I adduce the highest examples both of
prince and of state; because men, walking almost always in paths
beaten by others, and following by imitation their deeds, are yet
unable to keep entirely to the ways of others or attain to the power
of those they imitate. A wise man ought always to follow the paths
beaten by great men, and to imitate those who have been supreme, so
that if his ability does not equal theirs, at least it will savour
of it. Let him act like the clever archers who, designing to hit the
mark which yet appears too far distant, and knowing the limits to
which the strength of their bow attains, take aim much higher than the
mark, not to reach by their strength or arrow to so great a height,
but to be able with the aid of so high an aim to hit the mark they
wish to reach.
  I say, therefore, that in entirely new principalities, where there
is a new prince, more or less difficulty is found in keeping them,
accordingly as there is more or less ability in him who has acquired
the state. Now, as the fact of becoming a prince from a private
station presupposes either ability or fortune, it is clear that one or
other of these two things will mitigate in some degree many
difficulties. Nevertheless, he who has relied least on fortune is
established the strongest. Further, it facilitates matters when the
prince, having no other state, is compelled to reside there in person.
  But to come to those who, by their own ability and not through
fortune, have risen to be princes, I say that Moses, Cyrus, Romulus,
Theseus, and such like are the most excellent examples. And although
one may not discuss Moses, he having been a mere executor of the
will of God, yet he ought to be admired, if only for that favour which
made him worthy to speak with God. But in considering Cyrus and others
who have acquired or founded kingdoms, all will be found admirable;
and if their particular deeds and conduct shall be considered, they
will not be found inferior to those of Moses, although he had so great
a preceptor. And in examining their actions and lives one cannot see
that they owed anything to fortune beyond opportunity, which brought
them the material to mould into the form which seemed best to them.
Without that opportunity their powers of mind would have been
extinguished, and without those powers the opportunity would have come
in vain.
  It was necessary, therefore, to Moses that he should find the people
of Israel in Egypt enslaved and oppressed by the Egyptians, in order
that they should be disposed to follow him so as to be delivered out
of bondage. It was necessary that Romulus should not remain in Alba,
and that he should be abandoned at his birth, in order that he
should become King of Rome and founder of the fatherland. It was
necessary that Cyrus should find the Persians discontented with the
government of the Medes, and the Medes soft and effeminate through
their long peace. Theseus could not have shown his ability had he
not found the Athenians dispersed. These opportunities, therefore,
made those men fortunate, and their high ability enabled them to
recognize the opportunity whereby their country was ennobled and
made famous.
  Those who by valorous ways become princes, like these men, acquire a
principality with difficulty, but they it with ease. The
difficulties they have in acquiring it arise in part from the new
rules and methods which they are forced to introduce to establish
their government and its security. And it ought to be remembered
that there is nothing more difficult to take in hand, more perilous to
conduct, or more uncertain in its success, than to take the lead in
the introduction of a new order of things. Because the innovator has
for enemies all those who have done well under the old conditions, and
lukewarm defenders in those who may do well under the new. This
coolness arises partly from fear of the opponents, who have the laws
on their side, and partly from the incredulity of men, who do not
readily believe in new things until they have had a long experience of
them. Thus it happens that whenever those who are hostile have the
opportunity to attack they do it like partisans, whilst the others
defend lukewarmly, in such wise that the prince is endangered along
with them.
  It is necessary, therefore, if we desire to discuss this matter
thoroughly, to inquire whether these innovators can rely on themselves
or have to depend on others: that is to say, whether, to consummate
their enterprise, have they to use prayers or can they use force? In
the first instance they always succeed badly, and never compass
anything; but when they can rely on themselves and use force, then
they are rarely endangered. Hence it is that all armed prophets have
conquered, and the unarmed ones have been destroyed. Besides the
reasons mentioned, the nature of the people is variable, and whilst it
is easy to persuade them, it is difficult to fix them in that
persuasion. And thus it is necessary to take such measures that,
when they believe no longer, it may be possible to make them believe
by force.
  If Moses, Cyrus, Theseus, and Romulus had been unarmed they could
not have enforced their constitutions for long- as happened in our
time to Fra Girolamo Savonarola, who was ruined with his new order
of things immediately the multitude believed in him no longer, and
he had no means of keeping steadfast those who believed or of making
the unbelievers to believe. Therefore such as these have great
difficulties in consummating their enterprise, for all their dangers
are in the ascent, yet with ability they will overcome them; but
when these are overcome, and those who envied them their success are
exterminated, they will begin to be respected, and they will
continue afterwards powerful, secure, honoured, and happy.
  To these great examples I wish to add a lesser one; still it bears
some resemblance to them, and I wish it to suffice me for all of a
like kind: it is Hiero the Syracusan. This man rose from a private
station to be Prince of Syracuse, nor did he, either, owe anything
to fortune but opportunity; for the Syracusans, being oppressed, chose
him for their captain, afterwards he was rewarded by being made
their prince. He was of so great ability, even as a private citizen,
that one who writes of him says he wanted nothing but a kingdom to
be a king. This man abolished the old soldiery, organized the new,
gave up old alliances, made new ones; and as he had his own soldiers
and allies, on such foundations he was able to build any edifice:
thus, whilst he had endured much trouble in acquiring, he had but
little in keeping.
                             CHAPTER VII
           CONCERNING NEW PRINCIPALITIES WHICH ARE ACQUIRED
           EITHER BY THE ARMS OF OTHERS OR BY GOOD FORTUNE

  THOSE who solely by good fortune become princes from being private
citizens have little trouble in rising, but much in keeping atop; they
have not any difficulties on the way up, because they fly, but they
have many when they reach the summit. Such are those to whom some
state is given either for money or by the favour of him who bestows
it; as happened to many in Greece, in the cities of Ionia and of the
Hellespont, where princes were made by Darius, in order that they
might hold the cities both for his security and his glory; as also
were those emperors who, by the corruption of the soldiers, from being
citizens came to empire. Such stand simply upon the goodwill and the
fortune of him who has elevated them- two most inconstant and unstable
things. Neither have they the knowledge requisite for the position;
because, unless they are men of great worth and ability, it is not
reasonable to expect that they should know how to command, having
always lived in a private condition; besides, they cannot hold it
because they have not forces which they can keep friendly and
faithful.
  States that rise unexpectedly, then, like all other things in nature
which are born and grow rapidly, cannot have their foundations and
relations with other states fixed in such a way that the first storm
will not overthrow them; unless, as is said, those who unexpectedly
become princes are men of so much ability that they know they have
to be prepared at once to hold that which fortune has thrown into
their laps, and that those foundations, which others have laid
before they became princes, they must lay afterwards.
  Concerning these two methods of rising to be a prince by ability
or fortune, I wish to adduce two examples within our own recollection,
and these are Francesco Sforza and Cesare Borgia. Francesco, by proper
means and with great ability, from being a private person rose to be
Duke of Milan, and that which he had acquired with a thousand
anxieties he kept with little trouble. On the other hand, Cesare
Borgia, called by the people Duke Valentino, acquired his state during
the ascendancy of his father, and on its decline he lost it,
notwithstanding that he had taken every measure and done all that
ought to be done by a wise and able man to fix firmly his roots in the
states which the arms and fortunes of others had bestowed on him.
  Because, as is stated above, he who has not first laid his
foundations may be able with great ability to lay them afterwards, but
they will be laid with trouble to the architect and danger to the
building. If, therefore, all the steps taken by the duke be
considered, it will be seen that he laid solid foundations for his
future power, and I do not consider it superfluous to discuss them,
because I do not know what better precepts to give a new prince than
the example of his actions; and if his dispositions were of no
avail, that was not his fault, but the extraordinary and extreme
malignity of fortune.
  Alexander VI, in wishing to aggrandize the duke, his son, had many
immediate and prospective difficulties. Firstly, he did not see his
way to make him master of any state that was not a state of the
Church; and if he was willing to rob the Church he knew that the
Duke of Milan and the Venetians would not consent, because Faenza
and Rimini were already under the protection of the Venetians. Besides
this, he saw the arms of Italy, especially those by which he might
have been assisted, in hands that would fear the aggrandizement of the
Pope, namely, the Orsini and the Colonna and their following. It
behoved him, therefore, to upset this state of affairs and embroil the
powers, so as to make himself securely master of part of their states.
This was easy for him to do, because he found the Venetians, moved
by other reasons, inclined to bring back the French into Italy; he
would not only not oppose this, but he would render it more easy by
dissolving the former marriage of King Louis. Therefore the king
came into Italy with the assistance of the Venetians and the consent
of Alexander. He was no sooner in Milan than the Pope had soldiers
from him for the attempt on the Romagna, which yielded to him on the
reputation of the king. The duke, therefore, having acquired the
Romagna and beaten the Colonna, while wishing to hold that and to
advance further, was hindered by two things: the one, his forces did
not appear loyal to him, the other, the goodwill of France: that is to
say, he feared that the forces of the Orsini, which was using, would
not stand to him, that not only might they hinder him from winning
more, but might themselves seize what he had won, and that the King
might also do the same. Of the Orsini he had a warning when, after
taking Faenza and attacking Bologna, he saw them go very unwillingly
to that attack. And as to the king, he learned his mind when he
himself, after taking the duchy of Urbino, attacked Tuscany, and the
king made him desist from that undertaking; hence the duke decided
to depend no more upon the arms and the luck of others.
  For the first thing he weakened the Orsini and Colonna parties in
Rome, by gaining to himself all their adherents who were gentlemen,
making them his gentlemen, giving them good pay, and, according to
their rank, honouring them with office and command in such a way
that in a few months all attachment to the factions was destroyed
and turned entirely to the duke. After this he awaited an
opportunity to crush the Orsini, having scattered the adherents of the
Colonna. This came to him soon and he used it well; for the Orsini,
perceiving at length that the aggrandizement of the duke and the
Church was ruin to them, called a meeting at Magione, in the territory
of Perugia. From this sprung the rebellion at Urbino and the tumults
in the Romagna, with endless dangers to the duke, all of which he
overcame with the help of the French. Having restored his authority,
not to leave it at risk by trusting either to the French or other
outside forces, he had recourse to his wiles, and he knew so well
how to conceal his mind that, by the mediation of Signor Paolo
[Orsini]- whom the duke did not fail to secure with all kinds of
attention, giving him money, apparel, and horses- the Orsini were
reconciled, so that their simplicity brought them into his power at
Sinigaglia. Having exterminated the leaders, and turned their
partisans into his friends, the duke had laid sufficiently good
foundations to his power, having all the Romagna and the duchy of
Urbino; and the people now beginning to appreciate their prosperity,
he gained them all over to himself. And as this point is worthy of
notice, and to be imitated by others, I am not willing to leave it
out.
  When the duke occupied the Romagna he found it under the rule of
weak masters, who rather plundered their subjects than ruled them, and
gave them more cause for disunion than for union, so that the
country was full of robbery, quarrels, and every kind of violence; and
so, wishing to bring back peace and obedience to authority, he
considered it necessary to give it a good governor. Thereupon he
promoted Messer Ramiro d'Orco [de Lorqua], a swift and cruel man, to
whom he gave the fullest power. This man in a short time restored
peace and unity with the greatest success. Afterwards the duke
considered that it was not advisable to confer such excessive
authority, for he had no doubt but that he would become odious, so
he set up a court of judgment in the country, under a most excellent
president, wherein all cities had their advocates. And because he knew
that the past severity had caused some hatred against himself, so,
to clear himself in the minds of the people, and gain them entirely to
himself, he desired to show that, if any cruelty had been practised,
it had not originated with him, but in the natural sternness of the
minister. Under this pretence he took Ramiro, and one morning caused
him to be executed and left on the piazza at Cesena with the block and
a bloody knife at his side. The barbarity of this spectacle caused the
people to be at once satisfied and dismayed.
  But let us return whence we started. I say that the duke, finding
himself now sufficiently powerful and partly secured from immediate
dangers by having armed himself in his own way, and having in a
great measure crushed those forces in his vicinity that could injure
him if he wished to proceed with his conquest, had next to consider
France, for he knew that the king, who too late was aware of his
mistake, would not support him. And from this time he began to seek
new alliances and to temporize with France in the expedition which she
was making towards the kingdom of Naples against the Spaniards who
were besieging Gaeta. It was his intention to secure himself against
them, and this he would have quickly accomplished had Alexander lived.
  Such was his line of action as to present affairs. But as to the
future he had to fear, in the first place, that a new successor to the
Church might not be friendly to him and might seek to take from him
that which Alexander had given him, so he decided to act in four ways.
Firstly, by exterminating the families of those lords whom he had
despoiled, so as to take away that pretext from the Pope. Secondly, by
winning to himself all the gentlemen of Rome, so as to be able to curb
the Pope with their aid, as has been observed. Thirdly, by
converting the college more to himself. Fourthly, by acquiring so much
power before the Pope should die that he could by his own measures
resist the first shock. Of these four things, at the death of
Alexander, he had accomplished three. For he had killed as many of the
dispossessed lords as he could lay hands on, and few had escaped; he
had won over the Roman gentlemen, and he had the most numerous party
in the college. And as to any fresh acquisition, he intended to become
master of Tuscany, for he already possessed Perugia and Piombino,
and Pisa was under his protection. And as he had no longer to study
France (for the French were already driven out of the kingdom of
Naples by the Spaniards, and in this way both were compelled to buy
his goodwill), he pounced down upon Pisa. After this, Lucca and
Siena yielded at once, partly through hatred and partly through fear
of the Florentines; and the Florentines would have had no remedy had
he continued to prosper, as he was prospering the year that
Alexander died, for he had acquired so much power and reputation
that he would have stood by himself, and no longer have depended on
the luck and the forces of others, but solely on his own power and
ability.
  But Alexander died five years after he had first drawn the sword. He
left the duke with the state of Romagna alone consolidated, with the
rest in the air, between two most powerful hostile armies, and sick
unto death. Yet there were in the duke such boldness and ability,
and he knew so well how men are to be won or lost, and so firm were
the foundations which in so short a time he had laid, that if he had
not had those armies on his back, or if he had been in good health, he
would have overcome all difficulties. And it is seen that his
foundations were good, for the Romagna awaited him for more than a
month. In Rome, although but half alive, he remained secure; and
whilst the Baglioni, the Vitelli, and the Orsini might come to Rome,
they could not effect anything against him. If he could not have
made Pope him whom he wished, at least the one whom he did not wish
would not have been elected. But if he had been in sound health at the
death of Alexander, everything would have been easy to him. On the day
that Julius II was elected, he told me that he had thought of
everything that might occur at the death of his father, and had
provided a remedy for all, except that he had never anticipated
that, when the death did happen, he himself would be on the point to
die.
  When all the actions of the duke are recalled, I do not know how
to blame him, but rather it appears to me, as I have said, that I
ought to offer him for imitation to all those who, by the fortune or
the arms of others, are raised to government. Because he, having a
lofty spirit and far-reaching aims, could not have regulated his
conduct otherwise, and only the shortness of the life of Alexander and
his own sickness frustrated his designs. Therefore, he who considers
it necessary to secure himself in his new principality, to win
friends, to overcome either by force or fraud, to make himself beloved
and feared by the people, to be followed and revered by the
soldiers, to exterminate those who have power or reason to hurt him,
to change the old order of things for new, to be severe and
gracious, magnanimous and liberal, to destroy a disloyal soldiery
and to create new, to maintain friendship with kings and princes in
such a way that they must help him with zeal and offend with
caution, cannot find a more lively example than the actions of this
man.
  Only can he be blamed for the election of Julius II, in whom he made
a bad choice, because, as is said, not being able to elect a Pope to
his own mind, he could have hindered any other from being elected
Pope; and he ought never to have consented to the election of any
cardinal whom he had injured or who had cause to fear him if they
became pontiffs. For men injure either from fear or hatred. Those whom
he had injured, amongst others, were San Pietro ad Vincula, Colonna,
San Giorgio, and Ascanio.* Any one of the others, on becoming Pope,
would have had to fear him, Rouen and the Spaniards excepted; the
latter from their relationship and obligations, the former from his
influence, the kingdom of France having relations with him. Therefore,
above everything, the duke ought to have created a Spaniard Pope, and,
failing him, he ought to have consented to Rouen and not San Pietro ad
Vincula. He who believes that new benefits will cause great personages
to forget old injuries is deceived. Therefore, the duke erred in his
choice, and it was the cause of his ultimate ruin.

  * Julius II had been Cardinal of San Pietro ad Vincula; San
Giorgio was Raffaells Riaxis, and Ascanio was Cardinal Ascanio Sforza.
                             CHAPTER VIII
          CONCERNING THOSE WHO HAVE OBTAINED A PRINCIPALITY
                            BY WICKEDNESS

  ALTHOUGH a prince may rise from a private station in two ways,
neither of which can be entirely attributed to fortune or genius,
yet it is manifest to me that I must not be silent on them, although
one could be more copiously treated when I discuss republics. These
methods are when, either by some wicked or nefarious ways, one ascends
to the principality, or when by the favour of his fellow-citizens a
private person becomes the prince of his country. And speaking of
the first method, it will be illustrated by two examples- one ancient,
the other modern- and without entering further into the subject, I
consider these two examples will suffice those who may be compelled to
follow them.
  Agathocles, the Sicilian, became King of Syracuse not only from a
private but from a low and abject position. This man, the son of a
potter, through all the changes in his fortunes always led an infamous
life. Nevertheless, he accompanied his infamies with so much ability
of mind and body that, having devoted himself to the military
profession, he rose through its ranks to be Praetor of Syracuse. Being
established in that position, and having deliberately resolved to make
himself prince and to seize by violence, without obligation to others,
that which had been conceded to him by assent, he came to an
understanding for this purpose with Hamilcar, the Carthaginian, who,
with his army, was fighting in Sicily. One morning he assembled the
people and senate of Syracuse, as if he had to discuss with them
things relating to the Republic, and at a given signal the soldiers
killed all the senators and the richest of the people; these dead,
he seized and held the princedom of that city without any civil
commotion. And although he was twice routed by the Carthaginians,
and ultimately besieged, yet not only was he able to defend his
city, but leaving part of his men for its defence, with the others
he attacked Africa, and in a short time raised the siege of
Syracuse. The Carthaginians, reduced to extreme necessity, were
compelled to come to terms with Agathocles, and, leaving Sicily to
him, had to be content with the possession of Africa.
  Therefore, he who considers the actions and the genius of this man
will see nothing, or little, which can be attributed to fortune,
inasmuch as he attained pre-eminence, as is shown above, not by the
favour of any one, but step by step in the military profession,
which steps were gained with a thousand troubles and perils, and
were afterwards boldly held by him with many hazards and dangers.
Yet it cannot be called talent to slay fellow-citizens, to deceive
friends, to be without faith, without mercy, without religion; such
methods may gain empire, but not glory. Still, if the courage of
Agathocles in entering into and extricating himself from dangers be
considered, together with his greatness of mind in enduring overcoming
hardships, it cannot be seen why he should be esteemed less than the
most notable captain. Nevertheless, his barbarous cruelty and
inhumanity with infinite wickednesses do not permit him to be
celebrated among the most excellent men. What he achieved cannot be
attributed either to fortune or to genius.
  In our times, during the rule of Alexander VI, Oliverotto da
Fermo, having been left an orphan many years before, was brought up by
his maternal uncle, Giovanni Fogliani, and in the early days of his
youth sent to fight under Paolo Vitelli, that, being trained under his
discipline, he might attain some high position in the military
profession. After Paolo died, he fought under his brother
Vitellozzo, and in a very short time, being endowed with wit and a
vigorous body and mind, he became the first man in his profession. But
it appearing to him a paltry thing to serve under others, he resolved,
with the aid of some citizens of Fermo, to whom the slavery of their
country was dearer than its liberty, and with the help of the Vitelli,
to seize Fermo. So he wrote to Giovanni Fogliani that, having been
away from home for many years, he wished to visit him and his city,
and in some measure to look into his patrimony; and although he had
not laboured to acquire anything except honour, yet, in order that the
citizens should see he had not spent his time in vain, he desired to
come honourably, so would be accompanied by one hundred horsemen,
his friends and retainers; and he entreated Giovanni to arrange that
he should be received honourably by the citizens of Fermo, all of
which would be not only to his honour, but also to that of Giovanni
himself, who had brought him up.
  Giovanni, therefore, did not fail in any attentions due to his
nephew, and he caused him to be honourably received by the Fermans,
and he lodged him in his own house, where, having passed some days,
and having arranged what was necessary for his wicked designs,
Oliverotto gave a solemn banquet to which he invited Giovanni Fogliani
and the chiefs of Fermo. When the viands and all the other
entertainments that are usual in such banquets were finished,
Oliverotto artfully began certain grave discourses, speaking of the
greatness of Pope Alexander and his son Cesare, and of their
enterprises, to which discourse Giovanni and others answered; but he
rose at once, saying that such matters ought to be discussed in a more
private place, and he betook himself to a chamber, whither Giovanni
and the rest of the citizens went in after him. No sooner were they
seated than soldiers issued from secret places and slaughtered
Giovanni and the rest. After these murders Oliverotto, mounted on
horseback, rode up and down the town and besieged the chief magistrate
in the palace, so that in fear the people were forced to obey him, and
to form a government, of which he made himself the prince. He killed
all the malcontents who were able to injure him, and strengthened
himself with new civil and military ordinances, in such a way that, in
the year during which he held the principality, not only was he secure
in the city of Fermo, but he had become formidable to all his
neighbours. And his destruction would have been as difficult as that
of Agathocles if he had not allowed himself to be overreached by
Cesare Borgia, who took him with the Orsini and Vitelli at Sinigaglia,
as was stated above. Thus one year after he had committed this
parricide, he was strangled, together with Vitellozzo, whom he had
made his leader in valour and wickedness.
  Some may wonder how it can happen that Agathocles, and his like,
after infinite treacheries and cruelties, should live for long
secure in his country, and defend himself from external enemies, and
never be conspired against by his own citizens; seeing that many
others, by means of cruelty, have never been able even in peaceful
times to hold the state, still less in the doubtful times of war. I
believe that this follows from severities being badly or properly
used. Those may be called properly used, if of evil it is lawful to
speak well, that are applied at one blow and are necessary to one's
security, and that are not persisted in afterwards unless they can
be turned to the advantage of the subjects. The badly employed are
those which, notwithstanding they may be few in the commencement,
multiply with time rather than decrease. Those who practise the
first system are able, by aid of God or man, to mitigate in some
degree their rule, as Agathocles did. It is impossible for those who
follow the other to maintain themselves.
  Hence it is to be remarked that, in seizing a state, the usurper
ought to examine closely into all those injuries which it is necessary
for him to inflict, and to do them all at one stroke so as not to have
to repeat them daily; and thus by not unsettling men he will be able
to reassure them, and win them to himself by benefits. He who does
otherwise, either from timidity or evil advice, is always compelled to
keep the knife in his hand; neither can he rely on his subjects, nor
can they attach themselves to him, owing to their continued and
repeated wrongs. For injuries ought to be done all at one time, so
that, being tasted less, they offend less; benefits ought to be
given little by little, so that the flavour of them may last longer.
  And above all things, a prince ought to live amongst his people in
such a way that no unexpected circumstances, whether of good or
evil, shall make him change; because if the necessity for this comes
in troubled times, you are too late for harsh measures; and mild
ones will not help you, for they will be considered as forced from
you, and no one will be under any obligation to you for them.
                              CHAPTER IX
                   CONCERNING A CIVIL PRINCIPALITY

  BUT coming to the other point- where a leading citizen becomes the
prince of his country, not by wickedness or any intolerable
violence, but by the favour of his fellow citizens- this may be called
a civil principality: nor is genius or fortune altogether necessary to
attain to it, but rather a happy shrewdness. I say then that such a
principality is obtained either by the favour of the people or by
the favour of the nobles. Because in all cities these two distinct
parties are found, and from this it arises that the people do not wish
to be ruled nor oppressed by the nobles, and the nobles wish to rule
and oppress the people; and from these two opposite desires there
arises in cities one of three results, either a principality,
self-government, or anarchy.
  A principality is created either by the people or by the nobles,
accordingly as one or other of them has the opportunity; for the
nobles, seeing they cannot withstand the people, begin to cry up the
reputation of one of themselves, and they make him a prince, so that
under his shadow they can give vent to their ambitions. The people,
finding they cannot resist the nobles, also cry up the reputation of
one of themselves, and make him a prince so as to be defended by his
authority. He who obtains sovereignty by the assistance of the
nobles maintains himself with more difficulty than he who comes to
it by the aid of the people, because the former finds himself with
many around him who consider themselves his equals, and because of
this he can neither rule nor manage them to his liking. But he who
reaches sovereignty by popular favour finds himself alone, and has
none around him, or few, who are not prepared to obey him.
  Besides this, one cannot by fair dealing, and without injury to
others, satisfy the nobles, but you can satisfy the people, for
their object is more righteous than that of the nobles, the latter
wishing to oppress, whilst the former only desire not to be oppressed.
It is to be added also that a prince can never secure himself
against a hostile people, because of their being too many, whilst from
the nobles he can secure himself, as they are few in number. The worst
that a prince may expect from a hostile people is to be abandoned by
them; but from hostile nobles he has not only to fear abandonment, but
also that they will rise against him; for they, being in these affairs
more far-seeing and astute, always come forward in time to save
themselves, and to obtain favours from him whom they expect to
prevail. Further, the prince is compelled to live always with the same
people, but he can do well without the same nobles, being able to make
and unmake them daily, and to give or take away authority when it
pleases him.
  Therefore, to make this point clearer, I say that the nobles ought
to be looked at mainly in two ways: that is to say, they either
shape their course in such a way as binds them entirely to your
fortune, or they do not. Those who so bind themselves, and are not
rapacious, ought to be honoured and loved; those who do not bind
themselves may be dealt with in two ways; they may fail to do this
through pusillanimity and a natural want of courage, in which case you
ought to make use of them, especially of those who are of good
counsel; and thus, whilst in prosperity you honour yourself, in
adversity you have not to fear them. But when for their own
ambitious ends they shun binding themselves, it is a token that they
are giving more thought to themselves than to you, and a prince
ought to guard against such, and to fear them as if they were open
enemies, because in adversity they always help to ruin him.
  Therefore, one who becomes a prince through the favour of the people
ought to keep them friendly, and this he can easily do seeing they
only ask not to be oppressed by him. But one who, in opposition to the
people, becomes a prince by the favour of the nobles, ought, above
everything, to seek to win the people over to himself, and this he may
easily do if he takes them under his protection. Because men, when
they receive good from him of whom they were expecting evil, are bound
more closely to their benefactor; thus the people quickly become
more devoted to him than if he had been raised to the principality
by their favours; and the prince can win their affections in many
ways, but as these vary according to the circumstances one cannot give
fixed rules, so I omit them; but, I repeat, it is necessary for a
prince to have the people friendly, otherwise he has no security in
adversity.
  Nabis, Prince of the Spartans, sustained the attack of all Greece,
and of a victorious Roman army, and against them he defended his
country and his government; and for the overcoming of this peril it
was only necessary for him to make himself secure against a few, but
this would not have been sufficient if the people had been hostile.
And do not let any one impugn this statement with the trite proverb
that 'He who builds on the people, builds on the mud,' for this is
true when a private citizen makes a foundation there, and persuades
himself that the people will free him when he is oppressed by his
enemies or by the magistrates; wherein he would find himself very
often deceived, as happened to the Gracchi in Rome and to Messer
Giorgio Scali in Florence. But granted a prince who has established
himself as above, who can command, and is a man of courage, undismayed
in adversity, who does not fail in other qualifications, and who, by
his resolution and energy, keeps the whole people encouraged- such a
one will never find himself deceived in them, and it will be shown
that he has laid his foundations well.
  These principalities are liable to danger when they are passing from
the civil to the absolute order of government, for such princes either
rule personally or through magistrates. In the latter case their
government is weaker and more insecure, because it rests entirely on
the goodwill of those citizens who are raised to the magistracy, and
who, especially in troubled times, can destroy the government with
great ease, either by intrigue or open defiance; and the prince has
not the chance amid tumults to exercise absolute authority, because
the citizens and subjects, accustomed to receive orders from
magistrates, are not of a mind to obey him amid these confusions,
and there will always be in doubtful times a scarcity of men whom he
can trust. For such a prince cannot rely upon what he observes in
quiet times, when citizens had need of the state, because then every
one agrees with him; they all promise, and when death is far distant
they all wish to die for him; but in troubled times, when the state
has need of its citizens, then he finds but few. And so much the
more is this experiment dangerous, inasmuch as it can only be tried
once. Therefore a wise prince ought to adopt such a course that his
citizens will always in every sort and kind of circumstance have
need of the state and of him, and then he will always find them
faithful.
                              CHAPTER X
               CONCERNING THE WAY IN WHICH THE STRENGTH
              OF ALL PRINCIPALITIES OUGHT TO BE MEASURED

  IT IS necessary to consider another point in examining the character
of these principalities: that is, whether a prince has such power
that, in case of need, he can support himself with his own
resources, or whether he has always need of the assistance of
others. And to make this quite clear I say that I consider those are
able to support themselves by their own resources who can, either by
abundance of men or money, raise a sufficient army to join battle
against any one who comes to attack them; and I consider those
always to have need of others who cannot show themselves against the
enemy in the field, but are forced to defend themselves by
sheltering behind walls. The first case has been discussed, but we
will speak of it again should it recur. In the second case one can say
nothing except to encourage such princes to provision and fortify
their towns, and not on any account to defend the country. And whoever
shall fortify his town well, and shall have managed the other concerns
of his subjects in the way stated above, and to be often repeated,
will never be attacked without great caution, for men are always
adverse to enterprises where difficulties can be seen, and it will
be seen not to be an easy thing to attack one who has his town well
fortified, and is not hated by his people.
  The cities of Germany are absolutely free, they own but little
country around them, and they yield obedience to the emperor when it
suits them, nor do they fear this or any other power they may have
near them, because they are fortified in such a way that every one
thinks the taking of them by assault would be tedious and difficult,
seeing they have proper ditches and walls, they have sufficient
artillery, and they always keep in public depots enough for one year's
eating, drinking, and firing. And beyond this, to keep the people
quiet and without loss to the state, they always have the means of
giving work to the community in those labours that are the life and
strength of the city, and on the pursuit of which the people are
supported; they also hold military exercises in repute, and moreover
have many ordinances to uphold them.
  Therefore, a prince who has a strong city, and had not made
himself odious, will not be attacked, or if any one should attack he
will only be driven off with disgrace; again, because that affairs
of this world are so changeable, it is almost impossible to keep an
army a whole year in the field without being interfered with. And
whoever should reply: If the people have property outside the city,
and see it burnt, they will not remain patient, and the long siege and
self-interest will make them forget their prince; to this I answer
that a powerful and courageous prince will overcome all such
difficulties by giving at one time hope to his subjects that the
evil will not be for long, at another time fear of the cruelty of
the enemy, then preserving himself adroitly from those subjects who
seem to him to be too bold.
  Further, the enemy would naturally on his arrival at once burn and
ruin the country at the time when the spirits of the people are
still hot and ready for the defence; and, therefore, so much the
less ought the prince to hesitate; because after a time, when
spirits have cooled, the damage is already done, the ills are
incurred, and there is no longer any remedy; and therefore they are so
much the more ready to unite with their prince, he appearing to be
under obligations to them now that their houses have been burnt and
their possessions ruined in his defence. For it is the nature of men
to be bound by the benefits they confer as much as by those they
receive. Therefore, if everything is well considered, it wilt not be
difficult for a wise prince to keep the minds of his citizens
steadfast from first to last, when he does not fail to support and
defend them.
                              CHAPTER XI
               CONCERNING ECCLESIASTICAL PRINCIPALITIES

  IT ONLY remains now to speak of ecclesiastical principalities,
touching which all difficulties are prior to getting possession,
because they are acquired either by capacity or good fortune, and they
can be held without either; for they are sustained by the ordinances
of religion, which are so all-powerful, and of such a character that
the principalities may be held no matter how their princes behave
and live. These princes alone have states and do not defend them, they
have subjects and do not rule them; and the states, although
unguarded, are not taken from them, and the subjects, although not
ruled, do not care, and they have neither the desire nor the ability
to alienate themselves. Such principalities only are secure and happy.
But being upheld by powers, to which the human mind cannot reach, I
shall speak no more of them, because, being exalted and maintained
by God, it would be the act of a presumptuous and rash man to
discuss them.
  Nevertheless, if any one should ask of me how comes it that the
Church has attained such greatness in temporal power, seeing that from
Alexander backwards the Italian potentates (not only those who have
been called potentates, but every baron and lord, though the smallest)
have valued the temporal power very slightly- yet now a king of France
trembles before it, and it has been able to drive him from Italy,
and to ruin the Venetians- although this may be very manifest, it does
not appear to me superfluous to recall it in some measure to memory.
  Before Charles, King of France, passed into Italy, this country
was under the dominion of the Pope, the Venetians, the King of Naples,
the Duke of Milan, and the Florentines. These potentates had two
principal anxieties: the one, that no foreigner should enter Italy
under arms; the other, that none of themselves should seize more
territory. Those about whom there was the most anxiety were the Pope
and the Venetians. To restrain the Venetians the union of all the
others was necessary, as it was for the defence of Ferrara; and to
keep down the Pope they made use of the barons of Rome, who, being
divided into two factions, Orsini and Colonna, had always a pretext
for disorder, and, standing with arms in their hands under the eyes of
the Pontiff, kept the pontificate weak and powerless. And although
there might arise sometimes a courageous pope, such as Sixtus [IV],
yet neither fortune nor wisdom could rid him of these annoyances.
And the short life of a pope is also a cause of weakness; for in the
ten years, which is the average life of a pope, he can with difficulty
lower one of the factions; and if, so to speak, one pope should almost
destroy the Colonna, another would arise hostile to the Orsini, who
would support their opponents, and yet would not have time to ruin the
Orsini. This was the reason why the temporal powers of the pope were
little esteemed in Italy.
  Alexander VI arose afterwards, who of all the pontiffs that have
ever been showed how a pope with both money and arms was able to
prevail; and through the instrumentality of the Duke Valentino, and by
reason of the entry of the French, he brought about all those things
which I have discussed above in the actions of the duke. And
although his intention was not to aggrandize the Church, but the duke,
nevertheless, what he did contributed to the greatness of the
Church, which, after his death and the ruin of the duke, became the
heir to all his labours.
  Pope Julius came afterwards and found the Church strong,
possessing all the Romagna, the barons of Rome reduced to impotence,
and, through the chastisements Alexander, the factions wiped out; he
also found the way open to accumulate money in a manner such as had
never been practised before Alexander's time. Such things Julius not
only followed, but improved upon, and he intended to gain Bologna,
to ruin the Venetians, and to drive the French out of Italy. All of
these enterprises prospered with him, and so much the more to his
credit, inasmuch as he did everything to strengthen the Church and not
any private person. He kept also the Orsini and Colonna factions
within the bounds in which he found them; and although there was among
them some mind to make disturbance, nevertheless he held two things
firm: the one, the greatness of the church, with which he terrified
them; and the other, not allowing them to have their own cardinals,
who caused the disorders among them. For whenever these factions
have their cardinals they do not remain quiet for long, because
cardinals foster the factions in Rome and out of it, and the barons
are compelled to support them, and thus from the ambitions of prelates
arise disorders and tumults among the barons. For these reasons his
Holiness Pope Leo found the pontificate most powerful, and it is to be
hoped that, if others made it great in arms, he will make it still
greater and more venerated by his goodness and infinite other virtues.
                             CHAPTER XII
                HOW MANY KINDS OF SOLDIERY THERE ARE,
                      AND CONCERNING MERCENARIES

  HAVING discoursed particularly on the characteristics of such
principalities as in the beginning I proposed to discuss, and having
considered in some degree the causes of their being good or bad, and
having shown the methods by which many have sought to acquire them and
to hold them, it now remains for me to discuss generally the means
of offence and defence which belong to each of them.
  We have seen above how necessary it is for a prince to have his
foundations well laid, otherwise it follows of necessity he will go to
ruin. The chief foundations of all states, new as well as old or
composite, are good laws and good arms; and as there cannot be good
laws where the state is not well armed, it follows that where they are
well armed they have good laws. I shall leave the laws out of the
discussion and shall speak of the arms.
  I say, therefore, that the arms with which a prince defends his
state are either his own, or they are mercenaries, auxiliaries, or
mixed. Mercenaries and auxiliaries are useless and dangerous; and if
one holds his state based on these arms, he will stand neither firm
nor safe; for they are disunited, ambitious and without discipline,
unfaithful, valiant before friends, cowardly before enemies; they have
neither the fear of God nor fidelity to men, and destruction is
deferred only so long as the attack is; for in peace one is robbed
by them, and in war by the enemy. The fact is, they have no other
attraction or reason for keeping the field than a trifle of stipend,
which is not sufficient to make them willing to die for you. They
are ready enough to be your soldiers whilst you do not make war, but
if war comes they take themselves off or run from the foe; which I
should have little trouble to prove, for the ruin of Italy has been
caused by nothing else than by resting all her hopes for many years on
mercenaries, and although they formerly made some display and appeared
valiant amongst themselves, yet when the foreigners came they showed
what they were. Thus it was that Charles, King of France, was
allowed to seize Italy with chalk in hand;* and he who told us that
our sins were the cause of it told the truth, but they were not the
sins he imagined, but those which I have related. And as they were the
sins of princes, it is the princes who have also suffered the penalty.

  * With which to chalk up the billets for his soldiers.

  I wish to demonstrate further the infelicity of these arms. The
mercenary captains are either capable men or they are not; if they
are, you cannot trust them, because they always aspire to their own
greatness, either by oppressing you, who are their master, or others
contrary to your intentions; but if the captain is not skilful, you
are ruined in the usual way.
  And if it be urged that whoever is armed will act in the same way,
whether mercenary or not, I reply that when arms have to be resorted
to, either by a prince or a republic, then the prince ought to go in
person and perform the duty of captain; the republic has to send its
citizens, and when one is sent who does not turn out satisfactorily,
it ought to recall him, and when one is worthy, to hold him by the
laws so that he does not leave the command. And experience has shown
princes and republics, single-handed, making the greatest progress,
and mercenaries doing nothing except damage; and it is more
difficult to bring a republic, armed with its own arms, under the sway
of one of its citizens than it is to bring one armed with foreign
arms. Rome and Sparta stood for many ages armed and free. The Switzers
are completely armed and quite free.
  Of ancient mercenaries, for example, there are the Carthaginians,
who were oppressed by their mercenary soldiers after the first war
with the Romans, although the Carthaginians had their own citizens for
captains. After the death of Epaminondas, Philip of Macedon was made
captain of their soldiers by the Thebans, and after victory he took
away their liberty.
  Duke Filippo being dead, the Milanese enlisted Francesco Sforza
against the Venetians, and he, having overcome the enemy at
Caravaggio, allied himself with them to crush the Milanese, his
masters. His father, Sforza, having been engaged by Queen Johanna of
Naples, left her unprotected, so that she was forced to throw
herself into the arms of the King of Aragon, in order to save her
kingdom. And if the Venetians and Florentines formerly extended
their dominions by these arms, and yet their captains did not make
themselves princes, but have defended them, I reply that the
Florentines in this case have been favoured by chance, for of the able
captains, of whom they might have stood in fear, some have not
conquered, some have been opposed, and others have turned their
ambitions elsewhere. One who did not conquer was Giovanni Acuto,*
and since he did not conquer his fidelity cannot be proved; but
every one will acknowledge that, had he conquered, the Florentines
would have stood at his discretion. Sforza had the Bracceschi always
against him, so they watched each other. Francesco turned his ambition
to Lombardy; Braccio against the Church and the kingdom of Naples. But
let us come to that which happened a short while ago. The
Florentines appointed as their captain Paolo Vitelli, a most prudent
man, who from a private position had risen to the greatest renown.
If this man had taken Pisa, nobody can deny that it would have been
proper for the Florentines to keep in with him, for if he became the
soldier of their enemies they had no means of resisting, and if they
held to him they must obey him. The Venetians, if their achievements
are considered, will be seen to have acted safely and gloriously so
long as they sent to war their own men, when with armed gentlemen
and plebeians they did valiantly. This was before they turned to
enterprises on land, but when they began to fight on land they forsook
this virtue and followed the custom of Italy. And in the beginning
of their expansion on land, through not having much territory, and
because of their great reputation, they had not much to fear from
their captains; but when they expanded, as under Carmignola, they
had a taste of this mistake; for, having found him a most valiant
man (they beat the Duke of Milan under his leadership), and, on the
other hand, knowing how lukewarm he was in the war, they feared they
would no longer conquer under him, and for this reason they were not
willing, nor were they able, to let him go; and so, not to lose
again that which they had acquired, they were compelled, in order to
secure themselves, to murder him. They had afterwards for their
captains Bartolomeo da Bergamo, Roberto da San Severino, the Count
of Pitigliano, and the like, under whom they had to dread loss and not
gain, as happened afterwards at Vaila, where in one battle they lost
that which in eight hundred years they had acquired with so much
trouble. Because from such arms conquests come but slowly, long
delayed and inconsiderable, but the losses sudden and portentous.

  * As Sir John Hawkwood, the English leader of mercenaries, was
called by the Italians.

  And as with these examples I have reached Italy, which has been
ruled for many years by mercenaries, I wish to discuss them more
seriously, in order that, having seen their rise and progress, one may
be better prepared to counteract them. You must understand that the
empire has recently come to be repudiated in Italy, that the Pope
has acquired more temporal power, and that Italy has been divided up
into more states, for the reason that many of the great cities took up
arms against their nobles, who, formerly favoured by the emperor, were
oppressing them, whilst the Church was favouring them so as to gain
authority in temporal power: in many others their citizens became
princes. From this it came to pass that Italy fell partly into the
hands of the Church and of republics, and, the Church consisting of
priests and the republic of citizens unaccustomed to arms, both
commenced to enlist foreigners.
  The first who gave renown to this soldiery was Alberigo da Conio,
a native of the Romagna. From the school of this man sprang, among
others, Braccio and Sforza, who in their time were the arbiters of
Italy. After these came all the other captains who till now have
directed the arms of Italy; and the end of all their valour has
been, that she has been overrun by Charles, robbed by Louis, ravaged
by Ferdinand, and insulted by the Switzers. The principle that has
guided them has been, first, to lower the credit of infantry so that
they might increase their own. They did this because, subsisting on
their pay and without territory, they were unable to support many
soldiers, and a few infantry did not give them any authority; so
they were led to employ cavalry, with a moderate force of which they
were maintained and honoured; and affairs were brought to such a
pass that, in an army of twenty thousand soldiers, there were not to
be found two thousand foot soldiers. They had, besides this, used
every art to lessen fatigue and danger to themselves and their
soldiers, not killing in the fray, but taking prisoners and liberating
without ransom. They did not attack towns at night, nor did the
garrisons of the towns attack encampments at night; they did not
surround the camp either with stockade or ditch, nor did they campaign
in the winter. All these things were permitted by their military
rules, and devised by them to avoid, as I have said, both fatigue
and dangers; thus they have brought Italy to slavery and contempt.
                             CHAPTER XIII
        CONCERNING AUXILIARIES, MIXED SOLDIERY, AND ONE'S OWN

  AUXILIARIES, which are the other useless arm, are employed when a
prince is called in with his forces to aid and defend, as was done
by Pope Julius in the most recent times; for he, having, in the
enterprise against Ferrara, had poor proof of his mercenaries,
turned to auxiliaries, and stipulated with Ferdinand, King of Spain,
for his assistance with men and arms. These arms may be useful and
good in themselves, but for him who calls them in they are always
disadvantageous; for losing, one is undone, and winning, one is
their captive.
  And although ancient histories may be full of examples, I do not
wish to leave this recent one of Pope Julius II, the peril of which
cannot fall to be perceived; for he, wishing to get Ferrara, threw
himself entirely into the hands of the foreigner. But his good fortune
brought about a third event, so that he did not reap the fruit of
his rash choice; because, having auxiliaries routed at Ravenna, and
the Switzers having risen and driven out the conquerors (against all
expectation, both his and others), it so came to pass that he did
not become prisoner to his enemies, they having fled, nor to his
auxiliaries, he having conquered by other arms than theirs.
  The Florentines, being entirely without arms, sent ten thousand
Frenchmen to take Pisa, whereby they ran more danger than at any other
time of their troubles.
  The Emperor of Constantinople, to oppose his neighbours, sent ten
thousand Turks into Greece, who, on the war being finished, were not
willing to quit; this was the beginning of the servitude of Greece
to the infidels.
  Therefore, let him who has no desire to conquer make use of these
arms, for they are much more hazardous than mercenaries, because
with them the ruin is ready made; they are all united, all yield
obedience to others; but with mercenaries, when they have conquered,
more time and better opportunities are needed to injure you; they
are not all of one community, they are found and paid by you, and a
third party, which you have made their head, is not able all at once
to assume enough authority to injure you. In conclusion, in
mercenaries dastardy is most dangerous; in auxiliaries, valour. The
wise prince, therefore, has always avoided these arms and turned to
his own; and has been willing rather to lose with them than to conquer
with others, not deeming that a real victory which is gained with
the arms of others.
  I shall never hesitate to cite Cesare Borgia and his actions. This
duke entered the Romagna with auxiliaries, taking there only French
soldiers, and with them he captured Imola and Forli; but afterwards,
such forces not appearing to him reliable, he turned to mercenaries,
discerning less danger in them, and enlisted the Orsini and Vitelli;
whom presently, on handling and finding them doubtful, unfaithful, and
dangerous, he destroyed and turned to his own men. And the
difference between one and the other of these forces can easily be
seen when one considers the difference there was in the reputation
of the duke, when he had the French, when he had the Orsini and
Vitelli, and when he relied on his own soldiers, on whose fidelity
he could always count and found it ever increasing; he was never
esteemed more highly than when every one saw that he was complete
master of his own forces.
  I was not intending to go beyond Italian and recent examples, but
I am unwilling to leave out Hiero, the Syracusan, he being one of
those I have named above. This man, as I have said, made head of the
army by the Syracusans, soon found out that a mercenary soldiery,
constituted like our Italian condottieri, was of no use; and it
appearing to him that he could neither keep them nor let them go, he
had them all cut to pieces, and afterwards made war with his own
forces and not with aliens.
  I wish also to recall to memory an instance from the Old Testament
applicable to this subject. David offered himself to Saul to fight
with Goliath, the Philistine champion, and, to give him courage,
Saul armed him with his own weapons; which David rejected as soon as
he had them on his back, saying he could make no use of them, and that
he wished to meet the enemy with his sling and his knife. In
conclusion, the arms of others either fall from your back, or they
weigh you down, or they bind you fast.
  Charles VII, the father of King Louis XI, having by good fortune and
valour liberated France from the English, recognized the necessity
of being armed with forces of his own, and he established in his
kingdom ordinances concerning men-at-arms and infantry. Afterwards his
son, King Louis, abolished the infantry and began to enlist the
Switzers, which mistake, followed by others, is, as is now seen, a
source of peril to that kingdom; because, having raised the reputation
of the Switzers, he has entirely diminished the value of his own arms,
for he has destroyed the infantry altogether; and his men-at-arms he
has subordinated to others, for, being as they are so accustomed to
fight along with Switzers, it does not appear that they can now
conquer without them. Hence it arises that the French cannot stand
against the Switzers, and without the Switzers they do not come off
well against others. The armies of the French have thus become
mixed, partly mercenary and partly national, both of which arms
together are much better than mercenaries alone or auxiliaries
alone, yet much inferior to one's own forces. And this example
proves it, the kingdom of France would be unconquerable if the
ordinance of Charles had been enlarged or maintained.
  But the scanty wisdom of man, on entering into an affair which looks
well at first, cannot discern the poison that is hidden in it, as I
have said above of hectic fevers. Therefore, if he who rules a
principality cannot recognize evils until they are upon him, he is not
truly wise; and this insight is given to few. And if the first
disaster to the Roman Empire should be examined, it will be found to
have commenced only with the enlisting of the Goths; because from that
time the vigour of the Roman Empire began to decline, and all that
valour which had raised it passed away to others.
  I conclude, therefore, that no principality is secure without having
its own forces; on the contrary, it is entirely dependent on good
fortune, not having the valour which in adversity would defend it. And
it has always been the opinion and judgment of wise men that nothing
can be so uncertain or unstable as fame or power not founded on its
own strength. And one's own forces  are those which are composed
either of subjects, citizens, or dependants; all others are
mercenaries or auxiliaries. And the way to take ready one's own forces
will be easily found if the rules suggested by me shall be reflected
upon, and if one will consider how Philip, the father of Alexander the
Great, and many republics and princes have armed and organized
themselves, to which rules I entirely commit myself.
                             CHAPTER XIV
                     THAT WHICH CONCERNS A PRINCE
                   ON THE SUBJECT OF THE ART OF WAR

  A PRINCE ought to have no other aim or thought, nor select
anything else for his study, than war and its rules and