#include "d-gcc-includes.h"
#include "d-lang.h"
#include "d-irstate.h"

#include <assert.h>

#include "total.h"
#include "init.h"
#include "symbol.h"

#include "d-codegen.h"

IRBase::IRBase()
{
    parentIRState = 0;
    thisFunction = 0;
    loops = new Array();
    volatileDepth = 0;

    // declContextStack is composed... choose one..
}

IRState * 
IRBase::startFunction(FuncDeclaration * decl)
{
    IRState * new_irs = new IRState();
    new_irs->parentIRState = this;
    new_irs->thisFunction = decl;
    
    tree decl_tree = decl->toSymbol()->Stree;

    // Need to do this for GCC 3.4 or functions will not be emitted
    d_add_global_function( decl_tree );
    
    /*
      With this condition, nested functions are not output at -O3
    if ( ! DECL_CONTEXT( decl_tree  ) || TREE_CODE( DECL_CONTEXT( decl_tree  ) ) != FUNCTION_DECL ) {
	// wrapup_global_declarations knows not to ouput functions
	// that are not addressable and not public
	// // it probably handles the above condition as well
	//%%try//d_add_global_function( decl_tree );
    }
    */

    ModuleInfo & mi = gen.getModuleInfo(); // %% AOT 'this'
    if (decl->isStaticConstructor()) {
	mi.ctors.push(decl);
    } else if (decl->isStaticDestructor()) {
	mi.dtors.push(decl);
    } else if (decl->isUnitTestDeclaration()) {
	mi.unitTests.push(decl);
    }

    return new_irs;
}

IRState * 
IRBase::endFunction()
{
    thisFunction = 0; // make shouldDeferFunction return false
    while ( deferedFuncDecls.dim ) {
	FuncDeclaration * decl = (FuncDeclaration *) deferedFuncDecls.pop();
	decl->toObjFile();
    }
    return (IRState *) parentIRState; // %%
}

bool
IRBase::shouldDeferFunction(FuncDeclaration * decl)
{
    if (! thisFunction || decl->isNested()) {
	return false;
    } else {
	deferedFuncDecls.push(decl);
	return true;
    }
}

tree
IRBase::getLabelTree(LabelDsymbol * label)
{
    if (! label->statement->lblock) {
	tree label_decl = build_decl (LABEL_DECL, get_identifier(label->ident->string), void_type_node);

	assert(current_function_decl != 0);    
	DECL_CONTEXT( label_decl ) = current_function_decl;
	DECL_MODE( label_decl ) = VOIDmode; // Not sure why or if this is needed
	gen.setDeclLoc( label_decl, label->statement->loc ); // %% label->loc okay?
	label->statement->lblock = label_decl;
    }
    return label->statement->lblock;
}

nesting *
IRBase::getLoopForLabel(Identifier * ident)
{
    if (ident) {
	LabelStatement * lbl_stmt = getCurrentFunction()->searchLabel(ident)->statement;
	assert(lbl_stmt != 0);
	Statement * stmt = lbl_stmt->statement;

	for (int i = loops->dim - 2; i >= 0; i -= 2) {
	    if ( ((Statement *) loops->data[i]) == stmt )
		return (nesting *) loops->data[i + 1];
	}
	abort();
    } else {
	assert(loops->dim >= 2);
	return (nesting *) loops->tos();
    }
}

void
IRBase::beginLoop(Statement * stmt, nesting * loop)
{
    loops->push(stmt);
    loops->push(loop);
}

void
IRBase::endLoop()
{
    assert(loops->dim > 1);
    loops->pop();
    loops->pop();
}

tree
IRBase::getDeclContext()
{
    return declContextStack.dim ? (tree) declContextStack.tos() : (tree) 0;
}

void IRBase::pushDeclContext(tree context)
{
    declContextStack.push(context);
}

void IRBase::popDeclContext()
{
    declContextStack.pop();
}

extern "C" void pushlevel PARAMS ((int));
extern "C" tree poplevel PARAMS ((int, int, int));
extern "C" void set_block PARAMS ((tree));
extern "C" void insert_block PARAMS ((tree));

void IRBase::startBindings()
{
    pushlevel(0);
    tree block = make_node(BLOCK);
    set_block(block);

    // This is the key to getting local variables recorded for debugging.
    // Simplying having the whole tree of BLOCKs doesn't matter as
    // reorder_blocks will discard the whole thing.  
    expand_start_bindings_and_block(0, block);
}

void IRBase::endBindings()
{
    // %%TODO: reversing list causes problems with inf loops in unshare_all_decls
    tree block = poplevel(1,0,0);
    // %%TODO: DONT_JUMP_IN flag -- don't think DMD checks for
    // jumping past initializations yet..
    expand_end_bindings(NULL_TREE, 1, 0);

    // Because we used set_block, the popped level/block is not automatically recorded
    insert_block(block);
}


// This could be made more lax to allow better CSE (?)
static bool
needs_temp(tree t) {
    // %%TODO: check for anything with TREE_SIDE_EFFECTS?
    switch (TREE_CODE(t)) {
    case VAR_DECL:
    case FUNCTION_DECL:
    case PARM_DECL:
    case CONST_DECL:
    case SAVE_EXPR:
	return false;
    case ADDR_EXPR:
    case REFERENCE_EXPR:
    case INDIRECT_REF:
    case COMPONENT_REF:
    case NOP_EXPR:
    case NON_LVALUE_EXPR:
    case VIEW_CONVERT_EXPR:
	return needs_temp(TREE_OPERAND(t, 0));
    case ARRAY_REF:
	return true;
    default:
	return true;
    }
}

Expression *
IRBase::maybeMakeTemp(Expression * e)
{
    // %%TODO if no temp needed, juse return e...
    
    tree result = e->toElem( (IRState *) this); // %% ugly
    if (needs_temp(result))
	// save_expr can produce poorly performing code; eventually make a real temp for these cases
	// %%TODO: using e->op safe?
	result = save_expr(result);
    return new WrappedExp(e->loc, e->op, result, e->type);
}

tree
IRBase::maybeMakeTemp(tree t)
{
    // save_expr can produce poorly performing code; eventually make a real temp for these cases
    // e->op ok? eventually make TOKvar for a real temp
    return needs_temp(t) ? save_expr(t) : t;
}
