#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>

#include <netinet/in.h>

#include <arpa/inet.h>

#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <fdevent.h>
#include <signal.h>
#include <ctype.h>

#include <stdio.h>

#include "server.h"
#include "keyvalue.h"
#include "log.h"
#include "connections.h"
#include "joblist.h"
#include "http_chunk.h"

#include "plugin.h"

typedef struct {
	char **ptr;
	
	size_t size;
	size_t used;
} char_array;

typedef struct {
	pid_t *ptr;
	size_t used;
	size_t size;
} buffer_pid_t;

typedef struct {
	PLUGIN_DATA;
	buffer_pid_t cgi_pid;
	
	array *conf_cgi;
	
	buffer *tmp_buf;
	buffer *parse_response;
} plugin_data;

typedef struct {
	pid_t pid;
	int fd;
	int fde_ndx; /* index into the fd-event buffer */
	
	connection *remote_conn;  /* dump pointer */
	plugin_data *plugin_data; /* dump pointer */
	
	buffer *response;
	buffer *response_header;
} handler_ctx;

static handler_ctx * cgi_handler_ctx_init() {
	handler_ctx *hctx = calloc(1, sizeof(*hctx));
	
	hctx->response = buffer_init();
	hctx->response_header = buffer_init();
	
	return hctx;
}

static void cgi_handler_ctx_free(handler_ctx *hctx) {
	buffer_free(hctx->response);
	buffer_free(hctx->response_header);
	
	free(hctx);
}

enum {FDEVENT_HANDLED_UNSET, FDEVENT_HANDLED_FINISHED, FDEVENT_HANDLED_NOT_FINISHED, FDEVENT_HANDLED_ERROR};

INIT_FUNC(mod_cgi_init) {
	plugin_data *p;
	
	p = calloc(1, sizeof(*p));
	
	p->conf_cgi = array_init();
	p->tmp_buf = buffer_init();
	p->parse_response = buffer_init();
	
	return p;
}


FREE_FUNC(mod_cgi_free) {
	plugin_data *p = p_d;
	buffer_pid_t *r = &(p->cgi_pid);
	
	UNUSED(srv);

	if (r->ptr) free(r->ptr);
	
	array_free(p->conf_cgi);
	buffer_free(p->tmp_buf);
	buffer_free(p->parse_response);
	
	free(p);
	
	return HANDLER_GO_ON;
}

SETDEFAULTS_FUNC(mod_fastcgi_set_defaults) {
	plugin_data *p = p_d;
	size_t i = 0;
	
	config_values_t cv[] = { 
		{ "cgi.assign",                  NULL, T_CONFIG_ARRAY, T_CONFIG_SCOPE_CONNECTION },       /* 0 */
		{ NULL,                          NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET}
	};
	
	cv[i++].destination = p->conf_cgi;
	
	if (0 != config_insert_values(srv, cv)) {
		return HANDLER_ERROR;
	}
	
	return HANDLER_GO_ON;
}


static int cgi_pid_add(server *srv, plugin_data *p, pid_t pid) {
	int m = -1;
	size_t i;
	buffer_pid_t *r = &(p->cgi_pid);
	
	UNUSED(srv);

	for (i = 0; i < r->used; i++) {
		if (r->ptr[i] > m) m = r->ptr[i];
	}
	
	if (r->size == 0) {
		r->size = 16;
		r->ptr = malloc(sizeof(*r->ptr) * r->size);
	} else if (r->used == r->size) {
		r->size += 16;
		r->ptr = realloc(r->ptr, sizeof(*r->ptr) * r->size);
	}
	
	r->ptr[r->used++] = pid;
	
	return m;
}

static int cgi_pid_del(server *srv, plugin_data *p, pid_t pid) {
	size_t i;
	buffer_pid_t *r = &(p->cgi_pid);
	
	UNUSED(srv);

	for (i = 0; i < r->used; i++) {
		if (r->ptr[i] == pid) break;
	}
	
	if (i != r->used) {
		/* found */
		
		if (i != r->used - 1) {
			r->ptr[i] = r->ptr[r->used - 1];
		}
		r->used--;
	}
	
	return 0;
}

static int cgi_response_parse(server *srv, connection *con, plugin_data *p, buffer *in) {
	char *s, *ns;
	
	UNUSED(srv);
	
	/* \r\n -> \0\0 */
	
	buffer_copy_string_buffer(p->parse_response, in);
	
	for (s = p->parse_response->ptr; NULL != (ns = strstr(s, "\r\n")); s = ns + 2) {
		char *key, *value;
		int key_len;
		data_string *ds;
		
		ns[0] = '\0';
		ns[1] = '\0';
		
		key = s;
		if (NULL == (value = strstr(s, ": "))) {
			/* we expect: "<key>: <value>\n" */
			continue;
		}
		
		key_len = value - key;
		value += 2;
		
		if (NULL == (ds = (data_string *)array_get_unused_element(con->response.headers, TYPE_STRING))) {
			ds = data_response_init();
		}
		buffer_copy_string_len(ds->key, key, key_len);
		buffer_copy_string(ds->value, value);
		
		array_insert_unique(con->response.headers, (data_unset *)ds);
		
		switch(key_len) {
		case 4:
			if (0 == strncasecmp(key, "Date", key_len)) {
				con->parsed_response |= HTTP_DATE;
			}
			break;
		case 6:
			if (0 == strncasecmp(key, "Status", key_len)) {
				con->http_status = strtol(value, NULL, 10);
				con->parsed_response |= HTTP_STATUS;
			}
			break;
		case 8:
			if (0 == strncasecmp(key, "Location", key_len)) {
				con->parsed_response |= HTTP_LOCATION;
			}
			break;
		case 10:
			if (0 == strncasecmp(key, "Connection", key_len)) {
				con->response.keep_alive = (0 == strcasecmp(value, "Keep-Alive")) ? 1 : 0;
				con->parsed_response |= HTTP_CONNECTION;
			}
			break;
		case 14:
			if (0 == strncasecmp(key, "Content-Length", key_len)) {
				con->response.content_length = strtol(value, NULL, 10);
				con->parsed_response |= HTTP_CONTENT_LENGTH;
			}
			break;
		default:
			break;
		}
	}
	
	/* CGI/1.1 rev 03 - 7.2.1.2 */
	if ((con->parsed_response & HTTP_LOCATION) &&
	    !(con->parsed_response & HTTP_STATUS)) {
		con->http_status = 302;
	}
	
	return 0;
}


static int cgi_demux_response(server *srv, handler_ctx *hctx) {
	plugin_data *p    = hctx->plugin_data;
	connection  *con  = hctx->remote_conn;
	
	while(1) {
		int n;
		
		buffer_prepare_copy(hctx->response, 1024);
		if (-1 == (n = read(hctx->fd, hctx->response->ptr, hctx->response->size - 1))) {
			if (errno == EAGAIN || errno == EINTR) {
				/* would block, wait for signal */
				return FDEVENT_HANDLED_NOT_FINISHED;
			}
			/* error */
			log_error_write(srv, __FILE__, __LINE__, "sdd", strerror(errno), con->fd, hctx->fd);
			return FDEVENT_HANDLED_ERROR;
		}
		
		if (n == 0) {
			/* read finished */
			
			con->file_finished = 1;
			
			/* send final chunk */
			http_chunk_append_mem(srv, con, NULL, 0);
			joblist_append(srv, con);
			
			return FDEVENT_HANDLED_FINISHED;
		}
		
		hctx->response->ptr[n] = '\0';
		hctx->response->used = n+1;
		
		/* split header from body */
		
		if (con->file_started == 0) {
			char *c;
			
			buffer_append_string_buffer(hctx->response_header, hctx->response);
			
			/* search for the \r\n\r\n in the string */
			if (NULL != (c = buffer_search_string_len(hctx->response_header, "\r\n\r\n", 4))) {
				/* found */
				
				size_t hlen = c - hctx->response_header->ptr + 4;
				size_t blen = hctx->response_header->used - hlen - 1;
				
				/* a small hack: terminate after at the second \r */
				hctx->response_header->used = hlen + 1 - 2;
				hctx->response_header->ptr[hlen - 2] = '\0';
				
				/* parse the response header */
				cgi_response_parse(srv, con, p, hctx->response_header);
				
				/* enable chunked-transfer-encoding */
				if (con->request.http_version == HTTP_VERSION_1_1 &&
				    !(con->parsed_response & HTTP_CONTENT_LENGTH)) {
					con->response.transfer_encoding = HTTP_TRANSFER_ENCODING_CHUNKED;
				}
				
				if ((hctx->response->used != hlen) && blen > 0) {
					http_chunk_append_mem(srv, con, c + 4, blen + 1);
					joblist_append(srv, con);
				}
				
				con->file_started = 1;
			}
		} else {
			http_chunk_append_mem(srv, con, hctx->response->ptr, hctx->response->used);
			joblist_append(srv, con);
		}
		
#if 0		
		log_error_write(srv, __FILE__, __LINE__, "ddss", con->fd, hctx->fd, connection_get_state(con->state), b->ptr);
#endif
	}
	
	return FDEVENT_HANDLED_NOT_FINISHED;
}

static handler_t cgi_connection_close(server *srv, handler_ctx *hctx) {
	int status;
	pid_t pid;
	plugin_data *p;
	connection  *con;
	
	if (NULL == hctx) return HANDLER_GO_ON;
	
	p    = hctx->plugin_data;
	con  = hctx->remote_conn;
	
	if (con->mode != p->id) return HANDLER_GO_ON;
	
#if 0
	log_error_write(srv, __FILE__, __LINE__, "sdd", 
			"emergency exit: cgi", 
			con->fd,
			hctx->fd);
#endif
	
	/* the connection to the browser went away, but we still have a connection
	 * to the CGI script 
	 *
	 * close cgi-connection
	 */
	
	/* close connection to the cgi-script */
	fdevent_event_del(srv->ev, &(hctx->fde_ndx), hctx->fd);
	fdevent_unregister(srv->ev, hctx->fd);
	
	if (close(hctx->fd)) {
		log_error_write(srv, __FILE__, __LINE__, "sds", "cgi close failed ", hctx->fd, strerror(errno));
	}
	
	hctx->fd = -1;
	hctx->fde_ndx = -1;
	
	pid = hctx->pid;
	
	con->plugin_ctx[p->id] = NULL;
	
	/* is this a good idea ? */
	cgi_handler_ctx_free(hctx);
	
	/* if waitpid hasn't been called by response.c yet, do it here */
	if (pid) {
		/* check if the CGI-script is already gone */
		switch(waitpid(pid, &status, WNOHANG)) {
		case 0:
			/* not finished yet */
#if 0
			log_error_write(srv, __FILE__, __LINE__, "sd", "(debug) child isn't done yet, pid:", pid);
#endif
			break;
		case -1:
			/* */
			if (errno == EINTR) break;
			
			/* 
			 * errno == ECHILD happens if _subrequest catches the process-status before 
			 * we have read the response of the cgi process
			 * 
			 * -> catch status
			 * -> WAIT_FOR_EVENT
			 * -> read response
			 * -> we get here with waitpid == ECHILD
			 * 
			 */
			if (errno == ECHILD) return HANDLER_FINISHED;
			
			log_error_write(srv, __FILE__, __LINE__, "ss", "waitpid failed: ", strerror(errno));
			return HANDLER_ERROR;
		default:
			if (WIFEXITED(status)) {
#if 0
				log_error_write(srv, __FILE__, __LINE__, "sd", "(debug) cgi exited fine, pid:", pid);
#endif
				pid = 0;
				
				return HANDLER_FINISHED;
			} else {
				log_error_write(srv, __FILE__, __LINE__, "sd", "cgi died, pid:", pid);
				pid = 0;
				return HANDLER_FINISHED;
			}
		}
		
	
		kill(pid, SIGTERM);
		
		/* cgi-script is still alive, queue the PID for removal */
		cgi_pid_add(srv, p, pid);
	}
	
	return HANDLER_FINISHED;
}

static handler_t cgi_connection_close_callback(server *srv, connection *con, void *p_d) {
	plugin_data *p = p_d;
	
	return cgi_connection_close(srv, con->plugin_ctx[p->id]);
}


static handler_t cgi_handle_fdevent(void *s, void *ctx, int revents) {
	server      *srv  = (server *)s;
	handler_ctx *hctx = ctx;
	connection  *con  = hctx->remote_conn;
	
	joblist_append(srv, con);
	
	if (hctx->fd == -1) {
		log_error_write(srv, __FILE__, __LINE__, "ddss", con->fd, hctx->fd, connection_get_state(con->state), "invalid cgi-fd");
		
		return HANDLER_ERROR;
	}
	
	if (revents & FDEVENT_IN) {
		switch (cgi_demux_response(srv, hctx)) {
		case FDEVENT_HANDLED_NOT_FINISHED:
			break;
		case FDEVENT_HANDLED_FINISHED:
			/* we are done */
			
#if 0
			log_error_write(srv, __FILE__, __LINE__, "ddss", con->fd, hctx->fd, connection_get_state(con->state), "finished");
#endif
			
			break;
		case FDEVENT_HANDLED_ERROR:
			connection_set_state(srv, con, HANDLE_REQUEST);
			con->http_status = 500;
			con->mode = DIRECT;
			
			log_error_write(srv, __FILE__, __LINE__, "s", "demuxer failed: ");
			break;
		}
	}
	
	if (revents & FDEVENT_OUT) {
		/* nothing to do */
	}
	
	/* perhaps this issue is already handled */
	if (revents & FDEVENT_HUP) {
		/* check if we still have a unfinished header package which is a body in reality */
		if (con->file_started == 0 &&
		    hctx->response_header->used) {
			con->file_started = 1;
			http_chunk_append_mem(srv, con, hctx->response_header->ptr, hctx->response_header->used);
			joblist_append(srv, con);
		}
		
		if (con->file_finished == 0) {
			http_chunk_append_mem(srv, con, NULL, 0);
			joblist_append(srv, con);
		}
		
		con->file_finished = 1;
		
		if (chunkqueue_is_empty(con->write_queue)) {
			/* there is nothing left to write */
			connection_set_state(srv, con, RESPONSE_END);
		} else {
			/* used the write-handler to finish the request on demand */
			
		}
		
# if 0
		log_error_write(srv, __FILE__, __LINE__, "sddd", "got HUP from cgi", con->fd, hctx->fd, revents);
# endif
		
		/* rtsigs didn't liked the close */
		cgi_connection_close(srv, hctx);
	} else if (revents & FDEVENT_ERR) {
		con->file_finished = 1;
		
		/* kill all connections to the cgi process */
		cgi_connection_close(srv, hctx);
#if 1
		log_error_write(srv, __FILE__, __LINE__, "s", "cgi-FDEVENT_ERR");
#endif			
		return HANDLER_ERROR;
	}
	
	return HANDLER_FINISHED;
}


static int cgi_env_add(char_array *env, const char *key, const char *val) {
	int key_len, val_len;
	char *dst;
	
	if (!key || !val) return -1;
	
	key_len = strlen(key);
	val_len = strlen(val);
	
	dst = malloc(key_len + val_len + 3);
	memcpy(dst, key, key_len);
	dst[key_len] = '=';
	/* add the \0 from the value */
	memcpy(dst + key_len + 1, val, val_len + 1);
	
	if (env->size == 0) {
		env->size = 16;
		env->ptr = malloc(env->size * sizeof(*env->ptr));
	} else if (env->size == env->used) {
		env->size += 16;
		env->ptr = realloc(env->ptr, env->size * sizeof(*env->ptr));
	}
	
	env->ptr[env->used++] = dst;
	
	return 0;
}

static int cgi_create_env(server *srv, connection *con, plugin_data *p, buffer *cgi_handler) {
	pid_t pid;
	
#ifdef HAVE_IPV6
	char b2[INET6_ADDRSTRLEN + 1];
#endif
	
	int to_cgi_fds[2];
	int from_cgi_fds[2];
	
	if (pipe(to_cgi_fds)) {
		log_error_write(srv, __FILE__, __LINE__, "ss", "pipe failed: ", strerror(errno));
		return -1;
	}
	
	if (pipe(from_cgi_fds)) {
		log_error_write(srv, __FILE__, __LINE__, "ss", "pipe failed: ", strerror(errno));
		return -1;
	}
	
	if (con->request.content->used) {
		write(to_cgi_fds[1], con->request.content->ptr, con->request.content_length);
	}
	
	/* fork, execve */
	switch (pid = fork()) {
	case 0: {
		/* child */
		char **args;
		int argc;
		int i = 0;
		char buf[32];
		size_t n;
		char_array env;
		char *c;
		server_socket *srv_sock = con->srv_socket;
		
		/* move stdout to from_cgi_fd[1] */
		close(STDOUT_FILENO);
		dup2(from_cgi_fds[1], STDOUT_FILENO);
		close(from_cgi_fds[1]);
		/* not needed */
		close(from_cgi_fds[0]);
		
		/* move the stdin to to_cgi_fd[0] */
		close(STDIN_FILENO);
		dup2(to_cgi_fds[0], STDIN_FILENO);
		close(to_cgi_fds[0]);
		/* not needed */
		close(to_cgi_fds[1]);
		
		/* create environment */
		env.ptr = NULL;
		env.size = 0;
		env.used = 0;
		
#ifdef PACKAGE_NAME
		cgi_env_add(&env, "SERVER_SOFTWARE", PACKAGE_NAME"/"PACKAGE_VERSION);
#else
		cgi_env_add(&env, "SERVER_SOFTWARE", PACKAGE"/"VERSION);
#endif
		cgi_env_add(&env, "SERVER_NAME", 
#ifdef HAVE_IPV6
			    inet_ntop(srv_sock->addr.plain.sa_family, 
				      srv_sock->addr.plain.sa_family == AF_INET6 ? 
				      (const void *) &(srv_sock->addr.ipv6.sin6_addr) :
				      (const void *) &(srv_sock->addr.ipv4.sin_addr),
				      b2, sizeof(b2)-1)
#else
			    inet_ntoa(srv_sock->addr.ipv4.sin_addr)
#endif
			    );
		cgi_env_add(&env, "GATEWAY_INTERFACE", "CGI/1.1");
		
		cgi_env_add(&env, "SERVER_PROTOCOL", get_http_version_name(con->request.http_version));
		
		ltostr(buf, 
#ifdef HAVE_IPV6
			ntohs(srv_sock->addr.plain.sa_family == AF_INET6 ? srv_sock->addr.ipv6.sin6_port : srv_sock->addr.ipv4.sin_port)
#else
			ntohs(srv_sock->addr.ipv4.sin_port)
#endif
			);
		cgi_env_add(&env, "SERVER_PORT", buf);
		cgi_env_add(&env, "REQUEST_METHOD", get_http_method_name(con->request.http_method));
		if (con->request.pathinfo->used) {
			cgi_env_add(&env, "PATH_INFO", con->request.pathinfo->ptr);
		}
		cgi_env_add(&env, "REDIRECT_STATUS", "200");
		cgi_env_add(&env, "QUERY_STRING", con->uri.query->used ? con->uri.query->ptr : "");
		cgi_env_add(&env, "REMOTE_ADDR", 
#ifdef HAVE_IPV6
			    inet_ntop(con->dst_addr.plain.sa_family, 
				      con->dst_addr.plain.sa_family == AF_INET6 ? 
				      (const void *) &(con->dst_addr.ipv6.sin6_addr) :
				      (const void *) &(con->dst_addr.ipv4.sin_addr),
				      b2, sizeof(b2)-1)
#else
			    inet_ntoa(con->dst_addr.ipv4.sin_addr)
#endif
			    );
		
		/* request.content_length < SSIZE_MAX, see request.c */
		ltostr(buf, con->request.content_length);
		cgi_env_add(&env, "CONTENT_LENGTH", buf);
		cgi_env_add(&env, "SCRIPT_FILENAME", con->physical.path->ptr);
		cgi_env_add(&env, "SCRIPT_NAME", con->uri.path->ptr);
		
		/* for valgrind */
		cgi_env_add(&env, "LD_PRELOAD", getenv("LD_PRELOAD"));
		cgi_env_add(&env, "LD_LIBRARY_PATH", getenv("LD_LIBRARY_PATH"));
		
		for (n = 0; n < con->request.headers->used; n++) {
			data_string *ds;
			
			ds = (data_string *)con->request.headers->data[n];
			
			if (ds->value->used && ds->key->used) {
				size_t j;
				
				buffer_reset(p->tmp_buf);
				
				if (0 != strcasecmp(ds->key->ptr, "CONTENT_TYPE")) {
					buffer_copy_string(p->tmp_buf, "HTTP_");
				}
				
				p->tmp_buf->used--;
				buffer_prepare_append(p->tmp_buf, ds->key->used + 1);
				
				for (j = 0; j < ds->key->used - 1; j++) {
					p->tmp_buf->ptr[p->tmp_buf->used++] = 
						isalpha((unsigned char)ds->key->ptr[j]) ? 
						toupper((unsigned char)ds->key->ptr[j]) : '_';
				}
				p->tmp_buf->ptr[p->tmp_buf->used++] = '\0';
				
				cgi_env_add(&env, p->tmp_buf->ptr, ds->value->ptr);
			}
		}
		
		if (env.size == env.used) {
			env.size += 16;
			env.ptr = realloc(env.ptr, env.size * sizeof(*env.ptr));
		}
		
		env.ptr[env.used] = NULL;
		
		/* set up args */
		argc = 3;
		args = malloc(sizeof(*args) * argc);
		i = 0;
		
		if (cgi_handler->used > 1) {
			args[i++] = cgi_handler->ptr;
		}
		args[i++] = con->physical.path->ptr;
		args[i++] = NULL;

		/* search for the last / */
		if (NULL != (c = strrchr(con->physical.path->ptr, '/'))) {
			*c = '\0';
			
			/* change to the physical directory */
			if (-1 == chdir(con->physical.path->ptr)) {
				log_error_write(srv, __FILE__, __LINE__, "sss", "CGI failed: ", strerror(errno), con->physical.doc_root->ptr);
			}
			*c = '/';
		}

		/* we don't need the client socket */
		for (i = 3; i < 256; i++) {
			if (i != srv->log_error_fd) close(i);
		}
		
		/* exec the cgi */
		execve(args[0], args, env.ptr);
		
		log_error_write(srv, __FILE__, __LINE__, "sss", "CGI failed: ", strerror(errno), args[0]);
		
		/* */
		SEGFAULT();
		break;
	}
	case -1:
		/* error */
		log_error_write(srv, __FILE__, __LINE__, "ss", "fork failed: ", strerror(errno));
		break;
	default: {
		handler_ctx *hctx;
		/* father */
		
		close(from_cgi_fds[1]);
		
		close(to_cgi_fds[0]);
		close(to_cgi_fds[1]);
		
		/* register PID and wait for them asyncronously */
		con->mode = p->id;
		buffer_reset(con->physical.path);
		
		hctx = cgi_handler_ctx_init();
		
		hctx->remote_conn = con;
		hctx->plugin_data = p;
		hctx->pid = pid;
		hctx->fd = from_cgi_fds[0];
		hctx->fde_ndx = -1;
		
		con->plugin_ctx[p->id] = hctx;
		
		fdevent_register(srv->ev, hctx->fd, cgi_handle_fdevent, hctx);
		fdevent_event_add(srv->ev, &(hctx->fde_ndx), hctx->fd, FDEVENT_IN);
		
		if (-1 == fdevent_fcntl_set(srv->ev, hctx->fd)) {
			log_error_write(srv, __FILE__, __LINE__, "ss", "fcntl failed: ", strerror(errno));
			
			fdevent_event_del(srv->ev, &(hctx->fde_ndx), hctx->fd);
			fdevent_unregister(srv->ev, hctx->fd);
			
			log_error_write(srv, __FILE__, __LINE__, "sd", "cgi close:", hctx->fd);
			
			close(hctx->fd);
			
			cgi_handler_ctx_free(hctx);
			
			con->plugin_ctx[p->id] = NULL;
			
			return -1;
		}
		
		break;
	}
	}
	
	return 0;
}

static handler_t cgi_is_handled(server *srv, connection *con, void *p_d) {
	size_t k, s_len;
	plugin_data *p = p_d;
	buffer *fn = con->physical.path;
	
	if (fn->used == 0) return HANDLER_ERROR;
	
	s_len = fn->used - 1;
	
	for (k = 0; k < p->conf_cgi->used; k++) {
		data_string *ds = (data_string *)p->conf_cgi->data[k];
		size_t ct_len = ds->key->used - 1;
		
		if (ds->key->used == 0) continue;
		if (s_len < ct_len) continue;
		
		if (0 == strncmp(fn->ptr + s_len - ct_len, ds->key->ptr, ct_len)) {
			if (cgi_create_env(srv, con, p, ds->value)) {
				log_error_write(srv, __FILE__, __LINE__, "s", "(debug) cgi_create_env failed");
			}
			
			return HANDLER_FINISHED;
		}
	}
	
	return HANDLER_GO_ON;
}

TRIGGER_FUNC(cgi_trigger) {
	plugin_data *p = p_d;
	size_t ndx;
	/* the trigger handle only cares about lonely PID which we have to wait for */

	for (ndx = 0; ndx < p->cgi_pid.used; ndx++) {
		int status;
		
		switch(waitpid(p->cgi_pid.ptr[ndx], &status, WNOHANG)) {
		case 0:
			/* not finished yet */
#if 0
			log_error_write(srv, __FILE__, __LINE__, "sd", "(debug) child isn't done yet, pid:", p->cgi_pid.ptr[ndx]);
#endif
			break;
		case -1:
			log_error_write(srv, __FILE__, __LINE__, "ss", "waitpid failed: ", strerror(errno));
			
			return HANDLER_ERROR;
		default:

			if (WIFEXITED(status)) {
#if 0
				log_error_write(srv, __FILE__, __LINE__, "sd", "(debug) cgi exited fine, pid:", p->cgi_pid.ptr[ndx]);
#endif
			} else {
				log_error_write(srv, __FILE__, __LINE__, "s", "cgi died ?");
			}
			
			cgi_pid_del(srv, p, p->cgi_pid.ptr[ndx]);
			/* del modified the buffer structure 
			 * and copies the last entry to the current one
			 * -> recheck the current index
			 */
			ndx--;
		}
	}
	
	return HANDLER_GO_ON;
}

SUBREQUEST_FUNC(mod_cgi_handle_subrequest) {
	int status;
	plugin_data *p = p_d;
	handler_ctx *hctx = con->plugin_ctx[p->id];
	
	if (con->mode != p->id) return HANDLER_GO_ON;
	if (NULL == hctx) return HANDLER_GO_ON;
	
#if 0
	log_error_write(srv, __FILE__, __LINE__, "sdd", "subrequest, pid =", hctx, hctx->pid);
#endif	
	if (hctx->pid == 0) return HANDLER_FINISHED;
	
	switch(waitpid(hctx->pid, &status, WNOHANG)) {
	case 0:
		/* not finished yet */
		if (con->file_started) {
			return HANDLER_GO_ON;
		} else {
			return HANDLER_WAIT_FOR_EVENT;
		}
	case -1:
		if (errno == EINTR) return HANDLER_WAIT_FOR_EVENT;
		
		if (errno == ECHILD && con->file_started == 0) {
			/*
			 * second round but still not response 
			 */
			return HANDLER_WAIT_FOR_EVENT; 
		}
		
		log_error_write(srv, __FILE__, __LINE__, "ss", "waitpid failed: ", strerror(errno));
		con->mode = DIRECT;
		con->http_status = 500;
		
		hctx->pid = 0;
		
		fdevent_event_del(srv->ev, &(hctx->fde_ndx), hctx->fd);
		fdevent_unregister(srv->ev, hctx->fd);
		
		if (close(hctx->fd)) {
			log_error_write(srv, __FILE__, __LINE__, "sds", "cgi close failed ", hctx->fd, strerror(errno));
		}
		
		cgi_handler_ctx_free(hctx);
		
		con->plugin_ctx[p->id] = NULL;
		
		return HANDLER_FINISHED;
	default:
		/* cgi process exited cleanly 
		 * 
		 * check if we already got the response 
		 */
		
		if (!con->file_started) return HANDLER_WAIT_FOR_EVENT;
		
		if (WIFEXITED(status)) {
			/* nothing */
		} else {
			log_error_write(srv, __FILE__, __LINE__, "s", "cgi died ?");
			
			con->mode = DIRECT;
			con->http_status = 500;
			
		}
		
		hctx->pid = 0;
		
		fdevent_event_del(srv->ev, &(hctx->fde_ndx), hctx->fd);
		fdevent_unregister(srv->ev, hctx->fd);
		
		if (close(hctx->fd)) {
			log_error_write(srv, __FILE__, __LINE__, "sds", "cgi close failed ", hctx->fd, strerror(errno));
		}
		
		cgi_handler_ctx_free(hctx);
		
		con->plugin_ctx[p->id] = NULL;
		return HANDLER_FINISHED;
	}
}


int mod_cgi_plugin_init(plugin *p) {
	p->name        = buffer_init_string("cgi");

	p->handle_connection_close = cgi_connection_close_callback;
	p->handle_subrequest_start = cgi_is_handled;
	p->handle_subrequest = mod_cgi_handle_subrequest;
#if 0
	p->handle_fdevent = cgi_handle_fdevent;
#endif
	p->handle_trigger = cgi_trigger;
	p->init           = mod_cgi_init;
	p->cleanup        = mod_cgi_free;
	p->set_defaults   = mod_fastcgi_set_defaults;
	
	p->data        = NULL;
	
	return 0;
}
